(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/commands/acp.sml
 * description: command interface for confluence checker
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature ACP =  
sig
    val runDebug: bool ref
    val runProfile: bool ref
    val main: string * string list -> OS.Process.status
    val mkCommand: unit -> unit
    val install: unit -> unit
(*    val mkCommand2: int -> unit *)
end;

structure Acp: ACP  =
struct

val runDebug = ref false : bool ref
fun debug f = if !runDebug then f () else ()

val runProfile = ref false

local
    structure PU = PrintUtil
    val numOfSuccess = ref 0;
    val numOfFailure = ref 0;

    exception Abort;
    exception Unsupported;

    datatype option_result = HelpOption | Proceed | Error of string
    fun printOptionError [] = PU.printlnStdErr "option error?"
      | printOptionError ((Error str)::_) =  PU.printlnStdErr ("Error: " ^ str)
      | printOptionError (x::xs) = printOptionError xs

    val usage = "Usage: cmd <file or directory>, for details use --help"
    (* val usageFoSSaCS2020 = "Usage: cmd <file or directory>" *)

    val useDirectOption = ref true
    val useCommutativeOption = ref true
    val useNonMinimalCommutativeOption = ref false
    val useLayerOption = ref true

    val timeLimit = ref 0
    val verboseMode = ref 0
    val version = "0.74"

    val useCertifiableOutputOption = ref false
    val useManySortedInputOption = ref false

    fun setDebug () = 
	let val _ = if (!verboseMode) >= 3 then runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then Cr.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then CrSolver.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then CrCompletion.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then CrDiagram.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then CrOrd.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then NonCr.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then Unc.runDebug := true else ()
	    val _ = if (!verboseMode) >= 3 then Unr.runDebug := true else ()
	    val _ = if (!verboseMode) >= 4 
		    then  
			print ("Warning:  Verbosity level of > 3 is unofficial. \n"
				 ^ "You need to delete temporary files *manually*.\n")
		    else ()
	    val _ = if (!verboseMode) >= 4 then  Solver.runDebug := true else ()
	    val _ = if (!verboseMode) >= 4 then  DpSolver.runDebug := true else ()
	    val _ = if (!verboseMode) >= 4 then  Dp.runDebug := true else ()
	    val _ = if (!verboseMode) >= 5 then  PolynomialInterpretation.runDebug := true else ()
	    val _ = if (!verboseMode) >= 5 then  MatrixInterpretation.runDebug := true else ()
	    val _ = if (!verboseMode) >= 5 then  PoSolver.runDebug := true else ()
	    val _ = if (!verboseMode) >= 5 then  Order.runDebug := true else ()
	    val _ = if (!verboseMode) >= 5 then  TreeAutomata.runDebug := true else ()
	in ()
	end

    fun resetDebug () = 
	let val _ = runDebug := false
	    val _ = Cr.runDebug := false
	    val _ = CrSolver.runDebug := false
	    val _ = CrCompletion.runDebug := false
	    val _ = CrDiagram.runDebug := false
	    val _ = Unc.runDebug := false
	    val _ =  Solver.runDebug := false
	    val _ =  PoSolver.runDebug := false
	    val _ =  DpSolver.runDebug := false
	    val _ =  Dp.runDebug := false
	    val _ =  Order.runDebug := false
	    val _ = PolynomialInterpretation.runDebug := false
	    val _ = MatrixInterpretation.runDebug := false
	in ()
	end

    fun setInit () = (
        useDirectOption := true;  (* persistency  decomposition *)
        useCommutativeOption := true;
        useNonMinimalCommutativeOption := false;
        useLayerOption := true;
        useCertifiableOutputOption := false;
        useManySortedInputOption := false;
        CrSolver.useHuetToyamaOostrom:= true;
        CrSolver.useHuetStrongClosed:= true;
        CrSolver.useOyamaguchiOhta:= true;
        CrSolver.useToyamaParallel:= true;
        CrSolver.useOkuiSimultaneous:= true;
        CrSolver.useDecreasingDiagrams := true;
        CrSolver.useOhtaOyamaguchiToyama:= true;
        CrSolver.useGomiOyamaguchiOhtaDepth:= true;
        CrSolver.useGomiOyamaguchiOhtaWeight:= true;
        CrSolver.useSuzukiAotoToyama:= true;
        CrSolver.useUchidaAotoToyama:= true;
        CrSolver.useUchidaAotoToyamaStrong:= true;
        CrSolver.useSakaiOgawa:= true;
        CrSolver.useKnuthBendix:= true;
        CrSolver.useHuetModulo:= true;
	CrSolver.useToyamaOyamaguchiWeightDecreasing := true;
	CrSolver.useMartinNipkowOrderedRewriting := true;
	NonCr.useNonJoinableByApproximation := true;
	NonCr.useNonJoinableByTreeAutomata := true;
	NonCr.useNonJoinableByInterpretationAndOrder := true;
        CrSolver.useBidirectionalCompletion := true;
	Unc.useNonOmegaOverlapping := true;
	Unc.useParallelClosedLinearization := true;
	Unc.useStronglyClosedLinearization := true;
  	Unc.useWeightDecreasingJoinable := true;
  	Unc.useRightReducible := true;
  	Unc.useUncCompletionByStronglyClosed := true;
  	Unc.useUncCompletionByDevelopmentClosed  := true;
  	Unc.useGeneralUncCompletion  := true;
  	Unc.useNonUncByCps := true;
  	Unc.useRuleReversing := true;
	Unc.useConfluenceProver := true;
	Unc.useShallowDecProc := true;
	CrSolver.useYoshidaAotoToyama := true;
	CrSolver.useNonCommutation := true;
	resetDebug ();
        verboseMode := 1;
        timeLimit := 0)

    val _ = setInit ();

    fun setUseOption useOption str name =
	case name
            of "0" => (useOption := false; Proceed)
             | "1" => (useOption := true; Proceed)
             | "yes" => (useOption := true; Proceed)
             | "no" => (useOption := false; Proceed)
             | ow => Error ("illegal option: " ^ ow); 

    val tmpDir = ref (OS.FileSys.fullPath ".")

    fun setTmpDir path = 
        let val path' = OS.FileSys.fullPath path
            val _ = debug (fn _ => PU.printStdErr ("tmpDir: " ^ path' ^ "\n"))
        in if OS.FileSys.access (path', [OS.FileSys.A_WRITE])
           then (tmpDir := path'; Proceed)
           else Error ("directory not writable: " ^ path)
        end
        handle OS.SysErr (msg,_) => (Error ("directory not found: " ^ path))

    fun setExecutableFile name path =
        let val _ = debug (fn _ => PU.printStdErr ("[" ^ path ^ "]\n"))
	    val tokens = String.tokens (fn c => c = #" ") path
	    val command = hd tokens
	    val options = tl tokens
	    val path' = OS.FileSys.fullPath command
        in if OS.FileSys.access (path', [OS.FileSys.A_EXEC])
           then (name := String.concatWith " " (path'::options); Proceed)
           else Error ("file not executable: " ^ path)
        end
        handle OS.SysErr (msg,_) => Error ("file not found: " ^ path)

    val minisatPath = ref ""
    fun setMinisatPath path = setExecutableFile minisatPath path 

    val yicesPath = ref ""
    fun setYicesPath path = setExecutableFile yicesPath path 

    val terminationProver = ref ""
    fun setTerminationProver path = setExecutableFile terminationProver path 

    val relativeTerminationProver = ref ""
    fun setRelativeTerminationProver path = setExecutableFile relativeTerminationProver path 

    val certifiableOutputFile = ref ""

    fun setCertifiableOutputFile path =
        let val path' = OS.FileSys.fullPath path
        in if not (OS.FileSys.isDir path') 
	      andalso OS.FileSys.access (path', [OS.FileSys.A_WRITE])
           then (useCertifiableOutputOption := true;
		 debug (fn _ => PU.printStdErr ("certifiable output file: " ^ path' ^ "\n"));
		 certifiableOutputFile := path'; Proceed)
           else Error ("file not writable: " ^ path)
	end
	handle OS.SysErr (msg,_) => 
	       let val dir = OS.Path.getParent path
		   val file = OS.Path.file path
		   val dir' = OS.FileSys.fullPath dir
		   val path' = OS.Path.concat (dir',file)
               in if OS.FileSys.access (dir', [OS.FileSys.A_WRITE])
		  then (useCertifiableOutputOption := true;
			debug (fn _ => PU.printStdErr ("certifiable output file: " ^ path' ^ "\n"));
			certifiableOutputFile := path'; Proceed)
		  else Error ("file not writable: " ^ path)
	       end

    fun setCertifiableOutputOption path = 
	(CertifyUtil.versionNumber := version;
	 CertifyUtil.toolName := "ACP";
	 setCertifiableOutputFile path)
	
    datatype target_property = CR | UNC | UNR | COM
    val targetProperty = ref CR

    fun setPropertyOption keywords = 
	case keywords of
	    "cr" => (targetProperty := CR; Proceed)
	  | "unc" => (NonCr.useNonJoinableByApproximation := false; (* disable time consuming non-cr checks *)
		      NonCr.useNonJoinableByTreeAutomata := false;
		      NonCr.useNonJoinableByInterpretationAndOrder := false;
		      targetProperty := UNC; Proceed)
	  | "unr" => (NonCr.useNonJoinableByApproximation := false; (* disable time consuming non-cr checks *)
		      NonCr.useNonJoinableByTreeAutomata := false;
		      NonCr.useNonJoinableByInterpretationAndOrder := false;
		      targetProperty := UNR; Proceed)
	  |  "com" => (targetProperty := COM; Proceed)
	  | _ => Error ("illegal option: -p" ^ keywords)

    fun setTimerOption num  =
        case Int.fromString num
         of SOME i => (timeLimit := i; Proceed)
          | NONE => Error ("illegal option: -t" ^ num)

    fun setDirectOption () = 
        (useDirectOption := false;
         useCommutativeOption := false;
	 useNonMinimalCommutativeOption := false;
         useLayerOption := false;
         Proceed)

    fun setVerboseOption num  =
	case Int.fromString num
         of SOME i => (verboseMode := i; setDebug (); Proceed)
          | NONE => Error ("illegal option: -v" ^ num)

   fun setManySortedInputOption () = (useManySortedInputOption := true; Proceed)

    fun setResetOption () = 
		(CrSolver.useHuetToyamaOostrom:= false;
		 CrSolver.useHuetStrongClosed:= false;
		 CrSolver.useOyamaguchiOhta:= false;
		 CrSolver.useToyamaParallel:= false;
		 CrSolver.useOkuiSimultaneous:= false;
		 CrSolver.useDecreasingDiagrams := false;
		 CrSolver.useOhtaOyamaguchiToyama:= false;
		 CrSolver.useGomiOyamaguchiOhtaDepth:= false;
		 CrSolver.useGomiOyamaguchiOhtaWeight:= false;
		 CrSolver.useSuzukiAotoToyama:= false;
		 CrSolver.useUchidaAotoToyama:= false;
		 CrSolver.useUchidaAotoToyamaStrong:= false;
		 CrSolver.useSakaiOgawa:= false;
		 CrSolver.useKnuthBendix:= false;
		 CrSolver.useHuetModulo:= false;
		 CrSolver.useToyamaOyamaguchiWeightDecreasing := false;
		 CrSolver.useMartinNipkowOrderedRewriting := false;
		 NonCr.useNonJoinableByApproximation := false;
		 NonCr.useNonJoinableByTreeAutomata := false;
		 NonCr.useNonJoinableByInterpretationAndOrder := false;
		 CrSolver.useBidirectionalCompletion := false;
		 Unc.useNonOmegaOverlapping := false;
		 Unc.useParallelClosedLinearization := false;
		 Unc.useStronglyClosedLinearization := false;
  		 Unc.useWeightDecreasingJoinable := false;
  		 Unc.useRightReducible := false;
  		 Unc.useUncCompletionByStronglyClosed := false;
  		 Unc.useUncCompletionByDevelopmentClosed  := false;
  		 Unc.useGeneralUncCompletion  := false;
  		 Unc.useNonUncByCps := false;
  		 Unc.useRuleReversing := false;
		 Unc.useConfluenceProver := false;
		 Unc.useShallowDecProc := false;
		 CrSolver.useYoshidaAotoToyama := false;
		 CrSolver.useNonCommutation := false;
		 Proceed)

    val options = [{short = "h",
                    long = ["help"],
                    desc = GetOpt.NoArg (fn () => HelpOption),
                    help = "print this message"}
		  ,
                    {short = "v",
                     long = ["verbose"],
                     desc = GetOpt.ReqArg (setVerboseOption, "0|1|2|3"),
                     help = ("verbosity level [0:YES/NO/MAYBE,1(default):YES/NO/MAYBE+explanation,2--]")}
		  ,
                    {short = "p",
                     long = ["property"],
                     desc = GetOpt.ReqArg (setPropertyOption, "cr|unc|unr|com"),
                     help = "specify the target property to prove [cr(default),unc,unr,com]"}
		  ,
                    {short = "t",
                     long = ["timer"],
                     desc = GetOpt.ReqArg (setTimerOption, "int"),
                     help = "specify the time limit in seconds [default:0 (no-limit)]"}
		  ,
                    {short = "",
                     long = ["minisat-path"],
                     desc = GetOpt.ReqArg (setMinisatPath, "path"),
                     help = "specify an executable minisat [default:./minisat]"}
		  ,
                    {short = "",
                     long = ["yices-path"],
                     desc = GetOpt.ReqArg (setYicesPath, "path"),
                     help = "specify an executable yices [default:./yices]"}
		  ,
                    {short = "",
                     long = ["tmp-dir"],
                     desc = GetOpt.ReqArg (setTmpDir, "path"),
                     help = "specify a directory for putting temporal files [default:.]"}
		  ,
                    {short = "",
                     long = ["termination-prover"],
                     desc = GetOpt.ReqArg (setTerminationProver, "path"),
                     help = "specify an external termination prover [default:(internal)]"}
		  ,
                    {short = "",
                     long = ["relative-termination-prover"],
                     desc = GetOpt.ReqArg (setRelativeTerminationProver, "path"),
                     help = "specify an external relative termination prover [default:(internal)]"}
		  ,
                    {short = "",
                     long = ["certifiable-proofs-output"],
                     desc = GetOpt.ReqArg (setCertifiableOutputOption, "path"),
                     help = "cr: specify output file for certifiable proofs"}
(*** keep this, but not included to ACP
		  ,
                    {short = "",
                     long = ["mstrs"],
                     desc = GetOpt.NoArg (fn () => setManySortedInputOption ()),
                     help = "use many-sorted TRS as input"}
****)
		  ,
                    {short = "d",
                     long = ["direct"],
                     desc = GetOpt.NoArg (fn () => setDirectOption ()),
                     help = "cr/unc: equivalent to --ps=0 --lp=0 --cm=0"}
		  ,
                    {short = "",
                     long = ["ps"],
                     desc = GetOpt.ReqArg (setUseOption useDirectOption "ps","0|1|no|yes"),
                     help = "cr/unc: use persistent decomposition [default:yes]"}
		  ,
                    {short = "",
                     long = ["lp"],
                     desc = GetOpt.ReqArg (setUseOption useLayerOption "lp","0|1|no|yes"),
                     help = "cr/unc: use layer-preserving decomposition [default:yes]"}
		  ,
                    {short = "",
                     long = ["cm"],
                     desc = GetOpt.ReqArg (setUseOption useCommutativeOption "cm","0|1|no|yes"),
                     help = "cr: use commutative decomposition [default:yes]"}
		  ,
                    {short = "",
                     long = ["nm"],
                     desc = GetOpt.ReqArg (setUseOption useNonMinimalCommutativeOption 
							"nm","0|1|no|yes"),
                     help = "cr: with non-minimal commutative decomposition [default:no]"}
		  ,
                    {short = "r",
                     long = ["reset"],
                     desc = GetOpt.NoArg (fn () => setResetOption ()),
                     help = "turn off all criteria by default (use before enable options)"}
		  ,
                    {short = "",
                     long = ["enable-KB"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useKnuthBendix := true; Proceed)),
                     help = "cr: (Knuth&Bendix,1970)"}
		  ,
                    {short = "",
                     (* long = ["enable-strong"] 2014/4/14 overlap with strongly-quasi-linear is harmful *)
                     long = ["enable-strongly-closed"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useHuetStrongClosed := true; Proceed)),
                     help = "cr: (Huet,1980) strongly closed"}
		  ,
                    {short = "",
                     long = ["enable-modulo"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useHuetModulo := true; Proceed)),
                     help = "cr: (Huet,1980) Church-Rosser modulo"}
		  ,
                    {short = "",
                     long = ["enable-parallel"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useToyamaParallel := true; Proceed)),
                     help = "cr: (Toyama,1981) parallel critical pairs"}
		  ,
                    {short = "",
                     long = ["enable-simple"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useOhtaOyamaguchiToyama := true; Proceed)),
                     help = "cr: (Ohta&Oyamaguchi&Toyama,1995) simple-right-linear"}
		  ,
                    {short = "",
                     long = ["enable-development"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useHuetToyamaOostrom := true; 
						    CrSolver.useYoshidaAotoToyama := true;
						    Proceed)),
                     help = "cr: (Huet,1980)-(Toyama,1988)-(van Oostrom,1997) development closed\ncom: (Toyama,1988)-(Yoshida&Aoto&Toyama,2009) development closed"}
		  ,
                    {short = "",
                     long = ["enable-simultaneous"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useOkuiSimultaneous := true; Proceed)),
                     help = "cr: (Okui,1998) simultaneous critical pairs"}
		  ,
                    {short = "",
                     long = ["enable-depth"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useGomiOyamaguchiOhtaDepth := true; Proceed)),
                     help = "cr: (Gomi&Oyamaguchi&Ohta,1996/1998) strongly depth-preserving"}
		  ,
                    {short = "",
(*                     long = ["enable-weight"], --- fix 2018/11/4  option which is a prefix of other option does not work properly *)
                     long = ["enable-strongly-weight-preserving"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useGomiOyamaguchiOhtaWeight := true; Proceed)),
                     help = "cr: (Gomi&Oyamaguchi&Ohta,1998) strongly weight-preserving"}
		  ,
                    {short = "",
                     long = ["enable-upside"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useOyamaguchiOhta := true; Proceed)),
                     help = "cr: (Oyamaguchi&Ohta,2004) upside-parallel-closed and outside-closed "}
		  ,
                    {short = "",
                     long = ["enable-decreasing"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useDecreasingDiagrams := true; Proceed)),
                     help = "cr: (van Oostrom,2008)-(Aoto,2010) decreasing diagram with rule-labelling"}
		  ,
                    {short = "",
                     long = ["enable-shallow"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useSakaiOgawa := true; Proceed)),
                     help = "cr: (Sakai&Ogawa,2010) weakly-non-overlapping non-collapsing and shallow"}
		  ,
                    {short = "",
                     long = ["enable-completion"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useBidirectionalCompletion := true; Proceed)),
                     help = "cr: (Aoto&Toyama,2012) reduction-preserving completion"}
		  ,
                    {short = "",
                     long = ["enable-quasi-left-linear"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useSuzukiAotoToyama := true; Proceed)),
                     help = "cr: (Suzuki&Aoto&Toyama,2013) quasi-left-linear"}
		  ,
                    {short = "",
                     long = ["enable-quasi-linear"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useUchidaAotoToyama := true; Proceed)),
                     help = "cr: (Aoto&Toyama&Uchida,2014) quasi-linear"}
		  ,
                    {short = "",
                     long = ["enable-strongly-quasi-linear"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useUchidaAotoToyamaStrong := true; Proceed)),
                     help = "cr: (Aoto&Toyama&Uchida,2014) strongly quasi-linear"}
		  ,
                    {short = "",
                     long = ["enable-direct-approximation"],
                     desc = GetOpt.NoArg (fn () => (NonCr.useNonJoinableByApproximation := true; Proceed)),
                     help = "cr/com: disproving by direct approximations"}
		  ,
                    {short = "",
                     long = ["enable-tree-automata"],
                     desc = GetOpt.NoArg (fn () => (NonCr.useNonJoinableByTreeAutomata := true; Proceed)),
                     help = "cr/com: (Jacquemard,1996)-(Durand&Middeldorp,1997) disproving by tree-automata approximation"}
		  ,
                    {short = "",
                     long = ["enable-interpretation-and-order"],
                     desc = GetOpt.NoArg (fn () => (NonCr.useNonJoinableByInterpretationAndOrder := true; Proceed)),
                     help = "cr/com: (Aoto,2013) disproving by interpretation and ordering"}
		  ,
                    {short = "",
                     long = ["enable-ordered-rewriting"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useMartinNipkowOrderedRewriting := true; Proceed)),
                     help = "cr: (Martin&Nipkow,1990) confluence by ordered rewriting"}
		  ,
                    {short = "",
                     long = ["enable-non-omega-overlapping"],
                     desc = GetOpt.NoArg (fn () => (Unc.useNonOmegaOverlapping := true; Proceed)),
                     help = "unc: (Chew,1981)-(Mano&Ogawa,1997)-(Kahrs&Smith,2016) non-omega-overlapping"}
		  ,
                    {short = "",
                     long = ["enable-strongly-non-overlapping"],
                     desc = GetOpt.NoArg (fn () => (Unc.useStrongNonOverlapping := true; Proceed)),
                     help = "unc: (Klop&de Vrijer,1990)-(de Vrijer,1999) strongly-non-overlapping"}
		  ,
                    {short = "",
(*                     long = ["enable-parallel-closed-linearization"], --- fix 2018/11/4  option which is a prefix of other option does not work properly *)
                    long = ["enable-linearization-parallel-closed"],
                     desc = GetOpt.NoArg (fn () => (Unc.useParallelClosedLinearization := true; Proceed)),
                     help = "unc: (Aoto&Toyama,2018) parallel closed linearization"}
		  ,
                    {short = "",
(*                     long = ["enable-strongly-closed-linearization"], --- fix 2018/11/4  option which is a prefix of other option does not work properly *)
                     long = ["enable-linearization-strongly-closed"],
                     desc = GetOpt.NoArg (fn () => (Unc.useStronglyClosedLinearization := true; Proceed)),
                     help = "unc: (Aoto&Toyama,2018) strongly closed linearization"}
		  ,
                    {short = "",
                     long = ["enable-weight-decreasing-joinable"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useToyamaOyamaguchiWeightDecreasing := true;
		  				    Unc.useWeightDecreasingJoinable := true; Proceed)),
                     help = "cr/unc: (Toyama&Oyamaguchi,2001) weight-decreasing joinable"}
                  ,
		    {short = "",
                     long = ["enable-right-reducible"],
                     desc = GetOpt.NoArg (fn () => (Unc.useRightReducible := true;
		  				    Proceed)),
                     help = "unc: (Aoto&Toyama,2018) right-reducible"}
                  , 
		    {short = "",
                     long = ["enable-rule-reversing"],
                     desc = GetOpt.NoArg (fn () => (Unc.useRuleReversing := true;
		  				    Proceed)),
                     help = "unc: (Aoto&Toyama,2018) perform rule reversing transformation"}
                  ,
		    {short = "",
                     long = ["enable-reverse-critical-pairs"],
                     desc = GetOpt.NoArg (fn () => (Unc.useNonUncByCps := true;
		  				    Proceed)),
                     help = "unc: check non-UNC via critical pairs closure"}
                  ,
		    {short = "",
                     long = ["enable-unc-completion"],
                     desc = GetOpt.OptArg (fn opt => case opt of
		  					 SOME "s1" => (Unc.useUncCompletionByStronglyClosed := true;
		  						       Unc.maxCPclosingSteps := 1;
		  						       Proceed)
		  				       | SOME "s2" => (Unc.useUncCompletionByStronglyClosed := true;
		  						       Unc.maxCPclosingSteps := 2;
		  						       Proceed)
		  				       | SOME "s3" => (Unc.useUncCompletionByStronglyClosed := true;
		  						       Unc.maxCPclosingSteps := 3;
		  						       Proceed)
		  				       | SOME "d1" => (Unc.useUncCompletionByDevelopmentClosed := true;
		  						       Unc.maxCPclosingSteps := 1;
		  						    Proceed)
		  				       | SOME "d2" => (Unc.useUncCompletionByDevelopmentClosed := true;
		  						       Unc.maxCPclosingSteps := 2;
		  						    Proceed)
		  				       | SOME "d3" => (Unc.useUncCompletionByDevelopmentClosed := true;
		  						       Unc.maxCPclosingSteps := 3;
		  						    Proceed)
		  				       | SOME "0" => (Unc.useUncCompletionByStronglyClosed := true;
		  						      Unc.maxCPclosingSteps := 3;
		  						    Proceed)
		  				       | SOME "1" => (Unc.useUncCompletionByDevelopmentClosed := true;
		  						      Unc.maxCPclosingSteps := 3;
		  						    Proceed)
		  				       | SOME "2" => (Unc.useGeneralUncCompletion := true;
		  						      Unc.maxCPclosingSteps := 3;
		  						    Proceed)
		  				       | _ => (Unc.useUncCompletionByStronglyClosed := true;
		  						  Unc.useUncCompletionByDevelopmentClosed := true;
		  						  Unc.useGeneralUncCompletion := true;
		  						  Proceed),
		  			   "0|1|2"),
                     help = "unc: (Aoto&Toyama, 2018) UNC completion"}
		  ,
                    {short = "",
                     long = ["enable-non-commutation"],
                     desc = GetOpt.NoArg (fn () => (CrSolver.useNonCommutation := true; Proceed)),
                     help = "com: disproving commutation by searching a couterexample"}
		  ,
		    {short = "",
                     long = ["enable-decproc-shallow"],
                     desc = GetOpt.NoArg (fn () => (Unc.useShallowDecProc := true;
		  				    Proceed)),
                     help = "unc: (Radcliffe&Moreas&Verma, 2017)-(Yamaguchi&Aoto, 2019) decision procedure for shallow TRSs"}
                  ]

		    
    (* val optionsFoSSaCS2020 = [{short = "h", *)
    (*                 long = ["help"], *)
    (*                 desc = GetOpt.NoArg (fn () => HelpOption), *)
    (*                 help = "print this message"} *)
    (* 			     ] *)

    exception ReadFileError

    fun checkFile (path,filename) =
        let 
            fun main () = 
                let 
		    val _ = case (!targetProperty) of
				COM => (useDirectOption := false;
					useCommutativeOption := false;
					useNonMinimalCommutativeOption := false;
					useLayerOption := false)
			      | _ => ()
                    (* val _ = useDirectOption := !useDirectOption (* what this code ?? *) removed 2019/03/03 *)
                    (* val _ = useCommutativeOption := !useCommutativeOption (* what this code ?? *) removed 2019/03/03 *)
                    val _ = useNonMinimalCommutativeOption := (!useCommutativeOption andalso 
							       !useNonMinimalCommutativeOption)
                    (* val _ = useLayerOption := !useLayerOption (* what this code ?? *) removed 2019/03/03 *)

                    val ((crules,ss),(crules2,ss2),sigs) = (*** keep this, but not included to ACP
                                 if (!useManySortedInputOption)
				 then
				     let val (decls,rs,_) = case IOMsTpdb.rdFile filename of
								SOME ans => ans
							      | NONE => (print "Error: reading input\n";  raise Abort)
				     in (rs,[])
				     end
				 else ***)
				     (case IOTpdb.rdFile filename of
					  (SOME ans, SOME ans2, sigs ) => (ans, ans2,sigs)
				       | (SOME ans, NONE,sigs ) => (ans,([],[]),sigs)
				       | (NONE, _, _ ) => raise Unsupported)
				     handle IOTpdb.ReadFileError => raise Abort

		    val _ = if not (null ss)
			    then (PU.printlnStdErr ("Error: relative rules are found\n");
				  raise Abort)
			    else ()
                    val opt0 = {satSolver = (!minisatPath),
                                smtSolver = (!yicesPath),
                                terminationProver = (!terminationProver),
                                relativeTerminationProver = (!relativeTerminationProver),
                                tmpDir = (!tmpDir) }
                    val opt1 = DpSolver.defaultOptions
                    val opt2 = PoSolver.defaultOptions
                    val opt3 = {useDirect=(!useDirectOption),
                                useCommutative=(!useCommutativeOption),
                                useNonMinimalCommutative=(!useNonMinimalCommutativeOption),
                                useLayer=(!useLayerOption),
				useCertifiableOutputOption=(!useCertifiableOutputOption)}
		    val _ = CertifyUtil.outputFile := (!certifiableOutputFile) (* need this *)
                    val _ = debug (fn _ => PU.print ("\n"))

                    val _ = debug (fn _ => 
                                    let val mesg = 
                                             "Start Checking " 
                                              ^ (if path = "" 
                                                 then filename 
                                                 else (path ^ "/" ^ filename))
                                    in
                             PU.print (mesg ^ "\n");
                             PU.print (implode (List.tabulate (size mesg, fn x => #"=")));
                             PU.print "\n"
                                      end)

                    val _ = debug (fn _ => PU.print ("Persistent Decomposition: " ^ (if !useDirectOption 
                                                              then "on\n" else "off\n"))) 
                    val _ = debug (fn _ => PU.print ("Commutative Decompositoin: " ^ (if !useCommutativeOption 
                                                             then "on\n" else "off\n"))) 
                    val _ = debug (fn _ => PU.print ("Non-min Commutative Decomposition : " 
						     ^ (if !useNonMinimalCommutativeOption 
                                                             then "on\n" else "off\n"))) 
                    val _ = debug (fn _ => PU.print ("Layer-Preserving Decomposition: " ^ (if !useLayerOption 
                                                               then "on\n" else "off\n"))) 

                    val _ = debug (fn _ => PU.print ("Output Certifierable Proof Trees: " 
						     ^ (if !useCertifiableOutputOption 
                                                        then "on\n" else "off\n")))

                    val _ = debug (fn _ => PU.print ("Input of Many-Sorted TRS: " 
						     ^ (if !useManySortedInputOption
                                                        then "on\n" else "off\n")))

                    (* val result  = CrSolver.crSolver  opt0 opt1 opt2 opt3 rs *)

		    val isConditional = List.exists (fn (l,r,c) => not (null c)) crules
					orelse
					List.exists (fn (l,r,c) => not (null c)) crules2

		    val rules = List.map (fn (l,r,c) => (l,r)) crules
		    val rules2 = List.map (fn (l,r,c) => (l,r)) crules2
		in case (!targetProperty) of
		       CR => let val result  = if isConditional
					       then CrSolver.condCrSolver opt0 opt1 opt2 opt3 crules
					       else CrSolver.crSolver opt0 opt1 opt2 opt3 rules
				 val _ = PU.print (if path = "" then filename else (path ^ "/" ^ filename))
			     in case result of
				    Cr.CR => (numOfSuccess := !numOfSuccess + 1; 
					      PU.println ": Success(CR)";
					      PU.printlnStdOut "YES")
				  | Cr.NotCR => (numOfSuccess := !numOfSuccess + 1; 
						 PU.println ": Success(not CR)";
						 PU.printlnStdOut "NO")
				  | Cr.Unknown => (numOfFailure := !numOfFailure + 1; 
						PU.println ": Failure(unknown CR)";
						PU.printlnStdOut "MAYBE")
			     end
		     | UNC => let val result  = if isConditional
						then (PU.printlnStdErr ("Error: UNC proving for CTRSs is not supported\n");
						      raise Abort)
						else Unc.uncSolverWithDecomposition opt0 opt1 opt2 opt3 rules
				  val _ = PU.print (if path = "" then filename else (path ^ "/" ^ filename))
			     in case result of
				    Unc.UNC => (numOfSuccess := !numOfSuccess + 1; 
					      PU.println ": Success(UNC)";
					      PU.printlnStdOut "YES")
				  | Unc.NotUNC => (numOfSuccess := !numOfSuccess + 1; 
						 PU.println ": Success(not UNC)";
						 PU.printlnStdOut "NO")
				  | Unc.Unknown => (numOfFailure := !numOfFailure + 1; 
						PU.println ": Failure(unknown UNC)";
						PU.printlnStdOut "MAYBE")
			      end
		     | UNR => let val result  = if isConditional
						then (PU.printlnStdErr ("Error: UNR proving for CTRSs is not supported\n");
						      raise Abort)
						else Unr.unrSolver opt0 opt1 opt2 opt3 sigs rules
				  val _ = PU.print (if path = "" then filename else (path ^ "/" ^ filename))
			     in case result of
				    Unr.UNR => (numOfSuccess := !numOfSuccess + 1; 
					      PU.println ": Success(UNR)";
					      PU.printlnStdOut "YES")
				  | Unr.NotUNR => (numOfSuccess := !numOfSuccess + 1; 
						 PU.println ": Success(not UNR)";
						 PU.printlnStdOut "NO")
				  | Unr.Unknown => (numOfFailure := !numOfFailure + 1; 
						PU.println ": Failure(unknown UNR)";
						PU.printlnStdOut "MAYBE")
			      end

		     | COM => let val result  = if isConditional
						then (PU.printlnStdErr ("Error: COM proving for CTRSs is not supported\n");
						      raise Abort)
						else CrSolver.comSolver opt0 opt1 opt2 opt3 (rules,rules2)
				  val _ = PU.print (if path = "" then filename else (path ^ "/" ^ filename))
			      in case result of
				     Cr.COM => (numOfSuccess := !numOfSuccess + 1; 
						 PU.println ": Success(COM)";
						 PU.printlnStdOut "YES")
				   | Cr.NotCOM => (numOfSuccess := !numOfSuccess + 1; 
						    PU.println ": Success(not COM)";
						    PU.printlnStdOut "NO")
				   | Cr.UnknownCOM => (numOfFailure := !numOfFailure + 1; 
						 PU.println ": Failure(unknown COM)";
						 PU.printlnStdOut "MAYBE")
			      end
		end
            val timer = Timer.startCPUTimer ()
            val start = TimeUtil.checkTime timer 
            (*         val result = main ()   *)
            val result = TimeUtil.timeLimitRun (IntInf.fromInt (!timeLimit)) main () 
                handle TimeLimit.TimeOut => 
                       (if OS.FileSys.access (!Solver.inTmpFileName, [OS.FileSys.A_WRITE])
                        then (OS.FileSys.remove (!Solver.inTmpFileName);
                              Solver.inTmpFileName := "")
                        else ();
                        if OS.FileSys.access (!Solver.outTmpFileName, [OS.FileSys.A_WRITE])    
                        then (OS.FileSys.remove (!Solver.outTmpFileName);
                              Solver.outTmpFileName := "")
                        else ();
                        PU.print (if path = "" then filename else (path ^ "/" ^ filename));
                        numOfFailure := !numOfFailure + 1; 
                        PU.println ": Failure(timeout)";
			PU.printlnStdOut "MAYBE")
		     | ReadFileError =>
		       (PU.print (if path = "" then filename else (path ^ "/" ^ filename));
			numOfFailure := !numOfFailure + 1; 
			PU.println ": Failure(input file error)";
			PU.printlnStdOut "MAYBE")
		     | Unsupported => 
		       (PU.print (if path = "" then filename else (path ^ "/" ^ filename));
			 numOfFailure := !numOfFailure + 1;
			 PU.println ": Failure(unsupported)";
			 PU.printlnStdOut "UNSUPPORTED")

            val stop = TimeUtil.checkTime timer 
            val _ = PU.print ("("  ^ (TimeUtil.reportTime (start,stop)) ^ " msec.)\n")
        in
            result
        end


(*     fun checkFileFoSSaCS2020 (path,filename) = *)
(*         let *)
(*             fun main () = *)
(*                 let *)
(*     		    val _ = targetProperty := UNC *)
(*     		    val _ = setDirectOption () *)
(*                     val ((crules,ss),(crules2,ss2)) = *)
(*     			(case IOTpdb.rdFile filename of *)
(*     			     (SOME ans, SOME ans2) => (ans, ans2) *)
(*     			   | (SOME ans, NONE) => (ans,([],[])) *)
(*     			   | (NONE, _) => raise Unsupported) *)
(*     			handle IOTpdb.ReadFileError => raise Abort *)
(*     		    val _ = if not (null ss) *)
(*     			    then (PU.printlnStdErr ("Error: relative rules are found\n"); *)
(*     				  raise Abort) *)
(*     			    else () *)
(*     		    val isConditional = List.exists (fn (l,r,c) => not (null c)) crules *)
(*     					orelse *)
(*     					List.exists (fn (l,r,c) => not (null c)) crules2 *)
(*     		    val rules = List.map (fn (l,r,c) => (l,r)) crules *)
(*     		    val rules2 = List.map (fn (l,r,c) => (l,r)) crules2 *)
(*     		in let val result  = if isConditional *)
(*     				     then (PU.printlnStdErr ("Error: UNC proving for CTRSs is not supported\n"); *)
(*     					   raise Abort) *)
(*     				     else if Trs.areShallowRules rules *)
(* (*				     then (print "shallow:"; print filename; Unc.Unknown) *) *)
(* (* 				     else (print "non-shallow:"; Unc.Unknown) *) *)
(*     				     then let val _ = *)
(* 				     debug (fn ()=> PU.println "Call a decision procedure for shallow TRSs") *)
(*     					  in if Unc.runDecProcForShallow rules *)
(*     					     then Unc.UNC *)
(*     					     else Unc.NotUNC *)
(*     					  end *)
(*     				     else (PU.printlnStdErr ("Error: not a shallow TRS\n"); *)
(*     					   raise Abort) *)
(*     		       val _ = PU.print (if path = "" then filename else (path ^ "/" ^ filename)) *)
(*     		   in case result of *)
(*     			  Unc.UNC => (numOfSuccess := !numOfSuccess + 1; *)
(*     				      PU.println ": Success(UNC)"; *)
(*     				      PU.printlnStdOut "YES") *)
(*     			| Unc.NotUNC => (numOfSuccess := !numOfSuccess + 1; *)
(*     					 PU.println ": Success(not UNC)"; *)
(*     					 PU.printlnStdOut "NO") *)
(*     			| Unc.Unknown => (numOfFailure := !numOfFailure + 1; *)
(*     				      PU.println ": Failure(unknown UNC)"; *)
(*     				      PU.printlnStdOut "MAYBE") *)
(*     		   end *)
(*     		end *)
(*             val result = TimeUtil.timeLimitRun (IntInf.fromInt (!timeLimit)) main () *)
(*                 handle TimeLimit.TimeOut => *)
(*                        (if OS.FileSys.access (!Solver.inTmpFileName, [OS.FileSys.A_WRITE]) *)
(*                         then (OS.FileSys.remove (!Solver.inTmpFileName); *)
(*                               Solver.inTmpFileName := "") *)
(*                         else (); *)
(*                         if OS.FileSys.access (!Solver.outTmpFileName, [OS.FileSys.A_WRITE]) *)
(*                         then (OS.FileSys.remove (!Solver.outTmpFileName); *)
(*                               Solver.outTmpFileName := "") *)
(*                         else (); *)
(*                         PU.print (if path = "" then filename else (path ^ "/" ^ filename)); *)
(*                         numOfFailure := !numOfFailure + 1; *)
(*                         PU.println ": Failure(timeout)"; *)
(*     			PU.printlnStdOut "MAYBE") *)
(*     		     | ReadFileError => *)
(*     		       (PU.print (if path = "" then filename else (path ^ "/" ^ filename)); *)
(*     			numOfFailure := !numOfFailure + 1; *)
(*     			PU.println ": Failure(input file error)"; *)
(*     			PU.printlnStdOut "MAYBE") *)
(*     		     | Unsupported => *)
(*     		       (PU.print (if path = "" then filename else (path ^ "/" ^ filename)); *)
(*     			 numOfFailure := !numOfFailure + 1; *)
(*     			 PU.println ": Failure(unsupported)"; *)
(*     			 PU.printlnStdOut "UNSUPPORTED") *)
(*         in *)
(*             result *)
(*         end *)

    fun reportTotal timer = 
        let val _ = PU.print "\n=============================\n"
            val _ = PU.print ("Total: " 
                           ^ (Int.toString (!numOfSuccess)) ^ " Success, " 
                           ^ (Int.toString (!numOfFailure)) ^ " Failure: ")
            val _ = PU.print (Int.toString (LargeInt.toInt (TimeUtil.checkTime timer)) 
                           ^ " millisec.\n")
        in () end

in
fun main (name:string, args:string list) =
    let  
        val timer = Timer.startCPUTimer()
        val start = TimeUtil.checkTime timer 
        val commandDir = OS.FileSys.getDir();
	val _ = tmpDir := commandDir
        val _ = setInit ();

        val targets = 
            case (GetOpt.getOpt {argOrder = GetOpt.Permute,
                                 options = options,
                                 errFn = (fn str => (PU.printlnStdErr "option error\n"; 
						     PU.printlnStdErr usage; 
						     raise Abort))}
                                args) of
                (xs,args) =>  if List.exists (fn x => x = HelpOption) xs
                              then (PU.printlnStdErr (GetOpt.usageInfo 
                                                           {header=("acp v" ^ version),
                                                            options=options});  
                                    raise Abort)
			      else if not (List.all (fn x => x = Proceed) xs)
			      then (printOptionError xs; PU.printlnStdErr usage; raise Abort)
                              else case args of 
                                       [] =>(PU.printlnStdErr "file or directory not specified";
                                            PU.printlnStdErr usage; raise Abort)
                                     | xs => (List.map 
                                                 (fn x => OS.FileSys.realPath x
                                                     handle OS.SysErr (msg,_) => 
                                                            (PU.printlnStdErr ("Error: file or directory not found: " 
                                                                                ^ x);
							     PU.printlnStdErr usage;
                                                             raise Abort))
                                                 xs)

	val _ = debug (fn _ => PU.print ("commandDir: " ^ commandDir ^ "\n"))
	val _ = debug (fn _ => PU.print ("tmpDir: " ^ (!tmpDir) ^ "\n"))


        val _ = if ("" = !minisatPath) 
                then let val path' = OS.FileSys.fullPath "./minisat"
                                     handle OS.SysErr (msg,_) => 
                                            (PU.printlnStdErr ("Error: ./minisat not found; " 
                                                               ^ "use --minisat-path to specify minisat-path");
					     PU.printlnStdErr usage;
                                             raise Abort)
                     in if OS.FileSys.access (path', [OS.FileSys.A_EXEC])
                        then minisatPath := path'
                        else (PU.printlnStdErr ("Error: ./minisat not executable; "
                                                ^ "use --minisat-path to specify minisat-path");
			      PU.printlnStdErr usage;
                              raise Abort)
                     end
                else ()

        val _ = if ("" = !yicesPath) 
                then let val path' = OS.FileSys.fullPath "./yices"
                             handle OS.SysErr (msg,_) => 
                                    (PU.printlnStdErr ("Error: ./yices not found; " 
                                                        ^ "use --yices-path to specify yices-path");
				     PU.printlnStdErr usage;
                                     raise Abort)
                     in if OS.FileSys.access (path', [OS.FileSys.A_EXEC])
                        then yicesPath := path'
                        else (PU.printlnStdErr ("Error: ./yices not executable; "
                                                 ^ "use --yices-path to specify yices-path");
			      PU.printlnStdErr usage;
                              raise Abort)
                     end
                else ()

	val _ = if (!useCertifiableOutputOption)
		   andalso ((!useDirectOption) 
			    orelse (!useCommutativeOption)
			    orelse (!useLayerOption))
		then (PU.printlnStdErr ("Error: currently, certification and decomposition techniques can not combied;");
		      PU.printlnStdErr ("       specify -d and --certifiable-proofs-output together"); 
		      raise Abort)
		else ()


(**        val _ = if (!CrSolver.useHuetModulo)
                    andalso ("" = !relativeTerminationProver) 
                then 
                    (PU.println ("relative termination prover not specified; "
                                        ^ "disable some options or specify one using --relative-termination-prover");
                     raise Abort)
                else ()
**)

	val logFile = if ((!verboseMode) = 0)
		      then OS.Path.joinDirFile {dir="/dev",file="null"} 
		      else if ((!verboseMode) = 1)
		      then
			  let val tmpName = 
			      #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
			  val _ = debug (fn () => print ("logfile: " ^ tmpName ^ "\n"))
			  in OS.Path.joinDirFile {dir=(!tmpDir),file=tmpName}
			  end
		      else ""

	val saveOut = ref (!PU.outs)
	val _ = if (!runDebug) orelse (!verboseMode) >= 2
		then PU.outs := TextIO.stdOut
		else PU.outs := TextIO.openOut logFile

	fun readIns ins = 
	    if TextIO.endOfStream ins 
	    then ()
	    else case TextIO.inputLine ins of
		     SOME line => (PU.printStdOut line; readIns ins)
		   | NONE => readIns ins

	fun finish () =  
	    if (!verboseMode = 1)
	    then let
		    val _ = TextIO.closeOut (!PU.outs)
		    val ins = TextIO.openIn logFile
		    val _ = readIns ins
		    val _ = TextIO.closeIn ins; 
		    val _ = PU.outs := (!saveOut)
		    val _ = if !runDebug then () else OS.FileSys.remove logFile
		in ()
		end
	    else (PU.outs := (!saveOut); ())

    in
        if length targets = 1 
           andalso not (OS.FileSys.isDir (hd targets))
        then 
            (CommandUtil.applyCmdToDirTree checkFile ("", hd targets);
	     finish ();
             setInit ();
             OS.Process.success)
        else
            (List.app (fn item => 
                         CommandUtil.applyCmdToDirTree checkFile ("", item))
                      targets;
             reportTotal timer;
 	        finish ();
             setInit ();
             OS.Process.success)
            handle e  => (OS.FileSys.chDir commandDir; finish (); raise (e))
    end
    handle Abort => OS.Process.failure

(* fun mainFoSSaCS2020 (name:string, args:string list) = *)
(*     let *)
(*         val timer = Timer.startCPUTimer() *)
(*         val start = TimeUtil.checkTime timer *)
(*         val commandDir = OS.FileSys.getDir(); *)
(* 	val _ = tmpDir := commandDir *)
(*         val _ = setResetOption (); *)
(*         val targets = *)
(*             case (GetOpt.getOpt {argOrder = GetOpt.Permute, *)
(*                                  options = optionsFoSSaCS2020, *)
(*                                  errFn = (fn str => (PU.printlnStdErr "option error\n"; *)
(* 						     PU.printlnStdErr usageFoSSaCS2020; *)
(* 						     raise Abort))} *)
(*                                 args) of *)
(*                 (xs,args) =>  if List.exists (fn x => x = HelpOption) xs *)
(*                               then (PU.printlnStdErr (GetOpt.usageInfo *)
(*                                                           {header=("acp v" ^ version ^ ", specialized for FoSSaCS 2020"), *)
(*                                                            options=optionsFoSSaCS2020}); *)
(*                                     raise Abort) *)
(* 			      else if not (List.all (fn x => x = Proceed) xs) *)
(* 			      then (printOptionError xs; PU.printlnStdErr usage; raise Abort) *)
(*                               else case args of *)
(*                                        [] =>(PU.printlnStdErr "file or directory not specified"; *)
(*                                             PU.printlnStdErr usage; raise Abort) *)
(*                                      | xs => (List.map *)
(*                                                  (fn x => OS.FileSys.realPath x *)
(*                                                      handle OS.SysErr (msg,_) => *)
(*                                                             (PU.printlnStdErr ("Error: file or directory not found: " *)
(*                                                                                 ^ x); *)
(* 							     PU.printlnStdErr usage; *)
(*                                                              raise Abort)) *)
(*                                                  xs) *)

(* 	val _ = debug (fn _ => PU.print ("commandDir: " ^ commandDir ^ "\n")) *)
(* 	val _ = debug (fn _ => PU.print ("tmpDir: " ^ (!tmpDir) ^ "\n")) *)

(* 	val logFile = if ((!verboseMode) = 0) *)
(* 		      then OS.Path.joinDirFile {dir="/dev",file="null"} *)
(* 		      else if ((!verboseMode) = 1) *)
(* 		      then *)
(* 			  let val tmpName = *)
(* 			      #file (OS.Path.splitDirFile (OS.FileSys.tmpName ())) *)
(* 			  val _ = debug (fn () => print ("logfile: " ^ tmpName ^ "\n")) *)
(* 			  in OS.Path.joinDirFile {dir=(!tmpDir),file=tmpName} *)
(* 			  end *)
(* 		      else "" *)

(* 	val saveOut = ref (!PU.outs) *)
(* 	val _ = if (!runDebug) orelse (!verboseMode) >= 2 *)
(* 		then PU.outs := TextIO.stdOut *)
(* 		else PU.outs := TextIO.openOut logFile *)

(* 	fun readIns ins = *)
(* 	    if TextIO.endOfStream ins *)
(* 	    then () *)
(* 	    else case TextIO.inputLine ins of *)
(* 		     SOME line => (PU.printStdOut line; readIns ins) *)
(* 		   | NONE => readIns ins *)

(* 	fun finish () = *)
(* 	    if (!verboseMode = 1) *)
(* 	    then let *)
(* 		    val _ = TextIO.closeOut (!PU.outs) *)
(* 		    val ins = TextIO.openIn logFile *)
(* 		    val _ = readIns ins *)
(* 		    val _ = TextIO.closeIn ins; *)
(* 		    val _ = PU.outs := (!saveOut) *)
(* 		    val _ = if !runDebug then () else OS.FileSys.remove logFile *)
(* 		in () *)
(* 		end *)
(* 	    else (PU.outs := (!saveOut); ()) *)

(*     in *)
(*         if length targets = 1 *)
(*            andalso not (OS.FileSys.isDir (hd targets)) *)
(*         then *)
(*             (CommandUtil.applyCmdToDirTree checkFileFoSSaCS2020 ("", hd targets); *)
(* 	     finish (); *)
(*              setInit (); *)
(*              OS.Process.success) *)
(*         else *)
(*             (List.app (fn item => *)
(*                          CommandUtil.applyCmdToDirTree checkFileFoSSaCS2020 ("", item)) *)
(*                       targets; *)
(*              reportTotal timer; *)
(*  	        finish (); *)
(*              setInit (); *)
(*              OS.Process.success) *)
(*             handle e  => (OS.FileSys.chDir commandDir; finish (); raise (e)) *)
(*     end *)
(*     handle Abort => OS.Process.failure *)


end

fun mkCommand () = SMLofNJ.exportFn("../../work/acp/acp",main);
(** fun mkCommand () = SMLofNJ.exportFn("../../work/acp/acp-fossacs2020",mainFoSSaCS2020); **)
fun install () = SMLofNJ.exportFn("work/acp/acp",main);


(*
fun main2 1  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 2  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 3  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 4  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 5  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 6  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 7  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 8  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := false
in main (name,args)end

| main2 9  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2 10  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2 11  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2 12  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2 13  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2 14  (name:string, args:string list) = let 
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2  15  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true      
in main (name,args)end

| main2  16  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true      
in main (name,args)end

| main2  17  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true      
in main (name,args)end

| main2  18  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true      
in main (name,args)end

| main2  19  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true      
in main (name,args)end

| main2  20  (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := false
    val _ = CrCompletion.useCompletion := true      
in main (name,args)end

| main2  21 (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := true
    val _ = CrCompletion.useForward := true
    val _ = CrCompletion.useLinear := true
    val _ = CrCompletion.useParallel := true
    val _ = CrCompletion.useRelative := true
    val _ = CrCompletion.useHuet := true
    val _ = CrCompletion.useCompletion := true
in main (name,args)end

| main2  _ (name:string, args:string list) = let
    val _ = CrCompletion.usePCP := false
    val _ = CrCompletion.useForward := false
    val _ = CrCompletion.useLinear := false
    val _ = CrCompletion.useParallel := false
    val _ = CrCompletion.useRelative := false
    val _ = CrCompletion.useHuet := true
    val _ = CrCompletion.useCompletion := false
in main (name,args)end


fun mkCommand2 i  = SMLofNJ.exportFn("../../work/acp/acp",main2 i);

*)
end




