(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/commands/io_mstpdb.sml
 * description: reading a file of many-sorted TPDB format
 * author: AOTO Takahito
 * 
 ******************************************************************************)


signature IO_MSTPDB =  
  sig
      exception ReadFileError
(*      val rdFile: string -> (Term.decl list 
			     * (Term.term * Term.term) list 
			     * (Term.term * Term.term) list) option *)
      val rdFile: string -> (Term.decl list 
			     * (Term.term * Term.term) list 
			     * string option
			     * (Term.term * Term.term * (Term.term * Term.term) list) list) option
      val transToSigMs: string * string list -> OS.Process.status
(**
      val transFo2St: string * string list -> OS.Process.status
      val transFo2KBSeminar: string * string list -> OS.Process.status
      val transFo2Ms: string * string list -> OS.Process.status
**)
  end;

structure IOMsTpdb : IO_MSTPDB =  
  struct
  local
	  structure FIS = FunIntSet
	  structure FM = FunMap
	  structure L = List
	  structure LU = ListUtil
	  structure S = Sort
          structure PU = PrintUtil
  in

  exception ReadFileError

  (* ʸ:  *)
  structure MsTpdbLrVals = MsTpdbLrValsFun (structure Token = LrParser.Token);
  structure MsTpdbLex = MsTpdbLexFun (structure Tokens = MsTpdbLrVals.Tokens);
  structure MsTpdbParser = Join (structure ParserData = MsTpdbLrVals.ParserData;
                               structure Lex = MsTpdbLex;
                               structure LrParser = LrParser);

  (* ʸ: ȥ꡼ࢪʸڤؤѴ *)
  fun parserStream istream = 
	let val lexer = MsTpdbParser.makeLexer 
			    (fn n=>TextIO.inputN (istream, n));
	    val print_error = (fn (s, i, _) =>
	    			 (PU.printlnStdErr  (s^", line " ^ Int.toString(i) ^ "\n");
	    			  raise ReadFileError))
	in #1 (MsTpdbParser.parse (0, lexer, print_error, ()))
	end;

  fun rdFile filename = 
      let 
	  val ins = TextIO.openIn filename
	  val (sign,grules,ctypes) = parserStream ins
	  val _ = TextIO.closeIn ins

	  fun mkSort (ss,s) = Sort.Proc (ss,s)
	  fun mkDecls sign =  L.map (fn (f,ty) => {sym = f,sort=mkSort ty}) sign
	  val decls = mkDecls sign

	  val funs = L.map (fn (f,ty) => f) sign

	  fun transTerm (t as (Term.Fun (y,[],_))) = 
	      if ListUtil.member' Fun.equal y funs
	      then Term.Fun (y, [], Sort.null)
	      else Term.Var (Var.fromString (Fun.toString y), Sort.null)
	    | transTerm (Term.Fun (y,ys,_)) = Term.Fun (y, List.map transTerm ys, Sort.null)
	    | transTerm (Term.Var (y,_)) = Term.Var (y,Sort.null)

	  fun transRule (l,r) = (transTerm l, transTerm r)
	  fun transRules rs = L.map transRule rs
	  fun transCondRule (l,r,c) = (transTerm l, transTerm r, transRules c)
	  fun transCondRules rs = L.map transCondRule rs

	  val unsorted_rules = transCondRules grules

	  fun attachsort (l,r,[]) =
	      (case Trs.attachSortToRule decls (l,r) of
		   SOME (l',r') => SOME (l',r',[])
		 | NONE => (PrintUtil.printlnStdErr ("failed to attach sorts to rule: " ^ (Trs.prRule (l,r) ^ "\n"));
			    NONE))
	    | attachsort (l,r,c) =
	      (case Ctrs.attachSortToCondRule decls (l,r,c) of
		   SOME crule => SOME crule
		 | NONE => (PrintUtil.printlnStdErr ("failed to attach sorts to rule: " ^ (Ctrs.prRule (l,r,c) ^ "\n"));
			    NONE))

	  fun getConditionType [] = NONE
	    | getConditionType ([name]) = SOME name
	    | getConditionType names = (PrintUtil.printlnStdErr
					      ("multiple condition type declarations: "
					       ^ (LU.toStringComma (fn x=>x) names) ^ "\n");
					  NONE)

	  val conditionType = getConditionType ctypes

	  val rules = let val rop = L.map attachsort unsorted_rules
		      in if L.all isSome rop
			 then L.mapPartial (fn x=>x) rop
			 else raise ReadFileError
		      end
			  
	  val isConditional =
	      if L.exists (fn (l,r,c) => not (null c)) unsorted_rules
	      then case conditionType of 
		       SOME str => true
		     | NONE => (PrintUtil.printlnStdErr ("failed to get 'CONDITIONTYPE' declaration\n");
				raise ReadFileError)
	      else false

      in  if isConditional
	  then SOME (decls,[],conditionType,rules)
	  else SOME (decls,L.map (fn (l,r,c) => (l,r)) rules,NONE,[])
      end
      handle ReadFileError => (PU.printStdErr ("Skipped ... read file error\n"); NONE)
	   | MsTpdbParser.ParseError => (PU.printStdErr ("Skipped ... read file error\n"); NONE)

  (* my TPDB  MsTRS  agreed TPDB  MsTRS ؤѴ *)
  fun printInSigTPDB outs (decls,rules) =
      let
	  val lenall = L.map (fn decl => String.size (Fun.toString (#sym decl))) decls
	  val lenmax = L.foldr (fn (l,m) => Int.max (l,m)) 0 lenall
	  val _ = TextIO.output (outs, "(SIG\n")
	  val _ = List.app (fn decl
			       => TextIO.output (outs, 
						 ("   (" ^ 
						  Term.prDecl2 lenmax decl) ^ ")\n"))
			   decls
	  val _ = TextIO.output (outs, ")\n")
	  val _ = TextIO.output (outs, "(RULES\n")
	  val _ = List.app (fn (l,r) 
			       => TextIO.output (outs, 
						 ("   " ^ (Term.toStringWithoutQuestion l) ^ 
						  " -> " ^ (Term.toStringWithoutQuestion r)  ^ "\n")))
			   rules
	  val _ = TextIO.output (outs, ")\n")
      in
	  ()
      end

  exception Error
  fun checkFile prfun filename =
      case rdFile filename 
       of SOME (dcs,rs,ctype,crs) => (case ctype of
					 NONE => prfun TextIO.stdOut (dcs,rs)
				      | _ => (PU.printStdErr ("Skipped ... relative non-empty\n");
					      raise ReadFileError (* currently, incapable of relative rewriting *)))
	| NONE => raise ReadFileError


  exception Abort

  fun transCommand prfun (name:string, args:string list) =
      if length args = 1
      then (checkFile prfun (hd args); OS.Process.success)
      else OS.Process.failure

  val transToSigMs = transCommand printInSigTPDB

(* SMLofNJ.exportFn("../../work/agcp/transToSigMs",IOMsTpdb.transToSigMs) *)

  end (* of local *)

  end;
