(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/commands/io_tpdb.sml
 * description: reading a file of TPDB (Termination Problem Data Base) format
 * author: AOTO Takahito
 * 
 ******************************************************************************)


signature IO_TPDB =  
  sig
      exception ReadFileError
      val rdFile: string
		  -> (Ctrs.crules * Trs.rules) option
		     * (Ctrs.crules * Trs.rules) option
		     * ((Atom.atom * int) list) option
     
      (* disable to support CTRSs in 2018/01/25 *)
      (* val transFo2St: string * string list -> OS.Process.status *)
      (* val transFo2KBSeminar: string * string list -> OS.Process.status *)
      (* val transFo2Ms: string * string list -> OS.Process.status *)
  end;

structure IOTpdb : IO_TPDB =  
  struct
  local
	  structure FIS = FunIntSet
	  structure FM = FunMap
	  structure L = List
	  structure LU = ListUtil
	  structure S = Sort
          structure PU = PrintUtil
  in

  exception ReadFileError

  (* ʸ:  *)
  structure TpdbLrVals = TpdbLrValsFun (structure Token = LrParser.Token);
  structure TpdbLex = TpdbLexFun (structure Tokens = TpdbLrVals.Tokens);
  structure TpdbParser = Join (structure ParserData = TpdbLrVals.ParserData;
                               structure Lex = TpdbLex;
                               structure LrParser = LrParser);

  (* ʸ: ȥ꡼ࢪʸڤؤѴ *)
  fun parserStream istream = 
	let val lexer = TpdbParser.makeLexer 
			    (fn n=>TextIO.inputN (istream, n));
	    val print_error = fn (s, i, _) => 
				 (PU.print (s^", line " ^ Int.toString(i) ^ "\n");
				  raise ReadFileError)
	in #1 (TpdbParser.parse (0, lexer, print_error, ()))
	end;

  fun rdFile filename = 
      let val ins = TextIO.openIn filename
	  val result = parserStream ins
	  val _ = TextIO.closeIn ins
	  val (result1,result2,signitures) = case result of
				SOME ans => ans
			     |  NONE =>  raise ReadFileError
      in (result1,result2,signitures)
      end
      handle TpdbParser.ParseError => (PU.printlnStdErr ("Error: read file error"); raise ReadFileError)
	   | ReadFileError => (PU.printlnStdErr ("Error: read file error"); raise ReadFileError)

  fun replace s = String.implode 
		      (L.map (fn #"'"=> #"p"
			       | #"_" => #"b"
			       | #"#" => #"s"
			       | #"\"" => #"d"
			       | #"." => #"o"
			       | c=>c)
			     (String.explode s))

  fun connect [] = ""
    | connect (x::xs) = let val ys = (connect xs)
			in if String.size ys >= 1
			   then let val ys2 = String.extract (ys,1,NONE)
				    val c = hd (String.explode (String.substring (ys,0,1)))
				in x ^ (Char.toString (Char.toUpper c)) ^ ys2
				end
			   else x ^ ys
			end

  fun henkan "+" = "plus"
    | henkan "-" = "minus"
    | henkan "*" = "times"
    | henkan "\\" = "sub"
    | henkan "/" = "div"
    | henkan ">" = "gt"
    | henkan "<" = "lt"
    | henkan "<=" = "leq"
    | henkan ">=" = "geq"
    | henkan "=" = "eq"
    | henkan "==" = "equal"
    | henkan "--" = "mm"
    | henkan "__" = "floor"
    | henkan "||" = "par"
    | henkan "::" = "cons"
    | henkan ":" = "ap"
    | henkan "!" = "fact"
    | henkan "&" = "amp"
    | henkan "%" = "par"
    | henkan "[]" = "nil"
    | henkan ";" = "semi"
    | henkan "$" = "dollar"
    | henkan "@" = "app"
    | henkan "'" = "apply"
    | henkan "." = "conc"
    | henkan "++" = "concat"
    | henkan  s = connect (String.tokens (not o Char.isAlphaNum) (replace s))


  (* TPDB  FoTRS  KB-seminar  FoTRS ؤѴ *)
  local 
      open Term
  in
  fun printFoTRSinKBSeminarSpec outs rules =

      let 
	  fun fconv conv f = Fun.fromString (conv (Fun.toString f))
	  fun vconv conv x = Var.fromStringAndInt (conv (Var.name x), Var.index x)

	  fun convFirst ch s = let val xs = String.explode s
				 in case xs of 
				      [] => s
				    | (y::ys) => String.implode ((ch y)::ys)
			       end
	  fun prVarSym (x,_) = Atom.toString x

	  fun prTerm (Var (x,ty)) = prVarSym (vconv ((convFirst Char.toLower) o henkan) x)
	    | prTerm (Fun (f,[],ty)) = Fun.toString (fconv ((convFirst Char.toUpper) o henkan) f)
	    | prTerm (Fun (f,ts,ty)) = (Fun.toString (fconv ((convFirst Char.toUpper) o henkan) f))
				       ^ (prTermList ts)
	  and prTermList ts = LU.toStringCommaRound prTerm ts

	  fun prRule (l,r) = "\"" ^  (prTerm l) ^ " -> " ^ (prTerm r) ^ "\""
	  val _ = TextIO.output (outs, "val _ = test (IO.rdrules\n")
	  val _ = TextIO.output (outs, LU.toStringCommaLnSquare prRule rules)
	  val _ = TextIO.output (outs, "              )\n\n");
      in
	  ()
      end
  end

  (* TPDB  FoTRS  STSRS ؤѴ *)
  fun printFoTRSinStSpec outs rules =
	  let
		  open Term;
		  val faMap  = Trs.funArityMapInRules rules
		  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap

		  fun prType 0 = "o"
			| prType 1 = "o=>o"
			| prType n = "o*" ^ (prType (n-1))

		  fun prVarSym (x,0) = "?" ^ (replace (Atom.toString x))
		    | prVarSym (x,i) = "?" ^ (replace (Atom.toString x))  ^ "_" ^ (Int.toString i) 

		  fun prTerm (Var (x,ty)) = (replace (Var.toString x)) ^ ":" ^ (S.toString ty)
			| prTerm (Fun (f,[],ty)) = henkan (Fun.toString f) 
			| prTerm (Fun (f,ts,ty)) = "(" ^ (prTermList ts) ^ ")"
		  (* and prTermList ts = PrintUtil.prSpace prTerm ts *)
		  and prTermList ts = ListUtil.toStringSpace prTerm ts
			  
		  val osort = Sort.fromString "o"
		  val app = Fun.fromString "@"


		  fun foTerm2StTerm (Var (x,_)) = Var (x, osort)
			| foTerm2StTerm (t as (Fun (f,[],ty))) = t
			| foTerm2StTerm (Fun (f,ts,ty)) = 
							 let val hd = Fun (f,[],ty)
								 val ts' = L.map foTerm2StTerm ts
							 in Fun (app,hd::ts',ty)
							 end

		  val _ = TextIO.output (outs, "SORTS\n")
		  val _ = TextIO.output (outs, " o;\n\n")

		  val _ = TextIO.output (outs, "FUNCTIONS\n")
		  val _ = FIS.app  (fn (f,i) 
							 => TextIO.output (outs, " " ^ (henkan (Fun.toString f)) ^ " : " 
											   ^ (prType i) ^ ";\n"))
			  faSet
		  val _ = TextIO.output (outs, "\n")

		  val _ = TextIO.output (outs, "RULES\n")
		  val _ = List.app (fn (l,r) 
							=> TextIO.output (outs, 
											  (" " ^ (prTerm (foTerm2StTerm l)) ^ " -> " 
											   ^ (prTerm (foTerm2StTerm r))  ^ ";\n")))
			  rules
	  in
		  ()
	  end

  (* TPDB  FoTRS  TPDB  MsTRS ؤѴ *)
  fun printFoTRSinMsTRS outs rules =
	  let
		  open Term;
		  val faMap  = Trs.funArityMapInRules rules
		  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
		  val vSet  = Trs.varSetInRules rules

		  fun prType 0 = "Nat"
			| prType 1 = "Nat -> Nat"
			| prType n = "Nat," ^ (prType (n-1))

		  fun prVarSym (x,0) = "?" ^ (replace (Atom.toString x))
		    | prVarSym (x,i) = "?" ^ (replace (Atom.toString x))  ^ "_" ^ (Int.toString i) 

		  fun prTerm (Var (x,ty)) = (replace (Var.toString x)) ^ ":" ^ (S.toString ty)
			| prTerm (Fun (f,[],ty)) = henkan (Fun.toString f) 
			| prTerm (Fun (f,ts,ty)) = "(" ^ (prTermList ts) ^ ")"
		  (* and prTermList ts = PrintUtil.prSpace prTerm ts *)
		  and prTermList ts = ListUtil.toStringSpace prTerm ts
			  
		  val osort = Sort.fromString "o"
		  val app = Fun.fromString "@"

		  fun foTerm2StTerm (Var (x,_)) = Var (x, osort)
			| foTerm2StTerm (t as (Fun (f,[],ty))) = t
			| foTerm2StTerm (Fun (f,ts,ty)) = 
							 let val hd = Fun (f,[],ty)
								 val ts' = L.map foTerm2StTerm ts
							 in Fun (app,hd::ts',ty)
							 end

		  val _ = TextIO.output (outs, "(FUN\n")
		  val _ = FIS.app  (fn (f,i) 
				       => TextIO.output (outs, "   " ^ (Fun.toString f) ^ " : " 
							       ^ (prType i) ^ "\n"))
			  faSet
		  val _ = TextIO.output (outs, ")\n")

		  fun prVar x = let val n = Var.index x
				in if n = 0 then Var.name x
				   else (Var.name x) ^ "_" ^ (Int.toString n)
				end

		  val _ = TextIO.output (outs, "(Var\n")
		  val _ = VarSet.app  (fn x 
				       => TextIO.output (outs, "   " ^ (prVar x) ^ " : Nat\n"))
			  vSet
		  val _ = TextIO.output (outs, ")\n")

		  fun prTerm (Var (x,ty)) = (prVar x)
		    | prTerm (Fun (f,[],ty)) = (Fun.toString f)
		    | prTerm (Fun (f,ts,ty)) =  (Fun.toString f) ^ (ListUtil.toStringCommaRound prTerm ts)

		  val _ = TextIO.output (outs, "(RULES\n")
		  val _ = List.app (fn (l,r) 
				       => TextIO.output (outs, 
							 ("   " ^ (prTerm l) ^ " -> " ^ (prTerm r)  ^ "\n")))
			  rules
		  val _ = TextIO.output (outs, ")\n")
	  in
		  ()
	  end



  fun checkFile (prfun,suffix)  (path,filename) =
      let val decs = String.tokens (fn #".":char => true |_ => false) filename
		  val suffix = hd (rev decs)
		  fun getname [] = suffix
			| getname (x::xs) = x ^ "." ^ (getname xs)
		  val newname = getname (rev (tl (rev decs)))
		  exception ReadFileError
      in if suffix  = "trs"
		 then let val _ = PU.printStdErr ("Reading " ^ filename ^ "...\n")
				  val rules = case rdFile filename of
						  (SOME (rs,_), _, _ ) => rs
						| (NONE, _, _ ) => raise ReadFileError
				  val _ = PU.printStdErr ("Translating " ^ filename ^ "...\n")
				  val outFile = newname
				  val outs = TextIO.openOut outFile
				  val _ = prfun outs rules
				  val _ = TextIO.closeOut outs
			  in ()
			  end			
			  handle ReadFileError => PU.printStdErr ("Skipped ... read file error\n")
		 else ()
      end	 



  fun checkFile2 prfun (path,filename) =
      let val decs = String.tokens (fn #".":char => true |_ => false) filename
		  exception ReadFileError
		  val suffix = hd (rev decs)
      in if suffix  = "trs"
		 then
	     let
			 val rules = case rdFile filename of
					     (SOME (rs,_), _, _ )  => rs
					   | (NONE, _, _ ) =>raise ReadFileError
			 val _ = prfun TextIO.stdOut rules
	     in ()
	     end			
		 handle ReadFileError => PU.printStdErr ("Skipped ... read file error\n")
	 else 
	     PU.printStdErr ("Skipped ... not .trs file\n")
      end	 

  exception Abort


  (* fun transCommand (prfun,suffix) (name:string, args:string list) *)
  fun transCommand prfun (name:string, args:string list) =
      let  
	  val commandDir = OS.FileSys.getDir();
	  val targets = case args of 
			    [] =>(PrintUtil.println "file or directory not specified";
				  raise Abort)
			  | xs => List.map 
				      (fn x => OS.FileSys.realPath x
					  handle OS.SysErr (msg,_) => 
						 (PrintUtil.println ("file or directory not found: " 
								     ^ x);
						  raise Abort))
				      xs
      in
	  if length targets = 1 
	     andalso not (OS.FileSys.isDir (hd targets))
	  then 
	      (CommandUtil.applyCmdToDirTree 
(**		   (checkFile (prfun,suffix)) **)
		   (checkFile2 prfun)
		   ("", hd targets);
	       OS.Process.success)
	  else
	      (List.app (fn item => 
			    CommandUtil.applyCmdToDirTree 
(**				(checkFile (prfun,suffix)) **)	
			(checkFile2 prfun)
				("", item))
			targets;
	       OS.Process.success)
	      handle e  => (OS.FileSys.chDir commandDir; raise (e))
      end
      handle Abort => OS.Process.failure

  (* disable to support CTRSs in 2018/01/25 *)
  (* val transFo2St = transCommand printFoTRSinStSpec *)
  (* val transFo2KBSeminar = transCommand printFoTRSinKBSeminarSpec *)
  (* val transFo2Ms = transCommand printFoTRSinMsTRS *)

  (*  SMLofNJ.exportFn("trans",transFo2St) *)
  (*  SMLofNJ.exportFn("trans",IOTpdb.transFo2KBSeminar) *)
  (*  SMLofNJ.exportFn("trans",IOTpdb.transFo2Ms) *)

  end (* of local *)

  end;




