(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/cr.sml
 * description: ingredients for checking Chuch-Rosser Property
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature CR = 
   sig
    val runDebug: bool ref
   val criticalPeaks : (Term.term * Term.term) list 
		       -> (Term.term * Term.term * Term.term) list
   val criticalPeaksKB : (Term.term * Term.term) 
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term * Term.term) list
   val insideCriticalPeaks : 
       (Term.term * Term.term) list -> (Term.term * Term.term * Term.term) list
   val insideCriticalPeaksWithIndex : 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term * Term.term)) list
   val insideCriticalPeaksBetweenDifferentRules: 
       (Term.term * Term.term) list -> (Term.term * Term.term * Term.term) list
   val insideCriticalPeaksBetweenDifferentRulesWithIndex: 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term * Term.term)) list

   val outsideCriticalPeaks : 
       (Term.term * Term.term) list -> (Term.term * Term.term * Term.term) list
   val outsideCriticalPeaksInOneside : 
       (Term.term * Term.term) list -> (Term.term * Term.term * Term.term) list
   val outsideCriticalPeaksWithIndex: 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term * Term.term)) list
   val outsideCriticalPeaksInOnesideWithIndex : 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term * Term.term)) list

   val criticalPeaks2 : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
			-> (Term.term * Term.term * Term.term) list
   val insideCriticalPeaks2 : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
			       -> (Term.term * Term.term * Term.term) list
   val outsideCriticalPeaks2 : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
			       -> (Term.term * Term.term * Term.term) list

   val criticalPairs : (Term.term * Term.term) list -> (Term.term * Term.term) list
   val criticalPairsKB : (Term.term * Term.term) 
	   -> (Term.term * Term.term) list 
	   -> (Term.term * Term.term) list

   val insideCriticalPairs : 
       (Term.term * Term.term) list -> (Term.term * Term.term) list
   val insideCriticalPairsWithIndex: 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term)) list
   val insideCriticalPairsBetweenDifferentRules: 
       (Term.term * Term.term) list -> (Term.term * Term.term) list
   val insideCriticalPairsBetweenDifferentRulesWithIndex: 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term)) list

   val outsideCriticalPairs : 
       (Term.term * Term.term) list -> (Term.term * Term.term) list
   val outsideCriticalPairsInOneside : 
       (Term.term * Term.term) list -> (Term.term * Term.term) list
   val outsideCriticalPairsWithIndex: 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term)) list
   val outsideCriticalPairsInOnesideWithIndex: 
       (Term.term * Term.term) list -> ((int * int) * (Term.term * Term.term)) list

   val criticalPairs2 : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
			-> (Term.term * Term.term) list
   val insideCriticalPairs2 : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
			      -> (Term.term * Term.term) list
   val outsideCriticalPairs2 : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
			       -> (Term.term * Term.term) list

   val criticalPairs2WithRules:
       ((Term.term * Term.term) list * (Term.term * Term.term) list)
       -> ((Term.term * Term.term) * (Term.term * Term.term) * Term.term * Term.term) list

   val insideCriticalPairsWithIndexAndFs:
       (Term.term * Term.term) list -> ((int * int) * (Fun.ord_key list * Term.term * Term.term)) list

   val insideCriticalPeaksWithIndexAndPosFs:
       (Term.term * Term.term) list ->
       ((int * int) * (int list * Fun.ord_key list * Term.term * Term.term * Term.term)) list

   val insideCriticalPeaksWithIndexAndPos:
       (Term.term * Term.term) list ->
       ((int * int) * (int list * Term.term * Term.term * Term.term)) list
											 
   val insideCriticalPairsWithIndexAndPosFs:
       (Term.term * Term.term) list ->
       ((int * int) * (int list * Fun.ord_key list * Term.term * Term.term)) list

   val outsideCriticalPairsInOnesideWithIndexAndFs:
       (Term.term * Term.term) list -> ((int * int) * (Fun.ord_key list * Term.term * Term.term)) list

   val outsideCriticalPairsInOnesideWithIndexAndPosFs:
       (Term.term * Term.term) list -> 
       ((int * int) * (int list * Fun.ord_key list * Term.term * Term.term)) list

   val condCriticalPeaks: 
       Ctrs.crules -> (Term.term * Term.term * Term.term * (Term.term * Term.term) list) list

   val condCriticalPairs: 
       Ctrs.crules -> (Term.term * Term.term * (Term.term * Term.term) list) list

   val condInsideCriticalPeaks: 
       Ctrs.crules -> (Term.term * Term.term * Term.term * (Term.term * Term.term) list) list

   val condInsideCriticalPairs: 
       Ctrs.crules -> (Term.term * Term.term * (Term.term * Term.term) list) list

   val condOutsideCriticalPeaks: 
       Ctrs.crules -> (Term.term * Term.term * Term.term * (Term.term * Term.term) list) list

   val condOutsideCriticalPairs: 
       Ctrs.crules -> (Term.term * Term.term * (Term.term * Term.term) list) list

   val allInnerOverlaps: Term.term -> ((Term.term * Term.term) list)
		    -> (Term.position * Subst.subst * Term.term) list

   val allOverlaps: Term.term -> ((Term.term * Term.term) list)
		    -> (Term.position * Subst.subst * Term.term) list

   val makeParallelPositions: (Term.position * Subst.subst * Term.term) list
			      -> (Subst.subst * ((Term.position * Term.term) list)) option

   val innerProperParallelCriticalPairs: 
       (Term.term * Term.term) list -> (Term.term * Term.term) list

   val innerProperParallelCriticalPairsWithVarSet:
       (Term.term * Term.term) list -> (Term.term * Term.term * VarSet.set) list

   val innerProperParallelCriticalPairsWithVarSet2:
       (Term.term * Term.term) list * (Term.term * Term.term) list 
       -> (Term.term * Term.term * VarSet.set) list

   val simultaneousCriticalPairs0:
       (Term.term * Term.term) -> (Term.term * Term.term) list  -> (Term.term * Term.term) list

   val simultaneousCriticalPairs:
       (Term.term * Term.term) list  -> (Term.term * Term.term) list

   val isLocalConfluent : (Term.term * Term.term) list -> bool
   val isJoinableCps : (Term.term * Term.term) list -> 
		       (Term.term * Term.term) list -> bool
   val isInnerJoinableCps : (Term.term * Term.term) list -> 
			    (Term.term * Term.term) list -> bool

   val isLocalConfluentForNonTerminatingRules : (Term.term * Term.term) list -> bool
   val isNonOverlapping : (Term.term * Term.term) list -> bool
   val isOverlay : (Term.term * Term.term) list -> bool

   datatype ConfluenceResult = CR | NotCR | Unknown
   val report: ConfluenceResult -> ConfluenceResult

   datatype CommutativityResult = COM | NotCOM | UnknownCOM

   val checkConfluenceConditions:
	   ((Term.term * Term.term) list -> bool) 
	   -> (Term.term * Term.term) list 
	   ->  ConfluenceResult

   val isParallelCriticalPairClosed:
       (Term.term * Term.term) list 
       -> (Term.term * Term.term) list 
       -> bool

   val isStrongClosed:
       (Term.term * Term.term) list 
       -> Term.term * Term.term
       -> bool

   val isParallelClosed:
       (Term.term * Term.term) list 
       -> Term.term * Term.term
       -> bool

   val isParallelClosedCps:
       (Term.term * Term.term) list 
       -> (Term.term * Term.term) list 
       -> bool

   val isStrongClosedCps:
       (Term.term * Term.term) list 
       -> (Term.term * Term.term) list 
       -> bool

   val isDepthRestrictedClosedCps:
       (Term.term * Term.term) list 
       -> (int list * Term.term * Term.term) list 
       -> (Term.term * Term.term) list 
       -> bool

   val isOostromClosedInCp:
       (Term.term * Term.term) list 
       -> Term.term * Term.term
       -> bool

   val isOostromClosedOutCp:
       (Term.term * Term.term) list 
       -> Term.term * Term.term
       -> bool

   val isOostromClosedCps:
       (Term.term * Term.term) list 
       -> (Term.term * Term.term) list 
       -> (Term.term * Term.term) list 
       -> bool

   val isOkuiClosedCps:
       (Term.term * Term.term) list 
       -> bool

   val isHuetModulo:
       ((Term.term * Term.term) list * (Term.term * Term.term) list -> bool)
       -> (Term.term * Term.term) list 
       -> bool

   val findCommonReduct: (Term.term * Term.term) list 
			 -> int
			 -> (Term.term * Term.term) 
			 -> (Term.term list * Term.term list) option

   val findConvSequencesWithIndex: (Term.term * Term.term) list 
			 -> int
			 -> ((int * Term.term) * (int * Term.term))
			 -> ((int * int) list) list

   val findCommonReductWithIndexAndFs: (Term.term * Term.term) list 
			 -> int
			 -> ((Fun.ord_key list * int * Term.term)
			     * (Fun.ord_key list * int * Term.term))
			 -> ((Fun.ord_key list * int * Term.term) list 
			     * (Fun.ord_key list * int * Term.term) list) list

   val findCommonReductWithIndexAndPosFs: (Term.term * Term.term) list 
			 -> int
			 -> (((Fun.ord_key * int) list * int * Term.term)
			     * ((Fun.ord_key * int) list * int * Term.term))
			 -> (((Fun.ord_key * int) list * int * Term.term) list 
			     * ((Fun.ord_key *int) list * int * Term.term) list) list

   val isModuloJoinableCps: (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
                         -> bool		    

   val confluenceByDepthPreserving: (Term.term * Term.term) list -> bool

   val isWeightDecreasingJoinableTrs: (Term.term * Term.term) list -> bool

end;

structure Cr : CR = 
   struct

   local 
       open Term
       open Trs
       open Rewrite
       open Subst
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure TP = TermPair
       structure TPS = TermPairSet
       structure TS = TermSet
       open PrintUtil
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   exception CrError

   (* we follow the definition of 
    "An Introduction to Knuth-Bendix Completion" by J.W.Klop and A.Middeldorp 
    *)
   local
       fun cpsub (alpha,beta) C (t,delta) =
	   case (cpsubWhole (alpha,beta) C (t,delta)) of
		   SOME ans => ans::(cpsubProperSubterm (alpha,beta) C (t,delta))
		 | NONE => cpsubProperSubterm (alpha,beta) C (t,delta)
       and cpsubWhole (alpha,beta) C (t,delta) =
	   case t of 
	       Var _ => NONE
	     | Fun (f,ts,ty) 
	       => case (unify t alpha) of
		      SOME sigma => SOME (applySubst sigma (C alpha), 
					  applySubst sigma (C beta), 
					  applySubst sigma delta)
		    | NONE => NONE
       and cpsubProperSubterm (alpha,beta) C (t,delta) =
	   case t of 
	       Var _ => []
	     | Fun (f,ts,ty) 
	       => cpsubArguments (alpha,beta) 
				 (fn xs => C (Fun (f,xs,ty)))
				 (ts,delta) 
       and cpsubArguments (alpha,beta) C ([],delta) = []
	 | cpsubArguments (alpha,beta) C (t::ts,delta) =
	   List.@(cpsub (alpha,beta) (fn x => C (x::ts)) (t,delta),
		  cpsubArguments (alpha,beta) (fn xs => C (t::xs)) (ts,delta))

   in
   (* $B0[$J$k5,B'4V$N%k!<%H0LCV$G$N4m81BP$N(B2$BEY7W;;$r$J$/$7$?(B *)
   (* ==> ??? $B:81&$,0[$J$k$+$iN>J}I,MW$N$O$:!$=$@5(B 2017/12/2 *)
   fun criticalPeaks rs =
       let val rs' = renameRules rs
           val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
	   (* fun prPeak (u,v,w) = ("<" ^ Term.toString u ^ " <- " ^ Term.toString v ^ " -> " ^ Term.toString w ^ ">" ) *)
       in
		   LU.mapAppend
		   (fn n => LU.mapAppend
			(fn m => if n = m
				 then
				     let
					 val lr = List.nth (rs,n)
					 val [lr1,lr2] = renameRules [lr,lr]
					 (* val _ = println  ("cp " ^ Trs.prRule lr1 ^ " on " ^ Trs.prRule lr2) *)
					 (* $B<+J,<+?H$K$D$$$F$O(B proper subterm $B$G$N=E$J$j$@$1(B *)
					 val ans = cpsubProperSubterm lr1 (fn x => x) lr2
					 (* val _ = println (LU.toStringCommaLnSquare prPeak ans) *)
				     in 
					 ans
				     end
				 else 
				     let val lr1 = List.nth (rs',m)
					 val lr2 = List.nth (rs',n)
					 (* val _ = println  ("cp " ^ Trs.prRule lr1 ^ " on " ^ Trs.prRule lr2) *)
					 val ans = cpsub lr1 (fn x => x) lr2
					 (* val _ = println (LU.toStringCommaLnSquare prPeak ans) *)
				     in
					 ans
				     end
			        (**** 2017/12/2 cp$B$N8~$-$r9M$($k>l9g$b$"$k$N$G%@%a(B
                                     $B8~$-$r9M$($J$$>l9g$O!$(BcriticalPair2(rs,rs)$B$r;H$($P$h$$!%(B
                                 if n < m
				 then (* $BAj0[$J$k5,B'$K$D$$$F$O(B $BG$0U$N(B subterm $B$G$N=E$J$j(B *)
				     cpsub (List.nth (rs',m)) (fn x => x) (List.nth (rs',n))
				 else (* m < n *)
				     (* m,n $B4V$N%k!<%H0LCV$O(B n < m $B$N$H$-$K$d$C$F$$$k$N$G$$$i$J$$(B *)
				     cpsubProperSubterm (List.nth (rs',m)) (fn x => x) (List.nth (rs',n))
                                 ****)
			)
			nums)
		   nums
       end



  (*  r $B$H(B r::rs $B$N4V$N4m81BP=89g(B *)
   fun criticalPeaksKB r rs = 
       let val (r1::r2::rs') = renameRules (r::r::rs)
       in 
	   (* $B<+J,<+?H$K$D$$$F$O(B proper subterm $B$G$N=E$J$j$@$1(B *)
 	   L.@ (cpsubProperSubterm r1 (fn x => x) r2,
		(* rs $B$N5,B'$X$N(B r $B$N=E$J$j!$G$0U$N(B subterm $B$G(B *)
		L.@ (LU.mapAppend
 			 (fn lr => cpsub lr (fn x => x) r1)
 			 rs',
		     (* r $B$X$N(B rs $B$N5,B'$N=E$J$j!$G$0U$N(B subterm $B$G(B *)
		     LU.mapAppend
			 (fn lr => cpsub r1 (fn x => x) lr)
			 rs'))
       end

   fun insideCriticalPeaks rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
       in 
 	   L.@(LU.mapAppend 
		   (* $BAj0[$J$k5,B'$K$D$$$F$O(B renaming $BI,MW$J$7(B *)
		   (fn n => LU.mapAppend
				(fn lr => cpsubProperSubterm lr (fn x => x) (List.nth (rs',n)))
				(LU.exceptNth (rs',n)))
		   nums,
	       (* $B<+J,<+?H$K$D$$$F$O(B renaming $B$r$7$F$+$i(B *)
	       LU.mapAppend (fn lr => let val [lr1,lr2] = renameRules [lr,lr]
				   in cpsubProperSubterm lr1 (fn x => x) lr2
				   end)
			 rs')
       end

   fun insideCriticalPeaksWithIndex rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
       in 
 	   L.@(LU.mapAppend 
	       (* $BAj0[$J$k5,B'$K$D$$$F$O(B renaming $BI,MW$J$7(B *)
		   (fn n => 
		       (LU.mapAppend
			    (fn m => (L.map (fn cp => ((m,n),cp))
					    (cpsubProperSubterm (List.nth (rs',m))
								(fn x => x) 
								(List.nth (rs',n)))))
			    (LU.exceptNth (nums,n))))
		   nums,
		    (* $B<+J,<+?H$K$D$$$F$O(B renaming $B$r$7$F$+$i(B *)
	       LU.mapAppend (fn n => let val lr = List.nth (rs',n)
				      val [lr1,lr2] = renameRules [lr,lr]
				  in L.map (fn cp => ((n,n),cp))
					    (cpsubProperSubterm lr1 (fn x => x) lr2)
				  end)
			 nums)
       end

       (* outside CP $B$O(B $BBP>NE*$J$N$G!$JRB&$@$1$GNI$$>l9g$O$3$A$i(B *)
   fun outsideCriticalPeaksInOneside rs = 
       let val rs' = renameRules rs
	   fun loop [] = []
	     | loop (x::ys) = 
	       L.@(List.mapPartial (fn y => cpsubWhole x (fn i => i) y) ys,
		   loop ys)
       in 
	   loop rs'
       end

(*    fun outsideCriticalPeaks rs =  *)
(*        let val rs' = renameRules rs *)
(* 	   val nums = List.tabulate (List.length rs, fn i => i)   (\* 0,...,n-1 *\) *)
(*        in  *)
(*  	   LU.mapAppend  *)
(* 		   (fn n => List.mapPartial *)
(* 				(fn lr => cpsubWhole lr (fn x => x) (List.nth (rs',n))) *)
(* 				(LU.exceptNth (rs',n))) *)
(* 		   nums *)
(*        end *)

    fun outsideCriticalPeaks rs =  
	let val cps = outsideCriticalPeaksInOneside rs
	in cps @ (L.map (fn (x,y,z) => (x,z,y)) cps)
	end

   fun insideCriticalPeaksBetweenDifferentRules rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
       in 
 	   LU.mapAppend 
	   (fn n => LU.mapAppend
			(fn m => cpsubProperSubterm (List.nth (rs',m))
						    (fn x => x) (List.nth (rs',n)))
			(LU.exceptNth (nums,n)))
	   nums
       end;

   fun insideCriticalPeaksBetweenDifferentRulesWithIndex rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
       in 
 	   LU.mapAppend 
	   (fn n => LU.mapAppend
			(fn m => L.map (fn cp => ((m,n),cp))
				       (cpsubProperSubterm (List.nth (rs',m))
							   (fn x => x) (List.nth (rs',n))))
			(LU.exceptNth (nums,n)))
	   nums
       end;

   fun outsideCriticalPeaksInOnesideWithIndex rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
	   fun loop [] = []
	     | loop (n::ms) = 
	       L.@(List.mapPartial
				(fn m => case cpsubWhole (List.nth (rs',m))
							 (fn x => x) (List.nth (rs',n))
					  of SOME cp => SOME ((m,n),cp)
					   | NONE => NONE) ms,
		   loop ms)
       in 
	   loop nums
       end

(*    fun outsideCriticalPeaksWithIndex rs =  *)
(*        let val rs' = renameRules rs *)
(* 	   val nums = List.tabulate (List.length rs, fn i => i)   (\* 0,...,n-1 *\) *)
(*        in  *)
(*  	   LU.mapAppend  *)
(* 		   (fn n => List.mapPartial *)
(* 				(fn m => case cpsubWhole (List.nth (rs',m)) *)
(* 							 (fn x => x) (List.nth (rs',n)) *)
(* 					  of SOME cp => SOME ((m,n),cp) *)
(* 					   | NONE => NONE) *)
(* 				(LU.exceptNth (nums,n))) *)
(* 		   nums *)
(*        end *)

   fun outsideCriticalPeaksWithIndex rs = 
       let val cps = outsideCriticalPeaksInOnesideWithIndex rs
	in cps @ (L.map (fn ((m,n),(x,y,z)) => ((n,m),(x,z,y))) cps)
	end


  (* 2$B$D$N0[$J$k=q$-49$(%7%9%F%`$N85$N4m81D:(B <top, l, r >*)
   fun criticalPeaks2 (rs1,rs2)  = 
       let val (rs1',rs2') = renameRuleSet (rs1,rs2)
       in 
 	   LU.mapAppend 
		   (fn lr1' => LU.mapAppend
				(fn lr2' => cpsub lr1' (fn x => x) lr2')
				rs2')
		   rs1'
       end

   fun insideCriticalPeaks2 (rs1,rs2)  = 
       let val (rs1',rs2') = renameRuleSet (rs1,rs2)
       in 
 	   LU.mapAppend 
		   (fn lr1' => LU.mapAppend
				(fn lr2' => cpsubProperSubterm lr1' (fn x => x) lr2')
				rs2')
		   rs1'
       end

   fun outsideCriticalPeaks2 (rs1,rs2)  = 
       let val (rs1',rs2') = renameRuleSet (rs1,rs2)
       in
 	   LU.mapAppend 
		   (fn lr1' => List.mapPartial
				(fn lr2' => cpsubWhole lr1' (fn x => x) lr2')
				rs2')
		   rs1'
       end

   fun criticalPairs rs = L.map (fn (x,y,z) => (y,z)) (criticalPeaks rs)

   fun criticalPairsKB r rs = L.map (fn (x,y,z) => (y,z)) (criticalPeaksKB r rs)

   fun insideCriticalPairs rs = 
       L.map (fn (x,y,z) => (y,z)) (insideCriticalPeaks rs)

   fun insideCriticalPairsWithIndex rs = 
       L.map (fn (mn,(x,y,z)) => (mn,(y,z))) (insideCriticalPeaksWithIndex rs)

   fun insideCriticalPairsBetweenDifferentRules rs = 
       L.map (fn (x,y,z) => (y,z)) (insideCriticalPeaksBetweenDifferentRules rs)

   fun insideCriticalPairsBetweenDifferentRulesWithIndex rs = 
       L.map (fn (mn,(x,y,z)) => (mn,(y,z))) (insideCriticalPeaksBetweenDifferentRulesWithIndex rs)

   fun outsideCriticalPairsInOneside rs = 
       L.map (fn (x,y,z) => (y,z)) (outsideCriticalPeaksInOneside rs)

   fun outsideCriticalPairs rs = 
       L.map (fn (x,y,z) => (y,z)) (outsideCriticalPeaks rs)

   fun outsideCriticalPairsInOnesideWithIndex rs = 
       L.map (fn (mn,(x,y,z)) => (mn,(y,z))) (outsideCriticalPeaksInOnesideWithIndex rs)

   fun outsideCriticalPairsWithIndex rs = 
       L.map (fn (mn,(x,y,z)) => (mn,(y,z))) (outsideCriticalPeaksWithIndex rs)

   fun criticalPairs2 (rs1,rs2) = L.map (fn (x,y,z) => (y,z)) (criticalPeaks2 (rs1,rs2))
   fun insideCriticalPairs2 (rs1,rs2) = L.map (fn (x,y,z) => (y,z)) (insideCriticalPeaks2 (rs1,rs2))
   fun outsideCriticalPairs2 (rs1,rs2) = L.map (fn (x,y,z) => (y,z)) (outsideCriticalPeaks2 (rs1,rs2))
   end

   fun criticalPairs2WithRules (R1,R2) = 
       List.concat (ListXProd.mapX (fn (lr,lr') => 
				       L.map (fn (u,v) => (lr,lr',u,v)) 
					     (criticalPairs2 ([lr],[lr'])))
	       			   (R1,R2))

   (*** Pos, FS $BIU$-%P!<%8%g%s(B ***)
   (* we follow the definition of 
    "An Introduction to Knuth-Bendix Completion" by J.W.Klop and A.Middeldorp 
    *)
   local
       fun getPosFromContext C =
	   let val zero = Var.fromStringAndInt  ("", ~1)
	       val dummy = Var (zero, Sort.null)
	       val t = C dummy
	       fun pathToZero (Var (x,_)) =
		   if Var.equal (x,zero) then SOME [] else NONE
		 | pathToZero (Fun (f,ts,_)) =
		   case pathToZeroList 1 ts of
		       NONE => NONE
		     | SOME ps => SOME ps
	       and pathToZeroList _ [] = NONE
		 | pathToZeroList i (t::ts) =
		   case pathToZero t of
		       NONE => pathToZeroList (i+1) ts
		     | SOME ps => SOME (i::ps)
	   in valOf (pathToZero t)
	   end

       fun getFsFromContext C =
	   let val zero = Var.fromStringAndInt  ("", ~1)
	       val dummy = Var (zero, Sort.null)
	       val t = C dummy
	       fun pathToZero (Var (x,_)) = if Var.equal (x,zero)
					then SOME [] else NONE
		 |	pathToZero (Fun (f,ts,_)) =
			case pathToZeroList ts of
			    NONE => NONE
			  | SOME fs => SOME (f::fs)
	       and pathToZeroList [] = NONE
		 | pathToZeroList (t::ts) =
		   case pathToZero t of
		       NONE => pathToZeroList ts
		     | SOME fs => SOME fs
	   in valOf (pathToZero t)
	   end

(*        fun getFsPosFromContext C =  *)
(* 	   let val zero = Var.fromStringAndInt  ("", ~1) *)
(* 	       val dummy = Var (zero, Sort.null) *)
(* 	       val t = C dummy *)
(* 	       fun pathToZero (Var (x,_)) = if Var.equal (x,zero) *)
(* 					    then SOME [] else NONE *)
(* 		 |	pathToZero (Fun (f,ts,_)) =  *)
(* 			case pathToZeroList 1 ts of  *)
(* 			    NONE => NONE *)
(* 			  | SOME (fps,i) => SOME ((f,i)::fps) *)
(* 	       and pathToZeroList _ [] = NONE *)
(* 		 | pathToZeroList i (t::ts) =  *)
(* 		   case pathToZero t of *)
(* 		       NONE => pathToZeroList (i+1) ts  *)
(* 		     | SOME fps => SOME (fps,i) *)
(* 	   in valOf (pathToZero t) *)
(* 	   end *)


       fun cpsubFs (alpha,beta) C (t,delta) =
	   case (cpsubWholeFs (alpha,beta) C (t,delta)) of
		   SOME ans => ans::(cpsubProperSubtermFs (alpha,beta) C (t,delta))
		 | NONE => cpsubProperSubtermFs (alpha,beta) C (t,delta)
       and cpsubWholeFs (alpha,beta) C (t,delta) =
	   case t of 
	       Var _ => NONE
	     | Fun (f,ts,ty) 
	       => case (unify t alpha) of
		      SOME sigma => SOME (getPosFromContext C,
					  getFsFromContext C,
					  applySubst sigma (C alpha), 
					  applySubst sigma (C beta), 
					  applySubst sigma delta)
		    | NONE => NONE
       and cpsubProperSubtermFs (alpha,beta) C (t,delta) =
	   case t of 
	       Var _ => []
	     | Fun (f,ts,ty) 
	       => cpsubArgumentsFs (alpha,beta) 
				 (fn xs => C (Fun (f,xs,ty)))
				 (ts,delta) 
       and cpsubArgumentsFs (alpha,beta) C ([],delta) = []
	 | cpsubArgumentsFs (alpha,beta) C (t::ts,delta) =
	   List.@(cpsubFs (alpha,beta) (fn x => C (x::ts)) (t,delta),
		  cpsubArgumentsFs (alpha,beta) (fn xs => C (t::xs)) (ts,delta))

   in
   fun insideCriticalPeaksWithIndexAndPosFs rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
       in 
 	   L.@(LU.mapAppend 
	       (* $BAj0[$J$k5,B'$K$D$$$F$O(B renaming $BI,MW$J$7(B *)
		   (fn n => 
		       (LU.mapAppend
			    (fn m => (L.map (fn cp => ((m,n),cp))
					    (cpsubProperSubtermFs (List.nth (rs',m))
								(fn x => x) 
								(List.nth (rs',n)))))
			    (LU.exceptNth (nums,n))))
		   nums,
		    (* $B<+J,<+?H$K$D$$$F$O(B renaming $B$r$7$F$+$i(B *)
	       LU.mapAppend (fn n => let val lr = List.nth (rs',n)
				      val [lr1,lr2] = renameRules [lr,lr]
				  in L.map (fn cp => ((n,n),cp))
					    (cpsubProperSubtermFs lr1 (fn x => x) lr2)
				  end)
			 nums)
       end

   fun insideCriticalPeaksWithIndexAndFs rs = 
       L.map (fn (mn,(ps,fs,x,y,z)) => (mn,(fs,x,y,z))) (insideCriticalPeaksWithIndexAndPosFs rs)

   fun insideCriticalPeaksWithIndexAndPos rs = 
     L.map (fn (mn,(ps,fs,x,y,z)) => (mn,(ps,x,y,z))) (insideCriticalPeaksWithIndexAndPosFs rs)

   fun insideCriticalPairsWithIndexAndPosFs rs = 
       L.map (fn (mn,(ps,fs,x,y,z)) => (mn,(ps,fs,y,z))) (insideCriticalPeaksWithIndexAndPosFs rs)

   fun insideCriticalPairsWithIndexAndFs rs = 
       L.map (fn (mn,(ps,fs,x,y,z)) => (mn,(fs,y,z))) (insideCriticalPeaksWithIndexAndPosFs rs)

   fun outsideCriticalPeaksInOnesideWithIndexAndPosFs rs = 
       let val rs' = renameRules rs
	   val nums = List.tabulate (List.length rs, fn i => i)   (* 0,...,n-1 *)
	   fun loop [] = []
	     | loop (n::ms) = 
	       L.@(List.mapPartial
				(fn m => case cpsubWholeFs (List.nth (rs',m))
							 (fn x => x) (List.nth (rs',n))
					  of SOME cp => SOME ((m,n),cp)
					   | NONE => NONE) ms,
		   loop ms)
       in 
	   loop nums
       end

   fun outsideCriticalPairsInOnesideWithIndexAndPosFs rs = 
       L.map (fn (mn,(ps,fs,x,y,z)) => (mn,(ps,fs,y,z))) 
	     (outsideCriticalPeaksInOnesideWithIndexAndPosFs rs)

   fun outsideCriticalPairsInOnesideWithIndexAndFs rs = 
       L.map (fn (mn,(ps,fs,x,y,z)) => (mn,(fs,y,z))) 
	     (outsideCriticalPeaksInOnesideWithIndexAndPosFs rs)


   end


   (* $B$9$Y$F$N(B <p, sigma|V(t), r\sigma> such that simga = mgu (t/p,l), t/p \notin V $B$N%j%9%H$rJV$9(B *)
   (* $BJQ?t$O(B rename $B:Q$_$H2>Dj(B *)
   local
       val vnames = ref VS.empty
       fun rootOverlaps rp (Var _) rs = []
	 | rootOverlaps rp (t as (Fun _)) rs = 
	   L.mapPartial 
	       (fn (l,r) => 
		   case (unify t l) of 
		       SOME sigma => let val vsetOfL = Term.varSetInTerm l
				     in
					 if VS.isEmpty (VS.intersection (vsetOfL, !vnames))
					 then 
					     (vnames := VS.union (vsetOfL, !vnames);
					      SOME (rev rp, 
						    let val vset = Term.varSetInTerm t
						    in VM.filteri (fn (v,_) => VS.member (vset,v)) sigma
						    end,
						    Subst.applySubst sigma r))
					 else let val (l',r') = Trs.renameRuleDisjointFrom 
								    (VS.listItems (!vnames))
							            (l,r)
						  val sigma' = valOf (unify t l')
						  val vsetOfL = Term.varSetInTerm l'
						  val _ = vnames := VS.union (vsetOfL, !vnames)
					      in
						  SOME (rev rp, 
							let val vset = Term.varSetInTerm t
							in VM.filteri (fn (v,_) => VS.member (vset,v)) 
								      sigma'
							end,
							Subst.applySubst sigma r')
					      end
				     end

		     | NONE => NONE) rs
       fun innerOverlaps rp (Var _) rs = []  
	 | innerOverlaps rp (Fun (_,ts,_)) rs = 
	   let val rps = L.tabulate (length ts, fn x => (x+1)::rp)
	       val pts = LP.zip (rps,ts)
	   in 
	       L.@ (LU.mapAppend (fn (rq,t) => rootOverlaps rq t rs) pts,
		    LU.mapAppend (fn (rq,t) => innerOverlaps rq t rs) pts)
	   end
   in
       fun allInnerOverlaps t rs = (vnames := VS.empty; innerOverlaps [] t rs)
       fun allOverlaps t rs = (vnames := VS.empty; 
			       L.@ (rootOverlaps [] t rs, innerOverlaps [] t rs))
   end

   fun makeParallelPositions xs = 
       let
	   fun incomparable [] _ = false
	     | incomparable _ [] = false
	     | incomparable (n::ns) (m::ms) = (n <> m) orelse incomparable ns ms

	   fun merge (rho,u) (sigma,ts) =
	       let exception CanNotMerge
		   val domain = VS.addList (VS.addList (VS.empty, VM.listKeys rho), 
					    VM.listKeys sigma)
	       in
		   VS.foldl
		   (fn (v,(m,x,ys)) => 
		       case (VM.find (rho,v), VM.find (sigma,v)) of
			   (NONE, NONE) => (m,x,ys) (* this doesn't happen *)
			 | (SOME t, NONE) => (VM.insert (m,v,t),x,ys)
			 | (NONE, SOME t) => (VM.insert (m,v,t),x,ys)
			 | (SOME t1, SOME t2) => 
			   case unify t1 t2 of 
			       SOME rho => let val s = Subst.applySubst rho t1
					   in (VM.insert (m,v,s),
					       Subst.applySubst rho x,
					       L.map (Subst.applySubst rho) ys)
					   end
			     | NONE => raise CanNotMerge)
		   (VM.empty, u, ts)
		   domain 
	       end

	   fun parallelPositions [] (ps,ts,sigma)= SOME (sigma, LP.zip (ps,ts))
	     | parallelPositions ((p,rho,u)::ys) (ps,ts,sigma) =
	       (if L.all (incomparable p) ps
		then let val (phi,u',ts') = merge (rho,u) (sigma,ts)
		     in parallelPositions ys (p::ps,u'::ts',phi)
		     end
		else NONE)
	       handle CanNotMerge => NONE 
       in
	   parallelPositions xs ([],[],VM.empty)
       end


   fun nonSingleNonEmptySublists [] = []
     | nonSingleNonEmptySublists (x::xs) = 
       L.@ (L.map (fn y=> [x,y]) xs,
	    let val yss = nonSingleNonEmptySublists xs
	    in L.@ (yss, L.map (fn ys => x::ys) yss)
	    end)

   fun nonEmptySublists [] = []
     | nonEmptySublists (x::xs) = 
	   let val yss = nonEmptySublists xs
	   in L.@ (yss, [x]::L.map (fn ys => x::ys) yss)
	   end

   fun innerProperParallelCriticalPairs rs =
   (* include usual *)
       let
	   fun findPcps lr = 
	       let val ((l,r)::rs') = renameRules (lr::rs)
		   val overlaps = allInnerOverlaps l rs'
		   val idx = L.tabulate (length overlaps, fn x => x)
(*		   val idxes = nonSingleNonEmptySublists idx *)
		   val idxes = nonEmptySublists idx
		   val posAndReducts = 
		       L.mapPartial 
			   (fn ns => makeParallelPositions 
					 (L.map (fn i => L.nth (overlaps,i)) ns))
			   idxes
	       in
		   L.map (fn (sigma, pts) => 
			     (L.foldl
				  (fn ((p,u),t) => valOf (Term.replaceSubterm t p u))
				  l pts,
			      Subst.applySubst sigma r))
			 posAndReducts
	       end
       in
	   LU.mapAppend findPcps rs
       end

   fun innerProperParallelCriticalPairsWithVarSet rs =
       let
	   fun findPcps lr = 
	       let val ((l,r)::rs') = renameRules (lr::rs)
		   val overlaps = allInnerOverlaps l rs'
		   val idx = L.tabulate (length overlaps, fn x => x)
		   val idxes = nonEmptySublists idx
		   val posAndReducts = 
		       L.mapPartial 
			   (fn ns => makeParallelPositions 
					 (L.map (fn i => L.nth (overlaps,i)) ns))
			   idxes
	       in
		   L.map (fn (sigma, pts) => 
			     (L.foldl
				  (fn ((p,u),t) => valOf (Term.replaceSubterm t p u))
				  l pts,
			      Subst.applySubst sigma r,
			      let val lsigma = Subst.applySubst sigma l
				  val terms = L.map (fn (p,_) => valOf (Term.subterm p lsigma)) pts
			      in Term.varSetInTerms terms
			      end))
			 posAndReducts
	       end
       in
	   LU.mapAppend findPcps rs
       end


   fun innerProperParallelCriticalPairsWithVarSet2 (rs1,rs2) =
       let
	   fun findPcps lr = 
	       let val ((l,r)::rs1') = renameRules (lr::rs1)
		   val overlaps = allInnerOverlaps l rs1'
		   val idx = L.tabulate (length overlaps, fn x => x)
		   val idxes = nonEmptySublists idx
		   val posAndReducts = 
		       L.mapPartial 
			   (fn ns => makeParallelPositions 
					 (L.map (fn i => L.nth (overlaps,i)) ns))
			   idxes
	       in
		   L.map (fn (sigma, pts) => 
			     (L.foldl
				  (fn ((p,u),t) => valOf (Term.replaceSubterm t p u))
				  l pts,
			      Subst.applySubst sigma r,
			      let val lsigma = Subst.applySubst sigma l
				  val terms = L.map (fn (p,_) => valOf (Term.subterm p lsigma)) pts
			      in Term.varSetInTerms terms
			      end))
			 posAndReducts
	       end
       in
	   LU.mapAppend findPcps rs2
       end


   fun isJoinableCps rs cps =
       List.all (fn (u,v) => Term.equal (linf rs u,  linf rs v)) 
		cps

   fun isInnerJoinableCps rs cps =
       List.all (fn (u,v) => Term.equal (linf rs u,  linf rs v)) 
		cps

   fun isLocalConfluent rs =
       isJoinableCps rs (criticalPairs rs)	 

   fun isLocalConfluentForNonTerminatingRules rs =
       let val cps = criticalPairs rs
	   fun isjoinable (u,v) 
	     = not (TS.isEmpty 
		    (TermSet.intersection (developOneStepReductSet rs u,
					   developOneStepReductSet rs v)))
       in L.all isjoinable (criticalPairs rs)
       end

   fun isNonOverlapping rs = 
       null (criticalPairs rs)

   fun isOverlay rs = 
       null (insideCriticalPeaks rs)

   (* Huet $B$N(B Parallel-Closed $B>r7o(B *)
   fun isParallelClosed rs (t1,t2) =
       if Term.equal (t1,t2) 
       then true
       else TS.member (Rewrite.parallelOneStepReductSet rs t1, t2)
				   
   fun isParallelClosedCps rs cps =
       List.all (isParallelClosed rs) cps

   (* $B30;3$N>r7o!'4J0WHG(B *)
   fun isToyamaClosedCps rs inCps outCps =
       (List.all (fn (t1,t2) => 
		     Term.equal (t1,t2) 
		     orelse TS.member (Rewrite.parallelOneStepReductSet rs t1, t2))
		 inCps)
       andalso
       (List.all (fn (t1,t2) => 
		    Term.equal (t1,t2) 
		    orelse 		    
	            not (TS.isEmpty (TS.intersection (Rewrite.parallelOneStepReductSet rs t1,  
						      Rewrite.parallelOneStepReductSet rs t2))))
		 outCps)

   (* Oostrom$B>r7o!'4J0WHG(B *)
   fun isOostromClosedInCp rs (t1,t2) = Term.equal (t1,t2) 
					 orelse TS.member (Rewrite.developOneStepReductSet rs t1, t2)

   fun isOostromClosedOutCp rs (t1,t2) = Term.equal (t1,t2) 
					 orelse 		    
					 not (TS.isEmpty (TS.intersection (Rewrite.developOneStepReductSet rs t1,  
									   Rewrite.developOneStepReductSet rs t2)))

   fun isOostromClosedCps rs inCps outCps =
       List.all (isOostromClosedInCp rs) inCps
       andalso
       List.all (isOostromClosedOutCp rs) outCps


   (* Strong Closed $B>r7o!'4J0WHG(B *)
   fun isStrongClosed rs (t1,t2) =
       Term.equal (t1,t2)
       orelse (let val t2d = Rewrite.developOneStepReductSet rs t2
	       in TS.member (t2d, t1)
		  orelse not (TS.isEmpty (TS.intersection (t2d, Rewrite.oneStepReductSet rs t1)))
	       end
	       andalso
	       let val t1d = Rewrite.developOneStepReductSet rs t1
	       in TS.member (t1d, t2)
		  orelse  not (TS.isEmpty (TS.intersection (t1d, Rewrite.oneStepReductSet rs t2)))
	       end)

   fun isStrongClosedCps rs cps =  List.all (isStrongClosed rs) cps

   fun isDepthRestrictedClosedCps rs inCpsPos outCps =
       let fun checkOut (l,r) = 
	       let 
		   val _ = debug (fn _ => println ("check outer CP " ^ (Trs.prEq (l,r))))
		   val l1 = Rewrite.parallelOneStepReductSet rs l
		   val _ = debug (fn _ => println ("X1 = { s | l -||-> s } = " ^
						   (prSetInOneLine Term.toString (TS.listItems l1))))
		   val l2 = LU.mapAppend (Rewrite.nonRootParallelOneStepReducts rs) (TS.listItems l1)
		   val _ = debug (fn _ => println ("X2 = { s' | s -||Q-> s', s in X1, Q /= {e} } = " ^
						   (prSetInOneLine Term.toString l2)))
		   val _ = debug (fn _ => println ("check r in X2"))
	       in LU.member' Term.equal r l2
		  orelse
		  let val r1 = Rewrite.rootRewriteAllSet rs r
		      val _ = debug (fn _ => println ("Y1 = { s | r -e-> s } = " ^
						   (prSetInOneLine Term.toString (TS.listItems r1))))
		      val _ = debug (fn _ => println ("check X1 cap Y1 /= {} "))
		  in not (TS.isEmpty (TS.intersection (l1,r1)))
		     orelse
		     let val r2 = LU.mapAppend (Rewrite.rootRewriteAll rs) (TS.listItems r1)
			 val _ = debug (fn _ => println ("Y2 = { s' | s -e-> s', s in Y1 } = " ^
						   (prSetInOneLine Term.toString r2)))
			 val _ = debug (fn _ => println ("check l in Y2"))
		     in LU.member' Term.equal l r2
			orelse let val l3 = Rewrite.rootRewriteAll (L.map (fn (x,y) => (y,x)) rs) l
				   val _ = debug (fn _ => println ("X3 = { s | s -e-> l } = " ^
								   (prSetInOneLine Term.toString l3)))
				   val l4 = LU.mapAppend (Rewrite.nonRootParallelOneStepReducts rs) l3
				   val _ = debug (fn _ => println ("X4 = { s' | s -||Q-> s', s in X3, Q /= {e} } = " ^
						   (prSetInOneLine Term.toString l4)))
				   val _ = debug (fn _ => println ("check r in X4"))
			       in LU.member' Term.equal r l4
			       end
		     end
		  end
	       end

	   fun checkIn1 (pos,l,r) = 
	       let
		   val _ = debug (fn _ => println ("check inner CP " ^ (Trs.prEq (l,r))
						   ^ " at position p = " ^ (Pos.toString pos)))
		   val l1 = Rewrite.rootRewriteAllSet rs l
		   val _ = debug (fn _ => println ("X1 = { s | l -e-> s } = " ^
						   (prSetInOneLine Term.toString (TS.listItems l1))))
		   val _ = debug (fn _ => println ("check r in X1 "))
	       in
	       TS.member (l1,r)
	       orelse let val r1 = Rewrite.parallelOneStepReductsOfRestrictedDepthLength 
				       rs (length pos) r
			  val _ = debug (fn _ => println ("Y1 = { s | r -||Q-> s, |q| <= |p| for all q in Q} = " ^
							  (prSetInOneLine Term.toString r1)))
			  val _ = debug (fn _ => println ("check l in Y1 "))
		      in LU.member' Term.equal l r1
		      end
	       end

	   fun checkIn2 (pos,l,r) =
	       let  val _ = debug (fn _ => println ("check inner CP " ^ (Trs.prEq (l,r))
						    ^ " at position p = " ^ (Pos.toString pos)))
		    val l1 = Rewrite.rootRewriteAllSet rs l
		   val _ = debug (fn _ => println ("X1 = { s | l -e-> s } = " ^
						   (prSetInOneLine Term.toString (TS.listItems l1))))
		   val _ = debug (fn _ => println ("check r in X1 cup { l }"))
	       in
		   Term.equal (l,r)
		   orelse TS.member (l1,r)
	       orelse let val r1 = oneStepReductsWithRestrictedPosition rs pos r
			  val _ = debug (fn _ => println ("Y1 = { s | r -q-> s, q /> p } = " ^
							  (prSetInOneLine Term.toString r1)))
			  val _ = debug (fn _ => println ("check l in Y1 "))
		      in LU.member' Term.equal l r1
		      end 
	       end

       in
	   (List.all checkOut outCps)
	   andalso (List.all checkOut (L.map (fn (x,y) => (y,x)) outCps))
	   andalso ((println "inner CP cond (upside-parallel)"; (List.all checkIn1 inCpsPos))
		    orelse (println "innter CP Cond (outside)"; (List.all checkIn2 inCpsPos)))
       end


   (* Parallel Critical Pair $B>r7o(B *)
   fun isParallelCriticalPairClosed rs cps =
       if L.all (fn (t1,t2) =>
		    let val set1 = Rewrite.parallelOneStepReductSet rs t1
			val set2 = Rewrite.parallelOneStepReductSet rs t2
		    in not (TS.isEmpty (TS.intersection (set1,set2)))
		    end)
		cps
       then
	   let val pcps = innerProperParallelCriticalPairsWithVarSet rs
	       val _ = print "(inner) Parallel CPs:\n"
	       fun printPcpWithVar (s,t,vset) = 
		   ((IOFotrs.prEq (s,t))
		    ^ " "
		    ^ (LU.toStringCommaCurly Var.toString (VS.listItems vset)))

	       val _ = print (LU.toStringCommaLnSquare printPcpWithVar pcps)
	       fun check (s,t,vset) = 
		   let val cands = L.mapPartial (fn (u,vset') => 
						    if VS.isSubset (vset', vset)
						    then SOME u
						    else NONE)
						(parallelOneStepReductsWithVar rs t)
		       val sd = Rewrite.developOneStepReductSet rs s
		   in L.exists (fn u => TS.member (sd,u)) cands
		   end
	   in L.all check pcps
	   end
       else
	   (print "(inner) Parallel CPs: (not computed)\n"; false)


   fun simultaneousCriticalPairs0 (l,r) rs = 
       let 
	   fun fposInTerm t = Term.collectSubtermOCs Term.isFun t
	   val fposInL = Term.collectSubtermOCs Term.isFun l
	   (* isPrefix ns ms: ns <= ms in prefix ordering *)
	   fun isPrefix [] _ = true
	     | isPrefix (n::ns) [] = false
	     | isPrefix (n::ns) (m::ms) = n = m andalso (isPrefix ns ms)

	   fun removePrefixPositions p qs = L.filter (fn q => not (isPrefix q p)) qs

	   val rootOverlaps = 
	       let val ((l',r')::rs') = renameRules ((l,r)::rs)
	       in
	       L.mapPartial (fn (l1,r1) => 
				case unify l1 l' of
				    SOME rho => SOME (Subst.applySubst rho l',
						      Subst.applySubst rho r',
						      posListMinus (posListMinus (fposInL,[[]]), 
								    fposInTerm l1),
						      [([],l1,r1)],
						      rho)
				  | NONE => NONE)  rs'
	       end

	   exception DevelopCPError of string


	   fun overlapsAt _ _ _ [] _ = []
	     | overlapsAt (l,r) ps qs pos rs = 
	       let val ((l0,r0)::rs') = renameRules ((l,r)::rs)
	       in 
		   L.mapPartial 
		       (fn (x,y) 
			   => case subterm pos l0 of
				  NONE => raise DevelopCPError "subterm"
				| SOME subl =>
				  (case unify x subl of
				       SOME rho => 
						    SOME (Subst.applySubst rho l0,
							  Subst.applySubst rho r0,
							  
							  posListMinus (removePrefixPositions pos ps, 
									L.map (fn p => pos@p) 
									     (fposInTerm x)),
							 (* $B$3$l$+$i@h$K$^$@E83+$G$-$k0LCV(B *)
							 (pos,x,y)::qs, (* $B$3$l$^$G$NE83+0LCV$HE83+5,B'(B *)
							 rho)
				     |  NONE => NONE))
		       rs'
	       end


	   val result0 = rootOverlaps @ (LU.mapAppend 
					     (fn pos => overlapsAt (l,r) 
								   (posListMinus (fposInL, [[]]))
								   [] pos ((l,r)::rs))
					     fposInL)


(* 	   val _ = L.app (fn (l',r',ps,qs,rho) *)
(* 			     => (print "\n["; *)
(* 				 print (prRule (l',r')); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaCurly prPosition ps); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaLnCurly *)
(* 					    (fn (pos,x,y) => ((prPosition pos) *)
(* 							     ^ ": " ^ (prRule (x,y)))) *)
(* 					    qs); *)
(* 				 print "\n"; *)
(* 				 print "]\n")) *)
(* 			 result0 *)

	   fun loop done ans = 
	       let val (done1,done2) = L.partition (fn (_,_,ps,_,_) => null ps) done
		   fun repeat (l',r',ps,qs,rho) =
		       LU.mapAppend (fn pos => overlapsAt (l',r') ps qs pos ((l,r)::rs)) ps
	       in
		   if (null done2)
		   then (done1 @ ans)
		   else loop (LU.mapAppend repeat done2) (done @ ans)
	       end

	   val  result0a = loop result0 []

(* 	   val _ = print ("===============================\n"); *)

(* 	   val _ = L.app (fn (l',r',ps,qs,rho) *)
(* 			     => (print "\n["; *)
(* 				 print (prRule (l',r')); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaCurly prPosition ps); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaLnCurly *)
(* 					    (fn (pos,x,y) => ((prPosition pos) *)
(* 							     ^ ": " ^ (prRule (x,y)))) *)
(* 					    qs); *)
(* 				 print "\n"; *)
(* 				 print "]\n")) *)
(* 			 result0a *)

	   fun rewriteManySteps t [] = t
	     | rewriteManySteps t ((pos,l,r)::qs) = 
	       case subterm pos t of 
		   NONE => raise DevelopCPError "subterm"
		 | SOME tsub 
		   => case rootRewrite [(l,r)] tsub of
			  NONE => raise DevelopCPError "rootRewrite"
			| SOME reduct 
			  => case replaceSubterm t pos reduct of 
				 NONE => raise DevelopCPError "replaceSubterm"
			       | SOME ans => rewriteManySteps ans qs

	   val result0b = L.map (fn (l',r',ps,qs,rho)
				   => (rewriteManySteps l' qs, r')) 
			       result0a

(* 	   val _ = print ("===============================\n"); *)

(* 	   val _ = print (prEqs result0b) *)

	   (* t,l $B$O(B rename $B:Q$_$H2>Dj(B *)
	   fun getAboveRootOverlaps pos (t,s) l = 
	       case subterm pos t of 
		   NONE => raise DevelopCPError "subterm"
		 | SOME tsub 
		   => case unify tsub l of
			  NONE => NONE
			| SOME rho => SOME (case replaceSubterm (Subst.applySubst rho t)
								pos 
								(Subst.applySubst rho l) of
						NONE => raise DevelopCPError "replaceSubterm"
					      | SOME t' => t',
					    case replaceSubterm (Subst.applySubst rho t)
								pos 
								(Subst.applySubst rho r) of
						NONE => raise DevelopCPError "replaceSubterm"
					      | SOME t' => t',
					    posListMinus (L.map (fn p => pos @ p) fposInL,
							  fposInTerm t),
					    [([],t,s)],
					    rho)

	   val result3 = 
	       let val ((l0,r0)::(l',r')::rs') = renameRules ((l,r)::(l,r)::rs)
	       in LU.mapAppend
		      (fn (x,y) => L.mapPartial
				       (fn pos => getAboveRootOverlaps pos (x,y) l0)
				       (posListMinus (fposInTerm x,[[]])))
		      ((l',r')::rs')
	       end

(* 	   val _ = print ("vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv\n"); *)

(* 	   val _ = L.app (fn (l',r',ps,qs,rho) *)
(* 			     => (print "\n["; *)
(* 				 print (prRule (l',r')); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaCurly prPosition ps); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaLnCurly *)
(* 					    (fn (pos,x,y) => ((prPosition pos) *)
(* 							     ^ ": " ^ (prRule (x,y)))) *)
(* 					    qs); *)
(* 				 print "\n"; *)
(* 				 print "]\n")) *)
(* 			 result3 *)

	   val result3a = loop result3 []

(* 	   val _ = print ("vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv\n"); *)

(* 	   val _ = L.app (fn (l',r',ps,qs,rho) *)
(* 			     => (print "\n["; *)
(* 				 print (prRule (l',r')); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaCurly prPosition ps); *)
(* 				 print "\n"; *)
(* 				 print (LU.toStringCommaLnCurly *)
(* 					    (fn (pos,x,y) => ((prPosition pos) *)
(* 							     ^ ": " ^ (prRule (x,y)))) *)
(* 					    qs); *)
(* 				 print "\n"; *)
(* 				 print "]\n")) *)
(* 			 result3a *)

	   val result3b = L.map (fn (l',r',ps,qs,rho)
				   => (rewriteManySteps l' qs, r')) 
			       result3a

(* 	   val _ = print (prEqs result3b) *)

       in
	   result0b @ result3b
       end

   fun simultaneousCriticalPairs rs =
       LU.mapAppend 
	   (fn n =>
	       simultaneousCriticalPairs0 (L.nth (rs,n)) (LU.exceptNth (rs,n)))
	   (L.tabulate (length rs, fn x=>x))


   (* Okui $B>r7o(B *)
   fun isOkuiClosedCps rs =
       let val dcps = simultaneousCriticalPairs rs
	   val _ = print "Simultaneous CPs:\n"
	   val dcps' = LU.eliminateDuplication' 
			   (fn (x,y) => identicalModuloRenamingRule x y)
			   dcps
	   val _ = print (prEqs dcps')
	   fun check (s,t) = 
	       let val _ = debug (fn _ => print ("check CP  " ^ (prEq (s,t)) ^ "\n"))
		   val sd = Rewrite.developOneStepReductSet rs s
		   val td = Rewrite.developOneStepReductSet rs t
		   val _ = debug (fn _ => println ("X = { s | l -o-> s } = " ^ prSetInOneLine Term.toString (TS.listItems sd)))
		   val _ = debug (fn _ => println ("Y = { t | r -o-> t } = " ^ prSetInOneLine Term.toString (TS.listItems td)))
		   val _ = debug (fn _ => println ("check X cap Y /= {}"))
		   val cap = TS.intersection (sd,td)
	       in not (TS.isEmpty cap)
	       end
       in L.all check dcps'
       end

   (* Huet modulo $B>r7o(B: LL(rs), VP(es), SN(rs/es)$B$r2>Dj(B *)
   fun isHuetModulo isRelativeTerminating rs =
       let fun isCom (Fun (f,[t1,t2],_), Fun (g,[s1,s2],_)) =
	       Fun.equal (f,g) 
	       andalso
	       (List.all Term.isVar [t1,t2,s1,s2])
	       andalso 
	       Term.equal (t1,s2)
	       andalso 
	       Term.equal (t2,s1)
	     | isCom _ = false
	   fun isAssoc (Fun (f,[t1,t2],_), Fun (g,[s1,s2],_)) =
	       Fun.equal (f,g) 
	       andalso
	       (case (t1,s2) of 
		    (Fun (f2, [u1,u2], _), Fun (g2, [v1,v2], _))
		    => Fun.equal (f,f2) 
		       andalso Fun.equal (g,g2) 
		       andalso List.all Term.isVar [t2,u1,u2,s1,v1,v2]
		       andalso Term.equal (u1,s1)
		       andalso Term.equal (u2,v1)
		       andalso Term.equal (t2,v2)
		  | _ => false)
	     | isAssoc _ = false

	   val (es0,rs0) = List.partition
			     (fn (l,r) => isCom (l,r)
					  orelse isAssoc (l,r)
					  orelse isAssoc (r,l))
			     rs

          (* add refinement on 2013/12/17 *)
	   fun partAssocOnly xs =
		   let val cfuns = L.mapPartial
				       (fn (l,r) => if isCom (l,r) then Term.funRootOfTerm l
						    else NONE) xs
		       fun incl (l,r) = isCom (l,r)
					orelse 
					(LU.member' Fun.equal (valOf (Term.funRootOfTerm l)) cfuns)
		   in  List.partition incl xs end

	   val (es,rs) = let val (es',rs') = partAssocOnly es0
			   in (es',rs0 @ rs') end

	   val _ = debug (fn _ => print "non AC rules:\n")
	   val _ = debug (fn _ => print (Trs.prRules rs))
	   val _ = debug (fn _ => print "AC rules:\n")
	   val _ = debug (fn _ => print (Trs.prRules es))

	   fun check0 (t1,t2) = 
	       let val nf1 = Rewrite.linf rs t1
		   val nf2 = Rewrite.linf rs t2
	       in Rewrite.isEquivalent es (nf1,nf2)
	       end

	   fun check (t1,t2) = 
	       let val _ = debug (fn _ => print ("  <" ^ (Term.toString t1) ^ " = "
						 ^ (Term.toString t2) ^ ">" ))
		   val nf1 = Rewrite.linf rs t1
		   val nf2 = Rewrite.linf rs t2
		   val _ = debug (fn _ => (print (" ==> <" ^ (Term.toString nf1) ^ " = "
				  ^ (Term.toString nf2) ^ ">")))
		   val res = Rewrite.isEquivalent es (nf1,nf2)
		   val _ = debug (fn _ => if res 
			   then print ": true\n"
			   else print ": false\n")
	       in res
	       end

	   fun checkConditions () = 
	       let val es2 = LU.union' (fn (l,r) => Subst.identicalModuloRenamingRule l r)
				       (es, L.map (fn (l,r) => (r,l)) es)

	       in
		   if true
		   then let val cps1 = criticalPairs rs
			    val _ = debug (fn _ => print "CP(R):\n")
			    val res1 = L.map check cps1
			    val _ = debug (fn _ => print "CP(R,P U P^{-1}):\n")
			    val cps2 = criticalPairs2 (rs,es2)
			    val res2 = L.map check cps2
			    val _ = debug (fn _ =>  print "CP(P U P^{-1},R):\n")
			    val cps3 = criticalPairs2 (es2,rs)
			    val res3 = L.map check cps3
			in List.all (fn x=>x) (res1 @ res2 @ res3)
			end
		   else
		       L.all check0 (criticalPairs rs)
		       andalso L.all check0 (criticalPairs2 (rs,es2))
		       andalso L.all check0 (criticalPairs2 (es2,rs))
	       end
       in
	   if null es
	      orelse not (Rewrite.isReversibleRules es)
	   then false
	   else if not (isRelativeTerminating (rs,es))
	   then ((* print "unknown relatively terminating (nonAC/AC)\n"; *) false)
	   else ((* print "relatively terminating (nonAC/AC)\n";*) checkConditions ())
       end

  (* $B9gN.@-$N8!>Z7k2L(B *)
   datatype ConfluenceResult = CR | NotCR | Unknown

   fun report CR = (print ": CR\n"; CR)
     | report NotCR = (print ": not CR\n"; NotCR)
     | report Unknown = (print ": Can't judge\n"; Unknown)

  (* $B2D49@-$N8!>Z7k2L(B *)
   datatype CommutativityResult = COM | NotCOM | UnknownCOM


  (* $B9gN.@-$N4pK\H=Dj(B *)
   fun checkConfluenceConditions isTerminating rs = 
       let val inCps = insideCriticalPairs rs
	   val outCps = outsideCriticalPairsInOneside rs
	   val cps = inCps @ outCps
	   val _ = print "Check Termination...\n"
       in
	   if isTerminating rs
	   then let val _ = print "Terminating"
		    val joinableCps = isJoinableCps rs (inCps @ outCps)
		    val _ = if joinableCps
			    then print ", WCR"
			    else print ", not WCR"
		in
		    if joinableCps then report CR else report NotCR
		end
	   else 
	       let val _ = print "unknown Terminating"
	       in
		   if Trs.areLinearRules rs 
		   then let val _ = print ", Linear"
			    val oostromClosedCps = isOostromClosedCps rs inCps outCps
			    val _ = if oostromClosedCps
				    then print ", Oostrom"
				    else print ", unknown Oostrom"
			in
			    if oostromClosedCps 
			    then report CR 
			    else let val stronglyClosed = isStrongClosedCps rs cps
				     val _ = if stronglyClosed
					     then print ", strongly closed"
					     else print ", unknown strongly closed"
				 in if stronglyClosed
				    then report CR
				    else report Unknown
				 end
			end
		   else if Trs.areLeftLinearRules rs
		   then let val _ = print ", Left-Linear"
(*			    val _ = print (Trs.prRules rs) *)
(*			    val _ = print ("In  CPs:\n" ^ (prEqs inCps)) *)
(*			    val _ = print ("Out CPs:\n" ^ (prEqs outCps)) *)
			    val oostromClosedCps = isOostromClosedCps rs inCps outCps
			    val _ = if oostromClosedCps
				    then print ", Oostrom"
				    else print ", unknown Oostrom"
			in
			    if oostromClosedCps 
			    then report CR 
			    else report Unknown
			end
		   else let val _ = print ", not Left-Linear"
			in report Unknown
			end
	       end
       end

  (* find only one *)
   fun findCommonReduct R maxCount (s,t) =
       let val counter = ref 0
	   exception NotFound
	   fun checkOneStep (xss,yss) = 
	       (counter := !counter + 1;
		if !counter > maxCount * 2 
		then NONE
		else 
		    let val xss2 = Rewrite.strengthenRewriteSeq R xss
			fun member x [] = NONE
			  | member x (y::ys) = if Term.equal (x,y)
					       then SOME (y::ys)
					       else member x ys
			exception Found of Term.term list * Term.term list
		    in
			(case (L.find (fn xs => 
					 isSome (L.find (fn ys => case member (hd xs) ys of
							SOME ys' =>  raise Found (xs,ys')
						      | NONE => false)
					     yss))
				     xss2) of
			    SOME _ => SOME ([],[]) (* never here *)
			  | NONE => checkOneStep (yss,xss2))
			handle Found (xs,ys) => SOME (xs,ys)
		    end)
       in
	   case checkOneStep ([[s]],[[t]]) of
	       SOME (xs,ys) =>if (!counter) mod 2 <> 0
			      then SOME (xs,ys) else SOME (ys,xs)
	     | NONE => NONE
       end

  (* find only one *)
   (* fun findCommonReductWithIndex R maxCount (s,t) = *)
   (*     let val counter = ref 0 *)
   (* 	   exception NotFound *)
   (* 	   fun checkOneStep (xss,yss) =  *)
   (* 	       (counter := !counter + 1; *)
   (* 		if !counter > maxCount * 2  *)
   (* 		then NONE *)
   (* 		else  *)
   (* 		    let val xss2 = Rewrite.strengthenRewriteSeqWithIndex R xss *)
   (* 			fun member (_,x) [] = NONE *)
   (* 			  | member (j,x) ((i,y)::ys) = if Term.equal (x,y) *)
   (* 					       then SOME ((i,y)::ys) *)
   (* 					       else member (j,x) ys *)
   (* 			exception Found of (int * Term.term) list * (int * Term.term) list *)
   (* 		    in *)
   (* 			(case (L.find (fn xs =>  *)
   (* 					 isSome (L.find (fn ys => case member (hd xs) ys of *)
   (* 							SOME ys' =>  raise Found (xs,ys') *)
   (* 						      | NONE => false) *)
   (* 					     yss)) *)
   (* 				     xss2) of *)
   (* 			    SOME _ => SOME ([],[]) (* never here *) *)
   (* 			  | NONE => checkOneStep (yss,xss2)) *)
   (* 			handle Found (xs,ys) => SOME (xs,ys) *)
   (* 		    end) *)
   (*     in *)
   (* 	   case checkOneStep ([[s]],[[t]]) of *)
   (* 	       SOME (xs,ys) =>if (!counter) mod 2 <> 0 *)
   (* 			      then SOME (xs,ys) else SOME (ys,xs) *)
   (* 	     | NONE => NONE *)
   (*     end *)

  (* find all$B!$(BR$B$N5U8~$-$N5,B'$bMQ$$$k2~NI(B(?)$BHG!$(B
     $B$?$@$7(B ($BJRB&(B) maxCount $BD9$5J,$r:G=i$K5a$a$k(B *)
   fun findConvSequencesWithIndex R maxCount ((m0,s0),(n0,t0)) =
       if Term.equal (s0,t0)
       then [[]]
       else
	   let 
	   val RrevWithIdx  = L.mapPartial (fn i => let val (l,r) = L.nth (R,i)
						    in if Trs.isRewriteRule (r,l)
						       then SOME (i,(r,l))
						       else NONE
						    end)
					   (L.tabulate (L.length R, fn i => i))

	   val Rrev = L.map (fn (_,lr) => lr) RrevWithIdx
	   val Rext = R@Rrev

	   val seqFromS0 = Rewrite.nStepsRewriteSeqWithIndex Rext maxCount (m0,s0)
	   val seqFromT0 = Rewrite.nStepsRewriteSeqWithIndex Rext maxCount (n0,t0)

          (* $B5U8~$-$K;H$C$?5,B'$N%$%s%G%C%/%9$rLa$9(B *)
	   val lenR = L.length R
	   val renamelist = L.map (fn i => let val (j,_) = L.nth (RrevWithIdx,i)
					   in (lenR + i, j) 
					   end)
				  (L.tabulate (L.length Rrev, fn i=>i))

	   fun renameIndex0 [] (dir,i) =  (dir,i)
	     | renameIndex0 ((j,k)::xs) (dir,i) =  if i = j then ((~1)*dir,k)
						   else renameIndex0 xs (dir,i)

	   fun renameIndex us = L.map (renameIndex0 renamelist) us

	   (* (1,n): -> direction, (-1,n): <- direction *)
	   fun combine (us,vs) =  
	       let val us2 = renameIndex (L.map (fn (i,t) => (1,i)) (LU.exceptLast us))
		   val vs2 = renameIndex (L.map (fn (i,t) => (~1,i)) (LU.exceptLast vs))
		   (* val _ = print "->:" *)
		   (* val _ = L.app (fn (i,t) => print ("(" ^ (Int.toString i) ^ ")" *)
		   (* 				     ^ (Term.toString t) ^ ":")) *)
		   (* 		 (rev us) *)
		   (* val _ = print "\n" *)
		   (* val _ = print "->:" *)
		   (* val _ = L.app (fn (i,j) => print ("(" ^ (Int.toString j) ^ "):")) *)
		   (* 		 (rev us2) *)
		   (* val _ = print "\n" *)
		   (* val _ = print "<-:" *)
		   (* val _ = L.app (fn (i,t) => print ("(" ^ (Int.toString i) ^ ")" *)
		   (* 				     ^ (Term.toString t) ^ ":")) *)
		   (* 		 vs *)
		   (* val _ = print "\n" *)

	       in (rev us2) @ vs2
	       end

	   fun hasEqualTerms ((_,s),(_,t)) = Term.equal (s,t)

           (* same as critical pair making sequence *)
	   fun isCP [] = false
	     | isCP [x] = false
	     | isCP ws =  (let val (dir1,i1) = hd ws
			      val (dir2,i2) = hd (tl ws)
			  in dir1 = ~1 andalso i1 = m0
			     andalso dir2 = 1 andalso i2 = n0
			  end) orelse isCP (tl ws)
       
	   fun isJoins (x::xs, y::ys) = 
	       if hasEqualTerms (x,y) 
		  andalso LU.disjoint' (fn ((_,s),(_,t)) => Term.equal (s,t)) (xs,ys)
	       then let val ws = combine (x::xs, y::ys)
		    in if isCP ws then [] else [ws]
		    end
	       else []

	   fun checkJoins (uss,vss) = ListXProd.foldX 
					  (fn (us,vs,wss)  => (isJoins (us,vs)) @ wss)
					  (uss,vss)
					  []

	   (* index $BNs$,(B $BB>$NItJ,Ns$K$J$C$F$$$J$$(B join sequence $B$N$_$rJV$9(B *)
	   fun isLeq ([],_) = true
	     | isLeq ((d1,i1)::ns,(d2,i2)::ms) = ((d1 = d2) andalso (i1 = i2) andalso isLeq (ns,ms))
						 orelse isLeq ((d1,i1)::ns,ms)
	     | isLeq (_::ns,[])  = false

	   fun removeNonMinimal0 us wss = L.mapPartial (fn ws => if isLeq (us,ws) then NONE else SOME ws) 
					 	       wss
	   fun removeNonMinimal [] ans = ans
	     | removeNonMinimal (us::wss) ans = removeNonMinimal 
						    (removeNonMinimal0 us wss)
						    (us::(removeNonMinimal0 us ans))

	   in removeNonMinimal (checkJoins (seqFromS0,seqFromT0)) []
	   end


  (* find all *)
   fun findCommonReductWithIndexAndFs R maxCount (s,t) =
       let val counter = ref 0
	   fun hasEqualTerms ((_,_,s),(_,_,t)) = Term.equal (s,t)

	   fun printX (fs,i,s) = (L.app (print o Fun.toString) fs;
				  print (":" ^ (Int.toString i) ^ ":");
				  print ((Term.toString s) ^ "\n"))

	   fun checkOneStep (xss,oldxss) (yss,oldyss) = 
	       (counter := !counter + 1;
		if !counter > maxCount 
		then []
		else 
		    let val xss2 = Rewrite.strengthenRewriteSeqWithIndexAndFs R xss
			val yss2 = Rewrite.strengthenRewriteSeqWithIndexAndFs R yss
			val oldxss2 = xss @ oldxss
			val oldyss2 = yss @ oldyss
			fun isJoins (x::_, y::_) =  hasEqualTerms (x,y) 
			fun select (uss,vss) = 
			    ListXProd.foldX 
				(fn (us,vs,wss)  => if isJoins (us,vs)
						    then (us,vs)::wss
						    else wss)
				(uss,vss)
				[]

			val ans = select (xss2,oldyss2) 
				  @ select (oldxss2,yss2) 
				  @ select (xss2,yss2)

(* 			val _ = L.app (fn (XS,YS)  *)
(* 					  => ((L.app printX  XS);(L.app printX  YS))) *)
(* 				      ans  *)
		    in
			if null ans 
			then checkOneStep (xss2,oldxss2) (yss2,oldyss2)
			else ans
		    end)
       in
	   if hasEqualTerms (s,t)
	   then [([s],[t])]
	   else checkOneStep ([[s]],[]) ([[t]],[])
       end

  (* all *)
   fun findCommonReductWithIndexAndPosFs R maxCount (s,t) =
       let val counter = ref 0
	   fun hasEqualTerms ((_,_,s),(_,_,t)) = Term.equal (s,t)
	   fun prFunInt (f,i) = "(" ^ (Fun.toString f) ^ "," 
				^ (Int.toString i) ^ ")"

	   fun printX (fps,i,s) = (L.app (print o prFunInt) fps;
				   print (":" ^ (Int.toString i) ^ ":");
				   print ((Term.toString s) ^ "\n"))

	   fun checkOneStep (xss,oldxss) (yss,oldyss) = 
	       (counter := !counter + 1;
		if !counter > maxCount 
		then []
		else 
		    let val xss2 = Rewrite.strengthenRewriteSeqWithIndexAndPosFs R xss
			val yss2 = Rewrite.strengthenRewriteSeqWithIndexAndPosFs R yss
			val oldxss2 = xss @ oldxss
			val oldyss2 = yss @ oldyss
			fun isJoins (x::_, y::_) =  hasEqualTerms (x,y) 
			fun select (uss,vss) = 
			    ListXProd.foldX 
				(fn (us,vs,wss)  => if isJoins (us,vs)
						    then (us,vs)::wss
						    else wss)
				(uss,vss)
				[]

			val ans = select (xss2,oldyss2) 
				  @ select (oldxss2,yss2) 
				  @ select (xss2,yss2)

		    in
			if null ans 
			then checkOneStep (xss2,oldxss2) (yss2,oldyss2)
			else ans
		    end)
       in
	   if hasEqualTerms (s,t)
	   then [([s],[t])]
	   else checkOneStep ([[s]],[]) ([[t]],[])
       end


  (* SN(S/P), P^{-1} \subseteq P $B$r2>Dj(B *)
  (* ->S modulo ->P $B$G(B joinable $B$+(B *)
   fun isModuloJoinableCps S P cps =
       List.all (Rewrite.isModuloJoinable S P) cps
   
  (* SN(S/P), P^{-1} \subseteq P $B$r2>Dj(B *)
  (* CP condition *)

   fun check (S,P) () = 
       let fun deldupl ts = TPS.listItems (TPS.addList (TPS.empty,ts))
       	val cpSS = deldupl (criticalPairs S)
(*	val _ = print (ListUtil.toStringCommaCurly TP.toString cpSS) *)
	val cpPS2 = deldupl (insideCriticalPairs2 (P,S))
	val cpPS = deldupl (cpPS2 @ (outsideCriticalPairs2 (P,S)))
	val cpSP = deldupl (criticalPairs2 (S,P))
    in 
       List.all (fn (u,v) => not (Rewrite.isNormalForm S v)) cpSP
       andalso
       List.all (fn (u,v) => not (Rewrite.isNormalForm S u)) cpPS 
       andalso
       isModuloJoinableCps S P cpSS
       andalso
       isModuloJoinableCps S P cpSP
       andalso
       isModuloJoinableCps S P cpPS2
    end

(**

val Radd = IOFotrs.rdRules 
 [ "add(0,?y) -> ?y", "add(s(?x),?y) -> s(add(?x,?y))",
   "add(?x,0) -> ?x", "add(?x,s(?y)) -> s(add(?x,?y))" ];
val Rtimes = IOFotrs.rdRules 
 [ "times(0,?y) -> 0", "times(s(?x),?y) -> add(times(?x,?y),?y)",
   "times(?x,0) -> 0", "times(?x,s(?y)) -> add(times(?x,?y),?x)" ];
val Rdist = IOFotrs.rdRules 
 [ "times(add(?x,?y),?z) -> add(times(?x,?z),times(?y,?z))",
   "times(?x,add(?y,?z)) -> add(times(?x,?y),times(?x,?z))" ];
val ACadd = IOFotrs.rdRules 
 [ "add(?x,?y) -> add(?y,?x)",
   "add(add(?x,?y),?z) -> add(?x,add(?y,?z))",
   "add(?x,add(?y,?z)) -> add(add(?x,?y),?z)" ];
val ACtimes  = IOFotrs.rdRules 
  ["times(?x,?y) -> times(?y,?x)",
   "times(times(?x,?y),?z) -> times(?x,times(?y,?z))",
   "times(?x,times(?y,?z)) -> times(times(?x,?y),?z)"];

val _ = TimeUtil.profile (check (Radd,ACadd),"add+AC\n")
val _ = TimeUtil.profile (check (Radd@Rtimes@Rdist,ACadd@ACtimes),"times+ACD\n")

***)

   fun confluenceByDepthPreserving rs =
       let fun varPosInTerm vset pos (Var (x,_)) = 
	       if VS.member (vset, x) 
	       then [(length pos,x)]
	       else []
	     | varPosInTerm vset pos (Fun (_,ts,_)) = 
	       LU.mapAppend 
		   (fn i => varPosInTerm vset (pos @ [i+1]) (L.nth (ts,i)))
		   (List.tabulate (length ts, fn x=>x))
	   fun checkDepthPrevMaxRLeqMinL (l,r) =  
	       let val vset = VS.intersection (varSetInTerm l, varSetInTerm r)
		   val lv = varPosInTerm vset [] l
		   val rv = varPosInTerm vset [] r
		   fun refineLv [] zs = zs
		     | refineLv ((i,x)::ys) zs =
		       refineLv (L.filter (fn (j,y) => not (Var.equal (x,y)) 
						       orelse j <= i) ys)
				((i,x)::(L.filter (fn (j,y) => not (Var.equal (x,y)) 
							       orelse j <= i) zs))
		   fun checkVar [] _ = true
		     | checkVar ((i,x)::ys) zs = 
		       L.all (fn (j,z) => not (Var.equal (x,z))
					  orelse i >= j) zs
		       andalso checkVar ys zs
	       in (checkVar (refineLv lv []) rv)
	       end
	   fun checkStrongDepthPrev96 (l,r) = 
	       let fun subcheck [] = true
		     | subcheck ((i,x)::ys) =
		       L.all (fn (j,y) => not (Var.equal (x,y)) orelse i = j) ys
	       in subcheck (varPosInTerm (varSetInTerm l) [] l)
	       end
	   fun checkStrong98 rs = 
	       let val _ = debug (fn _ => (print "check all x in V(r), max {|p| | r/p = x } ";
					   print "<= min {|q| | l/q = x}"))
		   val ans2 = L.all checkDepthPrevMaxRLeqMinL rs 
		   val _ = debug (fn _ => print (if ans2 then " (yes)\n" else " (no)\n"))
	       in ans2
	       end
	   fun checkStrong96 rs = (* assuming that checkStrong98 is true *)
	       let val _ = debug (fn _ => print "check all x in V(l), |p| = |q| if l/p = l/q = x")
		   val ans1 = L.all checkStrongDepthPrev96 rs
		   val _ = debug (fn _ => print (if ans1 then " (yes)\n" else " (no)\n"))
	       in ans1
	       end

	   fun filterMinimal [] ans = LU.eliminateDuplication' Pos.equal ans
	     | filterMinimal (p::rest) ans = 
	       let val rest' = L.filter (fn q => not (Pos.isPrefix p q)) rest
		   val ans' = L.filter (fn q => not (Pos.isPrefix p q)) ans
	       in filterMinimal rest' (p::ans')
	       end

	   fun parallelReductsByReplace R (t as (Var _))  = 
	       t::(L.mapPartial (fn (l,r) => if Term.equal (l,t) then SOME r else NONE) R)
	     | parallelReductsByReplace R (t as (Fun (f,ts,ty))) = 
	       let val ih = L.map (fn ss => Fun (f,ss,ty))
				  (LU.allCombinations (map (parallelReductsByReplace R) ts))
		   val new = L.mapPartial (fn (l,r) => if Term.equal (l,t) then SOME r else NONE) R
	       in ih @ new
	       end

	   fun height (Var _) = 0
	     | height (Fun (_,[],_)) = 0
	     | height (Fun (_,ts,_)) = 1 + (L.foldr (fn (t,j) => Int.max (height t,j)) 0 ts)

	   fun checkConditionIofGOO98 rs outcps = 
	       let val rs' = Trs.renameRules rs
		   val _ = debug (fn _ => print "R (after renaming)\n")
		   val _ = debug (fn _ => print (Trs.prRules rs'))
		   val rs2 = rs' @ (L.map (fn (l,r) => (r,l)) rs')
		   fun check ((i,j),_) =
		       let val (alpha,beta) = L.nth (rs',i)
			   val _ = debug (fn _ => print "check a new strongly overlapping pair\n")
			   val _ = debug (fn _ => print ("alpha = " ^ (Term.toString alpha) ^ ", "))
			   val _ = debug (fn _ => print ("beta = " ^ (Term.toString beta) ^ "; "))
			   val (alpha',beta') = L.nth (rs',j)
			   val _ = debug (fn _ => print ("alpha' = " ^ (Term.toString alpha') ^ ", "))
			   val _ = debug (fn _ => print ("beta' = " ^ (Term.toString beta') ^ "; "))
			   val R' = let val OccXAlpha = Term.varPositionsInTerm alpha
					val OccXAlpha' = Term.varPositionsInTerm alpha'
					val minpos = filterMinimal (OccXAlpha @ OccXAlpha') []
					val _ = debug (fn _ => print "min positions:")
					val _ = debug (fn _ => print (LU.toStringCommaCurly Pos.toString minpos))
					val _ = debug (fn _ => print "\n")
				    in L.mapPartial
					   (fn u => case (Term.subterm u alpha, Term.subterm u alpha') of
							(SOME l',SOME r') => 
							if not (null (Term.varListInTerm l'))
							   andalso not (null (Term.varListInTerm r'))
							then SOME (l',r')
							else NONE
						      | _ => (print "checkConditionIofGOO98\n";
							      raise CrError))
					   minpos
				    end
			   val _ = debug (fn _ => print "R'\n")
			   val _ = debug (fn _ => print (Trs.prRules R'))
		       in ((height beta) <= Int.max (height alpha', height beta')
			   andalso
			   let
			       val redBeta = parallelReductsByReplace R' beta
			       val _ = debug (fn _ => print ("-|-> reducts of beta by R': " 
					      ^ (LU.toStringCommaCurly Term.toString redBeta)
					      ^ "\n"))
			       val redBeta' = Rewrite.parallelOneStepReducts rs beta'
			       val _ = debug (fn _ => print ("-|-> reducts of beta' by R: " 
					      ^ (LU.toStringCommaCurly Term.toString redBeta')
					      ^ "\n"))
			       val candForM1 = LU.intersection' Term.equal (redBeta,redBeta')
			       val candForM2 = L.filter
						   (fn t => LU.member' Term.equal
								       beta'
								       (Rewrite.parallelOneStepReducts rs t))
						   redBeta
			       (* val _ = debug (fn _ => print "check V(M) cap V(beta) = emptyset") *)
			       fun checkcondM M = 
				   LU.disjoint' Var.equal (Term.varListInTerm M,  Term.varListInTerm beta)
			   in case L.find checkcondM (candForM1 @ candForM2) of
				  SOME M => let val _ = debug (fn _ => print ("closure condition is satisfied (M = " ^ (Term.toString M) ^ ")\n"))
					    in true end
				| NONE => let val _ = debug (fn _=> print "closure condition is not satisfied\n")
					  in false end
			   end)
			  orelse
			  ((height beta') <= Int.max (height alpha, height beta)
			   andalso
			   let
			       val redBeta1 = Rewrite.parallelOneStepReducts rs beta
			       val _ = debug (fn _=> print ("-|-> reducts of beta by R: "
					      ^ (LU.toStringCommaCurly Term.toString redBeta1)
					      ^ "\n"))
			       val R2' = L.map (fn (l,r) => (r,l)) R'
			       val redBeta' = parallelReductsByReplace R2' beta'
			       val _ = debug (fn _=> print ("<-|- reducts of beta' by R': " 
					      ^ (LU.toStringCommaCurly Term.toString redBeta')
					      ^ "\n"))
			       val candForM1 = LU.intersection' Term.equal (redBeta1,redBeta')
			       val candForM2 = L.filter
						   (fn t => LU.member' Term.equal
								       beta
								       (Rewrite.parallelOneStepReducts rs t))
						   redBeta'
			       (* val _ = debug (fn _=> print "check V(M) cap V(beta') = emptyset") *)
			       fun checkcondM M = 
				   LU.disjoint' Var.equal (Term.varListInTerm M,  Term.varListInTerm beta')
			   in case L.find checkcondM (candForM1 @ candForM2) of
				  SOME M => let val _ = debug (fn _=> print ("closure condition is satisfied (M = " 
									     ^ (Term.toString M) ^ ")\n"))
					    in true end
				| NONE => let val _ = debug (fn _ => print "closure condition is not satisfied\n")
					  in  false end
			   end)
		       end
	       in L.all check outcps
	       end
       in if checkStrong98 rs
	  then let val _ = debug (fn _ => print ("depth-preserving (of GOO98)\n"))
	       in if checkStrong96 rs
		     andalso let val _ = debug (fn _ => print ("depth-preserving (of GOO96)\n"))
				 val _ = debug (fn _ => print ("check non-omega-overlapping"))
				 val omegaNO = InfTerm.omegaNonOverlappingTrs rs
				 val _ = debug (fn _ => print (if omegaNO then " (yes)\n" else " (no)\n"))
			     in omegaNO
			     end
	       then true
	       else let 
		       val linrs = L.map (fn (l,r) => (Subst.linearize l,r)) rs
		       val incps = insideCriticalPairs linrs
		       val _ = debug (fn _ => print ("inside CPs after linearization of lhs:\n"))
		       val _ = debug (fn _ => print (LU.toStringCommaLnSquare (IOFotrs.prEq) incps))
		   in if not (null incps)
		      then let val _ = debug (fn _ => print ("not stongly root-overlapping\n"))
			   in false 
			   end
		      else let val _ = debug (fn _ => print ("stongly root-overlapping\n"))
			       val _ = debug (fn _ => print ("check condition I of GOO98...\n"))
			       val outcps = outsideCriticalPeaksWithIndex linrs
			       val _ = debug (fn _ => print ("outside CPs after linearization of lhs:\n"))
			       val _ = debug (fn _ => print (LU.toStringCommaLnSquare 
								 (IOFotrs.prEq) 
								 (L.map (fn (_,(_,l,r))=> (l,r)) outcps)))
			   in if checkConditionIofGOO98 rs outcps
			      then (if !runDebug then print "...condition I holds\n" else ();
				    if !runDebug then print "root-E-closed or non-E-overlapping\n" else ();
				    true)
			      else (if !runDebug then print "...condition I does not hold\n" else (); false)
			   end
		   end
	       end
	  else false
       end


   (* conditional critical pairs *)
   local
       fun cpsub (alpha,beta,cond1) C (t,delta,cond2) =
	   case (cpsubWhole (alpha,beta,cond1) C (t,delta,cond2)) of
		   SOME ans => ans::(cpsubProperSubterm (alpha,beta,cond1) C (t,delta,cond2))
		 | NONE => cpsubProperSubterm (alpha,beta,cond1) C (t,delta,cond2)
       and cpsubWhole (alpha,beta,cond1) C (t,delta,cond2) =
	   case t of 
	       Var _ => NONE
	     | Fun (f,ts,ty) 
	       => case (unify t alpha) of
		      SOME sigma => let (* val _ = println (Subst.toString sigma) *)
					(* val _ = println (Term.toString (applySubst sigma (C alpha))) *)
					(* val _ = println (Term.toString (applySubst sigma (C beta))) *)
					(* val _ = println (Term.toString (applySubst sigma delta)) *)
				    in
					SOME (applySubst sigma (C alpha), 
					      applySubst sigma (C beta), 
					      applySubst sigma delta,
					      L.map (fn (x,y) => (applySubst sigma x,applySubst sigma y)) (cond1 @ cond2))
				    end
		    | NONE => NONE
       and cpsubProperSubterm (alpha,beta,cond1) C (t,delta,cond2) =
	   case t of 
	       Var _ => []
	     | Fun (f,ts,ty) 
	       => cpsubArguments (alpha,beta,cond1) 
				 (fn xs => C (Fun (f,xs,ty)))
				 (ts,delta,cond2) 
       and cpsubArguments (alpha,beta,cond1) C ([],delta,cond2) = []
	 | cpsubArguments (alpha,beta,cond1) C (t::ts,delta,cond2) =
	   List.@(cpsub (alpha,beta,cond1) (fn x => C (x::ts)) (t,delta,cond2),
		  cpsubArguments (alpha,beta,cond1) (fn xs => C (t::xs)) (ts,delta,cond2))

   in
   fun condCriticalPeaks crules =
       let val crules' = Ctrs.renameRules crules
	   (* val _ = print (Ctrs.prRules crules') *)
           val nums = List.tabulate (List.length crules, fn i => i)   (* 0,...,n-1 *)
       in LU.mapAppend
	      (fn n => LU.mapAppend
			   (fn m => if n = m
				    then let val crule = List.nth (crules,n)
					     val [crule1,crule2] = Ctrs.renameRules [crule,crule]
					 in (* $B<+J,<+?H$K$D$$$F$O(B proper subterm $B$G$N=E$J$j$@$1(B also in CTRSs *)
					     cpsubProperSubterm crule1 (fn x => x) crule2
					 end
				    else cpsub (List.nth (crules',m)) (fn x => x) (List.nth (crules',n)))
			   nums)
	      nums
       end

   fun condCriticalPairs crules = L.map (fn (x,y,z,cond) => (y,z,cond)) (condCriticalPeaks crules)

   fun condInsideCriticalPeaks crules = 
       let val crules' = Ctrs.renameRules crules
	   val nums = List.tabulate (List.length crules, fn i => i)   (* 0,...,n-1 *)
       in 
 	   L.@(LU.mapAppend 
		   (* $BAj0[$J$k5,B'$K$D$$$F$O(B renaming $BI,MW$J$7(B *)
		   (fn n => LU.mapAppend
				(fn lr => cpsubProperSubterm lr (fn x => x) (List.nth (crules',n)))
				(LU.exceptNth (crules',n)))
		   nums,
	       (* $B<+J,<+?H$K$D$$$F$O(B renaming $B$r$7$F$+$i(B *)
	       LU.mapAppend (fn crule => let val [crule1,crule2] = Ctrs.renameRules [crule,crule]
				   in cpsubProperSubterm crule1 (fn x => x) crule2
				   end)
			 crules')
       end

   fun condInsideCriticalPairs crules = L.map (fn (x,y,z,cond) => (y,z,cond)) (condInsideCriticalPeaks crules)

   fun condOutsideCriticalPeaks crules =
       let val crules' = Ctrs.renameRules crules
	   val nums = List.tabulate (List.length crules, fn i => i)   (* 0,...,n-1 *)
       in
 	   LU.mapAppend
		   (fn n => List.mapPartial
				(fn crule => cpsubWhole crule (fn x => x) (List.nth (crules',n)))
				(LU.exceptNth (crules',n)))
		   nums
       end

   fun condOutsideCriticalPairs crules = L.map (fn (x,y,z,cond) => (y,z,cond)) (condOutsideCriticalPeaks crules)

   end

(*** Toyama-Oyamaguchi's Weigth Decreasing Joinability ***)

(* $BEy<0CV$-49$((B *)
(*  { s | t|p = u & s=t[v]_p or t|p = v & s=t[u]_p } *)
fun oneStepEqReplacements (u,v) t = 
    if Term.equal (t, u) then [v]
    else if Term.equal (t,v) then [u]
    else case t of 
	     Fun (f,ts,ty) => L.map (fn us => Fun (f,us,ty)) (oneStepEqReplacementsList (u,v) ts)
	   | _  => []
and oneStepEqReplacementsList eq [] = []
  | oneStepEqReplacementsList eq (t::ts) = 
    L.map (fn t' => t'::ts) (oneStepEqReplacements eq t)
    @ L.map (fn ts' => t::ts') (oneStepEqReplacementsList eq ts)


(* cond$B$N>r7o$r(B1$B2s;H$C$?Ey<0CV$-49$(A4It(B *)
(*  { (cond\{u=v},s) | t|p = u & s=t[v]_p or t|p = v & s=t[u]_p for some u = v \in cond }*)
fun oneStepEqReplacementsAll ([],t) = []
|   oneStepEqReplacementsAll ((u,v)::cond,t) = 
    L.map (fn s => (cond,s)) (oneStepEqReplacements (u,v) t)
    @ L.map (fn (eqs,s) => ((u,v)::eqs, s)) (oneStepEqReplacementsAll (cond,t))

(* cond$B$N>r7o$rJ#?t2s(B(>=0$B2s(B)$B;H$C$?Ey<0CV$-49$(@hA4It(B *)
fun multiStepEqReplacementsAllSub (cond,t) = 
    let (* val _ = println ("do multi eq repl for " ^ (Term.toString t)) *)
	val reducts = oneStepEqReplacementsAll (cond,t)
    in (cond,t) :: LU.mapAppend multiStepEqReplacementsAllSub reducts
    end

(* (cond,u) $B$G!$F1$8(Bu$B$G>r7o(Bcond $B$,>/$J$$$b$N$d=EJ#$r$J$/$9(B *) 
fun removeRedundant ans = 
    let fun isRedundant (eqs,u) =
	    L.exists (fn (eqs2,u2) =>
			 let (* val _ = println (Term.toString u) *)
			     (* val _ = println (Term.toString u2) *)
			     (* val _ = println (LU.toStringCommaCurly TermPair.toString eqs) *)
			     (* val _ = println (LU.toStringCommaCurly TermPair.toString eqs2) *)
			     val result = Term.equal (u,u2)
					  andalso null (LU.differenceByOne' TermPair.equal (eqs,eqs2))
					  andalso not (null (LU.differenceByOne' TermPair.equal (eqs2,eqs)))
			     (* val _ = if result then println "true" else println "false" *)
			 in result
			 end)
		     ans
	val ans0 = L.filter (not o isRedundant) ans 
    in LU.eliminateDuplication' (fn ((_,u1),(_,u2)) => Term.equal (u1,u2)) ans0
    end

(* cond$B$N>r7o$rJ#?t2s(B(>=0$B2s(B)$B;H$C$?Ey<0CV$-49$(@hA4It!'=EJ#$dM>7W$J>r7o$r;H$C$?$b$N$r:o=|(B *)
fun multiStepEqReplacementsAll (cond,t) = removeRedundant (multiStepEqReplacementsAllSub (cond,t))

(* rank 0 $B$N(B 1$B%9%F%C%W$N>r7oIU$-=q$-49$(5,B'$r%k!<%H$GE,MQ$7$?7k2LA4It(B *)
(* rank 0 $B!%!%!%(B $B>r7oIt$O(B eqRepacements $B$N$_(B *)
fun rankZeroRootStepCondRule (l,r,eqs) (cond,t) = 
    let (* val _ = println ("rootRewrite: " ^ Term.toString t) *)
	val ans0 = 
	    case Subst.match l t of
		NONE => []
	      | SOME sigma => let val sigmaList = VM.listItemsi sigma
				  val images = L.map (fn (k,v) => v) sigmaList
				  val keys = L.map (fn (k,v) => k) sigmaList
				  val init = Term.mkFunTerm (Fun.fromString "Dummy", images, Sort.null)
				  val obtained = multiStepEqReplacementsAll (cond,init)
				  fun mkSubstFromList xs = L.foldl (fn ((x,t),map) => VM.insert (map,x,t)) VM.empty xs
				  val subList = L.map (fn (eqs,reduct) => (eqs, mkSubstFromList (ListPair.zip (keys,argsOfTerm reduct)))) 
						      obtained
				  val rho = mkSubstFromList (L.map (fn (s,t) => (valOf (varRootOfTerm s), t)) eqs)
				  val l' = Subst.applySubst rho l
			      in L.mapPartial (fn (cond',tau) => case Subst.match l' (Subst.applySubst tau l) of
								     SOME rho => SOME (cond', Subst.applySubst rho r)
								   | NONE => NONE)
					      subList
			      end
    in removeRedundant ans0
    end

(* rank 0 $B$N!$(B1$B%9%F%C%W$N=q$-49$($rE,MQ$7$?7k2LA4It(B *)
fun rankZeroRewriteStepCondRule (l,r,eqs) (cond,t) = 
    let val fromRoot = rankZeroRootStepCondRule (l,r,eqs) (cond,t)
	val fromArgs = case t of 
			   Var _ => []
			 | Fun (f,args,ty) => L.map (fn (eqs,ts) => (eqs,Fun (f,ts,ty))) 
						    (rankZeroRewriteStepCondRuleList (l,r,eqs) (cond,args))
    (* val _ = println (Term.toString t) *)
    in fromRoot @ fromArgs
    end
and rankZeroRewriteStepCondRuleList (l,r,eqs) (cond,[]) = []
  | rankZeroRewriteStepCondRuleList (l,r,eqs) (cond,t::ts) = 
    L.map (fn (cond',t') => (cond', t'::ts)) (rankZeroRewriteStepCondRule (l,r,eqs) (cond,t))
    @ L.map (fn (cond',ts') => (cond', t::ts')) (rankZeroRewriteStepCondRuleList (l,r,eqs) (cond,ts))


fun rankZeroRewriteStepCondRules crules (cond,t) = 
    removeRedundant (LU.mapAppend (fn crule => rankZeroRewriteStepCondRule crule (cond,t)) crules)

exception RankZeroOneStepConvertible of (Term.term * Term.term) list

(* check whether "u sim o <->^=[0] o sim v"  under cond *)
fun rankZeroOneStepConvertibleCondRules crules (u,v,cond) = 
    let (* val _ = println ("Check u sim o <->^=[0] o sim v for u = " 
			    ^ (Term.toString u) ^ " and v = " ^ (Term.toString v))  *)
	(* val _ = println "{ x | u sim x }"  *)
	val U = multiStepEqReplacementsAll (cond,u)
	(* val _ = L.app (fn (eqs,x) => println ((Term.toString x) ^ " ... remaining eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs))) U  *)

	val _ = case L.find (fn (eqs,x) => Term.equal (x,v)) U of
		    SOME (eqs,x) => ((* println ("u sim v with remained eqs:" ^ (Trs.prEqs eqs));  *)
		     raise RankZeroOneStepConvertible eqs)
		  | NONE => ()

	(* val _ = println "{ x | u sim o ->[0] x }"  *)
	val Uz = removeRedundant (LU.mapAppend (rankZeroRewriteStepCondRules crules) U)
	(* val _ = L.app (fn (eqs,x) => println ((Term.toString x) ^ " ... remaining eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs))) Uz  *)

	(* val _ = println "{ x | u sim o ->[0] o sim x }"  *)
	val Uz' = removeRedundant (LU.mapAppend multiStepEqReplacementsAll Uz)
	(* val _ = L.app (fn (eqs,x) => println ((Term.toString x) ^ " ... remaining eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs))) Uz'  *)

	val _ = case L.find (fn (_,x) => Term.equal (x,v)) Uz' of
		    SOME (eqs,x) => ((* println ("u sim o ->0 o sim v with remained eqs:" ^ (Trs.prEqs eqs));  *)
		     raise RankZeroOneStepConvertible eqs)
		  | NONE => ()

	(* val _ = println "{ y | y sim v }"  *)
	val V = multiStepEqReplacementsAll (cond,v)
	(* val _ = L.app (fn (eqs,x) => println ((Term.toString x) ^ " ... remaining eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs))) V  *)

	(* val _ = println "{ y | y [0]<- sim v }"  *)
	val Vz = removeRedundant (LU.mapAppend (rankZeroRewriteStepCondRules crules) V)
	(* val _ = L.app (fn (eqs,x) => println ((Term.toString x) ^ " ... remaining eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs))) Vz  *)

	(* val _ = println "{ y | y sim o [0]<- sim v }"  *)
	val Vz' = removeRedundant (LU.mapAppend multiStepEqReplacementsAll Vz)
	(* val _ = L.app (fn (eqs,x) => println ((Term.toString x) ^ " ... remaining eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs))) Vz'  *)

	val _ = case L.find (fn (_,y) => Term.equal (y,u)) Vz' of
		    SOME (eqs,y) => ((*println ("u sim o [0]<- o sim v with remained eqs:" ^ (Trs.prEqs eqs));   *)
		     raise RankZeroOneStepConvertible eqs)
		  | NONE => ()
    in NONE
    end
    handle RankZeroOneStepConvertible eqs => SOME eqs

(* check whether "u_i sim o <->^=[0] o sim v_i for all i"  under cond *)
fun rankZeroOneStepConvertibleCondRulesList crules cond [] = SOME cond
  | rankZeroOneStepConvertibleCondRulesList crules cond ((u,v)::problem) =
    (case rankZeroOneStepConvertibleCondRules crules (u,v,cond) of
	 NONE => NONE
       | SOME eqs => rankZeroOneStepConvertibleCondRulesList crules eqs problem)


(* (l,r,eqs) $B$K$h$k(B  cond |- u <->[1] v (rank 1 $B%9%F%C%W=q$-49$((B) $B$H$J$k$+(B *)
fun checkRankOneRewriteStep crules (l,r,eqs) (u,v,cond) = 
    let (* val _ = println ("Check u <->[1] v for u = " 
			    ^ (Term.toString u) ^ " and v = " ^ (Term.toString v)) *)
	fun check (l,r,eqs) (u,v,cond) = 
	    let val pat  = Term.mkFunTerm (Fun.fromString "Dummy", [l,r], Sort.null)
		val term = Term.mkFunTerm (Fun.fromString "Dummy", [u,v], Sort.null)
	    (* val _ = println ("pat: " ^ (Term.toString pat)) *)
	    (* val _ = println ("term: " ^ (Term.toString term)) *)
	    in case Subst.match pat term of
		   SOME sigma => let val problem = L.map (fn (u,v) => (Subst.applySubst sigma u, Subst.applySubst sigma v)) eqs
				 (* val _ =  println ("Matched!") *)
				 (* val _ =  println ("  rule: " ^ (Ctrs.prRule (l,r,eqs))) *)
				 (* val _ =  println ("  conditions: " ^ (LU.toStringComma Trs.prEq problem)) *)
				 in case rankZeroOneStepConvertibleCondRulesList crules cond problem of
					SOME eqs' => ((* println ("u <->[1] v with remained eqs: " ^ (LU.toStringCommaSquare Trs.prEq eqs'));*)
					 true)
				      | NONE => false
				 end
		 | NONE => false
	    end
    in check (l,r,eqs) (u,v,cond) 
       orelse check (l,r,eqs) (v,u,cond) 
       orelse (case (funRootOfTerm u, funRootOfTerm v) of
		   (SOME f, SOME g) => if Fun.equal (f,g) 
				       then checkRankOneRewriteStepList crules (l,r,eqs) (argsOfTerm u, argsOfTerm v, cond)
				       else false
		 | _ => false)
    end
and checkRankOneRewriteStepList crules (l,r,eqs) ([],[],cond) = true
  | checkRankOneRewriteStepList crules (l,r,eqs) (u::us,v::vs,cond) = 
    if Term.equal (u,v) 
    then checkRankOneRewriteStepList crules (l,r,eqs) (us,vs,cond)
    else if LP.all Term.equal (us,vs)
    then checkRankOneRewriteStep crules (l,r,eqs) (u,v,cond)
    else false



exception WeightDecreasingJoinable

fun checkWeightDecreasingCondition crules (u,v,cond) = 
    let val _ = debug (fn () => print "critical pair: ")
	val _ = debug (fn () => println (Ctrs.prEq (u,v,cond)))

        (*** check Condition I ***)
	val _ = case rankZeroOneStepConvertibleCondRules crules (u,v,cond) of
		    SOME _ => (debug (fn () => println "Weight Decreasing Joinable (Condition I)");
			       raise WeightDecreasingJoinable)
		  | NONE => (debug (fn () => println "...Condition I failed");NONE)

        (*** check Condition III ***)
	val _ = if L.exists (fn (l,r,eqs) => checkRankOneRewriteStep crules (l,r,eqs) (u,v,cond)) crules
		then (debug (fn () => println "Weight Decreasing Joinable (Condition III)");
		      raise WeightDecreasingJoinable)
		else (debug (fn () => println "...Condition III failed"))

        (*** check Condition II ***)
	val _ = if L.exists 
		       (fn (cond',v') => L.exists (fn (l,r,eqs) => checkRankOneRewriteStep crules (l,r,eqs) (u,v',cond')) crules)
		       (multiStepEqReplacementsAll (cond,v))
		   orelse L.exists (fn (cond',u') => 
				       isSome (rankZeroOneStepConvertibleCondRules crules (u',v,cond')))
				   (rankZeroRewriteStepCondRules crules (cond,u))
		then if L.exists 
			    (fn (cond',u') => L.exists (fn (l,r,eqs) => checkRankOneRewriteStep crules (l,r,eqs) (v,u',cond')) crules)
			    (multiStepEqReplacementsAll (cond,u))
			orelse L.exists (fn (cond',v') => 
					    isSome (rankZeroOneStepConvertibleCondRules crules (v',u,cond')))
					(rankZeroRewriteStepCondRules crules (cond,v))
		     then (debug (fn () => println "Weight Decreasing Joinable (Condition II)");
			   raise WeightDecreasingJoinable)
		     else debug (fn () => println "...Condition II(R) failed")
		else debug (fn () => println "...Condition II(L) failed")

    in (debug (fn () => println "...checking Weight Decreasing Joinability failed");  false)
    end
    handle WeightDecreasingJoinable => (debug (fn () => println "...is Weight Decreasing Joinable");  true)


   fun isWeightDecreasingJoinableTrs rs = 
       let val _ = debug (fn () => println "Check weight decreasing condition for LR-separated conditional linearization")
	   val _ = debug (fn () => println "LR-separated conditional linearlization:")
	   val crules =  Ctrs.condTOLinearizationRules rs
	   val _ = debug (fn () => print (Ctrs.prRules crules))

	   val _ = debug (fn () => println "conditional critical pairs:")
	   val ccp = condCriticalPairs crules
	   val _ = debug (fn () => print (Ctrs.prEqs ccp))
    in L.all (checkWeightDecreasingCondition crules) ccp
    end




   end (* of local *)


   end; (* of structure Cr *)

