(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/cr_com.sml
 * description: confluence check by commutative decomposition
 * author: YOSHIDA Junichi
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature CR_COM = 
   sig
   val proveConfluenceByCommutativeDecomposition:
	   ((Term.term * Term.term) list -> bool) 
	   -> (Term.term * Term.term) list 
	   ->  Cr.ConfluenceResult

   val minimalDecompositions: 
       (Term.term * Term.term) list
       -> int list list list

   val tryCommutativeDecomposition:
       (Term.term * Term.term) list
       -> ((Term.term * Term.term) list -> Cr.ConfluenceResult) 
	  * ((Term.term * Term.term) list -> Cr.ConfluenceResult) 
       -> Cr.ConfluenceResult

   val useNonMinimal: bool ref
end;

structure CrCom : CR_COM = 
   struct

   local 
       open Term
       open Cr
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure IS = IntSet
       structure L = List
       structure LP = ListPair
       structure TS = TermSet
       structure ILM = IntListMap2
       fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs
       fun exceptNth n [] = []
	 | exceptNth n (x::xs) = 
	   if n = 0 
	   then xs
	   else (x::(exceptNth (n-1) xs))
       fun prFunSet set = "{" ^ (PrintUtil.prSeq Fun.toString (FS.listItems set)) ^ "}"
       fun prIntSet set = "{" ^ (PrintUtil.prSeq Int.toString (IS.listItems set)) ^ "}"
       fun prIntList list = "{" ^ (PrintUtil.prSeq Int.toString list) ^ "}"
       open PrintUtil
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   val useNonMinimal = ref true: bool ref

   (***********************)
   (******* $BJd=u4X?t(B ********)
   (***********************)

   fun member x ys = List.exists (fn y => y = x) ys

   fun union [] ys = ys
     | union (x::xs) ys =
       if member x ys then union xs ys else (union xs (x::ys))

   fun listToSet xs = union xs []
		      
   fun unionall xss = 
       List.foldl (fn (xs,ys) => union xs ys)  [] xss
       
   fun subseteq [] ns2 = true
     | subseteq (n1::ns1) ns2 =
       if member n1 ns2 then subseteq ns1 ns2
       else false

   fun setequal s1 s2 = (subseteq s1 s2) andalso (subseteq s2 s1)

   fun filter p xs = List.filter p xs

   (* [X1,...,XN] << [Y1,...,YM] ?
     <=> for any Xi,  there exists Yj such that Xi subseteq  Yj *)
   fun setsOfSubsets xss yss =
       List.all 
	   (fn xs => List.exists (fn ys => subseteq xs ys) yss)
	   xss

   (* $B6K>.MWAG$N$_$r<h$j=P$9(B *)
   fun leaveMinimals smallerOrEq xss = 
       let
	   fun leaveMinimalSub [] mins = mins
	     | leaveMinimalSub (xs::xss) [] =  leaveMinimalSub xss [xs]
	     | leaveMinimalSub (xs::xss) (ys::yss) = 
	       if (smallerOrEq xs ys)
	       then leaveMinimalSub xss (ys::yss)
	       else if (smallerOrEq ys xs)
	       then leaveMinimalSub (xss @ yss) [ys]
	       else leaveMinimalSub xss (xs::ys::yss)
       in
	   leaveMinimalSub xss []
       end

   (*  equivalence relation [(a1,a2),(b1,b2),(a2,a3),...] $B$+$i!$(B
		   equivalence class [ [a1,a2,a3], [b1,b2], ... ] $B$r7W;;$9$k(B  *)
   fun makeEqClass nps = 
       let 
	   (* nsh $B$NMWAG$N$&$A(B n0 $B$r4^$`$b$N(B(1$B$D0J2<(B)$B$r(B n0::nsh $B$H9g$o$;$k(B *)
	   fun assoc_sub n0 nsh [] = [(n0::nsh)]
	     | assoc_sub n0 nsh (ns::nss) =
	       if member n0 ns 
	       then (nsh @ ns)::nss
	       else ns::(assoc_sub n0 nsh nss)
		    
	   (* (n1,n2) $B$H(B disjoint $B$J%j%9%H$,M?$($i$l$?$H$-$K!$(B
		      n1 $B!A(B n2 $B$H$7$?(B $B%j%9%H$XJQ49$9$k(B *)
	   fun assoc (n1,n2) [] = [[n1,n2]]
	     | assoc (n1,n2) (ns::nss) =
	       if member n1 ns 
	       then if member n2 ns 
		    then ns::nss
		    else assoc_sub n2 ns nss
	       else if member n2 ns 
	       then assoc_sub n1 ns nss
	       else ns :: (assoc (n1,n2) nss)

	   fun makeEqClassSub [] ans = ans
	     | makeEqClassSub (np::nps) ans = 
	       makeEqClassSub nps (assoc np ans)
       in 
	   makeEqClassSub nps []
       end

   fun delete [] ys = []
     | delete (x::xs) ys = if member x ys 
			   then delete xs ys
			   else x::(delete xs ys)
				
   fun ucons x ys = if member x ys then ys else x::ys

   fun orderPair (n1,n2) = if n1 <= n2 then (n1,n2) else (n2,n1)

  (* xs => { (i,j) |  i,j \in xs, i < j } *)
   fun makeOrderedIntPairs [] = []
     | makeOrderedIntPairs (n::ns) =
       (map (fn n0 => orderPair (n,n0)) ns) @ (makeOrderedIntPairs ns)

   (* powerlist: [0,1,2] 
       =>  [[],[0],[1],[2],[0,1],[0,2],[1,2],[0,1,2]]  *)
   fun powerlist [] = [[]]
     | powerlist (x::xs) = 
       let val yss = powerlist xs
       in yss @ (L.map (fn ys => x::ys) yss)
       end

   (* properPowerlist: [0,1,2] 
       =>  [[0],[1],[2],[0,1],[0,2],[1,2]]  *)
   fun properPowerlist [] = []
     | properPowerlist (x::[]) = []
     | properPowerlist (x::xs) = 
       let val yss = properPowerlist xs
       in [x]::
	  ((L.mapPartial (fn ys => if ys = xs then NONE else SOME (x::ys)) yss)
	   @ (yss @ [xs]))
       end

   (* powerlistWithComplement: 
       [0,1,2]  => 
            [  ([],[0,1,2]),
	       ([0],[1,2]),([1],[0,2]),([2],[0,1]),
               ([0,1],[2]),([0,2],[1]),([1,2],[0]),
	       ([0,1,2],[])   ]                *)
   fun powerlistWithComplement [] = [([],[])]
     | powerlistWithComplement (x::xs) = 
       let val yss = powerlistWithComplement xs
       in (L.map (fn (ys,zs) => (x::ys,zs)) yss)
	  @ (L.map (fn (ys,zs) => (ys,x::zs)) yss)
       end

   (* nonNullPowerlistWithComplement: 
       [0,1,2]  => 
            [  ([0],[1,2]),([1],[0,2]),([2],[0,1]),
               ([0,1],[2]),([0,2],[1]),([1,2],[0]) ]  *)
   fun nonNullPowerlistWithComplement [] = []
     | nonNullPowerlistWithComplement (x::xs) = 
       let val yss = powerlistWithComplement xs
       in (L.mapPartial (fn (ys,zs) => if null zs then NONE else SOME (x::ys,zs)) yss)
	  @ (L.mapPartial (fn (ys,zs) => if null ys then NONE else SOME (ys,x::zs)) yss)
       end

   fun concat [] = []
     | concat ([] :: xss) = concat xss
     | concat ((x::xs) :: xss) = x :: (concat (xs :: xss))

   fun isDisjoint xs ys = List.all (fn x => not (member x ys)) xs

   fun takeEqClosure [] = []
     | takeEqClosure ps =
       let
           fun strict_sub (f0,g0) [] ps0 = (f0,g0)::ps0
	     | strict_sub (f0,g0) ((f,g)::ps) ps0 =
	       if (g0 = f)
	       then strict_sub (f0,g0) ps ((orderPair (f0,g))::((f,g)::ps0))
	       else if (f0 = g)
	       then strict_sub (f0,g0) ps ((orderPair (f,g0))::((f,g)::ps0))
	       else if (f0 = f)
	       then strict_sub (f0,g0) ps ((orderPair (g0,g))::((f,g)::ps0))
	       else if (g0 = g)
	       then strict_sub (f0,g0) ps ((orderPair (f0,f))::((f,g)::ps0))
	       else strict_sub (f0,g0) ps ((f,g)::ps0)

           fun strict_step psb = 
               listToSet (strict_sub (hd psb) (tl psb) [])

           val results = strict_step ps
       in
           if (setequal ps results) then ps else takeEqClosure results
       end

   (**********************************)
   (******* $B4m81BP$N9gN.>r7o$N7W;;(B ********)
   (**********************************)
       
   (* inside CP <P,Q> where P <-_{m} X ->_{n}  Q $B$K$D$$$F!$(B *)
   (* P -o->_A Y <-o-_B Q $B$H$J$k=89gBP(B (A,B) $B$K$D$$$F(B (A,B,false) $B$N%j%9%H(B *)
   fun mem_develop_out2 (ps1,ps2) =
       let (* $B6K>.2r$N%j%9%H$r5a$a$k(B *)
           fun subset_cons (set1,set2) [] ans = (set1,set2,false)::ans
             | subset_cons (set1,set2) ((set3,set4,b)::xs) ans = 
               if IS.isSubset (set1,set3)           (* (set1,set2) $B$,6K>.2r$K4^$^$l$k>l9g(B *)
		  andalso IS.isSubset (set2,set4)
	       then subset_cons (set1,set2) xs ans
	       else if IS.isSubset (set3,set1)      (* (set1,set2) $B$r4^$`6K>.2r$,$"$k>l9g(B *)
		       andalso IS.isSubset (set4,set2)
	       then ((set3,set4,b)::xs) @ ans
	       else  (* (set1,set2) $B$,(B $B6K>.2r$N(B1$B$D$H(B incomparable $B$N>l9g(B *)
		   subset_cons (set1,set2) xs ((set3,set4,b)::ans) 
       in
	   ListXProd.foldX
	       (fn ((set1,t1),(set2,t2),xs) => if Term.equal (t1,t2) 
					       then subset_cons (set1,set2) xs []
					       else xs)
	       (ps1,ps2)
	       []
       end

   (* inside CP <P,Q> where P <-_m X ->_n  Q $B$K$D$$$F!$(B *)
   (*  P -o->_A Y <-o-_B Q $B$H$J$k=89gBP(B (A,B) $B$K$D$$$F(B,  (A\{n},B\{m},b) $B$rJV$9(B *)
   (* where b:  B <> emptyset $B$+(B ... $B8e$N%A%'%C%/$KI,MW(B *)

   fun mem_develop_in2 (m,n) (ps1,ps2) =
       let
           fun subset_cons1 set1 [] ans = (set1,IS.empty,false)::ans
             | subset_cons1 set1 ((set3,set4,b)::xs) ans =
               if IS.isSubset (set1,set3)
	       then subset_cons1 set1 xs ans
	       else if IS.isSubset (set3,set1) 
		       andalso (not b)
	       then  ((set3,set4,b)::xs) @ ans
	       else subset_cons1 set1 xs ((set3,set4,b)::ans)

           fun subset_cons2 (set1,set2) [] ans = (set1,set2,true)::ans
             | subset_cons2 (set1,set2) ((set3,set4,b)::xs) ans  =
               if IS.isSubset (set1,set3)
		  andalso IS.isSubset (set2,set4) 
		  andalso b
	       then
                   subset_cons2 (set1,set2) xs ans
	       else if IS.isSubset (set3,set1)
		       andalso IS.isSubset (set4,set2) 
	       then ((set3,set4,b)::xs) @ ans
	       else
		   subset_cons2 (set1,set2) xs ((set3,set4,b)::ans)

           fun subset_cons (set1,set2) xs ans =
               if IS.isEmpty set2 
	       then subset_cons1 (IS.difference (set1,IS.singleton n)) xs ans
	       else subset_cons2 (IS.difference (set1,IS.singleton n), 
				  IS.difference (set2,IS.singleton m)) xs ans
       in
	   ListXProd.foldX
	       (fn ((set1,t1),(set2,t2),xs) => if Term.equal (t1,t2) 
					       then subset_cons (set1,set2) xs []
					       else xs)
	       (ps1,ps2)
	       []
       end


   (* $B4m81BP$N9gN.>r7o$N7W;;(B *)
   fun makeInsideCPConditions rs =
       let val icps = insideCriticalPairsBetweenDifferentRulesWithIndex rs
       in List.map (fn ((m,n),(c1,c2)) =>
		       let 
			   val _ = print "Inside Critical Pair: <"
			   val _ = print ((Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">")
			   val _ = print (" by Rules <" ^ (Int.toString m)
					  ^ ", " ^ (Int.toString n) ^ ">\n")
			   val reds1 = Rewrite.developOneStepReductsWithIndex rs c1
			   val reds2 = Rewrite.developOneStepReductsWithIndex rs c2
			   val _ = print "develop reducts from lhs term...\n"
			   val _ = List.app (fn (set,t) => 
						(print ("<" ^ (prIntSet set) ^ ", ");
						 print ((Term.toString t) ^ ">");
						 print "\n"))
					    reds1
			   val _ = print "develop reducts from rhs term...\n"
			   val _ = List.app (fn (set,t) => 
						(print ("<" ^ (prIntSet set) ^ ", ");
						 print ((Term.toString t) ^ ">");
						 print "\n"))
					    reds2
			    val constraints = mem_develop_in2 (m,n) (reds1,reds2)
		       in
			   ((m,n), 
			    List.map 
				(fn (set1,set2,b)=> (IS.listItems set1, IS.listItems set2, b))
				constraints)
		       end)
	  icps
       end


   fun makeOutsideCPConditions rs =
       let (* val ocps = outsideCriticalPairsWithIndex rs *)
	   val ocps = outsideCriticalPairsInOnesideWithIndex rs  (* $BBP>NE*$J(BCP$B$N$&$A!$JRB&$@$1(B *)
       in List.map (fn ((m,n),(c1,c2)) =>
		       let 
			   val _ = print "Outside Critical Pair: <"
			   val _ = print ((Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">")
			    val _ = print (" by Rules <" ^ (Int.toString m)
					   ^ ", " ^ (Int.toString n) ^ ">\n")
			    val reds1 = Rewrite.developOneStepReductsWithIndex rs c1
			    val reds2 = Rewrite.developOneStepReductsWithIndex rs c2
			    val _ = print "develop reducts from lhs term...\n"
			    val _ = List.app (fn (set,t) => 
						 (print ("<" ^ (prIntSet set) ^ ", ");
						  print ((Term.toString t) ^ ">");
						  print "\n"))
					     reds1
			    val _ = print "develop reducts from rhs term...\n"
			    val _ = List.app (fn (set,t) => 
						 (print ("<" ^ (prIntSet set) ^ ", ");
						  print ((Term.toString t) ^ ">");
						  print "\n"))
					     reds2
			    val reds1'  =  List.map (fn (set,t) => (IS.difference (set,IS.singleton n),t)) 
						    reds1
			    val reds2'  =  List.map (fn (set,t) => (IS.difference (set,IS.singleton m),t)) 
						    reds2
			    val constraints = mem_develop_out2 (reds1', reds2')
			in
			    ((m,n), 
			     List.map 
				 (fn (set1,set2,b)=> (IS.listItems set1, IS.listItems set2, b))
				 constraints)
			end)
		    ocps
       end

   (****************************************)
   (******* $B2D49@-$rK~$?$96K>.$JJ,2r$N7W;;(B ********)
   (****************************************)

   (* insidecheck ($BF1(B)$B$,(B($BFb(B)$B>r7o$rK~$?$5$J$$$H$-(Bfalse *)
   (* - insidecheck [(1,2),(3,4)] [(3,1),(2,4)] => false *)
   (*  (3,1) :  inside CP, P <-{3} X ->{1} Q  $B$G!$(B P -o-> Q $B$H$J$C$F$$$J$$$b$N$,$"$k!%(B *)
   (*  (2,4) :  inside CP, P <-{2} X ->{4} Q  $B$G!$(B P -o-> Q $B$H$J$C$F$$$J$$$b$N$,$"$k!%(B *)
   (*  ===>   CP_in({1,2},{3,4}) $B$O(B Oostrom $B>r7o$rK~$?$5$J$$(B *)

   fun insidecheck nps1 nps3 =
       let
	   fun cpin_pair [] = []
	     | cpin_pair ((n1,n2)::nps) =
	       let
		   fun cpin_pair_sub (n11,n12) (n21,n22) = 
		       (orderPair (n11,n22),orderPair (n12,n21))
	       in
		   (map (cpin_pair_sub (n1,n2)) nps) @ (cpin_pair nps)
	       end
	   fun insidecheck_sub _ [] = true
	     | insidecheck_sub nps ((np1,np2)::npps) =
	       if (member np1 nps) andalso (member np2 nps) then false
	       else insidecheck_sub nps npps
       in
	   insidecheck_sub nps1 (cpin_pair nps3)
       end

   (* one step comcr *)
   (*
    - comcr_sub ((2,4),([4],[1],true)) [([(1,2)],[],[]),([(2,3)],[(1,2)],[])];
   val it =
       [([(2,4),(1,2)],  [],  []),
	([(1,2)], [(2,4)], [(2,4)]),
	([(2,4),(2,3)], [(1,2)], []),
	([(1,2),(2,3)], [(2,4),(1,2)], [(2,4)])]
          $BF1$8@.J,$K4^$^$l$kBP!$F1$8@.J,$K4^$^$l$F$$$J$$BP!$(Binsidecheck$BMQ$NBP(B
      $B$N%j%9%H$KDI2C(B
    *)

    (* crplist0   <m,n> (<Ai,Bi,bi> *)

   fun comcr_step _ [] = []
     | comcr_step ((_,_),([],[],false)) npsps0 = npsps0
     | comcr_step ((n1,n2),(ns1,ns2,b)) npsps0 =
       let
	   val (n1r,n2r) = orderPair (n1,n2)
           val n1ns2 = makeOrderedIntPairs (ucons n1 ns2)
           val n2ns1 = makeOrderedIntPairs (ucons n2 ns1)
           fun comcr_step_t [] = []
             | comcr_step_t ((nps1,nps2,nps3)::npsps) =
               (ucons (n1r,n2r) nps1,nps2,nps3) ::
	       (union n1ns2 (union n2ns1 nps1),ucons (n1r,n2r) nps2,ucons (n1,n2) nps3) ::
	       (comcr_step_t npsps)
           fun comcr_step_f [] = []
             | comcr_step_f ((nps1,nps2,nps3)::npsps) =
               (ucons (n1r,n2r) nps1,nps2,nps3) ::
	       (union n1ns2 (union n2ns1 nps1),ucons (n1r,n2r) nps2,nps3) ::
	       (comcr_step_f npsps)
       in
	   if b 
	   then comcr_step_t npsps0
	   else comcr_step_f npsps0
       end

   (* [([2],[],false),([2],[1],true)] ->  *)
   (*
    - comcr [((1,2),[([3],[1],false)]),((2,4),[([4],[1],true)])];
   val it =
       [([(2,4),(1,2)],[],[]),([(1,2)],[(2,4)],[(2,4)]),([(2,4),(2,3)],[(1,2)],[])]
    *)

   (* crplist0   <m,n> [<A1,B1,b1>,..,<Ak,Bk,bk>]  $B$N%j%9%H(B *)

   fun makeConstraints [] = [([],[],[])]
     | makeConstraints ((np0,nsps0)::crplist) =
       let fun comcr_check (es,ds,cs) = 
	       let val es' = takeEqClosure es
	       in (isDisjoint es' ds) andalso (insidecheck es' cs)
	       end
	   fun comcr_step1 np npsps = 
	       filter comcr_check
		      (map (fn (es,ds,cs) => (ucons (orderPair np) es, ds, cs)) 
			   npsps)
	   fun comcr_step2 (np,nsp) npsps = 
	       filter comcr_check  (comcr_step (np,nsp) npsps)
	   val rest = makeConstraints crplist
       in
	   if null nsps0 
	   then comcr_step1 np0 rest
	   else mapAppend (fn nsp => comcr_step2 (np0,nsp) rest)
			  nsps0
       end



   (* $B6K>.$N2D49J,2r$K4^$^$l$F$$$J$$%k!<%k$rDI2C$9$k(B ($B2D49J,2r@.J,$KF~$C$F$$$J$$$b$N$OC1FH$N@.J,$K$J$k(B)
       [[3,2,7],[4,5]]  => [[3,2,7],[4,5],[1],[6]]                          *)
   fun makeFullComponents nss rs = 
       let 
	   val singles = delete (L.tabulate (length rs, fn i=>i)) (unionall nss)
       in
	   nss @ (L.map (fn lr => [lr]) singles)
       end

   (* $B2D49@-$rK~$?$96K>.$JJ,2r$N7W;;(B *)
   fun minimalDecompositions rs =
       let 
	   val commutativeConstraints = 
	       makeConstraints ((makeOutsideCPConditions rs) @ (makeInsideCPConditions rs))
	   val components = List.map (fn (a,b,c) => makeEqClass a) commutativeConstraints
	   val minimals = leaveMinimals subseteq components
	   val full = List.map (fn nss => makeFullComponents nss rs) minimals
	   val full2 = L.filter (fn nss => length nss <> 1) full (* $BJ,2r$K$J$C$F$$$J$$$b$N$O=|$/(B *)
       in
	   full2
       end


   (****************************************)
   (******* $B9gN.@-$N2D49J,2r$rMQ$$$?H=Dj(B *********)
   (****************************************)


(*    (\* $B2D49@.J,$r(BCR$B$H$J$k$h$&$JAH$_9g$o$;$K$^$H$a$k(B *\) *)
(*    fun comapp checkCC [] = (true, []) *)
(*      | comapp checkCC (rs0::rss0) = *)
(*        let *)
(*            fun comapp_sub rs [] = (false, rs0::rss0) *)
(*              | comapp_sub rs ((rss1,rss2)::rssps) = *)
(*                if (checkCC rs) = CR then (true, (rs::(rss1 @ rss2))) *)
(*                else if checkCC (rs @ (concat rss1)) = CR *)
(*                then case comapp_sub (hd rss2) *)
(*                                     (kumi ((tl rss2) @ [(rs @ (concat rss1))])) *)
(*                      of (true, rss') => (true, rss') *)
(*                       | (false, _) => comapp_sub rs rssps *)
(*                else comapp_sub rs rssps; *)
(*        in *)
(*            comapp_sub rs0 (kumi rss0) *)
(*        end *)


   (* $B9gN.@-$N2D49J,2r$rMQ$$$?H=Dj(B *)
   fun proveConfluenceByCommutativeDecomposition isTerminating rs = 
       if null rs
       then report CR
       else 
	   let val _ = print "Rewrite Rules:\n"
	       val _ = print (Trs.prRules rs)
	       val _ = print "Check Confluence Conditions...\n"
	       fun getRules ns = L.map (fn i => L.nth (rs,i)) ns
	       fun printDecomposition nss =  L.app (print o Trs.prRules o getRules) nss
	       fun checkCC xs = checkConfluenceConditions isTerminating xs

	       (* CR$B$NFs=E%A%'%C%/$rHr$1$k$?$a!$J,2r@.J,$,(B CR $B$+$r5-O?(B *)
	       val rem = ref ILM.empty
	       fun lookup xs  = case ILM.find (!rem,xs) of
				    SOME b => b
				  | NONE => let val _ = print (prIntList xs)
						val b = checkCC (getRules xs) = CR
						val _ = ILM.insert (!rem, xs, b)
					    in b
					    end

	       fun checkNonMinimalDecompositions rss2 rss1 =
		   let 
		       val _  = List.app (fn xs => rem := ILM.insert (!rem, xs, false)) rss2
		       (* $B2D49$J(B CR $B@.J,$NOB$O(B CR *)
		       val _  = List.app (fn xss => 
					     rem := ILM.insert (!rem,
								ListMergeSort.sort (Int.>) (unionall xss),
								true))
					 (powerlist rss1)
		       fun check xs yss  = 
			   let val ys = ListMergeSort.sort (Int.>) (unionall (xs::yss))
			       val _ = print "Add "
			       val _ = List.app (print o prIntList) yss
			       val _ = print "\n"
			   in
			       lookup ys
			   end

		       fun main [] _ = true
			 | main (xs::xss) yss =
			   let val _ = print ("Try to add other components to ");
			       val _ = print (prIntList xs);
			       val _ = print "\n";
			       val result = 
				   L.exists (fn zss => 
						check xs zss
						andalso main (delete xss zss) (delete yss zss))
					    (properPowerlist (xss @ yss))
			       val _ = print "\n";
			   in result
			   end
		   in 
		       main rss2 rss1
		   end

	       fun crchecktry [] = (print "Not decomposable"; report Unknown)
		 | crchecktry (rss::rsss) =
		   let val _ = print ("Try A Minimal Decomposition ")
		       val _ = L.app (print o prIntList) rss
		       val _ = print ("\n")
		       val (rss1,rss2) = List.partition lookup rss
		   in
		       if null rss2 
		       then 
			   (print "Find commutative decomposition for CR\n";
			    printDecomposition rss1;
			    CR)
		       else
			   if checkNonMinimalDecompositions rss2 rss1
			   then (print "Find commutative decomposition for CR\n";
				 CR)
			   else crchecktry rsss
		   end
	   in
	       case checkCC rs of
		   CR => CR
		 | NotCR => NotCR
		 | Unknown => 
		   if Trs.areLeftLinearRules rs
		   then (print "\nTry Commutative Decomposition...\n";
			 rem := ILM.insert (!rem, L.tabulate (length rs, fn i=>i), false);
			 crchecktry (minimalDecompositions rs))
		   else (print "Not commutative decomposable"; report Unknown)
	   end

   fun tryCommutativeDecomposition rs (direct,decomp) = 
       let
	   val _ =  print "\nTry Commutative Decomposition for...\n"
	   val _ = print (Trs.prRules rs)
	   fun getRules ns = L.map (fn i => L.nth (rs,i)) ns
	   fun printDecomposition nss =  L.app (print o Trs.prRules o getRules) nss
	   fun checkCC rs = (print "(cm)"; direct rs)

	   (* CR$B$NFs=E%A%'%C%/$rHr$1$k$?$a!$J,2r@.J,$,(B CR $B$+$r5-O?(B *)
	   val rem = ref ILM.empty
	   fun lookup xs  = case ILM.find (!rem,xs) of
				SOME b => b
			      | NONE => let val _ = print ((prIntList xs)^"\n")
					    val b = checkCC (getRules xs) = CR
					    val _ = ILM.insert (!rem, xs, b)
					in b
					end
	   fun checkNonMinimalDecompositions rss2 rss1 =
	       let 
		   val _  = List.app (fn xs => rem := ILM.insert (!rem, xs, false)) rss2
		   (* $B2D49$J(B CR $B@.J,$NOB$O(B CR *)
		   val _  = List.app (fn xss => 
					 rem := ILM.insert (!rem,
							    ListMergeSort.sort (Int.>) (unionall xss),
							    true))
				     (powerlist rss1)
		   fun check xs yss  = 
		       let val ys = ListMergeSort.sort (Int.>) (unionall (xs::yss))
			   val _ = print "Add "
			   val _ = List.app (print o prIntList) yss
			   val _ = print "\n"
		       in
			   if lookup ys
			   then SOME ys
			   else NONE
		       end
		       
		   fun main [] yss = SOME yss
		     | main (xs::xss) yss =
		       let val _ = print ("Try to add other components to ");
			   val _ = print (prIntList xs);
			   val _ = print "\n";
		       in test xs xss yss (properPowerlist (xss @ yss))
		       end
		   and test _ _ _ [] = NONE
		     | test xs xss yss (zss::rest) = 
		       let val ans1 = check xs zss
		       in if isSome ans1 
			  then let val ans2 = main (delete xss zss) (delete yss zss)
			       in if isSome ans2
				  then SOME ((valOf ans1)::(valOf ans2))
				  else test xs xss yss rest
			       end
			  else test xs xss yss rest
		       end
	       in 
		   main rss2 rss1
	       end

	   fun crchecktry [] = (print "Commutative Decomposition failed";
				report Unknown;
				decomp rs)
	     | crchecktry (rss::rsss) =
	       let val _ = print ("Try A Minimal Decomposition ")
		   val _ = L.app (print o prIntList) rss
		   val _ = print ("\n")
		   val (rss1,rss2) = List.partition lookup rss
			in
		   if null rss2 
		   then 
		       (print "Find commutative decomposition for CR\n";
			printDecomposition rss1;
			CR)
		   else
		       if (!useNonMinimal) then 
			   case checkNonMinimalDecompositions rss2 rss1 of
			       SOME rss3 => (print "Find commutative decomposition for CR\n";
					     printDecomposition rss3;
					     CR)
			     | NONE => crchecktry rsss
		       else 
			   crchecktry rsss
	       end
       in
	   if Trs.areLeftLinearRules rs
	   then (rem := ILM.insert (!rem, L.tabulate (length rs, fn i=>i), false);
		 crchecktry (minimalDecompositions rs))
	   else (print "Commutative Decomposition failed (not left-linear)";
		 report Unknown;
		 decomp rs)
       end


   end
   end
