(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/cr_completion.sml
 * description: reduction-preserving completion
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature CR_COMPLETION = 
   sig
    val runDebug: bool ref
   val bidirectionalCompletion:
      ((Term.term * Term.term) list -> bool)
      -> ((Term.term * Term.term) list * (Term.term * Term.term) list -> bool)
      -> FunSet.set
      -> (Term.term * Term.term) list
      -> bool

   val useForward: bool ref
   val usePCP: bool ref
   val useLinear: bool ref
   val useParallel: bool ref
   val useRelative: bool ref
   val useHuet: bool ref
   val useCompletion: bool ref
end;

structure CrCompletion : CR_COMPLETION = 
   struct

   local 
       open Term
       open Trs
       open Rewrite
       open Subst
       open Cr
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure IS = IntSet
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure TP = TermPair
       structure TS = TermSet
       open PrintUtil   
in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()

   val useForward = ref true
   val usePCP = ref true
   val useLinear = ref true
   val useParallel = ref true
   val useRelative = ref true
   val useHuet = ref true
   val useCompletion = ref true
   val maxCompletionSteps = 20

   (* P => P \cup P^{-1} *)
   fun takeSym P = LU.union' 
		       (fn (l,r) => Subst.identicalModuloRenamingRule l r)
		       (P, L.map (fn (l,r) => (r,l)) P)

   (* P rule $BA*Br$N(B heuristics conditions *)
   fun isSuitableForPrule dSymSet (l,r) = 
       let val fset1 = Term.funSetInTerm l
	   val fset2 = Term.funSetInTerm r
       in
	   FS.equal (fset1,fset2)
	   (* andalso FS.isSubset (fset1,dSymSet) *)
	   andalso (case (funRootOfTerm l, funRootOfTerm r) of
			(SOME f, SOME g) => Fun.equal (f,g)
		      | _ => false)
       end


   (* decompose rs => (S,P) such a way that P: bidirectional and 
                      suitable or inverse rule exists *)
   fun decompose dSymSet rs = 
       let  
	   val (prules,srules) = 
	       L.partition (fn (l,r) => Trs.isBidirectionalRule (l,r)
					andalso 
					(isSuitableForPrule dSymSet (l,r)
					 orelse 
					 LU.member'
					     (fn (l,r) => Subst.identicalModuloRenamingRule l r)
					     (r,l) rs))
			   rs
	   val _ = debug (fn _ => print "R rules:\n")
	   val _ = debug (fn _ => print (Trs.prRules rs))
	   val _ = debug (fn _ => print "Selected P rules:\n")
	   val _ = debug (fn _ => print (Trs.prRules prules))
       in
	   (srules,prules)
       end

   (* SN(S) $B$r2>Dj(B *)
   fun joinOneStep S P (u,v) = 
       let val (u1,v1) = (Rewrite.linf S u, Rewrite.linf S v)
	   val _ = print ("<" ^ (Term.toString u) ^ ", " ^ (Term.toString v) ^ ">")
	   val _ = print (" --> <" ^ (Term.toString u1) ^ ", " ^ (Term.toString v1) ^ ">")
	   val result = Term.equal (u1,v1)
			orelse TS.member (Rewrite.oneStepReductSet P u1, v1)
	   val _ = print (if result then " => yes\n" else " => no\n")
       in
	   if result then NONE  else SOME (u,u1,v,v1)
       end

   (* SN(S) $B$r2>Dj(B *)
   fun joinOneStepWithParallel S P (u,v) = 
       let val (u1,v1) = (Rewrite.linf S u, Rewrite.linf S v)
	   val _ = print ("<" ^ (Term.toString u) ^ ", " ^ (Term.toString v) ^ ">")
	   val _ = print (" --> <" ^ (Term.toString u1) ^ ", " ^ (Term.toString v1) ^ ">")
	   val result = TS.member (Rewrite.parallelOneStepReductSet P u1, v1)
	   val _ = print (if result then " => yes\n" else " => no\n")
       in
	   if result then NONE  else SOME (u,u1,v,v1)
       end

   (* SN(S) $B$r2>Dj(B *)
   fun joinOneStepWithParallelWithVar S P (u,v,vset) = 
       let val (u1,v1) = (Rewrite.linf S u,  Rewrite.linf S v)
	   val _ = print ("<" ^ (Term.toString u) ^ ", " ^ (Term.toString v) ^ ">")
	   val _ = print (" --> <" ^ (Term.toString u1) ^ ", " ^ (Term.toString v1) ^ ">")
	   val cands = L.mapPartial (fn (u,vset') => 
					if VS.isSubset (vset', vset) then SOME u else NONE)
				    (Rewrite.parallelOneStepReductsWithVar P u1)
	   val result = LU.member' Term.equal v1 cands
	   val _ = print (if result then " => yes\n" else " => no\n")
       in
	   if result then NONE else SOME (u,u1,v,v1,vset)
       end

   (* idx -- Var ("x",idx) $B$O$^$@;H$o$l$F$$$J$$JQ?t(B *)
   (* makeInitialTerm idx s pos (l,r):
      s $B$N(B pos $B$N>l=j$r(B l $B$K$*$-$+$(!$(Bpos $B$+$i(B $B%k!<%H$X$N4X?t5-9f0J30$O(B
      $B$9$Y$FJQ?t$KCV$-BX$($?9`$rJV$9(B *)
   fun makeInitialTerm idx s [] (l,r) = SOME l
     | makeInitialTerm idx (s as (Var _)) (x::xs) (l,r) = NONE
     | makeInitialTerm idx (s as (Fun (f,ts,ty))) (x::xs) (l,r) = 
       if 0 < x andalso x <= L.length ts
       then let val len = L.length ts
		val ys = L.tabulate (len,
				  fn x=> Var.fromStringAndInt ("x",idx+x))
		val us = LP.map (fn (u,y)=> Var (y,sortOfTerm u)) (ts,ys)
		val candui = makeInitialTerm (idx + len) (L.nth (ts,x-1)) xs (l,r) 
	    in case candui of 
		   SOME ui => SOME (Fun (f, LU.replaceNth (us,x-1,ui), ty))
		 | NONE => NONE
	    end
       else NONE
   
   (* p::ps $B$O(B $B=q$-49$($?0LCV!$(B lr::rest $B$O=q$-49$($KMQ$$$?5,B'$G!$(B
      s $B$+$i;O$^$j!$%k!<%H=q$-49$($r4^$`=q$-49$(Ns$r;XDj$7$F$$$k!%(B
      $B$3$N$H$-:G=i$H:G8e$N9`$r7R$00lHLE*$JEy<0$r:n$k(B *)
   fun selectGeneralEqFromRewriteSeq s ([],[]) = NONE
     | selectGeneralEqFromRewriteSeq s (p::ps,lr::rest) =
       let val idx = (Term.maxVarIndexInTerm s) + 1
	   val init = makeInitialTerm idx s p lr
       in case init of
	      NONE => NONE
	    | SOME t0 => (case Rewrite.repeatNarrow t0 (p::ps,lr::rest) of
			     SOME (last,sub) => SOME (Subst.applySubst sub t0, last)
			   | NONE => NONE)
       end

   (* rs $B=q$-49$(5,B'!$(Bps $B$O(B $B=q$-49$($?0LCV!$(Bxs $B$O=q$-49$($KMQ$$$?5,B'$N%$%s%G%C%/%9!%(B
      $B=q$-49$($K$O%k!<%H$r4^$`$H$O8B$i$J$$$N$G!$$$$D$/$+0lHLE*$JEy<0$,$"$k$+$b!%(B*)
   (* $B%J%m!<%$%s%0$GB3$/>l9g$K$N$_0lHLE*$JEy<0$,9=@.$5$l$k(B *)
   fun getGeneralEq rs s (ps,xs) =
       let fun filt p ([],[]) = ([],[])
	     | filt p (q::qs,[]) = ([],[])
	     | filt p ([],y::ys) = ([],[])
	     | filt p (q::qs,y::ys) = 
	       if Pos.isPrefix p q 
	       then let val (qs',ys') = filt p (qs,ys)
		    in (valOf (Pos.cut p q)::qs', L.nth (rs,y)::ys')
		    end
	       else filt p (qs,ys)
       in L.mapPartial (fn p => selectGeneralEqFromRewriteSeq 
				    (valOf (Term.subterm p s)) (filt p (ps,xs)))
		       (Pos.selectMinimals ps)
       end

   fun makePEqsFromNonJoinables symP njs =
       LU.mapAppend
	   (fn (u,u1,v,v1) => 
	       let val xs = Rewrite.manyStepsReductsWithPosAndIndex symP 3 u1
		   val xs2 = L.mapPartial (fn (ps,xs,w) => if Term.equal (v1,w)
							   then SOME (ps,xs) else NONE)
				     	  xs
(*		   val _ = if null xs2 
			   then ()
			   else print (prEqs symP) *)
(*
		   val _ = L.app (fn (ps,js) =>
				     print ((LU.toStringCommaSquare Pos.toString ps)
				            ^ (LU.toStringCommaSquare Int.toString js)
				            ^ "\n"))
				 xs2
*)
		   val cand = LU.mapAppend (getGeneralEq symP u1) xs2 
		   val _ = if (null cand) then () 
			   else print ("Constructed Eq from " 
				       ^ (IOFotrs.prEq (u1,v1)) ^ ":\n"
				       ^ (prEqs cand))
	       in
		   cand
	       end)
	   njs

  (* term1 $B$H(B term2 $B$,(B R/E $B=q$-49$($G(B joinable $B$+(B
     SN(S) $B$r2>Dj!"(BR/E $B=q$-49$($NDd;_@-$O$^$@ITL@(B
     $BESCf$GMQ$$$?(B E $B$N(B subset $B$rJV$9(B  *)
   val maxLimitJoinabilityCondition = 2000
   fun isModuloJoinableWithLast
	   performLastStep
	   R E (term1,term2) =
       let (* heulistic to limit search *)
	   val maxSize = 3 + Int.max (Term.termSize term1,Term.termSize term2)
	   fun check [] [] n1 n2 tset1 tset2 all = (print "maybe not joinable\n"; NONE)
	     | check [] (t2::ts2) n1 n2 tset1 tset2 all = 
               (* do not use *) check (t2::ts2) [] n2 n1 tset2 tset1 all
	     | check ((t1,prevRuleSet)::ts1) ts2 n1 n2 tset1 tset2 all =
	       if maxLimitJoinabilityCondition <= n1 (* limit *)
	       then (print "maybe not joinable(limit exceeds)\n"; NONE)
	       else
	       if TS.member (tset1,t1)
	       then check ts1 ts2 n1 n2 tset1 tset2 all
	       else if (Term.termSize t1 > maxSize) (* heulistic *)
	       then check ts1 ts2 n1 n2 tset1 tset2 all
	       else 
		   case TS.find (fn x=>true)
				(TS.intersection 
				     (tset2,performLastStep t1)) of
		       SOME s1 =>
		       (let val (_,prevRuleSet2) = 
				valOf (List.find (fn (s,V) => Term.equal(s1,s)) all)
			    val used = IS.listItems (IS.union (prevRuleSet,prevRuleSet2))
			in (print "joinable by "; 
			    print (LU.toStringCommaCurly Int.toString used);
			    print "\n"; 
			    SOME used)
			end)
		     | NONE =>
			case linfPartial R t1 of
			    SOME nf1 => isModuloJoinableSub 
					    [(nf1,prevRuleSet)] ts2 
					    (n1+1) n2 
					    (TS.add (tset1,t1)) tset2 
					    ((t1,prevRuleSet)::all)
			  | NONE => 
			    isModuloJoinableSub 
				(ts1 @ (L.map (fn (i,t) => 
						  (t,IS.add(prevRuleSet,i)))
					      (Rewrite.oneStepReductsWithIndex E t1)))
				ts2 (n1+1) n2 
				(TS.add (tset1,t1)) tset2 
				((t1,prevRuleSet)::all)
				    
	   and isModuloJoinableSub ts1 ts2 n1 n2 tset1 tset2 all =
	       if n1 <= n2 andalso (not (null ts1))
	       then check ts1 ts2 n1 n2 tset1 tset2 all
	       else check ts2 ts1 n2 n1 tset2 tset1 all
	   val _ = print ("check modulo joinability of " 
			  ^ (Term.toString term1)
			  ^ " and "
			  ^ (Term.toString term2)
			  ^ ": ")

       (***
         in isModuloJoinableSub [(term1,IS.empty)] 
			      [(term2,IS.empty)] 
			      0 0 TS.empty TS.empty []
       ***)
      (* add heulistics *)
       in if (Term.termSize term2) > (Term.termSize term1)
 	  then
 	      isModuloJoinableSub [(term1,IS.empty)] 
 				  [(term2,IS.empty)] 
 				  0 0 TS.empty TS.empty []
 	  else
 	      isModuloJoinableSub [(term2,IS.empty)] 
 				  [(term1,IS.empty)] 
 				  0 0 TS.empty TS.empty []
       end

   val maxLimitReachabilityCondition = 2000
   fun isModuloReachableWithLast
	   performLastStep
	   R E (startTerm,goalTerm) =
       let val goalSet = performLastStep goalTerm
	   val maxSize = 3 + (Term.termSize goalTerm) (* heulistic to limit search *)
	   fun isModuloReachableSub [] n tset = (print "maybe not reachable\n"; NONE)
	     | isModuloReachableSub ((t,prevRuleSet)::ts) n tset =
	       ((* print ((Term.toString t) ^ "\n"); *)
	       if maxLimitReachabilityCondition <= n (* limit *)
	       then (print "maybe not reachable(limit exceeds)\n"; NONE)
	       else if TS.member (tset,t)
	       then isModuloReachableSub ts n tset
	       else if TS.member (goalSet,t)
	       then let val used = IS.listItems prevRuleSet
		    in (print "reachable by "; 
			print (LU.toStringCommaCurly Int.toString used);
			print "\n"; 
			SOME used)
		    end
	       else if Term.termSize t > maxSize (* heulistics *)
	       then isModuloReachableSub ts n tset  (* heulistics *)
	       else case linfPartial R t of
			SOME nf => if TS.member (tset,nf)
				   then isModuloReachableSub ts n tset
				   else isModuloReachableSub 
					    (ts @ [(nf,prevRuleSet)]) 
					    (n+1) (TS.add (tset,t))
		     | NONE => isModuloReachableSub 
				   (ts @ (L.map (fn (i,s) => 
						    (s,IS.add(prevRuleSet,i)))
						(Rewrite.oneStepReductsWithIndex E t)))
				   (n+1) (TS.add (tset,t)))
	   val _ = print ("check modulo reachablity from " 
			  ^ (Term.toString startTerm)
			  ^ " to "
			  ^ (Term.toString goalTerm)
			  ^ ": ")
       in isModuloReachableSub [(startTerm,IS.empty)] 0 TS.empty
       end

   (* Relative Termination $B$K4p$E$/9gN.@-H=Dj(B *)
   fun checkConfluenceConditionByRelativeTermination
	   isRelativeTerminating S symP
	   (ssNonJoinables,psNonJoinables,spNonJoinables) =
       let val _ = print "check joinability condition:\n"
	   fun performLastStep u = Rewrite.oneStepReductSet symP u
	   fun checkModuloJoinability (term1,term2) =
	       isModuloJoinableWithLast performLastStep S symP (term1,term2)
	   fun checkModuloReachability (startTerm,goalTerm) =
	       isModuloReachableWithLast performLastStep S symP (startTerm,goalTerm)
	   fun ssCheck (u,u1,v,v1) = checkModuloJoinability (u1,v1)
	   fun psCheck (u,u1,v,v1) =
	       if (Rewrite.isNormalForm S u)
	       then checkModuloReachability (v1,u)
	       else checkModuloJoinability (u1,v1)
	   fun spCheck (u,u1,v,v1) =
	       if (Rewrite.isNormalForm S v)
	       then checkModuloReachability (u1,v)
	       else checkModuloJoinability (u1,v1)
	   val ssResult = L.map ssCheck ssNonJoinables
	   val psResult = L.map psCheck psNonJoinables
	   val spResult = L.map spCheck spNonJoinables
       in if (List.all isSome ssResult)
	     andalso (List.all isSome psResult)
	     andalso (List.all isSome spResult)
	  then
	      let val _ = print "success\n";
		  val res = ssResult @ psResult @ spResult
		  val xs = LU.mapAppend (fn x=>x) 
					(L.mapPartial (fn x=>x) res)
		  val ys = LU.eliminateDuplication xs
			   val P' = L.map (fn i => List.nth (symP,i)) ys
			   val _ =  print "P':\n"
			   val _ =  print (Trs.prRules P')
			   val res = isRelativeTerminating (S,P')
			   (* val _ = debug (fn _ => if res 
				   then print "S/P': relatively terminating\n"
				   else print "S/P': unknown relative termination\n") *)
		       in res
		       end
	  else (print "failed\n";false)
       end

   (* Relative Termination $B$K4p$E$/9gN.@-H=Dj(B PCP version  *)
   fun checkConfluenceConditionByRelativeTerminationPCP
	   isRelativeTerminating S symPorg
	   (ssNonJoinables,psNonJoinables,spNonJoinables) =
       let val _ = print "check joinability condition:\n"
          (* heulistics for changing order of symP *)
	   val symP = ListMergeSort.sort (fn ((l1,r1),(l2,r2)) => 
 					     Int.< ((Term.termSize l1) - (Term.termSize r1),
 						    (Term.termSize l2) - (Term.termSize r2))) symPorg
	   fun performLastStep1 vset t = 
	       TS.addList (TS.empty,
			   L.mapPartial (fn (u,vset') => 
					    if VS.isSubset (vset', vset) 
					    then SOME u else NONE)
					(Rewrite.parallelOneStepReductsWithVar symP t))
	   fun performLastStep2 u = 
	       Rewrite.parallelOneStepReductSet symP u
	   fun checkModuloJoinability performLastStep (term1,term2) =
	       isModuloJoinableWithLast performLastStep S symP (term1,term2)
	   fun checkModuloReachability performLastStep (startTerm,goalTerm) =
	       isModuloReachableWithLast performLastStep S symP (startTerm,goalTerm)
	   fun ssCheck (u,u1,v,v1) = 
	       checkModuloJoinability performLastStep2 (u1,v1)
	   fun psCheck (u,u1,v,v1,vset) =
	       if (Rewrite.isNormalForm S u)
	       then checkModuloReachability (performLastStep1 vset) (v1,u)
	       else checkModuloJoinability (performLastStep1 vset) (u1,v1)
	   fun spCheck (u,u1,v,v1) =
	       if (Rewrite.isNormalForm S v)
	       then checkModuloReachability performLastStep2 (u1,v)
	       else checkModuloJoinability performLastStep2 (u1,v1)
	   val ssResult = L.map ssCheck ssNonJoinables
	   val psResult = L.map psCheck psNonJoinables
	   val spResult = L.map spCheck spNonJoinables
       in if (List.all isSome ssResult)
	     andalso (List.all isSome psResult)
	     andalso (List.all isSome spResult)
	  then
	      let val _ = print "success\n";
		  val res = ssResult @ psResult @ spResult
		  val xs = LU.mapAppend (fn x=>x) 
					(L.mapPartial (fn x=>x) res)
		  val ys = LU.eliminateDuplication xs
			   val P' = L.map (fn i => List.nth (symP,i)) ys
			   val _ =  print "P':\n"
			   val _ =  print (Trs.prRules P')
			   val res = isRelativeTerminating (S,P')
			   val _ = if res 
				   then print "S/P': relatively terminating\n"
				   else print "S/P': unknown relative termination\n"
		       in res
		       end
	  else (print "failed\n";false)
       end

   (* $BJdBj$N@8@.(B *)
   fun getCandidates symP ssNonJoinables psNonJoinables 
		     spNonJoinables spProblematicSRules =
       let val ssCand = LU.eliminateDuplication' 
			    (fn (l,r) => Subst.identicalModuloRenamingRule l r)
			    (makePEqsFromNonJoinables symP ssNonJoinables)
			
	   val spCand = LU.eliminateDuplication' 
			    (fn (l,r) => Subst.identicalModuloRenamingRule l r)
			    ((L.map (fn (u,u1,v,v1) => (u,v1)) psNonJoinables)
			     @ (L.map (fn (u,u1,v,v1) => (v,u1)) spNonJoinables))

	   val spProb = LU.eliminateDuplication' 
			    (fn (l,r) => Subst.identicalModuloRenamingRule l r)
			    spProblematicSRules
       in (ssCand,spCand,spProb)
       end

   (* SN(S), S:linear $B$J>l9g$N(B CP $B>r7o$N%A%'%C%/(B *)
   fun checkCPConditionLinear isRelativeTerminating S symP = 
       let 
	   val ssNonJoinables = 
	       let val cps = criticalPairs S
		   val _ = print ("CP(S,S):\n")
	       in L.mapPartial (joinOneStep S symP) cps
	       end

	   val psNonJoinables = 
	       let val cps = insideCriticalPairs2 (symP,S)
		   val _ = print ("CP_in(symP,S):\n")
	       in L.mapPartial (joinOneStep S symP) cps
	       end

	   val (spNonJoinables,spProblematicSRules) = 
	       let val llcps = criticalPairs2WithRules (S,symP)
		   val _ = print ("CP(S,symP):\n")
	       in LP.unzip (L.mapPartial 
				(fn (lr,lr',u,v) => 
				    case joinOneStep S symP (u,v) of
					SOME (u,u1,v,v1) => SOME ((u,u1,v,v1),lr)
				      | NONE => NONE)
				llcps)
	       end
       in if (null ssNonJoinables)
	      andalso (null psNonJoinables)
	      andalso (null spNonJoinables)
	  then SOME (true,symP,[],[],[])
	  else if (!useRelative)
		  andalso (checkConfluenceConditionByRelativeTermination
			       isRelativeTerminating S symP 
			       (ssNonJoinables,psNonJoinables,spNonJoinables))
	  then SOME (true,symP,[],[],[])
	  else if not (!useCompletion)
	  then SOME (false,symP,[],[],[])
	  else let val (ssCand,spCand,spProb) = 
		       getCandidates symP ssNonJoinables
				     psNonJoinables spNonJoinables 
				     spProblematicSRules
	       in SOME (false,symP,ssCand,spCand,spProb)			    
	       end
       end

   (* SN(S), S:left-linear $B$J>l9g$N(B parallel step $B$K4p$E$/>r7o$N%A%'%C%/(B *)
   fun checkCPConditionParallel isRelativeTerminating S symP = 
       let 
	   val ssNonJoinables = 
	       let val cps = criticalPairs S
		   val _ = print ("CP(S,S):\n")
	       in L.mapPartial (joinOneStepWithParallel S symP) cps
	       end

	   val cpPS = insideCriticalPairs2 (symP,S)
	   val _ = print ("CP_in(symP,S):\n")
	   val _ = print (prEqs cpPS)

	   val (spNonJoinables,spProblematicSRules) = 
	       let val llcps = criticalPairs2WithRules (S,symP)
		   val _ = print ("CP(S,symP):\n")
	       in LP.unzip (L.mapPartial 
				(fn (lr,lr',u,v) => 
				    case joinOneStepWithParallel S symP (u,v) of
					SOME (u,u1,v,v1) => SOME ((u,u1,v,v1),lr) 
				      | NONE => NONE)
				llcps)
	       end
       in if (null ssNonJoinables)
	     andalso (null cpPS)
	     andalso (null spNonJoinables)
	  then SOME (true,symP,[],[],[])
	  else if (!useRelative)
		  andalso (null cpPS)
		  andalso (checkConfluenceConditionByRelativeTerminationPCP
			       isRelativeTerminating S symP 
			       (ssNonJoinables,[],spNonJoinables))
	  then SOME (true,symP,[],[],[])
	  else if not (!useCompletion)
		  andalso (not (null cpPS))
	  then SOME (false,symP,[],[],[])
	  else let val (ssCand,spCand,spProb) = 
		       getCandidates symP ssNonJoinables
				     [] spNonJoinables 
				     spProblematicSRules
	       in SOME (false,symP,ssCand,spCand,spProb)			    
	       end
       end

   (* SN(S), S:left-linear $B$J>l9g$N(B PCP $B$K4p$E$/>r7o$N%A%'%C%/(B *)
   fun checkCPConditionParallelPCP isRelativeTerminating S symP = 
       let 
	   val ssNonJoinables = 
	       let val cps = criticalPairs S
		   val _ = print ("CP(S,S):\n")
	       in L.mapPartial (joinOneStepWithParallel S symP) cps
	       end

	   val psNonJoinables =
	       let val cps = innerProperParallelCriticalPairsWithVarSet2 (symP,S)
		   val _ = print ("PCP_in(symP,S):\n")
	       in L.mapPartial (joinOneStepWithParallelWithVar S symP) cps
	       end

	   val (spNonJoinables,spProblematicSRules) = 
	       let val llcps = criticalPairs2WithRules (S,symP)
		   val _ = print ("CP(S,symP):\n")
	       in LP.unzip (L.mapPartial 
				(fn (lr,lr',u,v) => 
				    case joinOneStepWithParallel S symP (u,v) of
					SOME (u,u1,v,v1) => SOME ((u,u1,v,v1),lr) 
				      | NONE => NONE)
				llcps)
	       end
       in if (null ssNonJoinables)
	      andalso (null psNonJoinables)
	      andalso (null spNonJoinables)
	  then SOME (true,symP,[],[],[])
	  else if (!useRelative)
		  andalso (checkConfluenceConditionByRelativeTerminationPCP
			       isRelativeTerminating S symP 
			       (ssNonJoinables,psNonJoinables,spNonJoinables))
	  then SOME (true,symP,[],[],[])
	  else if not (!useCompletion)
	  then SOME (false,symP,[],[],[])
	  else let val (ssCand,spCand,spProb) = 
		       getCandidates symP
				     ssNonJoinables
				     (L.map (fn (u,u1,v,v1,vset) => (u,u1,v,v1)) 
					    psNonJoinables)
				     spNonJoinables
				     spProblematicSRules
	       in SOME (false,symP,ssCand,spCand,spProb)			    
	       end
       end

   (* $BDI2C$5$l$k(BCP$B$N%R%e!<%j%9%F%#%/%9!'(B
	  $BN>JU$N%k!<%H$,F1$8Dj5A5-9f$J$i!$N>JU$K9=@.;R5-9f$r4^$^$J$$!$(B
          $BN>JU$N%k!<%H$,0[$J$kDj5A5-9f$J$i!$1&JU$K$O9=@.;R5-9f$r4^$^$J$$!$(B
	  $B:8JU$O@55,7A(B($B1&JU$O$$$D$b@55,7A(B)$B!$(B $B$r%A%'%C%/(B *)
   fun suitableCPforRules S dSymSet (l,r) = 
       let (* does not nicely *)
	   (* fun check (U,V) = (FS.numItems (FS.difference (V,U))) <= 1 *)
	   fun check (U,V) = FS.isSubset (U,V)
       in
	   Rewrite.isNormalForm S l
	   andalso
       case (funRootOfTerm l, funRootOfTerm r) of
	   (SOME f, SOME g) => if FS.member (dSymSet,f)
				  andalso Fun.equal (f,g)
			       then check (Term.funSetInTerm l,dSymSet)
				    andalso
				    check (Term.funSetInTerm r,dSymSet)
			       else if FS.member (dSymSet,f)
				       andalso FS.member (dSymSet,g)
			       then check (Term.funSetInTerm r,dSymSet)
			       else true
	 | _ => true
       end

   val maxRewriteStepsForAux = 3
   (* $BDI2C$9$k$H$-$K!$>iD9$J5,B'$O:o=|$7$?$$(B? *)
   fun deleteAuxiliaryRules rs = 
       let val steps = maxRewriteStepsForAux
	   fun subsumedBy1 (l,r) (s,t) =
	       TS.member (Rewrite.manyStepsReductSet [(l,r)] steps s, t)
	       
	   fun subsumedBy2 [] persist = persist
	     | subsumedBy2 ((l,r)::rest) persist =
	       if TS.member (Rewrite.manyStepsReductSet (rest @ persist) steps l,r)
	       then subsumedBy2 rest persist
	       else subsumedBy2 rest
				((l,r)::(L.filter (not o (subsumedBy1 (l,r))) persist))
       in
	   subsumedBy2 rs []
       end

   (* sn = 0|1 where 
               0:SN(S) check not yet tried
               1:SN(S) check already succeeded  *)

   fun checkConfluenceConditionParallel isTerminating isRelativeTerminating S P sn =
       if not (Trs.areLeftLinearRules S) 
       then (print "S: not left-linear\n"; NONE)
       else if not (sn = 1 orelse isTerminating S)
       then (print "S: unknown termination\n"; NONE)
       else (print "S: terminating\n";
	     if (!usePCP)
	     then checkCPConditionParallelPCP isRelativeTerminating S (takeSym P)
	     else checkCPConditionParallel isRelativeTerminating S (takeSym P))

   fun checkConfluenceConditionLinear isTerminating isRelativeTerminating S P sn =
       if not (Trs.areLinearRules S) 
       then (print "S: not linear\n"; NONE)
       else if not (sn = 1 orelse isTerminating S)
       then (print "S: unknown termination\n"; NONE)
       else (print "S: terminating\n";
	     checkCPConditionLinear isRelativeTerminating S (takeSym P))

   (* Relative Termination (Huet) $B$K4p$E$/9gN.@-H=Dj(B *)
   fun checkConfluenceConditionHuet
   	   isRelativeTerminating S P =
       if not (Trs.areLeftLinearRules S) 
       then NONE
       else if (isRelativeTerminating (S,P))
       then
	   let val _ = print "S/P: relatively terminating\n"
	       val _ = print "check CP condition:\n"
	       val symP = takeSym P
	       val ssCPs = criticalPairs S
	       val psCPs = insideCriticalPairs2 (symP,S)
	       val spCPs = criticalPairs2 (S,symP)
	       fun check0 (t1,t2) = 
		   let val nf1 = Rewrite.linf S t1
		       val nf2 = Rewrite.linf S t2
		   in Rewrite.isEquivalentWithLimit 
			  maxLimitJoinabilityCondition
			  symP (nf1,nf2)
		   end
	   in if
		   (List.all (fn (u,v) => not (Rewrite.isNormalForm S u))
   			     psCPs)
   		   andalso
   		   (List.all (fn (u,v) => not (Rewrite.isNormalForm S v))
   			     spCPs)
		   andalso L.all check0 ssCPs
		   andalso L.all check0 psCPs
		   andalso L.all check0 spCPs
	       then (print "success\n";SOME (true,symP,[],[],[]))
	       else (print "failed\n";NONE)
	   end
       else (print "S/P: unknown relative termination\n";NONE)

                
   (* $B8uJd$N@8@.(B *)
   fun getnext0 S P dSymSet usableKind maxTermSize ssCand spCand = 
       let fun removeAlready xs =
	       L.filter (fn lr =>
			    not (LU.member' (fn (lr,(l',r')) => 
					   (Subst.identicalModuloRenamingRule lr (l',r'))
					   orelse
					   (Subst.identicalModuloRenamingRule lr (r',l')))
				       lr (S @ P)) )
			xs

	   fun filterBySize xs = L.filter (fn (l,r) =>
					       (Term.termSize l <= maxTermSize)
					       andalso (Term.termSize r <= maxTermSize))
					  xs
	   val ps1 = removeAlready (filterBySize (L.filter (isSuitableForPrule dSymSet) ssCand))
	   val ys = removeAlready (filterBySize (L.filter (suitableCPforRules S dSymSet) spCand))
	   val _ = print (Trs.prRules ys)
	   val (ss2,ps2) = decompose dSymSet ys
	   val ps3 = LU.eliminateDuplication' 
			 (fn (l,r) => Subst.identicalModuloRenamingRule l r)
			 (ps1 @ ps2)
	   val ss3 = (* deleteAuxiliaryRules *) ss2
	   val newP = P @ ps3 
	   val isrev = Rewrite.isReversibleRules newP
	   val _ = print "Added S-Rules:\n"
	   val _ = print (Trs.prRules (if isrev then ss3 else ss3 @ ps3))
	   val _ = print "Added P-Rules:\n"
	   val _ = print (Trs.prRules (if isrev then ps3 else []))
	   val newS = S @ ss3
	   val newS' = newS @ ps3
       in
	   case (null ss3,isrev) of 
	       (true,  true) => L.map (fn i=>(newS,newP,1,i)) usableKind
	     | (true, false) => L.map (fn i=>(newS',P,0,i)) usableKind
	     | (false, true) => L.map (fn i=>(newS,newP,0,i)) usableKind
	     | (false,false) => L.map (fn i=>(newS,P,0,i)) usableKind
   (*    | (false,false) => [(newS,P,0,0),(newS,P,0,1),
			     (newS',P,0,0),(newS',P,0,1)] *)
       end					   

   (* inference by replacement rule *)
   fun replaceSRulesByOne S symP problematicSRules = 
       LU.mapAppend 
	   (fn i => let val (l,r) = L.nth (S,i)
		    in if LU.member' TP.equal (l,r) problematicSRules
		       then
			   L.map (fn r2=> 
				     (print ("replace: " ^ (Trs.prRule (l,r)) 
					     ^ " => " ^ (Trs.prRule (l,r2)) ^ "\n");
				      LU.replaceNth (S,i,(l,r2))))
				 (TS.listItems (Rewrite.oneStepReductSet symP r))
		       else []
		    end)
	   (L.tabulate (L.length S, fn x=>x))

   (* This does not work *)
   (* val maxRewriteStepsForElim = 3 *)
   (* (* replacement rule $B$K$h$kJdBj@8@.(B *) *)
   (* fun reduceByElimination symP [] (S,T) = (S,T) *)
   (*   | reduceByElimination symP ((l,r)::Q) (S,T) = *)
   (*     let val S' = LU.deleteOne' TP.equal (l,r) S *)
   (* 	   val set = Rewrite.manyStepsReductSet (S'@ symP) maxRewriteStepsForElim l *)
   (*     in if TS.member (set,r) *)
   (* 	  then reduceByElimination symP Q (S',T) *)
   (* 	  else reduceByElimination symP Q (S,(l,r)::T) *)
   (*     end *)

   fun getnext1 S P symP usableKind spProb = 
       if (!useForward)
       then
	   LU.mapAppend 
	       (fn S => L.map (fn i=> (S,P,0,i)) usableKind)
	       (replaceSRulesByOne S symP spProb)
          (**** This does not work
	   let val (S',spProb') = reduceByElimination symP spProb (S,[])
	   in LU.mapAppend 
		  (fn S' => L.map (fn i=> (S',P,0,i)) usableKind)
		  (replaceSRulesByOne S' symP spProb')
	   end **)
       else []


  (* $B3HBg9gN.@-H=Dj(B *)
   fun bidirectionalCompletion isTerminating isRelativeTerminating dSymSet R =
       let
         (* kind = 0|1|2 where 
                        0: try parallel version 
                        1: try linear version
                        2: try huet version
	  *)
          (* maxTermSize $B$h$jD9$$9`$r$b$DEy<0$OJdBj$KF~$l$J$$(B *)
	   val maxTermSize = let val n = List.foldl (fn ((l,r),i) => 
					    Int.max (i,
						     Int.max (Term.termSize l,
							      Term.termSize r)))
						    0 R
			     in n end

         val _ = (debug (fn _ =>
			    let val _ = print "options:"
				val _ = print " parallel("
				val xs = if (!useParallel) 
					 then print "on)"
					 else print "off)"
				val _ = if (!useParallel) 
					then if (!usePCP) 
				             then print "[with PCP]"
                                             else print "[w/o PCP]"
					else ()				      
				val _ = print ", linear("
				val ys = if (!useLinear) 
					 then print "on)"
					 else print "off)"
				val _ = print ", relative("
				val zs = if (!useRelative)
					 then print "on)"
					 else print "off)"
				val _ = print ", huet("
				val zs = if (!useHuet)
					 then print "on)"
                                      else print "off)"
				val _ = print ", completion("
				val _ = if (!useCompletion)
					then print "on)"
               				else print "off)"
				val _ = if (!useCompletion)
					then
					    if (!useForward)
					    then print "[with repl]"
               				    else print "[w/o repl]"
					else ()
				val _ = print "\n"
			    in ()
			    end))

         val usableKind = let  val xs = if (!useParallel)  then [0]  else []
                             val ys = if (!useLinear) then [1] else []
                             val zs = if (!useHuet)  then [2] else []
			  in xs @ ys @ zs
			  end

	   fun check i (S,P,sn,kind) = 
	       let val _ = print ("STEP: " ^ (Int.toString i))
		   val _ = case kind of
				   0 => print " (parallel)\n"
				 | 1 => print " (linear)\n"
				 | _ => print " (relative)\n"
                   val _ = print "S:\n"
		   val _ = print (Trs.prRules S)
		   val _ = print "P:\n"
		   val _ = print (Trs.prRules P)
	       (* val _ = print ("sn=" ^ (Int.toString sn) ^ "\n") *)
	       (* val _ = print ("kind=" ^ (Int.toString kind) ^ "\n") *)
	       in
		   if not (Rewrite.isReversibleRules P)
		   then (print "P: not reversible\n";NONE)
		   else
		       case kind of
			   0 => checkConfluenceConditionParallel isTerminating isRelativeTerminating S P sn
			 | 1 => checkConfluenceConditionLinear isTerminating isRelativeTerminating S P sn
			 | _ => checkConfluenceConditionHuet isRelativeTerminating S P
	       end

	   fun step _ [] _ = (print "failure(no possibility remains)\n"; false)
	     | step i ((S,P,sn,kind)::ps) done = 
	       if i > maxCompletionSteps
	       then (print "failure(limit exceeds)\n"; false)
	       else 
		   case check i (S,P,sn,kind) of 
		       NONE => (print ("failure(Step " ^ (Int.toString i) ^ ")\n"); 
				step (i+1) ps ((S,P,sn,kind)::done))
		     | SOME (true,_,_,_,_) => (print ("S:\n" ^ (Trs.prRules S));
					       print ("P:\n" ^ (Trs.prRules P));
					       print "Success\n"; true)
		     | SOME (false,symP,ssCand,spCand,spProb) =>
		       (print ("failure(Step " ^ (Int.toString i) ^ ")\n"); 
			if not (!useCompletion)
			then step (i+1) ps ((S,P,sn,kind)::done)
			else
			    let (* val next = getnext (S,P,sn,kind) dSymSet usableKind (ps@done)
						(symP,ssCand,spCand,spProb) *)
				val cand = getnext0 S P dSymSet usableKind maxTermSize ssCand spCand
				val cand2 = getnext1 S P symP usableKind spProb
				val next = 
				    L.filter 
					(fn (X,X',_,lin1) =>
					    L.all (fn (Y,Y',_,lin2) =>
						      kind <> lin2
						      orelse not (LP.allEq (fn (x,y) => 
									       Subst.identicalModuloRenamingRule x y)
 									   (X@X',Y@Y')))
						  ((S,P,sn,kind)::(ps @ done)))
					(cand @ cand2)
			    in  step (i+1) (ps @ next) ((S,P,sn,kind)::done)
			    end)

	   val (notP,maybeP) = 
	       List.partition 
	       (fn (l,r)=> not (isSuitableForPrule dSymSet (l,r)))
	       R
	  (* (Term.isVar r)
	   orelse
	   (not (FS.member (dSymSet,valOf (Term.funRootOfTerm r))))
	   orelse
	   not (Trs.isBidirectionalRule (l,r)) *)
				   
	   val (shouldP,maybeP2) = 
	       List.partition 
		   (fn (l,r)=> (isSome (Subst.match l r))
			       orelse
			       (LU.member' (fn (lr,(l',r')) => 
					       Subst.identicalModuloRenamingRule 
						   lr (r',l'))
					   (l,r) maybeP))
 		   maybeP
		 
	   val _ = debug (fn _ => print ("not P:\n" ^ (Trs.prRules notP)))
	   val _ = debug (fn _ => print ("maybe P:\n" ^ Trs.prRules maybeP2))
	   val _ = debug (fn _ => print ("should P:\n" ^ Trs.prRules shouldP))

	   val decomp = List.rev (LU.powerlistWithComplement maybeP2)

	   val decomp2 = List.filter (fn (S,P)=>
					 Rewrite.isReversibleRules (shouldP @ P))
				     decomp

       in let val (S,P) = decompose dSymSet R
	  in if null P then
		 (print "failure(empty P)\n"; false)
	     else step 1 (L.map (fn i=> (S,P,0,i)) usableKind) []
	  end
(*
step 1 (let val (S,P) = decompose dSymSet R
		  in L.map (fn i=> (S,P,0,i)) usableKind
		  end) []
*)
           (* step 1 (LU.mapAppend 
		   (fn (S,P) => L.map (fn i=> (notP@S,shouldP@P,0,i)) usableKind)
		   decomp)
		   [] *)
       end


   end (* of local *)

   end; (* of structure CrCompletion *)

