(******************************************************************************
 * Copyright (c) 2012-2014, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/cr_diagram.sml
 * description: confluence check by decreasing diagram techniques using SMT Solver
 * author: AOTO Takahito
 * 
 ******************************************************************************)


signature CR_DIAGRAM  = 
   sig
    val runDebug: bool ref
    val hasDecreasingCps: 
	string  (* minisat *)
	-> string  (* tmp dir *)
	-> (Term.term * Term.term) list 
	-> bool

    val fSymsOnVariablePathsInTerm: Term.term -> (Var.ord_key * (Fun.ord_key list)) list

   val isConfluentQuasiLinearSystem:
       (bool -> (Term.term * Term.term) list -> bool)
       -> string  (* smtSolverPath *)
       -> string  (* tmp dir *)
       -> (Term.term * Term.term) list 
       -> bool

   val isConfluentStronglyQuasiLinearSystem:
       (bool -> (Term.term * Term.term) list -> bool)
       -> string  (* smtSolverPath *)
       -> string  (* tmp dir *)
       -> (Term.term * Term.term) list 
       -> bool

end;

structure CrDiagram : CR_DIAGRAM= 
   struct

   local 
       open Term
       open Arith
(*        open Trs *)
(*        open Rewrite *)
(*        open Subst *)
(*        open Cr *)
       structure VS = VarSet
       structure VM = VarMap
       structure IM = IntMap
       structure FS = FunSet
       structure FM = FunMap
       structure FIM = FunIntMap
       structure IPM = IntPairMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure TS = TermSet
       open PrintUtil 
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   exception CrDiagramError

   val decreasingDiagramVersion = ref 5
   (*********************************
       1: basic version
       2: counting designated function symbol's occurrence
       3: with context weight
       4: 3 + extended comparison
       5: 4 + extended context weight
   *********************************)



   (* $B4m81BP$N9gN.Ns(B($BMQ$$$k=q$-49$(5,B'(B)$B$N@8@.(B *)
(*    fun joinSequenceOfCps rs = *)
(*        let  *)
(* 	   val icps = L.@ (Cr.insideCriticalPairsWithIndexAndFs rs, *)
(* 			   Cr.outsideCriticalPairsInOnesideWithIndexAndFs rs) *)

(* 	   val maxCount = 5 *)

(* 	   exception NonJoinableCPFound *)

(* 	   val anslist =  *)
(* 	       List.map (fn ((m,n),(fs,c1,c2)) => *)
(* 			    let  *)
(* 				val _ = print "Critical Pair <" *)
(* 				val _ = print ((Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">") *)
(* 				val _ = print (" by Rules <" ^ (Int.toString m) *)
(* 					       ^ ", " ^ (Int.toString n) ^ ">") *)
(* 				val _ = print (" preceded by [" ^ *)
(* 					       (PrintUtil.prSeq Fun.toString fs) ^ "]\n") *)
(* 			    in *)
(* 				case Cr.findCommonReductWithIndexAndFs rs maxCount *)
(* 								       ((fs,m,c1),([],n,c2)) of *)
(* 				    NONE => raise NonJoinableCPFound *)
(* 				  | SOME (ms,ns) =>  *)
(* 				    let val (_::ms') = L.map (fn (fs,i,_) => (fs,i)) (rev ms) *)
(* 					val (_::ns') = L.map (fn (fs,i,_) => (fs,i)) (rev ns) *)
(* 					val _ = print (" joinable by a reduction of rules <["  *)
(* 						       ^ (PrintUtil.prSeq  *)
(* 							      (fn (fs,i) =>  *)
(* 								  "(["  *)
(* 								  ^ (PrintUtil.prSeq Fun.toString fs)  *)
(* 								  ^ "]," ^ (Int.toString i) ^ ")") *)
(* 							      ms') *)
(* 						       ^ "], ["  *)
(* 						       ^ (PrintUtil.prSeq  *)
(* 							      (fn (fs,i) =>  *)
(* 								  "(["  *)
(* 								  ^ (PrintUtil.prSeq Fun.toString fs)  *)
(* 								  ^ "]," ^ (Int.toString i) ^ ")") *)
(* 							      ns') *)
(* 						       ^ "]>\n") *)
(* 				    in *)
(* 					((fs,m),([],n),ms',ns') *)
(* 				    end *)
(* 			    end) *)
(* 			icps *)

(*        in *)
(* 	   SOME anslist *)
(*        end *)
(*        handle NonJoinableCPFound => NONE *)


   (* $B4m81BP$N9gN.Ns(B($BMQ$$$k=q$-49$(5,B'(B)$B$N@8@.(B *)
   fun joinSequenceOfCps rs =
       let 
 	   val icps = L.@ (Cr.insideCriticalPairsWithIndexAndFs rs, 
 			   Cr.outsideCriticalPairsInOnesideWithIndexAndFs rs)

	   val maxCount = 5

	   exception NonJoinableCPFound

	   val anslist = 
	       List.map (fn ((m,n),(fs,c1,c2)) =>
			    let 
				val _ = print "Critical Pair <"
				val _ = print ((Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">")
				val _ = print (" by Rules <" ^ (Int.toString m)
					       ^ ", " ^ (Int.toString n) ^ ">")
				val _ = print (" preceded by [" ^
					       (PrintUtil.prSeq Fun.toString fs) ^ "]\n")

				val joinSeqs = Cr.findCommonReductWithIndexAndFs rs maxCount
										 ((fs,m,c1),([],n,c2))
				(* non-joinablity check *)
				val _ = if null joinSeqs
					then raise NonJoinableCPFound
					else ()
			    in
				L.map 
				    (fn (ms,ns) => 
					let val (_::ms') = L.map (fn (fs,i,_) => (fs,i)) (rev ms)
					    val (_::ns') = L.map (fn (fs,i,_) => (fs,i)) (rev ns)
					    val _ = print (" joinable by a reduction of rules <[" 
							   ^ (PrintUtil.prSeq 
								  (fn (fs,i) => 
								      "([" 
								      ^ (PrintUtil.prSeq Fun.toString fs) 
								      ^ "]," ^ (Int.toString i) ^ ")")
								  ms')
							   ^ "], [" 
							   ^ (PrintUtil.prSeq 
								  (fn (fs,i) => 
								      "([" 
								      ^ (PrintUtil.prSeq Fun.toString fs) 
								      ^ "]," ^ (Int.toString i) ^ ")")
								  ns')
							   ^ "]>\n")
					in
					    ((fs,m),([],n),ms',ns')
					end)
				    joinSeqs
			    end)
			icps
			
       in
	   SOME anslist
       end
       handle NonJoinableCPFound => NONE

   (* $B4m81BP$N9gN.Ns(B($BMQ$$$k=q$-49$(5,B'(B)$B$N@8@.(B *)
   fun joinSequenceOfCps2 rs =
       let 
	   val icps0 = L.@ (Cr.insideCriticalPairsWithIndexAndPosFs rs,
			    Cr.outsideCriticalPairsInOnesideWithIndexAndPosFs rs)

(* 	   val icps = L.@ (Cr.insideCriticalPairsWithIndexAndFs rs, *)
(* 			   Cr.outsideCriticalPairsInOnesideWithIndexAndFs rs) *)

(* 	   val _ = L.app   *)
(* 		       (fn ((m,n),(ps,fs,c1,c2)) => *)
(* 			   (print "Critical Pair <"; *)
(* 			    print ((Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">"); *)
(* 			    print (" by Rules <" ^ (Int.toString m) *)
(* 				   ^ ", " ^ (Int.toString n) ^ ">"); *)
(* 			    print (" preceded by [" ^ *)
(* 				   (PrintUtil.prSeq Fun.toString fs) ^ "]\n"); *)
(* 			    print (" position [" ^ *)
(* 				   (PrintUtil.prSeq Int.toString ps) ^ "]\n"))) *)
(* 		       icps0  *)

	   val icps = L.map (fn (mn,(ps,fs,y,z)) => (mn,(LP.zip (fs,ps),y,z))) icps0

(*	   val maxCount = 5 *)
	   val maxCount = 5

	   exception NonJoinableCPFound

	   fun prFunInt (f,i) = "(" ^ (Fun.toString f) ^ "," 
				^ (Int.toString i) ^ ")"

	   val anslist = 
	       List.map (fn ((m,n),(fps,c1,c2)) =>
			    let 
				val _ = print "Critical Pair <"
				val _ = print ((Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">")
				val _ = print (" by Rules <" ^ (Int.toString m)
					       ^ ", " ^ (Int.toString n) ^ ">")
				val _ = print (" preceded by [" ^
					       (PrintUtil.prSeq prFunInt fps) ^ "]\n")

				val joinSeqs = Cr.findCommonReductWithIndexAndPosFs rs maxCount
										 ((fps,m,c1),([],n,c2))
				(* non-joinablity check *)
				val _ = if null joinSeqs
					then raise NonJoinableCPFound
					else ()
			    in
				L.map 
				    (fn (ms,ns) => 
					let val (_::ms') = L.map (fn (fps,i,_) => (fps,i)) (rev ms)
					    val (_::ns') = L.map (fn (fps,i,_) => (fps,i)) (rev ns)
					    val _ = print (" joinable by a reduction of rules <[" 
							   ^ (PrintUtil.prSeq 
								  (fn (fps,i) => 
								      "([" 
								      ^ (PrintUtil.prSeq prFunInt fps) 
								      ^ "]," ^ (Int.toString i) ^ ")")
								  ms')
							   ^ "], [" 
							   ^ (PrintUtil.prSeq 
								  (fn (fps,i) => 
								      "([" 
								      ^ (PrintUtil.prSeq prFunInt fps) 
								      ^ "]," ^ (Int.toString i) ^ ")")
								  ns')
							   ^ "]>\n")
					in
					    ((fps,m),([],n),ms',ns')
					end)
				    joinSeqs
			    end)
			icps
			
       in
	   SOME anslist
       end
       handle NonJoinableCPFound => NONE

   fun mkFsMap counterRef fs = 
       L.foldl
           (fn (f,en) => (counterRef := 1 + !counterRef;
                          FM.insert (en,f,!counterRef)))
           FM.empty 
           fs

   fun mkFpsMap counterRef fps = 
       L.foldl
           (fn (fi,en) => (counterRef := 1 + !counterRef;
                           FIM.insert (en,fi,!counterRef)))
           FIM.empty 
           fps

   fun lookupFsMap (fsMap,f) = 
       case FM.find (fsMap,f) of
           SOME n => n
         | NONE => (print ("looking up fsMap fails: (" 
                           ^ (Fun.toString f) ^ ")\n");
                    raise CrDiagramError)


   fun lookupFpsMap (fpsMap,(f,i)) = 
       case FIM.find (fpsMap,(f,i)) of
           SOME n => n
         | NONE => (print ("looking up fpsMap fails: (" 
                           ^ (Fun.toString f) ^ ","
                           ^ (Int.toString i) ^ ")\n");
                    raise CrDiagramError)

   fun encodeGtOnFs fsMap (fs,gs) =
       let val polyFs = L.map (fn f => [Var (lookupFsMap (fsMap,f))]) fs
	   val polyGs = L.map (fn g => [Var (lookupFsMap (fsMap,g))]) gs
       in Atom (Gt (polyFs,polyGs))
       end

   fun encodeGeOnFs fsMap (fs,gs) =
       let val polyFs = L.map (fn f => [Var (lookupFsMap (fsMap,f))]) fs
	   val polyGs = L.map (fn g => [Var (lookupFsMap (fsMap,g))]) gs
       in Atom (Ge (polyFs,polyGs))
       end

   fun encodeEqOnFs fsMap (fs,gs) =
       let val polyFs = L.map (fn f => [Var (lookupFsMap (fsMap,f))]) fs
	   val polyGs = L.map (fn g => [Var (lookupFsMap (fsMap,g))]) gs
       in Atom (Eq (polyFs,polyGs))
       end

   fun encodeGtOnFps fpsMap (fps,gps) =
       let val polyFps = L.map (fn fp => [Var (lookupFpsMap (fpsMap,fp))]) fps
	   val polyGqs = L.map (fn gq => [Var (lookupFpsMap (fpsMap,gq))]) gps
       in Atom (Gt (polyFps,polyGqs))
       end

   fun encodeGeOnFps fpsMap (fps,gps) =
       let val polyFps = L.map (fn fp => [Var (lookupFpsMap (fpsMap,fp))]) fps
	   val polyGqs = L.map (fn gq => [Var (lookupFpsMap (fpsMap,gq))]) gps
       in Atom (Ge (polyFps,polyGqs))
       end

   fun encodeEqOnFps fpsMap (fps,gps) =
       let val polyFps = L.map (fn fp => [Var (lookupFpsMap (fpsMap,fp))]) fps
	   val polyGqs = L.map (fn gq => [Var (lookupFpsMap (fpsMap,gq))]) gps
       in Atom (Eq (polyFps,polyGqs))
       end

   fun encodeGtOnRs (i,j) = Atom (Gt ([[Var (i+1)]],[[Var (j+1)]]))
   fun encodeGeOnRs (i,j) = Atom (Ge ([[Var (i+1)]],[[Var (j+1)]]))
   fun encodeEqOnRs (i,j) = Atom (Eq ([[Var (i+1)]],[[Var (j+1)]]))

   fun encodeGt1 fsMap lenRs ((fs,m),(gs,x)) =
       if (!decreasingDiagramVersion >= 4)
       then
	   Disj [encodeGtOnRs (m,x),
 		 Conj [encodeEqOnRs (m,x),
 		       encodeGtOnFs fsMap (fs,gs)],
 		 Conj [encodeEqOnRs (m,x),
 		       encodeEqOnFs fsMap (fs,gs),
		       encodeGtOnRs (lenRs+m,lenRs+x)]]
       else 
	   Disj [encodeGtOnFs fsMap (fs,gs),
 		 Conj [encodeEqOnFs fsMap (fs,gs),
		       encodeGtOnRs (m,x)]]

   fun encodeEq1 fsMap lenRs ((fs,m),(gs,x)) =
       if (!decreasingDiagramVersion >= 4)
       then
	   Conj [encodeEqOnRs (m,x),
		 encodeEqOnFs fsMap (fs,gs),
		 encodeEqOnRs (lenRs+m,lenRs+x)]
       else
	   Conj [encodeEqOnFs fsMap (fs,gs),
		 encodeEqOnRs (m,x)]

   fun encodeEq2 fpsMap lenRs ((fps,m),(gqs,x)) =
       Conj [encodeEqOnRs (m,x),
	     encodeEqOnFps fpsMap (fps,gqs), 
	     encodeEqOnRs (lenRs+m,lenRs+x)]
		   
   fun encodeGt2 fpsMap lenRs ((fps,m),(gqs,x)) =
       Disj [encodeGtOnRs (m,x),
	     Conj [encodeEqOnRs (m,x),
		   encodeGtOnFps fpsMap (fps,gqs)],
	     Conj [encodeEqOnRs (m,x),
		   encodeEqOnFps fpsMap (fps,gqs),
		   encodeGtOnRs (lenRs+m,lenRs+x)]]

   (* $B4m81BP$N9gN.>r7o$N7W;;(B *)       
   fun makeDecreasingCondition0 rs =
       let
	   fun makeDecreasingCond0 (_,n) (_,y) = encodeEqOnRs (n,y)
	   fun makeDecreasingCond1 ((_,m),ms) =
	       Conj (L.map (fn (_,x) => encodeGtOnRs (m,x)) ms)
	   fun makeDecreasingCond2 ((_,m),(_,n),ms) =
	       Conj (L.map (fn (_,x) =>
			       Disj [encodeGtOnRs (m,x), 
				     encodeGtOnRs (n,x)])
			   ms)
	   fun makeCondSub (m,n,ms) =
	       let val len = length ms
		   val indexes = L.tabulate (len, fn i => i)
	       in
		   Disj ((makeDecreasingCond2 (m,n,ms))
			 ::
			 (L.map (fn i => let val xs = L.take (ms,i)
					     val (y::ys) = L.drop (ms,i)
					 in Conj [makeDecreasingCond1 (m,xs),
						  makeDecreasingCond0 n y,
						  makeDecreasingCond2 (m,n,ys) ]
					 end)
				(L.tabulate (length ms, fn i => i))))
	       end

	   fun makeCond (m,n,ms,ns) = Conj [makeCondSub (m,n,ms), makeCondSub (n,m,ns)]

	   fun makeCond2 xss = Disj (L.map makeCond xss)

       in case joinSequenceOfCps rs of
	      NONE => False
	    | SOME seqs => Conj (L.map makeCond2 seqs)
       end

   (* $B4m81BP$N9gN.>r7o$N7W;;(B *)
   fun makeDecreasingCondition1 fsMap lenRs rs =
       let
	   fun makeDecreasingCond1 (m,ms) =
	       Conj (L.map (fn x => encodeGt1 fsMap lenRs (m,x)) ms)
	   fun makeDecreasingCond2 (m,n,ms) =
	       Conj (L.map (fn x =>
			       Disj [encodeGt1 fsMap lenRs (m,x), 
				     encodeGt1 fsMap lenRs (n,x)])
			   ms)
	   fun makeCondSub (m,n,ms) =
	       let val len = length ms
		   val indexes = L.tabulate (len, fn i => i)
	       in
		   Disj ((makeDecreasingCond2 (m,n,ms))
			 ::
			 (L.map (fn i => let val xs = L.take (ms,i)
					     val (y::ys) = L.drop (ms,i)
					 in Conj [makeDecreasingCond1 (m,xs),
						  encodeEq1 fsMap lenRs (n,y),
						  makeDecreasingCond2 (m,n,ys) ]
					 end)
				(L.tabulate (length ms, fn i => i))))
	       end

	   fun makeCond (m,n,ms,ns) = Conj [makeCondSub (m,n,ms), makeCondSub (n,m,ns)]

	   fun makeCond2 xss = Disj (L.map makeCond xss)

       in case joinSequenceOfCps rs of
	      NONE => False
	    | SOME seqs => Conj (L.map makeCond2 seqs)
       end

   (* $B4m81BP$N9gN.>r7o$N7W;;(B *)
   fun makeDecreasingCondition2 fpsMap lenRs rs =
       let
	   fun makeDecreasingCond1 (m,ms) =
	       Conj (L.map (fn x => encodeGt2 fpsMap lenRs (m,x)) ms)
	   fun makeDecreasingCond2 (m,n,ms) =
	       Conj (L.map (fn x =>
			       Disj [encodeGt2 fpsMap lenRs (m,x), 
				     encodeGt2 fpsMap lenRs (n,x)])
			   ms)
	   fun makeCondSub (m,n,ms) =
	       let val len = length ms
		   val indexes = L.tabulate (len, fn i => i)
	       in
		   Disj ((makeDecreasingCond2 (m,n,ms))
			 ::
			 (L.map (fn i => let val xs = L.take (ms,i)
					     val (y::ys) = L.drop (ms,i)
					 in Conj [makeDecreasingCond1 (m,xs),
						  encodeEq2 fpsMap lenRs (n,y),
						  makeDecreasingCond2 (m,n,ys) ]
					 end)
				(L.tabulate (length ms, fn i => i))))
	       end

	   fun makeCond (m,n,ms,ns) = Conj [makeCondSub (m,n,ms), makeCondSub (n,m,ns)]

	   fun makeCond2 xss = Disj (L.map makeCond xss)

       in case joinSequenceOfCps2 rs of
	      NONE => False
	    | SOME seqs => Conj (L.map makeCond2 seqs)

       end


  (* root $B$+$i(B $BJQ?t=P8=$=$l$>$l$X;j$k%Q%9>e$N4X?t5-9fNs(B *)
   fun fSymsOnVariablePathsInTerm (Term.Var (y,_)) = [(y,[])]
     | fSymsOnVariablePathsInTerm (Term.Fun (f,ts,_)) =
       L.map (fn (x,fs) => (x,f::fs))
	     (LU.mapAppend fSymsOnVariablePathsInTerm ts)

   fun makeNonOverlappingCondition fsMap (l,r) =
       let val vfsInL = fSymsOnVariablePathsInTerm l
	   val vfsInR = fSymsOnVariablePathsInTerm r
	   val vs = VS.listItems (Term.varSetInTerm l)
       in
	   Conj (L.map
		     (fn x => let val lfs = hd (L.mapPartial (fn (y,fs) => if Var.equal (x,y)
									   then SOME fs
									   else NONE) 
							     vfsInL)
			      in  (* duplicating $B$JJQ?t(B $B$O(B > $B!$(Bnon-duplicating $B$J$i(B >= *)
				  case (L.mapPartial (fn (y,fs) => if Var.equal (x,y)
								   then SOME fs
								   else NONE) 
						     vfsInR) of
				      [] => True
				    | (rfs::[]) => encodeGeOnFs fsMap (lfs,rfs)
				    | rfss => Conj (L.map 
						       (fn rfs => 
							   encodeGtOnFs fsMap (lfs,rfs))
						       rfss)
			      end)
		     vs)
       end


  (* root $B$+$i(B $BJQ?t=P8=$=$l$>$l$X;j$k%Q%9>e$N4X?t5-9fNs(B *)
   fun fSymArgsOnVariablePathsInTerm (Term.Var (y,_)) = [(y,[])]
     | fSymArgsOnVariablePathsInTerm (Term.Fun (f,ts,_)) =
       LU.mapAppend (fn i => 
			L.map (fn (x,fps) => (x,(f,i+1)::fps))
			      (fSymArgsOnVariablePathsInTerm (L.nth (ts,i))))
		    (L.tabulate (length ts, fn x => x))

   fun makeNonOverlappingCondition2 fpsMap (l,r) =
       let val vfpsInL = fSymArgsOnVariablePathsInTerm l
	   val vfpsInR = fSymArgsOnVariablePathsInTerm r
	   val vs = VS.listItems (Term.varSetInTerm l)
       in
	   Conj (L.map
		     (fn x => let val lfps = hd (L.mapPartial (fn (y,fps) => if Var.equal (x,y)
									     then SOME fps
									     else NONE) 
							      vfpsInL)
			      in  (* duplicating $B$JJQ?t(B $B$O(B > $B!$(Bnon-duplicating $B$J$i(B >= *)
				  case (L.mapPartial (fn (y,fps) => if Var.equal (x,y)
								    then SOME fps
								    else NONE) 
						     vfpsInR) of
				      [] => True
				    | (rfps::[]) => encodeGeOnFps fpsMap (lfps,rfps)
				    | rpfss => Conj (L.map 
						       (fn rfps => 
							   encodeGtOnFps fpsMap (lfps,rfps))
						       rpfss)
			      end)
		     vs)
       end

  (* $B4pK\%k!<%k%i%Y%j%s%0(B *)
   fun hasDecreasingCps0 smtSolverPath tmpDir rs =
       if (Trs.areLinearRules rs)
       then
	   let 
	       val lenRs = List.length rs
	       val counter = ref lenRs  (* integer variable $B$N%+%&%s%?(B *)
	       val _ = print (Int.toString (!counter))
	       val _ = print "\n"
	       val prop = makeDecreasingCondition0 rs
	       val (result,assign) = Solver.smtSolver smtSolverPath tmpDir (prop,!counter)
	       fun lookupVal i = case L.find (fn (j,_) => i = j) assign of 
				  SOME (_,w) =>  w
				| NONE => (print ("no assingment for x" ^ (Int.toString i) ^ "\n");
					   raise CrDiagramError)
	       fun countOrder len = 
		   if result
		   then
		   let val sorted =
			   ListMergeSort.sort (fn ((i1,i2),(j1,j2)) => 
						  i1 > j1
						  orelse ((i1 = j1) andalso j1 > j2))
					      (L.tabulate (lenRs, fn i => (lookupVal (len+i+1),i+1)))
(* 		       val _ = print "\n" *)
(* 		       val _ = L.app  *)
(* 			       (fn (i,j) => (print ((Int.toString i) ^ ":=:" ^ (Int.toString j)))) *)
(* 			       sorted *)
(* 		       val _ = print "\n" *)
		   in case sorted of
			  [] => (0,"")
			| ((x1,x2)::xs) => L.foldl
						(fn ((i,j),(pre,str)) => 
						    if i > pre
						    then (i, (Int.toString j) ^  ">" ^ str)
						    else (i, (Int.toString j) ^  "," ^ str))
						(x1, Int.toString x2)
						xs
		   end
		   else (0,"")

	       val _ = if result then print ("Satisfiable by " ^ (#2 (countOrder 0)) ^ "\n")  else ()



	   in 
	       result
	   end
       else false

  (* $B=E$_IU$-%k!<%k%i%Y%j%s%0(B *)
   fun hasDecreasingCps1 smtSolverPath tmpDir rs =
       if (Trs.areLeftLinearRules rs)
       then
	   let 
	       val lenRs = List.length rs
	       val counter = ref (lenRs + lenRs) (* integer variable $B$N%+%&%s%?(B *)
	       val _ = print (Int.toString (!counter))
	       val _ = print "\n"
	       val fs = FS.listItems (Trs.funSetInRules rs)
               val fsMap = mkFsMap counter fs (* fsMap: $B4X?t5-9f(B |-> $BJQ?tHV9f(B *)
(* 	       val _ = List.app (fn f => (print (Fun.toString f);  *)
(* 					  print (Int.toString (lookupFsMap (fsMap,f))); *)
(* 					  print "\n"))  fs *)
(* 	       val _ = print "\n" *)
(* 	       val _ = print (Int.toString (!counter)) *)
(* 	       val _ = print "\n" *)


	       (* $B4X?t5-9f$N=E$_$O(B1$B0J2<(B *)
(*  	       val prop0 = Conj (L.tabulate (!counter - lenRs,  *)
(*  					  fn i => (Atom (Ge ([[Const 1]], [[Var (i+lenRs)]]))))) *)

	       val prop1 = Conj (L.map (makeNonOverlappingCondition fsMap) rs)
	       val prop2 = makeDecreasingCondition1 fsMap lenRs rs
(*	       val prop = Conj [prop0,prop1,prop2] *)
	       val prop = if (!decreasingDiagramVersion) = 2
			  then
			      let (* $B4X?t5-9f$N=E$_$NOB$O(B1$B0J2<(B *)
    				  val prop0 = Atom (Ge ([[Const 1]],  
  							L.tabulate (!counter - lenRs, 
								 fn i => [Var (i+lenRs)]))) 
			      in Conj [prop0,prop1,prop2]
			      end
			  else Conj [prop1,prop2] 


	       val (result,assign) = Solver.smtSolver smtSolverPath tmpDir (prop,!counter)

	       fun lookupVal i = case L.find (fn (j,_) => i = j) assign of 
				  SOME (_,w) =>  w
				| NONE => (print ("no assingment for x" ^ (Int.toString i) ^ "\n");
					   raise CrDiagramError)

	       fun countOrder len = 
		   if result
		   then
		   let val sorted =
			   ListMergeSort.sort (fn ((i1,i2),(j1,j2)) => 
						  i1 > j1
						  orelse ((i1 = j1) andalso j1 > j2))
					      (L.tabulate (lenRs, fn i => (lookupVal (len+i+1),i+1)))
(* 		       val _ = print "\n" *)
(* 		       val _ = L.app  *)
(* 			       (fn (i,j) => (print ((Int.toString i) ^ ":=:" ^ (Int.toString j)))) *)
(* 			       sorted *)
(* 		       val _ = print "\n" *)
		   in case sorted of
			  [] => (0,"")
			| ((x1,x2)::xs) => L.foldl
						(fn ((i,j),(pre,str)) => 
						    if i > pre
						    then (i, (Int.toString j) ^  ">" ^ str)
						    else (i, (Int.toString j) ^  "," ^ str))
						(x1, Int.toString x2)
						xs
		   end
		   else (0,"")

	       val _ = if result then print ("Satisfiable by " ^ (#2 (countOrder 0)) ^ "; ")  else ()

	       val fws
		 = if result
		   then L.foldl
			    (fn (f,str) => 
				(str ^ (Fun.toString f ^ "(" ^
					(Int.toString (lookupVal (lookupFsMap (fsMap,f))))
					^ ")")))
			    ""
			    fs
		   else ""

	       val _ = if result 
		       then if (!decreasingDiagramVersion) <  4
			    then print (fws ^ "\n") 
			    else print (fws ^ "; ") 
		       else ()

	       val _ = if result 
			  andalso (!decreasingDiagramVersion) =  4
		       then print ((#2 (countOrder lenRs)) ^ "\n") 
		       else ()

	   in 
	       result
	   end
       else false

  (* $B3HD%%k!<%k%i%Y%j%s%0(B *)
   fun hasDecreasingCps2 smtSolverPath tmpDir rs =
       if (Trs.areLeftLinearRules rs)
       then
	   let 
	       val lenRs = List.length rs
	       val counter = ref (lenRs + lenRs) (* integer variable $B$N%+%&%s%?(B *)
(* 	       val _ = print (Int.toString (!counter)) *)
(* 	       val _ = print "\n" *)
	       val fargs = FIS.listItems (Trs.funAritySetInRules rs)
	       val fps = LU.mapAppend 
			     (fn (f,arg) => L.tabulate (arg, fn i=>(f,i+1)))
			     fargs
               val fpsMap = mkFpsMap counter fps (* fpsMap: ($B4X?t5-9f(B,int) |-> $BJQ?tHV9f(B *)
(* 	       val _ = List.app (fn (f,i) =>  *)
(* 				    (print ("(" ^ (Fun.toString f) ^ "," *)
(* 					    ^ (Int.toString i) ^ ") |-> "); *)
(* 				     print (Int.toString (lookupFpsMap (fpsMap,(f,i)))); *)
(* 				     print "\n"))  fps *)
(* 	       val _ = print "\n" *)
(* 	       val _ = print (Int.toString (!counter)) *)
(* 	       val _ = print "\n" *)
	       val prop1 = Conj (L.map (makeNonOverlappingCondition2 fpsMap) rs)
	       val prop2 = makeDecreasingCondition2 fpsMap lenRs rs
	       val prop = Conj [prop1,prop2]


	       val (result,assign) = Solver.smtSolver smtSolverPath tmpDir (prop,!counter)


	       fun lookupVal i = case L.find (fn (j,_) => i = j) assign of 
				  SOME (_,w) =>  w
				| NONE => (print ("no assingment for x" ^ (Int.toString i) ^ "\n");
					   raise CrDiagramError)

	       fun countOrder len = 
		   if result
		   then
		   let val sorted =
			   ListMergeSort.sort (fn ((i1,i2),(j1,j2)) => 
						  i1 > j1
						  orelse ((i1 = j1) andalso j1 > j2))
					      (L.tabulate (lenRs, fn i => (lookupVal (len+i+1),i+1)))
(* 		       val _ = print "\n" *)
(* 		       val _ = L.app  *)
(* 			       (fn (i,j) => (print ((Int.toString i) ^ ":=:" ^ (Int.toString j)))) *)
(* 			       sorted *)
(* 		       val _ = print "\n" *)
		   in case sorted of
			  [] => (0,"")
			| ((x1,x2)::xs) => L.foldl
						(fn ((i,j),(pre,str)) => 
						    if i > pre
						    then (i, (Int.toString j) ^  ">" ^ str)
						    else (i, (Int.toString j) ^  "," ^ str))
						(x1, Int.toString x2)
						xs
		   end
		   else (0,"")

	       val _ = if result then print ("Satisfiable by " ^ (#2 (countOrder 0)) ^ "; ")  else ()

	       val fps = if result
			 then L.foldl
			     (fn ((f,arg),str) => 
				 if arg  = 0
				 then str
				 else
				     (str ^ (Fun.toString f ^ 
					    (ListUtil.toStringCommaRound
						 (fn j => Int.toString (lookupVal j))
						 (L.tabulate (arg, fn i=> lookupFpsMap (fpsMap,(f,i+1)))))
					    )))
			     ""
			     fargs
			 else ""

	       val _ = if result then print (fps ^ "; ") else ()
	       val _ = if result then print ((#2 (countOrder lenRs)) ^ "\n") else ()

	   (* 	       val _ = if result  *)
	   (* 		       then (print "Decreasing by: "; printInfo resultArray encoding) *)
	   (* 		       else () *)
	   in 
	       result
	   end
       else false


   fun hasDecreasingCps smtSolverPath tmpDir rs =
       case (!decreasingDiagramVersion) of
	   1 => hasDecreasingCps0 smtSolverPath tmpDir rs
	 | 2 => hasDecreasingCps1 smtSolverPath tmpDir rs
	 | 3 => hasDecreasingCps1 smtSolverPath tmpDir rs
	 | 4 => hasDecreasingCps1 smtSolverPath tmpDir rs
	 | _ => hasDecreasingCps2 smtSolverPath tmpDir rs


(* for test 

val rs = IOFotrs.rdRules [ "nats -> cons(0,inc(nats))",  
			   "inc(cons(?x,?ys)) -> cons(s(?x),inc(?ys))",
			   "hd(cons(?x,?ys)) -> ?x",
			   "tl(cons(?x,?ys)) -> ?ys",
			   "inc(tl(nats)) -> tl(inc(nats))"
			 ];

val rs20 = IOFotrs.rdRules [ "g(a) -> f(g(a))",  
			     "g(b) -> c",
			     "a -> b",
			     "f(?x) -> h(?x,?x)",
			     "h(?x,?y) -> c"
			   ];


val rs21 = IOFotrs.rdRules [ "b -> a",
			     "b -> c",
			     "c -> h(b)",
			     "c -> d",
			     "a -> h(a)",
			     "d -> h(d)" ];


val minisatPath = "../../work/tools/bin/minisat"

val tmpDir = "../../work/rwchecker/"

*)

(**************************************************)
(*** $B1JB3@-$H8:>/%@%$%d%0%i%`$K4p$E$/9gN.@->ZL@K!(B ***)
(**************************************************)

   exception NonJoinableCPFound

   val maxCount = 3 (* for checking convertibility of cp *)

   (* $B4m81BP$N9gN.Ns(B($BMQ$$$k=q$-49$(5,B'(B)$B$N@8@.(B *)
   fun getIndexOfConvSeq rs ((m,n),(c1,c2)) = 
       let val _ = print ("Critical Pair <" ^ (Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">")
	   val _ = print (" by Rules <" ^ (Int.toString m) ^ ", " ^ (Int.toString n) ^ ">\n")
	   val convSeqs = Cr.findConvSequencesWithIndex rs maxCount ((m,c1),(n,c2))
	   val _ = if null convSeqs then raise NonJoinableCPFound else ()
       in
	   L.map 
	       (fn ms => 
		   let val _ = print (" convertible by a reduction of rules [" 
				      ^ (PrintUtil.prSeq 
					     (fn (dir,i) => (
						 if dir = 1
						 then "->(" ^ (Int.toString i) ^ ")"
						 else "(" ^ (Int.toString i) ^ ")<-")) (* dir = ~1 *) 
					     ms)
				      ^ "]\n")
		   in (m,n,ms)
		   end)
	       convSeqs
       end

   fun revSeq (m,n,ms) = (n,m,rev (L.map (fn (dir,i) => (~1*dir,i)) ms))
   fun revSeqs us = L.map revSeq us

   fun checkProp smtSolverPath tmpDir rs prop =
       let
	   val lenRs = List.length rs
	   val counter = ref lenRs  (* integer variable $B$N%+%&%s%?(B *)
	   (* val _ = print (Int.toString (!counter)) *)
	   (* val _ = print "\n" *)
	   val (result,assign) = Solver.smtSolver smtSolverPath tmpDir (prop,!counter)
	   fun lookupVal i = case L.find (fn (j,_) => i = j) assign of
				 SOME (_,w) =>  w
			       | NONE => (print ("no assingment for x" ^ (Int.toString i) ^ "\n");
					  raise CrDiagramError)
	   fun countOrder len =
	       if result
	       then
		   let val sorted =
			   ListMergeSort.sort (fn ((i1,i2),(j1,j2)) =>
						  i1 > j1
						  orelse ((i1 = j1) andalso j1 > j2))
					      (L.tabulate (lenRs, fn i => (lookupVal (len+i+1),i+1)))
(* 		       val _ = print "\n" *)
(* 		       val _ = L.app  *)
(* 			       (fn (i,j) => (print ((Int.toString i) ^ ":=:" ^ (Int.toString j)))) *)
(* 			       sorted *)
(* 		       val _ = print "\n" *)
		   in case sorted of
			  [] => (0,"")
			| ((x1,x2)::xs) => L.foldl
						(fn ((i,j),(pre,str)) =>
						    if i > pre
						    then (i, (Int.toString (j-1)) ^  ">" ^ str)
						    else (i, (Int.toString (j-1)) ^  "," ^ str))
						(x1, Int.toString (x2-1))
						xs
		   end
		   else (0,"")

	   val _ = if result 
		   then print ("Satisfiable by " ^ (#2 (countOrder 0)) ^ "\n")  
		   else print ("Not Satisfiable\n")

       in result
       end

   fun getLeqTypes funEnv' types = 
       let fun addTypes tmp = LU.mapAppend (fn (f,args,ret) => if LU.member ret tmp then args else []) 
					   funEnv'
	   fun getClosure tmp = let val added = addTypes tmp
				in if L.all (fn ty => LU.member ty tmp) added
				   then tmp
				   else getClosure (LU.union (added, tmp))
				end
       in getClosure types 
       end

   fun makeSort (args,ty) = if (null args) 
			    then (Sort.fromString (Int.toString ty))
			    else Sort.Proc (L.map (fn sy => (Sort.fromString (Int.toString sy))) args,
					    Sort.fromString (Int.toString ty))

   fun makeDecl (f,args,ty) = {sym=f, sort=makeSort (args,ty)}

   (*** $B@~7A2=J]B86I=j8:>/@-$NH=Dj(B ***)

   fun isConfluentQuasiLinearSystem isTerminating
				   smtSolverPath tmpDir rs =
       let val (varEnv,funEnv,cs,rs') = CrDirect.makeInitialEnv rs
	   val _ = print (Trs.prRules rs')
	   val (varEnv',funEnv') = CrDirect.inferTypesFromTermConstraints (varEnv,funEnv) cs
	   val _ = print "Sort Assignment:\n"
	   val _ = L.app (fn (f,ms,m) => print (" " ^ (Fun.toString f)
						^ " : "
						^ (PrintUtil.prProd Int.toString ms)
						^ "=>"
						^ (Int.toString m) ^ "\n"))
			 funEnv'

	   val nonLinearVars = LU.mapAppend (fn (l,r) => 
						(Term.nonLinearVarListInTerm l) @ (Term.nonLinearVarListInTerm r))
					    rs'
	   val _ = print "non-linear variables: "
	   val _ = print ((LU.toStringCommaCurly Var.toString nonLinearVars) ^ "\n")

	   val nonLinearTypes = LU.eliminateDuplication (L.map (CrDirect.lookupVarEnv varEnv') nonLinearVars)
	   val _ = print "non-linear types: "
	   val _ = print ((LU.toStringCommaCurly Int.toString nonLinearTypes) ^ "\n")

	   val typesLeqNonLinearTypes = getLeqTypes funEnv' nonLinearTypes
	   val _ = print "types leq non-linear types: "
	   val _ = print ((LU.toStringCommaCurly Int.toString nonLinearTypes) ^ "\n")

	   val rulesApplicableToNonLinearTypes = 
	       L.filter (fn (l,r) => LU.member (CrDirect.typesOfTerm (varEnv',funEnv') l) 
					       typesLeqNonLinearTypes) rs'

	   val _ = print "rules applicable to terms of non-linear types:\n"
	   val _ = print (Trs.prRules rulesApplicableToNonLinearTypes)

	   val decls = L.map makeDecl funEnv' 
	   val sortedRules = case Trs.attachSortToRules decls rs of
				 SOME ss => ss
			       | NONE => (print "isConfluentWeakLeftLinearSystem\n";
					  raise CrDiagramError)

	   val indexesOfRulesApplicableToEachNonLinearTypes =
	       L.map (fn ty => let val letypes = getLeqTypes funEnv' [ty]
				   val indexes =
				       L.filter (fn i => let val (l,r) = L.nth (rs',i)
						      in LU.member (CrDirect.typesOfTerm (varEnv',funEnv') l) 
								   letypes
							 end)
						(L.tabulate (L.length rs', fn i => i))
			       in (ty,indexes)
			       end)
		     typesLeqNonLinearTypes

	   val indexesOfNLRforEachRule =
	       L.map (fn i => let val (l,r) = L.nth (rs',i)
				  val nlvars = (Term.nonLinearVarListInTerm l) @ (Term.nonLinearVarListInTerm r)
				  val nltypes = LU.eliminateDuplication (L.map (CrDirect.lookupVarEnv varEnv') nlvars)
				  val index0 = LU.mapAppend (fn (i,xs) => if LU.member i nltypes
									  then xs else [])
							    indexesOfRulesApplicableToEachNonLinearTypes 
			      in (i,LU.eliminateDuplication index0)
			      end)
		     (L.tabulate (L.length rs', fn i => i))

	   val _ = (print "Rnl:\n";
		    L.app (fn (i,xs) => print ((Int.toString i) ^ ": "
					       ^ (LU.toStringCommaCurly Int.toString xs)
					       ^"\n"))			 
			  indexesOfNLRforEachRule)

	   fun getIndexesOfNLRforRules xs =
	       LU.eliminateDuplication (LU.mapAppend (fn (i,ys) => if LU.member i xs
								   then ys else [])
						     indexesOfNLRforEachRule)

	   (* encoding local decreasing condition *)

   	   fun makeDecreasingCond0 (x,y) = encodeEqOnRs (x,y) 
   	   fun makeDecreasingCond1 (x,ys) = Conj (L.map (fn y => encodeGtOnRs (x,y)) ys)
   	   fun makeDecreasingCond2 (m,n,xs) = 
	       Conj (L.map (fn x => Disj [encodeGtOnRs (m,x), encodeGtOnRs (n,x)]) xs)
	   fun makeNLRCond1 (x,ys) =
	       Conj (L.map (fn y => encodeGtOnRs (x,y)) (getIndexesOfNLRforRules ys))
	   fun makeNLRCond2 (m,n,ys) =
	       Conj (L.map (fn y => Disj [encodeGtOnRs (m,y), encodeGtOnRs (n,y)])
			   (getIndexesOfNLRforRules ys))

	   fun makeCondH0V0 (m,n,ms) =
	       Conj [makeDecreasingCond2 (m,n,L.map (fn (_,x)=>x) ms),
   		     makeNLRCond2 (m,n,L.map (fn (_,x)=>x) ms)]

   	   fun makeCondV j (m,n,ms) =
	       let val xs = L.take (ms,j)
   		   val ((dir,y)::ys) = L.drop (ms,j)
   	       in if dir <> ~1 
		  then False
		  else Conj [makeDecreasingCond2 (m,n,L.map (fn (_,x)=>x) xs),
   			     makeNLRCond2 (m,n,L.map (fn (_,x)=>x) xs),
   			     (* makeDecreasingCond0 (n,y), bug fix 2014/6/28 *)
   			     makeDecreasingCond0 (m,y),
   			     makeNLRCond1 (n,y::L.map (fn (_,x)=>x) ys),
			     makeDecreasingCond1 (n,L.map (fn (_,x)=>x) ys)]
   	       end

   	   fun makeCondH0 (m,n,ms) = 
	       Disj ((makeCondH0V0 (m,n,ms))
		     :: (L.tabulate (L.length ms,fn j => makeCondV j (m,n,ms))))

   	   fun makeCondH i (m,n,ms) =
	       let val xs = L.take (ms,i)
   		   val ((dir,y)::ys) = L.drop (ms,i)
   	       in if dir <> 1 
		  then False
		  else Conj [makeDecreasingCond1 (m,L.map (fn (_,x)=>x) xs),
   			     makeDecreasingCond0 (n,y),
   			     makeNLRCond1 (m,y::L.map (fn (_,x)=>x) xs),
			     Disj (L.tabulate (L.length ys,fn j => makeCondV j (m,n,ys)))]
   	       end

   	   fun makeCond (m,n,ms) =
   	       Disj ((makeCondH0 (m,n,ms))
   		     :: (L.tabulate (length ms, fn i => makeCondH i (m,n,ms))))
   	   fun makeCond2 xss = Disj (L.map makeCond xss)

       in if isTerminating true rulesApplicableToNonLinearTypes (* SIN $B$N%A%'%C%/(B *)
	  then let val _ = print "terms of non-linear types are innermost terminating\n"
 		   val icps = L.@ (Cr.insideCriticalPairsWithIndex rs, 
 				   Cr.outsideCriticalPairsInOnesideWithIndex rs)
		   val seqs = List.map (getIndexOfConvSeq rs) icps (* may raise NonJoinableCPFound *)
		   val prop = Conj (L.map makeCond2 seqs)
		   val ans = checkProp smtSolverPath tmpDir rs prop
	       in ans
	       end
	  else (print "unknown innermost-termination for terms of non-linear types\n"; false)
       end
       handle NonJoinableCPFound => (print "no joinable sequence for some critical pairs\n"; false)



   (*** $B%=!<%9%i%Y%j%s%0$H%k!<%k%i%Y%j%s%0$rAH$_9g$o$;$?H=DjK!(B ***)

   fun isConfluentStronglyQuasiLinearSystem isTerminating
				   smtSolverPath tmpDir rs =
       let val (varEnv,funEnv,cs,rs') = CrDirect.makeInitialEnv rs
	   val _ = print (Trs.prRules rs')
	   val (varEnv',funEnv') = CrDirect.inferTypesFromTermConstraints (varEnv,funEnv) cs
	   val _ = print "Sort Assignment:\n"
	   val _ = L.app (fn (f,ms,m) => print (" " ^ (Fun.toString f)
						^ " : "
						^ (PrintUtil.prProd Int.toString ms)
						^ "=>"
						^ (Int.toString m) ^ "\n"))
			 funEnv'

	   val nonLinearVars = LU.mapAppend (fn (l,r) => 
						(Term.nonLinearVarListInTerm l) @ (Term.nonLinearVarListInTerm r))
					    rs'
	   val _ = print "non-linear variables: "
	   val _ = print ((LU.toStringCommaCurly Var.toString nonLinearVars) ^ "\n")

	   val nonLinearTypes = LU.eliminateDuplication (L.map (CrDirect.lookupVarEnv varEnv') nonLinearVars)
	   val _ = print "non-linear types: "
	   val _ = print ((LU.toStringCommaCurly Int.toString nonLinearTypes) ^ "\n")

	   val typesLeqNonLinearTypes = getLeqTypes funEnv' nonLinearTypes
	   val _ = print "types leq non-linear types: "
	   val _ = print ((LU.toStringCommaCurly Int.toString nonLinearTypes) ^ "\n")

	   val indexesOfNLR =
	       L.filter (fn i => let val (l,r) = L.nth (rs',i)
				 in LU.member (CrDirect.typesOfTerm (varEnv',funEnv') l) 
					      typesLeqNonLinearTypes
				 end)
			(L.tabulate (L.length rs', fn i => i))
	   val removeNLRfromSeq = L.filter (fn (dir,i) => not (LU.member i indexesOfNLR))
	   val rulesApplicableToNonLinearTypes = 
	       L.map (fn i => L.nth (rs',i)) indexesOfNLR
	   val _ = print "rules applicable to terms of non-linear types:\n"
	   val _ = print (Trs.prRules rulesApplicableToNonLinearTypes)

	   val decls = L.map makeDecl funEnv' 
	   val sortedRules = case Trs.attachSortToRules decls rs of
				 SOME ss => ss
			       | NONE => (print "isConfluentWeakLeftLinearSystem\n";
					  raise CrDiagramError)

	   (* encoding local decreasing condition *)

   	   fun makeDecreasingCond0 (x,y) = encodeEqOnRs (x,y) 
   	   fun makeDecreasingCond1 (x,ys) = Conj (L.map (fn y => encodeGtOnRs (x,y)) ys)
   	   fun makeDecreasingCond2 (m,n,xs) = 
	       Conj (L.map (fn x => Disj [encodeGtOnRs (m,x), encodeGtOnRs (n,x)]) xs)
	   fun makeCondH0V0 (m,n,ms) = makeDecreasingCond2 (m,n,L.map (fn (_,x)=>x) ms)
   	   fun makeCondV j (m,n,ms) =
	       let val xs = L.take (ms,j)
   		   val ((dir,y)::ys) = L.drop (ms,j)
   	       in if dir <> ~1 
		  then False
		  else Conj [makeDecreasingCond2 (m,n,L.map (fn (_,x)=>x) xs),
   			     (* makeDecreasingCond0 (n,y), bug fix 2014/6/28 *)
   			     makeDecreasingCond0 (m,y),
			     makeDecreasingCond1 (n,L.map (fn (_,x)=>x) ys)]
   	       end

   	   fun makeCondH0 (m,n,ms) = 
	       Disj ((makeCondH0V0 (m,n,ms))
		     :: (L.tabulate (L.length ms,fn j => makeCondV j (m,n,ms))))

   	   fun makeCondH i (m,n,ms) =
	       let val xs = L.take (ms,i)
   		   val ((dir,y)::ys) = L.drop (ms,i)
   	       in if dir <> 1 
		  then False
		  else Conj [makeDecreasingCond1 (m,L.map (fn (_,x)=>x) xs),
   			     makeDecreasingCond0 (n,y),
			     Disj (L.tabulate (L.length ys,fn j => makeCondV j (m,n,ys)))]
   	       end

   	   fun makeCond (m,n,ms) =
	       let val ms' = removeNLRfromSeq ms
	       in Disj ((makeCondH0 (m,n,ms'))
   		     :: (L.tabulate (length ms', fn i => makeCondH i (m,n,ms'))))
	       end

   	   fun makeCond2 xss = Disj (L.map makeCond xss)

   	   fun makeCondHforLvsH (n,ms) =
	       if null ms
	       then False
	       else let val ((dir,y)::ys) = ms
   		    in if dir <> 1 
		       then False
		       else if LU.member y indexesOfNLR
		       then makeCondHforLvsH (n,ys)
		       else Conj [makeDecreasingCond0 (n,y),
				  makeDecreasingCond1 (n,L.map (fn (_,x)=>x) ys)]
   		    end

   	   fun makeCondforLvsH (_,n,ms) =
	       Disj [makeDecreasingCond1 (n,L.map (fn (_,x)=>x) (removeNLRfromSeq ms)),
   			makeCondHforLvsH (n,ms)]

   	   fun makeCond1 xss = Disj (L.map makeCondforLvsH xss)

       in if isTerminating false rulesApplicableToNonLinearTypes (* SN $B$N%A%'%C%/(B *)
	  then let val _ = print "terms of non-linear types are terminating\n"
 		   val icps = L.@ (Cr.insideCriticalPairsWithIndex rs, 
 				   Cr.outsideCriticalPairsInOnesideWithIndex rs)
		   val (icpsL,icpsH) = L.partition (fn ((m,n),(u,v))=>LU.member m indexesOfNLR) icps
		   val (icpsLvsL,icpsLvsH) = L.partition (fn ((m,n),(u,v))=>LU.member n indexesOfNLR) icpsL
		   val (icpsHvsL,icpsHvsH) = L.partition (fn ((m,n),(u,v))=>LU.member n indexesOfNLR) icpsH
		   val _ = print "Check Joinablility of CP from NLR:\n"
		   val _ = L.app (fn ((m,n),(c1,c2)) => print ("Critical Pair <"
				  ^ (Term.toString c1) ^ ", " ^ (Term.toString c2) ^ ">"
				  ^ " by Rules <" ^ (Int.toString m)
				  ^ ", " ^ (Int.toString n) ^ ">\n"))  icpsLvsL
		   val _ = if Cr.isJoinableCps rs (L.map (fn (_,cp)=>cp) icpsLvsL)
			   then () else raise NonJoinableCPFound
		   val _ = print "done.\n"
		   val seqsLvsH = List.map (getIndexOfConvSeq rs) icpsLvsH (* may raise NonJoinableCPFound *)
		   val seqsHvsL = List.map (getIndexOfConvSeq rs) icpsHvsL (* may raise NonJoinableCPFound *)
		   val seqsHvsH = List.map (getIndexOfConvSeq rs) icpsHvsH (* may raise NonJoinableCPFound *)
	       (* 	   val _ =  L.app *)
	       (* (fn hs => L.app (fn (n,m,ms) => print ((PrintUtil.prSeq  *)
	       (* 				     (fn (dir,i) => ( *)
	       (* 					 if dir = 1 *)
	       (* 					 then "->(" ^ (Int.toString i) ^ ")" *)
	       (* 					 else "(" ^ (Int.toString i) ^ ")<-")) (* dir = ~1 *)  *)
	       (* 				     ms) *)
	       (* 			      ^ "\n") *)
	       (* 	   ) hs) *)
	       (* seqsLvsH *)
		   val prop = Conj [Conj (L.map makeCond2 seqsHvsH), 
				    Conj (L.map makeCond1 seqsLvsH),
				    Conj (L.map makeCond1 (L.map revSeqs seqsHvsL))]
		   val ans = checkProp smtSolverPath tmpDir rs prop
	       in ans
	       end
	  else (print "unknown innermost-termination for terms of non-linear types\n"; false)
       end
       handle NonJoinableCPFound => (print "no joinable sequence for some critical pairs\n"; false)


   (* val smtPath = "../../work/tools/bin/yices" *)
   (* val tmpDir = "../../work/acp/" *)
   (*  val _ = isConfluentWeakLinearSystem  *)
   (* 		(fn _ => (fn _ => true)) *)
   (* 		smtPath tmpDir  *)
   (* 		(IOFotrs.rdRules [ "f(?x,?y) -> f(g(?x),g(?x))",  *)
   (*  				   "f(g(?x),?x) -> f(?x,g(?x))",  *)
   (*  				   "g(?x) -> h(?x)" ]) *)

   end (* local *)
end (* struct *)

