(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/cr_layer.sml
 * description: confluence check by layer-preserving decomposition
 * author: YOSHIDA Junichi
 * 
 ******************************************************************************)

signature CR_LAYER  = 
   sig
       val clp_decompose: (Term.term * Term.term) list 
			  -> (Term.term * Term.term) list list
			     
       val proveConfluenceByLayerPreservingDecomposition:
	   ((Term.term * Term.term) list -> bool) 
	   -> (Term.term * Term.term) list 
	   ->  Cr.ConfluenceResult

       val tryLayerPreservingDecomposition:
	   (Term.term * Term.term) list
	   -> ((Term.term * Term.term) list -> Cr.ConfluenceResult) 
	      * ((Term.term * Term.term) list -> Cr.ConfluenceResult) 
	   -> Cr.ConfluenceResult

end;


structure CrLayer : CR_LAYER = 
   struct

   local 
       open Term
       open Trs
       open Rewrite
       open Subst
       open Cr
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure TS = TermSet
       fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs
       fun exceptNth n [] = []
	 | exceptNth n (x::xs) = 
	   if n = 0 
	   then xs
	   else (x::(exceptNth (n-1) xs))

	   fun donum rs0 =
		   let
			   fun donum_sub _ [] = []
				 | donum_sub n (r::rs) = (n,r) :: (donum_sub (n+1) rs);
		   in
			   donum_sub 1 rs0
		   end;


        open PrintUtil

   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()


fun donum2 rs0 =
    let
        fun donum2_sub _ [] = []
          | donum2_sub n ((l,r)::rs) = (n,l,r) :: (donum2_sub (n+1) rs);
    in
        donum2_sub 1 rs0
    end;


(*
- funlistr_rnr (1,"A(S(x))","M(J(x))");
val it = (1,["M","A"],["J","M","S","A"])
*)

 fun funlistrr (t1,t2) =
     let
         fun funlist (Fun (f,ts,ty)) fs =
             funlistlist ts (FunSet.add' (f,fs))
		   | funlist (Var _) fs = fs
         and funlistlist [] fs = fs
           | funlistlist (t::ts) fs = funlistlist ts (funlist t fs)
     in
         funlist t2 (funlist t1 FunSet.empty)
     end;

 fun funlistr_ndf (n,t1,t2) =
     let
         fun funlist_root (Fun (f,ts,ty)) (rootfs,fs) =
             funlistlist ts (FunSet.add' (f,rootfs),FunSet.add' (f,fs))
		   | funlist_root (Var _) (rootfs,fs) = (rootfs,fs)
         and funlist (Fun (f,ts,ty)) (rootfs,fs) =
             funlistlist ts (rootfs,FunSet.add' (f,fs))
		   | funlist (Var _) (rootfs,fs) = (rootfs,fs)
         and funlistlist [] (rootfs,fs) = (rootfs,fs)
           | funlistlist (t::ts) (rootfs,fs) = funlistlist ts (funlist t (rootfs,fs))
     in
         (fn (ds,fs) => (n,ds,fs)) (funlist_root t2 (funlist_root t1 (FunSet.empty,FunSet.empty)))
     end;


 fun allfunlistr_ndf ps = List.map funlistr_ndf ps;


(*
- allfunlistr_nf (donum2 (IO.rdrules ["W(x) -> x","B(S(x)) -> W(x)"]));
val it = ([2,1],["B","S","W"]) : int list * Symbol list
*)

 fun allfunlistr_nf nrs0 = 
	 let
		 fun allfunlistr_nf_sub [] (ns,sharef) = (ns,sharef)
		   | allfunlistr_nf_sub ((n,t1,t2)::nrs) (ns,sharef) =
			 allfunlistr_nf_sub nrs (IntSet.add' (n,ns), FunSet.union (funlistrr (t1,t2), sharef));
	 in
		 allfunlistr_nf_sub nrs0 (IntSet.empty,FunSet.empty)
	 end;


(*
$B1&JU$,JQ?t$N%k!<%k$r<h$j=P$9(B
- make_rsshare1 (donum2 (IO.rdrules ["W(W(x)) -> x","B(S(x)) -> W(x)"]));
([(1,"W(W(x))","x")],[(2,"B(S(x))","W(x)")])
*)

fun make_rsshare1 nrs0 =
	let
		fun make_rsshare1_sub [] (nrss1,nrss2) = (nrss1,nrss2)
		  | make_rsshare1_sub ((n,l,r as Fun(f,ts,ty))::nrs) (nrss1,nrss2) = 
            make_rsshare1_sub nrs (nrss1,(n,l,r)::nrss2)
		  | make_rsshare1_sub ((n,l,r as Var(f,ts))::nrs) (nrss1,nrss2) = 
			make_rsshare1_sub nrs ((n,l,r)::nrss1,nrss2)
	in
		make_rsshare1_sub nrs0 ([],[])
	end;


(*
$B1&JU$,JQ?t$N%k!<%k$+$i$D$J$,$k%k!<%k$r<h$j=P$9(B
- make_rsshare2 ([1],["W"]) [(2,["S","W"],["S","W"]),(3,["S"],["S"]),(4,["B"],["B","C"])];
val it = ([3,2,1],[(4,["B"],["B","C"])])
*)

 fun make_rsshare2 (ns0,sharef0) ps0 =
	 let
		 fun make_rsshare2_sub (ns, sharef) [] psh = (ns, psh)
		   | make_rsshare2_sub (ns, sharef) ((n,ds,fs)::ps) psh =
			 if FunSet.isEmpty (FunSet.intersection (sharef,ds))
				 then make_rsshare2_sub (ns, sharef) ps ((n,ds,fs)::psh)
			 else make_rsshare2_sub (IntSet.add' (n,ns), FunSet.union (sharef,fs)) (ps @ psh) []
	 in
		 make_rsshare2_sub (ns0,sharef0) ps0 []
	 end;


(*
- make_rsshare (donum2 ( IO.rdrules [
 "W(x) -> x",
 "B(S(x)) -> W(x)",
 "F(B(x)) -> F(x)",
 "J(S(x)) -> J(J(B(x)))"]) );
val it = ([2,1],[(4,["J"],["B","S","J"]),(3,["F"],["B","F"])])
*)

 fun make_rsshare nrs0 = 
	 (fn (nrs1,nrs2) => make_rsshare2 (allfunlistr_nf nrs1) (allfunlistr_ndf nrs2))
	 (make_rsshare1 nrs0);


(*
- make_rsn (2,["C"],["J","C"]) [(4,["J"],["B","S","J"]),(3,["F"],["B","F"])];
val it = [4,2]
*)

 fun make_rsn (na,dsa,fsa) ps0 =
	 let
		 fun make_rsn_sub (ns0,fs0) [] _ = ns0
		   | make_rsn_sub (ns0,fs0) ((n,ds,fs)::ps) psh =
			 if FunSet.isEmpty (FunSet.intersection (fs0,ds))
				 then make_rsn_sub (ns0,fs0) ps ((n,ds,fs)::psh)
			 else make_rsn_sub (IntSet.add' (n,ns0), FunSet.union (fs,fs0)) (ps@psh) []
	 in
		 make_rsn_sub (IntSet.singleton na,fsa) ps0 []
	 end;


(*
- psminus [(4,["J"],["B","S","J"]),(3,["F"],["B","F"])] [3];
val it = [(4,["J"],["B","S","J"])]
*)

 fun psminus [] _ = []
   | psminus ((n,ds,fs)::ps) ns =
	 if IntSet.member (ns,n) then psminus ps ns
	 else (n,ds,fs)::(psminus ps ns);


 fun ns2rs _ [] = []
   | ns2rs ns ((n,l,r)::nrs) = 
	 if IntSet.member (ns,n) then (l,r)::(ns2rs ns nrs)
	 else ns2rs ns nrs;


 fun minimal_ns [] [] = []
   | minimal_ns [] (nsh::nssh) = minimal_ns (nsh::nssh) []
   | minimal_ns (ns1::[]) nssh = ns1::(minimal_ns nssh [])
   | minimal_ns (ns1::ns2::nss) nssh =
	 if IntSet.isSubset (ns1,ns2)
		 then minimal_ns (ns2::nss@nssh) []
	 else if IntSet.isSubset (ns2,ns1)
			  then minimal_ns (ns1::nss) nssh
		  else minimal_ns (ns1::nss) (ns2::nssh);


 fun clp_decompose rs =
	 let
		 val nrs = donum2 rs
		 val (nsshare,pssingle) = make_rsshare nrs
		 fun clp_decompose_sub [] = []
		   | clp_decompose_sub ((nn,dsn,fsn)::ps0) =
			 let val nsn = make_rsn (nn,dsn,fsn) (psminus pssingle (IntSet.singleton nn))
                 val psrest = psminus ps0 nsn
			 in  nsn::(clp_decompose_sub psrest)  end
	 in
		 if List.null pssingle then [rs]
		 else List.map (fn nsa => ns2rs (IntSet.union (nsa,nsshare)) nrs)
			 (minimal_ns (clp_decompose_sub pssingle) [])
	 end;


  (* $B9gN.@-$NAXJ]B8J,2r$rMQ$$$?H=Dj(B *)
   fun proveConfluenceByLayerPreservingDecomposition isTerminating rs = 
       let val _ = print "\nTry Layer Preserving Decomposition...\n" 
	   val components = clp_decompose rs
       in
	   if length components = 1
	   then (print "Layer Preserving Decomposition failed"; report Unknown)
	   else
	       let val subresults = 
		       L.map (fn rs =>
				 let val _ = print (Trs.prRules rs)
				 in
				     checkConfluenceConditions isTerminating rs
				 end)
			     components
		   val _ = print "Proof by Layer Preserving Decomposition"
	       in 
		   if L.all (fn c => c = CR) subresults
		   then report CR
		   else if L.exists (fn c => c = NotCR) subresults
		   then report NotCR
		   else report Unknown
	       end
       end


   fun tryLayerPreservingDecomposition rs (direct,decomp) = 
       let val _ = print "\nTry Layer Preserving Decomposition for...\n"
	   val _ = print (Trs.prRules rs)
	   val components = clp_decompose rs
	   fun checkNCR [] = Unknown
	     | checkNCR (rs::rss) = 
	       let val _ = print "(lp)"
                   val _ = print (Trs.prRules rs)
	       in
		  case direct rs of
		      NotCR => NotCR
		    | _ => checkNCR rss
	       end
	   fun checkAll [] = CR
	     | checkAll (rs::rss) = 
	       let val _ = print "(lp)"
		   val _ = print (Trs.prRules rs) 
	       in
		   case direct rs of
		       CR => checkAll rss
		     | NotCR => NotCR
		     | Unknown => checkNCR rss
	       end
       in
	   if length components = 1
	   then (print "Layer Preserving Decomposition failed"; 
		 report Unknown;
		 decomp rs)
	   else
	       case checkAll components of
		   CR => (print "Result by Layer Preserving Decomposition";
			  report CR)
		 | NotCR => (print "Result by Layer Preserving Decomposition";
			     report NotCR)
		 | Unknown => (print "Result by Layer Preserving Decomposition";
			       report Unknown)
       end




end
end
