(******************************************************************************
 * Copyright (c) 2018-2019, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(**********ec********************************************************************
 * file: rwtools/rwchecker/cr_ordered.sml
 * description: confluence proof by ordered rewriting
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature CR_ORD = 
   sig
   val runDebug: bool ref

   val confluenceByOrderedRewriting:
       string  (* minisatPath *)
       -> string  (* smtSolverPath *)
       -> string  (* tmp dir *)
       -> (Term.term * Term.term) list 
       -> bool

end;

structure CrOrd : CR_ORD = 
   struct

   local 
       structure T = Term
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure TP = TermPair
       structure TPS = TermPairSet
       structure TS = TermSet
       structure YI = YicesInput
       open PrintUtil   
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   exception CrOrdError of string

  (* type prec  = Term.fun_key list *)
  (* val comparePrec: prec -> Term.fun_key * Term.fun_key -> order *)

   fun lex rel ([],[]) = false
     | lex rel (x::xs,y::ys) = if Term.equal (x,y)
			       then lex rel (xs,ys)
			       else rel (x,y)
     | lex rel _ = raise CrOrdError "Error: lex compares lists having different length"

   (* $BJQ?t$,(B skolemized $B$5$l$F$$$k$H$7$FHf3S(B *)
   (* $B%9%3!<%l%`JQ?t$O!$4X?t5-9f5-9f$h$j>.$5$$!%4X?t5-9f$NHf3S$O(B compare $B$G9T$&(B
      e.g. prec := [Fun.fromString "+", Fun.fromString "s", Fun.fromString "0"]
           compare  := Order.comparePrec prec  *)
   (* $B%9%3!<%l%`JQ?tF1;N$O!$(BVar.compare $B$G<-=q<0$KHf3S(B "" < "a" < "aa" < "ba" < ...*)
   fun lpoGtSkolem compare (term1,term2) = 
     let fun gt (T.Var (x,_), T.Var (y,_)) = (Var.compare (x,y) = GREATER)
           | gt (T.Var _ , T.Fun _) = false
           | gt (T.Fun (f,ss,_), t as (T.Var _)) = geqforsome ss t
           | gt (s as (T.Fun (f,ss,_)), t as (T.Fun (g,ts,_))) = 
             (case compare (f,g) of
		  EQUAL => (lex gt (ss,ts) andalso gtforall s ts)
			   orelse geqforsome ss t
	       |  GREATER => gtforall s ts orelse geqforsome ss t
	       |  LESS => geqforsome ss t)
         and geq (s,t) = Term.equal (s,t) orelse gt (s,t)
         and gtforall s ts = L.all (fn ti => gt (s,ti)) ts
         and geqforsome ss t = L.exists (fn si => geq (si,t)) ss
     in gt (term1,term2)
     end

   (**** rules for ordered rewriting ****
     (gt,R,E)
      gt: e.g. lpoGtSkolem
      R: a set of rewrite rules,   gt (l,r) = true for all l -> r in R )
      E: a set of equations, for any s = t in E, t = s in E (modulo renaming)
    Example R = [ +(s(x),y) -> s(+(x,y)),  +(0,y) -> y, +(+(x,y),z) -> +(x,+(y,z)) ]
            E = [ +(x,y) -> +(y,x),  +(x,+(y,z)) -> +(y,+(x,z)) ]
           > = lpo(+>s>0>...x_1>x_0)
    ****)

   fun skolemOrdRootRewriteByEpart ord [] term = NONE
     | skolemOrdRootRewriteByEpart ord ((l,r)::es) term = 
       (case Rewrite.rootRewrite [(l,r)] term of
	    SOME reduct  => if ord (term,reduct) then SOME reduct else skolemOrdRootRewriteByEpart ord es term
	 |  NONE => skolemOrdRootRewriteByEpart ord es term)

   fun skolemOrdRootRewrite (ord,R,E) term =
     (case Rewrite.rootRewrite R term of
	  SOME reduct => SOME reduct
       |  NONE => skolemOrdRootRewriteByEpart ord E term)

   fun skolemOrdLinf (ord,R,E) term =
     case term 
      of T.Var _ => term
       | T.Fun (f,ts,sort) => let val ss = map (skolemOrdLinf (ord,R,E)) ts
				  val _ = print "."
			      in case skolemOrdRootRewrite (ord,R,E) (T.Fun (f,ss,sort))
                                  of SOME reduct => skolemOrdLinf (ord,R,E) reduct
                                   | NONE => T.Fun (f,ss,sort)
			      end

  (**** test ***
   val _ = let val E = IOFotrs.rdRules ["plus(?x,?y) -> plus(?y,?x)",
   					"plus(?x,plus(?y,?z)) -> plus(?y,plus(?x,?z))" ]
   	       val R = IOFotrs.rdRules ["plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))",
   					"plus(succ(?x),?y) -> succ(plus(?x,?y))",
   					"plus(zero,?y) -> ?y" ]
   	       val prec = [Fun.fromString "plus", Fun.fromString "s", Fun.fromString "0"]
               val compare = Order.comparePrec prec
   	       val ord = lpoGtSkolem compare
   	       val term = IOFotrs.rdTerm "plus(?z,plus(?y,?x))"
   	       (* val term = IOFotrs.rdTerm "plus(?z,?x)" *)
   	       val term2 = IOFotrs.rdTerm "plus(?x,?z)"
   	       (* val term = IOFotrs.rdTerm "plus(?x,?x)" *)
   	       val _ = print ("term = " ^ (Term.toString term) ^ "\n")
   	       val _ = print ("term2 = " ^ (Term.toString term2) ^ "\n")
	       val _ = print "\n"
	       val _ = if ord (term,term2)
		       then print ((Term.toString term) ^ " > " ^ (Term.toString term2) ^ "\n")
		       else if ord (term2,term)
		       then print ((Term.toString term2) ^ " > " ^ (Term.toString term) ^ "\n")
		       else print ((Term.toString term) ^ " <> " ^ (Term.toString term2) ^ "\n")
	       val red = skolemOrdRootRewriteByEpart ord E term
   	       val _ = print ("red = " ^ (case red of SOME t => Term.toString t 
						    | NONE => "(none)") ^ "\n")
	       val red2 = skolemOrdRootRewrite (ord,R,E) term
   	       val _ = print ("red2 = " ^ (case red2 of SOME t => Term.toString t 
						    | NONE => "(none)") ^ "\n")
	       val red3 = skolemOrdLinf (ord,R,E) term
   	       val _ = print ("red3 = " ^ (Term.toString red3) ^ "\n")
				  
   	       (* val reduct = skolemOrdRootRewrite (ord,R,E) term *)

   	      (* val nf = skolemOrdLinf (ord,R,E) term  *)
   (* 	       val _ = print (Term.toString nf) *)
   	   in ()
   	   end
   ************)

   (* Crule: f(x,y) -> f(y,x) with distinct variables x,y *)
   fun isCrule (l,r) = case (T.funRootOfTerm l,T.funRootOfTerm r) of
			   (SOME f, SOME g) => Fun.equal (f,g)
					       andalso (case (T.argsOfTerm l, T.argsOfTerm r) of
							   ([u1,v1], [v2,u2]) => L.all T.isVar [u1,v1]
										 andalso not (T.equal (u1,v1))
										 andalso T.equal (u1,u2)
										 andalso T.equal (v1,v2)
							 | _ => false)
			|  _ => false

   local fun decompL t = case T.funRootOfTerm t of
			     NONE => NONE
			  |  SOME f => (case T.argsOfTerm t of
					 [u,l3] => (case T.funRootOfTerm u of
						      NONE => NONE
						    | SOME g => if Fun.equal (f,g)
								then case T.argsOfTerm u of
									  [l1,l2] => SOME [l1,l2,l3] | _ => NONE
								else NONE)
					 | _ => NONE)
	 fun decompR t = case T.funRootOfTerm t of
			   NONE => NONE
			|  SOME f => (case T.argsOfTerm t of
					 [r1,u] => (case T.funRootOfTerm u of
						      NONE => NONE
						    | SOME g => if Fun.equal (f,g)
								then case T.argsOfTerm u of [r2,r3] => SOME [r1,r2,r3] | _ => NONE
								else NONE)
					 | _ => NONE)
   in
   (* Arule: f(f(x,y),z) -> f(x,f(y,z)) *)
   fun isArule (l,r) =
     case (decompL l, decompR r) of
	 (SOME [l1,l2,l3], SOME [r1,r2,r3]) => L.all T.isVar [l1,l2,l3]
					       andalso T.equal (l1,r1) andalso T.equal (l2,r2)
					       andalso T.equal (l3,r3) andalso not (T.equal (l1,l2))
					       andalso not (T.equal (l1,l3)) andalso not (T.equal (l2,l3))
	 |  _ => false

   (* Srule: f(x,f(y,z)) -> f(y,f(x,z)) *)
   fun isSrule (l,r) =
     case (decompR l, decompR r) of
	    (SOME [l1,l2,l3], SOME [r1,r2,r3]) => L.all T.isVar [l1,l2,l3]
						  andalso T.equal (l1,r2) andalso T.equal (l2,r1)
						  andalso T.equal (l3,r3) andalso not (T.equal (l1,l2))
						  andalso not (T.equal (l1,l3)) andalso not (T.equal (l2,l3))
	 |  _ => false

   end

    (* i : InvFun for f iff there exists a rule of the form f(x,i(x)) -> ...  or f(i(x),x) -> ...*)
   fun getInvFunForAC f (l as (T.Var _), r) =  NONE
     | getInvFunForAC f (l as (T.Fun (g,[u,v],ty)), r ) =
       if Fun.equal (f,g)
       then (case (u,v) of
		 (T.Var _,  T.Fun (i,[u2],ty)) => if T.equal (u,u2) then SOME (f,i) else NONE
	      |  (T.Fun (i,[v2],ty), T.Var _ ) => if T.equal (v,v2) then SOME (f,i) else NONE
	      |  _ =>  NONE)
       else NONE
     | getInvFunForAC f _ = NONE
				
   fun prGtConstraint gtCnstr = LU.toStringCommaCurly (fn (u,v) => (Term.toString u)  ^ ">" ^ (Term.toString v)) gtCnstr
   fun prEqConstraint eqCnstr = LU.toStringCommaCurly (fn (u,v) => (Term.toString u)  ^ "=" ^ (Term.toString v)) eqCnstr
   fun prConstraint (eqCnstr,gtCnstr) =
     LU.toStringCommaCurly (fn (i,u,v) => if i = 0
					  then (Term.toString u)  ^ "=" ^ (Term.toString v)
					  else (Term.toString u)  ^ ">" ^ (Term.toString v))
			   ((L.map (fn (u,v) => (0,u,v)) eqCnstr) @ (L.map (fn (u,v) => (1,u,v)) gtCnstr))
   fun getGtConstraint (eqCnstr,gtCnstr) = gtCnstr
   fun getEqConstraint (eqCnstr,gtCnstr) = eqCnstr					       
   fun getVarConstraint (eqCnstr,gtCnstr) = (L.filter (fn (u,v) => T.isVar u andalso T.isVar v) eqCnstr,
					     L.filter (fn (u,v) => T.isVar u andalso T.isVar v) gtCnstr)
						     
   fun unionConstraint ((eqCnstr1,gtCnstr1), (eqCnstr2,gtCnstr2))
     = (LU.union' TP.equal (eqCnstr1,eqCnstr2), LU.union' TP.equal (gtCnstr1,gtCnstr2))

   fun areSameEqConstraint (eqCnstr1,eqCnstr2) =
       LU.setEqual' (fn ((x1,y1),(x2,y2)) => TP.equal ((x1,y1),(x2,y2)) orelse TP.equal ((x1,y1),(y2,x2)))
		    (eqCnstr1,eqCnstr2)
   fun areSameGtConstraint (gtCnstr1,gtCnstr2) = LU.setEqual' TP.equal (gtCnstr1,gtCnstr2)
   fun areSameConstraint ((eqCnstr1,gtCnstr1), (eqCnstr2,gtCnstr2)) =
       areSameEqConstraint (eqCnstr1,eqCnstr2) andalso areSameGtConstraint (gtCnstr1,gtCnstr2)

  (* C step or S step $B$GI,MW$J@)Ls(B *)
   fun gtCnstrInCstep redex = let val [u,v] = T.argsOfTerm redex in (u,v) end
   fun gtCnstrInSstep redex = let val [u1,v] = T.argsOfTerm redex
				  val [u2,u3] = T.argsOfTerm v
			      in (u1,u2) end

   fun gtCnstrInRootRewriteStepSub rule (redex,reduct) =
     if isCrule rule then gtCnstrInCstep redex
     else if isSrule rule then gtCnstrInSstep redex
     else (redex,reduct)

   (* count and drop outermost inverses *)
   fun dropInv inverse (t as (T.Var _)) = (0,t)
     | dropInv inverse (t as (T.Fun (f,[s],ty))) = if Fun.equal (f,inverse)
						   then let val (n,s2) = dropInv inverse s in (n+1,s2) end
						   else (0,t)
     | dropInv inverse t = (0,t)

 (* special treatment for inverse function *)
   fun gtCnstrInRootRewriteStep mayInvFun rule (redex,reduct) =
     let val (u,v) = gtCnstrInRootRewriteStepSub rule (redex,reduct)
     in case mayInvFun of
	    NONE => [(u,v)]
	 |  SOME inverse => let val (n,u2) = dropInv inverse u
				val (m,v2) = dropInv inverse v
			    in if n > m
			       then [(valOf (T.subterm (L.tabulate (m, fn x => 1)) u), v2)]
			       else [(u2,v2)]
			    end
     end	   
				       
   (*** test ***

   val _ = let val rule1 = IOFotrs.rdRule "plus(?x,?y) -> plus(?y,?x)"
	       val rule2 = IOFotrs.rdRule "plus(?x,plus(?y,?z)) -> plus(?y,plus(?x,?z))"
	       val rule3 = IOFotrs.rdRule "plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))"
	       fun ans b = if b then "true" else "false"
	       val _ = L.app (fn rule => (print (ans (isCrule rule) ^ ":");
					  print (ans (isArule rule) ^ ":");
					  print (ans (isSrule rule) ^ "\n")))   [rule1, rule2, rule3]
	       val step1 = IOFotrs.rdRule "plus(inv(inv(?x)),?y) -> plus(?y,inv(inv(?x)))"
	       val step2 = IOFotrs.rdRule "plus(?x,inv(?y)) -> plus(inv(?y),?x)"
	       val step3 = IOFotrs.rdRule "plus(inv(?x),inv(?y)) -> plus(inv(?y),inv(?x))"
               val inv = Fun.fromString "inv"
	       val _ = L.app (fn step =>
				 let val ((x,y)::_) = gtCnstrInRootRewriteStep (SOME inv) rule1 step
				 in  print (T.toString x ^ ">" ^ T.toString y ^ "\n")
				 end)
			     [step1, step2, step3]
	   in ()
	   end
    *******)	       


   fun mkTransitiveClosure cnstr = 
     let val idx = L.tabulate (L.length cnstr, fn x => x)
	 val additions = LU.mapAppend (fn i => let val (x,y) = L.nth (cnstr,i)
					       in L.mapPartial (fn (u,v) => if T.equal (u,y) then SOME (x,v) else NONE) cnstr
					       end) idx
	 val additions2 = LU.eliminateDuplication' TP.equal additions
     in if LU.subseteq' TP.equal (additions2,cnstr)
	then cnstr
	else mkTransitiveClosure (LU.union' TP.equal (additions2,cnstr))
     end

   (* We assume ord has a subterm property  *)
   fun mkGtClosure gtCnstr = L.rev (mkTransitiveClosure
				      (LU.mapAppend (fn (u,v) => (u,v) :: (if T.isFun v
									   then L.map (fn x => (v,x)) (T.varSubterms v)
									   else [])) 
						    gtCnstr))

   fun mkEqClosure eqCnstr = LU.eliminateDuplication'
				 (fn ((u1,v1),(u2,v2)) => (T.equal (u1,u2) andalso T.equal (v1,v2))
							  orelse (T.equal (u1,v2) andalso T.equal (v1,u2)))
				 (L.filter (not o T.equal)
					   (mkTransitiveClosure (eqCnstr @ (L.map (fn (u,v) => (v,u)) eqCnstr))))
						 
   fun mkClosure (eqCnstr,gtCnstr) = (mkEqClosure eqCnstr, mkGtClosure gtCnstr)

   fun isInconsistent (eqCnstr,gtCnstr) =
     L.exists (fn (u,v) => T.isASubterm u v) gtCnstr
     orelse L.exists (fn (u,v) => LU.member' TP.equal (u,v) gtCnstr
				  orelse LU.member' TP.equal (v,u) gtCnstr)
		     eqCnstr

   fun extendedCriticalPairs ord rules =
     let val rulesWithIdx = LP.zip (rules, L.tabulate (L.length rules, fn x => x))
	 val crulesIndex = L.mapPartial (fn (lr,i) => if isCrule lr then SOME i else NONE) rulesWithIdx
	 val srulesIndex = L.mapPartial (fn (lr,i) => if isSrule lr then SOME i else NONE) rulesWithIdx
	 fun isCrule i = LU.member i crulesIndex
	 fun isSrule i = LU.member i srulesIndex
	 fun getGtCnstr (i,redex)
	   = if isCrule i then [gtCnstrInCstep redex]
	     else if isSrule i then [gtCnstrInSstep redex]
	     else []

	 val inCPeaksWithIndexPos = Cr.insideCriticalPeaksWithIndexAndPos rules
	 val outCPeaksWithIndex = Cr.outsideCriticalPeaksInOnesideWithIndex rules (* symmetric cases are dropped *)

	 (* check without skolemization *)
	 fun checkOrderCond (redex,reduct) = not (ord (reduct,redex))

	 val outecps = L.mapPartial (fn ((i,j),(mid,lhs,rhs))
					=> if checkOrderCond (mid,rhs) andalso checkOrderCond (mid,lhs)
					   then SOME (i,j,[],(getGtCnstr (i,mid) @ getGtCnstr (j,mid)),lhs,rhs) 
					   else NONE)  outCPeaksWithIndex

	 val _ = debug (fn () => print "Constrained CPs:\n")
	 val _ = debug (fn () =>
			   print (LU.toStringCommaLnCurly
				      (fn (i,j,pos,gtCnstr,l,r) =>
					  "(" ^ (Int.toString i) ^ "," ^ (Int.toString j) ^ ") at " ^ Pos.toString pos ^ ": "
					  ^ (prGtConstraint gtCnstr) ^ "|- " ^ Trs.prEq (l,r)) outecps))
				 
	 fun takeSubterm ps t = case T.subterm ps t of
				    SOME t' => t'
				  | NONE => raise CrOrdError "Error: failed to take redex of critical peak steps"

	 val inecps = L.mapPartial (fn ((i,j),(pos,mid,lhs,rhs)) =>
				       let val top = takeSubterm pos mid
					   val left = takeSubterm pos lhs
					   val cnstr = getGtCnstr (i,top) @ getGtCnstr (j,mid)
				       in if checkOrderCond (mid,rhs) andalso checkOrderCond (mid,rhs)
					  then SOME (i,j,pos,cnstr,lhs,rhs)
					  else NONE
				       end)  inCPeaksWithIndexPos

	 val _ = debug (fn () => print (LU.toStringCommaLnCurly
					    (fn (i,j,pos,gtCnstr,l,r) =>
						"(" ^ (Int.toString i) ^ "," ^ (Int.toString j) ^ ") at " ^ Pos.toString pos ^ ": "
						^ (prGtConstraint gtCnstr) ^ "|- " ^ Trs.prEq (l,r)) inecps))

     in
	 (*  L.mapPartial (fn (i,j,pos,cnstr,l,r) => (cnstr,l,r)) (outecps @ inecps) *)
	 L.mapPartial (fn (i,j,pos,gtCnstr,l,r) =>
			  let val newGtCnstr = mkGtClosure gtCnstr
			  in if isInconsistent ([],newGtCnstr)
			     then NONE
			     else SOME (i,j,pos,newGtCnstr,l,r)
			  end)  (outecps @ inecps)
     end

  (* $B=g=x$N2>Dj(B gtCnstr $B$N$b$H$G$NHf3S(B *)
   fun lpoGtUnderHyp compare gtCnstr (term1,term2) = 
     let fun gt (s,t) = (LU.member' TermPair.equal (s,t) gtCnstr) orelse gtSub (s,t)
	 and gtSub (T.Var (x,_), _) = false
           | gtSub (T.Fun (f,ss,_), t as (T.Var _)) = geqforsome ss t
           | gtSub (s as (T.Fun (f,ss,_)), t as (T.Fun (g,ts,_))) = 
             (case compare (f,g) of
		  EQUAL => (lex gt (ss,ts) andalso gtforall s ts)
			   orelse geqforsome ss t
	       |  GREATER => gtforall s ts orelse geqforsome ss t
	       |  LESS => geqforsome ss t)
         and geq (s,t) = Term.equal (s,t) orelse gt (s,t)
         and gtforall s ts = L.all (fn ti => gt (s,ti)) ts
         and geqforsome ss t = L.exists (fn si => geq (si,t)) ss
     in gt (term1,term2)
     end

   (* $B30B&$N(B unary function 'inverse' $B$r:o=|$7$?$"$H$K!$Hf3S(B *)
   (* $BF1$8$H$-$O!$(B'inverse' $B$N?t$GHf3S$9$k(B *)
   fun lpoGtUnderHypUnarySpecial inverse compare gtCnstr (term1,term2) = 
     let fun gt (s,t) = let val (n,s2) = dropInv inverse s
			    val (m,t2) = dropInv inverse t
			in if T.equal (s2,t2) then n > m
			   else gtOrg (s2,t2) (s,t)
			end
	 and gtOrg (s,t) (orgS,orgT) = (LU.member' TermPair.equal (s,t) gtCnstr) orelse gtSub (s,t) (orgS,orgT)
	 and gtSub (T.Var (x,_), _) (orgS,orgT) = false
           | gtSub (T.Fun (f,ss,_), t as (T.Var _)) (orgS,orgT) = geqforsome ss orgT
           | gtSub (s as (T.Fun (f,ss,_)), t as (T.Fun (g,ts,_))) (orgS, orgT) = 
             (case compare (f,g) of
		  EQUAL => (lex gt (ss,ts) andalso gtforall orgS ts)
			   orelse geqforsome ss orgT
	       |  GREATER => gtforall orgS ts orelse geqforsome ss orgT
	       |  LESS => geqforsome ss orgT)
         and geq (s,t) = Term.equal (s,t) orelse gt (s,t)
         and gtforall s ts = L.all (fn ti => gt (s,ti)) ts
         and geqforsome ss t = L.exists (fn si => geq (si,t)) ss
     in gt (term1,term2)
     end


  (* $B=g=x$N2>Dj(B gtCnstr $B$N$b$H$G!$=g=x=q$-49$(!%(B *)
   fun ordRootRewriteByEpart preord [] gtCnstr term = NONE
     | ordRootRewriteByEpart preord ((l,r)::es) gtCnstr term = 
       (case Rewrite.rootRewrite [(l,r)] term of
	    SOME reduct  => if preord gtCnstr (term,reduct)
			    then SOME reduct
			    else ordRootRewriteByEpart preord es gtCnstr term
	 |  NONE => ordRootRewriteByEpart preord es gtCnstr term)

   fun ordRootRewrite (preord,R,E) gtCnstr term =
     (case Rewrite.rootRewrite R term of
	  SOME reduct => SOME reduct
       |  NONE => ordRootRewriteByEpart preord E gtCnstr term)

   fun ordLinf (preord,R,E) gtCnstr term =
       case term 
        of T.Var _ => term
         | T.Fun (f,ts,sort) => let val ss = map (ordLinf (preord,R,E) gtCnstr) ts
				in case ordRootRewrite (preord,R,E) gtCnstr (T.Fun (f,ss,sort))
                                    of SOME reduct => ordLinf (preord,R,E) gtCnstr reduct
                                     | NONE => T.Fun (f,ss,sort)
				end

   (*** test ***
   val _ = let val R = IOFotrs.rdRules ["plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))",
					"plus(?x,inv(?x)) -> zero",
					"plus(inv(?x),?x) -> zero",
					"inv(plus(?x,?y)) -> plus(inv(?y),inv(?x))",
					"plus(zero,?x) -> ?x",
					"plus(?x,zero) -> ?x",
					"inv(zero) -> zero",
					"inv(inv(?x)) -> ?x",
   					"plus(?x,plus(inv(?x),?y)) -> ?y",
   					"plus(inv(?x),plus(?x,?y)) -> ?y" ]

	       val E = IOFotrs.rdRules ["plus(?x,?y) -> plus(?y,?x)",
   					"plus(?x,plus(?y,?z)) -> plus(?y,plus(?x,?z))" ]
	       val rules = R @ E
   	       val prec = [Fun.fromString "plus"]
               val compare = Order.comparePrec prec
	       val inverse = Fun.fromString "inv"
   	       (* val preord = lpoGtUnderHyp compare *)
   	       val preord = lpoGtUnderHypUnarySpecial inverse compare
   	       val ord = preord []
	       val ecps = extendedCriticalPairs ord rules
	       val _ = L.app (fn (i,j,pos,cnstr,x,y)   =>
				 let val _ = print ("<" ^ (T.toString x) ^ " : " ^ (T.toString y) ^ ">\n")
				     val xnf = ordLinf (preord,R,E) cnstr x
				     val ynf = ordLinf (preord,R,E) cnstr y
				     val _ = print ("<" ^ (T.toString xnf) ^ " : " ^ (T.toString ynf) ^ ">")
				 in if T.equal (xnf,ynf)
				    then print "... Join\n"
				    else print "... Not Join\n"
				 end) ecps	
	   in ()
	   end
    **********)
				    
   fun prTermWithConstraint (term,cnstr) = (prConstraint cnstr) ^ "|- " ^ (Term.toString term)

  (* term $B$NJQ?t$r%9%3!<%l%`2=$7$J$$$G=g=x=q$-49$(!%(B
     $B$"$k@)Ls$N$b$H$G!$=g=x=q$-49$(2DG=$J$i!$(B $B$=$N@)Ls$H=q$-49$(7k2L$NBP$rJV$9(B *)

   fun rootRewriteToConstraintTermByE mayInvFun [] term = NONE
     | rootRewriteToConstraintTermByE mayInvFun ((l,r)::es) term = 
       (case Rewrite.rootRewrite [(l,r)] term of
	    SOME reduct  => SOME (reduct, ([],gtCnstrInRootRewriteStep mayInvFun (l,r) (term,reduct)))
	 |  NONE => NONE)

   fun rootRewriteToConstraintTermByR [] term = NONE
     | rootRewriteToConstraintTermByR ((l,r)::rs) term = 
       let val vs = Term.varListInTerm term
	   val [l',r'] = Subst.renameTermsDisjointFrom vs [l,r]
	   val lvars = Term.varListInTerm l'
     (* sometimes, identifying non linear variables may leads to perform a rewrite step *)
       in  case Subst.unify l' term of
	      NONE => NONE
	   |  SOME sigma => if L.exists (fn (x,t) => (LU.member' Var.equal x lvars
						      andalso not (null (LU.intersection' Var.equal (Term.varListInTerm t, lvars))))
						     orelse
						     (LU.member' Var.equal x vs
						      andalso (T.isFun t)
						      andalso not (null (LU.intersection' Var.equal (Term.varListInTerm t, lvars)))))
					(VM.listItemsi sigma)
			    then NONE
			    else let val sigmalist = VM.listItemsi sigma
				     val eqs = L.map (fn (x,u) => (T.Var (x, T.sortOfTerm u), u)) sigmalist
				     val preRho = L.filter (fn (x,u) => LU.member' Var.equal x lvars) sigmalist
	   			     val rho = L.foldl (fn ((v,t),map)=> VM.insert (map,v,t)) VM.empty preRho
				     val eqs2 = L.map (fn (u,v) => (Subst.applySubst rho u, Subst.applySubst rho v)) eqs
				     fun decomp (T.Var _, T.Fun _) ans = NONE
				       | decomp (T.Fun _, T.Var _) ans = NONE
				       | decomp (x as T.Var _, y as T.Var _) ans = SOME [(x,y)]
				       | decomp (T.Fun (f,ts1,ty1), T.Fun(g,ts2,ty2)) ans = if Fun.equal (f,g)
										    then decompList (ts1,ts2) ans
										    else NONE
				     and decompList ([],[]) ans = SOME ans
				       | decompList (x::xs,[]) ans =  raise CrOrdError "Error: arguments not syncronized."
				       | decompList ([],y::xs) ans =  raise CrOrdError "Error: arguments not syncronized."
				       | decompList (x::xs,y::ys) ans = case decomp (x,y) ans of
									    SOME ans2 => decompList (xs,ys) (ans2@ans)
									  | NONE => NONE
				     val decomposition = L.map (fn uv => decomp uv []) eqs2
	   			     val reduct = Subst.applySubst rho r'
				 in if L.all isSome decomposition
				    then SOME (reduct, (LU.mapAppend valOf decomposition, []))
				    else NONE
				 end
	   (* |  SOME sigma => if not (L.all T.isVar (VM.listItems sigma)) *)
	   (* 		    then NONE *)
	   (* 		    else let val eqs  = L.map (fn (x,u) => if LU.member' Var.equal x vs *)
	   (* 							   then (u, T.Var (x, T.sortOfTerm u)) *)
	   (* 							   else (T.Var (x, T.sortOfTerm u), u)) *)
	   (* 					      (VM.listItemsi sigma) *)
	   (* 			     val sigma' = L.foldl (fn ((v,t),map)=> VM.insert (map,valOf (T.varRootOfTerm v),t)) VM.empty eqs *)
	   (* 			     val eqs2 = L.map (fn (u,v) => (Subst.applySubst sigma' u, Subst.applySubst sigma' v)) eqs *)
	   (* 			     val eqs3 = LU.eliminateDuplication' TP.equal (L.filter (not o T.equal) eqs2) *)
	   (* 			     val reduct = Subst.applySubst sigma' r' *)
	   (* 			 in SOME (reduct, (eqs3,[])) *)
	   (* 			 end *)
       end
												  
   fun rootRewriteToConstraintTermAll (R,E,mayInvFun) term = 
       L.mapPartial (fn rule => rootRewriteToConstraintTermByE mayInvFun [rule] term) E
       @ L.mapPartial (fn rule => rootRewriteToConstraintTermByR [rule] term) R
       
   fun rewriteToConstraintTermAll (R,E,mayInvFun) (term,cnstr) =
       case term
	of T.Var _ => []
	 | T.Fun (f,ts,sort) =>
	   let val argReducts = LU.mapAppend 
				    (fn i => L.map (fn (u,newCnstr) => 
						       (T.Fun (f,
							     L.@(L.take (ts,i), u::(L.drop (ts,i+1))),
							     sort),
							newCnstr))
						   (rewriteToConstraintTermAll (R,E,mayInvFun) (L.nth (ts,i), cnstr)))
				    (L.tabulate (length ts, fn x=>x))
	       val rootReducts = L.map (fn (reduct,newCnstr) => (reduct,newCnstr))
				       (rootRewriteToConstraintTermAll (R,E,mayInvFun) term)
	   in
	       rootReducts @ argReducts
	   end

  (* assume term is normal under hyp, initial  hypothsis *)
  (* cnstr : additional constraints given along the path  *)
   fun mkSucceedents (preord,E,R,mayInvFun) hyp (term,cnstr) =
     let val ord = preord []
	 (* val _ = println ("hyp: " ^ (prGtConstraint hyp)) *)
	 val cands = rewriteToConstraintTermAll (R,E,mayInvFun) (term,cnstr)
         (* gt constraint $B$N1&JU$,JQ?t$G$J$$$N$O:o=|(B *)
	 val cands2 = L.filter (fn (term,cnstr) =>
				   let val (eqCnstr,gtCnstr) = cnstr
				   in L.all (fn (u,v) => T.isVar v) gtCnstr
				   end)
			       cands
         (* constraint $B$N(B closure $B$r$H$k!$(Binconsistent $B$J$N$O:o=|(B *)
	 val cands3 = L.mapPartial (fn (term,newCnstr) => let val cnstrUpdate = mkClosure (unionConstraint (newCnstr,cnstr))
						       in if isInconsistent cnstrUpdate
							  then NONE
							  else SOME (term,cnstrUpdate)
						       end) cands2
         (* term $B$r2>Dj$H(BR,E$B$N$b$H$G@55,2=(B *)
         (* val cands4 = L.map (fn (term,cnstr) => (ordLinf (preord,R,E) hyp term, cnstr)) cands3 *)
	 (* val _ = print ("Possible Child nodes of " ^ (prTermWithConstraint (term,cnstr)) ^ ":\n") *)
         (* not reflecing new constraint ... fixed 2019/02/25 *)
	 (* both reduct and source terms are useful ... 2019/03/04 *)

	 fun mergeCnstr (cnstr,hyp) = let val (eqCnstr,gtCnstr) = cnstr in gtCnstr@hyp end
         val cands4 = LU.mapAppend (fn (term,cnstr) =>
	 			       let val term2 = ordLinf (preord,R,E) (mergeCnstr (cnstr,hyp)) term
	 			       in if Term.equal (term,term2)
					  then [(term, cnstr)]
					  else [(term, cnstr), (term2, cnstr)]
	 			       end)
	 			   cands3
			   
	 val cands5 = LU.eliminateDuplication' (fn ((term1,cnstr1),(term2,cnstr2)) =>
						   Term.equal (term1,term2) andalso areSameConstraint (cnstr1,cnstr2))
					       cands4
	 (* val _ = print ("Possible Child nodes of " ^ (prTermWithConstraint (term, cnstr)) ^ ":\n") *)
	 (* val _ = print (LU.toStringCommaLnSquare prTermWithConstraint cands) *)
	 (* val _ = print (LU.toStringCommaLnSquare prTermWithConstraint cands2) *)
	 (* val _ = print (LU.toStringCommaLnSquare prTermWithConstraint cands3) *)
	 (* val _ = print (LU.toStringCommaLnSquare prTermWithConstraint cands4) *)
	 (* val _ = print (LU.toStringCommaLnSquare prTermWithConstraint cands5) *)
	 (* val _ = println ("-------------") *)
     in cands5
     end
										       
   fun expandTreeByOne (preord,E,R,mayInvFun) hyp (internal,[]) = NONE
     | expandTreeByOne (preord,E,R,mayInvFun) hyp (internal,leaves) =
       let val (termcnstr,pos) = hd leaves
	   val preSucceedents =  mkSucceedents (preord,E,R,mayInvFun) hyp termcnstr
	   val preSucceedents2 =
	       L.filter (fn (t,_) => not (L.exists (fn ((s,_),p) =>
						       Term.equal (t,s) andalso (Term.isPrefixPosition p pos))
						   internal))
			preSucceedents
	   val succeedents = LP.zip (preSucceedents2, L.tabulate (L.length preSucceedents2, fn x => pos@[(x+1)]))
       in SOME (internal@[(termcnstr,pos)], (tl leaves) @ succeedents)
       end

   fun expandTreeByDepth (preord,E,R,mayInvFun) hyp depth (internal,leaves) =
     let val (below,above) = L.partition (fn (_,pos) => L.length pos < depth) leaves
     in if null below
	then (internal,leaves)
	else case expandTreeByOne (preord,E,R,mayInvFun) hyp (internal,below@above) of
		 NONE => (internal,leaves) (* not comming here *)
	      |  SOME newTree => expandTreeByDepth (preord,E,R,mayInvFun) hyp depth newTree
     end

   fun constructTreeByDepth (preord,E,R,mayInvFun) depth (gtCnstr,x) =
     let val root = (x,  ([],gtCnstr)) 
	 val (xs,ys) = expandTreeByDepth (preord,E,R,mayInvFun) gtCnstr depth ([], [(root,[])])
     in xs @ ys
     end

   fun prTree nodes =
     let fun prNodes ((term,cnstr),i) = (T.prPosition i) ^ ": " ^ (prTermWithConstraint (term,cnstr))
     in LU.toStringCommaLnSquare prNodes nodes
     end

   fun connectNodes (preord,E,R,mayInvFun) (hyp,(term1,cnstr1),(term2,cnstr2)) =
       let val cnstrJoin = mkClosure (unionConstraint (cnstr1, cnstr2))
	   (* val _ = debug (fn _ => println ("check for " ^ (Term.toString term1) ^ " vs "  ^ (Term.toString term2))) *)
       in if isInconsistent cnstrJoin
	  then ((* println "inconsitent"; *) NONE)
	  else let val (eqCnstr,gtCnstr) = cnstrJoin
		   fun listToSubst eqCnstr = L.foldl (fn ((x,y),map) =>
							 case Term.compare (x,y) of
							     LESS => VM.insert (map,valOf (Term.varRootOfTerm x),y)
							   | GREATER => VM.insert (map,valOf (Term.varRootOfTerm y),x)
							   | EQUAL => map)
						     VM.empty eqCnstr
		   val sigma = listToSubst eqCnstr
		   fun mkTerm subst term = if Term.equal (term,Subst.applySubst subst term)
					  then term
					  else mkTerm subst (Subst.applySubst subst term)
		   val term1p = mkTerm sigma term1
		   val term2p = mkTerm sigma term2
		   val commonGtCnstr = gtCnstr @ hyp
		   val term1' = ordLinf (preord,R,E) commonGtCnstr term1p
		   val term2' = ordLinf (preord,R,E) commonGtCnstr term2p
		   (* val _ = println ("normalized to " ^ (Term.toString term1') ^ " vs " ^ (Term.toString term2')) *)
		   val invE = L.map (fn (l,r) => (r,l)) E
		   val X1p = Rewrite.oneStepReductSet (E@invE) term1p
		   val X1 = Rewrite.oneStepReductSet (E@invE) term1'
		   val XX = TS.union (X1p,X1) (* It looks like sometimes X1 is better and sometimes so does X1p  *)
	       in if TS.member (XX,term2') orelse TS.member (XX,term2p)
		  then SOME (getVarConstraint cnstrJoin)
		  else (case Subst.unify term1' term2' of 
		       SOME sigma => if Subst.isVarSubst sigma
				     then let val cnstr3 = (LU.eliminateDuplication' TP.equal (Subst.toEqs sigma), [])
					      val cnstrJoin3 = mkClosure (unionConstraint (cnstr3, cnstrJoin))
					      (* val _ = println (prConstraint cnstrJoin3) *)
					  in if isInconsistent cnstrJoin3
					     then NONE
					     else SOME (getVarConstraint cnstrJoin3)
					  end
				     else NONE
		    |  NONE => NONE)
	       end
       end
	 
   fun checkByDepth smtSolver tmpDir depth (preord,E,R,mayInvFun) (gtCnstr,x,y) =
     let val _ = debug (fn () => print ("Check joinability of " ^ (prGtConstraint gtCnstr)
			^ "|- <" ^ Term.toString x ^ ",  " ^ Term.toString y ^ ">\n"))
	 val xTree = constructTreeByDepth (preord,E,R,mayInvFun) depth (gtCnstr,x)
	 val _ = debug (fn () => print ("left tree:\n" ^ (prTree xTree)))
	 val yTree = constructTreeByDepth (preord,E,R,mayInvFun) depth (gtCnstr,y)
	 val _ = debug (fn () => print ("right tree:\n" ^ (prTree yTree)))
	 val constraintsAll = L.mapPartial (fn x => x)
					   (ListXProd.mapX (fn ((x,i),(y,j)) => connectNodes (preord,E,R,mayInvFun)
											     (gtCnstr,x,y))
							   (xTree,yTree))

        (* preparation for encoding *)
	 val vars = TermSet.listItems (Term.varTermSetInTerms ([x,y] @ ((fn (xs,ys) => xs @ ys) (LP.unzip gtCnstr))))
	 fun varIndex v =
	   let fun varIndexSub i [] =  (print "Error: failed to take var index\n";
					raise CrOrdError "Error: failed to take var index")
		 | varIndexSub i (w::ws) = if Term.equal (v,w) then i else varIndexSub (i+1) ws
	   in varIndexSub 0 vars
	   end
	 fun prVar v = "x_" ^ (Int.toString (varIndex v))
	 val vardefs = String.concat (L.map (fn v => YI.prDefInt (prVar v)) vars)
	 val positiveInt =  YI.prAnd (L.map (fn v => YI.prGe (prVar v, YI.prZero)) vars)
	 val encodingOfHyp = YI.prAnd (L.mapPartial (fn (u,v) => case (u,v) of
							   (T.Var _, T.Var _) => SOME (YI.prGt (prVar u, prVar v))
							 | (T.Var _, T.Fun _) => let val vs = TermSet.listItems (Term.varTermSetInTerm v)
										 in SOME (YI.prAnd (L.map (fn x => YI.prGe (prVar u, prVar x)) vs))
										 end
							 | _  =>  NONE) gtCnstr)

	 fun transEq (s,t) = YI.prEq (prVar s, prVar t)
	 fun transGt (s,t) = YI.prGt (prVar s, prVar t)
	 fun transCnstr (eqs,gts) = YI.prNot (YI.prAnd ((L.map transEq eqs) @ (L.map transGt gts)))
	 val encodingOfConnection  = YI.prAnd (L.map transCnstr constraintsAll)

         (* run smt solver and check *)
	 val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
	 val _ = debug (fn _ => print ("[" ^ tmpName ^ "]\n"))
	 val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
	 val outs = TextIO.openOut inTmpFile

         val _ = TextIO.output (outs, vardefs)
         val _ = TextIO.output (outs, YI.prAssert positiveInt)
         val _ = TextIO.output (outs, YI.prAssert encodingOfHyp)
         val _ = TextIO.output (outs, YI.prAssert encodingOfConnection)
			       
          val _ = TextIO.output (outs, "(set-evidence! true)\n")
          val _ = TextIO.flushOut outs

          val _ = TextIO.output (outs, "(check)\n")
          val _ = TextIO.flushOut outs

          val cmd = smtSolver ^ " " ^ inTmpFile ^ " 2>&1\n"
          val _ = debug (fn _ => print cmd)
          val proverCmd = ("/bin/sh", ["-c",cmd])
          val proc = Unix.execute proverCmd
          val (ins,outs) = Unix.streamsOf proc

          val answer = TextIO.inputLine ins
          val _ = if (isSome answer) 
                  then debug (fn _=> print ("output:" ^ (valOf answer)))
                  else ()

          val result =  (isSome answer) andalso (valOf answer = "unsat\n")

          (* $B%W%m%;%9=*N;=hM}(B*)
          fun finish () = 
              let val _ = debug (fn _=> print "finish the yices process\n")
                  val _ = TextIO.closeOut outs
                  val _ = TextIO.closeIn ins
                  val _ = Unix.reap proc  
                  val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 
              in () end


      in if not result 
         then let val _ =  debug (fn () => print " (fail)\n")
		  val _ = debug (fn _=> print "There may be a non-joinable instances ...\n")
                  val assign = YI.readAssigns ins
		  val table = L.map (fn (x,y) => (L.nth (vars,valOf (Int.fromString (String.extract (x,2,NONE)))),
						  valOf (Int.fromString y)))
				    (StringMap.listItemsi assign)
                  val _ = debug (fn _ => print "assignment:\n")
                  val _ = debug (fn _ => L.app (fn (x,y) => print ("[" ^ Term.toString x ^ "] |-> ["
								   ^ Int.toString y ^ "]\n"))
					       table)
              in (finish (); false)
              end
         else let val _ = debug (fn _ =>  print " (success)\n")
		  val _ = debug (fn _=> print "All instances are joinable ...\n")
	      in (finish (); true)
              end
     end

  (******  test   *****
   val _ = let val R = IOFotrs.rdRules ["plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))"]
	       val E = IOFotrs.rdRules ["plus(?x,?y) -> plus(?y,?x)",
   					"plus(?x,plus(?y,?z)) -> plus(?y,plus(?x,?z))" ]
	       val rules = R @ E
   	       val prec = [Fun.fromString "plus"]
               val compare = Order.comparePrec prec
   	       val preord = lpoGtUnderHyp compare
   	       val ord = preord []
	       val ecps = extendedCriticalPairs ord rules

	       val remainedEcps = L.mapPartial (fn (cnstr,x,y) =>
						   let val _ = print ("<" ^ (T.toString x) ^ " : " ^ (T.toString y) ^ ">\n")
						       val xnf = ordLinf (preord,R,E) cnstr x
						       val ynf = ordLinf (preord,R,E) cnstr y
						       val _ = print ("<" ^ (T.toString xnf) ^ " : " ^ (T.toString ynf) ^ ">")
						       val _ = if T.equal (xnf,ynf) then print "... Join\n" else print "... Not Join\n"
						   in if T.equal (xnf,ynf) then NONE else SOME (cnstr,xnf,ynf)
						   end) ecps
	       val smtSolver = "./yices"
	       val tmpDir = "./tmp/"

	       val remainedEcps2 = L.filter (fn ccp => not (checkByDepth smtSolver tmpDir 2 (preord,E,R) ccp)) remainedEcps

	   in null remainedEcps2
	   end
    *********)

   exception ProofFailed

   fun confluenceByOrderedRewritingStep num minisatPath smtSolverPath tmpDir rs =
     let val Crules = L.filter isCrule rs
	 val Srules = L.filter isSrule rs
	 val Arules = L.filter isArule rs

	 val Cfuns = L.mapPartial (fn (l,r) => Term.funRootOfTerm l) Crules
	 val Sfuns = L.mapPartial (fn (l,r) => Term.funRootOfTerm l) Srules
	 val Afuns = L.mapPartial (fn (l,r) => Term.funRootOfTerm l) Arules

	 val ACfuns = LU.intersection' Fun.equal (Cfuns, Afuns)
	 val SCfuns = LU.intersection' Fun.equal (Cfuns, Sfuns)
	 val additionalSfuns = LU.differenceByAll' Fun.equal (ACfuns, Sfuns)
	 val additionalAfuns = LU.differenceByAll' Fun.equal (SCfuns, Afuns)

	 val InvFunsForAC = LU.mapAppend (fn f => L.mapPartial (fn lr => getInvFunForAC f lr) rs) ACfuns
	 val mayInvFun = (case InvFunsForAC of [] => NONE | ((f,i)::_) => SOME i)
						   
	 fun getSort f = case L.find (fn (l,r) => case Term.funRootOfTerm l of
						      SOME g => Fun.equal (f,g)
						    | NONE => false) Crules of
			     SOME (l,r) => Term.sortOfTerm l
			   | NONE => raise CrOrdError "Error: failed to fix sort of a COM rule"

	 fun makeSrule f =
	   let val sort = getSort f
	       val x = Var.fromStringAndInt ("x",0)
	       val y = Var.fromStringAndInt ("y",0)
	       val z = Var.fromStringAndInt ("z",0)
	   in (T.Fun (f, [T.Var (x,sort), T.Fun (f, [T.Var (y,sort), T.Var (z,sort)], sort)], sort),
	       T.Fun (f, [T.Var (y,sort), T.Fun (f, [T.Var (x,sort), T.Var (z,sort)], sort)], sort))
	   end
	   
	 fun makeArule f =
	   let val sort = getSort f
	       val x = Var.fromStringAndInt ("x",0)
	       val y = Var.fromStringAndInt ("y",0)
	       val z = Var.fromStringAndInt ("z",0)
	   in (T.Fun (f, [T.Fun (f, [T.Var (x,sort), T.Var (y,sort)], sort), T.Var (z,sort)], sort),
	       T.Fun (f, [T.Var (x,sort), T.Fun (f, [T.Var (y,sort), T.Var (z,sort)], sort)], sort))
	   end
	   
	 fun makeCrule f =
	   let val sort = getSort f
	       val x = Var.fromStringAndInt ("x",0)
	       val y = Var.fromStringAndInt ("y",0)
	   in (T.Fun (f, [T.Var (x,sort), T.Var (y,sort)], sort),
	       T.Fun (f, [T.Var (y,sort), T.Var (x,sort)], sort))
	   end

	 val additionalArules = L.map makeArule additionalAfuns
	 val additionalSrules = L.map makeSrule additionalSfuns
				      
(*	 val (preE,preR) = L.partition (fn (l,r) => isCrule (l,r) orelse isSrule (l,r)) rs *)

	 fun isReachable l r = isCrule (l,r) orelse isSrule (l,r) orelse isArule (l,r)
			       orelse TS.member (Rewrite.manyStepsReductSet rs 10 l,r)

	 fun haveSameFunRoot (l,r) = case (T.funRootOfTerm l,T.funRootOfTerm r) of
					 (SOME f, SOME g) => Fun.equal (f,g) 
				      |  _ =>  false
	 fun lexDecreasingByProperSubterm (x::xs,y::ys) = T.isAProperSubterm y x
							  orelse (T.equal (x,y) andalso lexDecreasingByProperSubterm (xs,ys))
	   | lexDecreasingByProperSubterm _ = false

	 fun isNonDecreasingRule (l,r) =
	     L.exists (fn lr2 => Subst.identicalModuloRenamingRule (r,l) lr2) rs
	     orelse TS.member (Rewrite.oneStepReductSet rs r,l)
		      
	   (* isReachable r l
	       andalso not (haveSameFunRoot (l,r) andalso lexDecreasingByProperSubterm (T.argsOfTerm l, T.argsOfTerm r))
	   truns out NOT doable theoretially 
	    *)
					    
	 val (preE,preR) = L.partition isNonDecreasingRule rs

	 val preEfunSet = Trs.funSetInRules preE
	 val preRfunSet = Trs.funSetInRules preR

	 val (R,E') = if FS.isSubset (preEfunSet, preRfunSet)
		     then (preR @ additionalArules, preE @ additionalSrules)
		     else (preR @ preE @ additionalArules, additionalSrules)


	 fun isAuxiliary (l,r) =
	     not (isArule (l,r))
	     andalso not (isCrule (l,r))
	     andalso not (isSrule (l,r))
	     andalso let val F = LU.union' Fun.equal (ACfuns, SCfuns)
			 val fset = Trs.funSetInRule (l,r)
			 val f = hd (FS.listItems fset)
			 val AC = [makeCrule f, makeArule f]
		     in if (FS.numItems fset <> 1)
			   orelse not (LU.member' Fun.equal f F)
			then false
			else TS.member (Rewrite.manyStepsReductSet AC 10 l, r)
		     end

	 val E = L.filter (not o isAuxiliary) E'
	 val rules = R @ E

	 val _ = println "remove redundants rules and split"
	 val _ = print "R-part:\n"
	 val _ = print (Trs.prRules R)
	 val _ = print "E-part:\n"
	 val _ = print (Trs.prRules E)
		       

	 (* val _ = if L.all (fn (l,r) => isReachable r l) E *)
	 (* 	 then print "...reversibility of non-decreasing rules: ok\n" *)
	 (* 	 else (print "...reversibility of non-decreasing rules: failed\n"; *)
	 (* 	       raise ProofFailed) *)

	 val opt = { useQuasi=false, useLex = false, useMul = false, useAf = false }
         val compare = case PoSolver.poSolverForGcr minisatPath tmpDir opt (R,[]) of
			   NONE => (print "...failed to find a suitable LPO.\n";
				    raise ProofFailed)
			 | SOME (funqprec,mul,lex) => Order.compareQprec funqprec
	 val preord = case mayInvFun of
			  SOME i => lpoGtUnderHypUnarySpecial i compare
			| NONE => lpoGtUnderHyp compare
				
   	 val ord = preord []


	 (* val (R2,E2) = L.partition (fn (l,r) => ord (l,r)) E *)
	 (* val R = preR @ R2 *)
	 (* val E = E2 @ preE *)
	 (* val rules = R @ E *)
		
	 val ecps = extendedCriticalPairs ord rules

	 (* check joinability by normal forms w.r.t. R U E^{>(hyp)}  *)
	 val remainedEcps = L.mapPartial (fn (i,j,pos,gtCnstr,x,y) =>
					     let val _ = debug (fn _ => print ("<" ^ (T.toString x) ^ " : " ^ (T.toString y) ^ ">\n"))
						 val xnf = ordLinf (preord,R,E) gtCnstr x
						 val ynf = ordLinf (preord,R,E) gtCnstr y
						 val _ = debug (fn _ => print ("==> <" ^ (T.toString xnf) ^ " : " ^ (T.toString ynf) ^ ">"))
						 val _ = debug (fn _ => if T.equal (xnf,ynf) then print "... Join\n" else print "... Not Join\n")
					     in if T.equal (xnf,ynf)
						then NONE
						else SOME (i,j,pos,gtCnstr,xnf,ynf)
					     end) ecps

	 (* check joinality of **NF** by case analysis by ordering on sigma(x), sigma(y), ... *)
	 val _ = debug (fn _ => print "Check joinablity of non-join CCPs by case analysis...\n")
	 val remainedEcps2 = L.filter (fn (i,j,pos,gtCnstr,x,y) => not (checkByDepth smtSolverPath tmpDir 2 (preord,E,R,mayInvFun) (gtCnstr,x,y)))
				      remainedEcps

  
        (* $BN>JU$K6&DL$9$k??ItJ,9`$rCj>]2=$7$F!$%A%'%C%/(B *)
	 fun performAbstraction (l,r) =
	   let val subtermSetLeft = Term.setOfProperSubterms l
	       val subtermSetRight = Term.setOfProperSubterms r
	       val commons = L.filter (not o T.isVar) (TS.listItems (TS.intersection (subtermSetRight, subtermSetLeft)))
	       fun removeNonMaximal ([], maxs) = maxs
		 | removeNonMaximal (x::xs, maxs) =
		   if L.exists (fn y => T.isASubterm x y) maxs
		   then removeNonMaximal (xs, maxs)
		   else removeNonMaximal (xs, x::L.filter (fn y => not (T.isAProperSubterm y x)) maxs)
	       val maximals = removeNonMaximal (commons, [])
	       val vars = L.tabulate (L.length maximals, fn i => T.Var (Var.fromStringAndInt ("x",i),T.sortOfTerm (L.nth (maximals, i))))
	       val rel = (maximals,vars)
	       fun replace ([],[]) (xs2,ys2) t = (case t of
						      T.Fun (f,args,ty) => let val ((xs',ys'),ts') =  replaceList (xs2,ys2) args
									       val t' = T.Fun (f,ts',ty)
									   in ((xs',ys'), t')
									   end
						   | T.Var _ =>  ((xs2,ys2),t) )
		 | replace (x::xs,y::ys) (xs2,ys2) t = if T.equal (t,x)
						       then ((L.revAppend (xs2,xs), L.revAppend (ys2,ys)), y)
						       else replace (xs,ys) (x::xs2,y::ys2) t
		 | replace  _ _ t = raise CrOrdError "Error: non syncronizing term list pair"
	       and replaceList (xs,ys) [] = ((xs,ys), [])
		 | replaceList (xs,ys) (t::ts) = let val ((xs2,ys2),t') = replace (xs,ys) ([],[]) t
						     val ((xs3,ys3),ts') = replaceList (xs2,ys2) ts
						 in ((xs3,ys3),t'::ts')
						 end
	       val ((_,l'), (_,r')) = (replace rel ([],[]) l, replace rel ([],[]) r)
	       (* val _ = print (Trs. prRules [(l',r')]) *)
	   in (l',r')
	   end

	 val _ = if not (null remainedEcps2)
		 then print "Perform abstraction and check...\n"
		 else ()

	 val remainedEcps3 = L.filter (fn (i,j,pos,gtCnstr,x,y) => let val (x',y') = performAbstraction (x,y)
								   in if TP.equal ((x',y'), (x,y))
								      then ((* print "...failed to abstract\n" ; *) true)
								      else not (checkByDepth smtSolverPath tmpDir 2 (preord,E,R,mayInvFun) ([],x',y'))
								   end)
				      remainedEcps2

        (* $BLZ$N?<$5$rBg$-$/$7$F!$%A%'%C%/(B *)
	 fun vsize x = let val depth = VS.numItems (T.varSetInTerm x) + 1  (* this works! *)
		       in if depth > 4 then 3 else depth
		       end

	 val _ = if not (null remainedEcps2)
		 then print "Increase tree depth and check...\n"
		 else ()

	 val remainedEcps4 = L.filter (fn (i,j,pos,gtCnstr,x,y) => not (checkByDepth smtSolverPath tmpDir (vsize x) (preord,E,R,mayInvFun) (gtCnstr,x,y)))
				      remainedEcps3


	 val rulesToAdd = L.mapPartial (fn (i,j,pos,gtCnstr,x,y) =>
					   if VS.isSubset (Term.varSetInTerm y, Term.varSetInTerm x) andalso isReachable x y
					   then SOME (x,y)
					   else if VS.isSubset (Term.varSetInTerm x, Term.varSetInTerm y) andalso isReachable y x
					   then SOME (y,x)
					   else NONE)
				       remainedEcps4

	 fun removeRedundantRules ([], ans) = ans
	   | removeRedundantRules ((l,r)::rest, ans) =
	     if L.exists (fn lr2 => Subst.identicalModuloRenamingRule (l,r) lr2) ans
	     then removeRedundantRules (rest, ans)
	     else case L.find (fn (l2,r2) => Subst.isVariant l l2) ans of
		      SOME (l2,r2) => if Term.termSize l2 > Term.termSize l
				      then removeRedundantRules (rest, (l,r):: LU.deleteOne' TP.equal (l2,r2) ans)
				      else removeRedundantRules (rest, ans)
		    | NONE => removeRedundantRules (rest, (l,r)::ans)
			   
	 val rulesToAdd2 = removeRedundantRules (rulesToAdd, [])
	 val rulesToAdd3 = L.filter (fn lr => not (LU.member' (fn (lr1,lr2) => Subst.identicalModuloRenamingRule lr1 lr2) lr rules))
				    rulesToAdd2

	 fun removeNonReduced ss =
	     let val idx = L.tabulate (L.length ss, fn x => x)
	     in L.mapPartial (fn i => let val (l,r) = L.nth (ss, i)
					  val rest = LU.exceptNth (ss, i)
				      in if Rewrite.isNormalForm ss r
					    andalso Rewrite.isNormalForm rest l
					 then SOME (l,r) else NONE
				      end) idx
	     end
	 
	 val rulesToAdd4 = let val (nonDecreasingRules, decreasingRules) = L.partition isNonDecreasingRule rulesToAdd3
			       fun selectMin [] = (print "Error: failed to find a minimal non-decreasing rule.\n";
						   raise CrOrdError "Error: failed to find a minimal non-decreasing rule.") (* not coming here *)
				 | selectMin (x::[]) = x
				 | selectMin (x::xs) = let val y = selectMin xs
						       in if Trs.ruleSize x < Trs.ruleSize y then x else y
						       end
			       (* fun vsize (l,r) = VS.numItems (T.varSetInTerm l) *)
			       (* fun selectSizeMins [] = (print "Error: failed to find a minimal non-decreasing rule.\n"; *)
			       (* 			   raise CrOrdError "Error: failed to find a minimal non-decreasing rule.") (* not coming here *) *)
			       (* 	 | selectSizeMins (x::[]) = (vsize x, [x]) *)
			       (* 	 | selectSizeMins (x::xs) = let val (sizeY,ys) = selectSizeMins xs *)
			       (* 					val sizeX = vsize x *)
			       (* 				    in if sizeX  < sizeY then (sizeX, [x]) *)
			       (* 				       else if sizeX  = sizeY then (sizeY, x::ys) *)
			       (* 				       else (sizeY,ys) *)
			       (* 				    end *)
			       (* val decreasingRules2 = let val drs = removeNonReduced decreasingRules *)
			       (* 			      in if null drs then [] else ((fn (i,ys) => ys) (selectSizeMins drs)) *)
			       (* 			      end *)
			   (* in if null nonDecreasingRules *)
			   (*    then decreasingRules2 *)
			   (*    else (selectMin nonDecreasingRules)::decreasingRules2 *)
			   in if not (null decreasingRules)
			      then [selectMin decreasingRules]
			      else if not (null nonDecreasingRules)
			      then [selectMin nonDecreasingRules]
			      else []
			   (*    then decreasingRules2 *)
			   (*    else (selectMin nonDecreasingRules)::decreasingRules2 *)
			   end

	 val numberOfIteration = 5
			     
     in if null remainedEcps4
	then true
	else if not (null rulesToAdd4) andalso num < numberOfIteration
	then let val _ = print "Apply reduction-preserving transformation...\n"
		 val _ = print ("Rules to add:\n" ^ (Trs.prRules rulesToAdd4))
	     in confluenceByOrderedRewritingStep (num+1) minisatPath smtSolverPath tmpDir (rules @ rulesToAdd4)
	     end
	else false
     end
     handle ProofFailed => false

   fun confluenceByOrderedRewriting minisatPath smtSolverPath tmpDir rs =
     confluenceByOrderedRewritingStep 1 minisatPath smtSolverPath tmpDir rs


   end (* of local *)


   end; (* of structure CrOrd *)

