(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigta University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/cr_solver.sml
 * description: confluence checker
 * author: YOSHIDA Junichi
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature CR_SOLVER = 
sig
    val runProfile: bool ref
    val runDebug: bool ref
    val runCertification: bool ref

    type options = {
         useDirect:bool,   (*  $BD>OBJ,2r$r;H$&$+(B *)
         useCommutative:bool,   (*  $B2D49J,2r$r;H$&$+(B *)
         useNonMinimalCommutative:bool,   (* $BHs6K>.2D49J,2r$r;H$&$+(B *)
         useLayer:bool,    (*  $BAXJ]B8J,2r$r;H$&$+(B *)
         useCertifiableOutputOption:bool    (*  $B8!>Z=PNO$r=P$9$+(B *)
    }

    val crSolver: Solver.options
                  -> DpSolver.options
                  -> PoSolver.options
                  -> options
                  -> Trs.rules
                  ->  Cr.ConfluenceResult

    val condCrSolver: Solver.options
                  -> DpSolver.options
                  -> PoSolver.options
                  -> options
                  -> Ctrs.crules
                  ->  Cr.ConfluenceResult

    val comSolver: Solver.options
                  -> DpSolver.options
                  -> PoSolver.options
                  -> options
                  -> Trs.rules * Trs.rules
                  ->  Cr.CommutativityResult

    val useHuetToyamaOostrom: bool ref;
    val useHuetStrongClosed: bool ref;
    val useOyamaguchiOhta: bool ref;
    val useToyamaParallel: bool ref;
    val useOkuiSimultaneous: bool ref;
    val useDecreasingDiagrams: bool ref;
    val useOhtaOyamaguchiToyama: bool ref;
    val useGomiOyamaguchiOhtaDepth: bool ref;
    val useGomiOyamaguchiOhtaWeight: bool ref;
    val useSuzukiAotoToyama: bool ref;
    val useUchidaAotoToyama: bool ref;
    val useUchidaAotoToyamaStrong: bool ref;
    val useSakaiOgawa: bool ref;
    val useKnuthBendix: bool ref;
    val useHuetModulo: bool ref;
    val useBidirectionalCompletion: bool ref;
    val useToyamaOyamaguchiWeightDecreasing: bool ref;
    val useMartinNipkowOrderedRewriting: bool ref;
    val useYoshidaAotoToyama: bool ref;
    val useNonCommutation: bool ref;

(*     val proveGroundConfluence: bool ref; *)

end

structure CrSolver : CR_SOLVER =
struct
    local
	open Prop
	open Cr
	structure A = Atom
	structure CU = CertifyUtil
	structure L = List
	structure LP = ListPair
	structure LU = ListUtil
	structure FS = FunSet
	structure FIS = FunIntSet
	structure ILM = IntListMap2
	structure T = Term
	structure TS = TermSet
	structure VS = VarSet
	structure YI = YicesInput
	open PrintUtil (* PU.outs := 'log file' in acp.sml *)
    in
    
    exception CrSolverError
          
    val runProfile = ref false : bool ref
    val runDebug = ref false : bool ref
    val runCertification = ref false : bool ref

    fun debug f = if !runDebug then f () else ()
    fun cpout f = if !runCertification then f () else ()

    type options = {
         useDirect:bool,   (*  $BD>OBJ,2r$r;H$&$+(B *)
         useCommutative:bool,   (*  $B2D49J,2r$r;H$&$+(B *)
         useNonMinimalCommutative:bool,   (* $BHs6K>.2D49J,2r$r;H$&$+(B *)
         useLayer:bool,    (*  $BAXJ]B8J,2r$r;H$&$+(B *)
         useCertifiableOutputOption:bool    (*  $B8!>Z=PNO$r=P$9$+(B *)
    }

    val useHuetToyamaOostrom = ref true
    val useHuetStrongClosed = ref true
    val useOyamaguchiOhta = ref true
    val useToyamaParallel = ref true
    val useOkuiSimultaneous = ref true
    val useDecreasingDiagrams = ref true
    val useOhtaOyamaguchiToyama = ref true
    val useGomiOyamaguchiOhtaDepth = ref true
    val useGomiOyamaguchiOhtaWeight = ref true
    val useSuzukiAotoToyama = ref true
    val useUchidaAotoToyama = ref true
    val useUchidaAotoToyamaStrong = ref true
    val useSakaiOgawa = ref true
    val useKnuthBendix = ref true
    val useHuetModulo = ref false (* included in BidirectionalCompletion *)
    val useBidirectionalCompletion = ref true
    val useToyamaOyamaguchiWeightDecreasing = ref true
    val useMartinNipkowOrderedRewriting = ref true
    val useYoshidaAotoToyama = ref true
    val useNonCommutation = ref true
    (* val proveGroundConfluence = ref false *)

    val nameHuetToyamaOostrom = "Development Closed"
    val nameHuetStrongClosed = "Strongly Closed"
    val nameOyamaguchiOhta = "Upside-Parallel-Closed/Outside-Closed"
    val nameToyamaParallel = "Toyama (Parallel CPs)"
    val nameOkuiSimultaneous = "Okui (Simultaneous CPs)"
    val nameHuetModulo = "Huet (modulo AC)"
    val nameDecreasingDiagrams = "Diagram Decreasing"
    val nameOhtaOyamaguchiToyama = "Simple-Right-Linear"
    val nameGomiOyamaguchiOhtaDepth = "Strongly Depth-Preserving & Root-E-Closed/Non-E-Overlapping"
    val nameGomiOyamaguchiOhtaWeight = "Strongly Weight-Preserving & Root-E-Closed/Non-E-Overlapping"
    val nameSuzukiAotoToyama = "Quasi-Left-Linear & Parallel-Closed"
    val nameUchidaAotoToyama = "Quasi-Linear & Linearized-Decreasing"
    val nameUchidaAotoToyamaStrong = "Strongly Quasi-Linear & Hierarchically Decreasing"
    val nameSakaiOgawa = "Weakly-Non-Overlapping & Non-Collapsing & Shallow"
    val nameNonConfluence = "Non-Confluence"
    val nameKnuthBendix = "Knuth & Bendix"
    val nameToyamaOyamaguchiWeightDecreasing = "Weight-Decreasing Joinable"
    val nameYoshidaAotoToyama = "Development Closed"
    val nameNonCommutation = "Non-Commutation"

    fun unknown s = "unknown " ^ s

    val maxRewriteLenForNonConfluenceCandidates = ref 2
    val maxRewriteLenForDevelopmentClosedCandidates = ref 5

    local 
	open Term
(*	open Cr *)
(*	open PrintUtil *)
  (* $B9gN.@-$N4pK\H=Dj(B *)
    in
    fun checkConfluenceConditions2 satSolver smtSolver tmpDir 
				  isTerminating isRelativeTerminating rs = 
	let 
	   val _ = print "Apply Direct Methods...\n"


	   val inCpeaksWithIndexAndPosFs = Cr.insideCriticalPeaksWithIndexAndPosFs rs

	   val inCpsPos = L.map (fn (mn,(ps,fs,x,y,z)) => (ps,y,z)) inCpeaksWithIndexAndPosFs
	   val inCpeaks = L.map (fn (mn,(ps,fs,x,y,z)) => (x,y,z)) inCpeaksWithIndexAndPosFs
	   val outCpeaks = outsideCriticalPeaksInOneside rs

	   val inCps = L.map (fn (_,y,z) => (y,z)) inCpeaks
	   val outCps = L.map (fn (_,y,z) => (y,z)) outCpeaks

	   val _ = print ("Inner CPs:\n" ^ (LU.toStringCommaLnSquare (IOFotrs.prEq) inCps))
	   val _ = print ("Outer CPs:\n" ^ (LU.toStringCommaLnSquare (IOFotrs.prEq) outCps))

	   val (_,dSymSet,cSymSet,_) = Trs.fdcSetInRules rs

	   val developSetTable = ref TermMap.empty

	   fun insertDevelopSet term = 
	       let val tset = Rewrite.developOneStepReductSet rs term
		   val _ = developSetTable := TermMap.insert (!developSetTable,term,tset)
	       in tset
	       end

	   fun lookupDevelopSet term =
	       case TermMap.find (!developSetTable,term) of
		   SOME tset => tset
		 | NONE => insertDevelopSet term

	   val reductSetTable = ref TermMap.empty

	   fun insertReductSet term = 
	       let val maxLen = !maxRewriteLenForNonConfluenceCandidates
		   val tset = Rewrite.manyStepsReductSet rs maxLen term
		   val _ = reductSetTable := TermMap.insert (!reductSetTable,term,tset)
	       in tset
	       end

	   fun lookupReductSet term =
	       case TermMap.find (!reductSetTable,term) of
		   SOME tset => tset
		 | NONE => insertReductSet term

	   fun checkHuetToyamaOostrom _ = 
	       (!useHuetToyamaOostrom)
	       andalso
	       let val certificate = ref 0
		   fun checkIn (l,r) = Term.equal (l,r) 
				       orelse let val dl = lookupDevelopSet l
						  val _ = debug (fn _ => println ("check inner CP " ^ (Trs.prEq (l,r))))
						  val _ = debug (fn _ => println ("{ s | l -o-> s } = " ^
								 (prSetInOneLine Term.toString (TS.listItems dl))))
					      in TS.member (dl,r)
					      end

						  
		   fun checkOut (l,r) = 
		       Term.equal (l,r) 
		       orelse 
		       let val l1 = lookupDevelopSet l
			   val r1 = lookupDevelopSet r
			   val _ = debug (fn _ => println ("check outer CP " ^ (Trs.prEq (l,r))))
			   val _ = debug (fn _ => println ("X1 = { s | l -o-> s } = " ^ 
						  (prSetInOneLine Term.toString (TS.listItems l1))))
			   val _ = debug (fn _ => println ("Y1 = { t | r -o-> t } = " ^
						  (prSetInOneLine Term.toString (TS.listItems r1))))
			   val _ = debug (fn _ => println ("check X1 cap Y1 /= {} "))
		       in not (TS.isEmpty (TS.intersection (l1,r1)))
			  orelse let val l2 = lookupReductSet l
				     val r2 = lookupReductSet r
				     val _ = debug (fn _ => println ("X2 = { s | l -n-> s } = " ^
						    (prSetInOneLine Term.toString (TS.listItems l2))))
				     val _ = debug (fn _ => println ("Y2 = { s | r -n-> s } = " ^
								   (prSetInOneLine Term.toString (TS.listItems r2))))
				     val _ = debug (fn _ => println ("check X1 cap Y2 /= {} and X2 cap Y1 /= {}"))
				 in not (TS.isEmpty (TS.intersection (l1,r2)))
				    andalso not (TS.isEmpty (TS.intersection (l2,r1)))
				 end
		       end
	       in
		   if L.all checkIn inCps
		      andalso L.all checkOut outCps
		   then (println nameHuetToyamaOostrom;
			 cpout (fn _ => CU.output (CU.beginProof (CU.beginCrProof
  	                  (fn _ => CU.encloseProofLeafBy "developmentClosed" ""))));
			 true)
		   else (println (unknown nameHuetToyamaOostrom); false)
	       end

	   fun checkHuetStrongClosed _ = 
	       (!useHuetStrongClosed)
	       andalso
	       let val certificate = ref 0
		   fun strongClosed (l,r) = 
		       Term.equal (l,r) 
		       orelse 		    
		       let 
			   val _ = debug (fn _ => println ("check CP " ^ (Trs.prEq (l,r))))
			   val r1 = lookupDevelopSet r
			   val l1 = TS.add (Rewrite.oneStepReductSet rs l, l)
			   val _ = debug (fn _ => println ("X = { s | l -=-> s } = " ^
							   (prSetInOneLine Term.toString (TS.listItems l1))))
			   val _ = debug (fn _ => println ("Y1 = { s | r -o-> s } = " ^
							   (prSetInOneLine Term.toString (TS.listItems r1))))
			   val _ = debug (fn _ => println ("check X cap Y1 /= {} "))
		       in not (TS.isEmpty (TS.intersection (r1,l1)))
			      andalso (if !runCertification 
				       then certificate := Int.max(!certificate, TS.numItems r1)
				       else (); true)
			  orelse
			  let val r2 = lookupReductSet r
			   val _ = debug (fn _ => println ("Y2 = { s | r -n-> s } = " ^
							   (prSetInOneLine Term.toString (TS.listItems r2))))
			   val _ = debug (fn _ => println ("check X cap Y2 /= {} "))
			  in not (TS.isEmpty (TS.intersection (r2,l1)))
			     andalso (if !runCertification 
				      then certificate := Int.max(!certificate, TS.numItems r2)
				      else (); true)
			  end
		       end
	       in
		   if L.all strongClosed outCps
		      andalso L.all strongClosed (L.map (fn (x,y) => (y,x)) outCps)
		      andalso L.all strongClosed inCps
		      andalso L.all strongClosed (L.map (fn (x,y) => (y,x)) inCps)
		   then (println nameHuetStrongClosed; 
			 cpout (fn _ => CU.output (CU.beginProof (CU.beginCrProof
                               (fn _ => CU.encloseProofLeafBy "stronglyClosed" (Int.toString (!certificate))))));
			 true)
		   else (println (unknown nameHuetStrongClosed);false)
	       end


	   fun checkOyamaguchiOhta _ = 
	       (!useOyamaguchiOhta)
	       andalso
	       let val depthRestrectedClosedCps = isDepthRestrictedClosedCps rs inCpsPos outCps
		   val _ = if depthRestrectedClosedCps
			   then println nameOyamaguchiOhta
			   else println ("unknown " ^ nameOyamaguchiOhta)
	       in depthRestrectedClosedCps
	       end


	   fun checkToyamaParallel _ = 
	       (!useToyamaParallel)
	       andalso
	       let val parallelCriticalPairClosed = 
		       isParallelCriticalPairClosed rs (inCps @ outCps) 
		   val _ = if parallelCriticalPairClosed
			   then println nameToyamaParallel
			   else println (unknown nameToyamaParallel)
	       in parallelCriticalPairClosed
	       end

	   fun checkOkuiSimultaneous _ = 
	       (!useOkuiSimultaneous)
	       andalso
	       let val okuiClosedCps = isOkuiClosedCps rs
		   val _ = if okuiClosedCps
			   then println nameOkuiSimultaneous
			   else println (unknown nameOkuiSimultaneous)
	       in okuiClosedCps
	       end

	   fun checkHuetModulo _ = 
	       (!useHuetModulo)
	       andalso
	       if Cr.isHuetModulo isRelativeTerminating rs
	       then (println nameHuetModulo; true)
	       else (println (unknown nameHuetModulo);false)
	       
	   fun checkDecreasingDiagrams _ = 
	       (!useDecreasingDiagrams)
	       andalso
	       let val _ = print "check Locally Decreasing Diagrams by Rule Labelling...\n"
(*		   val decreasingCps = CrDiagram2.hasDecreasingCps0 smtSolver tmpDir rs *)
(*		   val decreasingCps = CrDiagram2.hasDecreasingCps smtSolver tmpDir rs  *)
		   val decreasingCps = CrDiagram.hasDecreasingCps smtSolver tmpDir rs  
		   val _ = if decreasingCps
			   then println nameDecreasingDiagrams
			   else println (unknown nameDecreasingDiagrams)
	       in decreasingCps
	       end

	   fun checkOhtaOyamaguchiToyama _ =
	       (!useOhtaOyamaguchiToyama)
	       andalso
	       let fun checkVar (l,r) = 
		       VS.isEmpty
			   (VS.intersection (		      
			    VS.difference (Term.varSetInTerm l, Term.linearVarSetInTerm l),
			    Term.varSetInTerm r))
		   val _ = debug (fn _ => println "check all x in V(r) is linear in l")
	       in if L.all checkVar rs
		     andalso 
		     let val cp = Cr.criticalPairs (L.map (fn (l,r) => (Subst.linearize l,r)) rs)
			 val _ = debug (fn _ => print ("CPs after linearization of lhs:\n"))
			 val _ = debug (fn _ => print (LU.toStringCommaLnSquare (IOFotrs.prEq) cp))
		     in  null cp
		     end
		  then (println nameOhtaOyamaguchiToyama; true)
		  else (println (unknown nameOhtaOyamaguchiToyama); false)
	       end

	   fun transformationForWightPreserving smtSolver rs =
	       let val faSet = Trs.funAritySetInRules rs
		   val funlist = L.map (fn (f,_) => f) (FIS.listItems faSet)

		   fun varWithFuncOnPath (Var (x,_)) =  [(x,[])]
		     | varWithFuncOnPath (Fun (f,ts,_)) =  
		       L.map (fn (x,fs) => (x,f::fs)) (LU.mapAppend varWithFuncOnPath ts)

		   fun prVarAndFuncs (x,fs) = "<" ^ (Var.toString x) ^ ","
					      ^ (LU.toStringCommaSquare Fun.toString fs) ^ ">"

		   fun prVarAndFuncsInRule (l,r) =
		       let (* val _ = print ((Term.toString l) ^ " -> " ^ (Term.toString r) ^ "\n") *)
			   val listl = varWithFuncOnPath l
			   (* val _ = print " lhs:" *)
			   (* val _ = print (LU.toStringSemicolonCurly prVarAndFuncs listl) *)
			   (* val _ = print "\n" *)
			   val listr = varWithFuncOnPath r
			   (* val _ = print " rhs:" *)
			   (* val _ = print (LU.toStringSemicolonCurly prVarAndFuncs listr) *)
			   (* val _ = print "\n" *)
		       in () end

		   (* val _ = L.app prVarAndFuncsInRule rs *)

		   fun prConstraint pr (l,r) =
		       let (* val _ = print ((Term.toString l) ^ " -> " ^ (Term.toString r) ^ "\n") *)
		   	   val listl = varWithFuncOnPath l
		   	   val listr = varWithFuncOnPath r
		   	   val vs = VS.listItems (VS.intersection (Term.varSetInTerm l, Term.varSetInTerm r))
		   	   fun prConstraint x =
			       let fun select xs = L.mapPartial 
						       (fn (y,fs) => if Var.equal (x,y) then SOME fs else NONE) 
						       xs
				   fun mkpoly fs = YI.prPlus (L.map Fun.toString fs)
				   val rstrings = L.map mkpoly (select listr)
				   val lstrings = L.map mkpoly (select listl)
			       in YI.prAnd (ListXProd.mapX YI.prLe (rstrings,lstrings))
			       end
		       in L.app (fn x => pr (YI.prAssert (prConstraint x))) vs
		       end

		   (*	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ())) *)
		   (*	  val _ = debug (fn _=> print ("[" ^ tmpName ^ "]\n")) *)
		   (*	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName} *)
		   (*	  val fouts = TextIO.openOut inTmpFile *)
		   (*     val cmd = smtSolver ^ " " ^ inTmpFile ^ " 2>&1\n"  *)
 		   val cmd = smtSolver ^ " 2>/dev/null\n"

 		   val proverCmd = ("/bin/sh", ["-c",cmd])
		   val proc = Unix.execute proverCmd
		   val (ins,outs) = Unix.streamsOf proc

		   (* fun yprint str = (TextIO.output (fouts, str); TextIO.flushOut fouts) *)
		   fun yprint str = (TextIO.output (outs, str); TextIO.flushOut outs)
		   val _ = yprint "(set-evidence! true)\n"

		   val _ = L.app (fn (f,ar) => 
				     let val fname = Fun.toString f
					 val _ = yprint (YI.prDefInt fname)
					 val _ = yprint (YI.prAssert (YI.prAnd [YI.prLe ("1",fname), 
									      YI.prLe (fname,"4")]))
				     in () end)
				 (FIS.listItems faSet)
		   val _ = L.app (prConstraint yprint) rs
		   val _ = yprint "(check)\n"
 

		   val answer = TextIO.inputLine ins

		   (* val _ = if (isSome answer)  *)
		   (* 	   then debug (fn _=> print ("output:" ^ (valOf answer))) *)
		   (* 	   else () *)

		   val result =  (isSome answer) andalso (valOf answer = "sat\n")

		   (* $B%W%m%;%9=*N;=hM}(B*)
		   fun finish () = 
		       let (* val _ = debug (fn _=> print "finish the yices process\n")  *)
			   val _ = TextIO.closeOut outs
			   val _ = TextIO.closeIn ins
			   val _ = Unix.reap proc  
			   (* val _ = OS.FileSys.remove inTmpFile *)
		       in () end

	       in if not result 
		  then let val _ = debug (fn _=> print "No suitable weight found ...\n")
		       in (finish ();NONE)
		       end
		  else let 
			  val assign = YI.readAssigns ins
			  (* val _ = print "assignment:\n" *)
			  (* val _ = StringMap.appi (fn (k,str) => print (k ^ ": " ^ str ^ "\n")) assign *)
			  (* val _ = print "end of assignment.\n" *)
			  val funNumlist = L.mapPartial
					       (fn (f,str) => let val i = valOf (Int.fromString str)
							      in if i > 1 then SOME (Fun.fromString f, i) else NONE
							      end)
					       (StringMap.listItemsi assign)
			  
			  fun mknewfun f (init,n) = 
			      let val newfs = L.tabulate (n, fn i => Fun.fromString (
								     (Fun.toString f) ^ "_" 
								     ^ (Int.toString (i+init))))
				  fun mkterm [] ts = (print "mkterm\n"; raise CrSolverError)
				    | mkterm [f] ts = Fun (f,ts,Sort.null)
				    | mkterm (f::fs) ts = Fun (f,[mkterm fs ts],Sort.null)
			      in if LU.disjoint' Fun.equal (newfs,funlist)
				 then (f,fn args => mkterm newfs args)
				 else mknewfun f (init+n,n)
			      end
			  val replacelist = L.map (fn (f,n)=> mknewfun f (1,n)) funNumlist

			  fun replace (t as (Var _)) = t
			    | replace (t as (Fun (f,args,ty))) = 
			      case L.find (fn (g,mk) => Fun.equal (f,g)) replacelist of
				  (* SOME (g,mk) => mk args | NONE => Fun (f,L.map replace args,ty) *)
				  (* bug fix 2019/03/04 *)
				  SOME (g,mk) => mk (L.map replace args) | NONE => Fun (f,L.map replace args,ty)
			  val ans = L.map (fn (l,r) => (replace l, replace r)) rs
		      in (finish (); SOME ans)
		      end
	       end
       
	   fun checkGomiOyamaguchiOhta _ = 
	       ((!useGomiOyamaguchiOhtaDepth) orelse (!useGomiOyamaguchiOhtaWeight))
	       andalso
	       if Cr.confluenceByDepthPreserving rs
	       then (println nameGomiOyamaguchiOhtaDepth; true)
	       else if (!useGomiOyamaguchiOhtaWeight)
	       then
		   let val _ = println (unknown nameGomiOyamaguchiOhtaDepth)
		       val _ = debug (fn _ => print "try to apply CR-preserving transformation\n")
		   in case transformationForWightPreserving smtSolver rs of
			  NONE => (println (unknown nameGomiOyamaguchiOhtaWeight); false)
			| SOME rs' => 
			  let val _ =  debug (fn _ => print "new rules obtained:\n")
			      val _ =  debug (fn _ => print (Trs.prRules rs'))
			  in if Cr.confluenceByDepthPreserving rs'
			     then (println nameGomiOyamaguchiOhtaWeight; true)
			     else (println (unknown nameGomiOyamaguchiOhtaWeight); false)
			  end
		   end
	       else (println (unknown nameGomiOyamaguchiOhtaDepth); false)


	   fun checkSuzukiAotoToyama _ = 
	       (!useSuzukiAotoToyama)
	       andalso
	       if CrDirect.isConfluentWeakLeftLinearSystem isTerminating (inCps,outCps) rs
	       then (println nameSuzukiAotoToyama; true)
	       else (println (unknown nameSuzukiAotoToyama); false)

	   fun checkUchidaAotoToyama _ = 
	       (!useUchidaAotoToyama)
	       andalso
	       if CrDiagram.isConfluentQuasiLinearSystem isTerminating smtSolver tmpDir rs
	       then (println nameUchidaAotoToyama; true)
	       else (println (unknown nameUchidaAotoToyama); false)

	   fun checkUchidaAotoToyamaStrong _ = 
	       (!useUchidaAotoToyamaStrong)
	       andalso
	       if CrDiagram.isConfluentStronglyQuasiLinearSystem isTerminating smtSolver tmpDir rs
	       then (println nameUchidaAotoToyamaStrong; true)
	       else (println (unknown nameUchidaAotoToyamaStrong); false)

	   fun checkSakaiOgawa _ = 
	       (!useSakaiOgawa)
	       andalso
	       if Trs.areShallowRules rs
		  andalso L.all Term.equal inCps
		  andalso L.all Term.equal outCps
		  andalso not (Trs.isCollapsingTrs rs)
	       then (println nameSakaiOgawa; true)
	       else (println (unknown nameSakaiOgawa); false)

	   fun checkToyamaOyamaguchiWeightDecreasing _ = 
	       (!useToyamaOyamaguchiWeightDecreasing)
	       andalso
	       if Trs.areRightLinearRules rs
		  andalso L.all  (fn (l,r) =>
				     let val nlvL = Term.nonLinearVarListInTerm l
					 val vsR =  Term.varListInTerm r
				     in  L.all (fn x => not (LU.member' Var.equal x vsR)) nlvL
				     end) rs
		  andalso Cr.isWeightDecreasingJoinableTrs rs
	       then (println nameToyamaOyamaguchiWeightDecreasing; true)
	       else (println (unknown nameToyamaOyamaguchiWeightDecreasing); false)

	   fun checkNonConfluence _ = 
	       ((!NonCr.useNonJoinableByApproximation)
		orelse
		(!NonCr.useNonJoinableByTreeAutomata)
		orelse
		(!NonCr.useNonJoinableByInterpretationAndOrder))
	       andalso
	       let val _ = print "check Non-Confluence...\n"
	       in if NonCr.checkNonConfluence satSolver smtSolver tmpDir rs
		  then true
		  else (println (unknown nameNonConfluence); false)
	       end

	   fun checkBidirectionalCompletion _ = 
	       (!useBidirectionalCompletion)
	       andalso
	       let val _ = print "check by Reduction-Preserving Completion...\n"
		   val result = CrCompletion.bidirectionalCompletion 
				    (isTerminating false)
				    isRelativeTerminating
				    dSymSet rs
		   val _ = if result
			   then print "Reduction-Preserving Completion\n"
			   else print "unknown Reduction-Preserving Completion\n"
	       in result
	       end

	   fun checkConfluenceByOrderedRewriting _ =
	     (!useMartinNipkowOrderedRewriting)
	     andalso
	     let val _ = print "check by Ordered Rewriting...\n"
		 val result = CrOrd.confluenceByOrderedRewriting satSolver smtSolver tmpDir rs
		 val _ = if result
			 then print "Confluence by Ordered Rewriting\n"
			 else print "unknown Confluence by Ordered Rewriting\n"
	       in result
	       end

	   fun checkConfluenceConditionsForLinearSystem _ = 
	       if checkHuetToyamaOostrom ()
		  orelse
		  checkHuetStrongClosed ()
		  orelse 
		  checkSakaiOgawa ()
		  orelse
		  checkOyamaguchiOhta ()
		  orelse
		  checkToyamaParallel ()
		  orelse
		  checkOkuiSimultaneous ()
		  orelse
		  checkGomiOyamaguchiOhta ()
		  orelse
		  checkDecreasingDiagrams ()
	       then (print "Direct Methods"; report CR)
	       else if checkNonConfluence ()
	       then (print "Direct Methods"; report NotCR)
	       else if
		       checkHuetModulo ()
		       orelse
		       checkBidirectionalCompletion ()
		       orelse
		       checkConfluenceByOrderedRewriting ()
		   then (print "Direct Methods"; report CR)
		   else (print "Direct Methods"; report Unknown)

	   fun checkConfluenceConditionsForLeftLinearSystem _ = 
	       if checkHuetToyamaOostrom ()
		  orelse 
		  checkSakaiOgawa ()
		  orelse
		  checkOyamaguchiOhta ()
		  orelse
		  checkToyamaParallel ()
		  orelse
		  checkOkuiSimultaneous ()
		  orelse
		  checkGomiOyamaguchiOhta ()
		  orelse
		  checkDecreasingDiagrams ()
		  orelse 
		  checkUchidaAotoToyama ()
		  orelse 
		  checkUchidaAotoToyamaStrong ()
	       then (print "Direct Methods"; report CR)
	       else if checkNonConfluence ()
	       then (print "Direct Methods"; report NotCR)
	       else if checkHuetModulo ()
		       orelse
		       checkBidirectionalCompletion ()
		       orelse
		       checkConfluenceByOrderedRewriting ()
	       then (print "Direct Methods"; report CR)
	       else (print "Direct Methods"; report Unknown)

	   fun checkConfluenceConditionsForRightLinearSystem _ = 
	       if checkOhtaOyamaguchiToyama ()
		  orelse 
		  checkSakaiOgawa ()
		  orelse
		  checkGomiOyamaguchiOhta ()
		  orelse 
		  checkSuzukiAotoToyama ()
		  orelse 
		  checkUchidaAotoToyama ()
		  orelse 
		  checkUchidaAotoToyamaStrong ()
		  orelse
		  checkToyamaOyamaguchiWeightDecreasing ()
	       then (print "Direct Methods"; report CR)
	       else if checkNonConfluence ()
	       then (print "Direct Methods"; report NotCR)
	       else if checkBidirectionalCompletion ()
		       orelse
		       checkConfluenceByOrderedRewriting ()
	       then (print "Direct Methods"; report CR)
	       else (print "Direct Methods"; report Unknown)

	   fun checkConfluenceConditionsForNonLeftRightLinearSystem _ = 
	       if checkSakaiOgawa ()
		  orelse
		  checkGomiOyamaguchiOhta ()
		  orelse 
		  checkSuzukiAotoToyama ()
		  orelse 
		  checkUchidaAotoToyama ()
		  orelse 
		  checkUchidaAotoToyamaStrong ()
	       then (print "Direct Methods"; report CR)
	       else if checkNonConfluence ()
	       then (print "Direct Methods"; report NotCR)
	       else if checkBidirectionalCompletion ()
		       orelse
		       checkConfluenceByOrderedRewriting ()
	       then (print "Direct Methods"; report CR)
	       else (print "Direct Methods"; report Unknown)

	   fun checkConfluenceConditionsForNonTerminatingSystem _ = 
	       case (Trs.areLeftLinearRules rs,
		     Trs.areRightLinearRules rs) of
		   (true,true) => (print "Linear\n";
				   checkConfluenceConditionsForLinearSystem ())
		 | (true,false) => (print "Left-Linear, not Right-Linear\n";
				    checkConfluenceConditionsForLeftLinearSystem ())
		 | (false,true) => (print "not Left-Linear, Right-Linear\n";
				    checkConfluenceConditionsForRightLinearSystem ())
		 | (false,false) => (print "not Left-Linear, not Right-Linear\n";
				     checkConfluenceConditionsForNonLeftRightLinearSystem ())

       in
	   if (!useKnuthBendix)
	   then
	       if null inCps  (* overlay *)
	       then (print "Overlay, check Innermost Termination...\n";
		     if isTerminating true rs
		    then let val _ = print "Innermost Terminating (hence Terminating)"
			     val joinableCps = isInnerJoinableCps rs outCps
			     val _ = if joinableCps
				     then println ", WCR"
				     else println ", not WCR"
			     val _ = println nameKnuthBendix
			 in
			     if joinableCps then 
				 (print "Direct Methods"; report CR)
			     else 
				 (print "Direct Methods"; report NotCR)
			 end
		    else 
			(print "unknown Innermost Terminating\n";
			 println (unknown nameKnuthBendix);
			 checkConfluenceConditionsForNonTerminatingSystem rs))
	       else (print "not Overlay, check Termination...\n";
		     if isTerminating false rs
		     then let val _ = print "Terminating"
			      val joinableCps = isJoinableCps rs (inCps @ outCps)
			      val _ = if joinableCps
				      then println ", WCR"
				      else println ", not WCR"
			      val _ = println nameKnuthBendix
			  in
			     if joinableCps then 
				 (print "Direct Methods"; report CR)
			     else 
				 (print "Direct Methods"; report NotCR)
			  end
		     else
			 (println  "unknown/not Terminating";
			  println (unknown nameKnuthBendix);
			  checkConfluenceConditionsForNonTerminatingSystem rs))
	   else
	       checkConfluenceConditionsForNonTerminatingSystem rs
       end

    end (* of local *)


   fun applyDirectCheck directCheck rs (n,fs) = 
       if null rs
       then report CR
       else let val _ = print "Rewrite Rules:\n"
		val _ = print (Trs.prRules rs)
		val checkCC = directCheck rs
	    in case checkCC of
		   CR => CR
		 | NotCR => NotCR
		 | Unknown => applyDecomposition directCheck rs (n,fs)
	    end
   and applyDecomposition directCheck rs (n,fs) = 
       if (null fs)
       then 
	   Unknown
       else if not (null fs)
	  andalso n = length fs
       then (print "No further decomposition possible\n\n";
	     Unknown)
       else 
	   let 
	       val fs' = (tl fs) @ [hd fs]
	       fun direct rs = applyDirectCheck directCheck rs (1,fs')
	       fun decomp rs = applyDecomposition directCheck rs (n+1,fs')
	   in case (hd fs) of
(*		0 => CrDirect.tryDirectSumDecomposition rs (direct,decomp) *)
		0 => CrDirect.tryPersistentDecomposition rs (direct,decomp)
	      | 1 => CrLayer.tryLayerPreservingDecomposition rs (direct,decomp)
	      | 2 => CrCom.tryCommutativeDecomposition rs (direct,decomp)
	      | _ => (print "Error at applyDecomposition\n";
		      raise CrSolverError)
	   end

   exception Failure


  (* $B9gN.@-%=%k%P(B *)
   fun crSolver (opt0:Solver.options) (opt1:DpSolver.options) 
		(opt2:PoSolver.options) (opt3:options) rs =
      let
	   val trs = Trs.rulesToTrs rs
	   val _ = debug (fn _ => print "Rewrite Rules:\n") 
	   val _ = debug (fn _ => print (Trs.prRules (#Rules trs))) 

           val _ = runCertification := (#useCertifiableOutputOption opt3)
           val _ = NonCr.runCertification := (!runCertification)
           val _ = PoSolver.runCertification := (!runCertification)

	   val _ = cpout (fn _ => CU.start ())
	   val _ = cpout (fn _ => CU.output (CU.beginInput (CU.inputTrs (Trs.prTrsInProofTree rs))))
	   val _ = cpout (fn _ => CU.output (CU.version "2.1"))

           val satSolver = (#satSolver opt0)
           val smtSolver = (#smtSolver opt0)
           val snProver = (#terminationProver opt0)
           val relSnProver = (#relativeTerminationProver opt0)

           val tmpDir = (#tmpDir opt0)

	   val _ = if  (#VarCond trs) then ()
		   else (print "Variable condition is not satisfied\n";
			 raise Failure)

          fun isTerminating isOverlay rs = 
	      if snProver = ""
	      then 
		  if (#useDp opt1)
		  then if isOverlay 
		       then DpSolver.dpSolverInner satSolver tmpDir opt1 opt2 rs
		       else DpSolver.dpSolver satSolver tmpDir opt1 opt2 rs
		  else PoSolver.poSolver satSolver tmpDir opt2 (rs,[])
	      else 
		  if isOverlay 
		  then Solver.sinSolver snProver tmpDir rs
		  else Solver.snSolver snProver tmpDir rs

	  fun isRelativeTerminating (rs,es) =
	      if relSnProver = ""
	      then 
		  MatrixInterpretation.relsnSolver 
		      (opt0:Solver.options) 
		      (opt1:DpSolver.options) 
		      (opt2:PoSolver.options) (rs,es)
	      else
		  (* print "relative termination proof\n";
		   print "rs:\n";
		   print (Trs.prRules rs);
		   print "es:\n";
		   print (Trs.prRules es);
		   print relSnProver;
		   print "\n"; *)
		  Solver.relsnSolver relSnProver tmpDir (rs,es)

	  fun indexOf (l,r) = 
	      let fun indexOfSub [] _ = (print "indexOf failed\n";
					 raise CrSolverError)
		    | indexOfSub ((l',r')::rest) i
		      = if Term.equal (l,l')
			   andalso Term.equal (r,r')
			then i
			else indexOfSub rest (i+1)
	      in indexOfSub rs 0 
	      end
	   fun getRules ns = L.map (fn i => L.nth (rs,i)) ns
	   val rem = ref ILM.empty

           (* $B<+A3?t%j%9%H$GF~NO(BTRS$B$NItJ,=89g$rBP1~$5$;$k(B *)
	   fun lookup xs  = 
	       case ILM.find (!rem,xs) of
		   SOME ans=> (print "(cached)"; ans)
		 | NONE => let val ans = checkConfluenceConditions2 
					     satSolver smtSolver tmpDir
					     isTerminating 
					     isRelativeTerminating 
					     (getRules xs) 
			       val _ = ILM.insert (!rem, xs, ans)
			   in ans
			   end
	   fun directCheck rs = lookup (L.map indexOf rs)

	   val _ = if #useNonMinimalCommutative opt3
		   then CrCom.useNonMinimal := true
		   else CrCom.useNonMinimal := false

(* (*temporal *)      val _ = NonCr.satSolver := (#satSolver opt0) *)
(* (*temporal *)      val _ = NonCr.smtSolver := (#smtSolver opt0) *)
(* (*temporal *)      val _ = NonCr.snProver := (#terminationProver opt0) *)
(* (*temporal *)      val _ = NonCr.relSnProver := (#relativeTerminationProver opt0) *)
(* (*temporal *)      val _ = NonCr.tmpDir := (#tmpDir opt0) *)

(* (*temporal *)	  val crResult = if NonCr.checkNonConfluence rs *)
(* 			 then NotCR *)
(* 			 else Unknown *)

	  val crResult = 
          case (#useDirect opt3, #useCommutative opt3, #useLayer opt3) of
               (false,false,false)
               => applyDirectCheck directCheck rs (0,[]) 
             | (true,false,false)
               => applyDirectCheck directCheck rs (0,[0]) 
             | (false,true,false)
               => applyDirectCheck directCheck rs (0,[2]) 
             | (false,false,true)
               => applyDirectCheck directCheck rs (0,[1]) 
             | (true,true,false)
               => applyDirectCheck directCheck rs (0,[0,2]) 
             | (true,false,true)
               => applyDirectCheck directCheck rs (0,[0,1]) 
             | (false,true,true)
               => applyDirectCheck directCheck rs (0,[1,2]) 
             | (true,true,true)
               => applyDirectCheck directCheck rs (0,[0,1,2]) 

	  val _ = cpout (fn _ => CU.finish ())
      in
	  case crResult of 
           CR => (print "\nCombined result: CR\n"; CR)
         | NotCR => (print "\nCombined result: not CR\n"; NotCR)
         | Unknown => (print "\nCombined result: Can't judge\n"; Unknown)
      end
      handle Failure => (print "\nCombined result: Can't judge\n"; Unknown)


  fun checkTerminationOfUncondRules  (opt0:Solver.options) (opt1:DpSolver.options) 
				     (opt2:PoSolver.options) rules =
      let val trs = Trs.rulesToTrs rules
      in if not (#VarCond trs) 
	 then false
	 else let val _ = debug (fn _ => print "Check termination of unconditional rules:\n") 
		  val _ = debug (fn _ => print (Trs.prRules (#Rules trs))) 
		  val satSolver = (#satSolver opt0)
		  val smtSolver = (#smtSolver opt0)
		  val snProver = (#terminationProver opt0)
		  val relSnProver = (#relativeTerminationProver opt0)
		  val tmpDir = (#tmpDir opt0)

		  fun isTerminating isOverlay rs = 
		      if snProver = ""
		      then 
			  if (#useDp opt1)
			  then if isOverlay 
			       then DpSolver.dpSolverInner satSolver tmpDir opt1 opt2 rs
			       else DpSolver.dpSolver satSolver tmpDir opt1 opt2 rs
			  else PoSolver.poSolver satSolver tmpDir opt2 (rs,[])
		      else 
			  if isOverlay 
			  then Solver.sinSolver snProver tmpDir rs
			  else Solver.snSolver snProver tmpDir rs
	      in isTerminating (Cr.isOverlay rules) rules
	      end
      end



  exception SolvedWith of ConfluenceResult

  (* $B9gN.@-%=%k%P(B for oriented conditional TRSs *)
  fun condCrSolver (opt0:Solver.options) (opt1:DpSolver.options) 
		   (opt2:PoSolver.options) (opt3:options) crules =
      let 
	   val _ = print "Conditional Rewrite Rules:\n"
	   val _ = print (Ctrs.prRules crules)

	   fun isTerminatingUncondRules rs = checkTerminationOfUncondRules opt0 opt1 opt2 rs

	   val _ = println "Check whether all rules are type 3"
	   val _ = case L.find (not o Ctrs.ruleOfTypeIII) crules of
		       NONE =>println "OK"
		     | SOME lrc => (print "No. Rule of not type 3: ";
				    println (Ctrs.prRule lrc);
				    raise SolvedWith Unknown)

	   val _ = println "Check whether the input is deterministic"
	   val isDeterministic = 
	       case L.find (not o Ctrs.isDeterministicRule) crules
		of NONE => (println "OK"; true)
		 | SOME lrc => (print "No. non-deterministic rule: ";
				println (Ctrs.prRule lrc); 
				false)
				 
	   val unraveledRules = Ctrs.unraveling crules

	   val _ = if isDeterministic
		   then (println "Result of unraveling:";
			 print (Trs.prRules unraveledRules))
		   else ()

	   val _ = if isDeterministic
		   then println "Check whether U(R) is terminating"
		   else ()

	   val isUterminating = 
	       if isDeterministic
	       then if isTerminatingUncondRules unraveledRules
		    then (println "OK"; true)
		    else (println "failed to show termination";false)
	       else false

	   val _ = if isDeterministic
		   then println "Check whether the input is weakly left-linear"
		   else ()

	   val isWeaklyLeftLinear = if isDeterministic
				    then if Ctrs.isWeaklyLeftLinearRules crules
					 then (println "OK"; true)
					 else (println "not weakly left-linear"; false)
				    else false

	   val isUconfluent = if isDeterministic andalso isUterminating
			      then (println "Check whether U(R) is confluent")
			      else ()

	   val cps = if isDeterministic andalso isUterminating
		     then criticalPairs unraveledRules
		     else []


	   val isUconfluent = if isDeterministic andalso isUterminating
			      then if isJoinableCps unraveledRules cps
				   then (println "OK"; true)
				   else (println "U(R) is not confluent";false)
			      else false

	   val _ = if isDeterministic andalso isUterminating
		      andalso isWeaklyLeftLinear andalso isUconfluent
		   then (println "R is deterministic, weakly left-linear and U(R) is confluent";
			 raise SolvedWith CR)
		   else ()


	   val ccps = condCriticalPairs crules
	   val _ = println "Conditional critical pairs (CCPs):"
	   val _ = print (Ctrs.prEqs ccps)

	   val _ = println "Check whether the input is almost orthogonale"
	   val isAlmostOrthogonal = 
	       if  L.all (fn (l,r,c) => T.isLinearTerm l) crules
		   andalso L.all (fn (u,v,c) => T.equal (u,v)) ccps
	       then (println "OK"; true)
	       else (println "not almost orthogonal"; false)

	   val _ = if isAlmostOrthogonal
		   then println "Check whether the input is properly oriented"
		   else ()

	   val isProperlyOriented  = if isAlmostOrthogonal
				     then if Ctrs.isProperlyOrientedRules crules
					  then (println "OK"; true)
					  else (println "not properly oriented"; false)
				     else false

	   val _ = if isAlmostOrthogonal andalso isProperlyOriented
		   then println "Check whether the input is right-stable"
		   else ()

	   val isRightStable = if isAlmostOrthogonal andalso isProperlyOriented
			       then if Ctrs.isRightStable crules
				    then (println "OK"; true)
				    else (println "not right-stable"; false)
			       else false

	   val _ = if isAlmostOrthogonal andalso isProperlyOriented
		      andalso isRightStable
		   then (println "R is almost orthogonal, properly oriented, right stable";
			 raise SolvedWith CR)
		   else ()

	   val isWeaklyLeftLinear = if isDeterministic
				    then if Ctrs.isWeaklyLeftLinearRules crules
					 then (println "OK"; true)
					 else (println "not weakly left-linear"; false)
				    else false


	   val _ = if isDeterministic andalso isUterminating
		   then println "Check whether the input is absolutely irreducible"
		   else ()

	   val isAbsolutelyIrreducible = 
	       if isDeterministic andalso isUterminating
	       then if Ctrs.isAbsolutelyIrreducible crules
		    then (println "OK"; true)
		    else (println "not absolutely irreducible"; false)
	       else false



        (* some preparations for ccp checking *)
	   fun isConstructor f = L.all (fn (l,r,c) => not (Fun.equal (valOf (Term.funRootOfTerm l), f))) crules
	   val funs = let val (us,vs) = LP.unzip (LU.mapAppend (fn (l,r,c) => (l,r)::c) crules)
		      in FS.listItems (Term.funSetInTerms (us @ vs))
		      end
	   val cfuns = L.filter isConstructor funs

	   val lhsOfUncondRules = L.mapPartial (fn (l,r,c) => if null c then SOME l else NONE) crules
	   fun searchInstantiation (s,t,cond) = 
	       LU.mapAppend (fn (u,v) => 
				if VS.isSubset (T.varSetInTerm u, T.varSetInTerm s)
				then let val vs = VS.listItems (Term.varSetInTerm u)
					 val t = T.Fun (Fun.fromString "Dummy", lhsOfUncondRules, Sort.null)
					 val ts = T.argsOfTerm (Subst.renameTermDisjointFrom vs t)
				     in L.mapPartial (fn l => Subst.unify u l) ts
				     end
				else []) cond 


	   val _ = if isDeterministic andalso isUterminating
		      andalso isAbsolutelyIrreducible
		   then let val _ = println "Check whether all CCPs are joinable"
			    fun checkInfeasibleI vset cond = false
                                (***
				L.exists (fn (u,v) => not (Term.equal (u,v))
						      andalso not (Term.isVar v)
						      andalso VS.isSubset (Term.varSetInTerm u, vset)
						      andalso Rewrite.isNormalForm unraveledRules u) cond
                                 **)
			    fun checkInfeasibleII  [] = false
			      | checkInfeasibleII ((u,v)::rest) = 
				if Rewrite.isNormalForm unraveledRules v
				then L.exists (fn (u2,v2) => Term.equal (u,u2) 
							     andalso not (Term.equal (v,v2))
							     andalso Rewrite.isNormalForm unraveledRules v2) rest
				else false
			    fun isInfeasible (u,v,cond) = if checkInfeasibleI (Term.varSetInTerm u)  cond
						    then (debug (fn () => println "infeasible by irreducible lhs condition"); 
							  true)
						    else if (isUconfluent andalso checkInfeasibleII cond)
						    then (debug (fn () => println "infeasible by two irreducible rhs's with same lhs");
							 true)
						    else false

			    fun findNonJoinableCCP ccps =
				let val _ = case L.find (fn (u,v,c) => null c) ccps of
						SOME (u,v,c) => (print "A feasible but not joinable CCP: ";
								 println (Ctrs.prEq (u,v,c));
								 raise SolvedWith NotCR)
					      | NONE => ()

				    fun checkByUnifier (s,t,cond) = 
					let val (us,vs) = ListPair.unzip cond
					    fun mkDummyTerm ts = Term.Fun (Fun.fromString "Dummy", ts, Sort.null)
					in case Subst.unify (mkDummyTerm us) (mkDummyTerm vs) of
					       NONE => ()
					     | SOME sigma => let val s2 = Subst.applySubst sigma s
								 val t2 = Subst.applySubst sigma t
								 val s3 = Ctrs.linf crules s2
								 val t3 = Ctrs.linf crules t2
							     in if not (Term.equal (s3,t3))
								then (print "A non-joinable pair: ";
								      println (Term.toString s3 ^ " <-*- o -*-> " ^ Term.toString t3);
								      println ("instanciated and reduced from " ^ (Ctrs.prEq (s,t,cond)));
								      raise SolvedWith NotCR)
								else ()
							     end
					end

				    val _ = L.app checkByUnifier ccps 

				    fun checkByReduction (s,t,cond) = 
					let fun isReachable u v = if Term.equal (u,v)
							    then true
							    else case Ctrs.rewriteOneStep crules u of
								     SOME u2 => isReachable u2 v
								   | NONE => false
					in if L.all (fn (u,v) => isReachable u v) cond
					      andalso let val s2 = Ctrs.linf crules s
							  val t2 = Ctrs.linf crules t
						      in not (Term.equal (s2,t2))
						      end
					   then (print "A feasible but not joinable CCP: ";
						 println (Ctrs.prEq (s,t,cond));
						 raise SolvedWith NotCR)
					   else ()
					end

				    val _ = L.app checkByReduction ccps 

				    fun instantiateAndCheckByReduction (s,t,cond) = 
					L.app checkByReduction
					      (L.map (fn sigma => (Subst.applySubst sigma s,
								   Subst.applySubst sigma t,
								   L.map (fn (x,y) => (Subst.applySubst sigma x,
										       Subst.applySubst sigma y)) cond))
						     (searchInstantiation (s,t,cond)))

				    val _ = L.app instantiateAndCheckByReduction ccps 
				in ()
				end


			    fun maybeJoinableCcp (u,v,cond) = 
				if isInfeasible (u,v,cond)
				then let val _ = debug (fn () => println (Ctrs.prEq (u,v,cond) ^ " : infeasible"))
				     in true
				     end
				else let 
				    fun decompose (s,t) = case (Term.funRootOfTerm s, Term.funRootOfTerm t) of
							      (SOME f, SOME g) => if Fun.equal (f,g)
										     andalso LU.member' Fun.equal f cfuns
										  then LP.zip (Term.argsOfTerm s, Term.argsOfTerm t)
										  else [(s,t)]
							    | _ => [(s,t)]
				    fun decompIte ps = let val ps2 = LU.mapAppend decompose ps
						       in if L.length ps2 = L.length ps
							     andalso LP.all TermPair.equal (ps,ps2) 
							  then ps
							  else decompIte ps2
						       end
				    val cond2 = decompIte cond
				    val (ts1,ts2)  = ListPair.unzip ((u,v)::cond2)
				    val (cvPairs, joined) = Ctrs.skolemizeTerms (ts1 @ ts2)
				    val len = (L.length joined) div 2
				    val ((u',v')::cond')  =  ListPair.zip (L.take (joined, len), L.drop (joined, len))
				    val _ = debug (fn () => println ("check joinablity of " ^  Term.toString u' ^ " and " ^ Term.toString v'))
				    val condRules = L.map (fn (u,v) => (u,v,[])) cond'
				    val crules2 = crules @ condRules
				    val _ = debug (fn () => print (Ctrs.prRules crules2))
				    val u2' = Ctrs.linf crules2 u'
				    val v2' = Ctrs.linf crules2 v'
				    val u2 = Ctrs.unSkolemize cvPairs u2'
				    val v2 = Ctrs.unSkolemize cvPairs v2'
				    val _ = debug (fn () => println ("resp. nf: " ^  Term.toString u2 ^ " and " ^ Term.toString v2))
				in Term.equal (u2,v2)
				end
			    val njCcps = L.filter (not o maybeJoinableCcp) ccps
			in (* just checking joinability, no feasibility check *)
			    if null njCcps 
			    then (println "OK"; raise SolvedWith CR)
			    else let val _ = println "Some ccp may be feasible but not joinable"
				     val _ =  print (Ctrs.prEqs njCcps)
				 in findNonJoinableCCP njCcps
				 end
			end
		   else ()

	   val _ = NonCr.useNonJoinableByApproximation := false
	   val _ = NonCr.useNonJoinableByTreeAutomata := false
	   val _ = NonCr.useNonJoinableByInterpretationAndOrder := false


	   val _ = if isDeterministic andalso (not isUterminating)
		      andalso isWeaklyLeftLinear
		   then let val _ = println "Check U(R) is confluent"
			    val ans = crSolver opt0 opt1 opt2 opt3 unraveledRules
			in if ans = CR
			   then (println "U(R) is confluent";
				 println "R is deterministic, weakly left-linear and U(R) is confluent";
				 raise SolvedWith CR)
			   else println "failed to show confluence of U(R)"
			end
		   else ()

      in Unknown
      end
      handle SolvedWith ans => ans



  (* $B2D49@-%=%k%P(B *)
  fun comSolver (opt0:Solver.options) (opt1:DpSolver.options) 
                (opt2:PoSolver.options) (opt3:options) (rs1,rs2) =
      let
	  val faSet = Trs.funAritySetInRules (rs1@rs2)
	  val faList = FunIntSet.listItems faSet

	   val trs1 = Trs.rulesToTrs rs1
	   val _ = print "Rewrite Rules (1):\n"
	   val _ = print (Trs.prRules (#Rules trs1))

	   val trs2 = Trs.rulesToTrs rs2
	   val _ = print "Rewrite Rules (2):\n"
	   val _ = print (Trs.prRules (#Rules trs2))

           val _ = runCertification := (#useCertifiableOutputOption opt3)
           val _ = NonCr.runCertification := (!runCertification)
           val _ = PoSolver.runCertification := (!runCertification)

	   val _ = cpout (fn _ => CU.start ())
	   val _ = cpout (fn _ => CU.output (CU.beginInput (CU.beginCommutation
								[Trs.prFunArityListInProofTree faList,
								 Trs.prTrsInProofTree rs1,
								 Trs.prTrsInProofTree rs2])))
	   val _ = cpout (fn _ => CU.output (CU.version "2.1"))


           val satSolver = (#satSolver opt0)
           val smtSolver = (#smtSolver opt0)
           val snProver = (#terminationProver opt0)
           val relSnProver = (#relativeTerminationProver opt0)

           val tmpDir = (#tmpDir opt0)

	   val _ = if  (#VarCond trs1) then ()
		   else (print "Variable condition is not satisfied for TRS (1)\n";
			 raise Failure)

	   val _ = if  (#VarCond trs2) then ()
		   else (print "Variable condition is not satisfied for TRS (2)\n";
			 raise Failure)

	   fun checkComConditionYoshida rules1 rules2 =
	       let val cpInRules1Rules2 = Cr.insideCriticalPairs2 (rules1,rules2)
		   val cpRules2Rules1 = Cr.criticalPairs2 (rules2,rules1)
		   val _ = println "check CPs are development closed..."
		   fun checkIn (l,r) =
		       Term.equal (l,r) 
		       orelse let val _ = debug (fn _ => println (Trs.prEq (l,r) ^ ":"))
				  val dl2 = Rewrite.developOneStepReductSet rules2 l
				  val _ = debug (fn _ => println ("X = { s | l -o->{R2} s } = " ^
								  (prSetInOneLine Term.toString (TS.listItems dl2))))
				  val _ = debug (fn _ => print ("check r in X..."))
				  val ans = TS.member (dl2,r)
				  val _ = debug (fn _ => println (if ans then "yes" else "no"))
			      in ans
			      end
		   fun checkInOut (l,r) =
		       Term.equal (l,r) 
		       orelse let val dl1 = Rewrite.developOneStepReductSet rules1 l
				  val _ = debug (fn _ => println (Trs.prEq (l,r) ^ ":"))
				  val _ = debug (fn _ => println ("X1 = { s | l -o->{R1} s } = " ^
								  (prSetInOneLine Term.toString (TS.listItems dl1))))
				  val dr2 = Rewrite.developOneStepReductSet rules2 r
				  val _ = debug (fn _ => println ("Y1 = { s | r -o->{R2} s } = " ^
								  (prSetInOneLine Term.toString (TS.listItems dr2))))
				  val _ = debug (fn _ => print ("check X1 cap Y1 /= {}..."))
				  val ans1 = not (TS.isEmpty (TS.intersection (dl1,dr2)))
				  val _ = debug (fn _ => println (if ans1 then "yes" else "no"))
			      in ans1
				 orelse let val maxLen = !maxRewriteLenForDevelopmentClosedCandidates
					  val nr2 = Rewrite.manyStepsReductSet rules2 maxLen r
					  val _ = debug (fn _ => println ("Y2 = { s | r -(<=n)->{R2} s } = " ^
									  (prSetInOneLine Term.toString (TS.listItems nr2))))
					  val _ = debug (fn _ => print ("check X1 cap Y2 /= {}..."))
					  val ans2 = not (TS.isEmpty (TS.intersection (dl1,nr2)))
					  val _ = debug (fn _ => println (if ans2 then "yes" else "no"))
					in ans2
				      end
			      end

		   val _ = debug (fn _ => println ("check for CPin(R2,R1)"))
		   val ansI = L.all checkIn cpInRules1Rules2

		   val result = if ansI
				then let val _ = debug (fn _ => println ("check for CP(R1,R2)"))
				     in if L.all checkInOut cpRules2Rules1
					then (println "...succeeded"; true)
					else (println "...failed"; false)
				     end
				else (println "...failed"; false)
	       in result
	       end 

	   val areLeftLinear = Trs.areLeftLinearRules rs1
			       andalso Trs.areLeftLinearRules rs2
	   val _ = if (!useYoshidaAotoToyama)
		   then if areLeftLinear
			then println "both left-linear"
			else println "some are non-left-linear"
		   else ()

	   val comResult = if (!useYoshidaAotoToyama) andalso areLeftLinear
			   then let val _ = println "check commutation with R1 := (1), R2 := (2)"
                                   (* the order of devleopment closed criterion is switeched in Ceta *)
				in checkComConditionYoshida rs1 rs2
				   andalso 
		   		   (cpout (fn _ => (CU.output o CU.beginProof o CU.beginComProof
						    o CU.beginSwapTRSs o CU.beginComProof)
						       (fn _ => CU.encloseProofLeafBy "developmentClosed"
     					(Int.toString (!maxRewriteLenForDevelopmentClosedCandidates))));
				    true)
				end
				orelse
				let val _ = println "check commutation with R1 := (2), R2 := (1)"
				in checkComConditionYoshida rs2 rs1
	                           andalso
		   		   (cpout (fn _ => (CU.output o CU.beginProof o CU.beginComProof)
						       (fn _ => CU.encloseProofLeafBy "developmentClosed"
				    (Int.toString (!maxRewriteLenForDevelopmentClosedCandidates))));
				    true)
				end
			   else false


	   fun constructNonCommutatationCandidate (l,r) x (l',r') =
               (* x .. non-linear variable in l *)
	       let val xpos =  L.filter (fn p => Var.equal
						     (valOf (Term.varRootOfTerm (valOf (Term.subterm p l))), x))
					(Term.varPositionsInTerm l)
		   val sigma = VarMap.singleton (x,l')
		   val top = Subst.applySubst sigma l
		   val lhs = valOf (Term.replaceSubterm top (hd xpos) r')
		   val rhs = Subst.applySubst sigma r
	       in (lhs,top,rhs)
	       end

	   val maxReductSetSize = 30

	   fun isComWitness (u,v) (rules1,rules2) =
	       (* u, v shoul satisfies u <-R2- o -R1-> v *)
	       let val _ = println ("check candidate: <" ^ (Term.toString u)
				    ^ ", " ^ (Term.toString v) ^ ">")
		   (* val maybeLreducts = Rewrite.minimalReductSet rules2 maxReductSetSize u *) (* bug fix 2019/04/07 *)
		   val maybeLreducts = Rewrite.minimalReductSet rules1 maxReductSetSize u
	       in case maybeLreducts of
		      NONE => false
		   |  SOME LreductSet =>
		      let val _ = debug (fn _ => println ("X = { s | l -*->{R1} s } = " ^
							  (prSetInOneLine Term.toString (TS.listItems LreductSet))))
			 (* val maybeRreducts = Rewrite.minimalReductSet rules1 maxReductSetSize v *) (* bug fix 2019/04/07 *)
			  val maybeRreducts = Rewrite.minimalReductSet rules2 maxReductSetSize v
		      in case maybeRreducts of
			     NONE => false
			   | SOME RreductSet =>
			     let val _ = debug (fn _ => println ("Y = { s | r -*->{R2} s } = " ^
								 (prSetInOneLine Term.toString (TS.listItems RreductSet))))
				 val _ = debug (fn _ => print ("check X cap Y = {}..."))
				 val ans = TS.isEmpty (TS.intersection (LreductSet,RreductSet))
				 val _ = debug (fn _ => println (if ans then "yes" else "no"))
			     in ans
			     end
		      end
	       end

	   exception Found of Term.term * Term.term * Term.term
	   fun checkNonComWitnessForX (l,r) x (rules1,rules2) =
           (* x .. non-linear variable in l *)
	   let val _ = println ("check for non-linear variable: " ^ (Var.toString x))
	       val cands = L.map (fn lr' => constructNonCommutatationCandidate (l,r) x lr') rules2
	   in case L.find (fn (u,_,v) => isComWitness (u,v) (rules1,rules2)) cands of
		  SOME cand => raise (Found cand)
		| NONE => ()
				     
	   end
			  
	   fun checkNonComWitness (l,r) (rules1,rules2) =
	   (* l -> r is non-left-linear rule i rules1 *)
	       let val nlvars = VS.listItems (Term.nonLinearVarSetInTerm l)
		   val _ = println ("check for non-left-linear rule: " ^ (Trs.prRule (l,r)))
	       in L.app (fn x => checkNonComWitnessForX (l,r) x (rules1,rules2)) nlvars
	       end
					 
	   fun simpleNonComCheck isUnSwitched (rules1,rules2) =
	       let val nonLLrules1 = L.filter (not o Trs.isLeftLinearRule) rules1
		   val _ = L.app (fn (l,r) => checkNonComWitness (l,r) (rules1,rules2)) nonLLrules1
	       in false
	       end
	       handle Found (l,top,r) => (print "Counter example: ";
					  print ((Term.toString l) ^ " <-R2- ");
					  print ((Term.toString top) ^ " -R1-> ");
					  println (Term.toString r);
			 		  if (!runCertification) 
					  then (NonCr.outputDisproof
						    (NonCr.ComProblem (isUnSwitched, rules1, rules2))
						    7 (top,r,l) 
						    (fn _ => CU.encloseProofLeafBy "finitelyReachable" ""))
					  else ();
					  true)
		   
	   fun nonComCheckByCPs isUnSwitched (rules1,rules2) =
	       let val cpRules2Rules1 = Cr.criticalPeaks2 (rules2,rules1)
		   val _ = println "check counter example from CPs..."
	       in case L.find (fn (top,l,r) => isComWitness (l,r) (rules1,rules2)) cpRules2Rules1 of
		      SOME (top,l,r) => (print "Counter example: ";
					 print ((Term.toString l) ^ " <-R2- ");
					 print ((Term.toString top) ^ " -R1-> ");
					 println (Term.toString r);
					 if (!runCertification) 
					 then (NonCr.outputDisproof
						   (NonCr.ComProblem (isUnSwitched, rules1, rules2))
						   7 (top,r,l) 
						   (fn _ => CU.encloseProofLeafBy "finitelyReachable" ""))
					  else ();
					 true)
		    | NONE => (println "...failed"; false)
	       end


	   fun checkNonCom () =
	       let fun checkMain isUnSwitched (xs,ys) =
		       (* Given that <-*-R1 o -*->R2, 
			  isUnSwitched = true  ==> xs:=R1 and ys:=R2
			  isUnSwitched = false ==> xs:=R2 and ys:=R1 *)
		       simpleNonComCheck isUnSwitched (xs,ys)
		       orelse nonComCheckByCPs isUnSwitched (xs,ys)
		       orelse NonCr.checkNonCommutative satSolver smtSolver tmpDir isUnSwitched (xs,ys)
		   exception Success
	       in
		   (if !useNonCommutation
		   then let val _ = println "check non-commutation"
			    val _ = println "check with R1 := (1), R2 := (2)"
			    val test1 = checkMain true (rs1,rs2)
			    val _ = if test1 then raise Success else ()
			    val _ = println "check with R1 := (2), R2 := (1)"
			    val test2 = checkMain false (rs2,rs1)
			    val _ = if test2 then raise Success else ()
			in false
			end
		   else false)
		   handle Success => true
	       end

	   val nonComResult = if comResult
			      then false (* not trying actually *)
			      else checkNonCom ()

	   val _ = cpout (fn _ => CU.finish ())
			 
      in if comResult
	 then (print "\nresult: COM\n"; COM)
	 else if nonComResult
	 then (print "\nresult: not COM\n"; NotCOM)
	 else (print "\nresult: Can't judge\n"; UnknownCOM)
      end
      handle Failure => (print "\nresult: Can't judge\n"; UnknownCOM)


    end (* of local *)


 
end (* of structre *)
