(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/dp_solver.sml
 * description: termination solver by dependency pair framework
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature DP_SOLVER = 
sig
    val runProfile: bool ref
    val runDebug: bool ref
    val inTmpFileName: string ref
    val outTmpFileName: string ref

    type options = {
	 useDp:bool,     (* dependency pair $B$r;H$&$+$I$&$+(B *)
	 useRp:bool,     (* reduction pair $B$r;H$&$+$I$&$+(B *)
	 usePi:bool,     (* polynomial interpretation $B$r;H$&$+$I$&$+(B *)
	 useSc:bool,     (* subterm criteria $B$r;H$&$+$I$&$+(B *)
	 useUr:bool,     (* usable rule criteria $B$r;H$&$+$I$&$+(B *)
	 useIn:bool      (* innermost termination $B$X$N@Z$jBX$($r;H$&$+$I$&$+(B *)
    }
    val defaultOptions : options

    datatype strategy = SN | SIN  (* termination or inntermost termination *)

    type dpproblem  = (Term.term * Term.term) list    (* dependency pairs *)
		      * Trs.trs * strategy
		      
	val defaultReport: bool ref  (* answer for unknown terminating case *)

    val dependencyGraphProcessor: dpproblem -> int list -> int list list
    val subtermCriteriaProcessor: string -> string -> dpproblem 
				  -> int list -> int list list

    val dpSolverGeneral: string -> string
		  -> options
		  -> PoSolver.options
                  -> strategy
		  -> (Term.term * Term.term) list
                  -> bool

    val dpSolver: string -> string
		  -> options
		  -> PoSolver.options
		  -> (Term.term * Term.term) list
                  -> bool

    val dpSolverInner: string -> string
		  -> options
		  -> PoSolver.options
		  -> (Term.term * Term.term) list
                  -> bool

end

structure DpSolver : DP_SOLVER =
struct
   local
       structure A = Atom
       structure AT = AtomTable
       structure ILM = IntListMap2
       structure IS = IntSet
       structure IM = IntMap
       structure L = List
       structure LP = ListPair
       structure FM = FunMap
       structure FS = FunSet
       structure FIS = FunIntSet
       structure FIT = FunIntTable
       structure FIIT = FunIntIntTable
       structure FPT = FunPairTable
(*       structure CP = Compiler.Profile *)
       structure TPM = TermPairMap
       fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs
       fun member x ys = isSome (L.find (fn y => x = y) ys) 
       fun selectIndexes sub all = 
	   let fun select [] _ _ ans = SOME (rev ans)
		 | select (x::xs) [] _ _ = NONE
		 | select (x::xs) (y::ys) i ans = 
		   if x = y 
		   then select xs ys (i+1) (i::ans)
		   else select (x::xs) ys (i+1) ans
	   in  
	       select sub all 0 []
	   end
       local
	   fun logMain n k ans = if n <= k then ans
				 else logMain n (k * 2) (ans + 1)
       in
       fun log n = logMain n 2 1
       end
   in 

   datatype strategy = SN | SIN  (* termination or inntermost termination *)

    type dpproblem  = (Term.term * Term.term) list    (* dependency pairs *)
		      * Trs.trs * strategy

    type options = {
	 useDp:bool,     (* dependency pair $B$r;H$&$+$I$&$+(B *)
	 usePi:bool,     (* polynomial interpretation $B$r;H$&$+$I$&$+(B *)
	 useRp:bool,    (* reduction pair $B$r;H$&$+$I$&$+(B *)
	 useSc:bool,    (* subterm criteria $B$r;H$&$+$I$&$+(B *)
	 useUr:bool,     (* usable rule criteria $B$r;H$&$+$I$&$+(B *)
	 useIn:bool     (* innermost termination $B$X$N@Z$jBX$($r;H$&$+$I$&$+(B *)
    }

    val defaultOptions = {
	 useDp = true,
	 useRp  = true,
	 usePi = true,
	 useSc = true,
	 useUr = true,
	 useIn = false
    }

   exception DpSolverError
			    
   val runProfile = ref false : bool ref
   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()

   val inTmpFileName = ref ""
   val outTmpFileName = ref ""

   val minisatPath = ref ""
   val tmpDir = ref ""

   fun divideDpProblem (_,trs,strat) (dps,idxList) = 
       L.map (fn idx => (L.map (fn i=> L.nth (dps,i)) idx,trs,strat)) idxList

   fun dependencyGraphProcessor (dps,trs:Trs.trs,strat) idxes =
       if null idxes
       then []
       else
	   let 
	       val _ = debug (fn _ => print "\nDependency Graph Processor...\n")

	       val _ = debug (fn _ => print "Rewrite Rules:\n")
	       val _ = debug (fn _ => print (Trs.prRules (#Rules trs)))

	       val _ = debug (fn _ => print "Dependency Pairs:\n")
	       val current = L.map (fn i => L.nth (dps,i)) idxes
	       val _ = debug (fn _ => print (Trs.prRules current))

	       val dependencyGraph = if strat = SN
				     then Dp.dependencyGraph (#DSymSet trs) current
				     else Dp.innermostDependencyGraph (#DSymSet trs) (#Rules trs) current

	       val _ = debug (fn _ => if strat = SN
				      then print "Dependency Graph:\n"
				      else print "Innermost Dependency Graph:\n")

	       val _ = debug (fn _ => print (Graph.toString dependencyGraph))

	       val scc = Graph.scc dependencyGraph (* list of int set *)
	       val _ = debug (fn _ => print "scc: ")
	       val _ = debug (fn _ => L.app (print o IntSet.toString) scc)
	       val _ = debug (fn _ => print "\n")

	   in
	       L.map (fn ns => L.map (fn n=> L.nth (idxes,n)) (IntSet.listItems ns)) scc
	   end

   fun symbolCountingProcessor path dir (dps,trs:Trs.trs,strat) idxes =
       if null idxes
       then []
       else let
	       val _ = debug (fn _ => print "\nSymbol Counting Processor...\n")
	       exception NotApplicable
	       val _ = if L.all (fn (l,r) => (Term.isLinearTerm r) 
					      andalso
					      (Term.numberOfFSymsInTerm l) 
					      >= (Term.numberOfFSymsInTerm r))
				(#Rules trs)
		       then ()
		       else raise NotApplicable
	       val rems = L.mapPartial
			      (fn i => let val (l,r) = L.nth (dps,i)
					   val sizel = Term.numberOfFSymsInTerm l
					   val sizer = Term.numberOfFSymsInTerm r
(* 					   val _ = print ((Term.toString l) ^ "("  *)
(* 							  ^ (Int.toString sizel) ^ ")") *)
(* 					   val _ = print ((Term.toString r) ^ "("  *)
(* 							  ^ (Int.toString sizer) ^ ")") *)
(* 					   val _ = print "\n"  *)
				       in
					   case Int.compare (sizel,sizer) of
					       LESS => raise NotApplicable
					     | GREATER => if Term.isLinearTerm r
						       then NONE
						       else raise NotApplicable
					     | EQUAL => if Term.isLinearTerm r
							then SOME i
							else raise NotApplicable
				       end)
	   		      idxes

	       val result = (length rems) < (length idxes)
	       val _ = debug (fn _ => print ("Remained Dps:\n"
						  ^ (Trs.prRules 
							 (L.map (fn i=> L.nth (dps,i)) rems))))
	   in
	       if result
	       then
		   mapAppend (subtermCriteriaProcessor path dir (dps,trs,strat))
			     (dependencyGraphProcessor (dps,trs,strat) rems)
	       else [rems]
	   end
	    handle NotApplicable => (debug (fn _ => print "...not applicable\n"); [idxes])


   and subtermCriteriaProcessor path dir (dps,trs:Trs.trs,strat) idxes =
       if null idxes
       then []
       else
	   let 
	       val _ = debug (fn _ => print "\nSubterm Criteria Processor...\n")

	       val current = L.map (fn i => L.nth (dps,i)) idxes

	       val _ = debug (fn _ => print "Dependency Pairs:\n")
	       val _ = debug (fn _ => print (Trs.prRules current))

	       val (result,remIdxes) = Dp.applySubtermCriteria path dir current

	       val _ = debug (fn _ => print ("=>" ^ (if result then "T" else "F") ^ "\n"))
	       val _ = debug (fn _ => if result 
				      then print ("Remained Dps:\n"
						  ^ (Trs.prRules 
							 (L.map (fn i=> L.nth (current,i)) remIdxes)))
				      else ())

	       val rems = L.map (fn n=> L.nth (idxes,n)) remIdxes
	   in
	       if result 
	       then if null rems
		    then []
		    else mapAppend (subtermCriteriaProcessor path dir (dps,trs,strat))
				   (dependencyGraphProcessor (dps,trs,strat) rems)
					     
	       else [ rems ] 
(*	       else (symbolCountingProcessor path dir (dps,trs:Trs.trs,strat) rems) *)
	   end
	 

   exception YesTerminating
   exception NotTerminating
   exception UnknownTerminating
   val defaultReport = ref false  (* answer for unknown terminating case *)

   fun dpSolverGeneral path dir (opt1:options) (opt2:PoSolver.options) strat rules =
       let
	   val _ = minisatPath := path
	   val _ = tmpDir := dir

(* 	   val rules = if strat = SIN *)
(* 		       then  *)
(* 			   L.filter *)
(* 			   (fn (l,r) => L.all *)
(* 					    (fn (lr' as (l',r')) => *)
(* 						(Rewrite.isNormalForm  [lr'] l) *)
(* 						orelse  *)
(* 						((Term.equal (l,l')) andalso (Term.equal (r,r')))) *)
(* 					    givenRules) *)
(* 			   givenRules *)
(* 		       else givenRules  *)

	   val trs = Trs.rulesToTrs rules
	   val _ = debug (fn _ => print "Rewrite Rules:\n")
	   val _ = debug (fn _ => print (Trs.prRules (#Rules trs)))

	   val _ =  if  (#VarCond trs) 
		    then ()
		    else (* print "Variable condition not satisfied\n" *)
			raise NotTerminating

	   val _ = if ((strat = SN) 
		       andalso (L.exists (fn (l,r) => L.exists (fn u => isSome (Subst.match l u) )
							      (Term.subterms r))
					rules))
		      orelse
		      ((strat = SIN ) 
			andalso (L.exists (fn (l,r) => L.exists (fn u => (isSome (Subst.match l u) 
								     andalso (L.all (Rewrite.isNormalForm rules) (Term.properSubterms u))))
							   (Term.subterms r))
					  rules))
		   then raise NotTerminating
		   else ()
			
	   val dSymSet = (#DSymSet trs) 
 	   val _ = debug (fn _ => print ("Def Syms: " ^ PrintUtil.prSetInOneLine
							    Fun.toString (FS.listItems dSymSet) ^ "\n"))
	   val dependencyPairs = Dp.dependencyPairs dSymSet rules 
	   val _ = debug (fn _ => print "Dependency Pairs:\n")
	   val _ = debug (fn _ => print (Trs.prRules dependencyPairs))


	   val problems = let val _ = debug (fn _ =>
						if strat = SIN
						then print ("Locally Confluence Overlay System..."
							    ^ "Suffices to Prove Innermost Termination\n")
						else ())
			      val idxes = L.tabulate(L.length dependencyPairs,
						      fn i=>i)
			  in
			      if #useSc opt1
			      then
 				  mapAppend (subtermCriteriaProcessor path dir
								      (dependencyPairs,trs,strat))
 					    (dependencyGraphProcessor 
 						 (dependencyPairs,trs,strat) idxes)
			      else
				  dependencyGraphProcessor
				      (dependencyPairs,trs,strat) idxes
			  end

	   val _ = if null problems 
		   then raise YesTerminating
		   else if not (#useRp opt1) andalso not (#usePi opt1)
		   then raise UnknownTerminating
		   else ()

	   val dpsIndexesAll = IS.listItems (L.foldl (fn (ns,set)=>IS.addList (set,ns)) IS.empty problems)
	   val idxesList = L.map (fn sub => 
				     case selectIndexes sub dpsIndexesAll of
					 SOME ns => ns
				       | NONE => (print "dpSolver: selectIndexes\n";
						  raise DpSolverError))
				 problems
	   val dplist = L.map (fn i => L.nth (dependencyPairs, i)) dpsIndexesAll

	   val faMap  = Trs.funArityMapInRules (rules @ dplist)
	   val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	   val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	   val faList  = FM.listItemsi faMap
	   val fs  = L.map #1 faList
	   val lenFs  = L.length fs
	   val rowLen  = log lenFs
	   val lenrules = length rules
	   val lendplist = length dplist

	   (* $BMxMQ2DG=5,B'$r7W;;$9$k$?$a$N%F!<%V%k(B *)
	   val usableFM = if #useUr opt1
			  then Dp.getUsableFunMap trs
			  else FM.empty


	   local

	   (****************************************)
	   (*** Prepare Reduction Pair Processor ***)
	   (****************************************)

	   val _ = if #useRp opt1
		   then debug (fn _ => print "\nPreparing for Reduction Pair Processor...\n")
		   else ()

	   val symCount = ref 0
	   val encoding = if #useRp opt1
			  then PoSolver.mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
			  else PoSolver.empty_info

	   fun strictPrecCond () = if #useRp opt1
				then PoSolver.mkStrictPrecCond opt2 encoding 
				else Prop.True
	   fun lexCond () = if #useRp opt1
			 then PoSolver.mkLexCond opt2 encoding 
			 else Prop.True
	   fun afCond () = if #useRp opt1
			then PoSolver.mkAfCond opt2 encoding 
			else Prop.True

	   fun quasiRpoCond () = if #useRp opt1
			      then PoSolver.mkQuasiRpoCond opt2 encoding 
			      else Prop.True

          (* DP $B$d(B rules $B$KBP1~$9$k(B index $B$rMQ0U(B *)
	   val count = !symCount + 1
	   val dplistForGt = if #useRp opt1
			     then LP.zip (L.tabulate (lendplist,fn x => count + x), dplist)
			     else []
	   val dplistForEq = if #useRp opt1
			     then LP.zip (L.tabulate (lendplist,fn x => count + lendplist + x), dplist)
			     else []
	   val rslistForGe = if #useRp opt1
			     then LP.zip (L.tabulate (lenrules,fn x => count + 2*lendplist + x), rules)
			     else []
 	   val _ = symCount := count + lendplist*2  + lenrules
           (* +1 for condition prop *)

	   val dsymsCountMap = if #useRp opt1
			       then
				   FS.foldl (fn (f,fmap) => FM.insert(fmap,f,
								      (symCount := (!symCount) + 1;
								       !symCount)))
					    FM.empty dSymSet
			       else FM.empty

           (* $B@)LsO@M}<0$KBP1~$7$?L?BjJQ?t$rMQ0U$7$F!$(BIff ($BL?BjJQ?t!$@)LsO@M}<0(B) $B$rEPO?(B  *)
	   (* $B<B:]$N(B encoding $B$O8zN($N$?$a!$8e$GI,MW$K$J$C$?$H$-$K$d$k(B *)
	   val propMap0 = 
	       List.foldr
		   (fn ((i,lr),mp) =>
		       IM.insert (mp,i,
			       fn () 
				  => Prop.eqCnf' symCount 
						 (Prop.simplifyProp 
						      (Prop.Iff (Prop.Atom i, 
								 PoSolver.encodeOrderConstraint 
								     Order.GT symCount encoding 
								     opt2 lr)))))
		   IM.empty
		   dplistForGt
		   
	   val propMap1 = 
	       List.foldr
	           (fn ((i,lr),mp) =>
		       IM.insert (mp,i,
			       fn ()
				  => Prop.eqCnf' symCount 
						 (Prop.simplifyProp 
						      (Prop.Iff (Prop.Atom i, 
								 PoSolver.encodeOrderConstraint 
								     Order.EQ symCount encoding 
								     opt2 lr)))))
		   propMap0
		   dplistForEq

	   val propMap2 = 
	       List.foldr
	           (fn ((i,lr),mp) =>
		       IM.insert(mp,i,
			      fn () =>
				 Prop.eqCnf' symCount
					     (Prop.simplifyProp
						  (Prop.Iff (Prop.Atom i,
							     PoSolver.encodeOrderConstraint
								 Order.GE symCount encoding 
								 opt2 lr)))))
		   propMap1
		   rslistForGe

	   val propMap3 = 
	       IM.insert (propMap2,
			  count + lendplist*2 + lenrules,
	               fn () => Prop.eqCnf' symCount 
					    (Prop.simplifyProp (Prop.Conj 
								    [strictPrecCond (), 
								     lexCond (), 
								     afCond (),
								     quasiRpoCond () ])))


          (* Iff ($BL?BjJQ?t!$@)LsO@M}<0(B) $B$N(B encoding $B$O(B lookup $B$,=i$a$F$"$C$?;~$K$9$k(B *)

	   val propMapDone = ref IM.empty


	   fun lookupProp i = 
	       case IM.find (!propMapDone,i) of
		   SOME p => p
		 | NONE => (case IM.find (propMap3, i) of
				SOME f => let val q = f ()
					  in (propMapDone := IM.insert (!propMapDone,i,q); q)
					  end
			      | NONE => (print ("Failing lookupProp: " ^ (Int.toString i)
						^ "(base " ^ (Int.toString count) ^ ")\n"); 
					 raise DpSolverError))

(* 	   val _ = debug (fn _ => (L.tabulate (lendplist,fn i => lookupProp (count + i)); ())) *)
(*  	   val _ = debug (fn _ => (L.tabulate (lendplist,fn i => lookupProp (count + lendplist + i)); ())) *)
(* 	   val _ = debug (fn _ => (L.tabulate (lenrules,fn i => lookupProp (count + lendplist*2 + i)); ())) *)
(*  	   val _ = debug (fn _ => (lookupProp (count + lendplist*2 + lenrules); ())) *)

	   (* $B0z?t%U%#%k%?%j%s%0$r$7$?8e$KMxMQ2DG=$J5,B'$N@)Ls$,@.N)$9$k$3$H$r<($9L?Bj$r@8@.(B *)
	   (* ts: dp $B$N1&JU(B *)
(* 	   fun mkUsableRulesProp2 ts rsIdxs = *)
(* 	       let *)
(* 		   open Term; *)
(* 		   fun funRulesIdx fset =  *)
(* 		       L.mapPartial  *)
(* 			   (fn i => let val (l,r) = L.nth (rules,i) *)
(* 				    in case l of  *)
(* 					   Fun (f,_,_) => if FS.member (fset,f) *)
(* 							   then SOME (Prop.Atom (count + 2*lendplist + i)) *)
(* 							   else NONE *)
(* 					 | Var _ => NONE *)
(* 				    end) *)
(* 			   rsIdxs *)

(* 		   fun usables (Var _)  = Prop.True *)
(* 		     | usables (Fun (f,ts,_)) = *)
(* 		       let  *)
(* 			   val ps = L.tabulate  *)
(* 					(length ts,  *)
(* 				      fn i => if #useAf opt2 *)
(* 								  then Prop.Imp (PoSolver.encodePiStatus encoding (f,i+1), *)
(* 												 usables (L.nth (ts,i))) *)
(* 							  else *)
(* 								  usables (L.nth (ts,i))) *)
(* 		       in *)
(* 			   case FM.find (usableFM,f) of  *)
(* 			       SOME fset => Prop.Conj (L.@(funRulesIdx fset, ps)) *)
(* 			     | NONE => Prop.Conj ps *)
(* 		       end *)
(* 	       in  *)
(* 		   Prop.eqCnf' symCount (Prop.simplifyProp (Prop.Conj (L.map usables ts))) *)
(* 	       end *)

	   fun mkUsableRulesProp ts rsIdxs =
	       let
		   open Prop
		   open Term

		   fun useFunArg (f,i) = PoSolver.encodePiStatus encoding (f,i+1)

		   fun usableTerm (Var _)  = True
		     | usableTerm (Fun (f,ts,_)) =
		       Conj ((case FM.find (dsymsCountMap,f) of
				 SOME i => Atom i
			       | NONE => True)
			     :: (L.tabulate (length ts, 
				      fn i => if #useAf opt2 
					      then Imp (useFunArg (f,i), usableTerm (L.nth (ts,i)))
					      else
						  usableTerm (L.nth (ts,i)))))

		   val usableRule = 
		       Conj (L.mapPartial 
				 (fn i => 
				     let val (l,r) = L.nth (rules,i)
				     in case l of 
					    Fun (f,_,_) => 
					    (case FM.find (dsymsCountMap,f) of
						SOME j => SOME (Imp (Atom j,
								     Conj [Atom (count + 2*lendplist + i),
									   usableTerm r]))
    					      | NONE => NONE)
					  | Var _ => NONE
				     end)
				 rsIdxs)

		   val uprop = Prop.Conj (L.@ (L.map usableTerm ts, [usableRule]))

(* 		   val _ = print (Prop.printProp uprop)  *)

	       in 
		   Prop.eqCnf' symCount (Prop.simplifyProp uprop)
	       end


	   in

	   fun reductionPairProcessor idxes = 
	       if null idxes
	       then SOME []
	       else 
		   let 
		       val _ = debug (fn _ => print "\nReduction Pair Processor...\n")

		       val dps = L.map (fn i=> L.nth (dplist,i)) idxes
		       val _ = debug (fn _ => print "Dependency Pairs:\n")
		       val _ = debug (fn _ => print (Trs.prRules dps))

		       (* AF$B$r9MN8$7$J$$4J0WHG$N7W;;$GMxMQ2DG=5,B'$r@)8B$7$F$*$/(B *)
		       val rsIdxes = if #useUr opt1
								 then Dp.getUsableRules usableFM rules dps
							 else 
								 L.tabulate (lenrules,fn x=>x)
		       val usableRS = L.map (fn i => L.nth (rules,i)) rsIdxes
		       val _ = debug (fn _ => print "Rules:\n")
		       val _ = debug (fn _ => print (Trs.prRules usableRS))

		       val gtPropsDP = L.map (fn i => lookupProp (count + i))  idxes
		       val eqPropsDP = L.map (fn i => lookupProp (count + lendplist + i)) idxes
		       val gePropsRS = L.map (fn i => lookupProp (count + lendplist*2 + i)) rsIdxes
		       val condProp = lookupProp (count + lendplist*2 + lenrules)
		       val usableRulesProp = if #useUr opt1
					     then mkUsableRulesProp (L.map (fn (l,r)=>r) dps) rsIdxes
					     else
						 Prop.Conj (L.tabulate (lenrules,
									fn i => Prop.Atom (count + 2*lendplist + i)))
							   
		       val props = 
			   Prop.appendCnf 
			       (mapAppend 
				    (fn x => x)
				    [ gtPropsDP,
				      eqPropsDP,
				      gePropsRS,
				      [condProp],
				      [usableRulesProp],
 				      [Prop.Disj 
					   (L.map (fn i => Prop.Atom (count + i)) idxes)],
 				      L.map (fn i => Prop.Disj 
							 [Prop.Atom (count + i), 
							  Prop.Atom (count + lendplist + i)])
					    idxes ])
			   
		       val (res,resAr) = Solver.cnfSolver (!minisatPath) (!tmpDir) 
							  (Prop.Conj props,length props,!symCount)

		       val _ = debug (fn _ => if res
					      then PoSolver.printInfo resAr encoding opt2
					      else ())

(* 		       val _ = debug (fn _=> if res then *)
(* 						 PoSolver.printPiMap resAr encoding *)
(* 					     else ()) *)

(* 		       val _ = debug (fn _=> if res then *)
(* 						 PoSolver.printColMap resAr encoding *)
(* 					     else ()) *)

(* 		       val _ = debug (fn _=> if res then *)
(* 						 PoSolver.printWeight resAr encoding *)
(* 					     else ()) *)

		       fun printVal resAr i  = if i < Array.length resAr then
						   print ("value: " ^ (Int.toString i) ^ "=>"
							  ^ (if PoSolver.isAssignedByTrue resAr i
							     then "T\n" else "F\n"))
					       else ()

(* 		       val _ = let val p = PoSolver.lookupGtPrecTable encoding (Fun.fromString "s", *)
(* 								       Fun.fromString "double") *)
(* 			       in if PoSolver.evalProp resAr encoding p  *)
(* 				  then print "s > double holds\n" *)
(* 				  else print "s > double does not holds\n" *)
(* 			       end *)


                       (*  strict order $B$,$D$$$F$$$J$$(B dp $B$rH4$-$@$9(B *)
		       val idxes2 = if res
				     then
					 L.filter (fn i => not (PoSolver.isAssignedByTrue 
								    resAr (count + i)))  
						  idxes
				     else []

                       (*  ge $B$H$J$C$F$$$J$$(B rules $B$rH4$-$@$9(B *)
		       val rsIdxes2 = if res
				     then
					 L.filter (fn i => not (PoSolver.isAssignedByTrue 
								    resAr (count + 2*lendplist + i)))
						  rsIdxes
				     else []

		       val _ = debug (fn _ => 
					 if res
					 then 
					     let val _ = print "Obtained:\n"
						 val colSet = PoSolver.getColSet resAr encoding 
						 val piMap = PoSolver.getPiMap resAr encoding 
						 fun af t  = Dp.applyAfToTerm piMap colSet t
					     in
						 (L.app (fn i => 
							    let val (l,r) = L.nth (dplist,i)
							    in
								if member i idxes2
								then
								    (print (" " 
									   ^ (Term.toString l) ^ 
									   " :(ge): "
									   ^ (Term.toString r) ^ "\n");
								    print ("  by " 
									   ^ (Term.toString (af l)) ^ 
									   " :(ge): "
									   ^ (Term.toString (af r)) ^ "\n"))
								else
								    (print (" " 
									   ^ (Term.toString l) ^ 
									   " :(gt): "
									   ^ (Term.toString r) ^ "\n");
								    print ("  by " 
									   ^ (Term.toString (af l)) ^ 
									   " :(gt): "
									   ^ (Term.toString (af r)) ^ "\n"))
							    end)
							idxes;
						  L.app (fn i => 
							    let val (l,r) = L.nth (rules,i)
							    in
								if member i rsIdxes2
								then
								    print (" " 
									   ^ (Term.toString l) 
									   ^ " :(not ge): "
									   ^ (Term.toString r) ^ "\n")
								else
								    (print (" " 
									   ^ (Term.toString l) 
									   ^ " :(ge): "
									   ^ (Term.toString r) ^ "\n");
								    print ("  by " 
									   ^ (Term.toString (af l)) ^ 
									   " :(ge): "
									   ^ (Term.toString (af r)) ^ "\n"))
							    end)
							rsIdxes)
					     end
					 else ())
		   in
		       if res 
		       then SOME idxes2
		       else NONE
		   end

	   end (* of local *)



	   (***************************************************)
	   (*** Prepare Polynomial Interpretation Processor ***)
	   (***************************************************)

	   local

	   val _ = if #usePi opt1
		   then debug (fn _ => print "\nPreparing for Polynomial Interpretation Processor...\n")
		   else ()

	   val symCount2 = ref 0
	   val rho = if #usePi opt1
		     then Poly.makeInitialMetaEnv symCount2 faList
		     else FM.empty

(* 	   val _ = FM.appi (fn (f,pp) => print ("[" ^ (Fun.toString f) ^ "] =" *)
(*  						^ (Poly.mToString pp) ^ "\n")) rho *)

          (* DP $B$d(B rules $B$KBP1~$9$k(B index $B$rMQ0U(B *)
	   val count2 = !symCount2 + 1
(* 	   val _ = print ("count2 :" ^ (Int.toString count2) ^ "\n"); *)
(* 	   val _ = print ("lendplist :" ^ (Int.toString lendplist) ^ "\n"); *)
(* 	   val _ = print ("lenrules :" ^ (Int.toString lenrules) ^ "\n"); *)
	   val polyDplistForGt = if #usePi opt1
				 then LP.zip (L.tabulate (lendplist,fn x => count2 + x), dplist)
				 else []
	   val polyDplistForGe = if #usePi opt1
				 then LP.zip (L.tabulate (lendplist,fn x => count2 + lendplist + x), dplist)
				 else []

	   val polyRslistForGe = if #usePi opt1
				 then LP.zip (L.tabulate (lenrules,fn x => count2 + 2*lendplist + x), rules)
				 else []
 	   val _ = symCount2 := count2 + lendplist*2  + lenrules - 1

	   val polyDsymsCountMap = if #usePi opt1
				   then
				       FS.foldl (fn (f,fmap) => FM.insert(fmap,f,
									  (symCount2 := (!symCount2) + 1;
									   !symCount2)))
						FM.empty dSymSet
				   else FM.empty


           (* $B@)LsO@M}<0$KBP1~$7$?L?BjJQ?t$rMQ0U$7$F!$(BIff ($BL?BjJQ?t!$@)LsO@M}<0(B) $B$rEPO?(B  *)
	   (* $B<B:]$N(B encoding $B$O8zN($N$?$a!$8e$GI,MW$K$J$C$?$H$-$K$d$k(B *)
	   val polyPropMap0 = 
	       List.foldr
		   (fn ((i,lr),mp) =>
		       IM.insert (mp,i,
			       fn () 
				  => Prop.eqCnf' symCount2
						 (Prop.simplifyProp 
						      (Prop.Iff (Prop.Atom i, 
								 Prop.Conj (L.map Poly.polyToProp 
										  (Poly.makeGreaterThanConstraints rho lr)))))))								 
		   IM.empty
		   polyDplistForGt
		   
	   val polyPropMap1 = 
	       List.foldr
	           (fn ((i,lr),mp) =>
		       IM.insert (mp,i,
			       fn ()
				  => Prop.eqCnf' symCount2
						 (Prop.simplifyProp 
						      (Prop.Iff (Prop.Atom i, 
								 Prop.Conj (L.map Poly.polyToProp 
										  (Poly.makeGreaterOrEqualConstraints rho lr)))))))
		   polyPropMap0
		   polyDplistForGe

	   val polyPropMap2 = 
	       List.foldr
	           (fn ((i,lr),mp) =>
		       IM.insert(mp,i,
			      fn () =>
				 Prop.eqCnf' symCount2
					     (Prop.simplifyProp
						  (Prop.Iff (Prop.Atom i,
							     Prop.Conj (L.map Poly.polyToProp 
									      (Poly.makeGreaterOrEqualConstraints rho lr)))))))
		   polyPropMap1
		   polyRslistForGe

          (* Iff ($BL?BjJQ?t!$@)LsO@M}<0(B) $B$N(B encoding $B$O(B lookup $B$,=i$a$F$"$C$?;~$K$9$k(B *)
	   val polyPropMapDone = ref IM.empty
	   fun polyLookupProp i = 
	       case IM.find (!polyPropMapDone,i) of
		   SOME p => p
		 | NONE => (case IM.find (polyPropMap2, i) of
				SOME f => let (* val _ = print ("Start encoding for var " 
							     ^ (Int.toString i) ^ "\n") *)
					      val q = f ()
					  in (polyPropMapDone := IM.insert (!polyPropMapDone,i,q); q)
					  end
			      | NONE => (print ("Failing polyLookupProp: " ^ (Int.toString i)
						^ "(base " ^ (Int.toString count2) ^ ")\n"); 
					 raise DpSolverError))

           (* f $B$N2r<a$N(B j $BHVL\$N0z?t(B(i>=1)$B$N78?t(B a_i $B$N(B i *)
           (*  { [x_j] |-> { [a_l] |-> 1 },  [x_1, x_j] |-> { [a_k] |-> 1 } }
                     ==> [l,k] 
           *)
	   fun usablePropVars (f,i) = 
	       case FM.find (rho,f) of
		   SOME pp => let val ys = L.filter (fn (xs,_) => member i xs) (ILM.listItemsi pp)
			      in mapAppend (fn (_,p) => hd (ILM.listKeys p))  ys
			      end
		 | NONE => (print "usablePropVar\n"; raise DpSolverError)


	   (* $B0z?t%U%#%k%?%j%s%0$r$7$?8e$KMxMQ2DG=$J5,B'$N@)Ls$,@.N)$9$k$3$H$r<($9L?Bj$r@8@.(B *)
	   (* ts: dp $B$N1&JU(B rsIdxs 0 ... *)
	   fun propMkUsableRulesProp ts rsIdxs =
	       let
		   open Prop
		   open Term

		   fun useFunArg (f,i) = Disj (L.map (fn j=> Atom j) (usablePropVars (f,i+1)))

		   fun usableTerm (Var _)  = True
		     | usableTerm (Fun (f,ts,_)) =
		       Conj ((case FM.find (polyDsymsCountMap,f) of
				 SOME i => Atom i
			       | NONE => True)
			     :: (L.tabulate (length ts, 
				      fn i => if #useAf opt2 
					      then Imp (useFunArg (f,i), usableTerm (L.nth (ts,i)))
					      else
						  usableTerm (L.nth (ts,i)))))

		   val usableRule = 
		       Conj (L.mapPartial 
				 (fn i => 
				     let val (l,r) = L.nth (rules,i)
				     in case l of 
					    Fun (f,_,_) => 
					    (case FM.find (polyDsymsCountMap,f) of
						SOME j => SOME (Imp (Atom j,
								     Conj [Atom (count2 + 2*lendplist + i),
									   usableTerm r]))
    					      | NONE => NONE)
					  | Var _ => NONE
				     end)
				 rsIdxs)

		   val uprop = Prop.Conj (L.@ (L.map usableTerm ts, [usableRule]))

(* 		   val _ = print (Prop.printProp uprop)  *)



	       in 
		   Prop.eqCnf' symCount2 (Prop.simplifyProp uprop)
	       end

	   in

	   fun polynomialInterpretationProcessor idxes =
	       if null idxes
	       then SOME []
	       else
		   let
		       val _ = debug (fn _ => print "\nPolynomial Interpretation Processor...\n")

		       val dps = L.map (fn i=> L.nth (dplist,i)) idxes
		       val _ = debug (fn _ => print "Dependency Pairs:\n")
		       val _ = debug (fn _ => print (Trs.prRules dps))

		       (* AF$B$r9MN8$7$J$$4J0WHG$N7W;;$GMxMQ2DG=5,B'$r@)8B$7$F$*$/(B *)
		       val rsIdxes = if #useUr opt1
				     then Dp.getUsableRules usableFM rules dps
				     else
					 L.tabulate (lenrules,fn x=>x)
		       val usableRS = L.map (fn i => L.nth (rules,i)) rsIdxes
		       val _ = debug (fn _ => print "Rules:\n")
		       val _ = debug (fn _ => print (Trs.prRules usableRS))

		       val gtPropsDP = L.map (fn i => polyLookupProp (count2 + i))  idxes
		       val gePropsDP = L.map (fn i => polyLookupProp (count2 + lendplist + i)) idxes
		       val gePropsRS = L.map (fn i => polyLookupProp (count2 + lendplist*2 + i)) rsIdxes

(*  		       val _ = print "RHSs of DP:\n"  *)
(*  		       val _ = L.app (fn (l,r)=> PrintUtil.println (Term.toString r)) dps  *)

		       val usableRulesProp =
			   if #useUr opt1
			   then propMkUsableRulesProp (L.map (fn (l,r)=>r) dps) rsIdxes
			   else
			       Prop.Conj (L.tabulate (lenrules,
						   fn i => Prop.Atom (count2 + 2*lendplist + i)))


		       val props =
			   Prop.appendCnf
			       (mapAppend
				    (fn x => x)
				    [ gtPropsDP,
				      gePropsDP,
				      gePropsRS,
 				      [usableRulesProp], 
 				      [Prop.Disj (L.map (fn i => Prop.Atom (count2 + i)) idxes)],
 				      L.map (fn i => Prop.Atom (count2 + lendplist + i)) idxes
			       ])
			   
		       val (res,resAr) = Solver.cnfSolver (!minisatPath) (!tmpDir)
							  (Prop.Conj props,length props,!symCount2)

		       fun assign i = if (PoSolver.isAssignedByTrue resAr i) then 1 else 0
		       val _ = debug (fn _ => if res
					      then
						  L.app (fn (f,_) =>
							    case FM.find (rho,f) of
								SOME pp => PrintUtil.println ( "[" ^ (Fun.toString f)  ^ "]"
											       ^ ":= "
											       ^ (Poly.toString (Poly.evalMetaPoly assign pp)))
							      | NONE => raise DpSolverError)
							faList
					      else ())

		       fun printVal resAr i  = if i < Array.length resAr
					       then
						   print ("value: " ^ (Int.toString i) ^ "=>"
							  ^ (if PoSolver.isAssignedByTrue resAr i
							     then "T\n" else "F\n"))
					       else ()


                       (*  strict order $B$,$D$$$F$$$J$$(B dp $B$rH4$-$@$9(B *)
		       val idxes2 = if res
				    then
					L.filter (fn i => not (PoSolver.isAssignedByTrue
								   resAr (count2 + i)))
						 idxes
				    else []

                       (*  ge $B$H$J$C$F$$$J$$(B rules $B$rH4$-$@$9(B *)
		       val rsIdxes2 = if res
				     then
					 L.filter (fn i => not (PoSolver.isAssignedByTrue
								    resAr (count2 + 2*lendplist + i)))
						  rsIdxes
				     else []

		       val _ = debug (fn _ =>
					 if res
					 then
					     let val _ = print "Obtained:\n"
					     in
						 (L.app (fn i =>
							    let val (l,r) = L.nth (dplist,i)
							    in
								if member i idxes2
								then
								    print (" "
									   ^ (Term.toString l) ^
									   " :(ge): "
									   ^ (Term.toString r) ^ "\n")
								else
								    print (" "
									   ^ (Term.toString l) ^
									   " :(gt): "
									   ^ (Term.toString r) ^ "\n")
							    end)
							idxes;
						  L.app (fn i =>
							    let val (l,r) = L.nth (rules,i)
							    in
								if member i rsIdxes2
								then
								    print (" "
									   ^ (Term.toString l)
									   ^ " :(not ge): "
									   ^ (Term.toString r) ^ "\n")
								else
								    print (" "
									   ^ (Term.toString l)
									   ^ " :(ge): "
									   ^ (Term.toString r) ^ "\n")
							    end)
							rsIdxes)
					     end
					 else ())
		   in
		       if res
		       then SOME idxes2
		       else NONE
		   end
		   
	   end (* of local *)

	   (************)
	   (*** Main ***)
	   (************)

	   fun rpStep [] = raise YesTerminating
	     | rpStep (p::ps) = 
	       case reductionPairProcessor p of
		   SOME ys => if null ys
			      then step ps
			      else step 
				       ((if #useSc opt1
					 then
 					     mapAppend (subtermCriteriaProcessor path dir
										 (dplist,trs,strat))
 						       (dependencyGraphProcessor 
							    (dplist,trs,strat) ys)
 					 else
					     dependencyGraphProcessor (dplist,trs,strat) ys)
					@ ps)
		 | NONE => raise UnknownTerminating
	   and piStep [] = raise YesTerminating
		 | piStep (p::ps) = 
		   case polynomialInterpretationProcessor p of
		       SOME ys => if null ys
				  then step ps
				  else step 
					   ((if #useSc opt1
					     then
 						 mapAppend (subtermCriteriaProcessor path dir
										     (dplist,trs,strat))
 							   (dependencyGraphProcessor 
								(dplist,trs,strat) ys)
 					     else
						 dependencyGraphProcessor (dplist,trs,strat) ys)
					    @ ps)
		     | NONE => if #useRp opt1 
			       then rpStep (p::ps)
			       else raise UnknownTerminating
	   and step [] = raise YesTerminating
	     | step (p::ps) =
		   if #usePi opt1 
		   then piStep (p::ps)
		   else if #useRp opt1 
		   then rpStep (p::ps)
		   else raise UnknownTerminating
       in
	   step idxesList
       end
       handle YesTerminating => true
			| UnknownTerminating => false
			| NotTerminating => (!defaultReport)



   fun dpSolver path dir (opt1:options) (opt2:PoSolver.options) rules =
       let
	   val strat = if #useIn opt1
			  andalso Cr.isOverlay rules
			  andalso Cr.isLocalConfluentForNonTerminatingRules rules
		       then SIN
		       else SN
       in
	   dpSolverGeneral path dir opt1 opt2 strat rules
       end

   fun dpSolverInner path dir (opt1:options) (opt2:PoSolver.options)  rules =
       dpSolverGeneral path dir opt1 opt2 SIN rules


   end (* of local *)
end (* of structre *)


