(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/kbo.sml
 * description: check of Knuth-Bendix ordering
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature KBO = 
sig

    type prec  = Term.fun_key list
    val comparePrec: prec -> Term.fun_key * Term.fun_key -> order

    type weight = int FunMap.map
    val compareWeight: weight -> Term.fun_key * Term.fun_key -> order

    val hasMoreVarOcc: Term.term -> Term.term -> bool

    (* KBO$B$N=E$_$dM%0L=g=x$,E,Ev$+$I$&$+$NH=Dj(B *)
    val isValidKboSpec: int 
			-> int FunMap.map
			-> (Term.fun_key * Term.fun_key -> order)
			-> Term.decl list
			-> bool

   (* $BM?$($i$l$?(B weight, precedence, lex status $B$N85$G$N=g=x%A%'%C%/(B *)
    val kboCheck: int 
		  -> int FunMap.map
		  -> (Term.fun_key * Term.fun_key -> order)
		  -> (Term.fun_key -> Term.term list -> Term.term list)
		  -> Term.term * Term.term
		  -> bool

end;

structure Kbo : KBO =
  struct

  local

  open Term
  open Trs

  structure S = Sort
  structure VS = VarSet
  structure VM = VarMap
  structure A = Atom
  structure FS = FunSet
  structure FM = FunMap
  structure L = List
  structure LP = ListPair

  exception KboError

  in 

(************************************************************************************)

  type prec  = Term.fun_key list (* $BBg$-$$=g$N%j%9%H(B *)

  fun comparePrec prec (f,g) = 
      if Fun.equal (f,g)
      then EQUAL
      else case (L.find (fn x => (Fun.equal (f,x)) orelse (Fun.equal (g,x))) prec)
	    of SOME y => if Fun.equal (f,y) then GREATER else LESS
	     | NONE => (print "comparePrec: function without prec.\n";
			raise KboError)

(************************************************************************************)

  type weight  = int FunMap.map  (* $B=E$_$E$1(B *)

  fun compareWeight wght (f,g) = 
      case (FunMap.find (wght,f),  FunMap.find (wght,g)) of
	  (SOME i, SOME j) => Int.compare (i,j)
	| _ => (print "compareWeight: function without weight.\n";
		raise KboError)

(************************************************************************************)

 (* r $B$NJQ?t%j%9%H(B $B"<(B l $B$NJQ?t%j%9%H(B *)

  fun hasMoreVarOcc l r  = 
      let val lvars = Term.varListInTerm l

	  exception ShortOfMember;

          fun deleteMember [] _ _ = raise ShortOfMember
	    | deleteMember (x::xs) y rem  = 
	      if Var.equal (x,y)
	      then SOME (xs @ rem)
	      else deleteMember xs y (x::rem)

          fun subtract xs (Var (x,_)) = deleteMember xs x []
	    | subtract xs (Fun (_,ts,_)) = subtractList xs ts
          and subtractList xs [] = SOME xs
	    | subtractList xs (t::ts) =
	      case subtract xs t of
		  SOME ys => subtractList ys ts
		| NONE => raise ShortOfMember

      in 
	  (subtract lvars r; true)
	  handle ShortOfMember => false
      end

  (* KBO$B$N=E$_$dM%0L=g=x$,E,Ev$+$I$&$+$NH=Dj(B *)

  fun isValidKboSpec varWeight weight compare (decls:Term.decl list) = 
      let fun weightOfFun f  = 
 	      case FM.find (weight, f) of
		  SOME w => w
		| NONE => ("weight is not defined for " ^ (Fun.toString f) ^ "\n"; 
			   raise KboError)
	  fun check (f, S.Base ty) = weightOfFun f >= varWeight
	    | check (f, S.Proc (types,_)) = 
	      weightOfFun f >= 0
	      andalso
	      (if (length types = 1 andalso weightOfFun f = 0)
           then L.all (fn x => compare (f, #sym x) <> LESS) decls
           else true)
      in
	  ((varWeight > 0) 
	  andalso (L.all (fn x => check (#sym x, #sort x))
			 (decls:Term.decl list)))
      handle KboError=>false
      end


(************************************************************************************)
(* kboCheck:  forall l -> r \in R. l >kbo r $B$+H=Dj(B                                  *)
(************************************************************************************)


  fun kboCheck varWeight weight compare lex (l,r) = 
      let fun weightOfFun f  = 
 	      case FM.find (weight, f) of
		  SOME w => w
		| NONE => ("weight is not defined for " ^ (Fun.toString f) ^ "\n"; 
			   raise KboError)

	  fun weightOfTerm (Var _)  =  varWeight
	    | weightOfTerm (Fun (f,ts,_))  =  
	      (weightOfFun f) + weightOfTermList ts  
	  and weightOfTermList [] = 0
	    | weightOfTermList (t::ts) = 
	      (weightOfTerm t) + (weightOfTermList ts)

	  fun grterKbo (l,r) =
	      let 
		  fun conditionA (Var _, _) = false
		    | conditionA (_, Fun _) = false
		    | conditionA (Fun (f,ts,_), s) = 
		      (weightOfFun f) = 0
		      andalso length ts = 1
		      andalso (Term.equal (hd ts, s) orelse conditionA (hd ts, s))
	      in
		  hasMoreVarOcc l r andalso
		  (let val wl = weightOfTerm l
		       val wr = weightOfTerm r
		   in
		       wl > wr orelse
		       (wl = wr andalso
			(conditionA (l,r)
			 orelse (case (l,r) of
				     (Fun (f,ss,_), Fun (g,ts,_)) 
				     => compare (f,g) = GREATER
					orelse (Fun.equal (f,g) 
						andalso grterKboLex (lex f ss) (lex f ts))
				   | (_, _) => false)))
		   end)
	      end
	  and grterKboLex [] [] = false
	    | grterKboLex [] (s0::ss) = false
	    | grterKboLex (t0::ts) [] = true
	    | grterKboLex (t0::ts) (s0::ss) = 
	      if Term.equal (t0,s0) 
	      then grterKboLex ts ss
	      else grterKbo (t0,s0) 
      in
	  grterKbo (l,r)
      end

  end (* of local *)
				     
  end (* of structre *)


