(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/matrix.sml
 * description: ingredients for matrix interpretation
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature YICES_INPUT = 
sig
   val runDebug: bool ref
   val prAssert: string -> string
   val prAssertPlus: string -> string
   val prRetract: int -> string

   val prTrue: string
   val prFalse: string
   val prNot: string -> string
   val prAnd: string list -> string
   val prOr: string list -> string
   val prXor: string list -> string
   val prIfThenElse: string -> string -> string -> string
   val prMoreThanTwo: string list -> string
   val prAtMostOne: string list -> string
   val prExactlyOne: string list -> string
   val prExactlyOneIsFalse: string list -> string
   val prExactlyTwo: string list -> string

   val prGe: string * string -> string
   val prGt: string * string -> string
   val prEq: string * string -> string
   val prNeq: string * string -> string
   val prLe: string * string -> string
   val prLt: string * string -> string

   val prZero: string
   val prPlus: string list -> string

   val prDefInt: string -> string
   val prDefBool: string -> string
   val readAssigns: TextIO.instream -> string StringMap.map
end

structure YicesInput : YICES_INPUT = 
struct
   local 
       structure L = List
       structure LU = ListUtil
      (* { {x,y} | x,y \in xs, x \neq y \} *)
       fun pairs xs = 
	   let fun trig [] [] = []
		 | trig (x::xs) (y::ys) = 
		   (L.map (fn y'=> (x,y')) ys) @ (trig xs ys)
	   in trig xs xs
	   end
       open PrintUtil
   in
   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   exception YicesError
   fun prAssert s = "(assert " ^ s ^ ")\n"
   fun prAssertPlus s = let val _ = debug (fn _ => print "assert+\n")
			in  "(assert+ " ^ s ^ ")\n" end
   fun prRetract n = let val _ = debug (fn _ => print ("retract " ^ (Int.toString n) ^ "\n"))
		     in  "(retract " ^ (Int.toString n) ^ ")\n" end

   val prTrue = "true"
   val prFalse = "false"
   fun prNot s = "(not " ^ s ^ ")"
   fun prAnd [] = prTrue
     | prAnd xs = "(and " ^ (LU.toStringSpace (fn x=>x) xs) ^ ")"
   fun prOr [] = prFalse
     | prOr xs = "(or " ^ (LU.toStringSpace (fn x=>x) xs) ^ ")"
   fun prXor [] = raise YicesError
     | prXor (x::xs) = L.foldl (fn (u,v) => "(/= " ^ u ^ " " ^ v ^ ")") x xs
   (* if then else *)
   fun prIfThenElse x y z = prOr [prAnd [x,y], prAnd [prNot x,z]]
   (* more than or equal to two *)
   fun prMoreThanTwo [] = prFalse
     | prMoreThanTwo (x::[]) = prFalse
     | prMoreThanTwo xs = prOr (L.map (fn (s1,s2) => (prAnd [s1,s2])) (pairs xs))
   fun prAtMostOne [] = prFalse
     | prAtMostOne (x::[]) = x
     | prAtMostOne xs = prAnd (L.map (fn (s1,s2) => (prNot (prAnd [s1,s2]))) (pairs xs))
   fun prExactlyOne [] = prFalse
     | prExactlyOne (x::[]) = x
     | prExactlyOne (x::xs) = prIfThenElse x (prNot (prOr xs)) (prExactlyOne xs)
   fun prExactlyOneIsFalse [] = prFalse
     | prExactlyOneIsFalse (x::[]) = prNot x
     | prExactlyOneIsFalse (x::xs) = prIfThenElse x (prExactlyOneIsFalse xs) (prAnd xs)
   fun prExactlyTwo [] = prFalse
     | prExactlyTwo (x::[]) = prFalse
     | prExactlyTwo (x::xs) = prIfThenElse x (prExactlyOne xs) (prExactlyTwo xs)

   fun prGe (s1,s2) = "(>= " ^ s1 ^ " " ^ s2 ^ ")"
   fun prGt (s1,s2) = "(> " ^ s1 ^ " " ^ s2 ^ ")"
   fun prEq (s1,s2) = "(= " ^ s1 ^ " " ^ s2 ^ ")"
   fun prNeq (s1,s2) = "(/= " ^ s1 ^ " " ^ s2 ^ ")"
   fun prLe (s1,s2) = "(<= " ^ s1 ^ " " ^ s2 ^ ")"
   fun prLt (s1,s2) = "(< " ^ s1 ^ " " ^ s2 ^ ")"

   val prZero = "0"
   fun prPlus [] = prZero
     | prPlus [x] = x
     | prPlus xs = "(+ " ^ (LU.toStringSpace (fn x=>x) xs) ^ ")"

   fun prDefInt s1 = "(define " ^ s1 ^ "::int)\n"
   fun prDefBool s1 = "(define " ^ s1 ^ "::bool)\n"

  fun readAssigns ins = 
      let fun readOneAssign line =
	      let 
		  val  toks = Substring.tokens (Char.contains " ()\n") 
					       (Substring.full line)
	      in case toks of 
		     [x,y,z] => if (Substring.string x) = "=" 
				then SOME (Substring.string y, Substring.string z) 
				else NONE
		   | _ => NONE
	      end

	  fun readAssignsSub ins ans = 
	      case TextIO.inputLine ins of
		  SOME line => ( (* print ("[" ^ line ^ "]\n"); *)
				if line = "\n"
				then (*  (print "..end..";ans) *)
				    ans
				else
				    (case readOneAssign line of
					 SOME (s1,s2) => 
					 readAssignsSub ins (StringMap.insert (ans,s1,s2))
				       | NONE => readAssignsSub ins ans))
		| NONE => ((* print ":"; *) readAssignsSub ins ans)
      in readAssignsSub ins StringMap.empty
      end
   end
end

signature ENCODE_INT_POLY =
sig
    val init: unit -> unit
   val lines: string ref
   val bitLength: int
   val bvar: int -> string
   val bvec: int * int -> string
   val intBvec: int * int -> string
   val bvecToInt: bool list -> int
   val encodeGeEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeGtEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeEvenEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeOddEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeEqualToZeroByModThreeEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeEqualToNonZeroByModThreeEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeMeqEquation: TextIO.outstream -> IntPoly.poly -> int
   val encodeMneqEquation: TextIO.outstream -> IntPoly.poly -> int
end

structure EncodeIntPoly: ENCODE_INT_POLY =
struct
   local
      structure IM = IntMap
      structure IP = IntPoly
      structure L = List
      structure LU = ListUtil
      open YicesInput
   in

   val lines = ref "" : string ref
   val bvecCounter = ref 0 (* counter for boolean vector *)

   datatype bvec = INT of int | NEW of int
   val intBvecTable = ref IM.empty : bvec IM.map ref
   val bvarCounter = ref 0 (* counter for boolean variable *)
   val constBvecTable = ref IM.empty : bvec IM.map ref

   fun bvecToString (INT i) = "b0_" ^ (Int.toString i)
     | bvecToString (NEW i) = "b1_" ^ (Int.toString i)

   fun init () = (lines := "";
		  bvecCounter := 0;
		  intBvecTable := IM.empty;
		  bvarCounter := 0;
		  constBvecTable := IM.empty)

(*   fun addlines outs newlines = (lines:= (!lines) ^ newlines) *)
   fun addlines outs newlines = TextIO.output (outs,newlines)

   (* 4-bit bool var : (i,0),...,(i,3)  *)
   val bitLength = 4

   fun intBvec (i,j) = "b0_" ^ (Int.toString i) ^ "_" ^ (Int.toString j)

   (* for each integer variable i,
      we prepare boolean vector <b0_i_0,b0_i_1,...> *)
   fun newIntBvec outs i = 
       case IM.find (!intBvecTable, i) of
	   SOME bv => bv
	 | NONE => let fun declare (i,j) = addlines outs (prDefBool (intBvec (i,j)))
		       val _ = L.app (fn j=> declare (i,j))
				     (L.tabulate (bitLength,fn x=>x))
		       val _ = intBvecTable := IM.insert (!intBvecTable, i, INT i)
		   in INT i
		   end

   fun bvec (i,j) = "b1_" ^ (Int.toString i) ^ "_" ^ (Int.toString j)

   (* boolean vector <b1_i_0,b1_i_0,...> *)
   fun newBvec outs () = 
       let fun declare (i,j) = addlines outs (prDefBool (bvec (i,j)))
	   val _ = bvecCounter := (!bvecCounter)+1
	   val _ = L.app (fn j=> declare (!bvecCounter,j))
			 (L.tabulate (bitLength,fn x=>x))
       in NEW (!bvecCounter)
       end


   fun bvar i = "b2_" ^ (Int.toString i)

   (* b2_0, b2_1, ... are used for auxiliary boolean variables *)
   fun newBvar outs () = 
       let val _ = bvarCounter := (!bvarCounter)+1
	   val _ = addlines outs (prDefBool (bvar (!bvarCounter)))
       in (!bvarCounter)
       end

   fun elem (INT i, j) = intBvec (i,j)
     | elem (NEW i, j) = bvec (i,j)

  (* bvecToInt [true,true,false,true] = 1*8+1*4+0*2+1 = 13 *)
   fun bvecToInt xs = 
       #2 (foldr (fn (b,(n,r))=> (n*2,r + (if b then n else 0))) (1,0)  xs)

   fun prEqVec (bv1,bv2) =
       prAnd (L.tabulate (bitLength,fn i=> prEq (elem (bv1,i),elem (bv2,i))))

   fun prEqZeroVec bv1 =
       prAnd (L.tabulate (bitLength,fn i=> prEq (elem (bv1,i),prFalse)))

   fun prEvenVec bv1 =
       prEq (elem (bv1,0),prFalse)

   fun prOddVec bv1 =
       prEq (elem (bv1,0),prTrue)

  (* 表わす整数が mod 3 で等しい for 4bits *)
   fun prEqualByModThreeVec bv1 bv2 = 
       prOr [prAnd [prOr [prAnd [elem (bv2,0),elem (bv2,2),elem (bv1,1),elem (bv1,3)],
			  prExactlyOne [elem (bv2,0),elem (bv2,2),elem (bv1,1),elem (bv1,3)]],
		    prOr [prAnd [elem (bv1,0),elem (bv1,2),elem (bv2,1),elem (bv2,3)],
			  prExactlyOne [elem (bv1,0),elem (bv1,2),elem (bv2,1),elem (bv2,3)]]],
	     prAnd [prOr [prExactlyOneIsFalse [elem (bv2,0),elem (bv2,2),elem (bv1,1),elem (bv1,3)],
			  prNot (prOr [elem (bv2,0),elem (bv2,2),elem (bv1,1),elem (bv1,3)])],
		    prOr [prExactlyOneIsFalse [elem (bv1,0),elem (bv1,2),elem (bv2,1),elem (bv2,3)],
			  prNot (prOr [elem (bv1,0),elem (bv1,2),elem (bv2,1),elem (bv2,3)])]],
	     prAnd [prExactlyTwo [elem (bv2,0),elem (bv2,2),elem (bv1,1),elem (bv1,3)],
		    prExactlyTwo [elem (bv1,0),elem (bv1,2),elem (bv2,1),elem (bv2,3)]]]

  (* 表わす整数が mod 3 で0に等しい for 4bits *)
   fun prEqualToZeroByModThreeVec bv =
       prOr [prAnd [elem (bv,0),elem (bv,1),elem (bv,2),elem (bv,3)],
	     prAnd [prXor [elem (bv,0),elem (bv,2)],prXor [elem (bv,1),elem (bv,3)]],
	     prNot (prOr [elem (bv,0),elem (bv,1),elem (bv,2),elem (bv,3)])]
  (* 表わす整数が mod 3 で1に等しい for 4bits *)
   fun prEqualToOneByModThreeVec bv =
       prOr [prAnd [prNot (elem (bv,0)),prNot (elem (bv,2)),elem (bv,1),elem (bv,3)],
	     prAnd [elem (bv,0),elem (bv,2),prXor [elem (bv,1),elem (bv,3)]],
	     prAnd [prXor [elem (bv,0),elem (bv,2)],prNot (elem (bv,1)),prNot (elem (bv,3))]]
  (* 表わす整数が mod 3 で2に等しい for 4bits *)
   fun prEqualToTwoByModThreeVec bv =
       prOr [prAnd [prNot (elem (bv,1)),prNot (elem (bv,3)),elem (bv,0),elem (bv,2)],
	     prAnd [elem (bv,1),elem (bv,3),prXor [elem (bv,0),elem (bv,2)]],
	     prAnd [prXor [elem (bv,1),elem (bv,3)],prNot (elem (bv,0)),prNot (elem (bv,2))]]

  (* 表わす整数の和が mod 3 で0/1/2に等しい for 4bits *)
   fun prEqualToZeroByModThreeVecList outs [] = prTrue
     | prEqualToZeroByModThreeVecList outs [bv] = prEqualToZeroByModThreeVec bv
     | prEqualToZeroByModThreeVecList outs (bv::rest) =
       let val bv0 = newBvar outs ()
	   val bv1 = newBvar outs ()
	   val bv2 = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert (prAnd [prEq (bvar bv0, prEqualToZeroByModThreeVecList outs rest),
				      prEq (bvar bv1, prEqualToOneByModThreeVecList outs rest),
				      prEq (bvar bv2, prEqualToTwoByModThreeVecList outs rest)]))
	   val ans = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert (prEq (bvar ans, 
				     prOr [prAnd [prEqualToZeroByModThreeVec bv, bvar bv0],
					   prAnd [prEqualToOneByModThreeVec bv, bvar bv2],
					   prAnd [prEqualToTwoByModThreeVec bv, bvar bv1]])))
       in bvar ans
       end

   and prEqualToOneByModThreeVecList outs [] = prFalse
     | prEqualToOneByModThreeVecList outs [bv] = prEqualToOneByModThreeVec bv
     | prEqualToOneByModThreeVecList outs (bv::rest) =
       let val bv0 = newBvar outs ()
	   val bv1 = newBvar outs ()
	   val bv2 = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert (prAnd [prEq (bvar bv0, prEqualToZeroByModThreeVecList outs rest),
				      prEq (bvar bv1, prEqualToOneByModThreeVecList outs rest),
				      prEq (bvar bv2, prEqualToTwoByModThreeVecList outs rest)]))
	   val ans = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert (prEq (bvar ans, 
				     prOr [prAnd [prEqualToZeroByModThreeVec bv, bvar bv1],
					   prAnd [prEqualToOneByModThreeVec bv, bvar bv0],
					   prAnd [prEqualToTwoByModThreeVec bv, bvar bv2]])))
       in bvar ans
       end

   and prEqualToTwoByModThreeVecList outs [] = prFalse
     | prEqualToTwoByModThreeVecList outs [bv] = prEqualToTwoByModThreeVec bv
     | prEqualToTwoByModThreeVecList outs (bv::rest) =
       let val bv0 = newBvar outs ()
	   val bv1 = newBvar outs ()
	   val bv2 = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert (prAnd [prEq (bvar bv0, prEqualToZeroByModThreeVecList outs rest),
				      prEq (bvar bv1, prEqualToOneByModThreeVecList outs rest),
				      prEq (bvar bv2, prEqualToTwoByModThreeVecList outs rest)]))
	   val ans = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert (prEq (bvar ans, 
				     prOr [prAnd [prEqualToZeroByModThreeVec bv, bvar bv2],
					   prAnd [prEqualToOneByModThreeVec bv, bvar bv1],
					   prAnd [prEqualToTwoByModThreeVec bv, bvar bv0]])))
       in bvar ans
       end

  (* xss - yss が mod 3 で 0 に等しい *)
   fun prDiffZeroByModThreeVecList outs (xbvs,ybvs) =
       let val b0x = newBvar outs ()
	   val b1x = newBvar outs ()
	   val b2x = newBvar outs ()
	   val b0y = newBvar outs ()
	   val b1y = newBvar outs ()
	   val b2y = newBvar outs ()
	   val _ =  addlines outs
			     (prAssert (prAnd
				       [prEq (bvar b0x, prEqualToZeroByModThreeVecList outs xbvs),
					prEq (bvar b1x, prEqualToOneByModThreeVecList outs xbvs),
					prEq (bvar b2x, prEqualToTwoByModThreeVecList outs xbvs),
					prEq (bvar b0y, prEqualToZeroByModThreeVecList outs ybvs),
					prEq (bvar b1y, prEqualToOneByModThreeVecList outs ybvs),
					prEq (bvar b2y, prEqualToTwoByModThreeVecList outs ybvs)]))
	   val ans = newBvar outs ()
	   val _ =  addlines outs
		    (prAssert(prEq (bvar ans, prOr [prAnd [bvar b0x,bvar b0y],
						    prAnd [bvar b1x,bvar b1y],
						    prAnd [bvar b2x,bvar b2y]])))
       in bvar ans
       end

  (* xss - yss が mod 3 で 1 に等しい *)
   fun prDiffOneByModThreeVecList outs (xbvs,ybvs) =
       let val b0x = newBvar outs ()
	   val b1x = newBvar outs ()
	   val b2x = newBvar outs ()
	   val b0y = newBvar outs ()
	   val b1y = newBvar outs ()
	   val b2y = newBvar outs ()
	   val _ =  addlines outs
			     (prAssert (prAnd
				       [prEq (bvar b0x, prEqualToZeroByModThreeVecList outs xbvs),
					prEq (bvar b1x, prEqualToOneByModThreeVecList outs xbvs),
					prEq (bvar b2x, prEqualToTwoByModThreeVecList outs xbvs),
					prEq (bvar b0y, prEqualToZeroByModThreeVecList outs ybvs),
					prEq (bvar b1y, prEqualToOneByModThreeVecList outs ybvs),
					prEq (bvar b2y, prEqualToTwoByModThreeVecList outs ybvs)]))
	   val ans = newBvar outs ()
	   val _ =  addlines outs
			     (prAssert
			     (prEq (bvar ans, prOr [prAnd [bvar b0x,bvar b2y],
						    prAnd [bvar b1x,bvar b0y],
						    prAnd [bvar b2x,bvar b1y]])))
       in bvar ans
       end

  (* xss - yss が mod 3 で 2 に等しい *)
   fun prDiffTwoByModThreeVecList outs (xbvs,ybvs) =
       let val b0x = newBvar outs ()
	   val b1x = newBvar outs ()
	   val b2x = newBvar outs ()
	   val b0y = newBvar outs ()
	   val b1y = newBvar outs ()
	   val b2y = newBvar outs ()
	   val _ =  addlines outs
			     (prAssert (prAnd
				       [prEq (bvar b0x, prEqualToZeroByModThreeVecList outs xbvs),
					prEq (bvar b1x, prEqualToOneByModThreeVecList outs xbvs),
					prEq (bvar b2x, prEqualToTwoByModThreeVecList outs xbvs),
					prEq (bvar b0y, prEqualToZeroByModThreeVecList outs ybvs),
					prEq (bvar b1y, prEqualToOneByModThreeVecList outs ybvs),
					prEq (bvar b2y, prEqualToTwoByModThreeVecList outs ybvs)]))
	   val ans = newBvar outs ()
	   val _ =  addlines outs
			     (prAssert (prEq (bvar ans, prOr [prAnd [bvar b0x,bvar b1y],
							      prAnd [bvar b1x,bvar b2y],
							      prAnd [bvar b2x,bvar b0y]])))
       in bvar ans
       end

   (* return k s.t. k = plus (i,j) *)
   fun plus outs (bv1,bv2) = 
       let val bv3 = newBvec outs ()
	   val car = newBvec outs ()
	   val _ = 
	       addlines outs
		   ((prAssert 
			 (prEq (elem (bv3,0),
				prXor [elem (bv1,0), elem (bv2,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,0),
				prAnd [elem (bv1,0), elem (bv2,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (bv3,1),
				prXor [elem (bv1,1), elem (bv2,1), elem (car,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,1),
				prMoreThanTwo 
				    [elem (bv1,1), elem (bv2,1), elem (car,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (bv3,2),
				prXor [elem (bv1,2), elem (bv2,2), elem (car,1)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,2),
				prMoreThanTwo 
				    [elem (bv1,2), elem (bv2,2), elem (car,1)])))
		    ^
		    (prAssert 
			 (prEq (elem (bv3,3),
				prXor [elem (bv1,3), elem (bv2,3), elem (car,2)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,3),
				prMoreThanTwo 
				    [elem (bv1,3), elem (bv2,3), elem (car,2)])))
		    ^
		    (prAssert (prNot (elem (car,3)))))
       in bv3
       end

   (* return k such that k <= plus (i,j) *)
   (* by neglecting carry over *)
   fun plus2 outs (bv1,bv2) = 
       let val bv3 = newBvec outs ()
	   val car = newBvec outs ()
	   val _ = 
	       addlines outs
		   ((prAssert 
			 (prEq (elem (bv3,0),
				prXor [elem (bv1,0), elem (bv2,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,0),
				prAnd [elem (bv1,0), elem (bv2,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (bv3,1),
				prXor [elem (bv1,1), elem (bv2,1), elem (car,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,1),
				prMoreThanTwo 
				    [elem (bv1,1), elem (bv2,1), elem (car,0)])))
		    ^
		    (prAssert 
			 (prEq (elem (bv3,2),
				prXor [elem (bv1,2), elem (bv2,2), elem (car,1)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,2),
				prMoreThanTwo 
				    [elem (bv1,2), elem (bv2,2), elem (car,1)])))
		    ^
		    (prAssert 
			 (prEq (elem (bv3,3),
				prXor [elem (bv1,3), elem (bv2,3), elem (car,2)])))
		    ^
		    (prAssert 
			 (prEq (elem (car,3),
				prMoreThanTwo 
				    [elem (bv1,3), elem (bv2,3), elem (car,2)]))))
(*	   val _ = print ("plus2:" 
			  ^ "bv1(" ^  (bvecToString bv1) ^ ")+"
			  ^ "bv2(" ^  (bvecToString bv2) ^ ")>="
			  ^ "bv3(" ^  (bvecToString bv3) ^ ")\n") *)
       in bv3
       end

   (* return k such that k = times (i,j) *)
   (* we request most insignificant bit of i,j are 0 *)
   fun shift outs n bv1 = 
       let val bv2 = newBvec outs ()
	   val _ = addlines outs
		       (prAssert (prAnd 
				      (L.tabulate (bitLength,
						  (fn k=> prEq (elem (bv2,k), 
								if k<n then prFalse 
								else elem (bv1,k-n)))))))
       in bv2
       end

(*
   fun land (bv1,bv2) = 
       let val bv3 = newBvec ()
	   val _ = addlines outs
		       (prAssert (L.tabulate (bitLength,
					   fn i=> prEq (elem (bv3,i), 
							prAnd [elem (bv1,i), elem (bv2,i)]))))
       in bv3
       end
*)

   fun times outs (bv1,bv2) =
       let val bv1_1 = shift outs 1 bv1
	   val bv1_2 = shift outs 2 bv1
	   val bv1_3 = shift outs 3 bv1
	   val bv3_0 = newBvec outs ()
	   val bv3_1 = newBvec outs ()
	   val bv3_2 = newBvec outs ()
	   val bv3_3 = newBvec outs ()
	   val _ = addlines outs
		   ((prAssert (prNot (prAnd [elem (bv2,1), elem (bv1,3)])))
		    ^ (prAssert (prNot (prAnd [elem (bv2,2), 
					       prOr [elem (bv1,2),elem (bv1,3)]])))
		    ^ (prAssert (prNot (prAnd [elem (bv2,3), 
					       prOr [elem (bv1,1),elem (bv1,2),elem (bv1,3)]])))
		    ^ (prAssert (prOr [prNot (elem (bv2,0)),prEqVec (bv3_0,bv1)]))
		    ^ (prAssert (prOr [elem (bv2,0), prEqZeroVec bv3_0]))
		    ^ (prAssert (prOr [prNot (elem (bv2,1)),prEqVec (bv3_1,bv1_1)]))
		    ^ (prAssert (prOr [elem (bv2,1), prEqZeroVec bv3_1]))
		    ^ (prAssert (prOr [prNot (elem (bv2,2)),prEqVec (bv3_2,bv1_2)]))
		    ^ (prAssert (prOr [elem (bv2,2), prEqZeroVec bv3_2]))
(*		    ^ (prAssert (prOr [prNot (elem (bv2,3)),prEqVec (bv3_2,bv1_3)])) bug fixed 2013/1/28 *)
		    ^ (prAssert (prOr [prNot (elem (bv2,3)),prEqVec (bv3_3,bv1_3)])) 
		    ^ (prAssert (prOr [elem (bv2,3), prEqZeroVec bv3_3])))
	   val sum1 = plus outs (bv3_0, bv3_1)
	   val sum2 = plus outs (bv3_2, bv3_3)
(*	   val sum2 = plus outs (sum1, sum2)
       in sum2  bug fixed 2013/1/28  *)
	   val sum3 = plus outs (sum1, sum2)
       in sum3
       end

   (* return k such that k <= times (i,j) *)
   fun times2 outs (bv1,bv2) =
       let val bv1_1 = shift outs 1 bv1
	   val bv1_2 = shift outs 2 bv1
	   val bv1_3 = shift outs 3 bv1
	   val bv3_0 = newBvec outs ()
	   val bv3_1 = newBvec outs ()
	   val bv3_2 = newBvec outs ()
	   val bv3_3 = newBvec outs ()
	   val _ = addlines outs
		   ((prAssert (prOr [prNot (elem (bv2,0)),prEqVec (bv3_0,bv1)]))
		    ^ (prAssert (prOr [elem (bv2,0), prEqZeroVec bv3_0]))
		    ^ (prAssert (prOr [prNot (elem (bv2,1)),prEqVec (bv3_1,bv1_1)]))
		    ^ (prAssert (prOr [elem (bv2,1), prEqZeroVec bv3_1]))
		    ^ (prAssert (prOr [prNot (elem (bv2,2)),prEqVec (bv3_2,bv1_2)]))
		    ^ (prAssert (prOr [elem (bv2,2), prEqZeroVec bv3_2]))
(*		    ^ (prAssert (prOr [prNot (elem (bv2,3)),prEqVec (bv3_2,bv1_3)]))  bug fixed 2013/1/28*)
		    ^ (prAssert (prOr [prNot (elem (bv2,3)),prEqVec (bv3_3,bv1_3)])) 
		    ^ (prAssert (prOr [elem (bv2,3), prEqZeroVec bv3_3])))
	   val sum1 = plus2 outs (bv3_0, bv3_1)
	   val sum2 = plus2 outs (bv3_2, bv3_3)
(* bug 	   val sum2 = plus2 outs (sum1, sum2)
       in sum2  bug fixed 2013/1/28 *)
	   val sum3 = plus2 outs (sum1, sum2)
(*	   val _ = print ("times2:" 
			  ^ "bv1(" ^  (bvecToString bv1) ^ ")*"
			  ^ "bv2(" ^  (bvecToString bv2) ^ ")>="
			  ^ "sum3(" ^  (bvecToString sum3) ^ ")\n") *)
       in sum3
       end


  (* n should be 0 <= n < 16 *)
  fun interpretConst outs n =
      case IM.find (!constBvecTable,n) of 
	  SOME bv => bv
	| NONE => let val bv = newBvec outs ()
		      val _ = constBvecTable := (IM.insert (!constBvecTable,n,bv))
		      fun f (x,k) = (Int.div (x,k), Int.mod (x,k))
		      val (m3,n0) = f (n,8)
		      val (m2,n1) = f (n0,4)
		      val (m1,m0) = f (n1,2)
		      fun mkBool m = if m = 0 then prFalse
				     else prTrue
		      val _ = addlines outs 
			  ((prAssert (prEq (elem (bv,0), mkBool m0)))
			  ^ (prAssert (prEq (elem (bv,1), mkBool m1)))
			  ^ (prAssert (prEq (elem (bv,2), mkBool m2)))
			  ^ (prAssert (prEq (elem (bv,3), mkBool m3))))
		   in bv
		   end

   fun interpretMonoLeft outs (xs,n) =
       let val ans = 
	       if (n >= 16)
	       then interpretMonoLeft outs (xs,15)
	       else if null xs
	       then interpretConst outs n
	       else let val bvecs = L.map (newIntBvec outs) xs
		    in if n = 1
		       then L.foldl (times2 outs) (hd bvecs) (tl bvecs)
		       else L.foldl (times2 outs) (interpretConst outs n) bvecs
		    end
	   (* val _ = print ("(mono[l]" ^ (LU.toStringCommaCurly Int.toString xs)
			       ^ ": " ^ (bvecToString ans) ^ ");\n")  *)
       in ans
       end

   fun interpretMonoRight outs (xs,n) =
       let val bvecs = L.map (newIntBvec outs) xs
	   val ans = 
	       if (n >= 16)
	       then if null bvecs
		    then (addlines outs (prAssert (prEq ("0","1"))); interpretConst outs 0)  
		    (* no possibility *)
		    else (* some variable should be zero *)
			(addlines outs (prAssert 
				       (prOr (L.map prEqZeroVec bvecs))); interpretConst outs 0)
	       else if null bvecs
	       then interpretConst outs n
	       else if n = 1
	       then L.foldl (times outs) (hd bvecs) (tl bvecs)
	       else L.foldl (times outs) (interpretConst outs n) bvecs
(*      val _ = print ("(mono[r]" ^ (LU.toStringCommaCurly Int.toString xs)
			  ^ ": " ^ (bvecToString ans) ^ ");\n")  *)
       in ans
       end

   (* (左辺,右辺) 係数はすべて1以上 *)
   fun interpretPolyLeft outs xss =
       if null xss
	  then interpretConst outs 0
	  else let val monos = L.map (interpretMonoLeft outs) xss
		   val sum = L.foldl (plus2 outs) (hd monos) (tl monos)
		 (*   val _ = print ("approx: " ^  (bvecToString sum) ^ " ; ") *)
	       in sum
	       end
	       
   (* (左辺,右辺) 係数はすべて1以上 *)
   fun interpretPolyRight outs xss =
       if null xss
       then interpretConst outs 0
       else let val monos = L.map (interpretMonoRight outs) xss
		val sum = L.foldl (plus outs) (hd monos) (tl monos)
		(* val _ = print ("exact: " ^  (bvecToString sum) ^ "\n") *)
	    in sum
	    end

   fun interpretPoly outs intPoly =
       let val xss = IP.toList intPoly
	   val (pos,neg) = L.partition (fn (_,n) => n >=0) xss
	   val pos2 = L.filter (fn (_,n) => n > 0) pos
	   val neg2 = L.map (fn (xs,n) => (xs,~n)) neg
	   (*  val _ = print ("pos: " ^  (IP.toString (IP.fromList pos2)) ^ "\n")  *)
	   (*  val _ = print ("neg: " ^  (IP.toString (IP.fromList neg2)) ^ "\n")  *)
       in (interpretPolyLeft outs pos2, interpretPolyRight outs neg2)
       end

   (*  多項式が \ge 0 となる条件を lines に
       エンコードし、それを示すブール変数を返す *)
   fun encodeGeEquation outs intPoly  =
       let val (bv1,bv2) = interpretPoly outs intPoly
	   val bge = newBvar outs ()
(* 	   val _ = print "poly and its boolean variable (for >=):\n"  *)
(* 	   val _ = print ((IP.toString intPoly)^ "\n")   *)
(* 	   val _ = print ((bvar bge) ^ "\n")   *)
	   val _ = addlines outs
		   ((prAssert (prEq (bvar bge,
				    prOr [ prGt (elem (bv1,3),elem (bv2,3)),
					   prAnd [prEq (elem (bv1,3),elem (bv2,3)),
						  prGt (elem (bv1,2),elem (bv2,2))],
					   prAnd [prEq (elem (bv1,3),elem (bv2,3)),
						  prEq (elem (bv1,2),elem (bv2,2)),
						  prGt (elem (bv1,1),elem (bv2,1))],
					   prAnd [prEq (elem (bv1,3),elem (bv2,3)),
						  prEq (elem (bv1,2),elem (bv2,2)),
						  prEq (elem (bv1,1),elem (bv2,1)),
						  prGe (elem (bv1,0),elem (bv2,0))]]))))
       in bge
       end

   (*  多項式が \gt 0 となる条件を lines に
       エンコードし、それを示すブール変数を返す *)
   fun encodeGtEquation outs intPoly  =
       let val (bv1,bv2) = interpretPoly outs intPoly
	   val bgt = newBvar outs ()
(* 	   val _ = print "poly and its boolean variable (for >):\n"  *)
(* 	   val _ = print ((IP.toString intPoly)^ "\n")  *)
(* 	   val _ = print ((bvar bgt) ^ "\n")   *)
	   val _ = addlines outs
		   ((prAssert (prEq (bvar bgt,
				    prOr [ prGt (elem (bv1,3),elem (bv2,3)),
					   prAnd [prEq (elem (bv1,3),elem (bv2,3)),
						  prGt (elem (bv1,2),elem (bv2,2))],
					   prAnd [prEq (elem (bv1,3),elem (bv2,3)),
						  prEq (elem (bv1,2),elem (bv2,2)),
						  prGt (elem (bv1,1),elem (bv2,1))],
					   prAnd [prEq (elem (bv1,3),elem (bv2,3)),
						  prEq (elem (bv1,2),elem (bv2,2)),
						  prEq (elem (bv1,1),elem (bv2,1)),
						  prGt (elem (bv1,0),elem (bv2,0))]]))))
       in bgt
       end

   (* 変数が偶数となる条件 *)
   fun encodeBvecEven  bv = prEq (elem (bv,0), prFalse)

   (* 変数が奇数となる条件 *)
   fun encodeBvecOdd  bv = prEq (elem (bv,0), prTrue)

   (* 変数が4の倍数となる条件 *)
   fun encodeBvecModFour bv = prAnd [prEq (elem (bv,0), prFalse), prEq (elem (bv,1), prFalse)]

   (* 変数が1となる条件 *)
   fun encodeBvecEqualOne bv = prAnd [prEq (elem (bv,0), prTrue), prEq (elem (bv,1), prFalse),
				      prEq (elem (bv,2), prFalse), prEq (elem (bv,3), prFalse)]

   (* 変数リストの積が偶数となる条件 *)
   fun encodeBvecListEven  bvs = prOr (L.map encodeBvecEven bvs)

   (* 変数リストの積が奇数となる条件 *)
   fun encodeBvecListOdd bvs = prAnd (L.map encodeBvecOdd bvs)

   (*  奇数係数をもつ単項式が even となる条件を lines に
       エンコードし、それを示すブール変数を返す *)
   fun encodeMonoEven outs (xs,n) =
       let val bvecs = L.map (newIntBvec outs) xs
	   val beven = newBvar outs ()
	   (* val _ = print "mono and its boolean variable (for even): " *)
	   (* val _ = print ((IP.toString (IP.fromList [(xs,n)]))^ " ") *)
	   (* val _ = print ((bvar beven) ^ "\n") *)
	   val _ = addlines outs (prAssert (prEq (bvar beven, encodeBvecListEven bvecs)))
       in beven
       end

   (*  多項式が Even となる条件を lines に   エンコードし、それを示すブール変数を返す *)
   fun encodeEvenEquation outs intPoly  =
       let val xss = IP.toList intPoly
	   val bev = newBvar outs ()
	   val (consts,yss) = L.partition (fn (xs,_) => null xs) xss
	   val (_,k) = if null consts
		       then ([],0)
		       else hd consts
	   val zss = L.filter (fn (xs,n) => n mod 2 <> 0) yss (* 係数が偶数でない単項を抜き出す *)
	   val bvars = L.map (fn ys => bvar (encodeMonoEven outs ys)) zss
	   (* val _ = print "poly and its boolean variable (for even): " *)
	   (* val _ = print ((IP.toString intPoly)^ " ") *)
	   (* val _ = print ((bvar bev) ^ "\n") *)

	   fun prEven [] = prTrue  (* 偽が偶数個 *)
	     | prEven (x::[]) = x
	     | prEven (x::xs) = let val bev = newBvar outs ()
				    val bod = newBvar outs ()
				    val _ = addlines outs (prAssert (prEq (bvar bev, prEven xs)))
				    val _ = addlines outs (prAssert (prEq (bvar bod, prOdd xs)))
				in prIfThenElse x (bvar bev) (bvar bod)
				end
	   and prOdd []  = prFalse  (* 偽が奇数個 *)
	     | prOdd (x::[]) = prNot x
	     | prOdd (x::xs) = (* prIfThenElse x (prOdd xs) (prEven xs)  *)
	       let val bev = newBvar outs ()
		   val bod = newBvar outs ()
		   val _ = addlines outs (prAssert (prEq (bvar bev, prEven xs)))
		   val _ = addlines outs (prAssert (prEq (bvar bod, prOdd xs)))
	       in prIfThenElse x (bvar bod) (bvar bev)
	       end
	   val _ = if k mod 2 = 0
		   then addlines outs (prAssert (prEq (bvar bev, prEven bvars))) (* 定数が偶数のとき *)
		   else addlines outs (prAssert (prEq (bvar bev, prOdd bvars)))  (* 定数が奇数のとき *)
       in bev
       end

   fun encodeOddEquation outs intPoly  =
       let val bod = newBvar outs ()
	   val bev = encodeEvenEquation outs intPoly
 	   (* val _ = print "poly and its boolean variable (for odd): "   *)
	   (* val _ = print ((IP.toString intPoly)^ " ")   *)
	   (* val _ = print ((bvar bod) ^ "\n")    *)
	   val _ = addlines outs (prAssert (prEq (bvar bod, prNot (bvar bev))))
       in bod
       end


   (*  多項式が3の倍数となる条件を lines に   エンコードし、それを示すブール変数を返す *)
   fun encodeEqualToZeroByModThreeEquation outs intPoly  =
       let val xss = IP.toList intPoly
	   val bmd = newBvar outs ()
(* 	   val _ = print "poly and its boolean variable (for = 0 (mod 3)): "  *)
(*	   val _ = print ((IP.toString intPoly)^ " ")  *)
(*	   val _ = print ((bvar bmd) ^ "\n")   *)

	   val (consts,yss) = L.partition (fn (xs,_) => null xs) xss (* 定数をだす *)
	   val (_,k) = if null consts then ([],0) else hd consts
	   val yss0 = L.filter (fn (xs,n) => n mod 3 <> 0) yss  (* 係数が3の倍数のは除外して考える *)
	   val (yss1,yss2) = L.partition (fn (xs,n) => n mod 3 = 1) yss0  (* 係数/3の余りで分割 *)
	   val bvs1 = L.map (fn (xs,_) => interpretMonoRight outs (xs,1)) yss1 (* 係数を1と考えてexactに乗算 *) 

(*	   val _ = print "bvs1: "  *)
(*	   val _ = print ((IP.toString (IP.fromList yss1))^ " ")  *)
(*	   val _ = print (LU.toStringCommaCurly bvecToString bvs1)  *)
(*	   val _ = print "\n" *)

	   val bvs2 = L.map (fn (xs,_) => interpretMonoRight outs (xs,1)) yss2 (* 係数を1を考えて乗算 *) 

(* 	   val _ = print "bvs2: "  *)
(*	   val _ = print ((IP.toString (IP.fromList yss2))^ " ")  *)
(*	   val _ = print (LU.toStringCommaCurly bvecToString bvs2)  *)
(*	   val _ = print "\n" *)

	   val _ = case (k mod 3) of
		       0 => addlines outs (prAssert (prEq (bvar bmd, prDiffZeroByModThreeVecList outs (bvs1,bvs2))))
		     | 1 => addlines outs (prAssert (prEq (bvar bmd, prDiffTwoByModThreeVecList outs (bvs1,bvs2))))
		     | n => addlines outs (prAssert (prEq (bvar bmd, prDiffOneByModThreeVecList outs (bvs1,bvs2))))
       in bmd
       end

   (*  多項式が3の倍数とならない条件を lines に   エンコードし、それを示すブール変数を返す *)
   fun encodeEqualToNonZeroByModThreeEquation outs intPoly  =
       let val xss = IP.toList intPoly
	   val bmd = newBvar outs ()
(* 	   val _ = print "poly and its boolean variable (for <> 0 (mod 3)): "  *)
(*	   val _ = print ((IP.toString intPoly)^ " ")  *)
(*	   val _ = print ((bvar bmd) ^ "\n")   *)
	   val (consts,yss) = L.partition (fn (xs,_) => null xs) xss (* 定数をだす *)
	   val (_,k) = if null consts then ([],0) else hd consts
	   val yss0 = L.filter (fn (xs,n) => n mod 3 <> 0) yss  (* 係数が3の倍数のは除外して考える *)
	   val (yss1,yss2) = L.partition (fn (xs,n) => n mod 3 = 1) yss0  (* 係数/3の余りで分割 *)

	   val bvs1 = L.map (fn (xs,_) => interpretMonoRight outs (xs,1)) yss1 (* 係数を1を考えて乗算 *) 

(*	   val _ = print "bvs1: "  *)
	   (* val _ = print ((IP.toString (IP.fromList yss1))^ " ")   *)
	   (* val _ = print (LU.toStringCommaCurly bvecToString bvs1)   *)
	   (* val _ = print "\n" *)

	   val bvs2 = L.map (fn (xs,_) => interpretMonoRight outs (xs,1)) yss2 (* 係数を1を考えて乗算 *) 

 	   (* val _ = print "bvs2: "   *)
	   (* val _ = print ((IP.toString (IP.fromList yss2))^ " ")  *)
	   (* val _ = print (LU.toStringCommaCurly bvecToString bvs2)   *)
	   (* val _ = print "\n" *)

	   val _ = case (k mod 3) of
		       0 => addlines outs (prAssert (prEq (bvar bmd, 
							   prNot (prDiffZeroByModThreeVecList outs (bvs1,bvs2)))))
		     | 1 => addlines outs (prAssert (prEq (bvar bmd, 
							   prNot (prDiffTwoByModThreeVecList outs (bvs1,bvs2)))))
		     | n => addlines outs (prAssert (prEq (bvar bmd, 
							   prNot (prDiffOneByModThreeVecList outs (bvs1,bvs2)))))
       in bmd
       end

   (*  単項式が4の倍数となる十分条件を lines にエンコードし、それを示すブール変数を返す *)
   fun encodeMeqMono outs (xs,n) =
       let val bvecs = L.map (newIntBvec outs) xs
	   val bmeq = newBvar outs ()
(* 	   val _ = print "mono and its boolean variable (for even): "  *)
(*	   val _ = print ((IP.toString (IP.fromList [(xs,n)]))^ " ") *)
(* 	   val _ = print ((bvar bmeq) ^ "\n")  *)
	   val _ = if n mod 4 = 0
		   then addlines outs (prAssert (prEq (bvar bmeq, prTrue)))
		   else if n mod 2 = 0
		   then addlines outs (prAssert (prEq (bvar bmeq, encodeBvecListEven bvecs)))
		   else addlines outs 
			(prAssert (prEq 
				   (bvar bmeq, 
				    prOr [prOr (L.map encodeBvecModFour bvecs),  (* どれかが4の倍数か *)
					  prMoreThanTwo (L.map encodeBvecEven bvecs)])))  (* 偶数が2つ以上 *)
       in bmeq
       end

   (* 係数が4の倍数でなく変数がすべて1，という条件を lines にエンコードし、それを示すブール変数を返す *)
   fun encodeMonoMeqSp outs (xs,n) =
       let val bvecs = L.map (newIntBvec outs) xs
	   val bmeq = newBvar outs ()
	   val _ = if n mod 4 = 0
		   then addlines outs (prAssert (prEq (bvar bmeq, prFalse)))
		   else addlines outs (prAssert (prEq (bvar bmeq, prAnd (L.map encodeBvecEqualOne bvecs))))
       in bmeq
       end

   (*  多項式が modulo 4 で等しくなる条件を lines に   エンコードし、それを示すブール変数を返す *)
   fun encodeMeqEquation outs intPoly  =
       let val xss = IP.toList intPoly
	   val bev = newBvar outs ()
	   val (consts,yss) = L.partition (fn (xs,_) => null xs) xss
	   val (_,k) = if null consts
		       then ([],0)
		       else hd consts
	   val _ = if k mod 4 <> 0
		   then addlines outs (prAssert (prEq (bvar bev, prFalse))) (* 定数が4の倍数でなければアウト *)
		   else
		       let val zss = L.filter (fn (xs,n) => n mod 4 <> 0) yss (* 係数が4の倍数でない単項を抜き出す *)
			   val bvars = L.map (fn zs => bvar (encodeMeqMono outs zs)) zss
		       in addlines outs (prAssert (prEq (bvar bev, prAnd bvars)))
		       end
       in bev
       end

   (*  多項式が modulo 4 にならない条件 *)
   fun encodeMneqEquation outs intPoly  =
       let val xss = IP.toList intPoly
	   val bmneq = newBvar outs ()
	   val (consts,yss) = L.partition (fn (xs,_) => null xs) xss
	   val mvecs = L.map (encodeMonoMeqSp outs) yss (* 係数が4の倍数でなく変数がすべて1という条件 *)
	   val (_,k) = if null consts
		       then ([],0)
		       else hd consts
	   val vars = LU.mapAppend (fn (xs,_) => xs) yss
	   val vvecs = L.map (newIntBvec outs) vars
	   val _ = if k mod 4 <> 0 
		   then addlines outs  (* 定数が4の倍数でないとき，変数がすべて4の倍数ならOK *)
				 (prAssert (prEq (bvar bmneq, prAnd (L.map encodeBvecModFour vvecs))))
		   else	addlines outs  (* そうでないとき，係数が4の倍数でなく変数がすべて1，となる
					      単項が丁度1つあればよい *)
				 (prAssert (prEq (bvar bmneq, prExactlyOne (L.map bvar mvecs))))
       in bmneq
       end


   end
end

signature POLYNOMIAL_INTERPRETATION =
sig
   val runDebug: bool ref
   val init: unit -> unit
   val prepareCheck:
       TextIO.outstream 
       ->    ((Term.term * Term.term) list * (Term.term * Term.term) list)
           -> (Fun.ord_key * int * IntPolyPoly.poly) list * string * int list
              * int list list * int list list * int list * int list

   val step: TextIO.instream * TextIO.outstream -> int 
             -> (Fun.ord_key * int * IntPolyPoly.poly) list * string * int list
		* int list list * int list list * int list * int list
             -> ((Term.term * Term.term) list  * (Term.term * Term.term) list)
             -> (int list * int list) -> (int list * int list) option

  val directCheck: string 
		   -> ((Term.term * Term.term) list * (Term.term * Term.term) list)
		   -> bool

  val njCheckEvenOdd: string -> string 
		      -> (Term.term * Term.term)
		      -> (Term.term * Term.term) list
		      -> bool * (unit -> string)

  val njCheckByOrder: string -> string 
		      -> (Term.term * Term.term)
		      -> (Term.term * Term.term) list
		      -> bool * (unit -> string)

  val njCheckModuloThree: string -> string 
		      -> (Term.term * Term.term)
		      -> (Term.term * Term.term) list
		      -> bool * (unit -> string)

  val njCheckModuloFour: string -> string 
		      -> (Term.term * Term.term)
		      -> (Term.term * Term.term) list
		      -> bool
end


structure PolynomialInterpretation : POLYNOMIAL_INTERPRETATION =
struct
   local
       open Term
       structure CU = CertifyUtil
       structure EIP = EncodeIntPoly
       structure FM = FunMap
       structure IM = IntMap
       structure ILM = IntListMap
       structure IP = IntPoly
       structure IPV = IntPolyVec
       structure IPM = IntPolyMatrix
       structure IPP = IntPolyPoly
       structure IPPM = IntPolyPolyMatrix
       structure IPPV = IntPolyPolyVec
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure YI = YicesInput
       open PrintUtil
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()

   val counter = ref 0
   fun init () = (counter:= 0)

   (* 単項式  f_J * \Prod_{j \in J} x_jの生成 *)
   fun newVarPoly xs = (counter := (!counter)+1; 
			IPP.fromList [(xs,IP.fromList [([!counter],1)])])

  (* 0 -- linear *)
  (* 1 -- simple *)
  (* 2 -- simple-mixed *)
   val polynomialClass = ref 2
   datatype poly_class = LIN | SIM | MIX

   (* 多項式の生成
     (n = 3 の場合)
     ** linear version **
            f_{} + f_{1} * x_1  + f_{2} * x_2  + f_{3} * x_3

     ** simple version **
            f_{} + f_{1} * x_1  + f_{2} * x_2  + f_{3} * x_3 
            + f_{1,2} * x_1 * x_2 + f_{1,3} * x_1 * x_3 + f_{2,3} * x_2 * x_3
            + f_{1,2,3} * x_1 * x_2 * x_3

     ** simple-mixed version **
            f_{} + f_{1} * x_1  + f_{2} * x_2  + f_{3} * x_3 
            + f_{1,2} * x_1 * x_2 + f_{1,3} * x_1 * x_3 + f_{2,3} * x_2 * x_3
            + f_{1,2,3} * x_1 * x_2 * x_3 
            + f_{1,1} * x_1 * x_1 + f_{2,2} * x_2 * x_2 + f_{3,3} * x_3 * x_3
   *)

   fun newPoly cls n = let val xss = case cls of
				     LIN => []::(L.tabulate (n,fn i =>[i+1]))
				   | SIM => LU.powerlist (L.tabulate (n,fn i =>i+1))
				   | MIX => (LU.powerlist (L.tabulate (n,fn i =>i+1)))
 					  @ (L.tabulate (n,fn i =>[i+1,i+1]))
		   in if null xss
		      then IPP.zero
		      else L.foldl (fn (ys,ans) => IPP.plus (newVarPoly ys,ans))
				   (newVarPoly (hd xss))
				   (tl xss)
		   end

   (* 関数記号fにたいして、[(f,arity_f,poly_f) | f \in F ] を割り当てる．
      このリストは fpassoc と参照される *)

   fun makeFunPolyAssoc cls faMap = 
       L.map (fn (f,arity) => (f,arity,newPoly cls arity))
	     (FM.listItemsi faMap)

   (* fpassoc の lookup *)
   fun lookup fpassoc f = valOf (L.find (fn (g,_,_) => Fun.equal (f,g)) fpassoc)

   (* 多項式解釈で必要な、x_i または x_i*x_i の係数が正という制約を表わす多項式 *)
   (* それぞれの x_i について，...+ f_j *x_i + f_k*x_i*xi ..+  => f_j + f_k を返す *)
   fun mkFunPolyCond1 (f,arity,poly) = 
       let 
(*	   val _ = print ((Fun.toString f) ^ ":\n") *)
	   val ipplist = IPP.toList poly
           (* 含まれる変数が x_k だけのものを取り出す *)
	   fun uniq k [] = false (* no constants *)
	     | uniq k (x::[]) = x = k
	     | uniq k (x::xs) = x = k andalso uniq k xs
(*       in LU.mapAppend (fn k => L.mapPartial (fn (xs,ip) => if uniq k xs
							   then SOME ip else NONE) ipplist)
		      (L.tabulate (arity,fn i=>i+1)) *)
	   fun makePoly k = 
	       let (* val _ = print ((Int.toString k) ^ ":\n") *)
		   val ip =
			IP.fromList
			    (L.concat
				(L.mapPartial (fn (xs,ip) => if uniq k xs
							     then SOME (IP.toList ip) 
							     else NONE) ipplist))
(*		   val _ = print (IP.toString ip) *)
(*		   val _ = print "\n" *)
	       in ip
	       end
       in L.tabulate (arity,fn i=>makePoly (i+1))
       end

   (* すべての多項式について，係数が正の制約を集めたリスト，要素は > 0 を満たす必要がある *)
   fun mkFunPolyCond fpassoc =  LU.mapAppend mkFunPolyCond1 fpassoc

  (* 項の解釈  vnames = [x,y,...] のとき、
         1  2   ...
         x  y   ...
    と変数番号をつける *)
   fun termToIPP fpassoc vnames (Var (x,_)) = 
       let val vnum = 1 + (valOf (LU.indexOf' Var.equal x vnames))
       in IPP.fromList [([vnum], IP.one)]
       end
     | termToIPP fpassoc vnames (Fun (f,ts,_)) =  
       let val (_,_,fpoly) = lookup fpassoc f
	   val args = L.map (termToIPP fpassoc vnames) ts
       in IPP.apply fpoly args
       end

(*
   val t1 = IOFotrs.rdTerm "f(?x,a)"
   val fpassoc = makeFunPolyAssoc (Term.funArityMapInTerm t1)
   val vnames = VarSet.listItems (Term.varSetInTerm t1)
   val poly = termToIPP fpassoc vnames t1
   val _ = print (IPP.toString poly)
*)


   (* 整係数多項式 (intPolyList1,intPoly2) s.t.
           intPolyList1 :  メタ多項式 l - r の定数以外の係数になっている多項式
           intPoly2 :  メタ多項式 l - r の定数になっている多項式
      を返す  *)

   fun encodeRule fpassoc (l,r) =
       let 
	   (*  val _ = print ((Term.toString l) ^ " -> " ^ (Term.toString r) ^ "\n")  *)
           val vnames = VarSet.listItems (Term.varSetInTerms [l,r])
	   val lpoly = termToIPP fpassoc vnames l
	   val rpoly = termToIPP fpassoc vnames r

	   val ipp = IPP.plus (lpoly, IPP.minus rpoly)
           (* val _ = print "meta poly for l - r:\n"   *)
	   (* val _ = print ((IPP.toString ipp) ^ "\n")  *)

	   (* 多項式の係数 *)
	   (* val _ = print "coefficients parts of l - r poly:\n"    *)
	   (* val _ = L.app (fn ip=> print ((IP.toString ip) ^ " [coef]\n")) intPolyList1 *)
	      val ipps = IPP.toList ipp
	   (* val _ = print "all coefficients:\n"    *)
	   (* val _ = L.app (fn (xs,ip) => 
			     print ((LU.toStringCommaCurly Int.toString xs)
			      ^ "  "
			      ^ (IP.toString ip) ^ "\n")) ipps *)
	   (* val intPolyList1 = IPP.coefficients ipp *)
	   val intPolyList1 = L.mapPartial (fn (xs,ip) => if null xs then NONE else SOME ip)
					   ipps
	   (* 定数項 *)
	   val intPoly2 = IPP.constant ipp
           (* val _ = print "constant part of l - r poly:\n"    *)
	   (* val _ = print ((IP.toString intPoly2) ^ " [const]\n")  *)
       in (intPolyList1,intPoly2) 
       end


   (*** decoding part ***)
   (* 割り当てのlookup by ブール値ベクトル *)
   fun lookupIntBvec assign (i,j) = 
       case StringMap.find (assign,EIP.intBvec (i,j))
	of NONE => false (* no value may be assinged *)
	 | SOME ans => valOf (Bool.fromString ans)

   (* 割り当てのlookup by 整数 *)
   fun lookupInt assign i = 
       EIP.bvecToInt (rev (L.tabulate (EIP.bitLength,
				    fn j=>lookupIntBvec assign (i,j))))

   (* 割り当てに基づく変換：多項式係数の多項式 => 整数係数の多項式 *)
   (* intPolyPolyToIntPoly: 
              assign (of f) = {1->2, 2->3, 3->1} 
              intPolyPoly = (2*f_1*x_1 + f_2*x_1*x_2 + f_3) = 
              = { [1]->f_1*2, [1,2]->f_2, [] -> f3 }
              = { [1]->{[1]->2]}, [1,2]->{[2]->1}, []->{[3]->1} }
              => { [1]->{2*2}, [1,2]->{3*1}, []->{1*1} }
              = { [1]->4, [1,2]->3, []->1 }
              = 4*x_1 + 3*x1*x_2 + 1 = intPoly
    *)
   fun intPolyPolyToIntPoly assign intPolyPoly =
       let val xs = IPP.toList intPolyPoly
     	   fun env i = lookupInt assign i
	   val poly = L.map (fn (xs,poly)=>(xs,IP.eval env poly)) xs
       in IP.fromList poly
       end

   (* 割り当てから関数記号の解釈をあたえる fpassoc を構成 *)
   (* fpassoc2 で参照する *)
   fun mkPolyIntepretation assign fpassoc =
       L.map (fn (f,arity,ipp) =>
		 let val ip = intPolyPolyToIntPoly assign ipp
		 in (f,arity,ip)
		 end) fpassoc


   (* 関数記号の解釈から項の解釈を構成 *)
   fun termToIP fpassoc2 vnames (Var (x,_)) = 
       let val vnum = valOf (LU.indexOf' Var.equal x vnames)
       in IP.fromList [([vnum+1], 1)]
       end
     | termToIP fpassoc2 vnames (Fun (f,ts,_)) =  
       let val (_,_,poly) = lookup fpassoc2 f
	   val args = L.map (termToIP fpassoc2 vnames) ts
       in IP.apply poly args
       end

   (* 関数記号の解釈から書き換え規則の解釈([l] - [r]を構成 *)
   fun transRule fpassoc2 (l,r) =
       let val vnames = VarSet.listItems (Term.varSetInTerms [l,r])
	   val lpoly = termToIP fpassoc2 vnames l
	   val rpoly = termToIP fpassoc2 vnames r
	   val diffpoly = IP.plus (lpoly, IP.minus rpoly)
       in diffpoly
       end

  (* 関数記号の解釈を表示 *)
   fun prPolyInterpretation fpassoc2 =
       (print "Polynomial Interpretation:\n";
	L.app (fn (f,arity,ip) =>
		  let val _ = print ("  " ^ (Fun.toString f) ^ ":= ")
		      val _ = print (IP.toString ip)
		      val _ = print "\n"
		  in ()
		  end) fpassoc2
       )

  (* 書き換え規則の解釈 [l] - [r] の整係数多項式を表示 *)
  (* 実際の解釈より多項式は近似されている場合があることに注意。
     (左辺のcarry over を無視するため) *)
   fun prPolyForRules fmassoc2 lookup R rs =
       let fun prPoly n =
	   let val (l,r) = L.nth (R,n)
	       val poly = transRule fmassoc2 (l,r)
	       val isNotStrict = lookup n
	       val _ = print (" [" ^ (Term.toString l) ^ "]-[" 
			      ^ (Term.toString r) ^ "] >= " ^ (IP.toString poly) 
			      ^ (if isNotStrict then " >= 0\n" else " > 0\n"))
	   in ()
	   end
       in L.app prPoly rs
       end

  (* polynomial interpretation ソルバ *)
  fun prepareCheck outs (R,S) =
      let
	   val _ = debug (fn _ => print "start of preparation...\n")
           val faMap = Trs.funArityMapInRules (R @ S)
	   val _ = EIP.init ()

	   val fpassoc = makeFunPolyAssoc MIX faMap

	   val _ = debug (fn _ =>
			     (L.app (fn (f,arity,ipp) =>
					let val _ = print ("  " ^ (Fun.toString f) ^ ":= ")
					    val _ = print (IPP.toString ipp)
					    val _ = print "\n"
					in () end) fpassoc))

	   val declFun = L.map (EIP.encodeGtEquation outs) (mkFunPolyCond fpassoc)

	   val (gePolyListR,gtPolyR) = LP.unzip (L.map (encodeRule fpassoc) R)
	   val (gePolyListS,gtPolyS) = LP.unzip (L.map (encodeRule fpassoc) S)

	   val geDeclsListR = L.map (L.map (EIP.encodeGeEquation outs)) gePolyListR
	   val geDeclsListS = L.map (L.map (EIP.encodeGeEquation outs)) gePolyListS

	   val gtDeclsR = L.map (EIP.encodeGtEquation outs) gtPolyR
	   val gtDeclsS = L.map (EIP.encodeGtEquation outs) gtPolyS
(*	   val eipCond = (!EIP.lines)  *)
	   val eipCond = ""
	   val _ = EIP.init ()
	   val _ = debug (fn _=> print "... end of preparation\n")
      in
	  (fpassoc,eipCond,declFun,geDeclsListR,geDeclsListS,gtDeclsR,gtDeclsS)
      end


   fun step (ins,outs) numStep spec (R,S) (rs,ss) =
      let
	  val (fpassoc,_,_,_,_,gtDeclsR,gtDeclsS) = spec

	  val _ = debug (fn _  => println ("Step " ^ (Int.toString numStep)))
	  val _ = debug (fn _ => println ("R = " ^ (prSetInOneLine Int.toString rs)))
	  val _ = debug (fn _ => println ("S = " ^ (prSetInOneLine Int.toString ss)))

	  fun lookupR assign i =
	      case StringMap.find (assign, EIP.bvar (L.nth (gtDeclsR, i)))
	       of NONE => (print ("i'th value (R) not found???"); false)
		 | SOME ans => ans = YI.prFalse
	  fun lookupS assign i =
	      case StringMap.find (assign,EIP.bvar (L.nth (gtDeclsS, i)))
	       of NONE => (print ("i'th value (S) not found???"); false)
		| SOME ans => ans = YI.prFalse

	  fun retract i =  (TextIO.output (outs, YI.prRetract i); TextIO.flushOut outs)
	  fun retractR i = (print ("retract " ^ (Trs.prRule (L.nth (R,i)))^ "\n");
			    retract (i + 1))
	  fun retractS i = (print ("retract " ^ (Trs.prRule (L.nth (S,i))) ^ "\n");
			    retract ((L.length R) + i + 1))
	  fun retractStep i = ((* print ("retract step " ^ (Int.toString i) ^ "\n"); *)
			       retract ((L.length R) + (L.length S) + i))
	  val listR = L.tabulate (L.length R, fn x=>x)
	  val listS = L.tabulate (L.length S, fn x=>x)
	  val _ = L.app (fn i => if LU.member i rs then () else retractR i) listR
	  val _ = L.app (fn i => if LU.member i ss then () else retractS i) listS

	  val gtCondR = YI.prAssertPlus (YI.prOr
					     (L.map (fn x=> EIP.bvar (L.nth (gtDeclsR,x))) rs))
	  val input = (gtCondR ^ "(check)\n" )
	  val _ = TextIO.output (outs, input)
	  val _ = debug (fn _=> print ("input:" ^ input))
	  val _ = TextIO.flushOut outs

	  val answer = TextIO.inputLine ins
	  val _ = if (isSome answer) 
                  then debug (fn _=> print ("output:" ^ (valOf answer)))
                  else ()

	  val result =  (isSome answer) andalso (valOf answer = "sat\n")

      in if not result 
	 then let val _ = debug (fn _=> print "No suitable polynomial interpretation found ...\n")
	      in NONE
	      end
	 else let 
		 val assign = YI.readAssigns ins
(*		 val _ = print "assignment:\n" *)
(*		 val _ = print (LU.toStringCommaSquare 
				    (fn i => Int.toString (lookupInt assign i)) 
				    [1,2,3,4,5,6,7,8,9,10,11,12,13]) *)
(*		 val _ = print "\n" *)
		 val fpassoc2 = mkPolyIntepretation assign fpassoc
		 val _ = prPolyInterpretation fpassoc2
		 val _ = debug (fn _=> print "polynomial(R):\n")
		 val _ = debug (fn _=> prPolyForRules fpassoc2 (lookupR assign) R rs)
		 val _ = debug (fn _=> print "polynomial(S):\n")
		 val _ = debug (fn _=> prPolyForRules fpassoc2 (lookupS assign) S ss)

		 val rs2 = L.filter (lookupR assign) rs 
		 val ss2 = L.filter (lookupS assign) ss
		 val _ = retractStep numStep
						
	     in SOME (rs2, ss2)
	     end
      end

  fun directCheck smtSolver (R,S) =
      let 
	  val listR = L.tabulate (L.length R, fn x=>x)
	  val listS = L.tabulate (L.length S, fn x=>x)

	 (* val cmd = smtSolver ^ "\n" *)
	  val cmd = smtSolver ^ " 2>/dev/null\n"
 	  val proverCmd = ("/bin/sh", ["-c",cmd])
	  val proc = Unix.execute proverCmd
	  val (ins,outs) = Unix.streamsOf proc

(*******************
	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
	  val outs2 = TextIO.openOut inTmpFile
          val _ = print ("tmpfile: " ^ tmpName ^ "\n")
          val _ = prepareCheck outs2 (R,S)
******************************************)

          val spec = prepareCheck outs (R,S)
	  val (_,eipCond,declFun,geDeclsListR,geDeclsListS,_,_) = spec

	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

	  val condF = YI.prAssert (YI.prAnd (L.map EIP.bvar declFun))
	  (* ids of asssert plus starts from 1 *)
	  val condR = String.concat (L.map (fn x => assert (L.nth (geDeclsListR,x))) listR)
	  val condS = String.concat (L.map (fn x => assert (L.nth (geDeclsListS,x))) listS)

	  val _ = TextIO.output (outs, eipCond ^ condF ^ condR ^ condS
				       (* ^ "(set-verbosity! 0)\n" *)
				       ^ "(set-evidence! true)\n")
	  val _ = TextIO.flushOut outs

(**
	  val _ = TextIO.output (outs2, eipCond ^ condF ^ condR ^ condS
				       (* ^ "(set-verbosity! 0)\n" *)
				       ^ "(set-evidence! true)\n")
**)

	  (* プロセス終了処理*)
	  fun finish () = 
	      let val _ = debug (fn _=> print "finish the yices process\n")
		  val _ = TextIO.closeOut outs
		  val _ = TextIO.closeIn ins
		  val _ = Unix.reap proc  
	      in () end

	  fun try ([],_) numStep = (finish (); true)
	    | try (rs,ss) numStep = 
	      case step (ins,outs) numStep spec (R,S) (rs,ss) of
		  NONE => (finish (); false)
		| SOME (rs2,ss2) => try (rs2,ss2) (numStep+1) 
      in if null R
	 then (finish ();true)
	 else try (listR,listS) 1 (* should start from '1' *)
      end

  (* polynomial interpretation ソルバ *)
  fun prepareCheckEvenOdd outs (term0,term1) R =
      let
	   val _ = debug (fn _ => print "start of preparation...\n")
           val faMap = Trs.funArityMapInRules ((term0,term1)::R)
	   val _ = EIP.init ()
	   val _ = init ()

	   val fpassoc = makeFunPolyAssoc LIN faMap

	   val _ = debug (fn _ =>
			     (L.app (fn (f,arity,ipp) =>
					let val _ = print ("  " ^ (Fun.toString f) ^ ":= ")
					    val _ = print (IPP.toString ipp)
					    val _ = print "\n"
					in () end) fpassoc))


	   (* l - rのメタ多項式の非定数係数，定数係数 *)
	   val (evPolyListR,evPolyR) = LP.unzip (L.map (encodeRule fpassoc) R)
	   val evDeclsListR = LU.mapAppend (L.map (EIP.encodeEvenEquation outs)) (evPolyR::evPolyListR)


	   (* term0 - term1 のメタ多項式の非定数係数，定数係数 *)
	   val (evPolyList,odPoly) = encodeRule fpassoc (term0,term1)

(*	   val _ = L.app (fn ip=> print ((IP.toString ip) ^ "\n")) evPolyList *)
(*	   val _ = print (LU.toStringCommaLnSquare IP.toString evPolyList) *)

	   val evDeclsList = L.map (EIP.encodeEvenEquation outs) evPolyList

	   val odDecl = EIP.encodeOddEquation outs odPoly

	   val _ = EIP.init ()
	   val _ = debug (fn _=> print "... end of preparation\n")
      in
	  (fpassoc,evDeclsListR,evDeclsList,odDecl)
      end

(* output certification proof *)
  fun outputMono (xs,n) = 
      let fun v () = CU.encloseProofLeafBy "natural" (Int.toString n)
	  fun getx x () = CU.encloseProofLeafBy "variable" (Int.toString x)
      in if null xs
	 then v
	 else (fn () => CU.encloseProofTreesBy "product" 
					       (L.map (fn g => 
						       fn () => 
							  CU.encloseProofTreeBy "arithFunction" g) 
						      (v::(L.map getx xs))))
      end

  fun outputPoly ip = 
      let val yss = IntPoly.toList ip
      in if null yss
	 then (fn () => CU.encloseProofLeafBy "natural" "0")
	 else if (L.length yss) = 1
	 then outputMono (hd yss)
	 else (fn () => 
		  CU.encloseProofTreesBy 
		      "sum"  
		      (L.map (fn (xs,n) => 
  			      fn () => CU.encloseProofTreeBy 
					   "arithFunction" 
					   (outputMono (xs,n))) yss))
      end

  fun outputIpt fpassoc =
      L.map  (fn (f,arity,ip) =>
	      fn () => CU.encloseProofTreesBy 
			   "interpret" 
			   [ fn () => CU.encloseProofLeafBy "name" (Fun.toString f),
			     fn () => CU.encloseProofLeafBy "arity" (Int.toString arity),
			     fn () => CU.encloseProofTreeBy "arithFunction" (outputPoly ip)])
             fpassoc


(*  fun njCheckEvenOdd (opt0:Solver.options) (term0,term1) R = *)
  fun njCheckEvenOdd smtSolver tmpDir (term0,term1) R =
      let (* val smtSolver = (#smtSolver opt0)
          val tmpDir = (#tmpDir opt0) *)

	  val listR = L.tabulate (L.length R, fn x=>x)

(**)	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
(**)	  val _ = debug (fn _=> print ("[" ^ tmpName ^ "]\n"))
(**)	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
(**)	  val outs = TextIO.openOut inTmpFile

          val spec = prepareCheckEvenOdd outs (term0,term1) R
	  val (fpassoc,evDeclsListR,evDeclsList,odDecl) = spec
	  
	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

	  val condReven = YI.prAssert (YI.prAnd (L.map EIP.bvar evDeclsListR))
	  val condEven = YI.prAssert (YI.prAnd (L.map EIP.bvar evDeclsList))
	  val condOdd = YI.prAssert (EIP.bvar odDecl)

(*	  val _ = TextIO.output (outs, condF ^ condE ^ condO *)
	  val _ = TextIO.output (outs, condReven ^ condEven ^ condOdd
				       ^ "(set-evidence! true)\n")
(*	  val _ = print condO *)
	  val _ = TextIO.flushOut outs

	  val input = "(check)\n"
	  val _ = TextIO.output (outs, input)
	  val _ = TextIO.flushOut outs

(***     val cmd = smtSolver ^ " 2>/dev/null\n" ***)
	  val cmd = smtSolver ^ " " ^ inTmpFile ^ " 2>&1\n"
 	  val _ = debug (fn _ => print cmd)
 	  val proverCmd = ("/bin/sh", ["-c",cmd])
	  val proc = Unix.execute proverCmd
	  val (ins,outs) = Unix.streamsOf proc

	  val answer = TextIO.inputLine ins
	  val _ = if (isSome answer) 
                  then debug (fn _=> print ("output:" ^ (valOf answer)))
                  else ()

	  val result =  (isSome answer) andalso (valOf answer = "sat\n")

	  (* プロセス終了処理*)
	  fun finish () = 
	      let val _ = debug (fn _=> print "finish the yices process\n")
		  val _ = TextIO.closeOut outs
		  val _ = TextIO.closeIn ins
		  val _ = Unix.reap proc  
		  val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 
	      in () end


      in if not result 
	 then let val _ = debug (fn _=> print "No suitable polynomial interpretation found ...\n")
	      in (finish (); (false, fn () => ""))
	      end
	 else let 
		 val _ =  print " (success)\n"
		 val assign = YI.readAssigns ins
		 val _ = debug (fn _ => print "assignment:\n")
		 val _ = debug (fn _ => L.app (fn (x,y) => print ("[" ^ x ^ "] |-> ["
						   ^ y ^ "]\n"))
			       (StringMap.listItemsi assign))

		 val fpassoc2 = mkPolyIntepretation assign fpassoc
		 val _ = prPolyInterpretation fpassoc2
		 fun poly (l,r) = transRule fpassoc2 (l,r)
		 val _ = L.app (fn (l,r) =>
				   print (" [" ^ (Term.toString l) ^ "]-[" 
					   ^ (Term.toString r) ^ "] = " 
					   ^ (IP.toString (poly (l,r)))
					   ^ " = 0 (mod 2)\n"))
			       R
(*		 val _ = print (" [" ^ (Term.toString term0) ^ "] = " 
				^ (IP.toString (termToIP fpassoc2 
							 (VarSet.listItems 
							      (Term.varSetInTerms 
								   [term0,term1])) term0))
				^ "\n")
		 val _ = print (" [" ^ (Term.toString term1) ^ "] = " 
				^ (IP.toString (termToIP fpassoc2 
							 (VarSet.listItems 
							      (Term.varSetInTerms 
								   [term0,term1])) term1))
				^ "\n") *)

		 val _ = print (" [" ^ (Term.toString term0) ^ "]-[" 
					   ^ (Term.toString term1) ^ "] = " 
					   ^ (IP.toString (poly (term0,term1)))
					   ^ " = 1 (mod 2)\n")

		 fun cpf () = 
		     (CU.encloseProofTreeBy "differentInterpretation"
		      (fn () => CU.encloseProofTreeBy "model" 
				(fn () => CU.encloseProofTreesBy "finiteModel" 
					  ([fn () => CU.encloseProofLeafBy "carrierSize" "2"]
					  @ outputIpt fpassoc2))))

	     in (finish (); (true, cpf))
	     end


      end

  (* polynomial interpretation ソルバ *)
  fun prepareCheckModuloThree outs (term0,term1) R =
      let
	   val _ = debug (fn _ => print "start of preparation...\n")
           val faMap = Trs.funArityMapInRules ((term0,term1)::R)
	   val _ = EIP.init ()
	   val _ = init ()

	   val fpassoc = makeFunPolyAssoc LIN faMap

	   val _ = debug (fn _ =>
			     (L.app (fn (f,arity,ipp) =>
					let val _ = print ("  " ^ (Fun.toString f) ^ ":= ")
					    val _ = print (IPP.toString ipp)
					    val _ = print "\n"
					in () end) fpassoc))

	   (* l - rのメタ多項式の非定数係数，定数係数 *)
	   val (meqPolyListR,meqPolyR) = LP.unzip (L.map (encodeRule fpassoc) R)

	   val meqDeclsListR = LU.mapAppend (L.map (EIP.encodeEqualToZeroByModThreeEquation outs))
					    (meqPolyR::meqPolyListR)


	   (* term0 - term1 のメタ多項式の非定数係数，定数係数 *)
	   val (meqPolyList,mneqPoly) = encodeRule fpassoc (term0,term1)

(*	   val _ = L.app (fn ip=> print ((IP.toString ip) ^ "\n")) evPolyList *)
(*	   val _ = print (LU.toStringCommaLnSquare IP.toString evPolyList) *)

	   val meqDeclsList = L.map (EIP.encodeEqualToZeroByModThreeEquation outs) meqPolyList

	   val mneqDecl = EIP.encodeEqualToNonZeroByModThreeEquation outs mneqPoly

	   val _ = EIP.init ()
	   val _ = debug (fn _=> print "... end of preparation\n")
      in
	  (fpassoc,meqDeclsListR,meqDeclsList,mneqDecl)
      end

(*  fun njCheckModuloThree (opt0:Solver.options) (term0,term1) R = *)
  fun njCheckModuloThree smtSolver tmpDir (term0,term1) R =
      let (* val smtSolver = (#smtSolver opt0)
          val tmpDir = (#tmpDir opt0) *)

	  val listR = L.tabulate (L.length R, fn x=>x)

(**)	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
(**)	  val _ = debug (fn _ => print ("[" ^ tmpName ^ "]\n"))
(**)	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
(**)	  val outs = TextIO.openOut inTmpFile

          val spec = prepareCheckModuloThree outs (term0,term1) R
	  val (fpassoc,meqDeclsListR,meqDeclsList,mneqDecl) = spec

	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

	  val condRmeq = YI.prAssert (YI.prAnd (L.map EIP.bvar meqDeclsListR))
	  val condMeq = YI.prAssert (YI.prAnd (L.map EIP.bvar meqDeclsList))
	  val condMneq = YI.prAssert (EIP.bvar mneqDecl)

	  val _ = TextIO.output (outs, condRmeq ^ condMeq ^ condMneq
				       ^ "(set-evidence! true)\n")
	  val _ = TextIO.flushOut outs

	  val input = "(check)\n"
	  val _ = TextIO.output (outs, input)
	  val _ = TextIO.flushOut outs

(***     val cmd = smtSolver ^ " 2>/dev/null\n" ***)
	  val cmd = smtSolver ^ " " ^ inTmpFile ^ " 2>&1\n"
 	  val _ = debug (fn _ => print cmd)
 	  val proverCmd = ("/bin/sh", ["-c",cmd])
	  val proc = Unix.execute proverCmd
	  val (ins,outs) = Unix.streamsOf proc

	  val answer = TextIO.inputLine ins
	  val _ = if (isSome answer) 
                  then debug (fn _=> print ("output:" ^ (valOf answer)))
                  else ()

	  val result =  (isSome answer) andalso (valOf answer = "sat\n")

	  (* プロセス終了処理*)
	  fun finish () = 
	      let val _ = debug (fn _=> print "finish the yices process\n")
		  val _ = TextIO.closeOut outs
		  val _ = TextIO.closeIn ins
		  val _ = Unix.reap proc  
		  val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 
	      in () end


      in if not result 
	 then let val _ = debug (fn _=> print "No suitable polynomial interpretation found ...\n")
	      in (finish (); (false, fn () => ""))
	     (* in (finish (); false) *)
	      end
	 else let 
		 val _ =  print " (success)\n"
		 val assign = YI.readAssigns ins
		 val _ = debug (fn _=> print "assignment:\n" )
		 val _ = debug (fn _=> L.app (fn (x,y) => print ("[" ^ x ^ "] |-> ["
						   ^ y ^ "]\n"))
			       (StringMap.listItemsi assign))

		 val fpassoc2 = mkPolyIntepretation assign fpassoc
		 val _ = prPolyInterpretation fpassoc2
		 fun poly (l,r) = transRule fpassoc2 (l,r)
		 val _ = L.app (fn (l,r) =>
				   print (" [" ^ (Term.toString l) ^ "]-[" 
					   ^ (Term.toString r) ^ "] = " 
					   ^ (IP.toString (poly (l,r)))
					   ^ " = 0 (mod 3)\n"))
			       R
		 val _ = print (" [" ^ (Term.toString term0) ^ "]-[" 
					   ^ (Term.toString term1) ^ "] = " 
					   ^ (IP.toString (poly (term0,term1)))
					   ^ " <> 0  (mod 3)\n")
		 fun cpf () = 
		     (CU.encloseProofTreeBy "differentInterpretation"
		      (fn () => CU.encloseProofTreeBy "model" 
				(fn () => CU.encloseProofTreesBy "finiteModel" 
					  ([fn () => CU.encloseProofLeafBy "carrierSize" "3"]
					  @ outputIpt fpassoc2))))

	     in (finish (); (true, cpf))
(*	     in (finish (); true) *)
	     end
      end


  (* polynomial interpretation ソルバ *)
  fun prepareCheckModuloFour outs (term0,term1) R =
      let
	   val _ = debug (fn _ => print "start of preparation...\n")
           val faMap = Trs.funArityMapInRules ((term0,term1)::R)
	   val _ = EIP.init ()
	   val _ = init ()

	   val fpassoc = makeFunPolyAssoc LIN faMap

	   val _ = debug (fn _ =>
			     (L.app (fn (f,arity,ipp) =>
					let val _ = print ("  " ^ (Fun.toString f) ^ ":= ")
					    val _ = print (IPP.toString ipp)
					    val _ = print "\n"
					in () end) fpassoc))

	   (* l - rのメタ多項式の非定数係数，定数係数 *)
	   val (meqPolyListR,meqPolyR) = LP.unzip (L.map (encodeRule fpassoc) R)
	   val meqDeclsListR = LU.mapAppend (L.map (EIP.encodeMeqEquation outs)) (meqPolyR::meqPolyListR)

	   (* term0 - term1 のメタ多項式の非定数係数，定数係数 *)
	   val (meqPolyList,mneqPoly) = encodeRule fpassoc (term0,term1)

(*	   val _ = L.app (fn ip=> print ((IP.toString ip) ^ "\n")) evPolyList *)
(*	   val _ = print (LU.toStringCommaLnSquare IP.toString evPolyList) *)

	   val meqDeclsList = L.map (EIP.encodeMeqEquation outs) meqPolyList
	   val mneqDecl = EIP.encodeMneqEquation outs mneqPoly

	   val _ = EIP.init ()
	   val _ = debug (fn _=> print "... end of preparation\n")
      in
	  (fpassoc,meqDeclsListR,meqDeclsList,mneqDecl)
      end


(*  fun njCheckModuloFour (opt0:Solver.options) (term0,term1) R = *)
  fun njCheckModuloFour smtSolver tmpDir (term0,term1) R =
      let (* val smtSolver = (#smtSolver opt0)
          val tmpDir = (#tmpDir opt0) *)

	  val listR = L.tabulate (L.length R, fn x=>x)

(**)	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
(**)	  val _ = print ("[" ^ tmpName ^ "]\n")
(**)	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
(**)	  val outs = TextIO.openOut inTmpFile

          val spec = prepareCheckModuloFour outs (term0,term1) R
	  val (fpassoc,meqDeclsListR,meqDeclsList,mneqDecl) = spec

	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

	  val condRmeq = YI.prAssert (YI.prAnd (L.map EIP.bvar meqDeclsListR))
	  val condMeq = YI.prAssert (YI.prAnd (L.map EIP.bvar meqDeclsList))
	  val condMneq = YI.prAssert (EIP.bvar mneqDecl)

(*	  val _ = TextIO.output (outs, condF ^ condE ^ condO *)
	  val _ = TextIO.output (outs, condRmeq ^ condMeq ^ condMneq
				       ^ "(set-evidence! true)\n")
(*	  val _ = print condO *)
	  val _ = TextIO.flushOut outs

	  val input = "(check)\n"
	  val _ = TextIO.output (outs, input)
	  val _ = TextIO.flushOut outs

(***     val cmd = smtSolver ^ " 2>/dev/null\n" ***)
	  val cmd = smtSolver ^ " " ^ inTmpFile ^ " 2>&1\n"
 	  val _ = debug (fn _ => print cmd)
 	  val proverCmd = ("/bin/sh", ["-c",cmd])
	  val proc = Unix.execute proverCmd
	  val (ins,outs) = Unix.streamsOf proc

	  val answer = TextIO.inputLine ins
	  val _ = if (isSome answer) 
                  then debug (fn _=> print ("output:" ^ (valOf answer)))
                  else ()

	  val result =  (isSome answer) andalso (valOf answer = "sat\n")

	  (* プロセス終了処理*)
	  fun finish () = 
	      let val _ = debug (fn _=> print "finish the yices process\n")
		  val _ = TextIO.closeOut outs
		  val _ = TextIO.closeIn ins
		  val _ = Unix.reap proc  
		  val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 
	      in () end


      in if not result 
	 then let val _ = debug (fn _=> print "No suitable polynomial interpretation found ...\n")
	      in (finish (); false)
	      end
	 else let 
		 val _ =  print " (success)\n"
		 val assign = YI.readAssigns ins
		 val _ = debug (fn _ => print "assignment:\n")
		 val _ = debug (fn _=> L.app (fn (x,y) => print ("[" ^ x ^ "] |-> ["
						   ^ y ^ "]\n"))
			       (StringMap.listItemsi assign))

		 val fpassoc2 = mkPolyIntepretation assign fpassoc
		 val _ = prPolyInterpretation fpassoc2
		 fun poly (l,r) = transRule fpassoc2 (l,r)
		 val _ = L.app (fn (l,r) =>
				   print (" [" ^ (Term.toString l) ^ "]-[" 
					   ^ (Term.toString r) ^ "] = " 
					   ^ (IP.toString (poly (l,r)))
					   ^ " = 0 (mod 4)\n"))
			       R
(*		 val _ = print (" [" ^ (Term.toString term0) ^ "] = " 
				^ (IP.toString (termToIP fpassoc2 
							 (VarSet.listItems 
							      (Term.varSetInTerms 
								   [term0,term1])) term0))
				^ "\n")
		 val _ = print (" [" ^ (Term.toString term1) ^ "] = " 
				^ (IP.toString (termToIP fpassoc2 
							 (VarSet.listItems 
							      (Term.varSetInTerms 
								   [term0,term1])) term1))
				^ "\n")  *)

		 val _ = print (" [" ^ (Term.toString term0) ^ "]-[" 
					   ^ (Term.toString term1) ^ "] = " 
					   ^ (IP.toString (poly (term0,term1)))
					   ^ " <> 0  (mod 4)\n")
	     in (finish (); true)
	     end
      end


  (* polynomial interpretation ソルバ *)
  fun prepareCheckByOrder outs (term0,term1) R =
      let
	   val _ = debug (fn _ => print "start of preparation...\n")
           val faMap = Trs.funArityMapInRules ((term0,term1)::R)
	   val _ = EIP.init ()
	   val _ = init ()

	   val fpassoc = makeFunPolyAssoc LIN faMap

	   val _ = debug (fn _ =>
			     (L.app (fn (f,arity,ipp) =>
					let val _ = print ("  " ^ (Fun.toString f) ^ ":= ")
					    val _ = print (IPP.toString ipp)
					    val _ = print "\n"
					in () end) fpassoc))

           (* weakly monotonic になるための条件 *)
	   val declFun = L.map (EIP.encodeGeEquation outs) (mkFunPolyCond fpassoc)

	   (* l - rのメタ多項式の非定数係数，定数係数 *)
	   val (polyListR,polyR) = LP.unzip (L.map (encodeRule fpassoc) R)

           (* l >= r となるための条件 *)
	   val geDeclsListR = LU.mapAppend (L.map (EIP.encodeGeEquation outs)) (polyR::polyListR)

	   (* term0 - term1 のメタ多項式の非定数係数，定数係数 *)
	   val (polyList0,poly0) = encodeRule fpassoc (term0,term1)

	   val geDeclsList = L.map (EIP.encodeGeEquation outs) polyList0
	   val gtDecl = EIP.encodeGtEquation outs poly0

	   val _ = EIP.init ()
	   val _ = debug (fn _=> print "... end of preparation\n")
      in
	  (fpassoc,declFun,geDeclsListR,geDeclsList,gtDecl)
      end

(*  fun njCheckByOrder (opt0:Solver.options) (term0,term1) R = *)
  fun njCheckByOrder smtSolver tmpDir (term0,term1) R =
      let 
          (* val smtSolver = (#smtSolver opt0)
          val tmpDir = (#tmpDir opt0) *)

	  val listR = L.tabulate (L.length R, fn x=>x)

	  val cmd = smtSolver ^ " 2>/dev/null\n"
 	  val proverCmd = ("/bin/sh", ["-c",cmd])
	  val proc = Unix.execute proverCmd
	  val (ins,outs) = Unix.streamsOf proc

	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
	  val _ = debug (fn _=> print ("[" ^ tmpName ^ "]\n"))
	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
	  val outsP = TextIO.openOut inTmpFile

          val spec = prepareCheckByOrder outs (term0,term1) R

	  val (fpassoc,declFun,geDeclsListR,geDeclsList,gtDecl) = spec
	  
	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

	  val condF = YI.prAssert (YI.prAnd (L.map EIP.bvar declFun))
	  val condR = YI.prAssert (YI.prAnd (L.map EIP.bvar geDeclsListR))
	  val condGe = YI.prAssert (YI.prAnd (L.map EIP.bvar geDeclsList))
	  val condGt = YI.prAssert (EIP.bvar gtDecl)

	  val _ = TextIO.output (outs, condF ^ condR ^ condGe ^ condGt
				       ^ "(set-evidence! true)\n")
	  val _ = TextIO.flushOut outs

	  (* プロセス終了処理*)
	  fun finish () = 
	      let val _ = debug (fn _=> print "finish the yices process\n")
		  val _ = TextIO.closeOut outs
		  val _ = TextIO.closeIn ins
		  val _ = Unix.reap proc  
		  val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 
	      in () end

	  val input = "(check)\n"
	  val _ = TextIO.output (outs, input)
	  val _ = TextIO.flushOut outs

	  val answer = TextIO.inputLine ins
	  val _ = if (isSome answer) 
                  then debug (fn _=> print ("output:" ^ (valOf answer)))
                  else ()

	  val result =  (isSome answer) andalso (valOf answer = "sat\n")


      in if not result 
	 then let val _ = debug (fn _=> print "No suitable polynomial interpretation found ...\n")
	      in (finish (); (false, fn () => ""))
	      end
	 else let 
		 val _ =  print " (success)\n"
		 val assign = YI.readAssigns ins
		 val _ = debug (fn _=> print "assignment:\n" )
		 val _ = debug (fn _=> L.app (fn (x,y) => print ("[" ^ x ^ "] |-> ["
						   ^ y ^ "]\n"))
			       (StringMap.listItemsi assign))

		 val fpassoc2 = mkPolyIntepretation assign fpassoc
		 val _ = prPolyInterpretation fpassoc2
		 fun poly (l,r) = transRule fpassoc2 (l,r)
		 val _ = L.app (fn (l,r) =>
				   print (" [" ^ (Term.toString l) ^ "]-[" 
					   ^ (Term.toString r) ^ "] = " 
					   ^ (IP.toString (poly (l,r)))
					   ^ " >= 0\n"))
			       R
		 val _ = print (" [" ^ (Term.toString term0) ^ "]-[" 
					   ^ (Term.toString term1) ^ "] = " 
					   ^ (IP.toString (poly (term0,term1)))
					   ^ " > 0\n")

		 fun prType () = 
		     CU.encloseProofTreeBy "type"
		     (fn _ => CU.encloseProofTreesBy "polynomial"
			      [fn _ => CU.encloseProofTreeBy "domain" 
				       (fn _ => CU.encloseProofLeafBy "naturals" ""),
			       fn _ => CU.encloseProofLeafBy "degree" "1"])

		 fun outputMono2 (xs,n) = 
		     let fun v () = CU.encloseProofTreeBy "coefficient"
				(fn () => CU.encloseProofLeafBy "integer" (Int.toString n))

			 fun getx x () = CU.encloseProofLeafBy "variable" (Int.toString x)
		     in if null xs
			then v
			else (fn () => CU.encloseProofTreesBy "product" 
							      (L.map (fn g => 
						       fn () => 
							  CU.encloseProofTreeBy "polynomial" g) 
						      (v::(L.map getx xs))))
		     end

		 fun outputPoly2 ip = 
		     let val yss = IntPoly.toList ip
		     in if null yss
			then (fn () => CU.encloseProofTreeBy "coefficient"
				     (fn () => CU.encloseProofLeafBy "integer"  "0"))
			else if (L.length yss) = 1
			then outputMono2 (hd yss)
			else (fn () => 
				 CU.encloseProofTreesBy 
				     "sum"  
				     (L.map (fn (xs,n) => 
  					     fn () => CU.encloseProofTreeBy 
							  "polynomial" 
					   (outputMono2 (xs,n))) yss))
		     end

		 fun outputIpt2 fpassoc =
		     L.map  (fn (f,arity,ip) =>
			     fn () => CU.encloseProofTreesBy 
					  "interpret" 
					  [ fn () => CU.encloseProofLeafBy "name" (Fun.toString f),
					    fn () => CU.encloseProofLeafBy "arity" (Int.toString arity),
					    fn () => CU.encloseProofTreeBy "polynomial" (outputPoly2 ip)])
			    fpassoc

		 fun cpf () = 
		     CU.encloseProofTreeBy "strictDecrease"
		     (fn _ => CU.encloseProofTreeBy "orderingConstraintProof"
			      (fn _ => CU.encloseProofTreeBy "redPair" 
				 (fn _ => CU.encloseProofTreesBy "interpretation" 
								 (prType::(outputIpt2 fpassoc2)))))

	     in (finish (); (true, cpf))
	     (* in (finish (); true) *)
	     end
      end

	    
(*
   val _ = if njCheckEvenOdd "/usr/local/bin/yices"
			     (IOFotrs.rdTerm "h(c)", IOFotrs.rdTerm "h(h(c))")
			     (IOFotrs.rdRules ["a -> h(c)", "a -> h(f(c))", "h(?x) -> h(h(?x))", "f(?x) -> f(g(?x))"])
	   then print "non-confluence proved\n"
	   else print "non-confluence unknown\n"
*)

(*   val _ = if njCheckEvenOdd "/usr/local/bin/yices"
			     (IOFotrs.rdTerm "f(a,b)", IOFotrs.rdTerm "f(f(a,b),b)")
			     (IOFotrs.rdRules [ "a -> f(a,b)", "f(a,b) -> f(b,a)" ])
	   then print "non-confluence proved\n"
	   else print "non-confluence unknown\n" *)



(*   val result2 = directCheck2 "/usr/local/bin/yices"
			    (IOFotrs.rdTerm "f(c)", IOFotrs.rdTerm "h(c)")
			    (IOFotrs.rdRules ["f(?x) -> h(g(?x))", "h(?x) -> f(g(?x))"]) *)

(*
   val result2 = directCheck2 "/usr/local/bin/yices"
			    (IOFotrs.rdTerm "f(h(c))", IOFotrs.rdTerm "h(c)")
			    (IOFotrs.rdRules ["h(?x) -> h(h(?x))", "f(?x) -> f(g(?x))"]) *)

(*
   val _ = if result2
	   then print "non-confluence proved\n"
	   else print "non-confluence unknown\n"
*)


(*
val result = directCheck "../../work/tools/bin/yices"
			    (IOFotrs.rdRules ["plus(s(?x),?y) -> s(plus(?x,?y))",
					      "plus(?x,s(?y)) -> s(plus(?x,?y))"],
			     IOFotrs.rdRules [ "plus(?x,?y) -> plus(?y,?x)",
						   "plus(?x,plus(?y,?z)) -> plus(plus(?x,?y), ?z)" ])
*)

(*
val result = directCheck "../../work/tools/bin/yices"
			    (IOFotrs.rdRules ["h(?x,?y,b) -> h(?x,?y,?y)",
					      "h(g,a,a) -> h(f,a,a)",
					      "h(?x,b,?y) -> h(?x,?y,?y)",
					      "g -> f"],
			     IOFotrs.rdRules [])
*)

(*
val result = directCheck "../../work/tools/bin/yices"
			    (IOFotrs.rdRules    [ "plus(s(?x),?y) -> plus(?x,s(?y))",
     "plus(?x,s(?y)) -> plus(s(?x),?y)",
     "times(?x,s(?y)) -> plus(?x,times(?x,?y))",
     "times(s(?x),?y) -> plus(times(?x,?y),?y)",
     "sq(?x) -> times(?x,?x)",
     "sq(s(?x)) -> plus(times(?x,?x),s(plus(?x,?x)))" ],
			    IOFotrs.rdRules    [ "plus(?x,plus(?y,?z)) -> plus(plus(?x,?y),?z)",
     "plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))",
     "plus(?x,?y) -> plus(?y,?x)",
     "times(?x,?y) -> times(?y,?x)" ])
*)


(*
   val result = directCheck "../../work/tools/bin/yices"
			    (IOFotrs.rdRules [ "f(?x) -> ?x" 
					     (* "plus(0,?y) -> ?y",
					      "plus(s(?x),?y) -> s(plus(?x,?y))",
					      "plus(?x,0) -> ?x",
					      "plus(?x,s(?y)) -> s(plus(?x,?y))" *) ],
			     IOFotrs.rdRules [ "a -> g(a)" 
                                              (* "plus(?x,?y) -> plus(?y,?x)",
					      (* "s(s(?x)) -> s(?x)", *)
					      (* "s(?x) -> s(s(?x))", *)
					      "plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))" *) ])
*)

(*
   val result = directCheck "./yices" "./tmp"
			    (IOFotrs.rdRules ["f(f(?x)) -> f(g(f(?x)))",
					      "f(?x) -> g(?x)"],
			     IOFotrs.rdRules ["f(?x) -> g(f(?x))"])
*)
(*
   val result = directCheck "./yices" "./tmp"
			    (IOFotrs.rdRules ["f(a,g(?y)) -> f(?y,a)"],
			     IOFotrs.rdRules [])
*)
(*
   val _ = if result 
	   then print "relatively terminating\n"
	   else print "not relatively terminating\n"

*)

   end

end

signature MATRIX_INTERPRETATION =
sig
    val runDebug: bool ref
   val prepareCheck:
       TextIO.outstream ->
           ((Term.term * Term.term) list * (Term.term * Term.term) list)
          -> (Fun.ord_key * IntPolyPolyVec.vec * IntPolyPolyMatrix.matrix list)
               list * string * int list * int list list * int list list * 
             int list * int list


   val step: TextIO.instream * TextIO.outstream -> int 
             -> (Fun.ord_key * IntPolyPolyVec.vec * IntPolyPolyMatrix.matrix list)
		    list * string * int list * int list list * int list list * 
		int list * int list
             -> ((Term.term * Term.term) list  * (Term.term * Term.term) list)
             -> (int list * int list) -> (int list * int list) option

  val directCheck: string 
		   -> ((Term.term * Term.term) list * (Term.term * Term.term) list)
		   -> bool

  val relsnSolver: Solver.options
                   -> DpSolver.options
                   -> PoSolver.options
                   -> ((Term.term * Term.term) list 
                       * (Term.term * Term.term) list)
                   -> bool
end

structure MatrixInterpretation : MATRIX_INTERPRETATION =
struct
   local
       open Term
       structure EIP = EncodeIntPoly
       structure FM = FunMap
       structure IM = IntMap
       structure ILM = IntListMap
       structure IP = IntPoly
       structure IPP = IntPolyPoly
(*     *** use here to use dimension 3 *** 
       structure IPV = IntPolyVec3
       structure IPM = IntPolyMatrix3
       structure IPPM = IntPolyPolyMatrix3
       structure IPPV = IntPolyPolyVec3 *)
(** **)
(*     *** use here to use dimension 2 *** *)
       structure IPV = IntPolyVec
       structure IPM = IntPolyMatrix
       structure IPPM = IntPolyPolyMatrix
       structure IPPV = IntPolyPolyVec

(** **)
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure YI = YicesInput
       structure PI = PolynomialInterpretation
       open PrintUtil
   in
   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()

(*   val dimension = 3  *)
   val dimension = 2

   val counter = ref 0
   (* 多項式 "f_i" の生成 *)
   fun newVarPoly () = (counter := (!counter)+1; 
			IPP.fromList [([],IP.fromList [([!counter],1)])])
   (* リスト [f_1,f_2] 等の生成 *)
   fun newVarList () = L.tabulate (dimension,fn i=>newVarPoly ())
   (* ベクトル [f_1,f_2] 等の生成 *)
   fun newVarVec () = IPPV.fromList (newVarList ())
   (* 行列 [[f_1,f_2],[f_3,f_4]] 等の生成 *)
   fun newVarMatrix () = IPPM.fromList (L.tabulate (dimension,fn i=>newVarList ()))

   (* 関数記号fにたいして、
      新しい変数 f_i,f_{i+1}を用いて
      ベクトル vec_f = [f_i,f_{i+1}] を割り当てる。

      関数記号とその引数1,...,nの対 (f,i)にたいして、
      新しい変数f_i,...,f_{i+3}を用いて、
      行列 A_{(f,i)} = [[f_i,f_{i+1}],[f_{i+2},f_{i+3}]] を割り当てる。

      リスト [ (f,vec_f,[A_{f,i}| 1 \le i \le arity(f)]) | f \in F ] を構成する
      このリストは fmassoc と参照される *)

   fun makeFunMatrixAssoc faMap = 
       L.map (fn (f,arity) => (f,newVarVec (),
			       L.tabulate (arity, fn x=> newVarMatrix ())))
	     (FM.listItemsi faMap)

   (* fmassoc の lookup *)
   fun lookup fmassoc f = valOf (L.find (fn (g,_,_) => Fun.equal (f,g)) fmassoc)

   (* 行列解釈で必要な、正方行列の第(1,1)要素が正という条件：
      すべての多項式が \ge 0 となれば条件を満たすような多項式リスト
      を返す  *)
   fun mkMatrixPoly fmassoc = 
       L.map (fn A=> IP.plus (IPP.constant (valOf (IPPM.element A (1,1))), 
			      IP.minus IP.one))
	     (LU.mapAppend (fn (_,_,ms) => ms) fmassoc)

  (* 項の解釈
     vnames = [x,y,...] のとき、例えば、dimention 2 なら、
         1   2   3  4  ...
         x_1 x_2 y_1 y_2 ...
    と変数番号をつける *)
   fun termToIPPVec fmassoc vnames (Var (x,_)) = 
       let val vnum = valOf (LU.indexOf' Var.equal x vnames)
       in IPPV.fromList (L.tabulate (dimension,
				  fn i=> IPP.fromList [([dimension*vnum+i+1], IP.one)]))
       end
     | termToIPPVec fmassoc vnames (Fun (f,ts,_)) =  
       let val (_,fvec,Aflist) = lookup fmassoc f
	   (* IntPolyVec および IntPolyPolyMatrixのリスト *)

	   val args = L.map (termToIPPVec fmassoc vnames) ts
       in LP.foldlEq (fn (Afi,argvec,ans) => 
			 IPPV.plus (IPPM.mvtimes (Afi,argvec), ans))
		     fvec
		     (Aflist,args)
       end

(***
   val _ = print "aaa\n"
   val t1 = IOFotrs.rdTerm "f(?x,a)"
   val fmassoc = makeFunMatrixAssoc (Term.funArityMapInTerm t1)
   val vnames = VarSet.listItems (Term.varSetInTerm t1)
   val _ = print "bbb\n"
   val poly = termToIPPVec fmassoc vnames t1
   val _ = print (IPPV.toString poly)
   val _ = print "ccc\n"
***)

   (* すべての係数が \ge 0 となるのが十分条件 *)
   fun absPositive intPolyPoly = IPP.coefficients intPolyPoly

   (* 整係数多項式 (intPolyList1,intPoly2) s.t.
            forall p \in intPolyList1.  p \ge 0 
                                 <==> l \ge r となる制約
            forall p \in intPolyList1.  p \ge 0  かつ intPolyList2 \gt 0
                                 <==> l \gt r となる制約
      を返す  *)

   fun encodeRule fmassoc (l,r) =
       let val vnames = VarSet.listItems (Term.varSetInTerms [l,r])
	   val lpvec = termToIPPVec fmassoc vnames l
	   val rpvec = termToIPPVec fmassoc vnames r
	   val vec = IPPV.plus (lpvec, IPPV.minus rpvec)
	   (* "ベクトルの各要素が \ge 0" の制約 *)
	   val intPolyList1 = LU.mapAppend (fn ipp => absPositive ipp) (IPPV.toList vec)
	   (* "ベクトルの第1要素について、その定数が \ge 1" の制約 *)
	   val intPoly2 = IPP.constant (hd (IPPV.toList vec))
       in (intPolyList1,intPoly2) 
       end

   (*** decoding part ***)

   fun lookupIntBvec assign (i,j) = 
       case StringMap.find (assign,EIP.intBvec (i,j))
	of NONE => false (* no value may be assinged *)
	 | SOME ans => valOf (Bool.fromString ans)

   fun lookupInt assign i = 
       EIP.bvecToInt (rev (L.tabulate (EIP.bitLength,
				    fn j=>lookupIntBvec assign (i,j))))

   fun mkIntPolyVec assign fvec =
       let val ipplist = IPPV.toList fvec
	   val iplist = L.map IPP.constant ipplist
	   fun getVar ip = hd (#1 (hd (IP.toList ip)))
	   val vnums = L.map getVar iplist 
	   val values = L.map (lookupInt assign) vnums
       in IPV.fromList (L.map (fn v=> IP.fromList [([],v)]) values)
       end
       
   fun mkIntPolyMatrix assign A =
       let val ipplistlist = IPPM.toList A
	   val iplistlist = L.map (L.map IPP.constant) ipplistlist
	   fun getVar ip = hd (#1 (hd (IP.toList ip)))
	   val vnums = L.map (L.map getVar) iplistlist
	   val values = L.map (L.map (lookupInt assign)) vnums
       in IPM.fromList (L.map (L.map (fn v=> IP.fromList [([],v)])) values)
       end

  (* 実際の割り当てから関数記号の解釈をあたえる fmassoc を構成 *)
  (* fmassoc2 で参照する *)
   fun mkMatrixIntepretation assign fmassoc =
       L.map (fn (f,fvec,Aflist) =>
		 let val mat = L.map (mkIntPolyMatrix assign) Aflist
		     val vec = mkIntPolyVec assign fvec
		 in (f,vec,mat)
		 end) fmassoc

   (* 関数記号の解釈から項の解釈を構成 *)
   fun termToIPVec fmassoc2 vnames (Var (x,_)) = 
       let val vnum = valOf (LU.indexOf' Var.equal x vnames)
       in IPV.fromList (L.tabulate (dimension,
				  fn i=> IP.fromList [([dimension*vnum+i+1], 1)]))
       end
     | termToIPVec fmassoc2 vnames (Fun (f,ts,_)) =  
       let val (_,fvec,Aflist) = lookup fmassoc2 f
	   val args = L.map (termToIPVec fmassoc2 vnames) ts
       in LP.foldlEq (fn (Afi,argvec,ans) => 
			 IPV.plus (IPM.mvtimes (Afi,argvec), ans))
		     fvec
		     (Aflist,args)
       end

   (* 関数記号の解釈から書き換え規則の解釈([l] - [r]を構成 *)
   fun transRule fmassoc2 (l,r) =
       let val vnames = VarSet.listItems (Term.varSetInTerms [l,r])
	   val lpvec = termToIPVec fmassoc2 vnames l
	   val rpvec = termToIPVec fmassoc2 vnames r
	   val vec = IPV.plus (lpvec, IPV.minus rpvec)
	   (* "ベクトルの各要素が \ge 0" の制約 *)
	   val intPolyList1 = IPV.toList vec
	   (* "ベクトルの第1要素について、その定数が \ge 1" の制約 *)
	   val intPoly2 = hd (IPV.toList vec)
       in (intPolyList1,intPoly2) 
       end

   fun prVec assign fvec =
       let val ipplist = IPPV.toList fvec
	   val iplist = L.map IPP.constant ipplist
	   fun getVar ip = hd (#1 (hd (IP.toList ip)))
	   val vnums = L.map getVar iplist 
	   val values = L.map (lookupInt assign) vnums
       in LU.toStringSpaceSquare Int.toString values
       end

   fun prMatrix assign A =
       let val ipplistlist = IPPM.toList A
	   val iplistlist = L.map (L.map IPP.constant) ipplistlist
	   fun getVar ip = hd (#1 (hd (IP.toList ip)))
	   val vnums = L.map (L.map getVar) iplistlist
	   val values = L.map (L.map (lookupInt assign)) vnums
       in LU.toStringSlashSquare 
	      (LU.toStringSpaceSquare Int.toString) values
       end

  (* 関数記号の解釈を表示 *)
   fun prMatrixIntepretation assign fmassoc =
       (debug (fn _=> print "Matrix Interpretation:\n");
	L.app (fn (f,fvec,Aflist) =>
		  let val _ = debug (fn _=> print ("  " ^ (Fun.toString f) ^ ":= "))
		      val vars = L.tabulate(L.length Aflist, 
					 fn x=> "x" ^ Int.toString (x+1))
		      val mat = LP.map (fn (x,y)=> (prMatrix assign x) ^ 
						   "(" ^ y ^ ")")
				       (Aflist,vars)
		      val vec = prVec assign fvec
		      val _ = debug (fn _=> print (LU.toStringPlus (fn x=>" " ^ x ^ " ") 
						     (mat @ [vec])))
		      val _ = debug (fn _=> print "\n")
		  in ()
		  end) fmassoc
       )

  (* 書き換え規則の解釈 [l] - [r] のベクトル成分(整係数多項式)を表示 *)
  (* 実際の解釈より多項式は近似されている場合があることに注意。
     (左辺のcarry over を無視するため) *)
   fun prPolyForRules fmassoc2 lookup R rs =
       let fun prPoly n =
	   let val (l,r) = L.nth (R,n)
	       val isNotStrict = lookup n
	       val _ = debug (fn _=> print (" " ^ (Trs.prRule (l,r)) ^ "\n"))
	       val (iplist,_) = transRule fmassoc2 (l,r)
	       val nums = L.tabulate (L.length iplist,fn x=> x+1)
	       fun prline (poly,i) = print (" [l]-[r]#" ^ (Int.toString i)
					    ^ " >= " ^ (IP.toString poly) 
					    ^ (if isNotStrict orelse i <> 1
					       then " >= 0\n"
					       else " > 0\n"))
	   in LP.app prline (iplist,nums) 
	   end
       in L.app prPoly rs
       end

  fun prepareCheck outs (R,S) = 
      let 
	  val _ = debug (fn _ => print "start of preparation...\n")
          val faMap = Trs.funArityMapInRules (R @ S)
	  val fmassoc = makeFunMatrixAssoc faMap
	  val _ = EIP.init ()

	  val geDeclsM = L.map (EIP.encodeGeEquation outs) (mkMatrixPoly fmassoc)

	  val (gePolyListR,gtPolyR) = LP.unzip (L.map (encodeRule fmassoc) R)
	  val (gePolyListS,gtPolyS) = LP.unzip (L.map (encodeRule fmassoc) S)

	  val geDeclsListR = L.map (L.map (EIP.encodeGeEquation outs)) gePolyListR
	  val geDeclsListS = L.map (L.map (EIP.encodeGeEquation outs)) gePolyListS

	  val gtDeclsR = L.map (EIP.encodeGtEquation outs) gtPolyR
	  val gtDeclsS = L.map (EIP.encodeGtEquation outs) gtPolyS

	  val condEIP = !EIP.lines
	  val _ = EIP.init ()
	  val _ = debug (fn _=> print "... end of preparation\n")

      in (fmassoc,condEIP,geDeclsM,geDeclsListR,geDeclsListS,gtDeclsR,gtDeclsS)
      end


   fun step (ins,outs) numStep spec (R,S) (rs,ss) =
      let
  	  val (fmassoc,condEIP,geDeclsM,geDeclsListR,geDeclsListS,gtDeclsR,gtDeclsS) = spec

	  val _ = debug (fn _  => println ("Step " ^ (Int.toString numStep)))
	  val _ = debug (fn _ => println ("R = " ^ (prSetInOneLine Int.toString rs)))
	  val _ = debug (fn _ => println ("S = " ^ (prSetInOneLine Int.toString ss)))

	  fun lookupR assign i =
	      case StringMap.find (assign, EIP.bvar (L.nth (gtDeclsR, i)))
	       of NONE => (print ("i'th value (R) not found???"); false)
		 | SOME ans => ans = YI.prFalse
	  fun lookupS assign i =
	      case StringMap.find (assign,EIP.bvar (L.nth (gtDeclsS, i)))
	       of NONE => (print ("i'th value (S) not found???"); false)
		| SOME ans => ans = YI.prFalse

	  fun retract i =  (TextIO.output (outs, YI.prRetract i); TextIO.flushOut outs)
	  fun retractR i = (print ("retract " ^ (Trs.prRule (L.nth (R,i)))^ "\n");
			    retract (i + 1))
	  fun retractS i = (print ("retract " ^ (Trs.prRule (L.nth (S,i))) ^ "\n");
			    retract ((L.length R) + i + 1))
	  fun retractStep i = ((* print ("retract step " ^ (Int.toString i) ^ "\n"); *)
			       retract ((L.length R) + (L.length S) + i))
	  val listR = L.tabulate (L.length R, fn x=>x)
	  val listS = L.tabulate (L.length S, fn x=>x)
	  val _ = L.app (fn i => if LU.member i rs then () else retractR i) listR
	  val _ = L.app (fn i => if LU.member i ss then () else retractS i) listS

	  val gtCondR = YI.prAssertPlus (YI.prOr
					     (L.map (fn x=> EIP.bvar (L.nth (gtDeclsR,x))) rs))

	  val input = (gtCondR ^ "(check)\n" )
	  val _ = TextIO.output (outs, input)
	  val _ = TextIO.flushOut outs

	  val answer = TextIO.inputLine ins
	  val result =  (isSome answer) andalso (valOf answer = "sat\n")

      in if not result 
 	 then let val _ = debug (fn _=> print "No matrix interpretation found ..." )
	      in NONE
	      end
	 else let 
		 val assign = YI.readAssigns ins
                 val _ = prMatrixIntepretation assign fmassoc 
                 val fmassoc2 = mkMatrixIntepretation assign fmassoc
		 val _ = debug (fn _=> print "polynomial(R):\n")
		 val _ = debug (fn _=> prPolyForRules fmassoc2 (lookupR assign) R rs)
		 val _ = debug (fn _=> print "polynomial(S):\n")
		 val _ = debug (fn _=> prPolyForRules fmassoc2 (lookupS assign) S ss)

		 val rs2 = L.filter (lookupR assign) rs
		 val ss2 = L.filter (lookupS assign) ss
		 val _ = retractStep numStep
	     in SOME (rs2, ss2)
	     end
      end

  fun directCheck smtSolver (R,S) =
      let val listR = L.tabulate (L.length R, fn x=>x)
	  val listS = L.tabulate (L.length S, fn x=>x)

	  val _ = print "aaa"
(*
	  val timer = Timer.startCPUTimer ()
	  val start = TimeUtil.checkTime timer 
*)
	 (* val cmd = smtSolver ^ "\n" *)
	  val cmd = smtSolver ^ " 2>/dev/null\n"
 	  val proverCmd = ("/bin/sh", ["-c",cmd])
	  val proc = Unix.execute proverCmd
	  val (ins,outs) = Unix.streamsOf proc

	  val spec = prepareCheck outs (R,S)
  	  val (_,condEIP,geDeclsM,geDeclsListR,geDeclsListS,_,_) = spec

(*
	  val stop = TimeUtil.checkTime timer 
	  val _ = print (" ("  ^ (TimeUtil.reportTime (start,stop)) ^ " msec.)\n")
*)
	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

  	  val condM = YI.prAssert (YI.prAnd (L.map EIP.bvar geDeclsM))
	  (* ids of asssert plus starts from 1 *)
	  val condR = String.concat (L.map (fn x => assert (L.nth (geDeclsListR,x))) listR)
	  val condS = String.concat (L.map (fn x => assert (L.nth (geDeclsListS,x))) listS)

	  val _ = TextIO.output (outs, condEIP ^ condM ^ condR ^ condS)
	  val _ = TextIO.flushOut outs

	  (* プロセス終了処理*)
	  fun finish () = 
	      let val _ = debug (fn _=> print "finish the yices process\n")
		  val _ = TextIO.closeOut outs
		  val _ = TextIO.closeIn ins
		  val _ = Unix.reap proc  
(*
		  val stop2 = TimeUtil.checkTime timer 
		  val _ = print (" ("  ^ (TimeUtil.reportTime (start,stop2)) ^ " msec.)\n")
*)
	      in () end

	  fun try ([],_) numStep = (finish (); true)
	    | try (rs,ss) numStep = 
	      case step (ins,outs) numStep spec (R,S) (rs,ss) of
		  NONE => (finish (); false)
		| SOME (rs2,ss2) => try (rs2,ss2) (numStep+1) 
      in if null R
	 then (finish ();true)
	 else try (listR,listS) 1 (* should start from '1' *)
      end




  (* relative termination ソルバ *)
  fun relsnSolver (opt0:Solver.options) 
		  (opt1:DpSolver.options) 
		  (opt2:PoSolver.options) (R,S) =
      let
      	  val usePolynomial = ref true
      	  val useMatrix = ref true
      	  val useTermination = ref false

      	  val _ = print "Check relative termination:\n"
      	  val _ = print (Trs.prRules R)
      	  val _ = print (Trs.prRules S)
	  val listR = L.tabulate (L.length R, fn x=>x)
	  val listS = L.tabulate (L.length S, fn x=>x)

          val satSolver = (#satSolver opt0)
          val smtSolver = (#smtSolver opt0)
          val snProver = (#terminationProver opt0)
          val tmpDir = (#tmpDir opt0)

	  (* a simple non-termination check *)
	  fun isNonTerminating rs = 
	      L.exists (fn (l,r) => L.exists (fn u => isSome (Subst.match l u) )
						(Term.subterms r))
		       rs

          fun isTerminating rs =
      	      if snProver = ""
      	      then
      		  if (#useDp opt1)
      		  then if Cr.isOverlay rs
      		       then DpSolver.dpSolverInner satSolver tmpDir opt1 opt2 rs
      		       else DpSolver.dpSolver satSolver tmpDir opt1 opt2 rs
      		  else PoSolver.poSolver satSolver tmpDir opt2 (rs,[])
      	      else
      		  if Cr.isOverlay rs
      		  then Solver.sinSolver snProver tmpDir rs
      		  else Solver.snSolver snProver tmpDir rs

(*	  val cmd = smtSolver ^ "\n" *)
	  val cmd = smtSolver ^ " 2>/dev/null\n"
 	  val proverCmd = ("/bin/sh", ["-c",cmd])

          (* process for checking matrix interpretation *)
	  val procM = Unix.execute proverCmd
	  val (insM,outsM) = Unix.streamsOf procM

          (* process for checking polynomial interpretation *)
	  val procP = Unix.execute proverCmd
	  val (insP,outsP) = Unix.streamsOf procP

(*
	  val tmpName = #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
	  val _ = print ("[" ^ tmpName ^ "]\n")
	  val inTmpFile = OS.Path.joinDirFile {dir=".",file=tmpName}
	  val outsP = TextIO.openOut inTmpFile
*)

	  fun assert xs = YI.prAssertPlus (YI.prAnd (L.map EIP.bvar xs))

	  fun initM specMat = let
	      val (_,condEIP,geDeclsM,geDeclsListR,geDeclsListS,_,_) = specMat
	      val condM = YI.prAssert (YI.prAnd (L.map EIP.bvar geDeclsM))
	      (* ids of asssert plus starts from 1 *)
	      val condR = String.concat (L.map (fn x => assert (L.nth (geDeclsListR,x))) listR)
	      val condS = String.concat (L.map (fn x => assert (L.nth (geDeclsListS,x))) listS)
	      val _ = TextIO.output (outsM, condEIP ^ condM ^ condR ^ condS
					   (* ^ "(set-verbosity! 0)\n" *) 
					   ^ "(set-evidence! true)\n" )
	      val _ = TextIO.flushOut outsM
	  in () end

	  fun initP specPoly = let
	      val (_,eipCond,declFun,geDeclsListR,geDeclsListS,_,_) = specPoly
	      val condF = YI.prAssert (YI.prAnd (L.map EIP.bvar declFun))
	      (* ids of asssert plus starts from 1 *)
	      val condR = String.concat (L.map (fn x => assert (L.nth (geDeclsListR,x))) listR)
	      val condS = String.concat (L.map (fn x => assert (L.nth (geDeclsListS,x))) listS)
	      val _ = TextIO.output (outsP, eipCond ^ condF ^ condR ^ condS
					   (* ^ "(set-verbosity! 0)\n" *)
					   ^ "(set-evidence! true)\n")
	      val _ = TextIO.flushOut outsP
	  in () end

	  val numStepM = ref 0
	  val numStepP = ref 0

	  fun tryPoly _  _ ([],_) = (print "relatively terminating\n"; true)
	    | tryPoly [] (specP,specM) (rs,ss) = try [] (specP,specM) (rs,ss)
	    | tryPoly (x::xs) (specP,specM) (rs,ss) =
	      let fun tryPolySub specPoly ([],_) = (print "relatively terminating\n"; true)
		    | tryPolySub specPoly (rs,ss) =
		      (numStepP := (!numStepP) + 1;
      		       case PI.step (insP,outsP) (!numStepP) specPoly (R,S) (rs,ss) of
			   NONE => try xs (SOME specPoly,specM) (rs,ss)
      			 | SOME (rs2,ss2) => 
			   if not (!useMatrix) orelse (LU.member 1 xs)
			   then tryPoly (x::xs) (SOME specPoly,specM) (rs2,ss2)
			   else tryPoly (x::[1]) (SOME specPoly,specM) (rs2,ss2))
	      in case specP of 
		     NONE => let val specPoly = PI.prepareCheck outsP (R,S)
				 val _ = initP specPoly
      			     in tryPolySub specPoly (rs,ss) end
		   | SOME specPoly => tryPolySub specPoly (rs,ss)
	      end

	  and tryMat _ _ ([],_) = (print "relatively terminating\n"; true)
	    | tryMat [] (specP,specM) (rs,ss) = try [] (specP,specM) (rs,ss)
	    | tryMat (x::xs) (specP,specM) (rs,ss) = 
	      let fun tryMatSub specMat ([],ss) = (print "relatively terminating\n"; true)
		    | tryMatSub specMat (rs,ss) =
		      (numStepM := (!numStepM) + 1;
      		       case step (insM,outsM) (!numStepM) specMat (R,S) (rs,ss) of
			   NONE => try xs (specP,SOME specMat) (rs,ss)
      			 | SOME (rs2,ss2) => 
			   if not (!usePolynomial) orelse (LU.member 0 xs)
			   then tryMat (x::xs) (specP, SOME specMat) (rs2,ss2)
			   else tryMat (x::[0]) (specP, SOME specMat) (rs2,ss2))
	      in case specM of 
		     NONE => let val specMat = prepareCheck outsM (R,S)
				 val _ = initM specMat
      			     in tryMatSub specMat (rs,ss) end
		   | SOME specMat => tryMatSub specMat (rs,ss)
	      end

      	  and try _  _ ([],_) = (print "relatively terminating\n"; true)
      	    | try [] _ (rs,ss) = 
	      if (!useTermination) andalso (null ss)
      	      then isTerminating (L.map (fn i=> L.nth (R,i)) rs)
      	      else (print "unknown relative termination\n"; false)
      	    | try (0::xs) (specP,specM) (rs,ss) =
      	      if (!usePolynomial) 
	      then tryPoly (0::xs) (specP,specM) (rs,ss)
	      else try xs (specP,specM) (rs,ss)
      	    | try (_::xs) (specP,specM) (rs,ss) =
      	      if (!useMatrix)
      	      then tryMat (1::xs) (specP,specM) (rs,ss) 
      	      else try xs (specP,specM) (rs,ss)

      in if null R
      	 then (print "relatively terminating\n"; true)
	 else if isNonTerminating R
	 then (print "not relatively terminatiing\n"; false)
      	 else try [0,1] (NONE,NONE) (listR,listS) 
      end




(*   val result = directCheck "../../work/tools/bin/yices"
			    (IOFotrs.rdRules ["f(f(?x)) -> f(g(f(?x)))",
					      "f(?x) -> g(?x)"],
			     IOFotrs.rdRules ["f(?x) -> g(f(?x))"])
*)
(*
   val result = directCheck "./yices" "./tmp"
			    (IOFotrs.rdRules ["plus(s(?x),?y) -> s(plus(?x,?y))",
					      "plus(?x,s(?y)) -> s(plus(?x,?y))" ],
			     IOFotrs.rdRules ["plus(?x,?y) -> plus(?y,?x)",
					      "s(s(?x)) -> s(?x)", 
					      "s(?x) -> s(s(?x))",
					      "plus(plus(?x,?y),?z) -> plus(?x,plus(?y,?z))" ])
*)
(*
   Val result = directCheck "./yices" "./tmp"
			    (IOFotrs.rdRules ["f(a,g(?y)) -> f(?y,a)"],
			     IOFotrs.rdRules [])
*)
(*
   val _ = if result 
	   then print "relatively terminating\n"
	   else print "not relatively terminating\n"
*)


   end

end

(*
signature PATH_ORDERING =
sig
    val directCheck: string 
		   -> ((Term.term * Term.term) list * (Term.term * Term.term) list)
		   -> bool

    val propToYices: Prop.prop -> string

end

structure PathOrdering : PATH_ORDERING 
struct

    local 
      open Term
      structure L = List
      structure YI = YicesInput
    in

    fun propToYices prop = ""

    fun directCheck smtSolver (R,S) = true


    end (* of local *)
end (* of structure *)
*)
