(******************************************************************************
 * Copyright (c) 2013-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/non_cr.sml
 * description: non-confluence check
 * author: AOTO Takahito
 * 
 * $Id: non_cr.sml,v 1.3 2013/06/10 07:13:36 aoto Exp $
 ******************************************************************************)

signature NON_CR = 
sig

    val checkNonConfluence: string -> string -> string
			    -> (Term.term * Term.term) list -> bool

    val checkNonGroundConfluence: string -> string -> string
				  -> Term.term list -> (Term.term * Term.term) list -> bool

   val checkNonCommutative: string -> string -> string
                            -> bool
			    -> (Term.term * Term.term) list * (Term.term * Term.term) list
			    -> bool

(* (*temporal *)  val satSolver: string ref *)
(* (*temporal *)  val smtSolver: string ref *)
(* (*temporal *)  val snProver: string ref *)
(* (*temporal *)  val relSnProver: string ref *)
(* (*temporal *)  val tmpDir: string ref *)
    val useNonJoinableByTreeAutomata: bool ref
    val useNonJoinableByApproximation: bool ref
    val useNonJoinableByInterpretationAndOrder: bool ref
    val runDebug: bool ref
    val runCertification: bool ref
    datatype nj_problem = CrProblem of Trs.rules | ComProblem of bool * Trs.rules * Trs.rules
    val outputDisproof: nj_problem -> int -> (Term.term * Term.term * Term.term)
			-> (unit -> string) -> unit

end

structure NonCr : NON_CR =
struct
    local
	open Prop
	open Cr
	structure A = Atom
	structure CU = CertifyUtil
	structure L = List
	structure LU = ListUtil
	structure LP = ListPair
	structure FIS = FunIntSet
	structure FS = FunSet
	structure ILM = IntListMap2
	structure TA = TreeAutomata
	structure TP = TermPair
	structure TPS = TermPairSet
	structure TS = TermSet
	structure VS = VarSet
	open PrintUtil
	open Term
    in
    
    exception NonCrError
    val runDebug = ref false: bool ref
    fun debug f = if !runDebug then f () else ()

    val runCertification = ref false : bool ref

    datatype nj_problem = CrProblem of Trs.rules | ComProblem of bool * Trs.rules * Trs.rules
    fun isCrProblem (CrProblem _) = true
      | isCrProblem (ComProblem _) = false
    fun isComProblem (CrProblem _) = false
      | isComProblem (ComProblem _) = true

    val useNonJoinableByApproximation = ref true
    val useNonJoinableByTreeAutomata = ref true
    val useNonJoinableByInterpretationAndOrder = ref true

    exception DisprovedByTcaps of (Term.term * Term.term)
    exception DisprovedByTcaps2 of (Term.term * Term.term * Term.term)
    exception DisprovedByRootApprox of (Term.term * Term.term)
    exception DisprovedByPathOrder of (Term.term * Term.term * Term.term)
    exception DisprovedByTA of (Term.term * Term.term)
    exception DisprovedByEvenOdd of (Term.term * Term.term * Term.term)
    exception DisprovedByApprox of (Term.term * Term.term * Term.term)
    exception DisprovedByPolyOrder of (Term.term * Term.term * Term.term)
    exception DisprovedByModThree of (Term.term * Term.term * Term.term)
    exception DisprovedByModFour of (Term.term * Term.term)

    fun tcapOfTerms rs ts = 
	let val num = ref (Term.maxVarIndexInTerms ts)
	    fun mkvarTerm ty = (num := (!num)+1; Term.mkVarTerm (Var.fromStringAndInt ("x",!num),ty))
	    (* val _ = print (Trs.prRulesWithVarSort rs) *)
	    fun tcapsub t = 
		if Term.isVar t 
		then mkvarTerm (Term.sortOfTerm t)
		else let val args = L.map tcapsub (Term.argsOfTerm t)
			 val f = valOf (Term.funRootOfTerm t)
			 val ty = Term.sortOfTerm t
			 val cap = Term.mkFunTerm (f,args,ty)
			 val max = (Term.maxVarIndexInTerm cap) + 1
		     in if L.exists (fn (l,r) => isSome (Subst.unify (Term.increaseVarIndexBy max l) cap))
				    rs
			then mkvarTerm ty
			else cap
		     end
	in L.map tcapsub ts 
	end

    fun tcapOfTerm rs t =  hd (tcapOfTerms rs [t])

    (* term should be ground *)
    local
	exception GetToVar
    in
    fun approximatedDescendants rs depth term = 
	let 
	    fun mkFreshVars ty = Term.mkVarTerm (Var.fromStringAndInt ("x",0),ty)
	    fun barRootTerm t = 
		if Term.isVar t then t
		else Term.linearize (Term.mkFunTerm (valOf (Term.funRootOfTerm t),
						     L.map mkFreshVars (L.map Term.sortOfTerm (Term.argsOfTerm t)),
						     Term.sortOfTerm t))
	    fun rootStep rs t = L.map (fn (u,_) => u) (Rewrite.rootNarrowAll rs (barRootTerm t))
	    fun expand [] ans = SOME ans
	      | expand ((depth,term)::rest) (rootSet,tcapSet) =
		if Term.isVar term 
		then raise GetToVar
		else if depth = 0
		then if FS.member (rootSet,valOf (Term.funRootOfTerm term))
		     then expand rest (rootSet, tcapSet)
		     else expand rest (rootSet, TS.add (tcapSet, tcapOfTerm rs term))
		else let val reducts = rootStep rs term
		     in expand (rest @ (L.map (fn t => (depth-1,t)) reducts))
			       (FS.add (rootSet, valOf (Term.funRootOfTerm term)), tcapSet)
		     end
	in expand [(depth,term)] (FS.empty,TS.empty)
	end
	handle GetToVar => NONE
    end


    (* term should be ground *)
    local
	exception GetToVar
    in
    fun getRootDescendants rs depth term = 
	let 
	    fun rootStep rs f = L.mapPartial (fn (l,r) => case Term.funRootOfTerm l of
						       SOME g  => if Fun.equal (f,g) then SOME r else NONE
						     | NONE => NONE) rs
	    fun expand [] ans = SOME ans
	      | expand ((depth,term)::rest) (rootSet,tcapSet) =
		if Term.isVar term 
		then raise GetToVar
		else if depth = 0
		then if FS.member (rootSet,valOf (Term.funRootOfTerm term))
		     then expand rest (rootSet, tcapSet)
		     else expand rest (rootSet, let val tcap = tcapOfTerm rs term
						in if Term.isVar tcap then raise GetToVar 
						   else TS.add (tcapSet, tcap)
						end)
		else let val reducts = rootStep rs (valOf (Term.funRootOfTerm term))
		     in expand (rest @ (L.map (fn t => (depth-1,t)) reducts))
			       (FS.add (rootSet, valOf (Term.funRootOfTerm term)), tcapSet)
		     end
	in expand [(depth,term)] (FS.empty,TS.empty)
	end
	handle GetToVar => NONE
    end

    fun getPureDescendants rs depth term = 
	let fun expand [] (interior, border) = (interior, TS.map (tcapOfTerm rs) (TS.difference (border,interior)))
	      | expand ((depth,term)::rest) (interior,border) =
		if depth = 0
		then expand rest (interior, TS.add (border, term))
		else let val reducts = Rewrite.oneStepReducts rs term
			 val new = L.filter (fn t => not (TS.member (border,t)) andalso not (TS.member (interior,t)))
					    reducts
		     in expand (rest @ (L.map (fn t => (depth-1,t)) new))
			       (TS.add (interior, term), border)
		     end
	in expand [(depth,term)] (TS.empty,TS.empty)
	end

    fun isNonUnifiableTcaps (t0,t1) = 
	let val max = (Term.maxVarIndexInTerm t0) + 1
	in not (isSome (Subst.unify (Term.increaseVarIndexBy max t1) t0))
	end

    fun nonConfluenceCheckByTcaps rs (term0,term1) =
	let val _ = debug (fn _=> print ("Prove non-joinability of "  ^ (Term.toString term0)
			   ^ " and "   ^ (Term.toString term1)   ^ " by tcaps\n"))
	    val cap0 = tcapOfTerm rs term0
	    val cap1 = tcapOfTerm rs term1
	in if isNonUnifiableTcaps (cap0,cap1) (* tcaps are not unifiable *)
	   then (debug (fn _=> print "tcaps not unifiable\n"); raise DisprovedByTcaps (term0,term1))
	   else 
               (*  TO DO: refining set of critical pairs... *)
	       (* if Term.isFun cap0 andalso Term.isFun cap1 andalso Term.haveSameRoots (cap0,cap1)
	       then let val args0 = Term.argsOfTerm term0
			val args1 = Term.argsOfTerm term1
			val _ = print "roots of tcaps are identical; check args recursively\n"
		    in LP.exists (nonConfluenceCheckByTcaps rs) (args0,args1)
		    end
	       else *) false
	end

    (* output CR disproof for certification *)
    fun outputDisproof njProblem max (mid,t0,t1) resultFn= 
	let val (rs0,rs1) = case njProblem of
			 CrProblem rs => (rs,rs)
		       | ComProblem (true,rs,ss) => (rs,ss)
		      (* | ComProblem (false,rs,ss) => (ss,rs) 2024/06/29  *)
		       | ComProblem (false,rs,ss) => (rs,ss)
	    (* val _ = debug (fn _=> print "R0:") *)
	    (* val _ = debug (fn _=> print (Trs.prRules rs0)) *)
	    (* val _ = debug (fn _=> print "R1:") *)
	    (* val _ = debug (fn _=> print (Trs.prRules rs1)) *)
	    (* val _ = debug (fn _=> print ("Witness for " ^ (Term.toString t0) *)
	    (* 			  ^ " <-*-R0 " ^ (Term.toString mid) *)
	    (* 			  ^ " -*->R1 " ^ (Term.toString t1))) *)
	    val rseq0 = let val res = (Rewrite.findRewriteSequence rs0 max mid t0)
			in if isSome res then valOf res
			   else (print "fail to find non-joinable-sequence";
				 raise NonCrError)
			end
	    val rseq1 = let val res = (Rewrite.findRewriteSequence rs1 max mid t1)
			in if isSome res then valOf res
			   else (print "fail to find non-joinable-sequence";
				 raise NonCrError)
			end
	    val beginDisproof = case njProblem of
			 CrProblem _ => CU.beginCrDisproof
		       | ComProblem (true ,_,_) => CU.beginComDisproof
		       | ComProblem (false,_,_)  => CU.beginComDisproof o CU.beginSwapTRSs o CU.beginComDisproof 
	in (CU.output o CU.beginProof o beginDisproof o CU.beginNonJoinableFork)
	       [Rewrite.prRewriteSeqInProofTree (mid,rseq1),
		Rewrite.prRewriteSeqInProofTree (mid,rseq0),
		resultFn]
	end

    (* raise DisprovedByTcaps2 if disproved, o.w. returning new candidates set *)
    fun nonJoinableCheckByTcaps2 njProblem (mid,term0,term1) =
	let val _ = debug (fn _=> print ("Prove non-joinability of "  ^ (Term.toString term0)
			   ^ " and "   ^ (Term.toString term1)   ^ " by tcaps\n"))
	    val (rs0,rs1) = case njProblem of
				CrProblem rs => (rs,rs)
			      | ComProblem (isUnSwitched,rs0,rs1) => (rs0,rs1)
	    val isUnSwitched = case njProblem of
				   CrProblem _ => true
				 | ComProblem (isUnSwitched,_,_) => isUnSwitched

	    val cap0 = tcapOfTerm rs1 term0
	    val cap1 = tcapOfTerm rs0 term1
	    val _ = debug (fn _ => println ("caps: " ^ (Term.toString cap0) ^ ", " ^ (Term.toString cap1)))
	in if isNonUnifiableTcaps (cap0,cap1) (* tcaps are not unifiable *)
	   then (print " (success)\n"; 
		 if (!runCertification) 
		 then (* if isUnSwitched 2024/06/29
		      then *)
		     (outputDisproof njProblem 7 (mid,term0,term1) 
				     (fn _ => CU.encloseProofLeafBy "capNotUnif" "");
		      raise DisprovedByTcaps2 (mid,term0,term1))
		 (* else (outputDisproof njProblem 7 (mid,term1,term0) 
		    (fn _ => CU.encloseProofLeafBy "capNotUnif" "");
		    raise DisprovedByTcaps2 (mid,term1,term0))) *)
		 else ())
	   else ()
(***  diable the following for including certification capability ***********************)
(************
	       if Term.isFun cap0 andalso Term.isFun cap1 andalso Term.haveSameRoots (cap0,cap1)
	       then let val args0 = Term.argsOfTerm term0
			val args1 = Term.argsOfTerm term1
			val ans = LP.map (nonJoinableCheckByTcaps2 njProblem) (args0,args1)
		    in  LU.mapAppend (fn x=>x) ans
		    end
	       else *************)
	     (*  [(term0,term1)] *)
	end


    val depthOfApproximation = ref 2
    val useNonJoinableByApprox = ref true

    fun nonJoinableCheckByApprox (ComProblem _) (mid,term0,term1) = false
      | nonJoinableCheckByApprox  njProblem (mid,term0,term1) =
	let (* currently, only works for CrProblem *)
	    (* to do: use a different TRS to compute Tcap approximation *)
	    val (rs0,rs1) = case njProblem of
				CrProblem rs => (rs,rs)
			      | ComProblem (isUnSwitched,rs0,rs1) => (rs0,rs1)

	    val _ = debug (fn _=> print ("Prove non-joinability of "  ^ (Term.toString term0)
			   ^ " and "   ^ (Term.toString term1)   ^ " by approx\n"))

	    val (tset1,caps1) =  getPureDescendants rs0 (!depthOfApproximation) term1
	    val (caps1v,caps1nv) = L.partition Term.isVar (TS.listItems caps1)
	    val _ = debug (fn _=> print ("tset1: " ^ (LU.toStringCommaCurly Term.toString (TS.listItems tset1)) ^ "\n"))
	    val _ = debug (fn _=> print ("caps1: " ^ (LU.toStringCommaCurly Term.toString (TS.listItems caps1)) ^ "\n"))

	    val (tset0,caps0) =  getPureDescendants rs1 (!depthOfApproximation) term0
	    val (caps0v,caps0nv) = L.partition Term.isVar (TS.listItems caps0)
	    val _ = debug (fn _=> print ("tset0: " ^ (LU.toStringCommaCurly Term.toString (TS.listItems tset0)) ^ "\n"))
	    val _ = debug (fn _=> print ("caps0: " ^ (LU.toStringCommaCurly Term.toString (TS.listItems caps0)) ^ "\n"))
	in if ListXProd.foldX 
		  (fn (c1,c0,ans) => ans orelse isNonUnifiableTcaps (c1,c0))
		  (caps1nv, caps0nv) false
	   then (debug (fn _=> print "found non-unifiable tcaps\n");
		 raise DisprovedByApprox (mid,term0,term1))
	   else if not (null caps1v) orelse not (null caps0v)
		   orelse (not (null caps0nv) andalso not (null caps1nv))
	   then (* add 2014/6/28  **)
	      if TS.isEmpty (TS.intersection (tset0,tset1))
		   andalso
		   L.all (fn t => TS.isSubset (Rewrite.oneStepReductSet rs0 t, tset0))
			 (TS.listItems tset0)
		   andalso
		   L.all (fn t => TS.isSubset (Rewrite.oneStepReductSet rs1 t, tset1))
			 (TS.listItems tset1)
	       then (print " (success)\n";
		     raise DisprovedByApprox (mid,term0,term1))
	       else false
	   else if TS.isEmpty (TS.intersection (tset0,tset1))
		   andalso ListXProd.foldX 
			       (fn (t0,c1,ans) => ans andalso (not (isSome (Subst.unify t0 c1))))
			       (TS.listItems tset0, caps1nv) true
		   andalso ListXProd.foldX 
			       (fn (t1,c0,ans) => ans andalso (not (isSome (Subst.unify t1 c0))))
			       (TS.listItems tset1, caps0nv) true
	   then (print " (success)\n";
		 raise DisprovedByApprox (mid,term0,term1))
 	   else false 
	end

    val useNonJoinableByRootApprox = ref false
    val depthOfRootApproximation = ref 2

    fun nonConfluenceCheckByRootApprox rs (_,term0,term1) =
	let val _ = debug (fn _=> print ("Prove non-joinability of "  ^ (Term.toString term0)
			   ^ " and "   ^ (Term.toString term1)   ^ " by root-approx\n"))
	    val ans1 = getRootDescendants rs (!depthOfRootApproximation) term1
	    val ans0 = getRootDescendants rs (!depthOfRootApproximation) term0
	in
	    case (ans0,ans1) of 
		(NONE,_) => false
	      | (_,NONE) => false
	      | _ => let
		    val (fset0,caps0) = valOf ans0
		    val (fset1,caps1) = valOf ans1
		    val _ = debug (fn _=> print ("fset1: " 
						 ^ (LU.toStringCommaCurly Fun.toString (FS.listItems fset1))
						 ^ ", caps1: " 
						 ^ (LU.toStringCommaCurly Term.toString (TS.listItems caps1))
						 ^ ", fset0: " 
						 ^ (LU.toStringCommaCurly Fun.toString (FS.listItems fset0)) 
						 ^ ", caps0: " 
						 ^ (LU.toStringCommaCurly Term.toString (TS.listItems caps0)) 
						 ^ "\n"))
		    val caps0' = TS.listItems caps0
		    val caps1' = TS.listItems caps1
		in if FS.isEmpty (FS.intersection (fset0,fset1))
		      andalso
		      L.all (fn c1 => not (FS.member (fset0, valOf (Term.funRootOfTerm c1)))) caps1'
		      andalso
		      L.all (fn c0 => not (FS.member (fset1, valOf (Term.funRootOfTerm c0)))) caps0'
		      andalso 
		      ListXProd.foldX 
			  (fn (c0,c1,ans) => ans andalso isNonUnifiableTcaps (c0,c1))
			  (caps0',caps1') true
		   then (print " (success)\n";
			 raise DisprovedByRootApprox (term0,term1))
		   else false
		end
	end

     fun degrowingApproximationRule (l,r) =
	let val num = ref (Int.max (Term.maxVarIndexInTerm r, Term.maxVarIndexInTerm l))
	    val xs = Term.varListInTerm l (* var_key list *)
	    val ys = case Term.varRootOfTerm r of
			 SOME x => [x]
		       | NONE => Term.varListInTerms (L.filter Term.isVar  (Term.argsOfTerm r))
            (* variables at depth 0 or 1  *)
	    val zs = ListUtil.multisetUnion' Var.equal (xs,ys)

	    fun mkrepl [] repl = repl
	      | mkrepl (x::xs) repl = 
		if LU.member' Var.equal x xs
		then (num := (!num)+1;
		      mkrepl xs ((x,Var.increaseIndexBy (!num) x)::repl))
		else mkrepl xs ((x,x)::repl)

	    val replacements = mkrepl zs []
	    (* repl =  [ (?x,?x), (?x,?x_4), (?x,?x_5) ]
                           meaning that first ?x should be replaced by ?x,
             second ?x should be replaced by ?x_4, etc.*)

	    fun mkVar ty = (num := (!num)+1; Term.mkVarTerm (Var.fromStringAndInt ("x",!num),ty))

	    fun lookup x ls = 
		let fun lookupSub x [] rest = raise NonCrError
		      | lookupSub x ((y,z)::rest) done = if Var.equal (x,y) 
							 then (z, List.revAppend (done,rest))
							 else lookupSub x rest ((y,z)::done)
		in lookupSub x ls [] end

	    fun replaceVar t repl = case varRootOfTerm t of
					SOME x => let val (x',repl') = lookup x repl
						  in (Term.mkVarTerm (x',Term.sortOfTerm t), repl') end
				      | NONE => let val (ts',repl') = replaceVars (Term.argsOfTerm t) repl
						in (Term.mkFunTerm (valOf (Term.funRootOfTerm t),
								    ts', Term.sortOfTerm t),
						    repl') end
	    and replaceVars [] repl = ([],repl)
	      | replaceVars (t::ts) repl = let val (t',repl') = replaceVar t repl
					       val (ts',repl'') = replaceVars ts repl'
					   in (t'::ts', repl'')
					   end

	    fun replaceLHS t repl = let val (s,_) = replaceVar l repl in s end

	    val newl = replaceLHS l replacements

	    val lsub = Term.nonVarSubterms newl

	    fun replaceWithFreshVar t = if Term.isVar t
					then mkVar (Term.sortOfTerm t)
					else Term.mkFunTerm (valOf (Term.funRootOfTerm t),
							     L.map replaceWithFreshVar (Term.argsOfTerm t),
							     Term.sortOfTerm t)

	    fun replaceVars2 [] repl = ([],repl)
	      | replaceVars2 (t::ts) repl = if Term.isVar t
					    then let val (t',repl') = replaceVar t repl
						     val (ts',repl'') = replaceVars2 ts repl'
						 in (t'::ts', repl'') end
					    else let val t' = if LU.member' Term.equal t lsub  (* !!! *)
							      then t
							      else replaceWithFreshVar t
						     val (ts',repl') = replaceVars2 ts repl
						 in (t'::ts', repl') end

	    fun replaceRHS t repl = if Term.isVar t
				    then t
				    else let val (ts,_) = replaceVars2 (Term.argsOfTerm t) repl
					 in Term.mkFunTerm (valOf (Term.funRootOfTerm t), ts, Term.sortOfTerm t)
					 end

	    val newr = replaceRHS r replacements

	in (newl, newr)
	end

    fun degrowingApproximationRules rs = L.map degrowingApproximationRule rs

    (* output CR disproof for certification *)
    fun outputDisproofTA (ta0,ta1) () = 
	CU.encloseProofTreesBy "emptyTreeAutomataIntersection"
       [fn _=> CU.encloseProofTreesBy "firstAutomaton"
				      [TA.toProofTree ta1, 
				       fn _ => CU.encloseProofTreeBy "criterion"
							     (fn _ => CU.encloseProofLeafBy "decisionProcedure" "")],
	fn _=> CU.encloseProofTreesBy "secondAutomaton"
				      [TA.toProofTree ta0, 
				       fn _ => CU.encloseProofTreeBy "criterion"
							     (fn _ => CU.encloseProofLeafBy "decisionProcedure" "")]
       ]

(*    fun nonJoinableCheckByTA rs (mid,t0,t1) =  *)
    fun nonJoinableCheckByTA njProblem (mid,t0,t1) = 
	let val (rs,rs0,rs1) = case njProblem of
			 CrProblem rs => (rs,rs,rs)
		       | ComProblem (isUnSwitched,rs0,rs1) => (rs0@rs1,rs0,rs1)

	    val isUnSwitched = case njProblem of
				   CrProblem _ => true
				 | ComProblem (isUnSwitched,_,_) => isUnSwitched

	    val ts = [t0,t1]
	    val rs' = degrowingApproximationRules rs
	    val _ = debug (fn _=> print "\n")
	    val _ = debug (fn _=> print (Trs.prRules rs'))
	    val (ta:TA.tree_automaton) = TA.degrowingCompletion
					     (FIS.union (Trs.funAritySetInRules rs', 
							 Term.funAritySetInTerms ts))
					     ts
					     rs'
	    val qs = L.tabulate (L.length ts, fn i => TA.qOfNum i)
	    val _ = debug (fn _=> print (TA.toString ta))

	    val ta0= { Signature = #Signature ta, 
			States = #States ta, 
			Final = TS.singleton (TA.qOfNum 0),
			Rules = #Rules ta }:TA.tree_automaton

	    val ta1= { Signature = #Signature ta, 
			States = #States ta, 
			Final = TS.singleton (TA.qOfNum 1),
			Rules = #Rules ta }:TA.tree_automaton

	    exception Success of TA.tree_automaton * TA.tree_automaton * TA.tree_automaton
	in (ListXProd.appX (fn (x,y) => if Term.equal(x,y)
				      then ()
				      else if TA.haveEmptyIntersection ta (x,y)
				      then raise Success (ta,ta0,ta1)
				      else ())
			 (qs,qs);
	    debug (fn _=> print "failed (non-empty intersection)\n");
	    false)
	   handle Success (ta,ta0,ta1) => (print " (success)\n";
					   print ((TA.toString ta));
					   if (!runCertification) 
					   then (* if isUnSwitched 2024/06/29
						then *) (outputDisproof njProblem 7 (mid,t0,t1) 
								     (outputDisproofTA (ta0,ta1));
						      raise DisprovedByTA (t0,t1))
						(* else (outputDisproof njProblem 7 (mid,t1,t0) 
								     (outputDisproofTA (ta1,ta0));
						      raise DisprovedByTA (t1,t0)) *)
					   else true)
	end

    fun usableInTerm rs (Var _) (use,nuse) = (use,nuse)
      | usableInTerm rs (Fun (f,ts,ty)) (use,nuse)  = 
	let val t' = Fun (f,tcapOfTerms rs ts,ty)
	    val (rs0,rs1) = L.partition (fn (l,r) => isSome (Subst.unify t' l)) nuse
	in usableInTerms rs ts (rs0@use,rs1) 
	end

    and usableInTerms rs [] (use,nuse) = (use,nuse)
      | usableInTerms rs (t::ts) (use,nuse) = usableInTerms rs ts (usableInTerm rs t (use,nuse))

    fun usableClosure rs (use,nuse) =
	let val rhs = L.map (fn (l,r) => r) use
	    val (use1,nuse1) =  usableInTerms rs rhs ([],nuse)
	in if null use1
	   then (use1,nuse1)
	   else let val (use2,nuse2) = usableClosure rs (use1, nuse1)
		in (use1 @ use2, nuse2)
		end
	end

    fun usableRules rs t = 
	let val (rs0,rs1) = usableInTerm rs t ([],rs)
	    val (rs1a,rs1b) = usableClosure rs (rs0,rs1)
	in rs0 @ rs1a end

    val useNonJoinableByEvenOdd = ref true

(*    fun nonJoinableCheckByEvenOdd smtSolver tmpDir rs (mid,t0,t1)  =  *)
    fun nonJoinableCheckByEvenOdd smtSolver tmpDir njProblem (mid,t0,t1)  = 
	let val _ = debug (fn _=> print ("Prove non-joinability of "
			   ^ (Term.toString t0)
			   ^ " and "
			   ^ (Term.toString t1)
			   ^ " by mod2\n"))
			  
	    val (rs,ss) = case njProblem of
			      CrProblem rs => (rs,rs)
			    | ComProblem (isUnSwitched,rs0,rs1) => (rs0,rs1)

	    val rs0 = usableRules ss t0
	    val rs1 = usableRules rs t1
	    val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	    
	    val (res,cpf) = PolynomialInterpretation.njCheckEvenOdd smtSolver tmpDir (t0,t1) (rs1 @ rs0rev)
	in 
	    (* if res 
	       then raise DisprovedByEvenOdd (t0,t1)
	       else false *)
           if res
	   then (if (!runCertification) 
		 then outputDisproof njProblem 7 (mid,t0,t1) 
				     (fn _ => CU.encloseProofTreeBy "usableRulesNonJoin" cpf)
		 else ();
		 print " (success)\n";
		 raise DisprovedByEvenOdd (mid,t0,t1))
	   else false
	end

    val useNonJoinableByModThree = ref false
    fun nonJoinableCheckByModuloThree smtSolver tmpDir njProblem (mid,t0,t1)  = 
	let val _ = debug (fn _=> print ("Prove non-joinability of "
			   ^ (Term.toString t0)
			   ^ " and "
			   ^ (Term.toString t1)
			   ^ " by mod3\n"))

	    val (rs,ss) = case njProblem of
			      CrProblem rs => (rs,rs)
			    | ComProblem (isUnSwitched,rs0,rs1) => (rs0,rs1)

	    val rs0 = usableRules ss t0
	    val rs1 = usableRules rs t1
	    val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	    val (res,cpf) = PolynomialInterpretation.njCheckModuloThree smtSolver tmpDir (t0,t1) (rs1 @ rs0rev)
	in if res
	   then (if (!runCertification) 
		 then outputDisproof njProblem 7 (mid,t0,t1) 
				     (fn _ => CU.encloseProofTreeBy "usableRulesNonJoin" cpf)
		 else ();
		 print " (success)\n";
		 raise DisprovedByModThree (mid,t0,t1))
	   else false

	end

    fun nonConfluenceCheckByModuloFour smtSolver tmpDir rs (t0,t1)  = 
	let val _ = debug (fn _=> print ("Prove non-joinability of "
			   ^ (Term.toString t0)
			   ^ " and "
			   ^ (Term.toString t1)
			   ^ " by mod4\n"))
	    val rs0 = usableRules rs t0
	    val rs1 = usableRules rs t1
	    val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	in if PolynomialInterpretation.njCheckModuloFour smtSolver tmpDir (t0,t1) (rs1 @ rs0rev)
	   then (print " (success)\n"; raise DisprovedByModFour (t0,t1))
	   else false
	end

    val useNonJoinableByPolyOrder = ref true
    fun nonJoinableCheckByPolyOrder smtSolver tmpDir njProblem (mid,t0,t1)  = 
	let val _ = debug (fn _=> print ("Prove non-joinability of "
			   ^ (Term.toString t0)
			   ^ " and "
			   ^ (Term.toString t1)
			   ^ " by poly\n"))

	    val (rs,ss) = case njProblem of
			      CrProblem rs => (rs,rs)
			    | ComProblem (isUnSwitched,rs0,rs1) => (rs0,rs1)
			  
	    val rs0 = usableRules ss t0
	    val rs1 = usableRules rs t1
	    val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	    val rs1rev = L.map (fn (l,r) => (r,l)) rs1

	    val (res1,cpf1) = PolynomialInterpretation.njCheckByOrder smtSolver tmpDir (t0,t1) (rs1 @ rs0rev)
	in if res1
	   then (if (!runCertification) 
		 then outputDisproof njProblem 7 (mid,t0,t1) 
				     (fn _ => CU.encloseProofTreeBy "usableRulesNonJoin" cpf1)
		 else ();
		 print " (success)\n";
		 raise DisprovedByPolyOrder (mid,t0,t1))
	   else let val (res2,cpf2) =  
			PolynomialInterpretation.njCheckByOrder smtSolver tmpDir (t1,t0) (rs0 @ rs1rev)	
		in if res2
		   then (if (!runCertification) 
			 then outputDisproof njProblem 7 (mid,t1,t0) 
					     (fn _ => CU.encloseProofTreeBy "usableRulesNonJoin" cpf2)
			 else ();
			 print " (success)\n";
			 raise DisprovedByPolyOrder (mid,t0,t1))
		   else false
		end
	end
  
    val useNonJoinableByPathOrder = ref true
(*    fun nonJoinableCheckByPathOrder satSolver tmpDir rs (mid,t0,t1)  =  *)
    fun nonJoinableCheckByPathOrder satSolver tmpDir njProblem (mid,t0,t1)  = 
	let  val (rs0a,rs1a) = case njProblem of
				   CrProblem rs => (rs,rs)
				 | ComProblem (isUnSwitched,rs0,rs1) => (rs0,rs1)
             (* t0 <-*-R0 o -*->R1 t1 *)
             (* Check "t0 -*->R1 o <-*->R0 t1" does not hold *)
	     val _ = debug (fn _=> print ("Prove non-joinability of "
					 ^ (Term.toString t0)  ^ " and "  ^ (Term.toString t1) ^ " by rpo\n"))

	     val isUnSwitched = case njProblem of
				   CrProblem _ => true (* not used *)
				 | ComProblem (isUnSwitched,_,_) => isUnSwitched

	    val useDescPair = true (* use  (>=, >) version *)
	    val useNgeTotal  = false  (* use  (>=,not >=) version with total precedence *)
	    val useNgePartial  = false  (* use  (>=,not >=) version with parital precedence  *)
	    val opt = { useQuasi = true, useLex = false, useMul = true, useAf = true  } : PoSolver.options
	    val optForNge = { useQuasi = true, useLex = true, useMul = false, useAf = true  } : PoSolver.options (* cannot use Mul *)

	    val rs1b = usableRules rs1a t0
	    val rs0b = usableRules rs0a t1
	    val rs1rev = L.map (fn (l,r) => (r,l)) rs1b
	    val rs0rev = L.map (fn (l,r) => (r,l)) rs0b

	    val _ = if not (Trs.areRewriteRules rs0rev)
		       andalso not (Trs.areRewriteRules rs1rev)
		    then (debug (fn _=> print "both of rs0^{-1} and rs1^{-1} are not TRSs, don't try.\n"))
		    else ()

	    fun main posolver opt' = 
		let fun report cpf (x,y) =
			if (!runCertification) 
			then (* if isUnSwitched 2024/06/26 
			     then *) outputDisproof njProblem 7 (mid,x,y)
						 (fn _ => CU.encloseProofTreeBy "usableRulesNonJoin" cpf)
			     (* else outputDisproof njProblem 7 (mid,y,x)
						 (fn _ => CU.encloseProofTreeBy "usableRulesNonJoin" cpf) *)
			else () 

		    val _ = if Trs.areRewriteRules rs0rev
			    then let val (res,cpf) = posolver satSolver tmpDir opt' (t0,t1) (rs1b,rs0b)
                                     (* check by t0>t1 & R0 \subset >=  & R1 \subset <= *)
				 in if res then (report cpf (t0,t1);
						 print " (success)\n";
						 (* if isUnSwitched 2024/06/29
						 then *) raise DisprovedByPathOrder (mid,t0,t1) 
						(* else raise DisprovedByPathOrder (mid,t1,t0) *)
						)
				    else ()
				 end
			    else ()
		    val _ = if Trs.areRewriteRules rs1rev
			    then let val _ = debug (fn _=> print (Trs.prRules rs1rev)) (* *)
				     val _ = debug (fn _=> print (Trs.prRules rs1b)) (* del *)
				     val _ = debug (fn _=> print (Trs.prRules rs0b)) (* del *)
				     val (res,cpf) = posolver satSolver tmpDir opt' (t1,t0) (rs0b,rs1b) 
                                     (* check by t1>t0 & R1 \subset >=  & R0 \subset <= *)
				 in if res then (report cpf (t0,t1); print " (success)\n";
						 (* if isUnSwitched 2024/06/24 
						 then *) raise DisprovedByPathOrder (mid,t1,t0)
						(* else raise DisprovedByPathOrder (mid,t0,t1) *)
						)
				    else ()
				 end
			    else ()
		in () end

	    val _ = if useDescPair then main PoSolver.poSolverForNj opt else ()
	    val _ = if useNgeTotal then main PoSolver.poSolverForNjwithNgeTotal optForNge else ()
	    val _ = if useNgePartial then main PoSolver.poSolverForNjwithNgePartial optForNge else ()

	in (* if (Trs.areRewriteRules rs0rev *)
	   (*     andalso PoSolver.poSolverForNj satSolver tmpDir opt (t0,t1) (rs0,rs1)) *)
	   (*    orelse  *)
	   (*    (Trs.areRewriteRules rs1rev *)
	   (*     andalso PoSolver.poSolverForNj satSolver tmpDir opt (t1,t0) (rs1,rs0)) *)
	   (* then raise DisprovedByPathOrder (t0,t1)) *)
	   (* else false *)
	    false
	end

    (* another TermPair structure, suitable for sorting *)
    structure TermPair2 : ORD_KEY = 
      struct 
      type ord_key = Term.term * Term.term
      fun compare ((x0,y0),(x1,y1)) =
	  let val sizeX0 = Term.termSize x0
	      val sizeY0 = Term.termSize y0
	      val sizeX1 = Term.termSize x1
	      val sizeY1 = Term.termSize y1
	      val sizeZ0 = sizeX0 + sizeY0
	      val sizeZ1 = sizeX1 + sizeY1
	  in if sizeZ0 > sizeZ1 
		orelse (sizeZ0 = sizeZ1 andalso sizeX0 > sizeX1)
		orelse (sizeZ0 = sizeZ1 andalso sizeX0 = sizeX1
			andalso Term.compare (x0,x1) = GREATER)
		orelse (sizeZ0 = sizeZ1 andalso sizeX0 = sizeX1
			andalso Term.compare (x0,x1) = EQUAL
			andalso Term.compare (y0,y1) = GREATER)
	     then GREATER
	     else if Term.compare (x0,x1) = EQUAL
		     andalso Term.compare (y0,y1) = EQUAL
	     then EQUAL
	     else LESS
	  end
      fun equal (ss,tt) = compare (ss,tt) = EQUAL
      fun toString (s,t) = ("(" ^ (Term.toString s) ^ ", " ^ (Term.toString t) ^ ")")
      end 

    structure TermPairSet2 = RedBlackSetFn (TermPair2) : ORD_SET
    structure TPS2 = TermPairSet2

    (* another TermPair structure, suitable for sorting *)
    structure TermTriple2 : sig
        include ORD_KEY 
	val fromTerm: Term.term * Term.term * Term.term -> ord_key option
	      end
    = struct 
    type ord_key = int * int * Term.term * Term.term * Term.term
    (*  total size, lhs size, mid term, lhs term,  rhs term *)
    fun fromTerm (m,x,y) = if Term.equal (x,y) then NONE
			   else let val sizeX = Term.termSize x
				    val sizeY = Term.termSize y
				    val sizeZ = sizeX + sizeY
				in if sizeX > sizeY then SOME (sizeZ,sizeX,m,x,y) 
				   else SOME (sizeZ,sizeY,m,y,x) 
				end

    fun compare ((sizeZ0,sizeX0,m0,x0,y0),(sizeZ1,sizeX1,m1,x1,y1)) =
	if sizeZ0 > sizeZ1 
	   orelse (sizeZ0 = sizeZ1 andalso sizeX0 > sizeX1)
	   orelse (sizeZ0 = sizeZ1 andalso sizeX0 = sizeX1
		   andalso Term.compare (x0,x1) = GREATER)
	   orelse (sizeZ0 = sizeZ1 andalso sizeX0 = sizeX1
		   andalso Term.compare (x0,x1) = EQUAL
		   andalso Term.compare (y0,y1) = GREATER)
	then GREATER
	else if Term.compare (x0,x1) = EQUAL
		andalso Term.compare (y0,y1) = EQUAL
	then EQUAL
	else LESS

    fun equal (ss,tt) = compare (ss,tt) = EQUAL
    fun toString (_,_,m,s,t) = ("(" ^ (Term.toString s) ^ " <- " ^ (Term.toString m) ^ " -> " ^ (Term.toString t) ^ ")")
    end 

    structure TermTripleSet2 = RedBlackSetFn (TermTriple2) : ORD_SET
    structure TTS2 = TermTripleSet2
    structure TT2 = TermTriple2

    fun finish (t0,t1) = (print ("Witness for Non-Confluence: <" ^ (Term.toString t0)
				 ^ ", " ^ (Term.toString t1)
				 ^ ">\n");true)

    fun finish2 (mid,t0,t1) = (print ("Witness for Non-Confluence: <" ^ (Term.toString t0)
				 ^ " <- " ^ (Term.toString mid)
				 ^ " -> " ^ (Term.toString t1)
				 ^ ">\n");true)


    fun mkNonJoinableCands satSolver smtSolver tmpDir forGround gterms rs = 
	let
	    (* val _ = print ("TRS:\n") *)
	    (* val _ = print (Trs.prRules rs) *)
	    val trs = Trs.rulesToTrs rs
	    val faset = Trs.funAritySetInRules rs
	    val rs0 = Rewrite.oneStepUnfoldingsOfRules rs
	    (* val rs2 = Trs.deleteIdenticalRules (rs @ rs0) *)

	   (*  val _ = debug (fn _ => print (Trs.prRulesWithVarSort rs0)) *)

	    val rs1 = TPS2.listItems (TPS2.addList (TPS2.empty, rs0 @ rs))
	    val rs1' = L.take (rs1, Int.min (L.length rs1, 20))

	    val rs1_0 = Rewrite.oneStepUnfoldingsOfRules rs1'
	    val rs1_1 = TPS2.listItems (TPS2.addList (TPS2.empty, rs1_0 @ rs1'))
	    val rs1_1' = L.take (rs1_1, Int.min (L.length rs1_1, 10))

	    val rs1_00 = Rewrite.oneStepUnfoldingsOfRules rs1_1'
	    val rs1_11 = TPS2.listItems (TPS2.addList (TPS2.empty, rs1_00 @ rs1_1'))
	    val rs1_11' = L.take (rs1_11, Int.min (L.length rs1_11, 10))

	    val rs1_000 = Rewrite.oneStepUnfoldingsOfRules rs1_11'
	    val rs1_111 = TPS2.listItems (TPS2.addList (TPS2.empty, rs1_000 @ rs1_11'))
	    val rs1_111' = L.take (rs1_111, Int.min (L.length rs1_111, 10))

(*	    val rs2 = rs @ rs1_111' *)
	    val rs2 = LU.eliminateDuplication' 
			  (fn (l,r) => Subst.identicalModuloRenamingRule l r) 
			  (rs @ rs1_111')

	    val _ = print ("obtain " ^ (Int.toString (L.length rs2)) ^ " rules by 3 steps unfolding\n")

	    (* val _ = debug (fn _ => print (Trs.prRulesWithVarSort rs2)) *)

	    (* val cps0 = Cr.criticalPairs rs2 *)
	    val cps0 = Cr.criticalPeaks rs2 
	   (* val _ = debug (fn _ => print "CPS0:\n") *)
	   (* val _ = debug (fn _=> print (LU.toStringCommaSquare IOMstrs.prEq (L.map (fn (_,x,y)=>(x,y)) cps0))) *)


	    (* val cps0' = TPS2.addList (TPS2.empty, *)
	    (* 			    L.mapPartial (fn (y,z) =>  *)
	    (* 					     if Term.equal (y,z) then NONE *)
	    (* 					     else if Term.termSize y > Term.termSize z *)
	    (* 					     then SOME (y,z) *)
	    (* 					     else SOME (z,y)) *)
	    (* 					 cps0) *)

	    val cps0' = TTS2.addList (TTS2.empty, L.mapPartial TT2.fromTerm cps0)

	    (* val _ = debug (fn _=> print ((Int.toString (TPS2.numItems cps0'))  *)
	    (* 				 ^ " critical pairs found (<= 100 will be examined).\n")) *)

	    val _ = debug (fn _=> print ((Int.toString (TTS2.numItems cps0')) 
					 ^ " critical pairs found (<= 100 will be examined).\n"))

	   val _ = debug (fn _=> print (LU.toStringCommaSquare IOMstrs.prEq (L.map (fn (_,_,_,x,y)=>(x,y)) (TTS2.listItems cps0'))))

	    val cps1 = let 
		val lenMax = L.foldr (fn ((l,r),z) => Int.max (z, Term.termSize l + Term.termSize r)) 0 rs
		val cps00 = List.filter (fn (n,_,_,_,_) => n <= lenMax*2) (TTS2.listItems cps0')
	    in L.take (cps00, Int.min (L.length cps00, 100)) end
				   
	   (* val _ = print "CPS1\n" *)
	   (* val _ = print (Trs.prEqs cps1) *)

	    val cps1' = LU.eliminateDuplication' 
			   (fn ((k0,l0,_,x0,x1),(k1,l1,_,y0,y1)) => 
			       k0 = k1 andalso l0 = l1 
			       andalso (Subst.identicalModuloRenamingRule (x0,x1) (y0,y1)
					orelse
					(k0 - l0 = l0
					 andalso Subst.identicalModuloRenamingRule (x0,x1) (y1,y0))))
			   cps1


	    fun mkSubst vterms = 
		let val maxConstIndex =
			let val cprefixed = L.mapPartial (fn (f,_) => if String.isPrefix "c_" (Fun.toString f)
								      then SOME (Fun.toString f) else NONE) (FIS.listItems faset)
			    val suffixes = L.map (implode o tl o tl o explode) cprefixed
			    val nums = L.mapPartial (Int.fromString) suffixes
			    val max = (L.foldr Int.max 0 nums) + 1
			in max
			end
		    val consts = L.tabulate (L.length vterms, 
					     fn i => Fun.fromString ("c_" ^ (Int.toString (maxConstIndex + i))))
		    val sigma = LP.foldr (fn (vt,c,phi) => VarMap.insert (phi,valOf (Term.varRootOfTerm vt),
									  Fun (c,[],Term.sortOfTerm vt)))
					 VarMap.empty
					 (vterms,consts)
		in sigma end

	    fun mkSubstByGterms vterms gts = 
		let val (sigma,_) = L.foldr (fn (vt,(phi,current)) => let val x =  valOf (Term.varRootOfTerm  vt)
								   val ty = Term.sortOfTerm  vt
								   (* 基底項は順番にとる．とったら最後へ回す． *)
								   val gt = valOf (L.find (fn t => Sort.equal (Term.sortOfTerm t,ty)) current)
								   val next  = (LU.deleteOne' Term.equal gt current) @ [gt]
							       in (VarMap.insert (phi,x,gt), next)
							       end)
					     (VarMap.empty,gts)
					     vterms
		    (* val _ = print ("Subst: " ^ (Subst.toString sigma) ^ "\n") *)
		in sigma end
 

	    (* fun instantiate (t0,t1) =  *)
	    (* 	let val vs = LU.eliminateDuplication' Var.equal  *)
	    (* 					      ((Term.varListInTerm t0) @  (Term.varListInTerm t1)) *)
	    (* 	    val sigma = mkSubst vs *)
	    (* 	in (Subst.applySubst sigma t0, Subst.applySubst sigma t1) *)
	    (* 	end *)

	    fun instantiate (l0,l1,m,t0,t1) = 
		let val sigma = if forGround
				then mkSubstByGterms (TermSet.listItems (Term.varTermSetInTerm m)) gterms
				(* else mkSubst (Term.varListInTerm m) *)
				else mkSubst (TermSet.listItems (Term.varTermSetInTerm m))
		in (l0,l1,Subst.applySubst sigma m, Subst.applySubst sigma t0, Subst.applySubst sigma t1)
		end

	    (* val cps2 = LU.eliminateDuplication' (fn (xx,yy) => TermPair.equal (instantiate xx, instantiate yy)) *)
	    (* 					cps1 *)


(***	    val cps2 = ListMergeSort.sort TTS2.compare (L.map instantiate cps1') ***)
	    val cps2 = rev (L.map instantiate cps1')  (*** **)

	    (* val _ = print "CPS2\n" *)
	    (* val _ = println (LU.toStringCommaSquare Trs.prEq (L.map (fn (_,_,_,t0,t1) =>(t0,t1)) cps2)) *)

	    val lenForManyStepReducts = ref 3 (* 6: lose 15.trs *) (* 2 *)
	    val numForManyStepReducts = ref 10

	    fun strengthenCPS (mid,t0,t1) = 
		let val tset0 = Rewrite.manyStepsReductSet rs (!lenForManyStepReducts) t0
		    val tset1 = Rewrite.manyStepsReductSet rs (!lenForManyStepReducts) t1
		    (* val _ = print ("strenghten " ^ (Term.toString t0)
				   ^ " and " ^ (Term.toString t1) ^ "\n") *)
		    fun cmp (s,t) = Term.termSize s > Term.termSize t
		    val ts0 = if (!numForManyStepReducts) < (TS.numItems tset0)
			      then L.take (ListMergeSort.sort cmp (TS.listItems tset0),
					   !numForManyStepReducts)
			      else TS.listItems tset0

		    (* val _ = if (!numForManyStepReducts) < (TS.numItems tset0)
			      then (print "============\n";
				    print (LU.toStringCommaSquare Term.toString (TS.listItems tset0));
				    print "------------\n";
				    print (LU.toStringCommaSquare Term.toString ts0))
			      else () *)

		    val ts1 = if (!numForManyStepReducts) < (TS.numItems tset1)
			      then L.take (ListMergeSort.sort cmp (TS.listItems tset1),
					   !numForManyStepReducts)
			      else TS.listItems tset1

		    (* val _ = if (!numForManyStepReducts) < (TS.numItems tset1)
			      then (print "============\n";
				    print (LU.toStringCommaSquare Term.toString (TS.listItems tset1));
				    print "------------\n";
				    print (LU.toStringCommaSquare Term.toString ts1))
			      else () *)
		    
		in L.mapPartial (fn x=>x)
				(ListXProd.mapX (fn (y,z) => if Term.equal (y,z) then NONE
							     else if Term.termSize y > Term.termSize z
							     then SOME (mid,y,z)
							     else SOME (mid,z,y))
						(ts0, ts1))
		end

	    (* val cps = (* ListMergeSort.sort comparePair *) *)
	    (* 				 (TPS2.listItems  *)
	    (* 				      (L.foldr (fn ((x,y),set) =>  *)
	    (* 						   if TPS2.numItems set > 200 *)
	    (* 						   then TPS2.add (set, (x,y)) *)
	    (* 						   else TPS2.addList (set,  *)
	    (* 						        (L.map instantiate (strengthenCPS (x,y))))) *)
	    (* 					       TPS2.empty (rev cps2))) *)

	    val cps = TTS2.listItems 
			   (L.foldr (fn ((k,l,m,x,y),set) => 
					if TTS2.numItems set > 200
					then TTS2.add (set, (k,l,m,x,y))
					else TTS2.addList (set, L.mapPartial TT2.fromTerm (strengthenCPS (m,x,y))))
				    TTS2.empty cps2)

	    val _ = debug (fn _=> print "candidates for finding a counterexample:\n" )
	    val _ = debug (fn _=> print (Trs.prEqs (L.map (fn (_,_,_,x,y) => (x,y)) cps)))

	    val InstantiatedCPS = L.map (fn (_,_,m,x,y)=> (m,x,y))
				  (L.take (cps, Int.min (L.length cps, 100)))

	    val _ = print ("obtain " ^ (Int.toString (L.length (InstantiatedCPS)))
			   ^ " candidates for checking non-joinability\n")

(*	    val _ = print ("Number of Cands: " ^ (Int.toString (L.length (InstantiatedCPS))) ^"\n") *)
(*	    val _ = debug (fn _ => print (Trs.prEqs InstantiatedCPS)) *)

	in
	    InstantiatedCPS
	end


    fun checkNonConfluence0 satSolver smtSolver tmpDir forGround gterms njProblem = 
	let val (rs,ss) = case njProblem of
			      CrProblem rs => (rs,rs)
			    | ComProblem (isUnSwitched,rs,ss) => (rs,ss)
						      
	    val InstantiatedCPS = case njProblem of
			 CrProblem rs => mkNonJoinableCands satSolver smtSolver tmpDir forGround gterms rs
		       | ComProblem (isUnSwitched,rs,ss) => Cr.criticalPeaks2 (rs,ss)

	    val _ = if (!useNonJoinableByApproximation) 
		    then (print "check by TCAP-Approximation"; debug (fn _=> print "\n"))
		    else ()

	    val _ = if (!useNonJoinableByApproximation)
		    then L.app (fn cand => nonJoinableCheckByTcaps2 njProblem cand)
				  InstantiatedCPS
		    else ()
				  
(**
	    val icps = if (!useNonJoinableByApproximation)
		       then TPS2.listItems
				(L.foldr (fn (xy,set) => TPS2.addList (set,nonJoinableCheckByTcaps2 njProblem xy))
					 TPS2.empty  (rev InstantiatedCPS))
		       else L.map (fn (_,x,y) => (x,y)) InstantiatedCPS
**)

	    val _ = if (!useNonJoinableByApproximation) 
		    then print " (failure)\n"
		    else ()

	    val _ = if (!useNonJoinableByApproximation) 
		       andalso (!useNonJoinableByRootApprox)
		    then (print "check by Root-Approximation"; debug (fn _=> print "\n"))
		    else ()
	       
	    val _ = (!useNonJoinableByApproximation) 
		    andalso (!useNonJoinableByRootApprox)
		    andalso (isCrProblem njProblem)
		    andalso
		    L.exists (fn cand =>
	       			 nonConfluenceCheckByRootApprox rs cand
	       		     ) InstantiatedCPS
			     
	    val _ = if (!useNonJoinableByApproximation) 
		       andalso (!useNonJoinableByRootApprox)
		    then print " (failure)\n"
		    else ()
			     
	    val _ = if (!useNonJoinableByInterpretationAndOrder) 
 		       andalso (!useNonJoinableByPathOrder)
		    then if (!useNonJoinableByTreeAutomata) 
			 then (print "check by Ordering(rpo), check by Tree-Automata Approximation";
			       debug (fn _=> print "\n"))
			    else (print "check by Ordering(rpo)"; debug (fn _=> print "\n"))
		       else if (!useNonJoinableByTreeAutomata) 
		       then (print "check by Tree-Automata Approximation"; debug (fn _=> print "\n"))
		       else ()

	       val _ = L.exists (fn cand =>
	       			    ((!useNonJoinableByInterpretationAndOrder) 
				     andalso (!useNonJoinableByPathOrder)
				     andalso (nonJoinableCheckByPathOrder satSolver tmpDir njProblem cand))
	       			    orelse
				    ((!useNonJoinableByTreeAutomata) 
				     andalso (nonJoinableCheckByTA njProblem cand)))
(*				(L.take (icps, (L.length icps) div 2)) *)
				(* icps *)
				(L.take (InstantiatedCPS, 
					 Int.min (L.length InstantiatedCPS, 50)))

	       val _ = if ((!useNonJoinableByInterpretationAndOrder) 
			   andalso (!useNonJoinableByPathOrder))
			  orelse (!useNonJoinableByTreeAutomata)
		       then print " (failure)\n"
		       else ()

	       val _ = if (!useNonJoinableByInterpretationAndOrder) 
			  andalso (isCrProblem njProblem) (* currently not supported for COM *)
		       then if (!useNonJoinableByEvenOdd)
			    then if (!useNonJoinableByModThree)
				 then (print "check by Interpretation(mod2,mod3)"; debug (fn _=> print "\n"))
				 else (print "check by Interpretation(mod2)"; debug (fn _=> print "\n"))
			    else ()
		       else ()

	       val _ = (!useNonJoinableByInterpretationAndOrder)
		       andalso (isCrProblem njProblem) (* currently not supported for COM *)
		       andalso L.exists (fn cand =>
					    ((!useNonJoinableByEvenOdd)
					     andalso
	       				     (nonJoinableCheckByEvenOdd smtSolver tmpDir njProblem cand))
	    				    orelse
					    ((!useNonJoinableByModThree)
					    andalso
					    (nonJoinableCheckByModuloThree smtSolver tmpDir njProblem cand))
	       				) 
					(* (L.take (icps, (L.length icps) div 4)) *)
				(L.take (InstantiatedCPS, 
					 Int.min (L.length InstantiatedCPS, 50)))

	       val _ = if (!useNonJoinableByInterpretationAndOrder)
		       then print " (failure)\n"
		       else ()

	       val _ = if (!useNonJoinableByApproximation)
			  andalso (!useNonJoinableByApprox )
			  andalso (isCrProblem njProblem) (* currently not supported for COM *)
		       then if (!useNonJoinableByInterpretationAndOrder)
			       andalso (!useNonJoinableByPolyOrder)
			    then (print "check by Descendants-Approximation, check by Ordering(poly)"; debug (fn _=> print "\n"))
			    else (print "check by Descendants-Approximation"; debug (fn _=> print "\n"))
		       else if (!useNonJoinableByInterpretationAndOrder)
			       andalso (!useNonJoinableByPolyOrder)
			       andalso (isCrProblem njProblem) (* currently not supported for COM *)		   
		       then (print "check by Ordering(poly)"; debug (fn _=> print "\n"))
		       else ()

	       val _ = if isCrProblem njProblem (* currently not supported for COM *)
		       then L.exists (fn cand =>
				    ((!useNonJoinableByApproximation)
				     andalso (!useNonJoinableByApprox )
				     andalso (nonJoinableCheckByApprox njProblem cand))
	    			    orelse
				    ((!useNonJoinableByInterpretationAndOrder)
				     andalso (!useNonJoinableByPolyOrder)
				     andalso
	    			     (nonJoinableCheckByPolyOrder smtSolver tmpDir njProblem cand))
				(*  orelse *)
				(*  (nonConfluenceCheckByModuloFour smtSolver tmpDir rs cand) *)
	    		    )  (** test (L.take (icps, (L.length icps) div 8)) **)
			(*	(L.take (icps, Int.min (L.length icps, 50))) *)
				(L.take (InstantiatedCPS, 
					 Int.min (L.length InstantiatedCPS, 50)))
		       else false


	       val _ = if ((!useNonJoinableByApproximation)
			   orelse (!useNonJoinableByInterpretationAndOrder))
			  andalso isCrProblem njProblem
		       then print " (failure)\n"
		       else ()
	   
               val maxNumOfSteps = 5
				       
	in false
	end
	handle DisprovedByTcaps2 (mid,t0,t1) => finish2 (mid,t0,t1)
	     | DisprovedByRootApprox (t0,t1) => finish (t0,t1)
	     | DisprovedByPathOrder (mid,t0,t1) => finish2 (mid,t0,t1)
	     | DisprovedByTA (t0,t1) => finish (t0,t1)
	     | DisprovedByEvenOdd (mid,t0,t1) => finish2 (mid,t0,t1)
	     | DisprovedByApprox (mid,t0,t1) => finish2 (mid,t0,t1)
	     | DisprovedByPolyOrder (mid,t0,t1) => finish2 (mid,t0,t1)
	     | DisprovedByModThree (mid,t0,t1) => finish2 (mid,t0,t1)
	     | DisprovedByModFour (t0,t1) => finish (t0,t1)

    fun checkNonConfluence satSolver smtSolver tmpDir rs = 
	checkNonConfluence0 satSolver smtSolver tmpDir false [] (CrProblem rs)

    fun checkNonCommutative satSolver smtSolver tmpDir isUnSwitched (rs,ss) = 
	checkNonConfluence0 satSolver smtSolver tmpDir false [] (ComProblem (isUnSwitched,rs,ss))

    fun checkNonGroundConfluence satSolver smtSolver tmpDir gterms rs = 
	checkNonConfluence0 satSolver smtSolver tmpDir true gterms (CrProblem rs)

			    

    end (* of local *)

end (* of structre *)
