(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/order.sml
 * description: check of recursive path ordering, SAT-encoding of RPO constraints
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature ORDER = 
sig
    datatype rpo_status = MUL | LEX
    datatype encoding_type = GT | GE | EQ
    val runDebug: bool ref

    val stringToRpoStatus: string -> rpo_status
    val eqRpoStatus: rpo_status * rpo_status -> bool
    val prRpoStatus: rpo_status -> string 
    val rdRpoStatus: string -> rpo_status

    type prec  = Term.fun_key list
    val comparePrec: prec -> Term.fun_key * Term.fun_key -> order

    type qprec  = int FunMap.map
    val compareQprec: qprec -> Term.fun_key * Term.fun_key -> order

    (* $BM?$($i$l$?(B precedence $B$N85$G$N=g=x%A%'%C%/(B *)
    (* lex with status on quasi precedence *)
    val qlposCheck: (Term.fun_key * Term.fun_key -> order)
		    -> (Term.fun_key -> Term.term list -> Term.term list)
		    -> encoding_type
		    -> Term.term * Term.term
		    -> bool

    (* mul on quasi precedence *)
    val qmpoCheck: (Term.fun_key * Term.fun_key -> order)
		   -> encoding_type
		   -> Term.term * Term.term
		   -> bool

    (* mul&lex with status on quasi precedence *)
    (* $BF1$8(B weight $B$N4X?t$O!$(Bmul or lex $B$,F1$8$G$J$$$H%@%a(B *)
    val qrpoCheck: (Term.fun_key * Term.fun_key -> order)
		   -> (Term.fun_key -> bool)
		   -> (Term.fun_key -> Term.term list -> Term.term list)
		   -> encoding_type
		   -> Term.term * Term.term
		   -> bool

    (* mul&lex with status on quasi precedence *)
    (* mul$B$NItJ,$N%A%'%C%/$K(B SAT$B%=%k%P$rMxMQ(B *)
    val encodeRpoCheck: (Term.fun_key * Term.fun_key -> order)
			-> (Term.fun_key -> bool)
			-> (Term.fun_key -> Term.term list -> Term.term list)
			-> int ref
			-> encoding_type
			-> Term.term * Term.term
			-> Prop.prop * int

    (* $B=g=x%A%'%C%/!'(Bprecedence $B$O<+M3$K$H$l$k(B *)
    (* SAT $B$X$N%(%s%3!<%I(B *)
    (* lex on strict-precedence *)
    val lpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
		     -> int ref
		     -> encoding_type
		     -> Term.term * Term.term
		     -> Prop.prop

    (* lex on quasi-precedence *)
    val qlpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
		      -> (Term.fun_key * Term.fun_key -> Prop.prop)
		      -> int ref
		      -> encoding_type
		      -> Term.term * Term.term
		      -> Prop.prop

    (* mul&lex on strict-precedence *)
    val rpoEncoding: FunSet.set * FunSet.set
		     -> (Term.fun_key * Term.fun_key -> Prop.prop)
		     -> (Term.fun_key -> Prop.prop)
		     -> (Term.fun_key * int * int -> Prop.prop)
		     -> int ref
		     -> encoding_type
		     -> Term.term * Term.term
		     -> Prop.prop

    (* mul&lex on quasi-precedence *)
    val qrpoEncoding: FunSet.set * FunSet.set
		      -> (Term.fun_key * Term.fun_key -> Prop.prop)
		      -> (Term.fun_key * Term.fun_key -> Prop.prop)
		      -> (Term.fun_key -> Prop.prop)
		      -> (Term.fun_key * int * int -> Prop.prop)
                      -> ((Term.fun_key * int) * (Term.fun_key * int) -> Prop.prop)
		      -> int ref
		      -> encoding_type
		      -> Term.term * Term.term
		      -> Prop.prop

    (* lex on strict-precedence with argument filtering *)
    val afLpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
		       -> (Term.fun_key -> Prop.prop)
		       -> (Term.fun_key * int -> Prop.prop)
		       -> int ref
		       -> encoding_type
		       -> Term.term * Term.term
		       -> Prop.prop

    (* lex on quasi-precedence with argument filtering *)
    val afQlpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key -> Prop.prop)
			-> (Term.fun_key * int -> Prop.prop)
			-> int ref
			-> encoding_type
			-> Term.term * Term.term
			-> Prop.prop

    (* lex with status on strict-precedence with argument filtering *)
    val afLposEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key * int * int -> Prop.prop)
			-> (Term.fun_key -> Prop.prop)
			-> (Term.fun_key * int -> Prop.prop)
			-> int ref
			-> encoding_type
			-> Term.term * Term.term
			-> Prop.prop

    (* lex with status on quasi-precedence with argument filtering *)
    val afQlposEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
			 -> (Term.fun_key * Term.fun_key -> Prop.prop)
			 -> (Term.fun_key * int * int -> Prop.prop)
			 -> (Term.fun_key -> Prop.prop)
			 -> (Term.fun_key * int -> Prop.prop)
			 -> int ref
			 -> encoding_type
			 -> Term.term * Term.term
			 -> Prop.prop

    (* mul status on strict-precedence with argument filtering *)
    val afMpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
		       -> (Term.fun_key -> Prop.prop)
		       -> (Term.fun_key * int -> Prop.prop)
		       -> int ref
		       -> encoding_type
		       -> Term.term * Term.term
		       -> Prop.prop

    (* mul status on quasi-precedence with argument filtering *)
    val afQmpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key -> Prop.prop)
			-> (Term.fun_key * int -> Prop.prop)
			-> int ref
			-> encoding_type
			-> Term.term * Term.term
			-> Prop.prop

    (* mul&lex status on strict-precedence with argument filtering *)
    val afRpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
		       -> (Term.fun_key -> Prop.prop)
		       -> (Term.fun_key * int * int -> Prop.prop)
		       -> (Term.fun_key -> Prop.prop)
		       -> (Term.fun_key * int -> Prop.prop)
		       -> int ref
		       -> encoding_type
		       -> Term.term * Term.term
		       -> Prop.prop

    (* mul&lex with status on quasi-precedence with argument filtering *)
    val afQrpoEncoding: (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key * Term.fun_key -> Prop.prop)
			-> (Term.fun_key -> Prop.prop)
			-> (Term.fun_key * int * int -> Prop.prop)
			-> (Term.fun_key -> Prop.prop)
			-> (Term.fun_key * int -> Prop.prop)
			-> int ref
			-> encoding_type
			-> Term.term * Term.term
			-> Prop.prop

    exception OrderError of string
end;

structure Order : ORDER =
struct

local

    open Term
    open Trs
    open Prop
    open PrintUtil (* diff *)

    structure VS = VarSet
    structure VM = VarMap
    structure A = Atom
    structure FS = FunSet
    structure AT = AtomTable
    structure FPS = FunPairSet
    structure FPT = FunPairTable
    structure L = List
    structure LP = ListPair
    structure FIS = FunIntSet
    structure FIT = FunIntTable
    structure IPM = IntPairMap
    structure FIIT = FunIntIntTable
    structure TP = TermPair
    structure TPM = TermPairMap
    structure ILM = IntListMap2
    structure ILPM = IntListPairMap
    (*  structure CP = Compiler.Profile *)

    exception OrderError of string

    fun member x ys = isSome (L.find (fn y => x = y) ys) 

    fun containVar [] _ = false
      | containVar ((Var (x,_))::ts) y =
	(Var.equal (x,y)) orelse (containVar ts y)
      | containVar ((Fun (_,ss,_))::ts) y =
	(containVar ss y) orelse (containVar ts y)

in 

val runDebug = ref false
fun debug f = if !runDebug then f () else ()

datatype rpo_status = MUL | LEX
datatype encoding_type = GT | GE | EQ

fun eqRpoStatus (MUL,MUL) = true
  | eqRpoStatus (LEX,LEX) = true
  | eqRpoStatus (_,_) = false

fun stringToRpoStatus str =
    if str = "Lex" then LEX
    else if str = "Mul" then MUL
    else (print str; raise OrderError "stringToRpoStatus: invalid string")

val rdRpoStatus = stringToRpoStatus

fun prRpoStatus MUL = "Mul"
  | prRpoStatus LEX = "Lex"

exception OrderError of string;

(************************************************************************************)

type prec  = Term.fun_key list (* $BBg$-$$=g$N%j%9%H(B *)

fun comparePrec prec (f,g) = 
    if Fun.equal (f,g)
    then EQUAL
    else case (L.find (fn x => (Fun.equal (f,x)) orelse (Fun.equal (g,x))) prec)
	  of SOME y => if Fun.equal (f,y) then GREATER else LESS
	   | NONE => raise OrderError "comparePrec: function without prec.\n"

(************************************************************************************)

type qprec  = int FunMap.map  (* $B=E$_$E$1(B *)

fun compareQprec qprec (f,g) = 
    case (FunMap.find (qprec,f),  FunMap.find (qprec,g)) of
	(SOME i, SOME j) => Int.compare (i,j)
      | _ => (print ("compareQprec: function without prec." ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "\n");
	      raise OrderError "comparePrec: function without prec.\n")

(************************************************************************************)
(* qlposCheck:  forall l -> r \in R. l >lpo r $B$+H=Dj(B                                   *)
(************************************************************************************)

fun qlposCheck compare lex checkType (l,r) = 
    let
	fun grterQlpos (Var _) _ = false
	  | grterQlpos (t as Fun (f,ts,_)) (s as (Var _)) = 
	    L.exists (fn ti => isSome (grterEqQlpos ti s)) ts
	  | grterQlpos (t as Fun (f,ts,_)) (s as Fun (g,ss,_)) = 
	    case compare (f,g) 
	     of GREATER => L.all (grterQlpos t) ss
	      | EQUAL => grterQlposLex t s (lex f ts) (lex g ss)
	      | LESS => L.exists (fn ti => isSome (grterEqQlpos ti s)) ts
	and grterEqQlpos (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then SOME EQUAL else NONE
	  | grterEqQlpos (Var _) (Fun _) = NONE
	  | grterEqQlpos (Fun (_,ts,_)) (s as (Var _)) =
	    if (L.exists (fn ti => isSome (grterEqQlpos ti s)) ts)
	    then SOME GREATER
	    else NONE
	  | grterEqQlpos (t as Fun (f,ts,_)) (s as Fun (g,ss,_)) =
	    case compare (f,g) 
	     of GREATER => if (L.all (grterQlpos t) ss)
			   then SOME GREATER
			   else NONE
	      | EQUAL => grterOrEqQlposLex t s (lex f ts) (lex g ss)
	      | LESS => if (L.exists (fn ti => isSome (grterEqQlpos ti s)) ts)
			then SOME GREATER
			else NONE
	and grterQlposLex t s [] [] = false
	  (* 	    | grterQlposLex t s [] (s0::ss) = L.all (grterQlpos t) (s0::ss);; corrected 08/07/30 *)
	  | grterQlposLex t s [] (s0::ss) = false
	  | grterQlposLex t s (t0::ts) [] = true
	  | grterQlposLex t s (t0::ts) (s0::ss) = 
	    case grterEqQlpos t0 s0
	     of SOME GREATER => L.all (grterQlpos t) ss
	      | SOME EQUAL => grterQlposLex t s ts ss
	      | NONE => L.exists (fn ti => isSome (grterEqQlpos ti s)) ts
	and grterOrEqQlposLex t s [] [] = SOME EQUAL
	  | grterOrEqQlposLex t s [] (s0::ss) = NONE
	  (* 	      if (L.all (grterQlpos t) (s0::ss)) ; corrected 08/07/30 *)
	  (* 	      then SOME GREATER *)
	  (* 	      else NONE *)
	  | grterOrEqQlposLex t s (t0::ts) [] = SOME GREATER
	  | grterOrEqQlposLex t s (t0::ts) (s0::ss) = 
	    case grterEqQlpos t0 s0
	     of SOME GREATER => if (L.all (grterQlpos t) ss)
				then SOME GREATER
				else NONE
	      | SOME EQUAL => grterOrEqQlposLex t s ts ss
	      | NONE => if (L.exists (fn ti => isSome (grterEqQlpos ti s)) ts)
			then SOME GREATER
			else NONE	  
    in
	case checkType of
	    GT => grterQlpos l r
	  | GE => isSome (grterEqQlpos l r)
	  | EQ => case (grterEqQlpos l r) of
		      SOME EQUAL => true | _ => false
    end


(************************************************************************************)
(* qmpoCheck:  forall l -> r \in R. l >qmpo r $B$+H=Dj(B                                 *)
(************************************************************************************)

fun qmpoCheck compare checkType (l,r) = 
    let
	fun grterQmpo (Var _) _ = false
	  | grterQmpo (t as Fun (f,ts,_)) (s as (Var _)) = 
	    L.exists (fn ti => grterEqQmpo ti s) ts
	  | grterQmpo (t as Fun (f,ts,_)) (s as Fun (g,ss,_)) = 
	    case compare (f,g) 
	     of GREATER => (majo t ss) orelse (alpha ts s)
	      | EQUAL => (grterQmpoMul ts ss) orelse (alpha ts s)
	      | LESS => alpha ts s

	and eqQmpo (Var (x,_)) (Var (y,_)) = Var.equal (x,y)
	  | eqQmpo (Var _) (Fun _) = false
	  | eqQmpo (Fun _) (Var _) = false
	  | eqQmpo (t as Fun (f,ts,_)) (s as Fun (g,ss,_)) =
	    case compare (f,g) 
	     of EQUAL => eqQmpoMul ts ss
	      | _ => false

	and grterEqQmpo s t = (eqQmpo s t) orelse (grterQmpo s t) 

 	and alpha ss t = List.exists (fn si => grterEqQmpo si t) ss  

 	and majo s ts = List.all (fn tj => grterQmpo s tj) ts  

	and findOne x ys = 
	    let exception GOT of (Term.term * (Term.term list)) option
		fun get _ [] _  = raise GOT NONE
		  | get x (y::ys) rest = 
		    if eqQmpo x y 
		    then raise GOT (SOME (y, L.revAppend (rest,ys)))
		    else get x ys (y::rest)
	    in get x ys []
	       handle GOT ans => ans
	    end

	and eqQmpoMul [] [] = true
	  | eqQmpoMul (t0::ts) [] = false
	  | eqQmpoMul [] (s0::ss) = false
	  | eqQmpoMul (t0::ts) (s0::ss) = 
	    let 
	    in
		if (length ss) <> (length ts)
		then false
		else case findOne t0 (s0::ss) of 
			 SOME (s0',ss') => eqQmpoMul ts ss'
		       | NONE => false
	    end

	and removeIntersect xs ys = 
	    let fun remove [] xrest ys = (L.rev xrest, ys)
		  | remove (x::xs) xrest ys =
		    case findOne x ys of
			SOME (y',ys') => remove xs xrest ys'
		      | NONE => remove xs (x::xrest) ys
	    in remove xs [] ys
	    end

	and grterQmpoMul ss ts = 
	    case removeIntersect ss ts of
		([],ys) => false
	      | (x::xs,[]) => true
	      | (x::xs,y::ys) => L.all
				     (fn yi => L.exists (fn xj => grterQmpo xj yi) (x::xs))
				     (y::ys)
    in
	case checkType of
	    GT => grterQmpo l r
	  | GE => grterEqQmpo l r
	  | EQ => eqQmpo l r
    end



(************************************************************************************)
(* qrpoCheck:  forall l -> r \in R. l >qrpo r $B$+H=Dj(B                                 *)
(************************************************************************************)

fun qrpoCheck compare isMul lex checkType (l,r) = 
    let
	fun grterQrpo (Var _) _ = false
	  | grterQrpo (t as Fun (f,ts,_)) (s as (Var _)) = 
	    L.exists (fn ti => grterEqQrpo ti s) ts
	  | grterQrpo (t as Fun (f,ts,_)) (s as Fun (g,ss,_)) = 
	    case compare (f,g) 
	     of GREATER => (L.all (grterQrpo t) ss) orelse (alpha ts s)
	      | EQUAL => if isMul f
			 then (grterQrpoMul ts ss) orelse (alpha ts s)
			 else (grterQrpoLex t s (lex f ts) (lex g ss)) orelse (alpha ts s)
	      | LESS => alpha ts s

	and eqQrpo (Var (x,_)) (Var (y,_)) = Var.equal (x,y)
	  | eqQrpo (Var _) (Fun _) = false
	  | eqQrpo (Fun _) (Var _) = false
	  | eqQrpo (t as Fun (f,ts,_)) (s as Fun (g,ss,_)) =
	    case compare (f,g) 
	     of EQUAL => (case (isMul f, isMul g) of
			      (true,true) => eqQrpoMul ts ss
			    | (false,false) => L.all (fn (ti,si)=> eqQrpo ti si)
						     (LP.zip (ts,ss))
			    | (_,_) => (print "eqQrpo: mul/lex not syncronized.\n"; 
					raise OrderError "eqQrpo:mul/lex not sycronized.\n"))
	      | _ => false

	and grterEqQrpo s t = (eqQrpo s t) orelse (grterQrpo s t) 

 	and alpha ss t = List.exists (fn si => grterEqQrpo si t) ss  

 	and majo s ts = List.all (fn tj => grterQrpo s tj) ts  

	and findOne x ys = 
	    let exception GOT of (Term.term * (Term.term list)) option
		fun get _ [] _  = raise GOT NONE
		  | get x (y::ys) rest = 
		    if eqQrpo x y 
		    then raise GOT (SOME (y, L.revAppend (rest,ys)))
		    else get x ys (y::rest)
	    in get x ys []
	       handle GOT ans => ans
	    end

	and eqQrpoMul [] [] = true
	  | eqQrpoMul (t0::ts) [] = false
	  | eqQrpoMul [] (s0::ss) = false
	  | eqQrpoMul (t0::ts) (s0::ss) = 
	    let 
	    in
		if (length ss) <> (length ts)
		then false
		else case findOne t0 (s0::ss) of 
			 SOME (s0',ss') => eqQrpoMul ts ss'
		       | NONE => false
	    end

	and removeIntersect xs ys = 
	    let fun remove [] xrest ys = (L.rev xrest, ys)
		  | remove (x::xs) xrest ys =
		    case findOne x ys of
			SOME (y',ys') => remove xs xrest ys'
		      | NONE => remove xs (x::xrest) ys
	    in remove xs [] ys
	    end

	and grterQrpoMul ss ts = 
	    case removeIntersect ss ts of
		([],ys) => false
	      | (x::xs,[]) => true
	      | (x::xs,y::ys) => L.all
				     (fn yi => L.exists (fn xj => grterQrpo xj yi) (x::xs))
				     (y::ys)

	(*** ss >rpo^lex tt and \forall ti \in tt. s >rpo ti ***)
	and grterQrpoLex s t [] [] =  false
	  | grterQrpoLex s t (x::xs) [] =  true
	  | grterQrpoLex s t [] (y::ys) = false
	  | grterQrpoLex s t (x::xs) (y::ys) = 
	    if eqQrpo x y 
	    then grterQrpoLex s t xs ys
	    else if grterQrpo x y 
	    then majo s ys
	    else alpha (x::xs) t

    in
	case checkType of
	    GT => grterQrpo l r
	  | GE => grterEqQrpo l r
	  | EQ => eqQrpo l r
    end



(************************************************************************************)
(* encodeRpoCheck:  forall l -> r \in R. l >qrpo r $B$+H=Dj(B                            *)
(************************************************************************************)

local
    open Prop
in
fun encodeRpoCheck compare isMul lex symCount checkType (l,r) = 
    let 
	(*** >rpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun rpoPropGt (Var _) _ = False
	  | rpoPropGt (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then True else False
	  | rpoPropGt (s as Fun (f,[],_)) (t as Fun (g,[],_)) =
	    (case compare (f,g) of 
		 GREATER => True
	       | _ => False)
	  | rpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else case compare (f,g) of 
		     GREATER => majoProp s ts
		   | EQUAL => if isMul f
			      then let val (p1,q1) = mulPropGe2 ss ts 
				   in if p1 = True andalso q1 = False
				      then True
				      else if p1 = False orelse q1 = True
				      then False
				      else if p1 = True
				      then Neg q1
				      else  if q1 = False
				      then p1
				      else Conj [p1, Neg q1]
				   end
			      else  
				  lexPropGt f s t (lex f ss) (lex g ts)
		   | LESS => alphaProp ss t

	(*** s >=rpo t $B$K$J$k@)Ls$rJV$9(B 
		       + s =rpo t $B$K$J$k@)Ls$rJV$9(B ***)
	and rpoPropGe2 (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then (True,True) else (False,False)
	  | rpoPropGe2 (Var _) (Fun _) = (False,False)
	  | rpoPropGe2 (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then (True,False) else (False,False)
	  | rpoPropGe2 (s as Fun (f,[],_)) (t as Fun (g,[],_)) = 
	    (case compare (f,g) of 
		 EQUAL => (True,True)
	       | GREATER => (True,False)
	       | LESS => (False, False))
	  | rpoPropGe2 (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) = 
	    if not (VS.isSubset (varSetInTerm t, varSetInTerm s)) then (False,False)
	    else if equal (s,t) then (True,True)
	    else case compare (f,g) of 
		     EQUAL => if isMul f
			      then let val (p1,q1) = mulPropGe2 ss ts 
				   in (p1, q1)
				   end
			      else let val p2 = lexPropGt f s t (lex f ss) (lex f ts)
				       val q2 = if not (VS.isSubset (varSetInTerm s, varSetInTerm t))
						   orelse (length ss) <> (length ts)
						then False
						else
						    let val res = LP.map (fn (x,y) => rpoPropEq x y) (ss,ts)
						    in if (L.all (fn r => r = True) res) then True
						       else if (L.exists (fn r => r = False) res) then False
						       else Conj res
						    end
				   in if q2 = True
				      then (True,True)
				      else if q2 = False
				      then (p2,False)
				      else if p2 = True
				      then (True,q2)
				      else if p2 = False
				      then (q2,q2)
				      else (Disj [p2,q2], q2)
				   end
		   | GREATER => (majoProp s ts, False)
		   | LESS => (alphaProp ss t, False)

	(*** s =rpo t $B$K$J$k@)Ls$rJV$9(B ***)
	and rpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | rpoPropEq (s as (Var _)) (t as (Fun _)) = False
	  | rpoPropEq (Fun (_,ts,_)) (Var (y,_)) = False
	  | rpoPropEq (s as Fun (f,[],_)) (t as Fun (g,[],_)) =
	    (case compare (f,g) of
		 EQUAL => True
	       | _ => False)
	  | rpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    let
		val M = length ss
		val N = length ts
	    in if equal (s,t) 
	       then True
	       else case compare (f,g) of
			GREATER => False
		      | LESS => False
		      | EQUAL => if not (M = N)
				    orelse not (VS.equal (varSetInTerm s,varSetInTerm t))
				 then False
				 else if isMul f
				 then
				     let
					 val count = !symCount+1
					 val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
					 val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
					 val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
					 val _ = symCount := (!symCount) + (M * N)
					 (* count, ..., count + M*N -1 are available *)
					 fun pvar (i,j) = count + i * N + j
				     in
					 (Conj
					      [Conj (L.tabulate (M,
								 fn i =>
								    Conj (L.tabulate (N,
										      fn j =>
											 Imp (Atom (pvar (i,j)),
											      rpoPropEq 
												  (L.nth (ss,i))
												  (L.nth (ts,j))))))),
					       (* it is injective and surjective *)
					       Conj (L.tabulate (N,
								 fn j => one (L.tabulate (M,
											  fn i => Atom (pvar (i,j)))))),
					       (* it is a function *)
					       Conj (L.tabulate (M,
								 fn i => Disj(L.tabulate (N,
											  fn j => Atom (pvar (i,j))))))])
				     end
				 else
				     let val res = LP.map (fn (x,y) => rpoPropEq x y) (ss,ts)
				     in
					 if (L.all (fn a => a = True) res) then True
					 else if (L.exists (fn a => a = False) res) then False
					 else Conj res
				     end
	    end


	(*** \exists si \in ss.  si >=rpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(* 	 and  alphaProp ss t = *)
	(*  	      if List.exists (fn si => Term.equal (si,t)) ss  *)
	(*  	      then True  *)
	(*  	      else Disj (L.map (fn si => let val (p,_) = rpoPropGe2 si t in p end) ss) *)
	(* $B0J2<$NJ}$,!$(BDisj [] $B$r@8@.$7$J$$$N$G8zN($,NI$$!%(B*)
	and alphaProp [] t = False
	  | alphaProp (si::ss) t = 
	    if Term.equal (si,t)
	    then True
	    else let val p1 = #1 (rpoPropGe2 si t)
		 in  if p1 = True 
		     then True
		     else if p1 = False 
		     then alphaProp ss t
		     else let val p2 = alphaProp ss t
			  in if p2 = True then True
			     else if p2 = False then p1
			     else Disj [p1,p2]
			  end
		 end
		     
	(*** \forall t \in ts. s >rpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(* 	  and majoProp s [] = True *)
	(* 	    | majoProp s (ti::[]) = rpoPropGt s ti *)
	(* 	    | majoProp s ts = *)
	(* 	      let val ps = L.map (fn ti => rpoPropGt s ti) ts *)
	(* 	      in if List.exists (fn p => p = False) ps *)
	(* 		 then False *)
	(* 		 else Conj ps *)
	(* 	      end *)
	(* $B0J2<$NJ}$,!$(BConj [] $B$r@8@.$7$J$$$N$G8zN($,NI$$!%(B*)
 	and majoProp s [] = True 
 	  | majoProp s (ti::tt) = 
	    let val p = rpoPropGt s ti
	    in if p = True then majoProp s tt
	       else if p = False then False
	       else let val p2 = majoProp s tt
		    in if p2 = True then p
		       else if p2 = False then False
		       else Conj [p,p2]
		    end
	    end

	(*** ss =>rpo^mul tt $B$H$J$k$?$a$N@)Ls(B
		 $B99$K(B ss =rpo^mul tt $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and mulPropGe2 [] [] = (True,True)
	  | mulPropGe2 (s1::ss) [] = (True,False)
	  | mulPropGe2 [] (t1::ts) = (False,False)
	  | mulPropGe2 [s] [t] = rpoPropGe2 s t
	  | mulPropGe2 [s1,s2] [t] = let val (p1,_) = rpoPropGe2 s1 t
					 val (p2,_) = rpoPropGe2 s2 t
				     in if p1 = True orelse p2 = True
					then (True,False)
					else if p1 = False 
					then (p2,False)
					else if p2 = False
					then (p1,False)
					else (Disj [p1,p2], False)
				     end
	  | mulPropGe2 ss ts =
	    let val M = length ss
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]>>="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
		fun pvar (i,j) = count + i * N + j
		val count2 = count + M*N
		fun pvar2 i = count2 + i
		(* val _ = print (">>=mul " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
		val p = 
		    Conj [
			Conj (L.tabulate (M,
					  fn i => 
					     Conj (L.tabulate (N,
							       fn j =>
								  Imp (Atom (pvar (i,j)),
								       IfThenElse (Atom (pvar2 i),
										   (rpoPropEq
											(L.nth (ss,i))
											(L.nth (ts,j))),
										   (rpoPropGt
											(L.nth (ss,i))
											(L.nth (ts,j))))))))),

			(* surjective *)
			Conj (L.tabulate (N,
					  fn j => Disj (L.tabulate (M,
								    fn i => Atom (pvar (i,j)))))),

			(* injective for equals *)
			Conj (L.tabulate (M, 
					  fn i => Imp (Atom (pvar2 i),
						       one (L.tabulate (N,
									fn j => Atom (pvar (i,j)))))))

		    ]

		val q = Conj (L.tabulate (M,
					  fn i => Atom (pvar2 i)))
	    in
		(p, q)
	    end

	(*** ss >rpo^lex tt and \forall ti \in tt. s >rpo ti  ***)
	(***      or  \exists si. si >=rpo t  $B$H$J$k$?$a$N@)Ls(B ***)

	and lexPropGt _ _ _ [] [] = False
	  | lexPropGt _ _ _ (si::ss) [] = True
	  (* 	    | lexPropGt _ s _ [] (ti::ts) = majoProp s (ti::ts); correced 08/07/30 *)
	  | lexPropGt _ _ _ [] (ti::ts) = False
	  | lexPropGt f s t (si::ss) (ti::ts) =
	    if Term.equal (si,ti)
	    then lexPropGt f s t ss ts
	    else let val p = rpoPropGt si ti
		 in if p = False 
		    then alphaProp ss t
		    else if p = True
		    then majoProp s ts
		    else let val p2 = alphaProp (si::ss) t
			 in if p2 = True  then True
			    else if p2 = False
			    then let val p1 = majoProp s ts
				 in if p1 = True then p
				    else if p1 = False then False
				    else Conj [p, p1]
				 end
			    else let val p1 = majoProp s ts
				 in if p1 = True then Disj [p, p2]
				    else if p1 = False then p2
				    else Disj [ Conj [p, majoProp s ts], p2 ]
				 end
			 end
		 end

    in
	case checkType of
	    GT => (rpoPropGt l r, !symCount)
	  | GE => (* bug fix 2015/01/17 *)
	    (* let val (p,q) = rpoPropGe2 l r *)
	    (* in (Conj [p,q], !symCount) *)
	    (* end *)
	    let val (p,_) = rpoPropGe2 l r
	    in (p, !symCount)
	    end
	  | EQ => (rpoPropEq l r, !symCount)
    end
end


(************************************************************************************)
(* lpoEncoding:  forall l -> r \in R. l >lpo r $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(************************************************************************************)

fun lpoEncoding precEncoding symCount encodingType (l,r)  = 
    let
	(*** >lpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun lpoPropGt (Var _) _ = False
	  | lpoPropGt (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then True else False
	  | lpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	       orelse not (VS.isSubset (varSetInTerm t, varSetInTerm s))
	    then False
	    else if Fun.equal (f,g) 
	    then lexPropGt f s t ss ts
	    else 
		Disj [Conj [precEncoding (f,g), majoProp s ts],
		      alphaProp ss t]

	(*** \exists si \in ss.  si >=lpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and alphaProp ss t =
 	    if List.exists (fn si => Term.equal (si,t)) ss 
 	    then True 
 	    else Disj (L.map (fn si => lpoPropGt si t) ss)
		      
	(*** \forall t \in ts. s >lpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp s ts =
	    Conj (L.map (fn ti => lpoPropGt s ti) ts)


	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)

	and lexPropGt _ _ _ [] [] = False
	  | lexPropGt _ _ _ (si::ss) [] = True
	  | lexPropGt _ _ _ [] (ti::ts) = False
	  | lexPropGt f s t (si::ss) (ti::ts) =
	    if Term.equal (si,ti)
	    then lexPropGt f s t ss ts
	    else let val p = lpoPropGt si ti
		 in
		     if p = False
		     then alphaProp ss t
		     else Disj [ Conj [p, majoProp s ts],
				 alphaProp (si::ss) t  ]
		 end

	val _ = symCount  := (!symCount) + 1;  
	val q = Prop.Atom (!symCount)
	val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
				      ^ (case encodingType of
					     GT => " :(gt): "
					   | GE => " :(ge): "
					   | EQ => " :(eq): ")
				      ^ (Term.toString r) ^ " by " 
				      ^ (Int.toString (!symCount)) ^ "\n"))

    in
	case encodingType of
	    GT => Conj [q, Prop.Iff (q, lpoPropGt l r)]
	  | GE => Conj [q, Prop.Iff (q, if Term.equal (l,r)  then True  else lpoPropGt l r)]
	  | EQ => Conj [q, Prop.Iff (q, if Term.equal (l,r) then True else False)]
    end


(************************************************************************************)
(* qlpoEncoding:  forall l -> r \in R. l >lpo r $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* base order $B$O(B quasi-order *)
(* $BF1$8(B precedence $B$G6&DL%"%j%F%#$^$GF1$80z?t$N$H$-!$(BSRS $B<0$K0z?t$,B?$$J}$,Bg(B      *)
(************************************************************************************)

fun qlpoEncoding precEncoding precEqEncoding symCount encodingType (l,r)  = 
    let
	(*** >lpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun qlpoPropGt (Var _) _ = False
	  | qlpoPropGt (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then True else False
	  | qlpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	       orelse not (VS.isSubset (varSetInTerm t, varSetInTerm s))
	    then False
	    else if Fun.equal (f,g) 
  	    then lexMAPropGt s t ss ts 
	    else let val (p1,p2,p3) = lexPropGt s t ss ts
		 in Disj [Conj [Disj [ precEncoding (f,g), 					
				       Conj [precEqEncoding (f,g), p1]],
				p2],
			  p3]
		 end

	(*** s =qlpo t $B$K$J$k@)Ls$rJV$9(B ***)
	and qlpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qlpoPropEq (s as (Var _)) (t as (Fun _)) = False
	  | qlpoPropEq (Fun (_,ts,_)) (Var (y,_)) = False
	  | qlpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if not (length ss = length ts)
	       orelse not (VS.equal (varSetInTerm s,varSetInTerm t))
	    then False
	    else if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then Conj (LP.map (fn (x,y) => qlpoPropEq  x y) (ss,ts))
	    else Conj (precEqEncoding (f,g)::
		       (LP.map (fn (x,y) => qlpoPropEq  x y) (ss,ts)))


	(*** \exists si \in ss.  si >=qlpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and alphaProp [] t = False
	  | alphaProp (si::ss) t = 
	    let val p = qlpoPropEq si t
	    in if p = True
	       then True
	       else let val q = qlpoPropGt si t
		    in
			if q = True
			then True
			else Disj [p,q,alphaProp ss t]
		    end
	    end
		
	(*** \forall t \in ts. s >qlpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp s ts =
	    Conj (L.map (fn ti => qlpoPropGt s ti) ts)


	(*** ss >qlpo^lex tt and \forall ti \in tt. s >qlpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
	and lexMAPropGt _ _ [] [] = False
	  | lexMAPropGt _ t  (si::ss) [] = alphaProp (si::ss) t
	  (* 	    | lexMAPropGt s _ [] (ti::ts) = majoProp s (ti::ts) ; corrected 08/07/30*)
	  | lexMAPropGt _ _ [] (ti::ts) = False
	  | lexMAPropGt s t (si::ss) (ti::ts) =
	    let val p = qlpoPropEq si ti
	    in if p = True
	       then lexMAPropGt s t ss ts
	       else if p = False
	       then let val q = qlpoPropGt si ti
		    in
			if q = False
			then alphaProp ss t
			else Disj [Conj [q, majoProp s ts], 
				   alphaProp (si::ss) t ]
		    end
	       else let val q = qlpoPropGt si ti
		    in
			if q = False
			then Disj [Conj [p, lexMAPropGt s t ss ts],
				   alphaProp ss t]
			else Disj [Conj [p, lexMAPropGt s t ss ts],
				   Conj [q, (majoProp s ts)],
				   alphaProp (si::ss) t ]
		    end
	    end

	(*** ss >qlpo^lex tt $B$H$J$k$?$a$N@)Ls(B***)
	(*** $B$H(B \forall ti \in tt. s >qlpo ti  $B$H$J$k$?$a$N@)Ls(B ***)
	(*** $B$H(B \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
	and lexPropGt s t [] [] = (False,True,False)
	  | lexPropGt s t (si::ss) [] = (True,True,alphaProp (si::ss) t)
	  | lexPropGt s t  [] (ti::ts) = (False,majoProp s (ti::ts),False)
	  | lexPropGt s t  (si::ss) (ti::ts) =
	    let val p = qlpoPropEq si ti
	    in if p = True
	       then lexPropGt s t ss ts
	       else if p = False
	       then let val q = qlpoPropGt si ti
		    in 
			if q = True
			then (True,majoProp s ts,alphaProp (si::ss) t)
			else (q,majoProp s (ti::ts),alphaProp (si::ss) t)
		    end
	       else let val q = qlpoPropGt si ti
		    in 
			if q = True
			then (True,majoProp s ts,alphaProp (si::ss) t)
			else if q = False
			then let val (r1,r2,r3) = lexPropGt s t ss ts
			     in
				 (Conj [p, r1], 
				  Conj [qlpoPropGt s ti, r2],
				  r3)
			     end
			else let val (r1,r2,r3) = lexPropGt s t ss ts
			     in
				 (Disj [Conj [p, r1], q],
				  Conj [qlpoPropGt s ti, r2],
				  Disj [alphaProp [si] t, r3])
			     end
		    end
	    end

	val _ = symCount  := (!symCount) + 1;  
	val q = Prop.Atom (!symCount)
	val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
				      ^ (case encodingType of
					     GT => " :(gt): "
					   | GE => " :(ge): "
					   | EQ => " :(eq): ")
				      ^ (Term.toString r) ^ " by " 
				      ^ (Int.toString (!symCount)) ^ "\n"))

    in
	case encodingType of
	    GT => Conj [q, Prop.Iff (q, qlpoPropGt l r)]
	  | GE => Conj [q, Prop.Iff (q, Disj [qlpoPropGt l r, qlpoPropEq l r])]
	  | EQ => Conj [q, Prop.Iff (q, qlpoPropEq l r)]
    end




(************************************************************************************)
(* rpoEncoding:  forall l -> r \in R. l >rpo r $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* Mul Lex $B$r%Q%i%a!<%?$K$7$?HG!((BCash$B$,;H$($J$$(B; Zantema Z30$B$K$b$=$l$[$I;~4V$,$+$+$i$J$$(B *)
(************************************************************************************)

fun rpoEncoding (mFunSet,lFunSet) retrieve isMulStatus lexStatus symCount encodingType (l,r) =
    let 
	(*** >rpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun rpoPropGt _ _ (Var _) _ = False
	  | rpoPropGt _ _ (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then True else False
	  | rpoPropGt Mul Lex (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	       orelse not (VS.isSubset (varSetInTerm t, varSetInTerm s))
	    then False
	    else if Fun.equal (f,g) 
	    then if FS.member (Mul, f)  
		 then let val (p1,q1) = mulPropGe2 Mul Lex ss ts 
		      in Conj [p1, Neg q1]
		      end
		 else if FS.member (Lex, f) 
		 then lexPropGt Mul Lex f s t ss ts
		 else let val (p1,q1) = mulPropGe2 (FS.add (Mul,f)) Lex ss ts 
			  val p2 = lexPropGt Mul (FS.add (Lex,f)) f s t ss ts
		      in
			  IfThenElse (isMulStatus f,Conj [p1, Neg q1],p2)
		      end
	    else 
		Disj [Conj [retrieve (f,g), majoProp Mul Lex s ts],
		      alphaProp Mul Lex ss t]

		     
	(*** s >=rpo t $B$K$J$k@)Ls$rJV$9(B 
		       + s =rpo t $B$K$J$k@)Ls$rJV$9(B ***)
	and rpoPropGe2 _ _ (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then (True,True) else (False,False)
	  | rpoPropGe2 _ _ (Var _) (Fun _) = (False,False)
	  | rpoPropGe2 _ _ (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then (True,False) else (False,False)
	  | rpoPropGe2 Mul Lex (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) = 
	    if not (VS.isSubset (varSetInTerm t, varSetInTerm s)) then (False,False)
	    else if equal (s,t) then (True,True)
	    else if Fun.equal (f,g)
	    then if FS.member (Mul, f)  
		 then let val (p1,q1) = mulPropGe2 Mul Lex ss ts 
		      in (p1, q1)
		      end
		 else 
		     let val p2 = lexPropGt Mul (FS.add (Lex,f))  f s t ss ts
			 val q2 = if not (VS.isSubset (varSetInTerm s, varSetInTerm t))
				     orelse (length ss) <> (length ts)
				  then False
				  else
				      Conj (LP.map (fn (x,y) => rpoPropEq Mul (FS.add (Lex,f)) x y) 
						   (ss,ts))
		     in
			 if FS.member (Lex, f) 
			 then (Disj [p2,q2], q2)
			 else let val (p1,q1) = mulPropGe2 (FS.add (Mul,f)) Lex ss ts 
			      in (IfThenElse (isMulStatus f, p1,Disj [p2,q2]),
				  IfThenElse (isMulStatus f, q1, q2))
			      end
		     end
	    else
		let val p = Disj [Conj [retrieve (f,g), majoProp Mul Lex s ts],
				  alphaProp Mul Lex ss t]
		in (p, False)
		end
		    
	(*** s =rpo t $B$K$J$k@)Ls$rJV$9(B ***)
	and rpoPropEq _ _ (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | rpoPropEq _ _ (s as (Var _)) (t as (Fun _)) = False
	  | rpoPropEq _ _ (Fun (_,ts,_)) (Var (y,_)) = False
	  | rpoPropEq Mul Lex (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    let
		val M = length ss
		val N = length ts
	    in if equal (s,t) then True
	       else if not (Fun.equal (f,g))
		       orelse not (M = N)
		       orelse not (VS.equal (varSetInTerm s,varSetInTerm t))
	       then False
	       else if FS.member (Lex,f) 
	       then Conj (LP.map (fn (x,y) => rpoPropEq Mul Lex x y) (ss,ts))
	       else 
		   let
		       val count = !symCount+1
		       val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
		       val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		       val _ = debug (fn _ => print ((Int.toString (M*N)) ^ " variables.\n"))
		       val _ = symCount := (!symCount) + (M * N)
		       (* count, ..., count + M*N - 1are available *)
		       fun pvar (i,j) = count + i * N + j
		       val prop =
			   (Conj
				[Conj (L.tabulate (M,
						   fn i =>
						      Conj (L.tabulate (N,
									fn j =>
									   Imp (Atom (pvar (i,j)),
										rpoPropEq (FS.add (Mul,f)) 
											  Lex
											  (L.nth (ss,i))
											  (L.nth (ts,j))))))),
				 (* it is injective and surjective *)
				 Conj (L.tabulate (N,
						   fn j => one (L.tabulate (M,
									    fn i => Atom (pvar (i,j)))))),
				 (* it is a function *)
				 Conj (L.tabulate (M,
						   fn i => Disj(L.tabulate (N,
									    fn j => Atom (pvar (i,j))))))])
		   in
		       if FS.member (Mul,f) 
		       then prop
		       else let val prop2 = Conj (LP.map (fn (x,y) => rpoPropEq Mul 
										(FS.add (Lex,f))
										x y) 
							 (ss,ts))
			    in IfThenElse (isMulStatus f, prop, prop2)
			    end
		   end
	    end


	(*** \exists si \in ss.  si >=rpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and alphaProp Mul Lex ss t =
 	    if List.exists (fn si => Term.equal (si,t)) ss 
 	    then True 
 	    else Disj (L.map (fn si => let val (p,_) = rpoPropGe2 Mul Lex si t in p end) ss)
		      
	(*** \forall t \in ts. s >rpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp Mul Lex s ts =
	    Conj (L.map (fn ti => rpoPropGt Mul Lex s ti) ts)

	(*** ss =>rpo^mul tt $B$H$J$k$?$a$N@)Ls(B
		 $B99$K(B ss =rpo^mul tt $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and mulPropGe2 _ _ [] [] = (True,True)
	  | mulPropGe2 _ _ (s1::ss) [] = (True,False)
	  | mulPropGe2 Mul Lex [s] [t] = rpoPropGe2 Mul Lex s t
	  | mulPropGe2 Mul Lex [s1,s2] [t] = let val (p1,_) = rpoPropGe2 Mul Lex s1 t
						 val (p2,_) = rpoPropGe2 Mul Lex s2 t
					     in (Disj [p1,p2], False)
					     end
	  | mulPropGe2 Mul Lex ss ts =
	    let val M = length ss
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]>>="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
		fun pvar (i,j) = count + i * N + j
		val count2 = count + M*N
		fun pvar2 i = count2 + i
		(* val _ = print (">>=mul " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
		val p = 
		    Conj [
			Conj (L.tabulate (M,
					  fn i => 
					     Conj (L.tabulate (N,
							       fn j =>
								  Imp (Atom (pvar (i,j)),
								       IfThenElse (Atom (pvar2 i),
										   (rpoPropEq Mul Lex
											      (L.nth (ss,i))
											      (L.nth (ts,j))),
										   (rpoPropGt Mul Lex
											      (L.nth (ss,i))
											      (L.nth (ts,j))))))))),

			(* surjective *)
			Conj (L.tabulate (N,
					  fn j => Disj (L.tabulate (M,
								    fn i => Atom (pvar (i,j)))))),

			(* injective for equals *)
			Conj (L.tabulate (M, 
					  fn i => Imp (Atom (pvar2 i),
						       one (L.tabulate (N,
									fn j => Atom (pvar (i,j)))))))

		    ]

		val q = Conj (L.tabulate (M,
					  fn i => Atom (pvar2 i)))
	    in
		(p, q)
	    end


	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
	(*** $B$HJL$K(B ss =rpo tt $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)

	and lexPropGt _ _ _ _ _ [] [] = False
	  | lexPropGt Mul Lex _ _ _ [s] [t] = rpoPropGt Mul Lex s t
	  | lexPropGt Mul Lex f s t ss ts =
	    let val L = length ss (* we assume length ss = length ts *)
		(* val _ = print (">>=lex " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
 		val ops = ref []
 		val gps = LP.map (fn (x,y) => rpoPropGt Mul Lex x y) (ss,ts)
		val lex = Disj (List.map
				    (fn i =>
					let val rest = L.filter (fn j => i <> j)
						       		(L.tabulate (L, fn j => j))
					in
					    if (L.nth (gps,i)) = False
					    then False
					    else
						Conj
					    	    ((L.nth (gps,i))
						     :: (L.map
							     (fn j =>
								 let val ej = rpoPropEq Mul Lex
											(L.nth (ss,j))
											(L.nth (ts,j))
								 in
								     if ej = True
								     then (ops := j::(!ops); True)
								     else
									 (IfThenElse
 									      (lexStatus (f,j,i),
									       ej,
									       Conj [ 
										   lexStatus (f,i,j),
										   rpoPropGt Mul Lex s
 											     (L.nth (ts,j))]))
								 end)
							     rest))
					end)
				    (L.tabulate (L, fn i => i)))


		val alpha = alphaProp Mul Lex
				      (L.mapPartial (fn i => if (member i (!ops))
								orelse (List.nth (gps,i) = False)
							     then NONE
							     else SOME (L.nth (ss,i)))
						    (L.tabulate (L, fn i => i)))
				      t
	    in
		Disj [lex,alpha]
	    end

	val _ = symCount  := (!symCount) + 1;  
	val q = Prop.Atom (!symCount)
	val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
				      ^ (case encodingType of
					     GT => " :(gt): "
					   | GE => " :(ge): "
					   | EQ => " :(eq): ")
				      ^ (Term.toString r) ^ " by " 
				      ^ (Int.toString (!symCount)) ^ "\n"))

    in
	case encodingType of
	    GT => Conj [q, Prop.Iff (q, rpoPropGt mFunSet lFunSet l r)]
	  | GE => (* bug fix 2015/01/17 *)
            (* let val (p1,p2) = rpoPropGe2 mFunSet lFunSet l r *)
            (* in Conj [q, Prop.Iff (q, Conj [p1,p2])] *)
            (* end *)
	    let val (p,_) = rpoPropGe2 mFunSet lFunSet l r
	    in Conj [q, Prop.Iff (q, p)]
	    end
	  | EQ => Conj [q, Prop.Iff (q, rpoPropEq mFunSet lFunSet l r)]
    end


(************************************************************************************)
(* qrpoEncoding:  forall l -> r \in R. l >qrpo r $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* f$B!A(Bg $B$N$H$-(B f,g$B$N(B isMulStatus $B$OF1$8$K8BDj(B                                            *)
(************************************************************************************)

fun qrpoEncoding (mFunSet,lFunSet) precEncoding precEqEncoding
		 isMulStatus lexStatus sameNthStatus symCount encodingType (l,r) =
    let 
	(*** >qrpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun qrpoPropGt _ _ (Var _) _ = False
	  | qrpoPropGt _ _ (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then True else False
	  | qrpoPropGt Mul Lex (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	       orelse not (VS.isSubset (varSetInTerm t, varSetInTerm s))
	    then False
	    else if isASubterm t s (* so that s is a proper subterm *)
	    then True
	    else if Fun.equal (f,g) 
	    then (***  f = g $B$N>l9g(B ***)
		(* mul $B$N>l9g(B *)
		if FS.member (Mul, f)  
		then let val (p1,q1) = mulPropGe2 Mul Lex ss ts 
		     in Conj [p1, Neg q1]
		     end
		(* lex $B$N>l9g(B *)
		else if FS.member (Lex, f) 
		then let val p2 = lexPropGt Mul Lex f s t ss ts
		     in p2
		     end
		(* $BITDj$N>l9g(B *)
		else let val (p1,q1) = mulPropGe2 (FS.add (Mul,f)) Lex ss ts 
			 val p2 = lexPropGt Mul (FS.add (Lex,f)) f s t ss ts
		     in
			 IfThenElse (isMulStatus f,Conj [p1, Neg q1],p2)
		     end
	    else (***  f$B!b(Bg $B$N>l9g(B ***)
		if (FS.member (Lex, f) andalso FS.member (Mul, g))
		   orelse (FS.member (Mul, f) andalso FS.member (Lex, g))
		then (* f$B!A(Bg $B$N2DG=@-$O$J$$(B *)
 		    Disj [Conj [precEncoding (f,g), majoProp Mul Lex s ts],
 			  alphaProp Mul Lex ss t] 
		(* f$B!A(Bg $B$N2DG=@-$"$j(B *)
		(* f$B!A(Bg $B$N$H$-(B f,g$B$N(B isMulStatus $B$OF1$8(B *)
		(* f,g $B$,(B mul $B$N>l9g(B *)
		else if FS.member (Mul,f) orelse FS.member (Mul,g)
		then
		    let val Mul' = FS.add (FS.add (Mul,f), g)
			val (p1,q1) = mulPropGe2 Mul' Lex ss ts 
			val statusF = if FS.member (Mul,f) then True else isMulStatus f
			val statusG = if FS.member (Mul,g) then True else isMulStatus g
		    in Disj [ Conj [statusF,statusG, precEqEncoding (f,g),p1, Neg q1],
			      Conj [precEncoding (f,g), majoProp Mul Lex s ts],
			      alphaProp Mul Lex ss t]
		    end
		else if FS.member (Lex,f) orelse FS.member (Lex,g)
		(* f,g $B$,(B lex $B$N>l9g(B *)
		then
		    let val Lex' = FS.add (FS.add (Lex,f), g)
  			val p2 = lexPropGtDiff Mul Lex' f g s t ss ts 
			val statusF = if FS.member (Lex,f) then True else Neg (isMulStatus f)
			val statusG = if FS.member (Lex,g) then True else Neg (isMulStatus g)
		    in Disj [ Conj [statusF,statusG,precEqEncoding (f,g), p2],
			      Conj [precEncoding (f,g), majoProp Mul Lex s ts],
			      alphaProp Mul Lex ss t]
		    end
		(* f,g $B$,(B $BITDj$N>l9g(B *)
		else 
		    let val Mul' = FS.add (FS.add (Mul,f), g)
			val Lex' = FS.add (FS.add (Lex,f), g)
			val (p1,q1) = mulPropGe2 Mul' Lex ss ts 
 			val p2 = lexPropGtDiff Mul Lex' f g s t ss ts
		    in Disj [ Conj [isMulStatus f, isMulStatus g,
				    precEqEncoding (f,g), p1, Neg q1],
			      Conj [Neg (isMulStatus f), Neg (isMulStatus g),
				    precEqEncoding (f,g), p2],
			      Conj [precEncoding (f,g), majoProp Mul Lex s ts],
			      alphaProp Mul Lex ss t]
		    end

			
	(*** s >=qrpo t $B$K$J$k@)Ls$rJV$9(B 
	       + s =qrpo t $B$K$J$k$?$a$NDI2C$N@)Ls$rJV$9(B ***)
	and qrpoPropGe2 _ _ (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then (True,True) else (False,False)
	  | qrpoPropGe2 _ _ (Var _) (Fun _) = (False,False)
	  | qrpoPropGe2 _ _ (Fun (_,ss,_)) (Var (y,_)) =
	    if (containVar ss y) then (True,False) else (False,False)
	  | qrpoPropGe2 Mul Lex (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) = 
	    if not (VS.isSubset (varSetInTerm t, varSetInTerm s)) then (False,False)
	    else if equal (s,t) then (True,True)
	    else if isASubterm t s (* so that s is a proper subterm *)
	    then (True,False)
	    else if Fun.equal (f,g)
	    then (***  f = g $B$N>l9g(B ***)
		(* mul $B$N>l9g(B *)
		if FS.member (Mul, f)  
		then let val (p1,q1) = mulPropGe2 Mul Lex ss ts 
		     in (p1, q1)
		     end
		(* lex $B$N>l9g(B *)
		else if FS.member (Lex, f) 
		then let val p2 = lexPropGt Mul Lex f s t ss ts
			 val q2 = if not (VS.isSubset (varSetInTerm s, varSetInTerm t))
				  then False
				  else
				      Conj (LP.map (fn (x,y) => qrpoPropEq Mul Lex x y)
						   (ss,ts))
		     in (Disj [p2,q2], q2)
		     end
		(* $BITDj$N>l9g(B *)
		else let val (p1,q1) = mulPropGe2 (FS.add (Mul,f)) Lex ss ts 
			 val p2 = lexPropGt Mul (FS.add (Lex,f)) f s t ss ts
			 val q2 = if not (VS.isSubset (varSetInTerm s, varSetInTerm t))
				     orelse (length ss) <> (length ts)
				  then False
				  else
				      Conj (LP.map (fn (x,y) => qrpoPropEq Mul (FS.add (Lex,f)) x y) 
						   (ss,ts))
		     in (IfThenElse (isMulStatus f, p1, Disj [p2,q2]),
			 IfThenElse (isMulStatus f, q1, q2))
		     end
	    else (***  f$B!b(Bg $B$N>l9g(B ***)
		if (FS.member (Lex, f) andalso FS.member (Mul, g))
		   orelse (FS.member (Mul, f) andalso FS.member (Lex, g))
		then (* f$B!A(Bg $B$N2DG=@-$O$J$$(B *)
		    let val p = Disj [Conj [precEncoding (f,g), majoProp Mul Lex s ts],
				      alphaProp Mul Lex ss t]
		    in (p,False)
		    end
		(* f$B!A(Bg $B$N2DG=@-$"$j(B *)
		(* f,g $B$,(B mul $B$N>l9g(B *)
		else if FS.member (Mul,f) orelse FS.member (Mul,g)
		then
		    let val Mul' = FS.add (FS.add (Mul,f), g)
			val (p1,q1) = mulPropGe2 Mul' Lex ss ts 
			val statusF = if FS.member (Mul,f) then True else isMulStatus f
			val statusG = if FS.member (Mul,g) then True else isMulStatus g
		    in (Disj [ Conj [statusF,statusG,precEqEncoding (f,g), p1],
			       Conj [precEncoding (f,g), majoProp Mul Lex s ts],
			       alphaProp Mul Lex ss t ],
			if (length ss) = (length ts)
			then Conj [precEqEncoding (f,g),q1]
			else False)
		    end
		else if FS.member (Lex,f) orelse FS.member (Lex,g)
		(* f,g $B$,(B lex $B$N>l9g(B *)
		then
		    let val Lex' = FS.add (FS.add (Lex,f), g)
   			val p2 = lexPropGtDiff Mul Lex' f g s t ss ts
			val q2 = if not (VS.isSubset (varSetInTerm s, varSetInTerm t))
				    orelse (length ss) <> (length ts)
				 then False
				 else
				     Conj (LP.map (fn (x,y) => qrpoPropEq Mul Lex' x y)
 						  (ss,ts))
			val statusF = if FS.member (Lex,f) then True else Neg (isMulStatus f)
			val statusG = if FS.member (Lex,g) then True else Neg (isMulStatus g)
		    in (Disj [Conj [statusF,statusG,precEqEncoding (f,g), Disj [p2,q2]],
			      Conj [precEncoding (f,g), majoProp Mul Lex s ts],
			      alphaProp Mul Lex ss t],
			if (length ss) = (length ts)
			then Conj [statusF,statusG,precEqEncoding (f,g), q2]
			else False)
		    end
		(* f,g $B$,(B $BITDj$N>l9g(B *)
		else 
		    let val Mul' = FS.add (FS.add (Mul,f), g)
			val Lex' = FS.add (FS.add (Lex,f), g)
			val (p1,q1) = mulPropGe2 Mul' Lex ss ts 
 			val p2 = lexPropGtDiff Mul Lex' f g s t ss ts 
			val q2 = if not (VS.isSubset (varSetInTerm s, varSetInTerm t))
				    orelse (length ss) <> (length ts)
				 then False
				 else
				     Conj (LP.map (fn (x,y) => qrpoPropEq Mul Lex' x y)
						  (ss,ts))
		    in (Disj [ Conj [isMulStatus f, isMulStatus g,
				     precEqEncoding (f,g), p1],
			       Conj [Neg (isMulStatus f), Neg (isMulStatus g),
				     precEqEncoding (f,g), Disj [p2,q2]],
			       Conj [precEncoding (f,g), majoProp Mul Lex s ts],
			       alphaProp Mul Lex ss t],
			if (length ss) = (length ts)
			then
			    Disj [Conj [isMulStatus f, isMulStatus g,
					precEqEncoding (f,g), q1],
				  Conj [Neg (isMulStatus f), Neg (isMulStatus g),
					precEqEncoding (f,g), q2]]
			else False)
		    end
			
	(*** s =qrpo t $B$K$J$k@)Ls$rJV$9(B ***)
	and qrpoPropEq _ _ (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qrpoPropEq _ _ (s as (Var _)) (t as (Fun _)) = False
	  | qrpoPropEq _ _ (Fun (_,ts,_)) (Var (y,_)) = False
	  | qrpoPropEq Mul Lex (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t) 
	    then True
	    else if (length ss) <> (length ts)
		    orelse not (VS.equal (varSetInTerm s,varSetInTerm t))
	    then False
	    else if isASubterm t s (* so that s is a proper subterm *)
	    then False
	    else if Fun.equal (f,g)
	    then (* f = g $B$N>l9g(B *)
		if FS.member (Lex,f) 
		then Conj (LP.map (fn (x,y) => qrpoPropEq Mul Lex x y) 
				  (ss,ts))
		else if FS.member (Mul,f) 
		then mulPropEq Mul Lex ss ts
		else let val Lex' = FS.add (Lex,f)
			 val Mul' = FS.add (Mul,f)
			 val propM = mulPropEq Mul' Lex ss ts 
			 val propL = Conj (LP.map (fn (x,y) => qrpoPropEq Mul Lex' x y) 
						  (ss,ts))
 		     in IfThenElse (isMulStatus f, propM, propL)
		     end
	    else (***  f$B!b(Bg $B$N>l9g(B ***)
		if (FS.member (Lex, f) andalso FS.member (Mul, g))
		   orelse (FS.member (Mul, f) andalso FS.member (Lex, g))
		then (* f$B!A(Bg $B$N2DG=@-$O$J$$(B *)
		    False
		(* f$B!A(Bg $B$N2DG=@-$"$j(B *)
		(* f$B!A(Bg $B$N$H$-(B f,g$B$N(B Status $B$OF1$8$K8BDj(B *)
		(* f,g $B$,(B mul $B$N>l9g(B *)
		else if FS.member (Mul,f) orelse FS.member (Mul,g)
		then
		    let val Mul' = FS.add (FS.add (Mul,f), g)
			val p = mulPropEq Mul' Lex ss ts
			val statusF = if FS.member (Mul,f) then True else isMulStatus f
			val statusG = if FS.member (Mul,g) then True else isMulStatus g
		    in 
			Conj [statusF,statusG,precEqEncoding (f,g),p]
		    end
		else if FS.member (Lex,f) orelse FS.member (Lex,g)
		(* f,g $B$,(B lex $B$N>l9g(B *)
		then
		    let (* val _  = print ((Fun.toString f) ^(Fun.toString g) ^ "\n") *)
			val Lex' = FS.add (FS.add (Lex,f), g)
    			val q2 = lexPropEq Mul Lex' f g ss ts
			val statusF = if FS.member (Lex,f) then True else Neg (isMulStatus f)
			val statusG = if FS.member (Lex,g) then True else Neg (isMulStatus g)
 		    in 
			Conj [statusF, statusG, 
 			      precEqEncoding (f,g), q2]
		    end
		(* f,g $B$,(B $BITDj$N>l9g(B *)
		else 
		    let val Mul' = FS.add (FS.add (Mul,f), g)
			val Lex' = FS.add (FS.add (Lex,f), g)
			val q1 = mulPropEq Mul' Lex ss ts 
			val q2 = lexPropEq Mul Lex' f g ss ts 
		    in Disj [Conj [isMulStatus f, isMulStatus g,
				   precEqEncoding (f,g), q1],
			     Conj [Neg (isMulStatus f), Neg (isMulStatus g),
				   precEqEncoding (f,g), q2]]
		    end


	and mulPropEq _ _ [] [] = True
	  | mulPropEq Mul Lex [s] [t] = qrpoPropEq Mul Lex s t
	  | mulPropEq Mul Lex ss ts = 
	    let
		val M = length ss (* assum length ss = length ts *)
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString (M*M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * M)
		(* count, ..., count + M*M -1 are available *)
		fun pvar (i,j) = count + i * M + j
		val prop =
		    (Conj
			 [Conj (L.tabulate (M,
					    fn i =>
					       Conj (L.tabulate (M,
								 fn j =>
								    Imp (Atom (pvar (i,j)),
									 qrpoPropEq Mul
										    Lex
										    (L.nth (ss,i))
										    (L.nth (ts,j))))))),
			  (* it is injective and surjective *)
			  Conj (L.tabulate (M,
					    fn j => one (L.tabulate (M,
								     fn i => Atom (pvar (i,j)))))),
			  (* it is a function *)
			  Conj (L.tabulate (M,
					    fn i => Disj(L.tabulate (M,
								     fn j => Atom (pvar (i,j))))))])
	    in
		prop
	    end

	and lexPropEq  _ _ _ _ [] [] = True
	  | lexPropEq Mul Lex _ _ [s] [t] = qrpoPropEq Mul Lex s t
	  | lexPropEq Mul Lex f g ss ts =
	    (* we assume length ss = length tt *)
	    let val xs = L.tabulate (length ss , fn x => x)
	    in Conj 
		   (ListXProd.mapX
			(fn (i,j) =>
			    Imp (sameNthStatus ((f,i),(g,j)),
				 qrpoPropEq Mul Lex (L.nth (ss,i)) (L.nth (ts,j))))
			(xs,xs))
	    end


	(*** \exists si \in ss.  si >=qrpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and alphaProp Mul Lex ss t =
 	    if List.exists (fn si => Term.equal (si,t)) ss 
 	    then True 
 	    else Disj (L.map (fn si => let val (p,_) = qrpoPropGe2 Mul Lex si t in p end) ss)
		      
	(*** \forall t \in ts. s >qrpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp Mul Lex s ts =
	    Conj (L.map (fn ti => qrpoPropGt Mul Lex s ti) ts)

	(*** ss =>qrpo^mul tt $B$H$J$k$?$a$N@)Ls(B
		 $B99$K(B ss =qrpo^mul tt $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and mulPropGe2 _ _ [] [] = (True,True)
	  | mulPropGe2 _ _ (s1::ss) [] = (True,False)
	  | mulPropGe2 Mul Lex [s] [t] = qrpoPropGe2 Mul Lex s t
	  | mulPropGe2 Mul Lex [s1,s2] [t] = let val (p1,_) = qrpoPropGe2 Mul Lex s1 t
						 val (p2,_) = qrpoPropGe2 Mul Lex s2 t
					     in (Disj [p1,p2], False)
					     end
	  | mulPropGe2 Mul Lex ss ts =
	    let val M = length ss
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]>>="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
		fun pvar (i,j) = count + i * N + j
		val count2 = count + M*N
		fun pvar2 i = count2 + i
		(* val _ = print (">>=mul " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
		val p = 
		    Conj [
			Conj (L.tabulate (M,
					  fn i => 
					     Conj (L.tabulate (N,
							       fn j =>
								  Imp (Atom (pvar (i,j)),
								       IfThenElse (Atom (pvar2 i),
										   (qrpoPropEq Mul Lex
											       (L.nth (ss,i))
											       (L.nth (ts,j))),
										   (qrpoPropGt Mul Lex
											       (L.nth (ss,i))
											       (L.nth (ts,j))))))))),

			(* surjective *)
			Conj (L.tabulate (N,
					  fn j => Disj (L.tabulate (M,
								    fn i => Atom (pvar (i,j)))))),

			(* injective for equals *)
			Conj (L.tabulate (M, 
					  fn i => Imp (Atom (pvar2 i),
						       one (L.tabulate (N,
									fn j => Atom (pvar (i,j)))))))

		    ]

		val q = if M = N 
			then Conj (L.tabulate (M, fn i => Atom (pvar2 i)))
			else False
	    in
		(p, q)
	    end

	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)

	and lexPropGt _ _ _ _ _ [] [] = False
	  | lexPropGt Mul Lex _ _ _ [s] [t] = qrpoPropGt Mul Lex s t
	  | lexPropGt Mul Lex f s t ss ts =
	    let val L = length ss (* we assume length ss = length ts *)
		(* val _ = print (">>=lex " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
 		val ops = ref []
 		val gps = LP.map (fn (x,y) => qrpoPropGt Mul Lex x y) (ss,ts)
		val lex = Disj (List.map
				    (fn i =>
					let val rest = L.filter (fn j => i <> j)
						       		(L.tabulate (L, fn j => j))
					in
					    if (L.nth (gps,i)) = False
					    then False
					    else
						Conj
					    	    ((L.nth (gps,i))
						     :: (L.map
							     (fn j =>
								 let val ej = qrpoPropEq Mul Lex
											 (L.nth (ss,j))
											 (L.nth (ts,j))
								 in
								     if ej = True
								     then (ops := j::(!ops); True)
								     else
									 (IfThenElse
 									      (lexStatus (f,j,i),
									       ej,
									       Conj [ 
										   lexStatus (f,i,j),
										   qrpoPropGt Mul Lex s
 											      (L.nth (ts,j))]))
								 end)
							     rest))
					end)
				    (L.tabulate (L, fn i => i)))


		val alpha = alphaProp Mul Lex
				      (L.mapPartial (fn i => if (member i (!ops))
								orelse (List.nth (gps,i) = False)
							     then NONE
							     else SOME (L.nth (ss,i)))
						    (L.tabulate (L, fn i => i)))
				      t
	    in
		Disj [lex,alpha] 
		     
	    end

	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)

	and lexPropGtDiff _ _ _ _ _ _ [] _ = False
	  | lexPropGtDiff _ _ _ _ _ _ _ [] = True
	  | lexPropGtDiff Mul Lex _ _ _ _ [s] ts  = majoProp Mul Lex s ts 
	  | lexPropGtDiff Mul Lex f g s t (ss as si::ss') [tj] = 
	    let val M = L.length ss
	    in Disj (L.tabulate (M,
				 fn i => let val (p,_) = qrpoPropGe2 Mul Lex (L.nth (ss,i)) tj
					 in Conj (p::(L.tabulate (M,
								  fn j => if i = j
									  then True
									  else lexStatus (f,i,j))))
					 end))
	    end
	  | lexPropGtDiff Mul Lex f g s t ss ts =
	    let val M = length ss 
		val N = length ts
		val xs = L.tabulate (M, fn x => x)
		val ys = L.tabulate (N, fn y => y)
 		val ops = ref []
 		val mps = ref []

		val gtMap = ref IPM.empty
		val eqMap = ref IPM.empty

 		val _ = ListXProd.appX
			    (fn (i,j) =>
				let val si = L.nth (ss,i)
				    val tj = L.nth (ts,j)
				    val p = qrpoPropGt Mul Lex si tj 
                                    val q = if p = True
                                            then False
                                            else qrpoPropEq Mul Lex si tj
                                    val _ = if p = True orelse q = False (* alpha $BI,MW$J$7(B *)
                                            then ops:= i::(!ops)
                                            else ()
                                    val _ = if p = True orelse q = True  (* majo $BI,MW$J$7(B *)
                                            then mps:= j::(!mps)  
                                            else ()
                                    val _ = if p <> False andalso q <> True
					    then gtMap := IPM.insert (!gtMap,(i,j),p)
					    else ()
                                    val _ = eqMap := IPM.insert (!eqMap,(i,j),q)
				in () end)
			    (xs,ys)

		(*  		  val _ = print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ ")\n") *)
		(* 		  val _ = print "[" *)
		(* 		  val _ = L.app (print o Term.toString) ss *)
		(* 		  val _ = print "]\n" *)
		(* 		  val _ = print "[" *)
		(* 		  val _ =  L.app (print o Term.toString) ts *)
		(* 		  val _ = print "]\n" *)
			    
                val mj = Conj (L.map 
 				   (fn j => if member j (!mps)
                                            then True
                                            else qrpoPropGt Mul Lex s (L.nth (ts,j)))
                                   ys)

		val lex = Disj (
			List.map
			    (fn ((i,j),p) =>
				let val xs' =  L.filter (fn x => i <> x) xs
				    val ys' =  L.filter (fn y => j <> y) ys
				in
				    Conj (sameNthStatus ((f,i),(g,j))::
					  p::
					  (ListXProd.mapX
					       (fn (i',j') =>
						   IfThenElse 
						       (Conj [lexStatus (f,i',i),
							      lexStatus (g,j',j),
							      sameNthStatus ((f,i'),(g,j'))],
							valOf (IPM.find (!eqMap,(i',j'))),
							Disj [lexStatus (f,i,i'),
							      lexStatus (g,j,j'),
							      Neg (sameNthStatus ((f,i'),(g,j')))]))
       					       (xs',ys')))
				end)
			    (IPM.listItemsi (!gtMap)))

		(* 		  val _ = print (Int.toString (length (IPM.listItemsi (!gtMap))) ^ "\n") *)

			       
                (* $B0z?t$NBg>.$G$O6hJL$,$D$+$J$$$,!$?t$G:9$,$D$/>l9g(B *)
		fun lexSub _ [] = True
 		  | lexSub xs (j::ys) = 
		    Disj (L.map
			      (fn i => let val p = valOf (IPM.find (!eqMap,(i,j)))
				       in if p = False 
					  then False
					  else Conj [p,
						     sameNthStatus ((f,i),(g,j)),
						     lexSub (L.filter (fn i' => i <> i') xs)
							    ys]
				       end)
			      xs)

		val lex2 = if M <= N
			   then False
			   else lexSub xs ys


		val alpha = alphaProp Mul Lex
  				      (L.mapPartial (fn i => if member i (!ops)
							     then NONE
							     else SOME (L.nth (ss,i)))
						    (L.tabulate (M, fn i => i)))
				      t
	    in
  		Disj [Conj [lex,mj],lex2,alpha] 
	    end

	val _ = symCount  := (!symCount) + 1;  
	val q = Prop.Atom (!symCount)

	val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
				      ^ (case encodingType of
					     GT => " :(gt): "
					   | GE => " :(ge): "
					   | EQ => " :(eq): ")
				      ^ (Term.toString r) ^ " by " 
				      ^ (Int.toString (!symCount)) ^ "\n"))
    in
	case encodingType of
	    GT => Conj [q, Prop.Iff (q,qrpoPropGt mFunSet lFunSet l r)]
	  | GE => (* bug fix 2015/01/17 *)
            (* let val (p1,p2) = qrpoPropGe2 mFunSet lFunSet l r *)
	    (* 	val p = Conj [p1,p2] *)
	    (* in Conj [q, Prop.Iff (q,p)] *)
	    (* end *)
            let val (p,_) = qrpoPropGe2 mFunSet lFunSet l r
            in Conj [q, Prop.Iff (q,p)]
	    end
	  | EQ => Conj [q, Prop.Iff (q, qrpoPropEq mFunSet lFunSet l r)]
    end



(************************************************************************************)
(* afLpoEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* permutation $B$J$7(B                                                                  *)
(************************************************************************************)


fun afLpoEncoding precEncoding isCollapse pi 
		  symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	fun lookupGt s t = 
	    case TPM.find (!gtMap, (s,t)) of 
		SOME p => p
	      | NONE => let 
		  val p = lpoPropGt s t
	      in 
		  if p = False orelse p = True
		  then
		      (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		  else
		      let
			  val _ = symCount  := (!symCount) + 1;
			  val q = Prop.Atom (!symCount)
			  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  val _ =  debug (fn _ => print ( "encode (" 
							  ^ (Term.toString s) ^ " :gt: "
							  ^ (Term.toString t) ^ ") by "
							  ^ (Int.toString (!symCount))
                                                          ^ ")\n" ))
		      in (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		      end
	      end

	and lookupEq s t = 
	    case TPM.find (!eqMap, (s,t)) of 
		SOME p => p
	      | NONE => let 
		  val p = lpoPropEq s t
	      in
		  if p = False orelse p = True
		  then
		      (debug (fn _ => print ( "encode (" 
					      ^ (Term.toString s) ^ " :eq: "
					      ^ (Term.toString t) ^ ") with " 
					      ^ (if p = True then "True" else "False")
					      ^ "\n"));
		       eqMap := TPM.insert (!eqMap, (s,t), p); p)
		  else
		      let 
			  val _ = symCount  := (!symCount) + 1;
			  val q = Prop.Atom (!symCount)
			  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  val _ =  debug (fn _ => print ( "encode (" 
							  ^ (Term.toString s) ^ " :eq: "
							  ^ (Term.toString t) ^ ") by "
							  ^ (Int.toString (!symCount))
                                                          ^ ")\n" ))
		      in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		      end
	      end

	(*** >lpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and lpoPropGt (Var _) _ = False
	  | lpoPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | lpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then let val gtArgs = L.tabulate (length ss, 
						  fn i =>lookupGt (L.nth (ss,i)) (L.nth (ts,i)))
		     in
			 IfThenElse (isCollapse f,
				     Disj (L.tabulate (length ss,
						       fn i => Conj [pi (f,i+1), L.nth (gtArgs,i)])),
				     lexPropGt f s t 0 ss ts gtArgs)
		     end
		else IfThenElse 
			 (isCollapse f,
			  Disj (L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  lookupGt (L.nth (ss,i)) t])),
			  Disj [IfThenElse
				    (isCollapse g,
				     Disj (L.tabulate (length ts,
						       fn j => Conj [pi (g,j+1), 
								     lookupGt s (L.nth (ts,j))])),
				     Conj [precEncoding (f,g), majoProp g s ts]),
 				alphaProp f 0 ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and lpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | lpoPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | lpoPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s 
	  | lpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else
		IfThenElse (isCollapse f,
 			    Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
 							   lookupEq (L.nth (ss,j)) t))),
 			    Conj (isCollapse g::
				  L.tabulate (length ts,
					      fn j => Imp (pi (g,j+1),
							   lookupEq s (L.nth (ts,j))))))

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
 	and lpoPropGe s t = Disj [lookupEq s t,lookupGt s t]

	(*** \exists si \in ss.  si >=lpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f j ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,j+i+1), lpoPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >lpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), lookupGt s (L.nth (ts,i)))))
		 
	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt _ _ _ _ [] [] gtArgs = False
	  | lexPropGt f s t i (si::ss) (ti::ts) gtArgs =
	    let val q2 = lexPropGt f s t (i+1) ss ts gtArgs
	    in
		IfThenElse (pi (f,i+1),
			    let val q = lookupEq si ti 
			    in
				if q = True
				then q2
				else let val p = L.nth (gtArgs, i) 
				     in
					 if p = False
					 then Disj [Conj [q,q2],
						    alphaProp f (i+1) ss t]
					 else Disj [ Conj [q,q2],
						     Conj [p, majoProp f s ts],
						     alphaProp f i (si::ss) t  ]
				     end
			    end,
			    q2)
	    end

    in
	case encodingType of 
	    GT => let val p = lookupGt l r 
		      val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
						    ^ " :(gt): " ^ (Term.toString r) ^ " by " 
						    ^ (Prop.printProp p) ^ "\n"))
		  in Conj (p::(!iffProps)) end
	  | GE => let val p = lpoPropGe l r 
		      val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
						    ^ " :(ge): " ^ (Term.toString r) ^ " by " 
						    ^ (Prop.printProp p) ^ "\n"))
		  in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r 
		      val _ = debug (fn _ => print ("end encoding: " ^ (Term.toString l)
						    ^ " :(eq): " ^ (Term.toString r) ^ " by " 
						    ^ (Prop.printProp p) ^ "\n"))
		  in Conj (p::(!iffProps)) end
		      
    end (* of let *)


(*****************************************************)
(*** afLpoEncoding2: OLD VERSION, NOT USED ANYMORE ***)
(*****************************************************)

fun afLpoEncoding2 precEncoding
		   isCollapse pi encodingType (l,r) =
    let 
	(*** >lpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun lpoPropGt _ _ (Var _) _ = False
	  | lpoPropGt Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if containVar ss y
	    then properVarSubterm Col NCol s t 
	    else False
	  | lpoPropGt Col NCol (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else if Fun.equal (f,g)
	    (*** f = g $B$N$H$-(B ***)
	    then if FS.member (Col,f)
		 then Disj (L.tabulate (length ss,
					fn i => Conj [pi (f,i+1),
						      lpoPropGt Col NCol
								(L.nth (ss,i))
								(L.nth (ts,i))]))
		 else if FS.member (NCol,f)
		 then lexPropGt Col NCol f s t 0 ss ts
		 else let val p1 = Disj (L.tabulate (length ss,
						     fn i => Conj [pi (f,i+1),
								   lpoPropGt (FS.add (Col,f)) NCol 
									     (L.nth (ss,i))
									     (L.nth (ts,i))]))
			  val p2 = lexPropGt Col (FS.add (NCol,f)) f s t 0 ss ts
		      in IfThenElse (isCollapse f,p1,p2)
		      end
	    (*** f $B!b(B g $B$N$H$-(B ***)
	    else if FS.member (Col,f)
	    then Disj (L.tabulate (length ss,
				   fn i => Conj [pi (f,i+1),
						 lpoPropGt Col NCol (L.nth (ss,i)) t]))
      	    else if FS.member (Col,g)
	    then Disj (L.tabulate (length ts,
				   fn j => Conj [pi (g,j+1),
						 lpoPropGt Col NCol s (L.nth (ts,j))]))
	    else let 
		val p1 = Disj (L.tabulate (length ss,
					   fn i => Conj [pi (f,i+1),
							 lpoPropGt (FS.add (Col,f)) NCol 
								   (L.nth (ss,i)) t]))
		val p2 = Disj (L.tabulate (length ts,
					   fn j => Conj [pi (g,j+1),
							 lpoPropGt (FS.add (Col,g)) NCol 
								   s (L.nth (ts,j))]))
		val NCol' = FS.add (FS.add (NCol,f), g)
		val p3 = Disj [(Conj [precEncoding (f,g), 
				      majoProp Col NCol' g s ts]),
			       alphaProp Col NCol' f 0 ss t]
	    in
		Disj [ Conj [isCollapse f, p1],
		       Conj [isCollapse g, p2],
		       Conj [Neg (isCollapse f), Neg (isCollapse g), p3] ]
	    end

        (* s $B$O(B $BJQ?t(B t $B$rItJ,9`$H$7$F4^$`$+(B *)
	and varSubterm (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then True else False
	  | varSubterm (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if not (containVar ss y)
	    then False
	    else
		Disj (L.tabulate (length ss,
				  fn i => Conj [pi (f,i+1), 
						varSubterm (L.nth (ss,i)) t]))

        (* s $B$O(B $BJQ?t(B t $B$r??ItJ,9`$H$7$F4^$`$+(B *)
	and properVarSubterm _ _ (Var (x,_)) (Var (y,_)) = False
	  | properVarSubterm Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if not (containVar ss y)
	    then False
	    else if FS.member (Col, f)
	    then Disj (L.tabulate (length ss,
				   fn i => Conj [pi (f,i+1), 
						 properVarSubterm Col NCol (L.nth (ss,i)) t]))
	    else if FS.member (NCol, f)
	    then Disj (L.tabulate (length ss,
				   fn i => Conj [pi (f,i+1), 
						 varSubterm (L.nth (ss,i)) t]))
	    else IfThenElse 
		     (isCollapse f,
		      Disj (L.tabulate (length ss,
					fn i => Conj [pi (f,i+1), 
						      properVarSubterm (FS.add (Col,f)) NCol 
								       (L.nth (ss,i)) t])),
		      Disj (L.tabulate (length ss,
					fn i => Conj [pi (f,i+1), 
						      varSubterm (L.nth (ss,i)) t])))

        (* s $B$O(B $B<B$OJQ?t(B t $B$K$J$k(B *)
	and equalToVar _ _ (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then True else False
	  | equalToVar Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if FS.member (NCol,f) orelse not (containVar ss y)
	    then False
	    else Conj [isCollapse f,
		       Disj (L.tabulate (length ss,
					 fn i => Conj [pi (f,i+1),
						       equalToVar (FS.add (Col,f)) NCol
								  (L.nth (ss,i)) t]))]

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and lpoPropEq _ _ (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | lpoPropEq Col NCol (s as (Var _)) (t as (Fun (g,ts,_))) = equalToVar Col NCol t s 
      	  | lpoPropEq Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = equalToVar Col NCol s t
	  | lpoPropEq Col NCol (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t) 
	    then True
 	    else if FS.isEmpty (FS.intersection (Term.funSetInTerm s, Term.funSetInTerm t)) 
 		    andalso VS.isEmpty (VS.intersection (Term.varSetInTerm s, Term.varSetInTerm t)) 
 	    then False 
	    else 
		if Fun.equal (f,g)
		then Conj (L.tabulate (length ss,
				       fn i => Imp (pi (f,i+1),
						    lpoPropEq Col NCol
							      (L.nth (ss,i)) (L.nth (ts,i)))))
		else if FS.member (Col,f)
		then Conj (L.tabulate (length ss,
				       fn i => Imp (pi (f,i+1), 
						    lpoPropEq Col NCol
							      (L.nth (ss,i)) t)))
		else if FS.member (Col,g)
		then Conj (L.tabulate (length ts,
				       fn j => Imp (pi (g,j+1), 
						    lpoPropEq Col NCol
							      s (L.nth (ts,j)))))
		else 
		    if FS.member (NCol,f) andalso FS.member (NCol,g)
		    then False
		    else if FS.member (NCol,f) 
		    then Conj (isCollapse g::
			       L.tabulate (length ts,
					   fn j => Imp (pi (g,j+1), 
							lpoPropEq (FS.add (Col,g)) NCol 
								  s (L.nth (ts,j)))))
		    else if FS.member (NCol,g) 
		    then Conj (isCollapse f::
			       L.tabulate (length ss,
					   fn i => Imp (pi (f,i+1), 
							lpoPropEq (FS.add (Col,f)) NCol 
								  (L.nth (ss,i)) t)))
		    else Disj [ Conj (isCollapse f::
				      L.tabulate (length ss,
						  fn i => Imp (pi (f,i+1), 
							       lpoPropEq (FS.add (Col,f)) NCol 
									 (L.nth (ss,i)) t))),
				Conj (isCollapse g::
				      L.tabulate (length ts,
						  fn j => Imp (pi (g,j+1), 
							       lpoPropEq (FS.add (Col,g)) NCol 
									 s (L.nth (ts,j)))))]

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
	and lpoPropGe _ _ (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | lpoPropGe Col NCol (s as (Var _)) (t as (Fun (g,ts,_))) = equalToVar Col NCol t s 
      	  | lpoPropGe Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = varSubterm s t
	  | lpoPropGe Col NCol (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t) then True
	    else Disj [lpoPropEq Col NCol s t,lpoPropGt Col NCol s t]

	(*** \exists si \in ss.  si >=lpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp Col NCol f j ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,j+i+1), lpoPropGe Col NCol (L.nth (ss,i)) t]))
		 
	(*** \forall t \in ts. s >lpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp Col NCol g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), lpoPropGt Col NCol s (L.nth (ts,i)))))

	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt _ _ _ _ _ _ [] [] = False
	  | lexPropGt Col NCol f s t i (si::ss) (ti::ts) =
	    let val q2 = lexPropGt Col NCol f s t (i+1) ss ts
	    in
		IfThenElse (pi (f,i+1),
			    let val q = lpoPropEq Col NCol si ti 
			    in
				if q = True
				then q2
				else let val p = lpoPropGt Col NCol si ti
				     in
					 if p = False
					 then Disj [Conj [q,q2],
						    alphaProp Col NCol f (i+1) ss t]
					 else Disj [ Conj [q,q2],
						     Conj [p, majoProp Col NCol f s ts],
						     alphaProp Col NCol f i (si::ss) t  ]
				     end
			    end,
			    q2)
	    end

    in
	let 
	    val result =
		case encodingType of 
		    GT => lpoPropGt FS.empty FS.empty l r
		  | GE => lpoPropGe FS.empty FS.empty l r
		  | EQ => lpoPropEq FS.empty FS.empty l r
	in 
	    result
	end 

    end (* of let *)

(************************************************************************************)
(* afQlpoEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >qlpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* permutation $B$J$7(B, quasi-order $BHG(B                                                   *)
(************************************************************************************)

fun afQlpoEncoding precEncoding precEqEncoding isCollapse pi 
		   symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	fun lookupGt s t = 
	    case TPM.find (!gtMap, (s,t)) of 
		SOME p => p
	      | NONE => let 
		  val p = qlpoPropGt s t
	      in 
		  if p = False orelse p = True
		  then
		      (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		  else
		      let
			  val _ = symCount  := (!symCount) + 1;
			  val q = Prop.Atom (!symCount)
			  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  val _ =  debug (fn _ => print ( "encode (" 
							  ^ (Term.toString s) ^ " :gt: "
							  ^ (Term.toString t) ^ ") by "
							  ^ (Int.toString (!symCount))
                                                          ^ ")\n" ))
		      in (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		      end
	      end

	and lookupEq s t = 
	    case TPM.find (!eqMap, (s,t)) of 
		SOME p => p
	      | NONE => let 
		  val p = qlpoPropEq s t
	      in
		  if p = False orelse p = True
		  then
		      (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		  else
		      let 
			  val _ = symCount  := (!symCount) + 1;
			  val q = Prop.Atom (!symCount)
			  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  val _ =  debug (fn _ => print ( "encode (" 
							  ^ (Term.toString s) ^ " :eq: "
							  ^ (Term.toString t) ^ ") by "
							  ^ (Int.toString (!symCount))
                                                          ^ ")\n" ))
		      in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		      end
	      end

	(*** >qlpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and qlpoPropGt (Var _) _ = False
	  | qlpoPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | qlpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then 
		    IfThenElse (isCollapse f,
				Disj (L.tabulate (length ss,
						  fn i => Conj [pi (f,i+1), 
								lookupGt (L.nth (ss,i)) 
									 (L.nth (ts,i)) ])),
				Disj [lexPropGt f f s t ss ts,
 				      alphaProp f 0 ss t])
      		else IfThenElse 
			 (isCollapse f,
			  Disj (L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  lookupGt (L.nth (ss,i)) t])),
			  Disj [IfThenElse
				    (isCollapse g,
				     Disj (L.tabulate (length ts,
						       fn j => Conj [pi (g,j+1), 
								     lookupGt s (L.nth (ts,j))])),
				     Disj [Conj [precEncoding (f,g), majoProp g 0 s ts],
					   Conj [precEqEncoding (f,g), 
						 lexPropGt f g s t ss ts] ]),
 				alphaProp f 0 ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and qlpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qlpoPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | qlpoPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s 
	  | qlpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else
		IfThenElse (isCollapse f,
 			    Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
 							   lookupEq (L.nth (ss,j)) t))),
 			    Disj [Conj (isCollapse g::
					L.tabulate (length ts,
						    fn j => Imp (pi (g,j+1),
								 lookupEq s (L.nth (ts,j))))),
				  Conj [precEqEncoding (f,g),
					lexPropEq f g s t ss ts]])

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
 	and qlpoPropGe s t = Disj [lookupEq s t,lookupGt s t]

	(*** \exists si \in ss.  si >=qlpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f j ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,j+i+1), qlpoPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >qlpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g j s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,j+i+1), lookupGt s (L.nth (ts,i)))))
		 
	(*** ss >qlpo^lex tt and \forall ti \in tt. s >qlpo ti  ***)
	(*** $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt f g s t ss ts = 
	    let 
 		(* val _ = debug (fn _=>  *)
		(* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "): " *)
		(* 			   ^ "[" ^ (PrintUtil.prSeq Term.toString  ss) ^ "] >> ["  *)
		(* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ "]\n")) *)
		val lexGtMap = ref IPM.empty
		val lenss  = length ss
		val lents  = length ts
		fun lookupLexGt i j  = 
		    if i >= lenss orelse j >= lents
		    then False
		    else case IPM.find (!lexGtMap, (i,j)) of 
			     SOME p => p
			   | NONE => let 
 			       val _ = debug (fn _=> 
						 print ("lookupLexGt (" ^ (Int.toString i) ^ "," ^ (Int.toString j) ^ ")\n"))
			       val p = lexGtSub i j
 			       val _ = debug (fn _=> 
						 print ("-lookupLexGt (" ^ (Int.toString i) ^ "," ^ (Int.toString j) ^ ")\n"))
			       val _ = symCount  := (!symCount) + 1
			       val q = Prop.Atom (!symCount)
			       val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			   in (lexGtMap := IPM.insert (!lexGtMap, (i,j), q); q)
			   end
		and lexGtSub i j = 
		    let val si = L.nth (ss,i)
			val tj = L.nth (ts,j)
			val ts0 = L.drop (ts,j)
 			val _ = debug (fn _=> 
					  print ("lexGtSub (" ^ (Int.toString i) ^ "," ^ (Int.toString j) ^ ")\n"))
		    in
			Disj [ Conj [pi (f,i+1), pi (g,j+1),  
				     Disj [Conj [ lookupLexGt (i+1) (j+1),
						  lookupEq si tj ],
					   Conj [ majoProp g j s ts0,
						  lookupGt si tj ]]],
			       Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
				      lookupLexGt i (j+1) ],
			       Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
				      lookupLexGt (i+1) j ],
			       Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)),
				      lookupLexGt (i+1) (j+1)] ]
		    end
	    in
		lookupLexGt 0 0
	    end

	(*** ss ==^lex tt $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropEq f g s t ss ts = 
	    let (* val _ = debug (fn _=>  *)
		(* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "): " *)
		(* 			   ^ "[" ^ (PrintUtil.prSeq Term.toString  ss) ^ "] == ["  *)
		(* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ "]\n")) *)
		val lexEqMap = ref IPM.empty
		val lenss  = length ss
		val lents  = length ts
		fun lookupLexEq i j  = 
		    if i = lenss andalso j = lents
		    then True
		    else if i >= lenss orelse j >= lents
		    then False
		    else case IPM.find (!lexEqMap, (i,j)) of 
			     SOME p => p
			   | NONE => let val p = lexEqsub i j
					 val _ = symCount  := (!symCount) + 1;
					 val q = Prop.Atom (!symCount)
					 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
				     in (lexEqMap := IPM.insert (!lexEqMap, (i,j), q); q)
				     end
		and lexEqsub i j = 
		    let val si = L.nth (ss,i)
			val tj = L.nth (ts,j)
		    in
			Disj [ Conj [pi (f,i+1), pi (g,j+1),  
				     lookupLexEq (i+1) (j+1),
				     lookupEq si tj ],
			       Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
				      lookupLexEq i (j+1) ],
			       Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
				      lookupLexEq (i+1) j ],
			       Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)),
				      lookupLexEq (i+1) (j+1)] ]
		    end
	    in
		lookupLexEq 0 0
	    end

    in
	case encodingType of 
	    GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
	  | GE => let val p = qlpoPropGe l r in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)



(*   val _ = CP.setProfMode true ; *)

fun afQlpoEncoding2 precEncoding precEqEncoding 
		    isCollapse pi encodingType (l,r) =
    let 
	(* 	  val _ = CP.setProfMode true  *)
	(*** >qlpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	fun qlpoPropGt _ _ _ _ (Var _) _ = False
	  | qlpoPropGt Col NCol Sim Nsim (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if containVar ss y
	    then properVarSubterm Col NCol s t 
	    else False
	  | qlpoPropGt Col NCol Sim Nsim (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else if Fun.equal (f,g)
	    (*** f = g $B$N$H$-(B ***)
	    then if FS.member (Col,f)
		 then Disj (L.tabulate (length ss,
					fn i => Conj [pi (f,i+1),
						      qlpoPropGt Col NCol Sim Nsim
								 (L.nth (ss,i))
								 (L.nth (ts,i))]))
		 else if FS.member (NCol,f)
		 then lexPropGt Col NCol Sim Nsim f s t 0 ss ts
		 else let val p1 = Disj (L.tabulate (length ss,
						     fn i => Conj [pi (f,i+1),
								   qlpoPropGt (FS.add (Col,f)) NCol 
									      Sim Nsim
									      (L.nth (ss,i))
									      (L.nth (ts,i))]))
			  val p2 = lexPropGt Col (FS.add (NCol,f)) Sim Nsim f s t 0 ss ts
		      in IfThenElse (isCollapse f,p1,p2)
		      end
	    (*** f $B!b(B g $B$N$H$-(B ***)
	    else if FS.member (Col,f)
	    then Disj (L.tabulate (length ss,
				   fn i => Conj [pi (f,i+1),
						 qlpoPropGt Col NCol Sim Nsim (L.nth (ss,i)) t]))
      	    else if FS.member (Col,g)
	    then Disj (L.tabulate (length ts,
				   fn j => Conj [pi (g,j+1),
						 qlpoPropGt Col NCol Sim Nsim s (L.nth (ts,j))]))
	    else let 
		val p1 = Disj (L.tabulate (length ss,
					   fn i => Conj [pi (f,i+1),
							 qlpoPropGt (FS.add (Col,f)) NCol 
								    Sim Nsim
								    (L.nth (ss,i)) t]))
		val p2 = Disj (L.tabulate (length ts,
					   fn j => Conj [pi (g,j+1),
							 qlpoPropGt (FS.add (Col,g)) NCol 
								    Sim Nsim
								    s (L.nth (ts,j))]))
		val NCol' = FS.add (FS.add (NCol,f), g)
		val p3 = Disj [(if (FPS.member (Sim, (f,g)))
				   orelse (FPS.member (Sim, (g,f)))
				then False
				else Conj [precEncoding (f,g), 
					   majoProp Col NCol' 
						    Sim (FPS.add (Nsim, (f,g)))
						    g s ts]),
			       (if (FPS.member (Nsim, (f,g)))
				   orelse (FPS.member (Nsim, (g,f)))
				then False
				else Conj [precEqEncoding (f,g),
					   lexMAPropGt Col NCol' 
						       (FPS.add (Sim, (f,g))) Nsim 
						       f g s t ss ts]),
			       alphaProp Col NCol' Sim Nsim f 0 ss t]
	    in
		Disj [ Conj [isCollapse f, p1],
		       Conj [isCollapse g, p2],
		       Conj [Neg (isCollapse f), Neg (isCollapse g), p3] ]
	    end

        (* s $B$O(B $BJQ?t(B t $B$rItJ,9`$H$7$F4^$`$+(B *)
	and varSubterm (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then True else False
	  | varSubterm (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if not (containVar ss y)
	    then False
	    else
		Disj (L.tabulate (length ss,
				  fn i => Conj [pi (f,i+1), 
						varSubterm (L.nth (ss,i)) t]))

        (* s $B$O(B $BJQ?t(B t $B$r??ItJ,9`$H$7$F4^$`$+(B *)
	and properVarSubterm _ _ (Var (x,_)) (Var (y,_)) = False
	  | properVarSubterm Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if not (containVar ss y)
	    then False
	    else if FS.member (Col, f)
	    then Disj (L.tabulate (length ss,
				   fn i => Conj [pi (f,i+1), 
						 properVarSubterm Col NCol (L.nth (ss,i)) t]))
	    else if FS.member (NCol, f)
	    then Disj (L.tabulate (length ss,
				   fn i => Conj [pi (f,i+1), 
						 varSubterm (L.nth (ss,i)) t]))
	    else IfThenElse 
		     (isCollapse f,
		      Disj (L.tabulate (length ss,
					fn i => Conj [pi (f,i+1), 
						      properVarSubterm (FS.add (Col,f)) NCol 
								       (L.nth (ss,i)) t])),
		      Disj (L.tabulate (length ss,
					fn i => Conj [pi (f,i+1), 
						      varSubterm (L.nth (ss,i)) t])))

        (* s $B$O(B $B<B$OJQ?t(B t $B$K$J$k(B *)
	and equalToVar _ _ (Var (x,_)) (Var (y,_)) = 
	    if Var.equal (x,y) then True else False
	  | equalToVar Col NCol (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if FS.member (NCol,f) orelse not (containVar ss y)
	    then False
	    else Conj [isCollapse f,
		       Disj (L.tabulate (length ss,
					 fn i => Conj [pi (f,i+1),
						       equalToVar (FS.add (Col,f)) NCol
								  (L.nth (ss,i)) t]))]

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and qlpoPropEq _ _ _ _ (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qlpoPropEq Col NCol _ _ (s as (Var _)) (t as (Fun (g,ts,_))) = equalToVar Col NCol t s 
      	  | qlpoPropEq Col NCol _ _ (s as Fun (f,ss,_)) (t as Var (y,_)) = equalToVar Col NCol s t
	  | qlpoPropEq Col NCol Sim Nsim (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t) 
	    then True
 	    else if FS.isEmpty (FS.intersection (Term.funSetInTerm s, Term.funSetInTerm t)) 
 		    andalso VS.isEmpty (VS.intersection (Term.varSetInTerm s, Term.varSetInTerm t)) 
 	    then False 
	    else 
		if Fun.equal (f,g)
		then Conj (L.tabulate (length ss,
				       fn i => Imp (pi (f,i+1),
						    qlpoPropEq Col NCol Sim Nsim
							       (L.nth (ss,i)) (L.nth (ts,i)))))
		else if FS.member (Col,f)
		then Conj (L.tabulate (length ss,
				       fn i => Imp (pi (f,i+1), 
						    qlpoPropEq Col NCol Sim Nsim 
							       (L.nth (ss,i)) t)))
		else if FS.member (Col,g)
		then Conj (L.tabulate (length ts,
				       fn j => Imp (pi (g,j+1), 
						    qlpoPropEq Col NCol Sim Nsim 
							       s (L.nth (ts,j)))))
		else if (FPS.member (Nsim, (f,g)))
			orelse (FPS.member (Nsim, (g,f)))
		then (* f \neg\sim  g $B$,$o$+$C$F$$$k>l9g(B *)
		    if FS.member (NCol,f) andalso FS.member (NCol,g)
		    then False
		    else if FS.member (NCol,f) 
		    then Conj (isCollapse g::
			       L.tabulate (length ts,
					   fn j => Imp (pi (g,j+1), 
							qlpoPropEq (FS.add (Col,g)) NCol 
								   Sim Nsim
								   s (L.nth (ts,j)))))
		    else if FS.member (NCol,g) 
		    then Conj (isCollapse f::
			       L.tabulate (length ss,
					   fn i => Imp (pi (f,i+1), 
							qlpoPropEq (FS.add (Col,f)) NCol 
								   Sim Nsim
								   (L.nth (ss,i)) t)))
		    else Disj [ Conj (isCollapse f::
				      L.tabulate (length ss,
						  fn i => Imp (pi (f,i+1), 
							       qlpoPropEq (FS.add (Col,f)) NCol 
									  Sim Nsim
									  (L.nth (ss,i)) t))),
				Conj (isCollapse g::
				      L.tabulate (length ts,
						  fn j => Imp (pi (g,j+1), 
							       qlpoPropEq (FS.add (Col,g)) NCol 
									  Sim Nsim
									  s (L.nth (ts,j)))))]
		else (* f \neg\sim  g $B$+$o$+$i$J$$>l9g(B *)
		    if FS.member (NCol,f) andalso FS.member (NCol,g)
		    then 
			Conj [ precEqEncoding (f,g), 
			       qlpoPropEqList Col NCol Nsim Sim f g ss ts ]
		    else if FS.member (NCol,f) 
		    then Disj [ Conj [Neg (isCollapse g), precEqEncoding (f,g), 
				      qlpoPropEqList Col (FS.add (NCol,g)) Sim Nsim f g ss ts ],
				Conj (isCollapse g::
				      L.tabulate (length ts,
						  fn j => Imp (pi (g,j+1), 
							       qlpoPropEq (FS.add (Col,g)) NCol 
									  Sim Nsim
									  s (L.nth (ts,j)))))]
		    else if FS.member (NCol,g) 
		    then Disj [ Conj [Neg (isCollapse f), precEqEncoding (f,g), 
				      qlpoPropEqList Col (FS.add (NCol,f)) Sim Nsim f g ss ts ],
				Conj (isCollapse f::
				      L.tabulate (length ss,
						  fn i => Imp (pi (f,i+1), 
							       qlpoPropEq (FS.add (Col,f)) NCol 
									  Sim Nsim
									  (L.nth (ss,i)) t))) ]
		    else Disj [ Conj [Neg (isCollapse f), Neg (isCollapse g), precEqEncoding (f,g), 
				      qlpoPropEqList Col (FS.add (FS.add (NCol,f),g)) 
						     Sim Nsim
						     f g ss ts ],
				Conj (isCollapse f::
				      L.tabulate (length ss,
						  fn i => Imp (pi (f,i+1), 
							       qlpoPropEq (FS.add (Col,f)) NCol 
									  Sim Nsim
									  (L.nth (ss,i)) t))),
				Conj (isCollapse g::
				      L.tabulate (length ts,
						  fn j => Imp (pi (g,j+1), 
							       qlpoPropEq (FS.add (Col,g)) NCol 
									  Sim Nsim
									  s (L.nth (ts,j)))))]


	(* f \sim g$B!$(Bf \neq g, f,g : non-collapse $B$r2>Dj(B *)

	and qlpoPropEqList Col NCol Sim Nsim f g ss ts = 
	    let
		val mapEq = ref IPM.empty
		val mapSub = ref IPM.empty

		fun sub [] [] = True
		  | sub [] (j::js) = False
		  | sub (i::is) [] = False
		  | sub (i::is) (j::js) = 
		    case IPM.find (!mapSub, (i,j)) of
			SOME prop => prop
		      | NONE => let val prop =
					Disj [Conj  [ pi (f,i+1), pi (g,j+1),  
						      case IPM.find (!mapEq, (i,j))
						       of SOME p => p
							| NONE => let 
							    val p = qlpoPropEq Col NCol Sim Nsim 
									       (L.nth (ss,i)) 
									       (L.nth (ts,j))
							    val _ = mapEq := IPM.insert 
										 (!mapEq, (i,j), p)
							in p end ],
					      Conj  [ pi (f,i+1), Neg (pi (g,j+1)), sub (i::is) js],
					      Conj  [ Neg (pi (f,i+1)), pi (g,j+1), sub is (j::js)],
					      Conj  [ Neg (pi (f,i+1)), Neg (pi (g,j+1)), sub is js] ]
				    val _ = mapSub := IPM.insert (!mapSub, (i,j), prop)
				in prop end
	    in
		sub (L.tabulate (length ss, fn i=>i)) (L.tabulate (length ts, fn j=>j))
	    end
		

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
	and qlpoPropGe _ _ _ _ (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qlpoPropGe Col NCol _ _ (s as (Var _)) (t as (Fun (g,ts,_))) = equalToVar Col NCol t s 
      	  | qlpoPropGe Col NCol _ _ (s as Fun (f,ss,_)) (t as Var (y,_)) = varSubterm s t
	  | qlpoPropGe Col NCol Sim Nsim (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t) then True
	    else Disj [qlpoPropEq Col NCol Sim Nsim s t,qlpoPropGt Col NCol Sim Nsim s t]

	(*** \exists si \in ss.  si >=qlpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp Col NCol Sim Nsim f j ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,j+i+1), qlpoPropGe Col NCol Sim Nsim (L.nth (ss,i)) t]))
		 

	(*** \forall t \in ts. s >qlpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp Col NCol Sim Nsim g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), qlpoPropGt Col NCol Sim Nsim s (L.nth (ts,i)))))
		 

	(*** ss >lpo^lex tt and \forall ti \in tt. s >lpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt _ _ _ _ _ _ _ _ [] [] = False
	  | lexPropGt Col NCol Sim Nsim f s t i (si::ss) (ti::ts) =
	    let val q2 = lexPropGt Col NCol Sim Nsim f s t (i+1) ss ts
	    in
		IfThenElse (pi (f,i+1),
			    let val q = qlpoPropEq Col NCol Sim Nsim si ti 
			    in
				if q = True
				then q2
				else let val p = qlpoPropGt Col NCol Sim Nsim si ti
				     in
					 if p = False
					 then Disj [Conj [q,q2],
						    alphaProp Col NCol Sim Nsim f (i+1) ss t]
					 else Disj [ Conj [q,q2],
						     Conj [pi (f,i+1), p, 
							   majoProp Col NCol Sim Nsim f s ts],
						     alphaProp Col NCol Sim Nsim f i (si::ss) t  ]
				     end
			    end,
			    q2)
	    end

	(*** ss >qlpo^lex tt and \forall ti \in tt. s >qlpo ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f \neq g, f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexMAPropGt Col NCol Sim Nsim f g s t ss ts =
	    let
		val mapEq = ref IPM.empty
		val mapGt = ref IPM.empty

		val alphaList = L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  qlpoPropGe Col NCol Sim Nsim 
								     (L.nth (ss,i)) t])
		fun alpha i = Disj (L.drop (alphaList,i))

		val majoList = L.tabulate (length ts,
					   fn i => Imp (pi (g,i+1), 
							qlpoPropGt Col NCol Sim Nsim 
								   s (L.nth (ts,i))))
		fun majo j = Disj (L.drop (majoList,j))

		val mapLex = ref IPM.empty

		fun lexsub [] [] _ _ = False
		  | lexsub (i::is) [] _ _ = False
		  | lexsub [] (j::js) _ _ = False
		  | lexsub (i::is) (j::js) (ui::us) (vj::vs) =
		    case IPM.find (!mapLex, (i,j)) of
			SOME prop => prop
		      | NONE => 
			let val prop = 
				let val p =  lexsub is js us vs
				in
				    if p = False
				    then 
					Disj [ 
					    Conj [pi (f,i+1), pi (g,j+1),  
						  Disj [
						      Conj [majo (j+1),
							    case IPM.find (!mapGt, (i,j))
							     of SOME q => q
							      | NONE => let 
								  val q = qlpoPropGt Col NCol Sim Nsim
										     (L.nth (ss,i)) 
										     (L.nth (ts,j))
								  val _ = mapGt:= IPM.insert 
										      (!mapGt, (i,j), q)
							      in q end ],
						      alpha i ] ],
					    Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
						   lexsub (i::is) js (ui::us) vs],
					    Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
						   lexsub is (j::js) us (vj::vs)] ]
				    else
					Disj [
					    Conj [pi (f,i+1), pi (g,j+1),  
						  Disj [alpha i,
							Conj [ p, 
							       case IPM.find (!mapEq, (i,j))
								of SOME q => q
								 | NONE => let 
								     val q = 
									 qlpoPropEq Col NCol 
										    Sim Nsim
										    (L.nth (ss,i)) 
										    (L.nth (ts,j))
								     val _ = mapEq :=
									     IPM.insert 
										 (!mapEq, (i,j), q)
								 in q end ],
							Conj [ majo (j+1),
							       case IPM.find (!mapGt, (i,j))
								of SOME q => q
								 | NONE => let 
								     val q = 
									 qlpoPropGt Col NCol 
										    Sim Nsim
										    (L.nth (ss,i)) 
										    (L.nth (ts,j))
								     val _ = mapGt := 
									     IPM.insert 
										 (!mapGt, (i,j), q)
								 in q end ] ] ],
					    Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
						   lexsub (i::is) js (ui::us) vs],
					    Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
						   lexsub is (j::js) us (vj::vs)],
					    Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)), p ] ]
				end
			    val _ = mapLex := IPM.insert (!mapLex, (i,j), prop)
			in prop end
	    in
		lexsub (L.tabulate (length ss, fn i=>i))
		       (L.tabulate (length ts, fn j=>j))
		       ss ts
	    end
    (* 	  val _ = CP.setProfMode false *)
    in
	let (* val _ = CP.setTimingMode true *)
	    val result =
		case encodingType of 
		    GT => qlpoPropGt FS.empty FS.empty FPS.empty FPS.empty l r
		  | GE => qlpoPropGe FS.empty FS.empty FPS.empty FPS.empty l r
		  | EQ => qlpoPropEq FS.empty FS.empty FPS.empty FPS.empty l r
	(* val _ = CP.report TextIO.stdOut *)
	(* val _ = CP.setTimingMode false *)
	in 
	    result
	end 

    end (* of let *)

(*    val _ = CP.setProfMode false  *)

(************************************************************************************)
(* afLposEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* permutation $B$"$j(B                                                                  *)
(************************************************************************************)

fun afLposEncoding precEncoding lexStatus isCollapse pi 
		   symCount encodingType (l,r) =
    let 
        val gtMap= ref TermPairMap.empty
        val eqMap= ref TermPairMap.empty
        val iffProps = ref []

        fun lookupGt s t = 
	    case TPM.find (!gtMap, (s,t)) of 
                SOME p => p
              | NONE => let val p = lposPropGt s t
			in if p = True orelse p = False
			   then 
			       (gtMap := TPM.insert (!gtMap, (s,t), p); p)
			   else
			       let val _ = symCount  := (!symCount) + 1;
                                   val q = Prop.Atom (!symCount)
                                   val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
                                   val _ =  debug (fn _ => print ( "encode (" 
                                                                   ^ (Term.toString s) ^ " :gt: "
								   ^ (Term.toString t) ^ ") by "
								   ^ (Int.toString (!symCount))
                                                                   ^ ")\n" ))
                               in (gtMap := TPM.insert (!gtMap, (s,t), q); q)
                               end
			end

        and lookupEq s t = 
	    case TPM.find (!eqMap, (s,t)) of 
                SOME p => p
              | NONE => let 
                  val p = lposPropEq s t
	      in if p = True orelse p = False
		 then
		     (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
                         val q = Prop.Atom (!symCount)
                         val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
                         val _ = debug (fn _ => print ( "encode (" 
                                                        ^ (Term.toString s) ^ " :eq: "
							^ (Term.toString t) ^ ") by "
							^ (Int.toString (!symCount))
                                                        ^ ")\n" ))
                     in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
                     end
	      end

	(*** >lpos $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and lposPropGt (Var _) _ = False
	  | lposPropGt (Fun (f,ss,_)) (t as Var (y,_)) = 
	    if containVar ss y
            then
                let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
                    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
                    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
                    val ns = L.tabulate (length ss,fn x=>x)
                in
                    IfThenElse (isCollapse f,
                                Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
                                Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
                end
            else False
	  | lposPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then IfThenElse (isCollapse f,
				 Disj (L.tabulate (length ss,
						   fn i => Conj [pi (f,i+1),
								 lookupGt (L.nth (ss,i))
									  (L.nth (ts,i))])),
				 lexPropGt f s t ss ts)
		else IfThenElse 
			 (isCollapse f,
			  Disj (L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  lookupGt (L.nth (ss,i)) t])),
			  Disj [IfThenElse
				    (isCollapse g,
				     Disj (L.tabulate (length ts,
						       fn j => Conj [pi (g,j+1), 
								     lookupGt s (L.nth (ts,j))])),
				     Conj [(precEncoding (f,g)), (majoProp g s ts)]),
 				alphaProp f (LP.zip (L.tabulate (length ss, fn x=>x), ss)) t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and lposPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
          | lposPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
            if containVar ts x
            then
                Conj [isCollapse g,
                      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
                                                      lookupEq s (L.nth (ts,j))]))]
            else False
          | lposPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s
          | lposPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
            if equal (s,t) then True
            else if Fun.equal (f,g)
            then
                Conj (L.tabulate (length ss,
                                  fn j => Imp (pi (f,j+1),
                                               lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
            else
                IfThenElse (isCollapse f,
                            Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
							   lookupEq (L.nth (ss,j)) t))),
                            Conj (isCollapse g::
                                  L.tabulate (length ts,
                                              fn j => Imp (pi (g,j+1),
                                                           lookupEq s (L.nth (ts,j))))))

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
	and lposPropGe s t = Disj [lookupEq s t,lookupGt s t]

	(*** \exists si \in ss.  si >=lpos t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and alphaProp f ss t = (* f is assumed not collapsing *)
	    Disj (List.map (fn (i,si) => Conj [pi (f,i+1), lposPropGe si t])
			   ss)

	(*** \forall t \in ts. s >lpos t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), lookupGt s (L.nth (ts,i)))))

	(*** ss >lpos^lex tt and \forall ti \in tt. s >lpos ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
	(*  f $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt f s t ss ts = 
	    let val L = length ss (* we assume length ss = length ts *)
		(* val _ = print (">>=lex " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
 		val gps = LP.map (fn (x,y) => lookupGt x y) (ss,ts)
		val lex = L.tabulate (L, 
				      fn i => 
					 Conj 
					     ((List.nth (gps,i))::
					      (pi (f,i+1))::
					      (L.map 
						   (fn j => 
						       IfThenElse (lexStatus (f,j,i),
								   Imp (pi (f,j+1),
									lookupEq (L.nth (ts,j))
										 (L.nth (ss,j))),
								   Conj [lexStatus (f,i,j),
									 Imp (pi (f,j+1),
									      lookupGt s (L.nth (ts,j)))]))
						   (L.filter (fn j => i <> j) 
						       	     (L.tabulate (L, fn j => j))))))
				     
		val alpha = alphaProp
				f
				(L.mapPartial (fn (i,(ti,si)) => if Term.equal (ti,si)
								 then NONE
								 else SOME (i,si))
					      (LP.zip (L.tabulate (L, fn i=>i), LP.zip (ts,ss))))
				t
	    in
		Disj (alpha::lex)
	    end


    in
        case encodingType of 
            GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
          | GE => let val p = lposPropGe l r in Conj (p::(!iffProps)) end 
          | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)


(************************************************************************************)
(* afQlposEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(* permutation $B$"$j(B, quasi-order $BHG(B                                                   *)
(************************************************************************************)

fun afQlposEncoding precEncoding precEqEncoding lexStatus isCollapse pi 
		    symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	fun lookupGt s t = 
	    case TPM.find (!gtMap, (s,t)) of 
		SOME p => p
	      | NONE => let 
		  val p = qlposPropGt s t
	      in 
		  if p = False orelse p = True
		  then
		      (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		  else
		      let
			  val _ = symCount  := (!symCount) + 1;
			  val q = Prop.Atom (!symCount)
			  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  val _ =  debug (fn _ => print ( "encode (" 
							  ^ (Term.toString s) ^ " :gt: "
							  ^ (Term.toString t) ^ ") by "
							  ^ (Int.toString (!symCount))
							  ^ "\n" ))
		      in (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		      end
	      end

	and lookupEq s t = 
	    case TPM.find (!eqMap, (s,t)) of 
		SOME p => p
	      | NONE => let 
		  val p = qlposPropEq s t
	      in
		  if p = False orelse p = True
		  then
		      (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		  else
		      let 
			  val _ = symCount  := (!symCount) + 1;
			  val q = Prop.Atom (!symCount)
			  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  val _ =  debug (fn _ => print ( "encode (" 
							  ^ (Term.toString s) ^ " :eq: "
							  ^ (Term.toString t) ^ ") by "
							  ^ (Int.toString (!symCount))
							  ^ "\n" ))
		      in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		      end
	      end

	(*** >qlpos $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and qlposPropGt (Var _) _ = False
	  | qlposPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | qlposPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then 
		    IfThenElse (isCollapse f,
				Disj (L.tabulate (length ss,
						  fn i => Conj [pi (f,i+1), 
								lookupGt (L.nth (ss,i)) 
									 (L.nth (ts,i)) ])),
				Disj [lexPropGt f f s t ss ts,
 				      alphaProp f 0 ss t])
      		else IfThenElse 
			 (isCollapse f,
			  Disj (L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  lookupGt (L.nth (ss,i)) t])),
			  Disj [IfThenElse
				    (isCollapse g,
				     Disj (L.tabulate (length ts,
						       fn j => Conj [pi (g,j+1), 
								     lookupGt s (L.nth (ts,j))])),
				     Disj [Conj [precEncoding (f,g), 
						 majoProp g s (LP.zip 
								   (L.tabulate (length ts,fn x=>x),
								    ts))],
					   Conj [precEqEncoding (f,g), 
						 lexPropGt f g s t ss ts] ]),
 				alphaProp f 0 ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and qlposPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qlposPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | qlposPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s 
	  | qlposPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else if null ss andalso null ts
	    then precEqEncoding (f,g) (* added 2015/01/17 *)
	    else
		IfThenElse (isCollapse f,
 			    Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
 							   lookupEq (L.nth (ss,j)) t))),
			    (************* fix 2015/01/17 *
 			      Disj [Conj (isCollapse g::
					  L.tabulate (length ts,
						   fn j => Imp (pi (g,j+1),
								lookupEq s (L.nth (ts,j))))),
				    Conj [precEqEncoding (f,g),
					  lexPropEq f g s t ss ts]])
			     *******)
 			    IfThenElse (isCollapse g,
					Conj (L.tabulate (length ts, fn j => Imp (pi (g,j+1), lookupEq s (L.nth (ts,j))))),
					Conj [precEqEncoding (f,g),lexPropEq f g s t ss ts]))

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
 	and qlposPropGe s t = (* Disj [lookupEq s t, lookupGt s t] *)
	    let val p1 = lookupEq s t
	    in if p1 = True then True
	       else let val p2 = lookupGt s t
		    in if p2 = True then True
		       else if p1 = False then p2
		       else if p2 = False then p1
		       else Disj [p1, p2]
		    end
	    end

	(*** \exists si \in ss.  si >=qlpos t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f j ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,j+i+1), qlposPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >qlpos t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.map
		      (fn (i,ti) => Imp (pi (g,i+1), lookupGt s ti))
		      ts)

	(*** ss >qlpos^lex tt and \forall ti \in tt. s >qlpos ti  ***)
	(*** $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt f g s t ss ts = 
	    if Fun.equal (f,g)
	    then lexPropGt2 f s t ss ts
	    else 
		let 
 		    (* val _ = debug (fn _=>  *)
		    (* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "): " *)
		    (* 			   ^ "[" ^ (PrintUtil.prSeq Term.toString  ss) ^ "] >> ["  *)
		    (* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ "]\n")) *)
		    val lexGtMap = ref ILPM.empty
		    val lenss  = length ss
		    val ssidx = L.tabulate (lenss,fn x=>x)
		    val lents  = length ts
		    val tsidx = L.tabulate (lents,fn y=>y)
		    fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		    fun lookupLexGt (i::xs) (j::ys)  = 
			let val xs' = ListMergeSort.sort Int.> xs
			    val ys' = ListMergeSort.sort Int.> ys
			in
			    case ILPM.find (!lexGtMap, (i::xs',j::ys')) of 
				SOME p => p
			      | NONE => let 
 				  (* val _ = debug (fn _=>  *)
				  (* 		     print ("lookupLexGt ("  *)
				  (* 			    ^ (PrintUtil.prSeq Int.toString (i::xs'))  *)
				  (* 			    ^ "/"  *)
				  (* 			    ^ (PrintUtil.prSeq Int.toString (j::ys')) *)
				  (* 			    ^ ")\n")) *)
				  val p = lexGtSub (i::xs') (j::ys')
				  val _ = symCount  := (!symCount) + 1
				  val q = Prop.Atom (!symCount)
				  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			      in (lexGtMap := ILPM.insert (!lexGtMap, (i::xs',j::ys'), q); q)
			      end
			end
			    
		    and lexGtSub (i::xs) (j::ys) = (* i,j $B0z?t$rHf3S!%(Bxs,ys $B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
			let val si = L.nth (ss,i)
			    val tj = L.nth (ts,j)
			    val gt = lookupGt si tj
			    val eq = lookupEq si tj
			in
			    Disj [ (* if gt <> False orelse eq <> False
				 then *)
				Conj [pi (f,i+1), pi (g,j+1),  
				      Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
				      Conj (L.map (fn y=> lexStatus (g,j,y)) ys),
				      Disj [ (* if gt <> False then *)
					  Conj [gt,
						majoProp g s 
							 (L.map (fn y=> (y,L.nth (ts,y))) ys)]
					  (* else False *),
					  (* if eq <> False then *)
					  Conj [eq,
						Disj (ListXProd.mapX 
							  (fn (x,y) => 
							      lookupLexGt (x::(oth (x,xs)))
									  (y::(oth (y,ys))))
							  (xs,ys)) ]
				     (* else False *)]]
				(* else False *),
				Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
				       Disj (List.map 
						 (fn y => lookupLexGt (i::xs) (y::(oth (y,ys)))) 
						 ys)],
				Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
				       Disj (List.map 
						 (fn x => lookupLexGt (x::(oth (x,xs))) (j::ys)) 
						 xs)],
				Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)),
				       Disj (ListXProd.mapX 
						 (fn (x,y) => 
						     lookupLexGt (x::(oth (x,xs))) (y::(oth (y,ys))))
						 (xs,ys)) ] ]
			end
		in
		    Disj (ListXProd.mapX 
			      (fn (i,j) => lookupLexGt (i::(oth (i,ssidx))) (j::(oth (j,tsidx))))
			      (ssidx,tsidx))
		end

	and lexPropGt2 f s t ss ts = 
	    let 
 		(* val _ = debug (fn _=>  *)
		(* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString f) ^ "): " *)
		(* 			   ^ "[" ^ (PrintUtil.prSeq Term.toString  ss) ^ "] >> ["  *)
		(* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ "]\n")) *)
		val lexGtMap = ref ILM.empty
		val len  = length ss

	        (* si = ti $B$J$i!$$=$N0z?t$O9M$($J$/$F$h$$(B *)
		val idx = L.filter (fn i=> not (Term.equal (L.nth(ss,i), L.nth(ts,i))))
				   (L.tabulate (len,fn x=>x))

		fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		fun lookupLexGt (i::xs)  = 
		    let val xs' = ListMergeSort.sort Int.> xs
		    in
			case ILM.find (!lexGtMap, i::xs') of 
			    SOME p => p
			  | NONE => let 
 			      (* val _ = debug (fn _=>  *)
			      (* 		     print ("lookupLexGt ("  *)
			      (* 			    ^ (PrintUtil.prSeq Int.toString (i::xs'))  *)
			      (* 			    ^ "/"  *)
			      (* 			    ^ (PrintUtil.prSeq Int.toString (i::xs')) *)
			      (* 			    ^ ")\n")) *)
			      val p = lexGtSub (i::xs') 
			      val _ = symCount  := (!symCount) + 1
			      val q = Prop.Atom (!symCount)
			      val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  in (lexGtMap := ILM.insert (!lexGtMap, i::xs', q); q)
			  end
		    end
			
		and lexGtSub (i::xs) = (* i$B0z?t$rHf3S!%(Bxs $B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
		    let val si = L.nth (ss,i)
			val ti = L.nth (ts,i)
			val gt = lookupGt si ti
			val eq = lookupEq si ti
		    in
			Disj [ if gt <> False orelse eq <> False
			       then
				   Conj [pi (f,i+1), 
					 Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					 Disj [ if gt <> False then
						    Conj [gt,
							  majoProp f s 
								   (L.map (fn x=> (x,L.nth (ts,x))) xs)]
						else False,
						if eq <> False then
						    Conj [eq,
							  Disj (List.map
								    (fn x => 
									lookupLexGt (x::(oth (x,xs))))
								    xs) ]
						else False ]]
			       else False,
			       Conj  [Neg (pi (f,i+1)), 
				      Disj (List.map
						(fn x => lookupLexGt (x::(oth (x,xs))))
						xs) ] ]
		    end
	    in
		Disj (List.map
			  (fn i => lookupLexGt (i::(oth (i,idx))))
			  idx)
	    end

	(*** ss ==^lex tt $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropEq f g s t ss ts = 
	    if Fun.equal (f,g)
	    then lexPropEq2 f s t ss ts
	    else 
		let 
 		    (* val _ = debug (fn _=>  *)
		    (* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "): " *)
		    (* 			   ^ "[" ^ (PrintUtil.prSeq Term.toString  ss) ^ "] >> ["  *)
		    (* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ "]\n")) *)
		    val lexEqMap = ref ILPM.empty
		    val lenss  = length ss
		    val ssidx = L.tabulate (lenss,fn x=>x)
		    val lents  = length ts
		    val tsidx = L.tabulate (lents,fn y=>y)
		    fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		    fun lookupLexEq (i::xs) (j::ys)  = 
			let val xs' = ListMergeSort.sort Int.> xs
			    val ys' = ListMergeSort.sort Int.> ys
			in
			    case ILPM.find (!lexEqMap, (i::xs',j::ys')) of 
				SOME p => p
			      | NONE => let 
				  val p = lexEqSub (i::xs') (j::ys')
				  val _ = symCount  := (!symCount) + 1
				  val q = Prop.Atom (!symCount)
				  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			      in (lexEqMap := ILPM.insert (!lexEqMap, (i::xs',j::ys'), q); q)
			      end
			end

		    and lexEqSub (i::xs) (j::ys) = (* i,j $B0z?t$rHf3S!%(Bxs,ys $B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
			let val si = L.nth (ss,i)
			    val tj = L.nth (ts,j)
			    val eq = lookupEq si tj
			in
			    Disj [ if eq <> False
				   then
				       Conj [pi (f,i+1), pi (g,j+1),  
					     Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					     Conj (L.map (fn y=> lexStatus (g,j,y)) ys),
					     Conj [eq,
						   if (null xs) andalso (null ys)
						   then True
						   else
						       Disj (ListXProd.mapX 
								 (fn (x,y) => 
								     lookupLexEq (x::(oth (x,xs)))
										 (y::(oth (y,ys))))
								 (xs,ys)) ]]
				   else False,
				   Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
					  Disj (List.map 
						    (fn y => lookupLexEq (i::xs) (y::(oth (y,ys)))) 
						    ys)],
				   Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
					  Disj (List.map 
						    (fn x => lookupLexEq (x::(oth (x,xs))) (j::ys)) 
						    xs)],
				   Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)),
					  if (null xs) andalso (null ys)
					  then True
					  else
					      Disj (ListXProd.mapX 
							(fn (x,y) => 
							    lookupLexEq (x::(oth (x,xs))) (y::(oth (y,ys))))
							(xs,ys)) ] ]
			end
		in
		    Disj (ListXProd.mapX 
			      (fn (i,j) => lookupLexEq (i::(oth (i,ssidx))) (j::(oth (j,tsidx))))
			      (ssidx,tsidx))
		end

	and lexPropEq2 f s t ss ts = 
	    let 
 		(* val _ = debug (fn _=>  *)
		(* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString f) ^ "): " *)
		(* 			   ^ "[" ^ (PrintUtil.prSeq Term.toString  ss) ^ "] >> ["  *)
		(* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ "]\n")) *)
		val lexEqMap = ref ILM.empty
		val len  = length ss

	        (* si = ti $B$J$i!$$=$N0z?t$O9M$($J$/$F$h$$(B *)
		val idx = L.filter (fn i=> not (Term.equal (L.nth(ss,i),L.nth(ts,i))))
				   (L.tabulate (len,fn x=>x))

		fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		fun lookupLexEq (i::xs) = 
		    let val xs' = ListMergeSort.sort Int.> xs
		    in
			case ILM.find (!lexEqMap, i::xs') of 
			    SOME p => p
			  | NONE => let 
			      val p = lexEqSub (i::xs') 
			      val _ = symCount  := (!symCount) + 1
			      val q = Prop.Atom (!symCount)
			      val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  in (lexEqMap := ILM.insert (!lexEqMap, i::xs', q); q)
			  end
		    end

		and lexEqSub (i::xs) = (* i$B0z?t$rHf3S!%(Bxs$B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
		    let val eq = lookupEq (L.nth (ss,i)) (L.nth (ts,i))
		    in
			Disj [ if eq <> False
			       then
				   Conj [pi (f,i+1), 
					 Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					 Conj [eq,
					       if (null xs) 
					       then True
					       else
						   Disj (List.map
							     (fn x => 
								 lookupLexEq (x::(oth (x,xs))))
							     xs) ]]
			       else False,
			       Conj  [Neg (pi (f,i+1)),
				      if (null xs) 
				      then True
				      else
					  Disj (List.map
						    (fn x => 
							lookupLexEq (x::(oth (x,xs))))
						    xs) ] ]
		    end
	    in
		Disj (List.map
			  (fn i => lookupLexEq (i::(oth (i,idx))))
			  idx)
	    end

    in
	case encodingType of 
	    GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
	  | GE => let val p = qlposPropGe l r in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)




(************************************************************************************)
(* afMpoEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(*                                                                                  *)
(************************************************************************************)

fun afMpoEncoding precEncoding isCollapse pi 
		  symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	(* $B$"$H$G!$(B True $B$d(B False $B$N>l9g$O$=$N$^$^EPO?$9$k$h$&$K$9$k$3$H(B *)
	fun lookupGt s t = (* mpoPropGt s t *)
	    case TPM.find (!gtMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = mpoPropGt s t
	      in if p = True orelse p = False
		 then
		     (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ =>
					    print ( "encode ("
						    ^ (Term.toString s) ^ " :gt: "
						    ^ (Term.toString t) ^ ") by "
						    ^ (Int.toString (!symCount))
						    ^ "\n" ))
		     in
			 (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		     end
	      end

	and lookupEq s t =
	    case TPM.find (!eqMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = mpoPropEq s t
	      in if p = True orelse p = False
		 then
		     (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ => print ( "encode ("
							 ^ (Term.toString s) ^ " :eq: "
							 ^ (Term.toString t) ^ ") by "
							 ^ (Int.toString (!symCount))
							 ^ "\n" ))
		     in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		     end
	      end


	(*** >mpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and mpoPropGt (Var _) _ = False
	  | mpoPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | mpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then IfThenElse (isCollapse f,
				 Disj (L.tabulate (length ss,
						   fn i => Conj [pi (f,i+1), 
								 lookupGt (L.nth (ss,i)) (L.nth (ts,i))])),
				 Disj [ mulPropGt f g ss ts,
					alphaProp f ss t ])

		else IfThenElse 
			 (isCollapse f,
			  Disj (L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  lookupGt (L.nth (ss,i)) t])),
			  Disj [IfThenElse
				    (isCollapse g,
				     Disj (L.tabulate (length ts,
						       fn j => Conj [pi (g,j+1), 
								     lookupGt s (L.nth (ts,j))])),
				     Conj [precEncoding (f,g), majoProp g s ts]),
 				alphaProp f ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and mpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | mpoPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | mpoPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s
	  | mpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else
		IfThenElse (isCollapse f,
 			    Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
 							   lookupEq (L.nth (ss,j)) t))),
 			    Conj [isCollapse g,
				  Disj (L.tabulate (length ts,
						    fn j => Conj [pi (g,j+1),
								  lookupEq s (L.nth (ts,j))]))])
	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
 	and mpoPropGe s t = Disj [lookupEq s t,lookupGt s t]

	(*** \exists si \in ss.  si >=mpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,i+1), mpoPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >mpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), lookupGt s (L.nth (ts,i)))))
		 
	(*** ss >mpo^mul tt $B$H$J$k$?$a$N@)Ls(B  ***)
	and mulPropGt f g [] [] = False
	  | mulPropGt f g ss ts =
	    let val M = length ss 
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M )) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
	        (*  si $B$H(B tj $B$,BP1~!$(B  si >mpo tj $B$J$i(B tj $B$O$$$/$D$"$C$F$b$h$$(B *)
		fun pvar (i,j) = count + i * N + j  (* 0 <= i < M, 0 <= j < N *)
		val count2 = count + M*N
	        (*  si $B$KBP1~$9$k(B tj $B$K$D$$$F!$(B si =mpo tj *)
		fun pvar2 i = count2 + i
	    in
		Conj (
		    (Disj (L.tabulate (M,
				       fn i => Conj [ pi (f,i+1), 
						      Neg (Atom (pvar2 i)) ])))
		    ::(L.@ 
		       (L.tabulate (M,
				    fn i => 
				       Imp (pi (f,i+1),
					    Conj (L.tabulate (N,
							      fn j =>
								 Imp (Conj [ pi (g,j+1), Atom (pvar (i,j)) ],
								      IfThenElse (Atom (pvar2 i),
										  (lookupEq
										       (L.nth (ss,i))
										       (L.nth (ts,j))),
										  (lookupGt
										       (L.nth (ss,i))
										       (L.nth (ts,j))))))))),
			(* surjective *)
			L.@ (L.tabulate (N,
					 fn j => Imp (pi (g,j+1),
						      Disj (L.tabulate (M,
									fn i => Conj [pi (f,i+1),
										      Atom (pvar (i,j))])))),
			     (* injective for equals *)
			     L.tabulate (M, 
					 fn i => Imp (Conj [pi (f,i+1), Atom (pvar2 i)],
						      one (L.tabulate (N,
								       fn j => Conj [pi (g,j+1),
										     Atom (pvar (i,j))]))))))))
	    end

    in
	case encodingType of 
	    GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
	  | GE => let val p = mpoPropGe l r in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)

(************************************************************************************)
(* afQmpoEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(*                                                                                  *)
(************************************************************************************)

fun afQmpoEncoding precEncoding precEqEncoding isCollapse pi 
		   symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	(* $B$"$H$G!$(B True $B$d(B False $B$N>l9g$O$=$N$^$^EPO?$9$k$h$&$K$9$k$3$H(B *)
	fun lookupGt s t = (* qmpoPropGt s t *)
	    case TPM.find (!gtMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = qmpoPropGt s t
	      in if p = True orelse p = False
		 then
		     (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ =>
					    print ( "encode ("
						    ^ (Term.toString s) ^ " :gt: "
						    ^ (Term.toString t) ^ ") by "
						    ^ (Int.toString (!symCount))
						    ^ "\n" ))
		     in
			 (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		     end
	      end

	and lookupEq s t =
	    case TPM.find (!eqMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = qmpoPropEq s t
	      in if p = True orelse p = False
		 then
		     (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ => print ( "encode ("
							 ^ (Term.toString s) ^ " :eq: "
							 ^ (Term.toString t) ^ ") by "
							 ^ (Int.toString (!symCount))
							 ^ "\n" ))
		     in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		     end
	      end


	(*** >qmpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and qmpoPropGt (Var _) _ = False
	  | qmpoPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | qmpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then IfThenElse (isCollapse f,
				 Disj (L.tabulate (length ss,
						   fn i => Conj [pi (f,i+1), 
								 lookupGt (L.nth (ss,i)) (L.nth (ts,i))])),
				 Disj [ mulPropGt f g ss ts,
					alphaProp f ss t ])
                else IfThenElse 
                         (isCollapse f,
                          Disj (L.tabulate (length ss,
                                            fn i => Conj [pi (f,i+1), 
                                                          lookupGt (L.nth (ss,i)) t])),
                          Disj [IfThenElse
                                    (isCollapse g,
                                     Disj (L.tabulate (length ts,
                                                       fn j => Conj [pi (g,j+1), 
                                                                     lookupGt s (L.nth (ts,j))])),
                                     Disj [Conj [precEncoding (f,g), majoProp g s ts],
                                           Conj [precEqEncoding (f,g), 
                                                 mulPropGt f g ss ts] ]),
                                alphaProp f ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and qmpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qmpoPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | qmpoPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s
	  | qmpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else if null ss andalso null ts
	    then precEqEncoding (f,g) (* added 2015/01/17 *)
	    else
                IfThenElse (isCollapse f,
                            Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
							   lookupEq (L.nth (ss,j)) t))),
			    (************* fix 2015/01/17 *
                            Disj [Conj (isCollapse g::
                                        L.tabulate (length ts,
                                                   fn j => Imp (pi (g,j+1),
                                                                lookupEq s (L.nth (ts,j))))),
                                    Conj [precEqEncoding (f,g),
                                          mulPropEq f g ss ts]])
			     ******)
                            IfThenElse (isCollapse g,
                                        Conj (L.tabulate (length ts,fn j => Imp (pi (g,j+1), lookupEq s (L.nth (ts,j))))),
					Conj [precEqEncoding (f,g),mulPropEq f g ss ts]))


	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
 	and qmpoPropGe s t = Disj [lookupEq s t,lookupGt s t]

	(*** \exists si \in ss.  si >=qmpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,i+1), qmpoPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >qmpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), lookupGt s (L.nth (ts,i)))))
		 
	(*** ss >qmpo^mul tt $B$H$J$k$?$a$N@)Ls(B  ***)
	and mulPropGt f g [] [] = False
	  | mulPropGt f g ss ts =
	    let val M = length ss 
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M )) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
	        (*  si $B$H(B tj $B$,BP1~!$(B  si >qmpo tj $B$J$i(B tj $B$O$$$/$D$"$C$F$b$h$$(B *)
		fun pvar (i,j) = count + i * N + j  (* 0 <= i < M, 0 <= j < N *)
		val count2 = count + M*N
	        (*  si $B$KBP1~$9$k(B tj $B$K$D$$$F!$(B si =qmpo tj *)
		fun pvar2 i = count2 + i
	    in
		Conj (
		    (Disj (L.tabulate (M,
				       fn i => Conj [ pi (f,i+1), 
						      Neg (Atom (pvar2 i)) ])))
		    ::(L.@ 
		       (L.tabulate (M,
				    fn i => 
				       Imp (pi (f,i+1),
					    Conj (L.tabulate (N,
							      fn j =>
								 Imp (Conj [ pi (g,j+1), Atom (pvar (i,j)) ],
								      IfThenElse (Atom (pvar2 i),
										  (lookupEq
										       (L.nth (ss,i))
										       (L.nth (ts,j))),
										  (lookupGt
										       (L.nth (ss,i))
										       (L.nth (ts,j))))))))),
			(* surjective *)
			L.@ (L.tabulate (N,
					 fn j => Imp (pi (g,j+1),
						      Disj (L.tabulate (M,
									fn i => Conj [pi (f,i+1),
										      Atom (pvar (i,j))])))),
			     (* injective for equals *)
			     L.tabulate (M, 
					 fn i => Imp (Conj [pi (f,i+1), Atom (pvar2 i)],
						      one (L.tabulate (N,
								       fn j => Conj [pi (g,j+1),
										     Atom (pvar (i,j))]))))))))
	    end

	(*** ss =qmpo^mul tt $B$H$J$k$?$a$N@)Ls(B  ***)
	and mulPropEq f g [] [] = False
	  | mulPropEq f g ss ts =
	    let val M = length ss 
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString (M*N)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N)
		(* count, ..., count + M*N - 1 are available *)
	        (*  si $B$H(B tj $B$,BP1~!$(B  si =qmpo tj *)
		fun pvar (i,j) = count + i * N + j  (* 0 <= i < M, 0 <= j < N *)
	    in
		Conj (L.@ (
			 L.tabulate (M,
				     fn i => 
					Imp (pi (f,i+1),
					     Conj (L.tabulate (N,
							       fn j =>
								  Imp (Conj [ pi (g,j+1), Atom (pvar (i,j)) ],
								       (lookupEq (L.nth (ss,i)) 
										 (L.nth (ts,j)))))))),
			 
			 (* surjective *)
			 L.@ (L.tabulate (N,
					  fn j => Imp (pi (g,j+1),
						       Disj (L.tabulate (M,
									 fn i => Conj [pi (f,i+1),
										       Atom (pvar (i,j))])))),
			      (* injective for equals *)
			      L.tabulate (M, 
					  fn i => Imp (pi (f,i+1),
						       one (L.tabulate (N,
									fn j => Conj [pi (g,j+1),
										      Atom (pvar (i,j))])))))))
	    end


    in
	case encodingType of 
	    GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
	  | GE => let val p = qmpoPropGe l r in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)





(************************************************************************************)
(* afRpoEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(*                                                                                  *)
(************************************************************************************)

fun afRpoEncoding precEncoding isMul lexStatus isCollapse pi 
		  symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	(* $B$"$H$G!$(B True $B$d(B False $B$N>l9g$O$=$N$^$^EPO?$9$k$h$&$K$9$k$3$H(B *)
	fun lookupGt s t = (* rpoPropGt s t *)
	    case TPM.find (!gtMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = rpoPropGt s t
	      in if p = True orelse p = False
		 then
		     (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ =>
					    print ( "encode ("
						    ^ (Term.toString s) ^ " :gt: "
						    ^ (Term.toString t) ^ ") by "
						    ^ (Int.toString (!symCount))
						    ^ "\n" ))
		     in
			 (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		     end
	      end

	and lookupEq s t =
	    case TPM.find (!eqMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = rpoPropEq s t
	      in if p = True orelse p = False
		 then
		     (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ => print ( "encode ("
							 ^ (Term.toString s) ^ " :eq: "
							 ^ (Term.toString t) ^ ") by "
							 ^ (Int.toString (!symCount))
							 ^ "\n" ))
		     in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		     end
	      end


	(*** >rpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and rpoPropGt (Var _) _ = False
	  | rpoPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | rpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then IfThenElse (isCollapse f,
				 Disj (L.tabulate (length ss,
						   fn i => Conj [pi (f,i+1), 
								 lookupGt (L.nth (ss,i)) (L.nth (ts,i))])),
				 Disj [ IfThenElse (isMul f,
						    mulPropGt f g ss ts,
						    lexPropGt f s t ss ts),
					alphaProp f ss t ])
		else IfThenElse 
			 (isCollapse f,
			  Disj (L.tabulate (length ss,
					    fn i => Conj [pi (f,i+1), 
							  lookupGt (L.nth (ss,i)) t])),
			  Disj [IfThenElse
				    (isCollapse g,
				     Disj (L.tabulate (length ts,
						       fn j => Conj [pi (g,j+1), 
								     lookupGt s (L.nth (ts,j))])),
				     Conj [precEncoding (f,g), majoProp g s ts]),
 				alphaProp f ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and rpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | rpoPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | rpoPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s
	  | rpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else
		IfThenElse (isCollapse f,
 			    Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
 							   lookupEq (L.nth (ss,j)) t))),
 			    Conj [isCollapse g,
				  Disj (L.tabulate (length ts,
						    fn j => Conj [pi (g,j+1),
								  lookupEq s (L.nth (ts,j))]))])

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
 	and rpoPropGe s t = Disj [lookupEq s t,lookupGt s t]

	(*** \exists si \in ss.  si >=rpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,i+1), rpoPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >rpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.tabulate (length ts, 
			      fn i => Imp (pi (g,i+1), lookupGt s (L.nth (ts,i)))))
		 
	(*** ss >rpo^mul tt $B$H$J$k$?$a$N@)Ls(B  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
	and mulPropGt f g [] [] = False
	  | mulPropGt f g ss ts =
	    let val M = length ss 
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]>>"))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
	        (*  si $B$H(B tj $B$,BP1~!$(B  si >rpo tj $B$J$i(B tj $B$O$$$/$D$"$C$F$b$h$$(B *)
		fun pvar (i,j) = count + i * N + j  (* 0 <= i < M, 0 <= j < N *)
		val count2 = count + M*N
	        (*  si $B$KBP1~$9$k(B tj $B$K$D$$$F!$(B si =rpo tj *)
		fun pvar2 i = count2 + i
	    in
		Conj (
		    (Disj (L.tabulate (M,
				       fn i => Conj [ pi (f,i+1), 
						      Neg (Atom (pvar2 i)) ])))
		    ::(L.@ 
		       (L.tabulate (M,
				    fn i => 
				       Imp (pi (f,i+1),
					    Conj (L.tabulate (N,
							      fn j =>
								 Imp (Conj [ pi (g,j+1), Atom (pvar (i,j)) ],
								      IfThenElse (Atom (pvar2 i),
										  (lookupEq
										       (L.nth (ss,i))
										       (L.nth (ts,j))),
										  (lookupGt
										       (L.nth (ss,i))
										       (L.nth (ts,j))))))))),
			(* surjective *)
			L.@ (L.tabulate (N,
					 fn j => Imp (pi (g,j+1),
						      Disj (L.tabulate (M,
									fn i => Conj [pi (f,i+1),
										      Atom (pvar (i,j))])))),
			     (* injective for equals *)
			     L.tabulate (M, 
					 fn i => Imp (Conj [pi (f,i+1), Atom (pvar2 i)],
						      one (L.tabulate (N,
								       fn j => Conj [pi (g,j+1),
										     Atom (pvar (i,j))]))))))))
	    end

	(*** ss >rpo^lex tt and \forall ti \in tt. s >lpos ti  ***)
	(***      or  \exists si. si >= t  $B$H$J$k$?$a$N@)Ls(B ***)
	(*  f $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt f s t ss ts = 
	    let val L = length ss (* we assume length ss = length ts *)
		(* val _ = print (">>=lex " ^ (prTerms ss) ^ "," ^ (prTerms ts) ^ "\n") *)
 		val gps = LP.map (fn (x,y) => lookupGt x y) (ss,ts)
	    in Disj 
		   (L.tabulate (L, 
				fn i => 
				   Conj 
				       ((List.nth (gps,i))::
					(pi (f,i+1))::
					(L.map 
					     (fn j => 
						 IfThenElse (lexStatus (f,j,i),
							     Imp (pi (f,j+1),
								  lookupEq (L.nth (ts,j))
									   (L.nth (ss,j))),
							     Conj [lexStatus (f,i,j),
								   Imp (pi (f,j+1),
									lookupGt s (L.nth (ts,j)))]))
					     (L.filter (fn j => i <> j) 
						       (L.tabulate (L, fn j => j)))))))
	    end
		
    in
	case encodingType of 
	    GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
	  | GE => let val p = rpoPropGe l r in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)


(************************************************************************************)
(* afQrpoEncoding:                                                                   *)
(*         forall l -> r \in R. pi(l) >lpo pi(r) $B$H(B $BF1Ey$J(B satisfiablity $B$NO@M}<0$KJQ49(B  *)
(*                                                                                  *)
(************************************************************************************)

fun afQrpoEncoding precEncoding precEqEncoding isMul lexStatus
                   isCollapse pi symCount encodingType (l,r) =
    let 
	val gtMap= ref TermPairMap.empty
	val eqMap= ref TermPairMap.empty
	val iffProps = ref []

	(* $B$"$H$G!$(B True $B$d(B False $B$N>l9g$O$=$N$^$^EPO?$9$k$h$&$K$9$k$3$H(B *)
	fun lookupGt s t = (* qrpoPropGt s t *)
	    case TPM.find (!gtMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = qrpoPropGt s t
	      in if p = True orelse p = False
		 then
		     (gtMap := TPM.insert (!gtMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ =>
					    print ( "encode ("
						    ^ (Term.toString s) ^ " :gt: "
						    ^ (Term.toString t) ^ ") by "
						    ^ (Int.toString (!symCount))
						    ^ "\n" ))
		     in
			 (gtMap := TPM.insert (!gtMap, (s,t), q); q)
		     end
	      end

	and lookupEq s t =
	    case TPM.find (!eqMap, (s,t)) of
		SOME p => p
	      | NONE => let 
		  val p = qrpoPropEq s t
	      in if p = True orelse p = False
		 then
		     (eqMap := TPM.insert (!eqMap, (s,t), p); p)
		 else
		     let val _ = symCount  := (!symCount) + 1;
			 val q = Prop.Atom (!symCount)
			 val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			 val _ =  debug (fn _ => print ( "encode ("
							 ^ (Term.toString s) ^ " :eq: "
							 ^ (Term.toString t) ^ ") by "
							 ^ (Int.toString (!symCount))
							 ^ "\n" ))
		     in (eqMap := TPM.insert (!eqMap, (s,t), q); q)
		     end
	      end


	(*** >qrpo $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and qrpoPropGt (Var _) _ = False
	  | qrpoPropGt (s as Fun (f,ss,_)) (t as Var (y,_)) = 
	    if (containVar ss y)
	    then
		let val ps1 = L.tabulate (length ss,fn i => lookupGt (L.nth (ss,i)) t)
		    val ps2 = L.tabulate (length ss,fn i => lookupEq (L.nth (ss,i)) t)
		    val ps = LP.map (fn (p,q) => Disj [p,q]) (ps1,ps2)
		    val ns = L.tabulate (length ss,fn x=>x)
		in
		    IfThenElse (isCollapse f,
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps1)),
				Disj (LP.map (fn (i,p) => Conj [pi (f,i+1),p]) (ns,ps)))
		end
	    else False
	  | qrpoPropGt (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if equal (s,t)
	    then False
	    else
		if Fun.equal (f,g)
		then IfThenElse (isCollapse f,
				 Disj (L.tabulate (length ss,
						   fn i => Conj [pi (f,i+1), 
								 lookupGt (L.nth (ss,i)) (L.nth (ts,i))])),
				 Disj [ IfThenElse (isMul f,
						    mulPropGt f g ss ts,
						    lexPropGt f g s t ss ts),
					alphaProp f ss t ])
                else IfThenElse 
                         (isCollapse f,
                          Disj (L.tabulate (length ss,
                                            fn i => Conj [pi (f,i+1), 
                                                          lookupGt (L.nth (ss,i)) t])),
                          Disj [IfThenElse
                                    (isCollapse g,
                                     Disj (L.tabulate (length ts,
                                                       fn j => Conj [pi (g,j+1), 
                                                                     lookupGt s (L.nth (ts,j))])),
                                     Disj [Conj [precEncoding (f,g), 
						 majoProp g s 
							  (LP.zip (L.tabulate (length ts,
									       fn x=>x),
								   ts))],
                                           Conj [precEqEncoding (f,g), 
						 IfThenElse (isMul f,
							     mulPropGt f g ss ts,
							     lexPropGt f g s t ss ts) ] ]),
                                alphaProp f ss t])

	(*** s = t $B$K$J$k@)Ls$rJV$9(B ***)
	and qrpoPropEq (Var (x,_)) (Var (y,_)) =
	    if Var.equal (x,y) then True else False
	  | qrpoPropEq (s as (Var (x,_))) (t as (Fun (g,ts,_))) =
	    if containVar ts x
	    then
		Conj [isCollapse g,
		      Disj (L.tabulate (length ts,
					fn j => Conj [pi (g,j+1),
						      lookupEq s (L.nth (ts,j))]))]
	    else False
	  | qrpoPropEq (s as Fun (f,ss,_)) (t as Var (y,_)) = lookupEq t s
	  | qrpoPropEq (s as Fun (f,ss,_)) (t as Fun (g,ts,_)) =
	    if Term.equal (s,t)
	    then True
	    else if Fun.equal (f,g)
	    then
		Conj (L.tabulate (length ss,
				  fn j => Imp (pi (f,j+1),
					       lookupEq (L.nth (ss,j)) (L.nth (ts,j)))))
	    else
                IfThenElse (isCollapse f,
                            Conj (L.tabulate (length ss,
					      fn j => Imp (pi (f,j+1),
							   lookupEq (L.nth (ss,j)) t))),
                            Disj [Conj (isCollapse g::
                                        L.tabulate (length ts,
                                                    fn j => Imp (pi (g,j+1),
                                                                 lookupEq s (L.nth (ts,j))))),
                                  Conj [precEqEncoding (f,g),
                                        IfThenElse (isMul f,
						    mulPropEq f g ss ts,
						    lexPropEq f g s t ss ts) ]])

	(*** s >= t $B$K$J$k@)Ls$rJV$9(B ***)
	(* 	  and qrpoPropGe s t = Disj [lookupEq s t,lookupGt s t] *)
 	and qrpoPropGe s t = 
	    let val p1 = lookupEq s t
	    in if p1 = True then True
	       else let val p2 = lookupGt s t
		    in if p2 = True then True
		       else if p1 = False then p2
		       else if p2 = False then p1
		       else Disj [p1, p2]
		    end
	    end

	(*** \exists si \in ss.  si >=qrpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	(*** ss $B$N@hF,$O(B j-th (>=0) argument of f *)
	and alphaProp f ss t = (* f is assumed not collapsing *)
	    Disj (L.tabulate (length ss,
			      fn i => Conj [pi (f,i+1), qrpoPropGe (L.nth (ss,i)) t]))

	(*** \forall t \in ts. s >qrpo t $B$H$J$k$?$a$N@)Ls$rJV$9(B ***)
	and majoProp g s ts =
	    Conj (L.map
		      (fn (i,ti) => Imp (pi (g,i+1), lookupGt s ti))
		      ts)

	(*** ss >qrpo^mul tt $B$H$J$k$?$a$N@)Ls(B  ***)
	and mulPropGt f g [] [] = False
	  | mulPropGt f g ss ts =
	    let val M = length ss 
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]>>"))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N) + M 
		(* count, ..., count + M*N + M - 1 are available *)
	        (*  si $B$H(B tj $B$,BP1~!$(B  si >qrpo tj $B$J$i(B tj $B$O$$$/$D$"$C$F$b$h$$(B *)
		fun pvar (i,j) = count + i * N + j  (* 0 <= i < M, 0 <= j < N *)
		val count2 = count + M*N
	        (*  si $B$KBP1~$9$k(B tj $B$K$D$$$F!$(B si =qrpo tj *)
		fun pvar2 i = count2 + i
	    in
		Conj (
		    (Disj (L.tabulate (M,
				       fn i => Conj [ pi (f,i+1), 
						      Neg (Atom (pvar2 i)) ])))
		    ::(L.@ 
		       (L.tabulate (M,
				    fn i => 
				       Imp (pi (f,i+1),
					    Conj (L.tabulate (N,
							      fn j =>
								 Imp (Conj [ pi (g,j+1), Atom (pvar (i,j)) ],
								      IfThenElse (Atom (pvar2 i),
										  (lookupEq
										       (L.nth (ss,i))
										       (L.nth (ts,j))),
										  (lookupGt
										       (L.nth (ss,i))
										       (L.nth (ts,j))))))))),
			(* surjective *)
			L.@ (L.tabulate (N,
					 fn j => Imp (pi (g,j+1),
						      Disj (L.tabulate (M,
									fn i => Conj [pi (f,i+1),
										      Atom (pvar (i,j))])))),
			     (* injective for equals *)
			     L.tabulate (M, 
					 fn i => Imp (Conj [pi (f,i+1), Atom (pvar2 i)],
						      one (L.tabulate (N,
								       fn j => Conj [pi (g,j+1),
										     Atom (pvar (i,j))]))))))))
	    end

	(*** ss =qrpo^mul tt $B$H$J$k$?$a$N@)Ls(B  ***)
	and mulPropEq f g [] [] = False
	  | mulPropEq f g ss ts =
	    let val M = length ss 
		val N = length ts
		val count = !symCount+1
		val _ = debug (fn _ => print ("encoding [" ^ (PrintUtil.prSeq Term.toString ss) ^ "]=="))
		val _ = debug (fn _ => print ("[" ^ (PrintUtil.prSeq Term.toString ts) ^ "] with "))
		val _ = debug (fn _ => print ((Int.toString ((M*N) + M)) ^ " variables.\n"))
		val _ = symCount := (!symCount) + (M * N)
		(* count, ..., count + M*N - 1 are available *)
	        (*  si $B$H(B tj $B$,BP1~!$(B  si =qrpo tj *)
		fun pvar (i,j) = count + i * N + j  (* 0 <= i < M, 0 <= j < N *)
	    in
		Conj (L.@ (
			 L.tabulate (M,
				     fn i => 
					Imp (pi (f,i+1),
					     Conj (L.tabulate (N,
							       fn j =>
								  Imp (Conj [ pi (g,j+1), Atom (pvar (i,j)) ],
								       (lookupEq (L.nth (ss,i)) 
										 (L.nth (ts,j)))))))),
			 
			 (* surjective *)
			 L.@ (L.tabulate (N,
					  fn j => Imp (pi (g,j+1),
						       Disj (L.tabulate (M,
									 fn i => Conj [pi (f,i+1),
										       Atom (pvar (i,j))])))),
			      (* injective for equals *)
			      L.tabulate (M, 
					  fn i => Imp (pi (f,i+1),
						       one (L.tabulate (N,
									fn j => Conj [pi (g,j+1),
										      Atom (pvar (i,j))])))))))
	    end


	(*** ss >qlpos^lex tt and \forall ti \in tt. s >qlpos ti  ***)
	(*** $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropGt f g s t ss ts = 
	    if Fun.equal (f,g)
	    then lexPropGt2 f s t ss ts
	    else
		let 
 		    (* val _ = debug (fn _=>  *)
		    (* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "): " *)
		    (* 			   ^ "<" ^ (PrintUtil.prSeq Term.toString  ss) ^ "> >> <"  *)
		    (* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ ">\n")) *)
		    val lexGtMap = ref ILPM.empty
		    val lenss  = length ss
		    val ssidx = L.tabulate (lenss,fn x=>x)
		    val lents  = length ts
		    val tsidx = L.tabulate (lents,fn y=>y)
		    fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		    fun lookupLexGt (i::xs) (j::ys)  = 
			let val xs' = ListMergeSort.sort Int.> xs
			    val ys' = ListMergeSort.sort Int.> ys
			in
			    case ILPM.find (!lexGtMap, (i::xs',j::ys')) of 
				SOME p => p
			      | NONE => let 
 				  (* val _ = debug (fn _=>  *)
				  (* 		     print ("lookupLexGt ("  *)
				  (* 			    ^ (PrintUtil.prSeq Int.toString (i::xs'))  *)
				  (* 			    ^ "/"  *)
				  (* 			    ^ (PrintUtil.prSeq Int.toString (j::ys')) *)
				  (* 			    ^ ")\n")) *)
				  val p = lexGtSub (i::xs') (j::ys')
				  val _ = symCount  := (!symCount) + 1
				  val q = Prop.Atom (!symCount)
				  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			      in (lexGtMap := ILPM.insert (!lexGtMap, (i::xs',j::ys'), q); q)
			      end
			end
			    
		    and lexGtSub (i::xs) (j::ys) = (* i,j $B0z?t$rHf3S!%(Bxs,ys $B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
			let val si = L.nth (ss,i)
			    val tj = L.nth (ts,j)
			    val gt = lookupGt si tj
			    val eq = lookupEq si tj
			in
			    Disj [ if gt <> False orelse eq <> False
				   then
				       Conj [pi (f,i+1), pi (g,j+1),  
					     Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					     Conj (L.map (fn y=> lexStatus (g,j,y)) ys),
					     Disj [ if gt <> False then
							Conj [gt,
							      majoProp g s 
								       (L.map (fn y=> (y,L.nth (ts,y))) ys)]
						    else False,
						    if eq <> False then
							Conj [eq,
							      Disj (ListXProd.mapX 
									(fn (x,y) => 
									    lookupLexGt (x::(oth (x,xs)))
											(y::(oth (y,ys))))
									(xs,ys)) ]
						    else False ]]
				   else False,
				   Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
					  Disj (List.map 
						    (fn y => lookupLexGt (i::xs) (y::(oth (y,ys)))) 
						    ys)],
				   Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
					  Disj (List.map 
						    (fn x => lookupLexGt (x::(oth (x,xs))) (j::ys)) 
						    xs)],
				   Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)),
					  Disj (ListXProd.mapX 
						    (fn (x,y) => 
							lookupLexGt (x::(oth (x,xs))) (y::(oth (y,ys))))
						    (xs,ys)) ] ]
			end
		in
		    Disj (ListXProd.mapX 
			      (fn (i,j) => lookupLexGt (i::(oth (i,ssidx))) (j::(oth (j,tsidx))))
			      (ssidx,tsidx))
		end

	and lexPropGt2 f s t ss ts = 
	    let 
 		(* val _ = debug (fn _=>  *)
		(* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString f) ^ "): " *)
		(* 			   ^ "<" ^ (PrintUtil.prSeq Term.toString  ss) ^ "> >> <"  *)
		(* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ ">\n")) *)
		val lexGtMap = ref ILM.empty
		val len  = length ss

	        (* si = ti $B$J$i!$$=$N0z?t$O9M$($J$/$F$h$$(B *)
		val idx = L.filter (fn i=> not (Term.equal (L.nth(ss,i), L.nth(ts,i))))
				   (L.tabulate (len,fn x=>x))

		fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		fun lookupLexGt (i::xs)  = 
		    let val xs' = ListMergeSort.sort Int.> xs
		    in
			case ILM.find (!lexGtMap, i::xs') of 
			    SOME p => p
			  | NONE => let 
 			      (* val _ = debug (fn _=>  *)
			      (* 		     print ("lookupLexGt ("  *)
			      (* 			    ^ (PrintUtil.prSeq Int.toString (i::xs'))  *)
			      (* 			    ^ "/"  *)
			      (* 			    ^ (PrintUtil.prSeq Int.toString (i::xs')) *)
			      (* 			    ^ ")\n")) *)
			      val p = lexGtSub (i::xs') 
			      val _ = symCount  := (!symCount) + 1
			      val q = Prop.Atom (!symCount)
			      val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  in (lexGtMap := ILM.insert (!lexGtMap, i::xs', q); q)
			  end
		    end
			
		and lexGtSub (i::xs) = (* i$B0z?t$rHf3S!%(Bxs $B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
		    let val si = L.nth (ss,i)
			val ti = L.nth (ts,i)
			val gt = lookupGt si ti
			val eq = lookupEq si ti
		    in
			Disj [ if gt <> False orelse eq <> False
			       then
				   Conj [pi (f,i+1), 
					 Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					 Disj [ if gt <> False then
						    Conj [gt,
							  majoProp f s 
								   (L.map (fn x=> (x,L.nth (ts,x))) xs)]
						else False,
						if eq <> False then
						    Conj [eq,
							  Disj (List.map
								    (fn x => 
									lookupLexGt (x::(oth (x,xs))))
								    xs) ]
						else False ]]
			       else False,
			       Conj  [Neg (pi (f,i+1)), 
				      Disj (List.map
						(fn x => lookupLexGt (x::(oth (x,xs))))
						xs) ] ]
		    end
	    in
		Disj (List.map
			  (fn i => lookupLexGt (i::(oth (i,idx))))
			  idx)
	    end

	(*** ss ==^lex tt $B$H$J$k$?$a$N@)Ls(B ***)
        (*  f,g $B$O(B not collapsing $B$H2>Dj(B *)
	and lexPropEq f g s t ss ts = 
	    if Fun.equal (f,g)
	    then lexPropEq2 f s t ss ts
	    else
		let 
 		    (* val _ = debug (fn _=>  *)
		    (* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString g) ^ "): " *)
		    (* 			   ^ "<" ^ (PrintUtil.prSeq Term.toString  ss) ^ "> >> <"  *)
		    (* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ ">\n")) *)
		    val lexEqMap = ref ILPM.empty
		    val lenss  = length ss
		    val ssidx = L.tabulate (lenss,fn x=>x)
		    val lents  = length ts
		    val tsidx = L.tabulate (lents,fn y=>y)
		    fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		    fun lookupLexEq (i::xs) (j::ys)  = 
			let val xs' = ListMergeSort.sort Int.> xs
			    val ys' = ListMergeSort.sort Int.> ys
			in
			    case ILPM.find (!lexEqMap, (i::xs',j::ys')) of 
				SOME p => p
			      | NONE => let 
				  val p = lexEqSub (i::xs') (j::ys')
				  val _ = symCount  := (!symCount) + 1
				  val q = Prop.Atom (!symCount)
				  val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			      in (lexEqMap := ILPM.insert (!lexEqMap, (i::xs',j::ys'), q); q)
			      end
			end

		    and lexEqSub (i::xs) (j::ys) = (* i,j $B0z?t$rHf3S!%(Bxs,ys $B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
			let val si = L.nth (ss,i)
			    val tj = L.nth (ts,j)
			    val eq = lookupEq si tj
			in
			    Disj [ if eq <> False
				   then
				       Conj [pi (f,i+1), pi (g,j+1),  
					     Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					     Conj (L.map (fn y=> lexStatus (g,j,y)) ys),
					     Conj [eq,
						   if (null xs) andalso (null ys)
						   then True
						   else
						       Disj (ListXProd.mapX 
								 (fn (x,y) => 
								     lookupLexEq (x::(oth (x,xs)))
										 (y::(oth (y,ys))))
								 (xs,ys)) ]]
				   else False,
				   Conj  [pi (f,i+1), Neg (pi (g,j+1)), 
					  Disj (List.map 
						    (fn y => lookupLexEq (i::xs) (y::(oth (y,ys)))) 
						    ys)],
				   Conj  [Neg (pi (f,i+1)), pi (g,j+1), 
					  Disj (List.map 
						    (fn x => lookupLexEq (x::(oth (x,xs))) (j::ys)) 
						    xs)],
				   Conj  [Neg (pi (f,i+1)), Neg (pi (g,j+1)),
					  if (null xs) andalso (null ys)
					  then True
					  else
					      Disj (ListXProd.mapX 
							(fn (x,y) => 
							    lookupLexEq (x::(oth (x,xs))) (y::(oth (y,ys))))
							(xs,ys)) ] ]
			end
		in
		    Disj (ListXProd.mapX 
			      (fn (i,j) => lookupLexEq (i::(oth (i,ssidx))) (j::(oth (j,tsidx))))
			      (ssidx,tsidx))
		end

	and lexPropEq2 f s t ss ts = 
	    let 
 		(* val _ = debug (fn _=>  *)
		(* 		    print ("(" ^ (Fun.toString f) ^ "," ^ (Fun.toString f) ^ "): " *)
		(* 			   ^ "<" ^ (PrintUtil.prSeq Term.toString  ss) ^ "> >> <"  *)
		(* 			   ^ (PrintUtil.prSeq Term.toString  ts) ^ ">\n")) *)
		val lexEqMap = ref ILM.empty
		val len  = length ss

	        (* si = ti $B$J$i!$$=$N0z?t$O9M$($J$/$F$h$$(B *)
		val idx = L.filter (fn i=> not (Term.equal (L.nth(ss,i),L.nth(ts,i))))
				   (L.tabulate (len,fn x=>x))

		fun oth (i, xs) = L.filter (fn x=> x<>i) xs

		fun lookupLexEq (i::xs) = 
		    let val xs' = ListMergeSort.sort Int.> xs
		    in
			case ILM.find (!lexEqMap, i::xs') of 
			    SOME p => p
			  | NONE => let 
			      val p = lexEqSub (i::xs') 
			      val _ = symCount  := (!symCount) + 1
			      val q = Prop.Atom (!symCount)
			      val _ = iffProps := (Prop.Iff (q,p))::(!iffProps)
			  in (lexEqMap := ILM.insert (!lexEqMap, i::xs', q); q)
			  end
		    end

		and lexEqSub (i::xs) = (* i$B0z?t$rHf3S!%(Bxs$B$K$D$$$F$O$^$@Hf3S$7$F$$$J$$(B *)
		    let val eq = lookupEq (L.nth (ss,i)) (L.nth (ts,i))
		    in
			Disj [ if eq <> False
			       then
				   Conj [pi (f,i+1), 
					 Conj (L.map (fn x=> lexStatus (f,i,x)) xs),
					 Conj [eq,
					       if (null xs)
					       then True
					       else
						   Disj (List.map
							     (fn x => 
								 lookupLexEq (x::(oth (x,xs))))
							     xs) ]]
			       else False,
			       Conj  [Neg (pi (f,i+1)),
				      if (null xs)
				      then True
				      else
					  Disj (List.map
						    (fn x => 
							lookupLexEq (x::(oth (x,xs))))
						    xs) ] ]
		    end
	    in
		Disj (List.map
			  (fn i => lookupLexEq (i::(oth (i,idx))))
			  idx)
	    end


    in
	case encodingType of 
	    GT => let val p = lookupGt l r in Conj (p::(!iffProps)) end
	  | GE => let val p = qrpoPropGe l r in Conj (p::(!iffProps)) end 
	  | EQ => let val p = lookupEq l r in Conj (p::(!iffProps)) end

    end (* of let *)



end (* of local *)
    
end (* of structre *)


