(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/po_solver.sml
 * description: termination solver by path ordering
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature PO_SOLVER = 
sig
    val runProfile: bool ref
    val runDebug: bool ref
    val runCertification: bool ref

    type options = {
	 useQuasi:bool,     (* quasi precedence $B$r;H$&$+$I$&$+(B *)
	 useLex:bool,       (* lexicographic status $B$r;H$&$+$I$&$+(B *)
	 useMul:bool,       (* multiset status $B$r;H$&$+$I$&$+(B *)
	 useAf:bool         (* argument filtering $B$r;H$&$+$I$&$+(B *)
    }

    val defaultOptions : options

    val poSolver: string -> string
		  -> options
		  -> (Term.term * Term.term) list 
		     * (Term.term * Term.term) list
                  -> bool

    val poSolverForGcrWithCpf: string -> string
		  -> options
		  -> (Term.term * Term.term) list 
		     * (Term.term * Term.term) list
                  -> (((int FunMap.map) * (Term.fun_key list) * ((Term.fun_key * int list) list))
			 * (unit -> string)) option

    val poSolverForGcr: string -> string
		  -> options
		  -> (Term.term * Term.term) list 
		     * (Term.term * Term.term) list
                  -> ((int FunMap.map) * (Term.fun_key list) * ((Term.fun_key * int list) list))   option

    val poSolverForGcrMulti: string -> string
		  -> options
		  -> (Term.term * Term.term list) list 
                  -> (int FunMap.map * (Term.term * Term.term) list) option

    val poSolverForGcrMulti2: string -> string
		  -> options
		  -> Term.term list
		  -> (Term.term * Term.term) list
		  -> (Term.term * Term.term) list list list
                  (* -> (int FunMap.map * (Term.term * Term.term) list) option *)
                  -> (int FunMap.map * (Fun.ord_key -> bool) * 
                      (Fun.ord_key -> Term.term list -> Term.term list) * 
                      (Term.term * Term.term) list) option



    val poSolverForNj: string -> string
		       -> options
		       -> Term.term * Term.term
		       -> (Term.term * Term.term) list * (Term.term * Term.term) list
                       -> bool * (unit -> string)

    val poSolverForNjwithNgeTotal: string -> string
		       -> options
		       -> Term.term * Term.term
		       -> (Term.term * Term.term) list * (Term.term * Term.term) list
                       -> bool * (unit -> string)

    val poSolverForNjwithNgePartial: string -> string
		       -> options
		       -> Term.term * Term.term
		       -> (Term.term * Term.term) list * (Term.term * Term.term) list
                       -> bool * (unit -> string)

    type encoding_info = { Fs : Fun.ord_key list,
			   LenFs : int,
			   RowLen : int,
			   FaList : (Fun.ord_key * int) list,
			   FaSet : FunIntSet.set,
			   FaMap : int FunMap.map,
			   PrecMap : int FunIntMap.map,
			   PrecMap2 : int FunPairMap.map, (* added for symbolic encoding *)
                           MulMap : int FunMap.map,
                           LexMap : int FunIntIntMap.map,
                           ColMap : int FunMap.map,
                           PiMap : int FunIntMap.map,
			   GtPrecTable : Prop.prop FunPairTable.hash_table,
			   EqPrecTable : Prop.prop FunPairTable.hash_table,
			   LexTable : Prop.prop FunIntIntTable.hash_table,
			   SameNthTable : Prop.prop FunIntPairTable.hash_table
			 }

    val mkSymMap: int ref -> Fun.ord_key list -> int FunMap.map
    val mkPrecMap: int ref -> (Fun.ord_key list * int) -> int FunIntMap.map
    val mkMulMap: int ref -> Fun.ord_key list -> int FunMap.map
    val mkLexMap: int ref -> FunIntSet.set -> int FunIntIntMap.map
    val mkColMap: int ref -> Fun.ord_key list -> int FunMap.map
    val mkPiMap: int ref -> FunIntSet.set -> int FunIntMap.map

    val empty_info: encoding_info

    val mkEncodingInfo: int ref 
			-> Fun.ord_key list * int * int * (Fun.ord_key * int) list * 
			   FunIntSet.set * int FunMap.map
			-> encoding_info

    val mkLexCond: options -> encoding_info -> Prop.prop
    val mkStrictPrecCond: options -> encoding_info -> Prop.prop
    val mkAfCond: options -> encoding_info -> Prop.prop
    val mkQuasiRpoCond: options -> encoding_info -> Prop.prop
    val encodeMulStatus: encoding_info -> Fun.ord_key -> Prop.prop
    val encodeLexStatus: encoding_info -> Fun.ord_key * int * int -> Prop.prop
    val encodeSameNthStatus: encoding_info -> (Fun.ord_key * int) * (Fun.ord_key * int) -> Prop.prop
    val encodeColStatus: encoding_info -> Fun.ord_key -> Prop.prop
    val encodePiStatus: encoding_info -> Fun.ord_key * int -> Prop.prop
    val encodeGtPrec: encoding_info -> Fun.ord_key * Fun.ord_key -> Prop.prop
    val encodeEqPrec: encoding_info -> Fun.ord_key * Fun.ord_key -> Prop.prop
    val lookupFaMap: (int FunMap.map) * Fun.ord_key -> int
    val lookupGtPrecTable: encoding_info -> Fun.ord_key * Fun.ord_key -> Prop.prop
    val lookupEqPrecTable: encoding_info -> Fun.ord_key * Fun.ord_key -> Prop.prop
    val lookupLexTable: encoding_info -> Fun.ord_key * int * int -> Prop.prop
    val lookupSameNthTable: encoding_info -> (Fun.ord_key * int) * (Fun.ord_key * int) -> Prop.prop

    val isAssignedByTrue: int array -> int -> bool
    val evalProp: int array -> encoding_info -> Prop.prop -> bool
    val printWeight: int array -> encoding_info -> unit
    val printPiMap : int array -> encoding_info -> unit
    val getPiMap: int array -> encoding_info -> (int list) FunMap.map
    val printColMap : int array -> encoding_info -> unit
    val getColSet: int array -> encoding_info -> FunSet.set
    val mkPrecList: int array -> encoding_info -> (Fun.ord_key list) list
    val prArgStatusAF: int array -> encoding_info -> options -> (Fun.ord_key * int) -> string
    val prArgStatus: int array -> encoding_info -> options -> (Fun.ord_key * int) -> string
    val printInfo: int array -> encoding_info -> options -> unit

    val encodeOrderConstraint: Order.encoding_type 
			       -> int ref -> encoding_info
			       -> options 
			       -> Term.term * Term.term -> Prop.prop
			       
    val mkPrecMap2: int ref -> Fun.ord_key list -> int FunPairMap.map

end

structure PoSolver : PO_SOLVER =
struct
local
    open Prop
    structure A = Atom
    structure CU = CertifyUtil
    structure L = List
    structure LP = ListPair
    structure LU = ListUtil
    structure FS = FunSet
    structure FPM = FunPairMap
    structure FPT = FunPairTable
    structure FIT = FunIntTable
    structure FIPT = FunIntPairTable
    structure FM = FunMap
    structure FIS = FunIntSet
    structure FIM = FunIntMap
    structure FII = FunIntInt
    structure FIIM = FunIntIntMap
    structure FIIT = FunIntIntTable
    structure TPM = TermPairMap
    fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs
    local
	fun logMain n k ans = if n <= k then ans
			      else logMain n (k * 2) (ans + 1)
    in
    fun log n = logMain n 2 1
    end
    open PrintUtil
in


    exception PoSolverError
	  
    val runProfile = ref false : bool ref
    val runDebug = ref false : bool ref
    fun debug f = if !runDebug then f () else ()
    val runCertification = ref false : bool ref

    type options = {
	 useQuasi:bool,     (* quasi precedence $B$r;H$&$+$I$&$+(B *)
	 useLex:bool,       (* lexicographic status $B$r;H$&$+$I$&$+(B *)
	 useMul:bool,       (* multiset status $B$r;H$&$+$I$&$+(B *)
	 useAf:bool         (* argument filtering $B$r;H$&$+$I$&$+(B *)
    }

    val defaultOptions = {
	 useQuasi = true,
	 useLex = false,  (** why not true ?? 2017 **)
	 useMul = false,  (** why not true ?? 2017 **)
	 useAf = true
    } : options

    fun lookupFaMap (faMap,f) = 
	case FM.find (faMap,f) of
	    SOME i => i
	  | NONE => (print ("looking up faMap fails: (" ^ (Fun.toString f)  ^ "\n");
		     raise PoSolverError)

    (****************)
    (** Symbol Map **)
    (****************)
    (* +-+------+ *)
    (* |f|!c    | *)
    (* |g|!c + 1| *)
    (* |h|!c + 2| *)
    (* |s|!c + 3| *)
    (* +-+------+ *)
    (*  lenFs = 4      *)

    fun mkSymMap counterRef fs =
	L.foldr 
	    (fn (f,en) => (debug (fn _ => (print ("Sym " ^ (Fun.toString f) ^ " : "  
						  ^ (Int.toString (1 + !counterRef)) ^ "\n")));
			   counterRef := 1 + !counterRef;
			   FM.insert (en,f,!counterRef)))
	    FM.empty 
	    fs

    fun lookupSymMap (symMap,f) = 
	case FM.find (symMap,f) of
	    SOME i => i
	  | NONE => (print ("looking up symMap fails: "  ^ (Fun.toString f) ^ "\n");
		     raise PoSolverError)


    (************************)
    (*** ENCODING SECTION ***)
    (************************)

   type encoding_info = { Fs : Fun.ord_key list,
			   LenFs : int,
			   RowLen : int,
			   FaList : (Fun.ord_key * int) list,
			   FaSet : FIS.set,
			   FaMap : int FM.map,
			   PrecMap : int FIM.map,
			   PrecMap2 : int FPM.map,
                           MulMap : int FM.map,
                           LexMap : int FIIM.map,
                           ColMap : int FM.map,
                           PiMap : int FIM.map,
			   GtPrecTable : Prop.prop FPT.hash_table,
			   EqPrecTable : Prop.prop FPT.hash_table,
			   LexTable : Prop.prop FIIT.hash_table,
			   SameNthTable : Prop.prop FIPT.hash_table
			 }

    val empty_info = { Fs = [],
		       LenFs = 0,
		       RowLen = 0,
		       FaList = [],
		       FaSet = FIS.empty,
		       FaMap = FM.empty,
		       PrecMap = FIM.empty,
		       PrecMap2 = FPM.empty, (* added *)
                       MulMap = FM.empty,
                       LexMap = FIIM.empty,
		       ColMap = FM.empty,
                       PiMap = FIM.empty,
		       GtPrecTable = FPT.mkTable (0,PoSolverError),
		       EqPrecTable = FPT.mkTable (0,PoSolverError),
		       LexTable = FIIT.mkTable (0,PoSolverError),
		       SameNthTable = FIPT.mkTable (0,PoSolverError)
		     } : encoding_info




    (******************)
    (** Codish Table **)
    (******************)
    (** f > g iff w(f) > w(g), where w(f) is a natunal numbe weight      **)
    (** each weight w(f) is represented by a binary number (f_1 f_2 f_3) **)
    (** where f_i is a boolean value                                     **)
    (* --+---------------------    *)
    (*row    1      2      3       *)
    (* --+---------------------    *)
    (* f | init   init+1  init+2   *)
    (* g | init+3 init+4  init+5   *)
    (* h | init+6 init+7  init+8   *)
    (* s | init+9 init+10 init+11  *)
    (* --+---------------------    *)
    (*  lowLen = 3, lenFs = 4      *)

    fun mkPrecMap counterRef (fs,rowLen) = 
	let
	    val rows = List.tabulate (rowLen, fn n => n + 1)
	    val products = ListXProd.mapX (fn (f,i) => (f,i)) (fs,rows)
	in
	    L.foldr 
		(fn ((f,i),en) => (debug (fn _ => (print ("Codish (" ^ (Fun.toString f) ^ ", " ^ (Int.toString i) 
						     ^ ") : "  ^ (Int.toString (1 + !counterRef)) ^ "\n")));
				   counterRef := 1 + !counterRef;
				   FIM.insert (en,(f,i),!counterRef)))
		FIM.empty 
		products
	end

    fun lookupPrecMap (precMap,(f,i)) = 
	case FIM.find (precMap,(f,i)) of
	    SOME n => n
	  | NONE => (print ("looking up precMap fails: (" 
			    ^ (Fun.toString f)  ^ "," ^ (Int.toString i)  ^ ")\n");
		     raise PoSolverError)


    (* f > g $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeGtPrec (encoding:encoding_info) (f,g) =
	let val rowLen = #RowLen encoding
	    val precMap = #PrecMap encoding
	in
	    Disj (L.tabulate (rowLen,
		    fn i => Conj
				(Atom (lookupPrecMap (precMap, (f,i+1)))::
				 (* Atom (~(lookupPrecMap (precMap, (g,i+1))))::*)  (* fixed 2015/1/15 *)
				 Neg (Atom (lookupPrecMap (precMap, (g,i+1))))::
				 (L.tabulate (i,
					   fn j => (Iff (Atom (lookupPrecMap (precMap, (f,j+1))),
							 Atom (lookupPrecMap (precMap, (g,j+1))))))))))
	end	    

    (* $B%a%b(B & Look Up *)
    fun mkGtPrecTable lenFs = FPT.mkTable (lenFs * lenFs, PoSolverError)

    fun lookupGtPrecTable encoding (f,g) =
	let val gtPrecTable = #GtPrecTable encoding 
	in
	    case (FPT.find gtPrecTable (f,g))
	     of SOME p => p
	      | NONE => let val p = encodeGtPrec encoding (f,g)
			in (FPT.insert gtPrecTable ((f,g),p); p)
			end
	end

    (* f $B$H(B g $B$,(B $BF1$8(B precedence $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeEqPrec (encoding:encoding_info) (f,g) =
	let val rowLen = #RowLen encoding
	    val precMap = #PrecMap encoding
	in
	    Conj (L.tabulate (rowLen,
			   fn i => (Iff (Atom (lookupPrecMap (precMap, (f,i+1))),
					 Atom (lookupPrecMap (precMap, (g,i+1)))))))
	end

    (* $B%a%b(B & Look Up *)
    fun mkEqPrecTable lenFs = FPT.mkTable (lenFs * lenFs, PoSolverError)

    fun lookupEqPrecTable encoding (f,g) =
	let val eqPrecTable = #EqPrecTable encoding
	in
	   case (FPT.find eqPrecTable (f,g))
	    of SOME p => p
	       | NONE => let val p = encodeEqPrec encoding (f,g)
			 in (FPT.insert eqPrecTable ((f,g),p); p)
			 end
	end

    (* f $B$H(B g $B$,(B $BF1$8(B precedence $B$G$J$$$3$H$rI=$o$9O@M}<0(B *)
    fun encodeNeqPrec (encoding:encoding_info) (f,g) = 
	Neg (lookupEqPrecTable encoding (f,g))

    (* strict precedence $B$K$9$k$?$a$N@)Ls(B *)
    fun mkStrictPrecCond (opt:options) (encoding:encoding_info) =
	if #useQuasi opt
	then Prop.True
	else let 
		val eqPrecTable = #EqPrecTable encoding
		fun prod [] = []
		  | prod (x::ys) = L.@ (L.map (fn y => (x,y)) ys, prod ys)
		val fs = #Fs encoding
	    in
		Conj (List.map (encodeNeqPrec encoding) (prod fs))
	    end

    (******************)
    (** Status Table **)
    (******************)
    (* +-+------+ *)
    (* |f|init  | *)
    (* |g|init+1| *)
    (* |h|init+2| *)
    (* |s|init+3| *)
    (* +-+------+ *)

    fun mkMulMap counterRef fs = 
	L.foldr 
	    (fn (f,en) => (debug (fn _ => (print ("Mul " ^ (Fun.toString f) ^ " : "  
						  ^ (Int.toString (1 + !counterRef)) ^ "\n")));
			   counterRef := 1 + !counterRef;
			   FM.insert (en,f,!counterRef)))
	    FM.empty 
	    fs

    fun lookupMulMap (mulMap,f) = 
	case FM.find (mulMap,f) of
	    SOME i => i
	  | NONE => (print ("looking up mulMap fails: " ^ (Fun.toString f)  ^ "\n");
		     raise PoSolverError)

    (* f $B$,(B Staus MUL $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeMulStatus (encoding:encoding_info) f = 
	Atom (lookupMulMap (#MulMap encoding, f))

    (* f ~= g $B$J$i(B MUL status $B$OF1$8(B *)
    (* Lex option $B$N$_$J$i!$(B MUL status $B$O(B LEX $B$N$_(B *)
    (* Mul option $B$N$_$J$i!$(B MUL status $B$O(B MUL $B$N$_(B *)
    fun mkQuasiRpoCond (opt:options) (encoding:encoding_info) =
	let val fs = #Fs encoding
	    val p = if #useLex opt 
		       andalso #useMul opt 
		       andalso #useQuasi opt
		    then let fun mkCond [] = []
			       | mkCond (f::xs) = 
				 L.@ (L.map (fn g => 
						Imp (lookupEqPrecTable encoding (f,g), 
						     Iff (encodeMulStatus encoding f, 
							  encodeMulStatus encoding g)))
					    xs,
				      mkCond xs)
			 in Conj (mkCond fs)
			 end
		    else True
	    (* added 2015/01/17 *)
	    val q = case (#useLex opt, #useMul opt) of  
			(true,true) => True
		      | (false,false) => True
		      | (true,false) => Conj (L.map (fn f => Neg (encodeMulStatus encoding f)) fs)
		      | (false,true) => Conj (L.map (fn f => encodeMulStatus encoding f) fs)
	in Conj [p,q]
	end
	


    (************************************)
    (** $B<-=q<0%9%F!<%?%9$N%(%s%3!<%G%#%s%0(B *****)
    (*****************  CODISH ENCODING ****************)
    (*  f      1 2    prec     prop vars               *)
    (*      +-+---+          --+------------           *)
    (*      |1|1 1| 2+1=3    f1| x31 x32 x33           *)
    (*      +-+---+            |                       *)
    (*      |2|1 0| 2+0=2    f2| x34 x35 x36           *)
    (*      +-+---+            |                       *)
    (*      |3|0 1| 0+1=1    f3| x37 x38 x39           *)
    (*      +-+---+            |                       *)
    (*      |4|0 0| 0+0=0    f4| x40 x41 x42           *)
    (*      +-+---+                                    *)
    (*  arity(f) = 4                                   *)
    (*  arity(f) = 1 $B$N$H$-$O$J$7(B                       *)
    (*  (f,i) $B$,(B (f,j) $B$h$j@h(B <=> w(f,i) > w(f,j)       *)
    (***************************************************)

    fun mkLexMap counterRef faSet = 
	FIS.foldr (fn ((f,ar),map) => 
		      if ar <= 1 then map
		      else
			  ListXProd.foldX
			      (fn (i,j,m) => (debug (fn _ => (print ("Lex (" ^ (Fun.toString f) ^ ", " ^ (Int.toString i) 
						     ^ ", " ^ (Int.toString j) 
						     ^ ") : "  ^ (Int.toString (1 + !counterRef)) ^ "\n")));
					      counterRef := 1 + (!counterRef);
					      FIIM.insert (m,(f,i,j),!counterRef)))
			      (L.tabulate (ar,fn i=>i+1),
			       L.tabulate (log ar,fn i=>i+1))
			      map)
		  FIIM.empty
		  faSet

    fun lookupLexMap (lexMap, (f,i,j)) = 
	if i > 0 andalso j > 0
	then
	    case FIIM.find (lexMap,(f,i,j)) of
		SOME i => i
	      | NONE => (print ("looking up lexMap fails: (" ^ (Fun.toString f)  ^ ","
				^ (Int.toString i)  ^ "," ^ (Int.toString j)  ^ ")\n");
			 raise PoSolverError)
	else  (print ("looking up lexMap fails: (" ^ (Fun.toString f)  ^ ","
		      ^ (Int.toString i)  ^ "," ^ (Int.toString j)  ^ ")\n");
	       raise PoSolverError)

    (* $BBh(Bi$B0z?t$OBh(Bj$B0z?t$h$j@h$KHf3S$9$k$3$H$rI=$o$9O@M}<0(B ($B$?$@$7!$(Bi <> j) *)
    fun encodeLexStatus (encoding:encoding_info) (f,i,j) =  
	let val faMap = #FaMap encoding
	    val lexMap = #LexMap encoding
	in
	    if i <> j 
	    then
		Disj (L.tabulate (log (lookupFaMap (faMap, f)),
                           (* k$BHVL\$N7e$G!$(B(f,i)=true, (f,j)=false $B$+$D>e$N7e$OEy$7$$(B *)
			       fn k => Conj
					   (Atom (lookupLexMap (lexMap, (f,i+1,k+1)))::
					   (* Atom (~(lookupLexMap (lexMap, (f,j+1,k+1))))::  *) (* fixed 2015/1/15 *)
					    Neg (Atom (lookupLexMap (lexMap, (f,j+1,k+1))))::
					    (L.tabulate (k,
						      fn l => 
							 (Iff (Atom (lookupLexMap (lexMap,(f,i+1,l+1))),
							       Atom (lookupLexMap (lexMap,(f,j+1,l+1))))))))))
	    else (print ("encodeLexStatus Error: (" ^ (Int.toString i) ^ ","
			 ^ (Int.toString j) ^ ")\n");
		  raise PoSolverError)
	end
      
    (* $B%a%b(B & Look Up *)
    fun mkLexTable faList = 
	let val len = L.foldr (fn ((_,ar),m) => ar * (log ar) + m) 0 faList
	in FIIT.mkTable (len, PoSolverError)
	end

    fun lookupLexTable encoding fij =
	let val lexTable = #LexTable encoding
	in
	   case (FIIT.find lexTable fij)
	    of SOME p => p
	     | NONE => let val p = encodeLexStatus encoding fij
		       in (FIIT.insert lexTable (fij,p); p)
		       end
	end

   (* f$B$NBh(Bi$B0z?t$H!$(B g$B$NBh(Bj$B0z?t$,!$F1$8HVL\$KHf3S$5$l$k(B *)
    fun encodeSameNthStatus (encoding:encoding_info) ((f,i),(g,j)) =  
	let 
	    val faMap = #FaMap encoding
	    val lexMap = #LexMap encoding
	    val arF = lookupFaMap (faMap, f)
	    val arG = lookupFaMap (faMap, g)
	    val logArF = log arF
	    val logArG = log arG
	    val _ = print ("encodeSameNthStatus " ^ (Fun.toString f) ^ "/" ^ (Int.toString i) ^ 
			   ", " ^ (Fun.toString g) ^ "/" ^ (Int.toString j) ^ "\n")
	    val _ = print ("encodeSameNthStatus(sub) logArF:" ^ (Int.toString logArF) ^ 
			   ", logArG:" ^ (Int.toString logArG) ^ "\n")

            (* fixed 2017/3/7 *)
	    fun sameNthStatusSub (fmax,gmax) ((f,i),(g,j)) =  
		let val lf = log fmax
		    val lg = log gmax
		    val diff = lg - lf
		in
		Conj [Conj (L.tabulate (lf, (* lf$B$NJ}$,>.$5$$!$(B*)
				     fn k => Iff (Atom (lookupLexMap (lexMap, (f,i+1,k+1))), 
						  Atom (lookupLexMap (lexMap, (g,j+1,diff+k+1)))))),
		      Conj (L.tabulate (diff, (* $B:9J,$N$H$3$m$O(B 1 $B$N$O$:(B *)
				     fn k => Iff (True, Atom (lookupLexMap (lexMap,(g,j+1,k+1))))))]
		end
	in
	    if arF <= arG
	    then sameNthStatusSub (arF,arG) ((f,i),(g,j))
	    else sameNthStatusSub (arG,arF) ((g,j),(f,i))
	end

    (* $B%a%b(B & Look Up *)
    fun mkSameNthTable faList  = 
	let val len = L.foldr (fn ((_,ar),m) => ar + m) 0 faList
	in FIPT.mkTable (len*len, PoSolverError)
	end

    fun lookupSameNthTable encoding (fi,gj) =
	let val sameNthTable = #SameNthTable encoding
	in
	    case (FIPT.find sameNthTable (fi,gj))
	     of SOME p => p
	      | NONE => let val p = encodeSameNthStatus encoding (fi,gj)
			in (FIPT.insert sameNthTable ((fi,gj),p); p)
			end
	end


   (* $B<-=q<0=g=x$N0z?t$N=E$_$,(B strict $B$J=g=x$K$J$C$F$$$k(B *)
    fun mkLexCond (opt:options) (encoding:encoding_info) =
	if #useLex opt
	then let val faList = #FaList encoding
		 val sameNthTable = #SameNthTable encoding
	     in Conj (L.map
			  (fn (f,ar) =>
			      let val xs = L.tabulate (ar,fn x=>x)
				  val p2 = Neg (encodeMulStatus encoding f)  (* added 2015/01/17 *)
			      in Imp (p2, Conj (ListXProd.mapX
						    (fn (i,j) 
							=> if i < j then Neg (lookupSameNthTable encoding ((f,i),(f,j)))
							   else True)
						    (xs,xs)))
			      end)
			  faList)
	     end
	else Prop.True



    (*********************************************)
    (** $B0z?t%U%#%k%?%j%s%0$N%9%F!<%?%9$N%(%s%3!<%G%#%s%0(B ****)
    (*********************************************)
    (* encodeColStatus f                              *)
    (*  \pi(f) = i => true                       *)
    (*  \pi(f) = [...] => false                  *)
    (*********************************************)

    fun mkColMap counterRef fs = 
	L.foldr 
	    (fn (f,en) => (debug (fn _ => (print ("Col " ^ (Fun.toString f) ^ " : "  
						  ^ (Int.toString (1 + !counterRef)) ^ "\n")));
			   counterRef := 1 + !counterRef;
			   FM.insert (en,f,!counterRef)))
	    FM.empty 
	    fs

    fun lookupColMap (colMap,f) = 
	case FM.find (colMap,f) of
	    SOME i => i
	  | NONE => (print ("looking up colMap fails: (" ^ (Fun.toString f)  ^ "\n");
		     raise PoSolverError)

    fun encodeColStatus (encoding:encoding_info) f = 
	let val colMap = #ColMap encoding
	in Atom (lookupColMap (colMap,f))
	end

    (**************************************************)
    (****** $B0z?t%U%#%k%?%j%s%0$N%9%F!<%?%9$N%(%s%3!<%G%#%s%0(B *****)
    (**************************************************)
    (* pi (f,i) = true iff  pi(f) = i            ***)
    (*                      or i  \in pi(f)       ***)
    (* (plus,1), (plus,2), ... *)
		    
    fun mkPiMap counterRef faSet =
	FIS.foldr (fn ((f,ar),pm) => 
		      List.foldr
			  (fn (i,m) => (debug (fn _ => (print ("AF (" ^ (Fun.toString f) ^ ", " ^ (Int.toString i) 
					       ^ ") : "  ^ (Int.toString (1 + !counterRef)) ^ "\n")));
					counterRef := 1 + !counterRef;
					FIM.insert (m,(f,i),!counterRef)))
			  pm
 			  (L.tabulate (ar,fn i=>i + 1)))
		  FIM.empty
		  faSet

    fun lookupPiMap (piMap,(f,i)) = 
	case FIM.find (piMap,(f,i))
	 of SOME n => n
	  | NONE =>  (print ("looking up piMap fails: (" ^ (Atom.toString f) ^ "," 
			     ^ (Int.toString i) ^ ")\n");
		      raise PoSolverError)

    fun encodePiStatus (encoding:encoding_info) (f,i) = 
	let val piMap = #PiMap encoding
	in Atom (lookupPiMap (piMap,(f,i)))
	end

    fun mkAfCond (opt:options) (encoding:encoding_info) =
	let val faList = #FaList encoding
	    val faMap = #FaMap encoding
	    val fs = #Fs encoding
            (****** useAF = false $B$J$i!$(BencodeColStatus f = false and encodePiStatus (f,i) = true for all i *****)
	    val p0 = if #useAf opt
		     then True
		     else Conj (L.map (fn (f,ar) => Conj (Iff (encodeColStatus encoding f, False)::
							  (L.tabulate (ar, fn i => Iff (encodePiStatus encoding (f,i+1), True)))))
				      faList)
	   (****** encodeColStatus f = true $B$J$i(B pi(f)$B$,(Btrue$B$H$J$k$N$O(B1$B$D$@$1(B*****)
	    val p1 = if #useAf opt
		     then Conj (L.map (fn (f,ar) =>
				     Imp (encodeColStatus encoding f,
					  one (L.tabulate (ar,fn j => encodePiStatus encoding (f,j+1)))))
				 faList)
		     else True
	    val products0 = ListXProd.mapX (fn (f,g) => (f,g)) (fs,fs)
	    val products = L.filter (not o Fun.equal) products0
	    (*  f ~= g  $B$J$i(B encodeColStatus f = encodeColStatus g *)
	    val p2 = if #useAf opt
		     then Conj (L.map (fn (f,g) =>
				     Imp (lookupEqPrecTable encoding (f,g),
					  Iff (encodeColStatus encoding f, encodeColStatus encoding g)))
				 products)
		     else True
            (*  f ~= g  $B$J$i(B |pi(f)| = |pi(g)| *)
	    val p3 = Conj (L.map (fn (f,g) =>
				     let val arF = lookupFaMap (faMap, f)
					 val arG = lookupFaMap (faMap, g)
				     in Imp (lookupEqPrecTable encoding (f,g),
					     equalNumber (L.tabulate (arF,fn j => encodePiStatus encoding (f,j+1)),
							  L.tabulate (arG,fn j => encodePiStatus encoding (g,j+1))))
				     end)
				 products)
	in Conj [p0,p1,p2,p3]
	end


    (***************************************)
    (*** $BA4It$N(B encoding map & table $B$N:n@.(B ***)
    (***************************************)

   (* couterRef: $B$9$G$K;H$C$F$$$k%+%&%s%?HV9f$N:GBgCM(B *)
    fun mkEncodingInfo counterRef (fs,lenFs,rowLen,faList,faSet,faMap) =
	let val precMap = mkPrecMap counterRef (fs,rowLen)
	    val mulMap = mkMulMap counterRef fs
	    val lexMap = mkLexMap counterRef faSet
	    val colMap = mkColMap counterRef fs
	    val piMap = mkPiMap counterRef faSet
	    val gtPrecTable = mkGtPrecTable lenFs
	    val eqPrecTable = mkEqPrecTable lenFs
	    val lexTable = mkLexTable faList
	    val sameNthTable = mkSameNthTable faList
	in
	    { Fs = fs,
	      LenFs = lenFs,
	      RowLen = rowLen,
	      FaList = faList,
	      FaSet = faSet,
	      FaMap = faMap,
	      PrecMap = precMap,
	      PrecMap2 = FPM.empty,
              MulMap = mulMap,
              LexMap = lexMap,
              ColMap = colMap,
              PiMap = piMap,
	      GtPrecTable = gtPrecTable,
	      EqPrecTable = eqPrecTable,
	      LexTable = lexTable,
	      SameNthTable = sameNthTable } : encoding_info
	end


    (*******************)
    (*** $B=PNO7k2L$N2r@O(B ***)
    (*******************)

    (* $BBh(Bi$BHVL\$NL?BjJQ?t$,??$+(B *)
    fun isAssignedByTrue resultAr i = 
	if i <= Array.length resultAr
	   andalso 0 < i
	then Array.sub (resultAr,i-1) >  0
	else (print ("isAssingedByTrue: index out of bouds: " 
		     ^ (Int.toString (i-1))
		     ^ "\n"); raise PoSolverError)

    fun evalProp resultAr (encoding:encoding_info) p = 
	case p of
	    Prop.Atom i => if i > 0 then isAssignedByTrue resultAr i 
		      else if i < 0 then not (isAssignedByTrue resultAr (~i))
		      else (print "zero is not used as var index.\n"; raise PoSolverError)
	  | Prop.Conj ps => L.all (evalProp resultAr encoding) ps
	  | Prop.Disj ps => L.exists (evalProp resultAr encoding) ps
	  | Prop.Neg p => not (evalProp resultAr encoding p)
	  | Prop.Imp (p,q)  => if evalProp resultAr encoding p
			       then evalProp resultAr encoding q
			       else true
	  | Prop.Iff (p,q) => (evalProp resultAr encoding p)
			      = (evalProp resultAr encoding q)
	  | Prop.IfThenElse (c,p,q) => 
	    if evalProp resultAr encoding c
	    then evalProp resultAr encoding p
	    else evalProp resultAr encoding q


    (*** $B4X?t5-9f$N=E$_(B ***)
    fun weightOfFun resultAr (encoding:encoding_info) f =
	let val rowLen = #RowLen encoding
	    val precMap = #PrecMap encoding
	    val ns = L.map (fn i => lookupPrecMap (precMap, (f,i+1)))
			   (L.tabulate (rowLen, fn x => x))
	in
	    List.foldl (fn (k,sum) 
			   => 2 * sum + (if isAssignedByTrue resultAr k
					 then 1 else 0)) 
		       0 ns
	end

    fun printWeight resultAr (encoding:encoding_info) = 
	let val fs = #Fs encoding
	    val _ = debug (fn _ => print "weight:\n")
	in
	    L.app (fn f =>
		      (print (" " ^ (Fun.toString f) ^ ": "
			      ^  (Int.toString (weightOfFun resultAr encoding f))
			      ^ " ")))
		  fs;
	    print "\n"
	end


    (*** $B4X?t5-9f$N=g0L%j%9%H$N:n@.(B ***)	  
    fun mkPrecList resultAr (encoding:encoding_info) = 
	let val fs = #Fs encoding
	    val mp = List.foldr 
			 (fn (f,map) 
			     => FM.insert (map,f,weightOfFun resultAr encoding f))
			 FM.empty
			 fs
	    val sorted = 
		ListMergeSort.sort  
		    (fn (f,g) => Int.< (valOf (FM.find (mp,f)), valOf (FM.find (mp,g))))
		    fs
	    (* [1,2,2,3,3,3] => [[1],[2,2],[3,3,3]] *)
	    fun divideToEqClass equal [] = []
	      | divideToEqClass equal (a::sorted) =
		let
		    fun divide a [] = [[a]]
		      | divide a (b::xs) = 
			let val pre = divide b xs
			in if null pre
			   then (print "divideToEqClass\n"; raise PoSolverError)
			   else if equal (a, b)
			   then  (a::hd pre)::(tl pre)
			   else  ([a]::pre)
			end
		in
		    divide a sorted
		end
	    fun equal (f,g) = (valOf (FM.find (mp,f))) = (valOf (FM.find (mp,g)))
	in
	    divideToEqClass equal sorted
	end


    (*** Lex status $B$N2r@O(B ***)
    fun greaterLex resultAr (encoding:encoding_info) f (i,j) =  (* i <> j, i,j = 1,.. *)
	let val faMap = #FaMap encoding
	    val lexMap = #LexMap encoding
	    val ar = lookupFaMap (faMap, f)
	    val lnAr = log ar
	    val w_i = List.foldl (fn (k,sum) 
				     => 2 * sum 
					+ (if isAssignedByTrue resultAr k 
					   then 1 else 0))   
					   (* then 0 else 1))  *** bug fix 2015/01/17  ***)
				 0 
				 (L.tabulate (lnAr, fn k => lookupLexMap (lexMap,(f,i,k+1))))
	    val w_j = List.foldl (fn (k,sum) 
				     => 2 * sum + (if isAssignedByTrue resultAr k 
						   then 1 else 0)) 
					   (* then 0 else 1))  *** bug fix 2015/01/17  ***)
				 0 
				 (L.tabulate (lnAr, fn k => lookupLexMap (lexMap, (f,j,k+1))))
	in
	    w_i > w_j
	end


    fun printPiMap resultAr (encoding:encoding_info) = 
	let val piMap = #PiMap encoding
	    val faList = #FaList encoding
	    val _ = debug (fn _ => print "piMap:\n")
	in
	    L.map (fn (f,ar) =>
		      (print (" " ^ (Fun.toString f) ^ ": ");
		       L.app (fn i =>
				 if isAssignedByTrue resultAr (lookupPiMap (piMap,(f,i+1)))
				 then print ((Int.toString (i+1)) ^ " ")
				 else ())
			     (L.tabulate (ar, fn i=>i))))
		  faList;
	    print "\n"
	end

    fun getPiMap resultAr (encoding:encoding_info) = 
	let val piMap = #PiMap encoding
	    val faList = #FaList encoding
	in
	    L.foldl (fn ((f,ar),map) =>
		      FM.insert (map, f, 
				L.filter (fn i => isAssignedByTrue resultAr (lookupPiMap (piMap,(f,i+1))))
					 (L.tabulate (ar, fn i=>i))))
		    FM.empty
		    faList
	end

    fun getMulList resultAr (encoding:encoding_info) = 
	let val mulMap = #MulMap encoding
	    val fs = #Fs encoding
	in L.filter (fn f => isAssignedByTrue resultAr (lookupMulMap (mulMap,f))) fs
	end

    fun printColMap resultAr (encoding:encoding_info) = 
	let val colMap = #ColMap encoding
	    val fs = #Fs encoding
	    val _ = debug (fn _ => print "colMap:\n")
	in
	    L.map (fn f =>
		      (print (" " ^ (Fun.toString f) ^ ": ");
		       if isAssignedByTrue resultAr (lookupColMap (colMap,f))
		       then print "Y "
		       else print "N "))
		  fs;
	    print "\n"
	end

    fun getColSet resultAr (encoding:encoding_info) = 
	let val colMap = #ColMap encoding
	    val fs = #Fs encoding
	in
	    L.foldl (fn (f,set) =>
		      if isAssignedByTrue resultAr (lookupColMap (colMap,f))
		      then FS.add (set,f)
		      else set)
		    FS.empty
		    fs
	end

    (*** $B4X?t5-9f$N0z?tHf3S=g=x%j%9%H$N:n@.(B[AF$B$J$7$N>l9g(B] ***)	  
    fun mkLexList resultAr (encoding:encoding_info) (f,ar) = 
	ListMergeSort.sort 
	    (fn (i,j) => greaterLex resultAr encoding f (i+1,j+1))
	    (L.tabulate (ar, fn i=>i))

    (*** $B4X?t5-9f$N0z?tHf3S=g=x%j%9%H$N:n@.(B[AF$B$"$j$N>l9g(B] ***)	  
    fun mkLexListAF resultAr (encoding:encoding_info) (f,ar) = 
	let
	    val piMap = #PiMap encoding
	    val ns = List.filter 
			 (fn i => isAssignedByTrue resultAr (lookupPiMap (piMap,(f,i+1))))
			 (L.tabulate (ar, fn i=>i))
	in
	    ListMergeSort.sort 
		(fn (i,j) => greaterLex resultAr encoding f (i+1,j+1))
		ns
	end

    fun getLexListMapAF resultAr (encoding:encoding_info) = 
	let val faList = #FaList encoding
	in L.foldl (fn ((f,ar),map) =>
		       FM.insert (map, f, mkLexListAF resultAr (encoding:encoding_info) (f,ar)))
		   FM.empty
		   faList
	end


    (*** Lex/Mul$BIU$-$N0z?tHf3S=g=x%j%9%H=PNO(B[AF$B$J$7$N>l9g(B] ***)
    fun prArgStatus resultAr (encoding:encoding_info) (opt:options) (f,ar)  = 
	let val mulMap = #MulMap encoding
	    val ns = L.map (fn x => x+1) (mkLexList resultAr encoding (f,ar))
	in
	    if (#useMul opt) andalso (#useLex opt)
	    then
		if isAssignedByTrue resultAr (lookupMulMap (mulMap, f))
  		then "Mul" 
		else "Lex [" ^ (PrintUtil.prSeq Int.toString ns) ^ "]"
	    else if (#useMul opt) 
	    then "Mul" 
	    else 
		"Lex [" ^  (PrintUtil.prSeq Int.toString ns) ^ "]"
	end

    (*** Lex/Mul$BIU$-$N0z?tHf3S=g=x%j%9%H=PNO(B[AF$B$"$j$N>l9g(B] ***)
    fun prArgStatusAF resultAr (encoding:encoding_info) (opt:options) (f,ar)  = 
	let val ns = L.map (fn x => x+1) (mkLexListAF resultAr encoding (f,ar))
	    val colMap = #ColMap encoding
	    val mulMap = #MulMap encoding
	    val isCol = isAssignedByTrue resultAr (lookupColMap (colMap,f))
	    val ms = ListMergeSort.sort Int.> ns
	in
	    if isCol 
	    then if length ns <> 1 
		 then (print "forget AF condition?\n"; raise PoSolverError)
		 else "!" ^ (Int.toString (hd ns))
	    else (* non collapsing *)
		if (#useMul opt) andalso (#useLex opt)
		then
		    if isAssignedByTrue resultAr (lookupMulMap (mulMap,f))
  		    then "Mul [" ^ (PrintUtil.prSeq Int.toString ms) ^ "]"
		    else "Lex [" ^ (PrintUtil.prSeq Int.toString ns) ^ "]"
		else if (#useMul opt) 
		then "Mul [" ^ (PrintUtil.prSeq Int.toString ms) ^ "]"
		else 
		    "Lex [" ^  (PrintUtil.prSeq Int.toString ns) ^ "]"
	end

    (*** $B4X?t5-9f$NM%@h=g0L!$(BLex/Mul$BIU$-$N0z?tHf3S=g=x%j%9%H$N=PNO(B ***)
    fun printInfo resultAr (encoding:encoding_info) (opt:options) = 
	let
	    val faMap = #FaMap encoding
	    val prec = mkPrecList resultAr encoding
	    fun prWithArg f = let 
				  val ar = lookupFaMap (faMap, f)
				  
			      in
				  " " ^ (Fun.toString f) ^ " : " 
				  ^ (if #useAf opt
				     then prArgStatusAF resultAr encoding
							(opt:options) (f,ar)
				     else prArgStatus resultAr encoding
						      (opt:options) (f,ar))
			      end
	in
	    print "Precedence:\n";
	    print (PrintUtil.prSeqWith 
		       (fn xs => PrintUtil.prSeqWith 
				     prWithArg ", " xs)

		       ";\n" prec);
            print ";\n"
	end

   (* getInfo for without AF *)
    fun getInfo resultArray (encoding:encoding_info) (opt:options) = 
	let val fs = #Fs encoding
	    val faList = #FaList encoding
            (* $B4X?t5-9f$N=E$_(B *)
	    val prec = L.foldr (fn (f,fm) => FM.insert (fm,f,weightOfFun resultArray encoding f)) FM.empty fs
	    val mulMap = #MulMap encoding
	    (* list of functions with MUL status *)
	    val mul = if (#useMul opt) 
		      then L.filter (fn f => isAssignedByTrue resultArray (lookupMulMap (mulMap,f))) fs
		      else [] 
	    fun getlex (f,ar) = mkLexList resultArray encoding (f,ar)
	    val lex = if (#useLex opt) 
		      then L.map (fn (f,ar) =>  (f,getlex (f,ar))) faList
		      else []
	in (prec,mul,lex)
	end

    fun encodeOrderConstraint encodingType counterRef encoding 
			      (opt:options) rule =
	let 
	    val faSet = #FaSet encoding
	    val funSet = FIS.foldr (fn ((f,i),set)=> FS.add (set,f)) FS.empty faSet
	    fun prec (f,g) = lookupGtPrecTable encoding (f,g)
	    fun preceq (f,g) = lookupEqPrecTable encoding (f,g)
	    fun mul f = encodeMulStatus encoding f
	    fun lex (f,i,j) = lookupLexTable encoding (f,i,j)
	    fun same (fi,gj) = lookupSameNthTable encoding (fi,gj)
	    fun col f = encodeColStatus encoding f
	    fun pi (f,i) = encodePiStatus encoding (f,i)

	    val _ = debug (fn _ => let val (l,r) = rule
				   in print ("do encoding: " ^ (Term.toString l)
					     ^ (case encodingType of
						    Order.GT => " :(gt): "
						  | Order.GE => " :(ge): "
						  | Order.EQ => " :(eq): ")
					     ^ (Term.toString r) ^ "\n")
				   end)
	in case (#useAf opt,#useQuasi opt,#useLex opt,#useMul opt) of
	       (false,false,false,false)
	       => Order.lpoEncoding prec counterRef encodingType rule
	     | (false,true,false,false)
	       => Order.qlpoEncoding prec preceq counterRef encodingType rule
	     | (false,false,true,false)
	       => Order.rpoEncoding (FS.empty,funSet) prec mul
				    lex counterRef encodingType rule
	     | (false,false,false,true)
	       => Order.rpoEncoding (funSet,FS.empty) prec mul
				    lex counterRef encodingType rule
	     | (false,false,true,true)
	       => Order.rpoEncoding (FS.empty,FS.empty) prec mul
				    lex counterRef encodingType rule
	     | (false,true,true,false)
	       => Order.qrpoEncoding (FS.empty,funSet) prec preceq
				     mul lex same counterRef
				     encodingType rule
	     | (false,true,false,true)
	       => Order.qrpoEncoding (funSet,FS.empty) prec preceq
				     mul lex same counterRef
				     encodingType rule
	     | (false,true,true,true)
	       => Order.qrpoEncoding (FS.empty,FS.empty) prec preceq
				     mul lex same counterRef
 				     encodingType rule
 	     | (true,false,false,false)
	       => Order.afLpoEncoding  prec col pi counterRef
				       encodingType rule
	     | (true,true,false,false)
	       => Order.afQlpoEncoding prec preceq col pi counterRef
				       encodingType rule
	     | (true,false,true,false)
	       => Order.afLposEncoding prec lex col pi counterRef
				       encodingType rule
	     | (true,true,true,false)
	       => Order.afQlposEncoding prec preceq lex col pi counterRef
				       encodingType rule
	     |(true,false,false,true)
	      => Order.afMpoEncoding prec col pi  counterRef
				     encodingType rule

	     |(true,true,false,true)
	      => Order.afQmpoEncoding prec preceq col pi counterRef
				     encodingType rule
	     |(true,false,true,true)
	      => Order.afRpoEncoding prec mul lex col pi  counterRef
				     encodingType rule

	     |(true,true,true,true)
	      => Order.afQrpoEncoding prec preceq mul lex col pi  counterRef
				     encodingType rule

(*	     | _ => (print "Option not supported\n"; False) *)
	end


 (* certificates $B=PNOMQ$N%3!<%I(B *)


  fun outputArgumentFilteringEntry (colSet,lexListMap) (f,ar) () = 
      let fun outputArgs () = 
	      case FM.find (lexListMap, f) of 
		  NONE => (debug (fn _ => print ("outputArgumentFilteringEntry: (" ^ (Fun.toString f) ^ "\n"));
			   raise PoSolverError)
		| SOME ns =>
		  if FS.member (colSet, f)
		  then CU.encloseProofLeafBy "collapsing" (Int.toString ((hd ns) + 1))
		  else if null ns
		  then CU.encloseProofLeafBy "nonCollapsing" ""
		  else CU.encloseProofTreesBy "nonCollapsing" 
					      (L.map (fn i => fn () => 
							 CU.encloseProofLeafBy "position" (Int.toString (i+1)))
						     ns)
      in CU.encloseProofTreesBy "argumentFilterEntry" 
				[fn () => CU.encloseProofLeafBy "name" (Fun.toString f),
				 fn () => CU.encloseProofLeafBy "arity" (Int.toString ar),
				 outputArgs]
      end


  fun outputArgmentFiltering (colSet,lexListMap) faList () =
      CU.encloseProofTreesBy "argumentFilter" 
			     (L.map (outputArgumentFilteringEntry (colSet,lexListMap)) faList)


  (* argument filtering $B$r$7$?8e$N(B arity $B$r=q$/(B *) 
  fun outputPrecEntry0
	  (colSet,mulList,piMap) (f,w) =
      if FS.member (colSet,f)
      then NONE
      else let val ar = case FM.find (piMap, f) of 
			    NONE  => (debug (fn _ => print ("outputArgumentFilteringEntry0: (" 
							    ^ (Fun.toString f) ^ "\n"));
				      raise PoSolverError)
			  | SOME ns => L.length ns
	   in SOME (fn () => 
		       CU.encloseProofTreesBy "statusPrecedenceEntry"
					      [fn () => CU.encloseProofLeafBy "name" (Fun.toString f),
					       fn () => CU.encloseProofLeafBy "arity" (Int.toString ar),
					       fn () => CU.encloseProofLeafBy "precedence" (Int.toString w),
					       fn () => if LU.member' Fun.equal f mulList
							then CU.encloseProofLeafBy "mul" ""
							else CU.encloseProofLeafBy "lex" ""])
	   end


  fun outputPrecEntry (colSet,mulList,piMap) (xs,w) =
      L.mapPartial (fn f => outputPrecEntry0 (colSet,mulList,piMap) (f,w)) xs

  fun outputPrec (colSet,precList,mulList,piMap) () =
      CU.encloseProofTreeBy "pathOrder"
			    (fn _ => CU.encloseProofTreesBy "statusPrecedence"
							    (LU.mapAppend
								 (fn i => outputPrecEntry 
									      (colSet,mulList,piMap)
									      (L.nth (precList,i), (L.length precList) - i))
								 (L.tabulate (L.length precList, fn i=>i))))


  fun outputDecrease (colSet,precList,mulList,piMap) () =
      CU.encloseProofTreeBy "strictDecrease"
			    (fn _ => CU.encloseProofTreeBy "orderingConstraintProof"
							   (fn _ => CU.encloseProofTreeBy "redPair" 
											  (outputPrec (colSet,precList,mulList,piMap))))
			    
  fun outputDisproofDecrease (colSet,precList,mulList,piMap) () =
      CU.encloseProofTreeBy "usableRulesNonJoin" (outputDecrease (colSet,precList,mulList,piMap))
			    

  (* $B7PO)=g=x$K4p$E$/(B (>,\simeq) $B$N=<B-2DG=@-H=Dj4o(B *)
  fun poSolver minisatPath tmpDir (opt:options) (rs,eqs) = 
      let val _ = debug (fn _ => print (Trs.prRules rs))
	  val _ = debug (fn _ => print (Trs.prEqs eqs))
	  val symCount = ref 0
	  val faMap  = Trs.funArityMapInRules (eqs @ rs)
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val strictPrecCond = mkStrictPrecCond opt encoding 
	  val lexCond = mkLexCond opt encoding 
	  val quasiRpoCond = mkQuasiRpoCond opt encoding 
	  val afCond = mkAfCond opt encoding 
	  val condProp = simplifyProp (Conj [strictPrecCond, lexCond, quasiRpoCond, afCond])
 	  val gtProps = L.map (simplifyProp 
			       o (encodeOrderConstraint Order.GT symCount encoding opt))
			      rs
 	  val eqProps = L.map (simplifyProp 
			       o (encodeOrderConstraint Order.EQ symCount encoding opt))
			      eqs
   	  
	  val prop = Prop.Conj [ condProp,
				 Prop.Conj gtProps,
				 Prop.Conj eqProps]
	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)
	  val _ = debug (fn _ => if result
				 then printInfo resultArray encoding opt
				 else ())
	  (* val prec =  L.foldr (fn (f,fm) => FM.insert (fm,f,weightOfFun resultArray encoding f)) FM.empty fs *)
      in
	  result
      end


  (* $B7PO)=g=x$K4p$E$/(B (>,\simeq) $B$N=<B-2DG=@-H=Dj4o(B *)
  (* GCR  $BMQ$K(B prec $B$rJV$9$h$&$K$7$?$@$1(B  *)
  fun poSolverForGcrWithCpf  minisatPath tmpDir (opt:options) (rs,eqs) = 
      let val _ = debug (fn _ => print (Trs.prRules rs))
	  val _ = debug (fn _ => print (Trs.prEqs eqs))
	  val symCount = ref 0
	  val faMap  = Trs.funArityMapInRules (eqs @ rs)
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val strictPrecCond = mkStrictPrecCond opt encoding 
	  val lexCond = mkLexCond opt encoding 
	  val quasiRpoCond = mkQuasiRpoCond opt encoding 
	  val afCond = mkAfCond opt encoding 
	  val condProp = simplifyProp (Conj [strictPrecCond, lexCond, quasiRpoCond, afCond])
 	  val gtProps = L.map (simplifyProp 
			       o (encodeOrderConstraint Order.GT symCount encoding opt))
			      rs
 	  val eqProps = L.map (simplifyProp 
			       o (encodeOrderConstraint Order.EQ symCount encoding opt))
			      eqs
   	  
	  val prop = Prop.Conj [ condProp,
				 Prop.Conj gtProps,
				 Prop.Conj eqProps]
	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)
	  val _ = debug (fn _ => if result
				 then printInfo resultArray encoding opt
				 else ())

	  fun cpf () = if result 
		       andalso (!runCertification)
		       then let val colSet = getColSet resultArray encoding 
				val precList = mkPrecList resultArray encoding
				val mulList = getMulList resultArray encoding 
				val lexListMap = getLexListMapAF resultArray encoding
				val piMap = getPiMap resultArray encoding
			    in outputDecrease (colSet,precList,mulList,piMap) ()
			    end
		       else ""

      in
	 if result 
	 then SOME (getInfo resultArray encoding opt, cpf)
	 else NONE
      end

 (* drop cpf part *)
  fun poSolverForGcr  minisatPath tmpDir (opt:options) (rs,eqs) = 
      case poSolverForGcrWithCpf  minisatPath tmpDir (opt:options) (rs,eqs) of
	  SOME (ans,cpf) => SOME ans
       |  NONE => NONE


  (* $B7PO)=g=x$K4p$E$/(B (>,\simeq) $B$N=<B-2DG=@-H=Dj4o(B *)
  (* Conj_i (l_i > r_{i,1} or ...or l_i > r_{i,n}) $B$r%A%'%C%/(B *)
  fun poSolverForGcrMulti  minisatPath tmpDir (opt:options) rss = 
      let (* val _ = debug (fn _ => print (Trs.prRules rs)) *)
	  val symCount = ref 0
	  val faMap  = Trs.funArityMapInRules 
			   (LU.mapAppend (fn (l,rs) => L.map (fn r=>(l,r)) rs) rss)
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val strictPrecCond = mkStrictPrecCond opt encoding 
	  val lexCond = mkLexCond opt encoding 
	  val quasiRpoCond = mkQuasiRpoCond opt encoding 
	  val afCond = mkAfCond opt encoding 
	  val condProp = simplifyProp (Conj [strictPrecCond, lexCond, quasiRpoCond, afCond])
 	  val gtProp = simplifyProp (
	      Prop.Conj (L.map 
			     (fn (l,rs)=>
				 Prop.Disj (L.map (fn r => simplifyProp (encodeOrderConstraint Order.GT symCount encoding opt (l,r))) rs))
			     rss)
	      )
	  val prop = Prop.Conj [ condProp, gtProp ]
	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)
	  val _ = debug (fn _ => if result
				 then printInfo resultArray encoding opt
				 else ())


	  val qprec = if result 
		  then L.foldr (fn (f,fm) => FM.insert (fm,f,weightOfFun resultArray encoding f)) FM.empty fs
		  else FM.empty

	  val rules = if result
		      then L.map (fn (l,rs)
					=> (l, valOf (L.find (fn r => Order.qmpoCheck (Order.compareQprec qprec) Order.GT (l,r)) rs)))
				    rss
		      else []
      in
	 if result 
	 then SOME (qprec,rules)
	 else NONE
      end





  (* $B7PO)=g=x$K4p$E$/(B (>,\simeq) $B$N=<B-2DG=@-H=Dj4o(B *)
  (* Conj_i (l_i > r_{i,1} or ...or l_i > r_{i,n}) $B$r%A%'%C%/(B *)
  fun poSolverForGcrMulti2  minisatPath tmpDir (opt:options) wPat Crules rsss = 
      let (* val _ = debug (fn _ => print (Trs.prRules rs)) *)
	  val symCount = ref 0
	  (* val faMap  = Trs.funArityMapInRules (LU.mapAppend (fn rss => LU.mapAppend (fn rs => rs) rss) rsss) *)
	  val faMap =  Term.funArityMapInTerms wPat
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  (* val _ = println (LU.toStringCommaCurly Fun.toString fs) *)
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val strictPrecCond = mkStrictPrecCond opt encoding 
	  val lexCond = mkLexCond opt encoding 
	  val quasiRpoCond = mkQuasiRpoCond opt encoding 
	  val afCond = mkAfCond opt encoding 
	  val condProp = simplifyProp (Conj [strictPrecCond, lexCond, quasiRpoCond, afCond])
 	  val gtProp = simplifyProp (
	      Prop.Conj (L.map 
			     (fn rss =>
				 Prop.Disj (L.map (fn rs => 
						      Prop.Conj 
							  (L.map (fn (l,r) => encodeOrderConstraint Order.GT symCount encoding opt (l,r))
								 rs))
						  rss))
			     rsss)
	      )
 	  val geProp = if null Crules
		       then Prop.True
		       else simplifyProp (Prop.Disj (* one of the crules suffices to be orientable, as others can be conjecture *)
					 (L.map (fn (l,r) => encodeOrderConstraint Order.GE symCount encoding opt (l,r))
						Crules))
           (*** for test
	      simplifyProp (Prop.Conj
				(L.map (fn (l,r) => encodeOrderConstraint Order.GT symCount encoding opt (l,r))
				       Crules))
 ***)


	  val prop = Prop.Conj [ condProp, gtProp, geProp ]

	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)


         (*
	  val qprec = if result 
		  then L.foldr (fn (f,fm) => FM.insert (fm,f,weightOfFun resultArray encoding f)) FM.empty fs
		  else FM.empty
         *)

      in
	  if result 
	  then let val _ = print "order successfully found\n"
		   val (qprec,mul,lexList) = getInfo resultArray encoding opt
		   fun isMul f = if #useMul opt
				 then LU.member' Fun.equal f mul
				 else false
		   fun lexFun f xs = if not (#useLex opt)
				     then xs
				     else case  L.find (fn (g,_) => Fun.equal (f,g)) lexList of
					      NONE => (print ("Could not find the lex status: " ^ (Fun.toString f) ^"\n");
						       raise PoSolverError)
					    | SOME (_,lex) => 
					      let val weighted = LP.zip (lex,xs)
						  (* val _ = print (LU.toStringCommaCurly Int.toString lex) *)
						  val ordered = ListMergeSort.sort (fn ((i,_),(j,_)) => Int.< (i,j)) weighted
						  val ans =  L.map (fn (_,t) => t) ordered
						  (* val _ = print (LU.toStringCommaCurly Term.toString ans) *)
					      in ans
					      end
						  
                   (* val _ = PrintUtil.println (LU.toStringCommaSquare  *)
		   (* 				  (fn (f,n) => "(" ^ (Fun.toString f) ^ "," ^ (Int.toString n) ^ ")") *)
		   (* 				  (FunMap.listItemsi qprec)) *)
		   val _ = printInfo resultArray encoding opt 

		   (* val _  =  L.app (fn (f,ar) =>  *)
		   (* 		       if isMul f *)
		   (* 		       then (print ((Fun.toString f) ^ ": mul\n")) *)
		   (* 		       else  let val _ = print ((Fun.toString f) ^ ": lex: (") *)
		   (* 				 val args = lexFun f (L.tabulate (ar, fn i => IOFotrs.rdTerm ("a" ^ (Int.toString i)))) *)
		   (* 				 val _ = print (LU.toStringComma Term.toString args) *)
		   (* 				 val _ = print ")\n" *)
		   (* 			     in () *)
		   (* 			     end) *)
		   (* 		   faList *)

		   (* val _ = L.app (fn rss => *)
		   (* 		     L.app (fn rs => (print "----\n"; *)
		   (* 			       L.app (fn (l,r) => let val _ = print (Trs.prRule (l,r)) *)
		   (* 						  in if Order.qrpoCheck (Order.compareQprec qprec) isMul lexFun Order.GT (l,r) *)
		   (* 						     then  print ": true\n" *)
		   (* 						     else  print ": no\n" *)
		   (* 						  end)) rs) *)
		   (* 			   rss) *)
		   (* 		   rsss *)

		   val rules = LU.mapAppend
		   		   (fn rss => valOf (L.find (fn rs
		   						=> (L.all (fn (l,r) =>
		   		      Order.qrpoCheck (Order.compareQprec qprec) isMul lexFun Order.GT (l,r)) rs))
  		   		 rss))
		   		   rsss

	      in SOME (qprec,isMul,lexFun, rules)
	      end
	 else NONE
      end

(**
      in
	 if result 
	 then SOME (qprec,rules)
	 else NONE
      end
*)





  (* $B7PO)=g=x$K4p$E$/(B (>,\gesim) $B$N=<B-2DG=@-H=Dj4o(B for non-joinability, 
    Assume rs0 U (rs0)^{-1} U rs1 is TRSs *)
  (* Check 1) term0 > term1 
           2) term0 -*->rs0  s0 implies term0 <= s0
           3) term1 -*->rs1  s1 implies s1 <= term1
     !!! Note that index is not corresponding for non-commutativity check  *)
  fun poSolverForNj minisatPath tmpDir (opt:options) (term0,term1) (rs0,rs1) = 
      let (* val _ = print "Prove non-joinability of " *)
	  (*  val _ = print ((Term.toString term0) ^ " and " ^ (Term.toString term1) ^ "\n") *)
	  val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	  val rs = rs0 @ rs1
	  val _ = debug (fn _ => print "(Current) usable rules:\n")
	  val _ = debug (fn _ => print (Trs.prRules rs0))
	  val _ = debug (fn _ => print (Trs.prRules rs1))

	  val symCount = ref 0

	  val faMap  = Trs.funArityMapInRules ((term0,term1)::rs)
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val strictPrecCond = mkStrictPrecCond opt encoding
	  val lexCond = mkLexCond opt encoding 
	  val quasiRpoCond = mkQuasiRpoCond opt encoding 
	  val afCond = mkAfCond opt encoding 

	   val count0 = !symCount + 1
	   val len0 = L.length rs0
	   val rsIdxs0 = LP.zip (rs0, L.tabulate (len0,fn x => count0 + x))

	   val count1 = !symCount + len0
	   val len1 = L.length rs1
	   val rsIdxs1 = LP.zip (rs1, L.tabulate (len1,fn x => count1 + x))

 	   val _ = symCount := count1 + len1
           
	   val dSyms = LU.eliminateDuplication' Fun.equal (L.map (fn (l,r) => valOf (Term.funRootOfTerm l)) rs)

	   val dsymsCountMap = L.foldr
				   (fn (f,fmap) => FM.insert(fmap,f,(symCount := (!symCount) + 1;!symCount)))
				   FM.empty 
				   dSyms
			      
	  fun mkUsableRulesProp term rsIdxs =
	      if #useAf opt
	      then
	      let open Prop
		  open Term
		  fun useFunArg (f,i) = encodePiStatus encoding (f,i+1)
		  fun isFunCol f = encodeColStatus encoding f
	

		  fun usableTerm (Var _)  = True
		    | usableTerm (Fun (f,ts,_)) =
		      Conj ((case FM.find (dsymsCountMap,f) of
				 SOME i => Atom i
			       | NONE => True)
			    :: (L.tabulate (length ts, 
					 fn i => Imp (useFunArg (f,i), usableTerm (L.nth (ts,i))))))

		   val usableRule = 
       Conj (L.mapPartial 
		 (fn ((l,r),i) => 
		     let val (Fun (f,ts,_)) = l
		     in case FM.find (dsymsCountMap,f) of
			    SOME j => SOME 
					  (Conj[
					   Imp (Atom j, Conj [Atom i, usableTerm r]),
					   Imp (isFunCol f, 
						Conj (L.map (fn k => Imp (Conj[useFunArg (f,k),
									       (if Term.isVar (L.nth (ts,k))
										then True
										else False)],
									  Conj [Atom i, usableTerm r]))
							    (L.tabulate (L.length ts, fn k=>k))))])
    			  | NONE => NONE
		     end)
		 rsIdxs)
              (* bug fix 2015/01/15 *)
	      in Prop.simplifyProp (Prop.Conj [usableTerm term, usableRule])
	      end		      
	      else True	      

	  val uprop0 = mkUsableRulesProp term0 rsIdxs0
	  val uprop1 = mkUsableRulesProp term1 rsIdxs1

	  val condProp = simplifyProp (Conj [strictPrecCond, lexCond, quasiRpoCond, afCond,uprop0,uprop1])

 	  val gtProp = encodeOrderConstraint Order.GT symCount encoding opt (term0,term1)

	  val leProp0 = Conj (L.map (fn ((l,r),i) => 
					let val p = encodeOrderConstraint Order.GE symCount encoding opt (r,l)
									                   (*** note (r,l) ***)
					in if #useAf opt then Imp (Atom i,p) else p
					end)   
				    rsIdxs0)

	  val geProp1 = Conj (L.map (fn ((l,r),i) => 
					let val p = encodeOrderConstraint Order.GE symCount encoding opt (l,r)
					in  if #useAf opt then Imp (Atom i,p) else p
					end)
				    rsIdxs1)

	  val prop = Prop.Conj [condProp, gtProp,leProp0,geProp1]
	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)
	  val _ = debug (fn _ => if result
		  then (print " (success)\n"; printInfo resultArray encoding opt)
		  else (print " (failure)\n"))

	  local  open Term
	  in
	  fun applyAfToTerm piMap colSet (t as (Var _)) = t
	    | applyAfToTerm piMap colSet (t as (Fun (f,ts,ty))) =
	      case FM.find (piMap, f) of 
		  SOME ns => if FS.member (colSet, f)
			     then applyAfToTerm piMap colSet (L.nth (ts, hd ns))
			     else Fun (f,
				       L.map (fn i => applyAfToTerm piMap colSet (L.nth (ts,i))) ns,
				       ty)
		| _=> (debug (fn _ => print ("applyAfToTerm: (" ^ (Term.toString t) ^ "\n"));
		       raise PoSolverError)
	  end

	  val _ = debug (fn _ =>    if result
		  then 
		      let val _ = print "Obtained:\n"
			  val colSet = getColSet resultArray encoding 
			  val piMap = getPiMap resultArray encoding 
			  fun af t  = applyAfToTerm piMap colSet t
			  val _ = print (" " ^ (Term.toString (af term0)) ^ " :(gt): " 
					 ^ (Term.toString (af term1)) ^ "\n" )
			  val _ = L.app (fn ((l,r),i) => 
					    if isAssignedByTrue resultArray i
					    then print (" " ^ (Term.toString (af l)) ^ " :(le): " 
						 ^ (Term.toString (af r)) ^ "\n")
					    else print (" (" ^ (Term.toString l) ^ " -> " 
							^ (Term.toString r) ^ ": not usable)\n"))
					rsIdxs0
			  val _ = L.app (fn ((l,r),i) => 
					    if isAssignedByTrue resultArray i
					    then print (" " ^ (Term.toString (af l)) ^ " :(ge): " 
							^ (Term.toString (af r)) ^ "\n")
					    else print (" (" ^ (Term.toString l) ^ " -> " 
						 ^ (Term.toString r) ^ ": not usable)\n"))
					rsIdxs1
		      in ()
		      end
		  else ())

	 (*  fun outputDisproofAfEntry (colSet,lexListMap) (f,ar) () =  *)
	 (*      let fun outputArgs () =  *)
	 (* 	      case FM.find (lexListMap, f) of  *)
	 (* 		  NONE => (debug (fn _ => print ("outputDisproofAfEntry: (" ^ (Fun.toString f) ^ "\n")); *)
	 (* 			   raise PoSolverError) *)
	 (* 		| SOME ns => *)
	 (* 		  if FS.member (colSet, f) *)
	 (* 		  then CU.encloseProofLeafBy "collapsing" (Int.toString ((hd ns) + 1)) *)
	 (* 		  else if null ns *)
	 (* 		  then CU.encloseProofLeafBy "nonCollapsing" "" *)
	 (* 		  else CU.encloseProofTreesBy "nonCollapsing"  *)
	 (* 		       (L.map (fn i => fn () =>  *)
	 (* 					  CU.encloseProofLeafBy "position" (Int.toString (i+1))) *)
	 (* 			      ns) *)
	 (*      in CU.encloseProofTreesBy "argumentFilterEntry"  *)
	 (* 				[fn () => CU.encloseProofLeafBy "name" (Fun.toString f), *)
	 (* 				 fn () => CU.encloseProofLeafBy "arity" (Int.toString ar), *)
	 (* 				 outputArgs] *)
	 (*      end *)

	 (*  fun outputDisproofAf (colSet,lexListMap) faList () = *)
	 (*      CU.encloseProofTreesBy "argumentFilter"  *)
	 (* 			     (L.map (outputDisproofAfEntry (colSet,lexListMap)) faList) *)


         (* (* argument filtering $B$r$7$?8e$N(B arity $B$r=q$/(B *)  *)
	 (*  fun outputDisproofPrecEntry0 *)
	 (* 	  (colSet,mulList,piMap) (f,w) = *)
	 (*      if FS.member (colSet,f) *)
	 (*      then NONE *)
	 (*      else let val ar = case FM.find (piMap, f) of  *)
	 (* 			    NONE  => (debug (fn _ => print ("outputDisproofAfEntry0: ("  *)
	 (* 							    ^ (Fun.toString f) ^ "\n")); *)
	 (* 				      raise PoSolverError) *)
	 (* 			  | SOME ns => L.length ns *)
	 (* 	   in SOME (fn () =>  *)
	 (* 		       CU.encloseProofTreesBy "statusPrecedenceEntry" *)
	 (* 					      [fn () => CU.encloseProofLeafBy "name" (Fun.toString f), *)
	 (* 					       fn () => CU.encloseProofLeafBy "arity" (Int.toString ar), *)
	 (* 					       fn () => CU.encloseProofLeafBy "precedence" (Int.toString w), *)
	 (* 					       fn () => if LU.member' Fun.equal f mulList *)
	 (* 							then CU.encloseProofLeafBy "mul" "" *)
	 (* 							else CU.encloseProofLeafBy "lex" ""]) *)
	 (* 	   end *)


	 (*  fun outputDisproofPrecEntry (colSet,mulList,piMap) (xs,w) = *)
	 (*      L.mapPartial (fn f => outputDisproofPrecEntry0 (colSet,mulList,piMap) (f,w)) xs *)

	 (*  fun outputDisproofPrec (colSet,precList,mulList,piMap) () = *)
	 (*      CU.encloseProofTreeBy "pathOrder" *)
	 (*      (fn _ => CU.encloseProofTreesBy "statusPrecedence" *)
	 (* 				      (LU.mapAppend *)
	 (* 					   (fn i => outputDisproofPrecEntry  *)
	 (* 							(colSet,mulList,piMap) *)
	 (* 							(L.nth (precList,i), (L.length precList) - i)) *)
	 (* 					   (L.tabulate (L.length precList, fn i=>i)))) *)

	 (*  fun outputDisproofDecrease (colSet,precList,mulList,piMap) () = *)
	 (*      CU.encloseProofTreeBy "usableRulesNonJoin" *)
	 (*      (fn _ => CU.encloseProofTreeBy "strictDecrease" *)
	 (* 	       (fn _ => CU.encloseProofTreeBy "orderingConstraintProof" *)
	 (* 			(fn _ => CU.encloseProofTreeBy "redPair"  *)
	 (* 						       (outputDisproofPrec (colSet,precList,mulList,piMap))))) *)
	      
	  fun cpf () = if result 
		       andalso (!runCertification)
		       then let val colSet = getColSet resultArray encoding 
				val precList = mkPrecList resultArray encoding
				val mulList = getMulList resultArray encoding 
				val lexListMap = getLexListMapAF resultArray encoding
				val piMap = getPiMap resultArray encoding
			    in CU.encloseProofTreesBy "argumentFilterNonJoin"
						      [ outputArgmentFiltering (colSet,lexListMap) faList,
							outputDisproofDecrease (colSet,precList,mulList,piMap) ]
			    end
		       else ""

      in
	  (result,cpf)
      end

  (* $B<-=q<07PO)=g=x$K4p$E$/(B (gesim,not gesim) $B$N=<B-2DG=@-H=Dj4o(B for non-joinability, 
    Assume rs0 U (rs0)^{-1} U rs1 is TRSs 
    multiset status $B$N%3!<%F%#%s%0$O!$(Bnot gesim $B$KBP1~$7$F$$$J$$$N$G!$(BMPO$B$OMxMQIT2D(B
   *)
  fun poSolverForNjwithNgeTotal minisatPath tmpDir (opt:options) (term0,term1) (rs0,rs1) = 
      let (* val _ = print "Prove non-joinability of " *)
	  (*  val _ = print ((Term.toString term0) ^ " and " ^ (Term.toString term1) ^ "\n") *)
	  val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	  val rs = rs0 @ rs1
	  val _ = debug (fn _ => print "(Current) usable rules:\n")
	  val _ = debug (fn _ => print (Trs.prRules rs0))
	  val _ = debug (fn _ => print (Trs.prRules rs1))

	  val symCount = ref 0

	  (* encoding $B$K;H$&Dj?t(B *)
	  val faMap  = Trs.funArityMapInRules ((term0,term1)::rs)
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val strictPrecCond = mkStrictPrecCond opt encoding
	  val lexCond = mkLexCond opt encoding 
	  val quasiRpoCond = mkQuasiRpoCond opt encoding 
	  val afCond = mkAfCond opt encoding 

	  (* encoding $B$K;H$&%+%&%s%?(B *)
	   val count0 = !symCount + 1
	   val len0 = L.length rs0
	   val rsIdxs0 = LP.zip (rs0, L.tabulate (len0,fn x => count0 + x))

	   val count1 = !symCount + len0
	   val len1 = L.length rs1
	   val rsIdxs1 = LP.zip (rs1, L.tabulate (len1,fn x => count1 + x))

 	   val _ = symCount := count1 + len1
	   val dSyms = LU.eliminateDuplication' Fun.equal (L.map (fn (l,r) => valOf (Term.funRootOfTerm l)) rs)
	   val dsymsCountMap = L.foldr
				   (fn (f,fmap) => (debug (fn _ => print ("Usable DFun " ^ (Fun.toString f) 
								   ^ " : " ^ (Int.toString ((!symCount) + 1)) ^ "\n"));
						    FM.insert(fmap,f,(symCount := (!symCount) + 1;!symCount))))
				   FM.empty 
				   dSyms
			      
	  (*  usable $B%k!<%k$N(B encoding function *)
	  fun mkUsableRulesProp term rsIdxs =
	      if #useAf opt
	      then
	      let open Prop
		  open Term
		  fun useFunArg (f,i) = encodePiStatus encoding (f,i+1)
		  fun isFunCol f = encodeColStatus encoding f
		  fun usableTerm (Var _)  = True
		    | usableTerm (Fun (f,ts,_)) =
		      Conj ((case FM.find (dsymsCountMap,f) of
				 SOME i => Atom i
			       | NONE => True)
			    :: (L.tabulate (length ts, fn i => Imp (useFunArg (f,i), usableTerm (L.nth (ts,i))))))
		  val usableRule = 
		      Conj (L.mapPartial 
				(fn ((l,r),i) => 
				    let val (Fun (f,ts,_)) = l
				    in case FM.find (dsymsCountMap,f) of
					   SOME j => SOME 
							 (Conj[
							       Imp (Atom j, Conj [Atom i, usableTerm r]),
							       Imp (isFunCol f,  (* $B$3$32?$r$d$C$F$$$k$s$@$C$1(B ? *)
								    Conj (L.map (fn k => Imp (Conj[useFunArg (f,k),
												   (if Term.isVar (L.nth (ts,k))
												    then True
												    else False)],
											      Conj [Atom i, usableTerm r]))
										(L.tabulate (L.length ts, fn k=>k))))])
    					 | NONE => NONE
				    end)
				rsIdxs)

		   fun isUsedVarInTerm (Var (x,_) ) y = if Var.equal (x,y) then True else False
		     | isUsedVarInTerm (Fun (f,ts,_)) y = 
		       Disj (L.mapPartial
				 (fn i => let val ti = L.nth (ts,i)
					  in if LU.member' Var.equal y (Term.varListInTerm ti)
					     then SOME (Conj [useFunArg (f,i), isUsedVarInTerm ti y])
					     else NONE
					  end)
				 (L.tabulate (L.length ts, fn i => i)))
		   
		   val varCond = Conj (L.map (fn ((l,r),_) => 
						 Conj (L.map (fn y => Imp (isUsedVarInTerm r y, isUsedVarInTerm l y)) 
							     (LU.eliminateDuplication' Var.equal 
										       (Term.varListInTerm r))))
					     rsIdxs)


	      in Prop.simplifyProp (Prop.Conj [usableTerm term, usableRule, varCond])
	      end		      
	      else True

	  (*  usable $B%k!<%k$N(B encoding  *)
	  val uprop0 = mkUsableRulesProp term0 rsIdxs0
	  val uprop1 = mkUsableRulesProp term1 rsIdxs1

	  (*  $B$3$l$^$G$N=`Hw$r$^$H$a$?(B proposition  *)
	  val condProp = simplifyProp (Conj [strictPrecCond,lexCond,quasiRpoCond,afCond,uprop0,uprop1])

 	  val ngeProp = let val ps = encodeOrderConstraint Order.GE symCount encoding opt (term1,term0)
			in case ps of Conj (p0::defs) => Conj ((Neg p0):: defs)
			end
	  val _ = debug (fn _ => println (Prop.printProp ngeProp))

	  val leProp0 = Conj (L.map (fn ((l,r),i) => 
					let val p = encodeOrderConstraint Order.GE symCount encoding opt (r,l)
					in if #useAf opt then Imp (Atom i,p) else p
					end)   
				    rsIdxs0)

	  val geProp1 = Conj (L.map (fn ((l,r),i) => 
					let val p = encodeOrderConstraint Order.GE symCount encoding opt (l,r)
					in  if #useAf opt then Imp (Atom i,p) else p
					end)
				    rsIdxs1)

	  val prop = Prop.Conj [condProp, ngeProp,leProp0,geProp1]
	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)

	  val _ = Solver.propSolver2 "../tools/bin/yices" tmpDir (prop,!symCount) 

	  val _ = debug (fn _ => if result
		  then (print " (success)\n"; printInfo resultArray encoding opt)
		  else (print " (failure)\n"))

	  local  open Term
	  in
	  fun applyAfToTerm piMap colSet (t as (Var _)) = t
	    | applyAfToTerm piMap colSet (t as (Fun (f,ts,ty))) =
	      case FM.find (piMap, f) of 
		  SOME ns => if FS.member (colSet, f)
			     then applyAfToTerm piMap colSet (L.nth (ts, hd ns))
			     else Fun (f,
				       L.map (fn i => applyAfToTerm piMap colSet (L.nth (ts,i))) ns,
				       ty)
		| _=> (debug (fn _ => print ("applyAfToTerm: (" ^ (Term.toString t) ^ "\n"));
		       raise PoSolverError)
	  end

	  val _ = if result
		  then 
		      let val _ = print "Obtained:\n"
			  val colSet = getColSet resultArray encoding 
			  val piMap = getPiMap resultArray encoding 
			  fun af t  = if (#useAf opt)
			      	      then applyAfToTerm piMap colSet t
				      else t
		  	  val _ = print (" " ^ (Term.toString (af term1)) ^ " :(not ge): " 
					 ^ (Term.toString (af term0)) ^ "\n" )

			  val _ = L.app (fn ((l,r),i) => 
					    if isAssignedByTrue resultArray i orelse (not (#useAf opt))
					    then print (" " ^ (Term.toString (af l)) ^ " :(le): " 
						 ^ (Term.toString (af r)) ^ "\n")
					    else print (" (" ^ (Term.toString l) ^ " -> " 
							^ (Term.toString r) ^ ": not usable)\n"))
					rsIdxs0
			  val _ = L.app (fn ((l,r),i) =>  
					    if isAssignedByTrue resultArray i
					    then print (" " ^ (Term.toString (af l)) ^ " :(ge): " 
							^ (Term.toString (af r)) ^ "\n")
					    else print (" (" ^ (Term.toString l) ^ " -> " 
						 ^ (Term.toString r) ^ ": not usable)\n"))
				         rsIdxs1
		      in ()
		      end
		  else ()

	 (*  fun outputDisproofAfEntry (colSet,lexListMap) (f,ar) () =  *)
	 (*      let fun outputArgs () =  *)
	 (* 	      case FM.find (lexListMap, f) of  *)
	 (* 		  NONE => (debug (fn _ => print ("outputDisproofAfEntry: (" ^ (Fun.toString f) ^ "\n")); *)
	 (* 			   raise PoSolverError) *)
	 (* 		| SOME ns => *)
	 (* 		  if FS.member (colSet, f) *)
	 (* 		  then CU.encloseProofLeafBy "collapsing" (Int.toString ((hd ns) + 1)) *)
	 (* 		  else if null ns *)
	 (* 		  then CU.encloseProofLeafBy "nonCollapsing" "" *)
	 (* 		  else CU.encloseProofTreesBy "nonCollapsing"  *)
	 (* 		       (L.map (fn i => fn () =>  *)
	 (* 					  CU.encloseProofLeafBy "position" (Int.toString (i+1))) *)
	 (* 			      ns) *)
	 (*      in CU.encloseProofTreesBy "argumentFilterEntry"  *)
	 (* 				[fn () => CU.encloseProofLeafBy "name" (Fun.toString f), *)
	 (* 				 fn () => CU.encloseProofLeafBy "arity" (Int.toString ar), *)
	 (* 				 outputArgs] *)
	 (*      end *)

	 (*  fun outputDisproofAf (colSet,lexListMap) faList () = *)
	 (*      CU.encloseProofTreesBy "argumentFilter"  *)
	 (* 			     (L.map (outputDisproofAfEntry (colSet,lexListMap)) faList) *)


         (* (* argument filtering $B$r$7$?8e$N(B arity $B$r=q$/(B *)  *)
	 (*  fun outputDisproofPrecEntry0 *)
	 (* 	  (colSet,mulList,piMap) (f,w) = *)
	 (*      if FS.member (colSet,f) *)
	 (*      then NONE *)
	 (*      else let val ar = case FM.find (piMap, f) of  *)
	 (* 			    NONE  => (debug (fn _ => print ("outputDisproofAfEntry0: ("  *)
	 (* 							    ^ (Fun.toString f) ^ "\n")); *)
	 (* 				      raise PoSolverError) *)
	 (* 			  | SOME ns => L.length ns *)
	 (* 	   in SOME (fn () =>  *)
	 (* 		       CU.encloseProofTreesBy "statusPrecedenceEntry" *)
	 (* 					      [fn () => CU.encloseProofLeafBy "name" (Fun.toString f), *)
	 (* 					       fn () => CU.encloseProofLeafBy "arity" (Int.toString ar), *)
	 (* 					       fn () => CU.encloseProofLeafBy "precedence" (Int.toString w), *)
	 (* 					       fn () => if LU.member' Fun.equal f mulList *)
	 (* 							then CU.encloseProofLeafBy "mul" "" *)
	 (* 							else CU.encloseProofLeafBy "lex" ""]) *)
	 (* 	   end *)


	 (*  fun outputDisproofPrecEntry (colSet,mulList,piMap) (xs,w) = *)
	 (*      L.mapPartial (fn f => outputDisproofPrecEntry0 (colSet,mulList,piMap) (f,w)) xs *)

	 (*  fun outputDisproofPrec (colSet,precList,mulList,piMap) () = *)
	 (*      CU.encloseProofTreeBy "pathOrder" *)
	 (*      (fn _ => CU.encloseProofTreesBy "statusPrecedence" *)
	 (* 				      (LU.mapAppend *)
	 (* 					   (fn i => outputDisproofPrecEntry  *)
	 (* 							(colSet,mulList,piMap) *)
	 (* 							(L.nth (precList,i), (L.length precList) - i)) *)
	 (* 					   (L.tabulate (L.length precList, fn i=>i)))) *)

	 (*  fun outputDisproofDecrease (colSet,precList,mulList,piMap) () = *)
	 (*      CU.encloseProofTreeBy "usableRulesNonJoin" *)
	 (*      (fn _ => CU.encloseProofTreeBy "strictDecrease" *)
	 (* 	       (fn _ => CU.encloseProofTreeBy "orderingConstraintProof" *)
	 (* 			(fn _ => CU.encloseProofTreeBy "redPair"  *)
	 (* 						       (outputDisproofPrec (colSet,precList,mulList,piMap))))) *)
	      
	  fun cpf () = if result 
		       andalso (!runCertification)
		       then let val colSet = getColSet resultArray encoding 
				val precList = mkPrecList resultArray encoding
				val mulList = getMulList resultArray encoding 
				val lexListMap = getLexListMapAF resultArray encoding
				val piMap = getPiMap resultArray encoding
			    in CU.encloseProofTreesBy "argumentFilterNonJoin"
						      [ outputArgmentFiltering (colSet,lexListMap) faList,
							outputDisproofDecrease (colSet,precList,mulList,piMap) ]
			    end
		       else ""

      in
	  (result,cpf)
      end



    (************************************************************************)
    (* non-joinability check $B$N$?$a$N(B partial precedence $BMQ$N%(%s%3!<%G%#%s%0(B *)
    (************************************************************************)

    (*****************************)
    (*** ENCODING SECTION      ***)
    (*** FOR SYMBOLIC ENCODING ***)
    (*****************************)

    (********************)
    (** Symbolic Table **)
    (********************)
    (* --+---------------------   *)
    (*      f      g      h       *)
    (* --+---------------------   *)
    (* f |  -     (f,g)   (f,h)   *)
    (* g | (g,f)    -     (g,h)   *)
    (* h | (h,f)  (h,g)     -     *)
    (* --+---------------------   *)
    (** Constrast to Codish Encoding, precedence needs not to be total. **)
    (** f >= g iff (f,g) = true                                         **)
    (** f ~= g iff (f,g) = (g,f) = true                                 **)
    (** f > g iff  (f,g) = true and (g,f) = false                       **)
    (* --+---------------------   *)
    (*      f      g      h       *)
    (* --+---------------------   *)
    (* f |  -     init   init+1   *)
    (* g | init+2  -     init+3   *)
    (* h | init+4 init+5  -       *)
    (* --+---------------------   *)

    fun mkPrecMap2 counterRef fs = 
	let val products0 = ListXProd.mapX (fn (f,g) => (f,g)) (fs,fs)
	    val products = L.filter (not o Fun.equal) products0
	in L.foldr  (fn ((f,g),en) => (debug (fn _ => (print ("(" ^ (Fun.toString f)  ^ " >= " ^ (Fun.toString g)  ^ ") : " 
							      ^ (Int.toString (1 + !counterRef)) ^ " \n")));
				       counterRef := 1 + !counterRef; 
				       FPM.insert (en,(f,g),!counterRef)))
		    FPM.empty 
		    products
	end

    (* f >= g $B$N%V!<%kJQ?t(B, f \neq g $B$r2>Dj(B *)
    fun lookupPrecMap2 precMap2 (f,g)  = 
	case FPM.find (precMap2,(f,g)) of
	    SOME n => n
	  | NONE => (print ("looking up precMap2 fails: (" 
			    ^ (Fun.toString f)  ^ "," ^ (Fun.toString g)  ^ ")\n");
		     raise PoSolverError)

    (* f > g $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeGtPrec2 (encoding:encoding_info) (f,g) = 
	let val precMap2 = #PrecMap2 encoding
	in if Fun.equal (f,g) then False
	   else Conj [Atom (lookupPrecMap2 precMap2 (f,g)), Neg (Atom (lookupPrecMap2 precMap2 (g,f)))]
	end

    (* f >= g $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeGePrec2 (encoding:encoding_info) (f,g) =
	let val precMap2 = #PrecMap2 encoding
	in if Fun.equal (f,g) then True
	   else Atom (lookupPrecMap2 precMap2 (f,g))
	end

    (* f ~= g $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeEqPrec2 (encoding:encoding_info) (f,g) =
	let val precMap2 = #PrecMap2 encoding
	in if Fun.equal (f,g) then True
	   else Conj [Atom (lookupPrecMap2 precMap2 (f,g)), Atom (lookupPrecMap2 precMap2 (g,f))]
	end

    (* not (f >= g) $B$G$"$k$3$H$rI=$o$9O@M}<0(B *)
    fun encodeNgePrec2 (encoding:encoding_info) (f,g) = 
	let val precMap2 = #PrecMap2 encoding
	in if Fun.equal (f,g) then False
	   else Neg (Atom (lookupPrecMap2 precMap2 (f,g)))
	end

    (* $B%a%b(B & Look Up *)
    fun mkGtPrecTable2 lenFs = FPT.mkTable (lenFs * lenFs, PoSolverError)

    fun lookupGtPrecTable2 encoding (f,g) =
	let val gtPrecTable = #GtPrecTable encoding 
	in
	    case (FPT.find gtPrecTable (f,g))
	     of SOME p => p
	      | NONE => let val p = encodeGtPrec2 encoding (f,g)
			in (FPT.insert gtPrecTable ((f,g),p); p)
			end
	end

    (* $B%a%b(B & Look Up *)
    fun mkEqPrecTable2 lenFs = FPT.mkTable (lenFs * lenFs, PoSolverError)

    fun lookupEqPrecTable2 encoding (f,g) =
	let val eqPrecTable = #EqPrecTable encoding
	in
	   case (FPT.find eqPrecTable (f,g))
	    of SOME p => p
	       | NONE => let val p = encodeEqPrec2 encoding (f,g)
			 in (FPT.insert eqPrecTable ((f,g),p); p)
			 end
	end

    (* transitive $B$K$9$k$?$a$N@)Ls(B *)
    fun mkPrecCond2 (opt:options) (encoding:encoding_info) = 
	let val fs = #Fs encoding 
	    val precMap2 = #PrecMap2 encoding
	    val products0 = ListXProd.mapX (fn (f,g) => (f,g)) (fs,fs)
	    val products = L.filter (not o Fun.equal) products0
	    val triples0 = ListXProd.mapX (fn (fg,h) => (fg,h)) (products,fs)
	    val triples = L.filter (fn ((f,g),h) => not (Fun.equal (g,h)) andalso not (Fun.equal (f,h))) triples0
	in  
	    Conj (List.map (fn ((f,g),h) => 
			       Imp (Conj [Atom (lookupPrecMap2 precMap2 (f,g)), Atom (lookupPrecMap2 precMap2 (g,h))],
				    Atom (lookupPrecMap2 precMap2 (f,h)))) 
			   triples)
	end

    (* strict precedence $B$K$9$k$?$a$N@)Ls(B *)
    fun mkStrictPrecCond2 (opt:options) (encoding:encoding_info) = 
	if #useQuasi opt
	then Prop.True
	else let val fs = #Fs encoding
		 val precMap2 = #PrecMap2 encoding
		 val products0 = ListXProd.mapX (fn (f,g) => (f,g)) (fs,fs)
		 val products = L.filter (not o Fun.equal) products0
	     in
		 Conj (List.map (fn (f,g) => 
				    Imp (Atom (lookupPrecMap2 precMap2 (f,g)), Neg (Atom (lookupPrecMap2 precMap2 (g,f)))))
				products)
	     end

    fun mkAfCond2 (opt:options) (encoding:encoding_info) =
	let val faList = #FaList encoding
	    val faMap = #FaMap encoding
	    val fs = #Fs encoding
            (****** useAF = false $B$J$i!$(BencodeColStatus f = false and encodePiStatus (f,i) = true for all i *****)
	    val p0 = if #useAf opt
		     then True
		     else Conj (L.map (fn (f,ar) => Conj (Iff (encodeColStatus encoding f, False)::
							  (L.tabulate (ar, fn i => Iff (encodePiStatus encoding (f,i+1), True)))))
				      faList)
	   (****** encodeColStatus f = true $B$J$i(B pi(f)$B$,(Btrue$B$H$J$k$N$O(B1$B$D$@$1(B*****)
	    val p1 = if #useAf opt
		     then Conj (L.map (fn (f,ar) =>
				     Imp (encodeColStatus encoding f,
					  one (L.tabulate (ar,fn j => encodePiStatus encoding (f,j+1)))))
				 faList)
		     else True
	    val products0 = ListXProd.mapX (fn (f,g) => (f,g)) (fs,fs)
	    val products = L.filter (not o Fun.equal) products0
	    (*  f ~= g  $B$J$i(B encodeColStatus f = encodeColStatus g *)
	    val p2 = if #useAf opt
		     then Conj (L.map (fn (f,g) =>
				     Imp (lookupEqPrecTable2 encoding (f,g),
					  Iff (encodeColStatus encoding f, encodeColStatus encoding g)))
				 products)
		     else True
            (*  f ~= g  $B$J$i(B |pi(f)| = |pi(g)| *)
	    val p3 = Conj (L.map (fn (f,g) =>
				     let val arF = lookupFaMap (faMap, f)
					 val arG = lookupFaMap (faMap, g)
				     in Imp (lookupEqPrecTable2 encoding (f,g),
					     equalNumber (L.tabulate (arF,fn j => encodePiStatus encoding (f,j+1)),
							  L.tabulate (arG,fn j => encodePiStatus encoding (g,j+1))))
				     end)
				 products)
	in Conj [p0,p1,p2,p3]
	end



    (***************************************)
    (*** $BA4It$N(B encoding map & table $B$N:n@.(B ***)
    (***************************************)

   (* couterRef: $B$9$G$K;H$C$F$$$k%+%&%s%?HV9f$N:GBgCM(B *)
   (* rowLen $B$O;H$o$J$$$,(B mkEncodingInfo $B$H9g$o$;$k$?$a$N%@%_!<(B  *)
    fun mkEncodingInfo2 counterRef (fs,lenFs,rowLen,faList,faSet,faMap) =
	let val precMap2 = mkPrecMap2 counterRef fs
	    val gtPrecTable = mkGtPrecTable2 lenFs
	    val eqPrecTable = mkEqPrecTable2 lenFs
	    val mulMap = mkMulMap counterRef fs
	    val lexMap = mkLexMap counterRef faSet
	    val colMap = mkColMap counterRef fs
	    val piMap = mkPiMap counterRef faSet
	    val lexTable = mkLexTable faList
	    val sameNthTable = mkSameNthTable faList
	in
	    { Fs = fs,
	      LenFs = lenFs,
	      RowLen = rowLen, (* not used *)
	      FaList = faList,
	      FaSet = faSet,
	      FaMap = faMap,
	      PrecMap = FIM.empty, (* not used *)
	      PrecMap2 = precMap2,
              MulMap = mulMap,
              LexMap = lexMap,
              ColMap = colMap,
              PiMap = piMap,
	      GtPrecTable = gtPrecTable,
	      EqPrecTable = eqPrecTable,
	      LexTable = lexTable,
	      SameNthTable = sameNthTable } : encoding_info
	end
	    
    fun encodeOrderConstraint2 encodingType counterRef encoding 
			      (opt:options) rule =
	let 
	    val faSet = #FaSet encoding
	    val funSet = FIS.foldr (fn ((f,i),set)=> FS.add (set,f)) FS.empty faSet
	    fun prec (f,g) = lookupGtPrecTable2 encoding (f,g)
	    fun preceq (f,g) = lookupEqPrecTable2 encoding (f,g)
	(*    fun prec (f,g) = encodeGtPrec2 encoding (f,g) *)
	(*    fun preceq (f,g) = encodeEqPrec2 encoding (f,g) *)
	    fun mul f = encodeMulStatus encoding f
	    fun lex (f,i,j) = lookupLexTable encoding (f,i,j)
	    fun same (fi,gj) = lookupSameNthTable encoding (fi,gj)
	    fun col f = encodeColStatus encoding f
	    fun pi (f,i) = encodePiStatus encoding (f,i)

	    val _ = debug (fn _ => let val (l,r) = rule
				   in print ("do encoding: " ^ (Term.toString l)
					     ^ (case encodingType of
						    Order.GT => " :(gt): "
						  | Order.GE => " :(ge): "
						  | Order.EQ => " :(eq): ")
					     ^ (Term.toString r) ^ "\n")
				   end)
	in case (#useAf opt,#useQuasi opt,#useLex opt,#useMul opt) of
	       (false,false,false,false)
	       => Order.lpoEncoding prec counterRef encodingType rule
	     | (false,true,false,false)
	       => Order.qlpoEncoding prec preceq counterRef encodingType rule
	     | (false,false,true,false)
	       => Order.rpoEncoding (FS.empty,funSet) prec mul
				    lex counterRef encodingType rule
	     | (false,false,false,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	     | (false,false,true,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	     | (false,true,true,false)
	       => Order.qrpoEncoding (FS.empty,funSet) prec preceq
				     mul lex same counterRef
				     encodingType rule
	     | (false,true,false,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	     | (false,true,true,true)
	       => (print "Option not supported\n"; raise PoSolverError)
 	     | (true,false,false,false)
	       => Order.afLpoEncoding  prec col pi counterRef
				       encodingType rule
	     | (true,true,false,false)
	       => Order.afQlpoEncoding prec preceq col pi counterRef
				       encodingType rule
	     | (true,false,true,false)
	       => Order.afLposEncoding prec lex col pi counterRef
				       encodingType rule
	     | (true,true,true,false)
	       => Order.afQlposEncoding prec preceq lex col pi counterRef
				       encodingType rule
	     |(true,false,false,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	     |(true,true,false,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	     |(true,false,true,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	     |(true,true,true,true)
	       => (print "Option not supported\n"; raise PoSolverError)
	end

    (*** $B4X?t5-9f$N(Bprecedence$B$N=PNO(B ***)
    fun printPrec resultAr (encoding:encoding_info) =
	let val fs = #Fs encoding
	    val precMap2 = #PrecMap2 encoding
	    val products0 = ListXProd.mapX (fn (f,g) => (f,g)) (fs,fs)
	    val products = L.filter (not o Fun.equal) products0
	    fun gePrec (f,g) = if Fun.equal (f,g) then true
			       else isAssignedByTrue resultAr (lookupPrecMap2 precMap2 (f,g))
	    fun leFuns f = L.filter (fn g => Fun.equal (f,g) orelse gePrec (f,g)) fs
	    val _ = print "Precedence:\n"
	in  L.app (fn f => print (" " ^ (Fun.toString f) ^ " >= : " ^ (LU.toStringComma Fun.toString (leFuns f)) ^ "\n"))
		  fs
	end

    fun printStatus resultAr (encoding:encoding_info) (opt:options) = 
	let val fs = #Fs encoding
	    val faMap = #FaMap encoding
	    val _ = print "Status:\n"
	in L.app (fn f => let val ar = lookupFaMap (faMap, f)
			  in print (" " ^ (Fun.toString f) ^ " : " 
				    ^ (if #useAf opt
				       then prArgStatusAF resultAr encoding (opt:options) (f,ar)
				       else prArgStatus resultAr encoding (opt:options) (f,ar))
				    ^ "\n")
			  end)
		 fs
	end

    (*** $B4X?t5-9f$NM%@h=g0L!$(BLex/Mul$BIU$-$N0z?tHf3S=g=x%j%9%H$N=PNO(B ***)
    fun printInfo2 resultAr (encoding:encoding_info) (opt:options) = 
	(printPrec resultAr encoding; printStatus resultAr encoding opt)


  (* $B<-=q<07PO)=g=x$K4p$E$/(B (gesim,not gesim) $B$N=<B-2DG=@-H=Dj4o(B for non-joinability, 
    Assume rs0 U (rs0)^{-1} U rs1 is TRSs 
    multiset status $B$N%3!<%F%#%s%0$O!$(Bnot gesim $B$KBP1~$7$F$$$J$$$N$G!$(BMPO$B$OMxMQIT2D(B
   *)
  fun poSolverForNjwithNgePartial minisatPath tmpDir (opt:options) (term0,term1) (rs0,rs1) = 
      let (* val _ = print "Prove non-joinability of " *)
	  (*  val _ = print ((Term.toString term0) ^ " and " ^ (Term.toString term1) ^ "\n") *)
	  val rs0rev = L.map (fn (l,r) => (r,l)) rs0
	  val rs = rs0 @ rs1
	  val _ = debug (fn _ => print "(Current) usable rules:\n")
	  val _ = debug (fn _ => print (Trs.prRules rs0))
	  val _ = debug (fn _ => print (Trs.prRules rs1))

	  val symCount = ref 0

	  (* encoding $B$K;H$&Dj?t(B *)
	  val faMap  = Trs.funArityMapInRules ((term0,term1)::rs)
	  val faSet  = FM.foldri (fn (k,v,xs)=> FIS.add (xs,(k,v))) FIS.empty faMap
	  val funSet  = FM.foldri (fn (k,v,xs)=> FS.add (xs,k)) FS.empty faMap
	  val faList  = FM.listItemsi faMap
	  val fs  = L.map #1 faList
	  val lenFs  = L.length fs
	  val rowLen  = log lenFs

	  val encoding = mkEncodingInfo2 symCount (fs,lenFs,rowLen,faList,faSet,faMap)
	  val precCond2 = mkPrecCond2 opt encoding
	  val _ = debug (fn _ => println (Prop.printProp precCond2))

	  val strictPrecCond2 = mkStrictPrecCond2 opt encoding
	  val _ = debug (fn _ => println (Prop.printProp strictPrecCond2))

	  val lexCond = mkLexCond opt encoding 
	  val _ = debug (fn _ => println (Prop.printProp lexCond))

	  val afCond = mkAfCond2 opt encoding 
	  val _ = debug (fn _ => println (Prop.printProp afCond))

	  (* encoding $B$K;H$&%+%&%s%?(B *)
	   val count0 = !symCount + 1
	   val len0 = L.length rs0
	   val rsIdxs0 = LP.zip (rs0, L.tabulate (len0,fn x => count0 + x))
	   val _ = debug (fn _ => print (LU.toStringCommaLnSquare (fn (lr,i) => (Trs.prRule lr) ^ " : " ^ (Int.toString i)) rsIdxs0))

	   val count1 = !symCount + len0
	   val len1 = L.length rs1
	   val rsIdxs1 = LP.zip (rs1, L.tabulate (len1,fn x => count1 + x))
	   val _ = debug (fn _ => print (LU.toStringCommaLnSquare (fn (lr,i) => (Trs.prRule lr) ^ " : " ^ (Int.toString i)) rsIdxs1))

 	   val _ = symCount := count1 + len1
	   val dSyms = LU.eliminateDuplication' Fun.equal (L.map (fn (l,r) => valOf (Term.funRootOfTerm l)) rs)
	   val dsymsCountMap = L.foldr
				   (fn (f,fmap) => 
				       (debug (fn _ => print ("dSym: " ^ (Fun.toString f) ^ " : " ^ (Int.toString ((!symCount) + 1)) ^ "\n"));
					FM.insert(fmap,f,(symCount := (!symCount) + 1;!symCount))))
				   FM.empty 
				   dSyms
			      

	  (*  usable $B%k!<%k$N(B encoding function *)
	  fun mkUsableRulesProp term rsIdxs =
	      if #useAf opt
	      then
	      let open Prop
		  open Term
		  fun useFunArg (f,i) = if #useAf opt then encodePiStatus encoding (f,i+1) else True
		  fun isFunCol f = if #useAf opt then encodeColStatus encoding f else False

		  fun usableTerm (Var _)  = True
		    | usableTerm (Fun (f,ts,_)) =
		      Conj ((case FM.find (dsymsCountMap,f) of
				 SOME i => Atom i
			       | NONE => True)
			    :: (L.tabulate (length ts, 
					 fn i => Imp (useFunArg (f,i), usableTerm (L.nth (ts,i))))))


		   val usableRule = 
       Conj (L.mapPartial 
		 (fn ((l,r),i) => 
		     let val (Fun (f,ts,_)) = l
		     in case FM.find (dsymsCountMap,f) of
			    SOME j => SOME 
					  (Conj[
					   Imp (Atom j, Conj [Atom i, usableTerm r]),
					   Imp (isFunCol f, 
						Conj (L.map (fn k => Imp (Conj[useFunArg (f,k),
									       (if Term.isVar (L.nth (ts,k))
										then True
										else False)],
									  Conj [Atom i, usableTerm r]))
							    (L.tabulate (L.length ts, fn k=>k))))])
    			  | NONE => NONE
		     end)
		 rsIdxs)

		   fun isUsedVarInTerm (Var (x,_) ) y = if Var.equal (x,y) then True else False
		     | isUsedVarInTerm (Fun (f,ts,_)) y = 
		       Disj (L.mapPartial
				 (fn i => let val ti = L.nth (ts,i)
					  in if LU.member' Var.equal y (Term.varListInTerm ti)
					     then SOME (Conj [useFunArg (f,i), isUsedVarInTerm ti y])
					     else NONE
					  end)
				 (L.tabulate (L.length ts, fn i => i)))
		   
		   val varCond = Conj (L.map (fn ((l,r),_) => 
						 Conj (L.map (fn y => Imp (isUsedVarInTerm r y, isUsedVarInTerm l y)) 
							     (Term.varListInTerm r)))
					     rsIdxs)

	      in Prop.simplifyProp (Prop.Conj [usableTerm term, usableRule, varCond])
	      end		      
	      else True

	  (*  usable $B%k!<%k$N(B encoding  *)
	  val uprop0 = mkUsableRulesProp term0 rsIdxs0
	  val _ = debug (fn _ => println (Prop.printProp uprop0))

	  val uprop1 = mkUsableRulesProp term1 rsIdxs1
	  val _ = debug (fn _ => println (Prop.printProp uprop1))


	  (*  $B$3$l$^$G$N=`Hw$r$^$H$a$?(B proposition  *)
	  val condProp = simplifyProp (Conj [precCond2,strictPrecCond2,lexCond,afCond,uprop0,uprop1])

 	  val ngeProp = let val ps = encodeOrderConstraint2 Order.GE symCount encoding opt (term1,term0)
			in case ps of 
			       Conj (p0::defs) => Conj ((Neg p0):: defs)
 			     | _ => (print "negation of order encoding failed\n" ;
 				     raise PoSolverError)
			end
	  val _ = debug (fn _ => println (Prop.printProp ngeProp))

	  val leProp0 = Conj (L.map (fn ((l,r),i) => 
					let val p = encodeOrderConstraint2 Order.GE symCount encoding opt (r,l)
					in if #useAf opt then Imp (Atom i,p) else p
					end)   
				    rsIdxs0)
	  val _ = debug (fn _ => println (Prop.printProp leProp0))

	  val geProp1 = Conj (L.map (fn ((l,r),i) => 
					let val p = encodeOrderConstraint2 Order.GE symCount encoding opt (l,r)
					in  if #useAf opt then Imp (Atom i,p) else p
					end)
				    rsIdxs1)
	  val _ = debug (fn _ => println (Prop.printProp geProp1))

	  val prop = Prop.Conj [condProp,ngeProp,leProp0,geProp1]

	  val (result,resultArray) = Solver.propSolver minisatPath tmpDir (prop,!symCount)

	  val _ = Solver.propSolver2 "../tools/bin/yices" tmpDir (prop,!symCount)

	  val _ = debug (fn _ => if result
		  then (print " (success)\n"; printInfo2 resultArray encoding opt)
		  else (print " (failure)\n"))

	  local  open Term
	  in
	  fun applyAfToTerm piMap colSet (t as (Var _)) = t
	    | applyAfToTerm piMap colSet (t as (Fun (f,ts,ty))) =
	      case FM.find (piMap, f) of 
		  SOME ns => if FS.member (colSet, f)
			     then applyAfToTerm piMap colSet (L.nth (ts, hd ns))
			     else Fun (f,
				       L.map (fn i => applyAfToTerm piMap colSet (L.nth (ts,i))) ns,
				       ty)
		| _=> (debug (fn _ => print ("applyAfToTerm: (" ^ (Term.toString t) ^ "\n"));
		       raise PoSolverError)
	  end

	  val _ = debug (fn _=> if result
		  then 
		      let val _ = print "Obtained:\n"
			  val colSet = getColSet resultArray encoding 
			  val piMap = getPiMap resultArray encoding 
			  fun af t  = if (#useAf opt)
			      	      then applyAfToTerm piMap colSet t
				      else t
		  	  val _ = print (" " ^ (Term.toString (af term1)) ^ " :(not ge): " 
					 ^ (Term.toString (af term0)) ^ "\n" )

			  val _ = L.app (fn ((l,r),i) => 
					    if isAssignedByTrue resultArray i orelse (not (#useAf opt))
					    then print (" " ^ (Term.toString (af l)) ^ " :(le): " 
						 ^ (Term.toString (af r)) ^ "\n")
					    else print (" (" ^ (Term.toString l) ^ " -> " 
							^ (Term.toString r) ^ ": not usable)\n"))
					rsIdxs0
			  val _ = L.app (fn ((l,r),i) =>  
					    if isAssignedByTrue resultArray i
					    then print (" " ^ (Term.toString (af l)) ^ " :(ge): " 
							^ (Term.toString (af r)) ^ "\n")
					    else print (" (" ^ (Term.toString l) ^ " -> " 
						 ^ (Term.toString r) ^ ": not usable)\n"))
				         rsIdxs1
		      in ()
		      end
		  else ())

	 (*  fun outputDisproofAfEntry (colSet,lexListMap) (f,ar) () =  *)
	 (*      let fun outputArgs () =  *)
	 (* 	      case FM.find (lexListMap, f) of  *)
	 (* 		  NONE => (debug (fn _ => print ("outputDisproofAfEntry: (" ^ (Fun.toString f) ^ "\n")); *)
	 (* 			   raise PoSolverError) *)
	 (* 		| SOME ns => *)
	 (* 		  if FS.member (colSet, f) *)
	 (* 		  then CU.encloseProofLeafBy "collapsing" (Int.toString ((hd ns) + 1)) *)
	 (* 		  else if null ns *)
	 (* 		  then CU.encloseProofLeafBy "nonCollapsing" "" *)
	 (* 		  else CU.encloseProofTreesBy "nonCollapsing"  *)
	 (* 		       (L.map (fn i => fn () =>  *)
	 (* 					  CU.encloseProofLeafBy "position" (Int.toString (i+1))) *)
	 (* 			      ns) *)
	 (*      in CU.encloseProofTreesBy "argumentFilterEntry"  *)
	 (* 				[fn () => CU.encloseProofLeafBy "name" (Fun.toString f), *)
	 (* 				 fn () => CU.encloseProofLeafBy "arity" (Int.toString ar), *)
	 (* 				 outputArgs] *)
	 (*      end *)

	 (*  fun outputDisproofAf (colSet,lexListMap) faList () = *)
	 (*      CU.encloseProofTreesBy "argumentFilter"  *)
	 (* 			     (L.map (outputDisproofAfEntry (colSet,lexListMap)) faList) *)


         (* (* argument filtering $B$r$7$?8e$N(B arity $B$r=q$/(B *)  *)
	 (*  fun outputDisproofPrecEntry0 *)
	 (* 	  (colSet,mulList,piMap) (f,w) = *)
	 (*      if FS.member (colSet,f) *)
	 (*      then NONE *)
	 (*      else let val ar = case FM.find (piMap, f) of  *)
	 (* 			    NONE  => (debug (fn _ => print ("outputDisproofAfEntry0: ("  *)
	 (* 							    ^ (Fun.toString f) ^ "\n")); *)
	 (* 				      raise PoSolverError) *)
	 (* 			  | SOME ns => L.length ns *)
	 (* 	   in SOME (fn () =>  *)
	 (* 		       CU.encloseProofTreesBy "statusPrecedenceEntry" *)
	 (* 					      [fn () => CU.encloseProofLeafBy "name" (Fun.toString f), *)
	 (* 					       fn () => CU.encloseProofLeafBy "arity" (Int.toString ar), *)
	 (* 					       fn () => CU.encloseProofLeafBy "precedence" (Int.toString w), *)
	 (* 					       fn () => if LU.member' Fun.equal f mulList *)
	 (* 							then CU.encloseProofLeafBy "mul" "" *)
	 (* 							else CU.encloseProofLeafBy "lex" ""]) *)
	 (* 	   end *)


	 (*  fun outputDisproofPrecEntry (colSet,mulList,piMap) (xs,w) = *)
	 (*      L.mapPartial (fn f => outputDisproofPrecEntry0 (colSet,mulList,piMap) (f,w)) xs *)

	 (*  fun outputDisproofPrec (colSet,precList,mulList,piMap) () = *)
	 (*      CU.encloseProofTreeBy "pathOrder" *)
	 (*      (fn _ => CU.encloseProofTreesBy "statusPrecedence" *)
	 (* 				      (LU.mapAppend *)
	 (* 					   (fn i => outputDisproofPrecEntry  *)
	 (* 							(colSet,mulList,piMap) *)
	 (* 							(L.nth (precList,i), (L.length precList) - i)) *)
	 (* 					   (L.tabulate (L.length precList, fn i=>i)))) *)

	 (*  fun outputDisproofDecrease (colSet,precList,mulList,piMap) () = *)
	 (*      CU.encloseProofTreeBy "usableRulesNonJoin" *)
	 (*      (fn _ => CU.encloseProofTreeBy "strictDecrease" *)
	 (* 	       (fn _ => CU.encloseProofTreeBy "orderingConstraintProof" *)
	 (* 			(fn _ => CU.encloseProofTreeBy "redPair"  *)
	 (* 						       (outputDisproofPrec (colSet,precList,mulList,piMap))))) *)
	      
	  fun cpf () = if result 
		       andalso (!runCertification)
		       then let val colSet = getColSet resultArray encoding 
				val precList = mkPrecList resultArray encoding
				val mulList = getMulList resultArray encoding 
				val lexListMap = getLexListMapAF resultArray encoding
				val piMap = getPiMap resultArray encoding
			    in CU.encloseProofTreesBy "argumentFilterNonJoin"
						      [ outputArgmentFiltering (colSet,lexListMap) faList,
							outputDisproofDecrease (colSet,precList,mulList,piMap) ]
			    end
		       else ""

      in
	  (result,cpf)
      end




  end (* of local *)

  end (* of structre *)

