(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/poly.sml
 * description: ingredients for polynomial ordering
 * author: AOTO Takahito
 * author: SEKO Hiroyuki
 * 
 ******************************************************************************)


signature POLY =
sig
   type poly
   val toString: poly -> string
   val plus: poly * poly -> poly
   val isZero: poly -> bool
   val isEven: poly -> bool
   val minus: poly -> poly
   val times: poly * poly -> poly
   val apply: poly -> poly list -> poly

   type env = poly FunMap.map
   val termToPoly: env -> Var.ord_key list -> Term.term -> poly
   val isGreaterThan: env -> (Term.term * Term.term) ->  bool
   val isGreaterOrEqual: env -> (Term.term * Term.term) ->  bool
   val isEqual: env -> (Term.term * Term.term) ->  bool
   val isEvalEven: env -> (Term.term * Term.term) ->  bool

   val term_to_poly : (Fun.ord_key * poly) list 
                      -> Term.term 
                      -> poly
   val rule_to_poly : (Fun.ord_key * poly) list
                      -> (Term.term * Term.term)
                      -> (poly * poly)
   val poly_strict  : (Fun.ord_key * poly) list
                      -> (Term.term * Term.term) 
                      -> bool
   val poly_eq      : (Fun.ord_key * poly) list
                      -> (Term.term * Term.term) 
                      -> bool

   type meta_poly
   val mToString: meta_poly -> string
   val polyToMetaPoly: poly -> meta_poly
   val mPlus: meta_poly * meta_poly -> meta_poly
   val mIsZero: meta_poly -> bool
   val mIsEven: meta_poly -> bool
   val mMinus: meta_poly -> meta_poly
   val mTimes: meta_poly * meta_poly -> meta_poly
   val mApply: meta_poly -> meta_poly list -> meta_poly

   val evalPoly:  (int -> int) -> poly -> int
   val evalMetaPoly:  (int -> int) -> meta_poly -> poly
   val makeInitialMetaEnv: int ref 
          -> (Fun.ord_key * int) list
             -> int IntListMap2.map IntListMap2.map FunMap.map
(*    val makeInitialMetaEnv: int ref  *)
(* 			   -> (Fun.ord_key * int) list  *)
(* 			   -> (Fun.ord_key * (int list * int) list) list *)
   val makeGreaterThanConstraints:  meta_poly FunMap.map -> Term.term * Term.term -> poly list
   val makeGreaterOrEqualConstraints:  meta_poly FunMap.map -> Term.term * Term.term -> poly list
   val polyToProp: poly -> Prop.prop


  (* solver for R, strictly monotone  *)
   val polySolver: string 
		   -> string 
		   -> (Term.term * Term.term) list
		   -> bool

  (* solver for DP, weakly monotone  *)
   val polySolver2: string 
		   -> string 
		   -> (Term.term * Term.term) list * (Term.term * Term.term) list
		   -> bool

  (* solver for Non-Confluence constr(>) constr(>=) constr(<=) *)
   val polySolver3: string 
		   -> string 
		   -> (Term.term * Term.term) list
		   -> (Term.term * Term.term) list
		   -> (Term.term * Term.term) list
		   -> bool

end

structure Poly : POLY = 
struct

   exception PolyError

   local 
       open Term
       structure L = List
       structure ILM = IntListMap2	
       structure FM = FunMap
       fun merge([],ys) = ys
	 | merge(xs,[]) = xs
	 | merge(x::xs,y::ys) =
	   if x > y then y::merge(x::xs,ys) else x::merge(xs,y::ys)
     fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()

   exception Nonterm;

   type poly = int ILM.map

   (* $BB?9`<0$NI=<((B *)
   fun toStringSub str p = 
       let fun prVarList xs = PrintUtil.prProd (fn n => str ^ (Int.toString n)) xs
	   fun toStringMono (xs,n) = 
	       if null xs then (Int.toString n)
	       else (Int.toString n) ^ "*" ^ (prVarList xs)
	   val str =  PrintUtil.prSeqWith toStringMono "+" (ILM.listItemsi p)
       in
	   if str = "" then "0" else str
       end

   fun toString p =  toStringSub "x" p

   (* $BB?9`<0$NB-$7;;(B *)
   fun plus (p,q) = ILM.filter (fn x => x <> 0)  (ILM.unionWith (fn (x,y) => x + y) (p,q))

   (* $BB?9`<0$N(B0$BH=Dj(B *)
   fun isZero p = ILM.isEmpty (ILM.filter (fn x => x <> 0) p)

   (* $BB?9`<0$N$9$Y$F78?t$,6v?t$+$NH=Dj(B *)
   fun isEven p = ILM.isEmpty (ILM.filter (fn x => x mod 2 = 0) p)

   (* $BB?9`<0$N5UId9f(B *)
   fun minus p = ILM.map (fn x => ~x) p

   (* $BB?9`<0$N3]$1;;(B *)
   local
       fun times1 ((xs,n), p) = ILM.foldli 
				    (fn (ys,m,q) => ILM.insert (q, merge (xs,ys), m*n))
				    ILM.empty
				    p
   in
   fun times (p,q) = ILM.foldli
			 (fn (xs,n,r) => plus(times1((xs,n),q), r))
			 ILM.empty
			 p
   end

   (* $BB?9`<0$N9g@.(B *)
   (* $BNc$($P!$(Bapply (x + 2y + 1) [q1,q2] = q1 + 2*q2 + 1 *)
   local
       fun apply1 (xs,n) qs = L.foldl
				  (fn (x,p) => times (L.nth (qs,x-1),p))
				  (ILM.singleton ([],n))
				  xs
   in
   fun apply p qs = ILM.foldli
			(fn (xs,n,r) => plus (apply1 (xs,n) qs, r))
			ILM.empty
			p
   end

  (* $B4D6-!'4X?t5-9f(B $B"*(B $BB?9`<0(B *)
   type env = poly FunMap.map

   local 
       exception OutOfBounds
       exception InterpretationUndefined
       fun indexOf x ys = 
	   let fun indexOfSub x _ [] = (print ("Not found: " ^ (Var.toString x) ^ "\n"); raise OutOfBounds)
		 | indexOfSub x n (y::ys) = if Var.equal (x,y) then n else indexOfSub x (n+1) ys
	   in indexOfSub x 1 ys
	   end
   in
   (* $B4D6-(B rho $B$K$*$1$k$N9`$NB?9`<02r<a(B where $BJQ?t(B [x1,x2,..] = vnames $B$H$7$F!%(B *) 
   fun termToPoly rho vnames (Var (x,_)) = ILM.singleton ([indexOf x vnames],1)
     | termToPoly rho vnames (Fun (f,ts,_)) = 
       case FunMap.find (rho,f) of
	   SOME p => apply p (L.map (fn t => termToPoly rho vnames t) ts)
	 | NONE => (print ("Interpretation not found: " ^ (Fun.toString f) ^ "\n");
		    raise InterpretationUndefined)
   end

   (* $BB?9`<0$K$h$k=g=xIU$1H=Dj(B *) 
   local
       fun absolutelyPositive p = L.all (fn (x,n) => n >= 0) (ILM.listItemsi p)
       val one = ILM.singleton ([],1)
   in 
   fun isGreaterThan rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val p = termToPoly rho vnames s
	   val q = termToPoly rho vnames t
       in absolutelyPositive (plus (p, minus (plus (q, one))))
       end
   fun isGreaterOrEqual rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val p = termToPoly rho vnames s
	   val q = termToPoly rho vnames t
       in absolutelyPositive (plus (p, minus q))
       end
   fun isEqual rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val p = termToPoly rho vnames s
	   val q = termToPoly rho vnames t
       in isZero (plus (p, minus q))
       end
   fun isEvalEven rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val p = termToPoly rho vnames s
	   val q = termToPoly rho vnames t
       in isEven (plus (p, minus q))
       end
   end


   (************************************************)
   (*** $BB?9`<0$N@)Ls2r>C%=%k%P(B **************************)
   (************************************************)
   local
       open Prop
   in


   fun member x [] = false
     | member x (y::ys) = (x = y) orelse (member x ys)

   exception NotFound
   fun delete x [] = raise NotFound
     | delete x (y::ys) = if (x = y) then ys
			  else y::(delete x ys)

   fun eliminate xs [] = xs
     | eliminate xs (y::ys) = eliminate (delete y xs) ys

   fun intersection ([], _) = []
     | intersection (_, []) = []
     | intersection (x::xs,ys) = 
       let fun delx [] _ = NONE
	     | delx (z::zs) ex = if x = z
				 then SOME (List.revAppend (ex,zs))
				 else delx zs (z::ex)
       in case delx ys [] of
	      SOME rest => x::(intersection (xs,rest))
	    | NONE => intersection (xs,ys)
       end

  (* [ ~1*x*y, 2*y*y, ~3*z*y ] => [y] *)
   fun intersectionOfList [] =  []
     | intersectionOfList (ys::[]) =  ys
     | intersectionOfList (ys::yss) = intersection (ys, intersectionOfList yss)


   (* [2*y, 3*z] \le 2  => $BO@M}<0@8@.(B *)
   fun atMost m []  = if m < 0 then False else True
     | atMost m ((xs,n)::[])  =
       if m < 0 then False
       else if m >= n 
       then True 
       else Disj (L.map (fn x=>Atom (~x)) xs)
     | atMost m ((xs,n)::yss)  =
       if m < 0 then False
       else if null xs
       then atMost (m - n) yss
       else IfThenElse (Conj (L.map (fn x=>Atom x) xs),
			atMost (m - n) yss,
			atMost m yss)

   (* [2*y, 3*z] \ge 2  => $BO@M}<0@8@.(B *)
   fun atLeast m []  = if m <= 0 then True else False
     | atLeast m ((xs,n)::[])  = 
       if m <= 0 then True
       else if m <= n 
       then Conj (L.map (fn x=>Atom x) xs)
       else False
     | atLeast m ((xs,n)::yss)  =
       if m <= 0 then True
       else if null xs 
       then atLeast (m - n) yss
       else IfThenElse (Conj (L.map (fn x=>Atom x) xs),
			atLeast (m - n) yss,
			atLeast m yss)

   (*****
   [(([2*y, 3*z],2),(2,[1*x,2*z])), (([2*y, 3*z],1),(1,[1*x,2*z])) ] 
  => Disj [Conj [atMost 2 [2*y, 3*z], atLeat 2 [1*x,2*z]],
	   Conj [atMost 1 [2*y, 3*z], atLeat 1 [1*x,2*z]]]
   *****)
   fun atLeastAndAtMost xss = 
       Disj (L.map (fn ((ps,n),(m,ns)) =>  Conj [atLeast n ps,atMost m ns]) xss)
				   
   (*  [] => SOME 0,  [([],i)] => SOME i *)
   fun isConstant xs = if null xs
		       then SOME 0
		       else if (List.length xs) = 1 andalso (null ((fn (x,y) => x) (hd xs)))
		       then SOME ((fn (x,y) => y) (hd xs))
		       else NONE 

   (*  [([],2),([1],1),([1,2],2)] => 4$B!(<h$jF@$k:GBgCM$O78?t$NOB(B *)
   fun upperBound ps = L.foldl (fn ((_,n),sum) => n + sum) 0 ps

   (*  [([],2),([1],1),([1,2],2)] => 2$B!(<h$jF@$k:G>.CM$ODj?t$N78?t(B *)
   fun lowerBound ps = case L.find (fn (xs,_) => null xs) ps of
			   SOME (_,n) => n | NONE => 0
			 
   (***
     4$B%?%$%W$N=hM}!'(B
     (A)  ([2], [1])  => True
     (B)  ([2*y, 3*z], [1, 2*z]) => atLeastAndAtMost [(([2*y, 3*z],2),(2,[1,2*z])), 
                                                      (([2*y, 3*z],1),(1,[1,2*z])) ] 
     (C)  ([2*y, 3*z], [1]) => atLeast 1 [2*y, 3*z]
     (D)  ([2], [2*x, 1*y]) => atMost 2 [2*x, 1*y]
    ***)
   fun dividePoly (ps,ns) = 
       case (isConstant ps,isConstant ns) of
	   (SOME i, SOME j) => if i >= j then True else False
	 | (NONE, SOME j) => atLeast j ps
	 | (SOME i, NONE) => atMost i ns
	 | (NONE, NONE) => 
	   let val minval = Int.max (lowerBound ps, lowerBound ns)
	       val maxval = Int.min (upperBound ps, upperBound ns)
	   in if minval <= maxval
	      then atLeastAndAtMost (L.tabulate (maxval - minval + 1, 
					      fn i => ((ps,minval + i),(minval + i,ns))))
	      else False
	   end

  (* $B6&DLJQ?t$N=hM}!%;D$j$N=hM}$O(B dividePoly $B$G!%(B*)
   fun removeCommons xs (ps,ns) = 
       if not (null xs) 
	  then let val ps' = L.map (fn (ys,i) => (eliminate ys xs,i)) ps
		   val ns' = L.map (fn (ys,i) => (eliminate ys xs,i)) ns
		   val prop = dividePoly (ps',ns')
	       in  Disj (prop::(L.map (fn x => Atom (~x)) xs))
	       end
       else dividePoly (ps,ns)
	       

   (* [ ~1*x, 2*y, ~3*z ] => ([ 2*y ], [1*x,3*z ]) *)
   (* [ ([x],~1), ([y],2) ([z],~3) ] => ([ ([y],2) ], [ ([x],1), ([z],3) ]) *)
   fun partitionPoly xs = 
       let fun part (ps,ns) [] = (ps,ns)
	     | part (ps,ns) ((xs,k)::ys) = 
	       if k > 0 then part ((xs,k)::ps,ns) ys
	       else if k < 0 then part (ps, (xs,~k)::ns) ys
	       else part (ps,ns) ys
       in
	   part ([],[]) xs
       end

   fun polyToProp p =
       let val plist = ILM.listItemsi p
	   val xs = intersectionOfList (List.map (fn (xs,_) => xs) plist)
	   val (ps,ns) = partitionPoly plist
	   val prop = removeCommons xs (ps,ns)
	   val _ = debug (fn _ => PrintUtil.println (toString p))
	   val _ = debug (fn _ => PrintUtil.println (printProp prop))
       in
	   prop
       end

   fun polyToProp2 p =
       let val plist = ILM.listItemsi p
	   val xs = intersectionOfList (List.map (fn (xs,_) => xs) plist)
	   val (ps,ns) = partitionPoly plist
	   val prop = removeCommons xs (ps,ns)
	   val _ = debug (fn _ => PrintUtil.println (toString p))
	   val _ = debug (fn _ => PrintUtil.println (printProp prop))
       in
	   prop
       end

   end  (* end of local *)


   (* $B9`$+$iB?9`<0$X(B *) 
   local
       fun term_to_poly1 ps xs (Var ((x,i),_)) =  
           (case (L.find (fn (x1,n) => Var.equal (x1,(x,i))) xs)
             of SOME (x1,n) => ILM.insert (ILM.empty,[n],1)
              | NONE   => raise Nonterm)
	 | term_to_poly1 ps xs (Fun (f,ts,_))  =
           let 
               val subpoly = term_to_polylist ps xs ts
           in 
               case (L.find (fn (x1,y1) => Fun.compare (x1,f) = EQUAL) ps)
		of SOME (f1,p1) => apply p1 subpoly
		 | NONE         => raise Nonterm 
           end 
       and term_to_polylist ps xs [] = []
	 | term_to_polylist ps xs (t::ts) =
           (term_to_poly1 ps xs t)::(term_to_polylist ps xs ts);
       fun varlist_term t = VarSet.listItems (Term.varSetInTerm t);
       fun varlist_rule (s,t) = 
           VarSet.listItems (VarSet.union (Term.varSetInTerm s,
                                           Term.varSetInTerm t));
       fun var_list_number n []      = []
	 | var_list_number n (x::xs) = (x,n)::(var_list_number (n + 1) xs);
   in 
   fun term_to_poly ps t = 
       term_to_poly1 ps (var_list_number 1 (varlist_term t)) t;
   fun rule_to_poly ps (s,t) = 
       let 
           val xs = var_list_number 1 (varlist_rule (s,t))
       in
           (term_to_poly1 ps xs s,term_to_poly1 ps xs t)
       end 
   end;
       
   (* $BB?9`<0$K$h$k=g=xIU$1H=Dj(B *) 
   local
       fun positive_check p = L.all (fn (x,n) => n >= 0) 
                                       (ILM.listItemsi p);
       val minus1 = ILM.insert (ILM.empty,[],~1);
   in
   fun poly_strict ps (s,t) =
       let
           val (poly_of_s,poly_of_t) = rule_to_poly ps (s,t)
	   (*               val _ = PrintUtil.println
					("[" ^ (Term.toString s) 
					 ^ "] = " ^ (toString poly_of_s))
            val _ = PrintUtil.println
			("[" ^ (Term.toString t) 
			 ^ "] = " ^ (toString poly_of_t)) *)
           val minus_t = times (poly_of_t,minus1)
           val const_poly = plus (plus (poly_of_s,minus_t),minus1)
           val _ = PrintUtil.println
		       ("[" ^ (Term.toString s) ^ "] - [" ^ (Term.toString t)
			^ "] - 1 = " ^ (toString const_poly))
       in
           positive_check const_poly
       end;

   fun poly_eq     ps (s,t) =
       let
           val (poly_of_s,poly_of_t) = rule_to_poly ps (s,t)
           val minus_s = times (poly_of_s,minus1)
           val minus_t = times (poly_of_t,minus1)
           val const_poly1 = plus (poly_of_s,minus_t)
           val _  = PrintUtil.println (toString const_poly1)
           val const_poly2 = plus (poly_of_t,minus_s)
           val _  = PrintUtil.println (toString const_poly2)
       in
           (positive_check const_poly1) andalso (positive_check const_poly2)
       end
   end       



   (*** $B%a%?B?9`<0(B  
       ax + (b+2)y + c  --> (a1)*x1 + (a1+2)*x2 + (a3) 
   ***)
   type meta_poly = poly ILM.map

   (* $B%a%?B?9`<0$NI=<((B *)
   fun mToStringSub stringForVar stringForMVar pp = 
       let fun prVarList xs = PrintUtil.prProd (fn n => stringForVar ^ (Int.toString n)) xs
	   fun toStringMono (xs,str) = 
	       if null xs then str
	       else str ^ "*" ^ (prVarList xs)
	   val str0 = L.map (fn (xs,p) => (xs, "(" ^ (toStringSub stringForMVar p) ^ ")"))
			    (ILM.listItemsi pp)
	   val str =  PrintUtil.prSeqWith toStringMono "+" str0
       in
	   if str = "" then "0" else str
       end


   fun mToString pp =  mToStringSub "x" "a" pp

   (* $BB?9`<0(B => $B%a%?B?9`<0(B *)
   fun polyToMetaPoly p = ILM.foldli
			      (fn (xs,n,rr) => ILM.insert(rr, xs, ILM.singleton ([],n)))
			      ILM.empty
			      p


   (* $B%a%?B?9`<0$N(B0$BH=Dj(B *)
   fun mIsZero pp = ILM.isEmpty (ILM.filter (fn p => not (isZero p)) pp)

   (* $B%a%?B?9`<0$N6v?tH=Dj(B *)
   fun mIsEven pp = ILM.isEmpty (ILM.filter (fn p => not (isEven p)) pp)

   (* $B%a%?B?9`<0$NB-$7;;(B *)
   fun mPlus (pp,qq) = ILM.filter (fn p => not (isZero p))
				  (ILM.unionWith (fn (p,q) => plus (p,q)) (pp,qq))

   (* $B%a%?B?9`<0$N5UId9f(B *)
   fun mMinus pp = ILM.map minus pp

   (* $B%a%?B?9`<0$N3]$1;;(B *)
   local
       fun mTimes1 ((xs,p), qq) = ILM.foldli 
				    (fn (ys,q,rr) => ILM.insert (rr, merge (xs,ys), times (p,q)))
				    ILM.empty
				    qq
   in
   fun mTimes (pp,qq) = ILM.foldli
			 (fn (xs,p,rr) => mPlus(mTimes1((xs,p),qq), rr))
			 ILM.empty
			 pp
   end

   (* $B%a%?B?9`<0$N9g@.(B *)
   (* $BNc$($P!$(Bapply (x + 2y + 1) [q1,q2] = q1 + 2*q2 + 1 *)
   local
       fun mApply1 (xs,p) qqs = L.foldl
				  (fn (x,rr) => mTimes (L.nth (qqs,x-1),rr))
				  (ILM.singleton ([],p))
				  xs
   in
   fun mApply pp qqs = ILM.foldli
			(fn (xs,p,rr) => mPlus (mApply1 (xs,p) qqs, rr))
			ILM.empty
			pp
   end

  (* $B%a%?4D6-!'4X?t5-9f(B $B"*(B $B%a%?B?9`<0(B *)
   type meta_env = meta_poly FunMap.map

   local 
       exception OutOfBounds
       exception InterpretationUndefined
       fun indexOf x ys = 
	   let fun indexOfSub x _ [] = (print ("Not found: " ^ (Var.toString x) ^ "\n"); raise OutOfBounds)
		 | indexOfSub x n (y::ys) = if Var.equal (x,y) then n else indexOfSub x (n+1) ys
	   in indexOfSub x 1 ys
	   end
   in
   (* $B%a%?4D6-(B rho $B$K$*$1$k$N9`$N%a%?B?9`<02r<a(B where $BJQ?t(B [x1,x2,..] = vnames $B$H$7$F!%(B *) 
   fun termToMetaPoly rho vnames (Var (x,_)) = 
       ILM.singleton ([indexOf x vnames],ILM.singleton ([],1))
     | termToMetaPoly rho vnames (Fun (f,ts,_)) = 
       case FunMap.find (rho,f) of
	   SOME p => mApply p (L.map (fn t => termToMetaPoly rho vnames t) ts)
	 | NONE => (print ("Interpretation not found: " ^ (Fun.toString f) ^ "\n");
		    raise InterpretationUndefined)
   end


   (* $B%a%?4D6-(B rho $B$N85$G$N!$B?9`<0@)Ls$H$J$k(B [p1 \ge 0, p2 \ge 0, ...] 
         [p1, p2, ...] $B$N@8@.(B *) 
   local
       fun absolutelyPositive pp = ILM.listItems pp
       fun absolutelyEven pp = ILM.listItems pp
       val one = ILM.singleton([], ILM.singleton ([],1))
   in 
   fun makeGreaterThanConstraints rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val _ = debug (fn () => print ("Do encoding " ^  (Term.toString s) 
					  ^ " :gt: " ^ (Term.toString t) ^ "\n" ))
	   val pp = termToMetaPoly rho vnames s
	   val qq = termToMetaPoly rho vnames t
       in absolutelyPositive (mPlus (pp, mMinus (mPlus (qq, one))))
       end
   fun makeGreaterOrEqualConstraints rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val _ = debug (fn () => print ("Do encoding " ^  (Term.toString s) 
					  ^ " :ge: " ^ (Term.toString t) ^ "\n" ))
	   val pp = termToMetaPoly rho vnames s
	   val qq = termToMetaPoly rho vnames t
       in absolutelyPositive (mPlus (pp, mMinus qq))
       end
  (* $B$9$Y$F$N78?t$O6v?t$J$i!$I,$:6v?t$K$J$k(B *)
   fun makeEvenConstraints rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val _ = debug (fn () => print ("Do encoding " ^  (Term.toString s) 
					  ^ " - " ^ (Term.toString t) ^ " = 0 (mod 2)\n" ))
	   val pp = termToMetaPoly rho vnames s
	   val qq = termToMetaPoly rho vnames t
       in absolutelyEven (mPlus (pp, mMinus qq))
       end
  (* $B78?t$O6v?t$GDj?t$O4q?t$J$i!$I,$:4q?t$K$J$k(B *)
   fun makeOddConstraints rho (s,t) =
       let val vnames = VarSet.listItems (varSetInTerms [s,t])
	   val _ = debug (fn () => print ("Do encoding " ^  (Term.toString s) 
					  ^ " - " ^ (Term.toString t) ^ " = 0 (mod 2)\n" ))
	   val pp = termToMetaPoly rho vnames s
	   val qq = termToMetaPoly rho vnames t
       in absolutelyEven (mPlus (pp, mMinus (mPlus (qq, one))))
       end
   end

  (* x_1 + 2*x_2 + ~3, [x1:=1,x2:=2]  ---> 2 *)
   fun evalPoly assign p = 
       let fun prod xs k = foldl (fn (x,n) => (assign x)*n) k xs
       in ILM.foldli (fn (xs,k,sum) => sum + (prod xs k)) 0 p
       end

  (* a1*x_1 + 2*a2*x_2, [a1:=1,a2:=1]  ---> 1*x1 + 2*x2 *)
   fun evalMetaPoly assign pp = 
       ILM.foldli (fn (xs,p,q) => ILM.insert (q,xs,evalPoly assign p))
		  ILM.empty
		  pp

  (* $BB?9`<0=g=x$,@.N)$9$k$?$a$NB?9`<0@)Ls(B [p1 \ge 0, p2 \ge 0, ...] 
     $B$N@8@.!$(B [p1, p2, ...] $B$rJV$9(B *) 
   local
       fun makeLinearPoly count n = 
	   let fun new () = (count := (!count) + 1; !count)
	   in L.foldl (fn (xs,map) => ILM.insert (map, xs, ILM.singleton ([new ()],1)))
		      ILM.empty
		      ([] :: (L.tabulate (n, fn i => [i+1])))
(* 	   in L.map (fn xs => (xs, new ())) ([] :: (L.tabulate (n, fn i => [i+1]))) *)
	   end

       fun makeSimplePoly count n = 
	   let fun new () = (count := (!count) + 1; !count)
	       fun simpleList [] = [[]]
		 | simpleList (x::xs) = let val yss = simpleList xs
					in (L.map (fn ys => x::ys) yss) @ yss
					end
	   in L.foldl (fn (xs,map) => ILM.insert (map, xs, ILM.singleton ([new ()],1)))
		      ILM.empty
		      (simpleList (L.tabulate (n, fn i => i+1)))
(* 	   in L.map (fn xs => (xs, new ())) (simpleList (L.tabulate (n, fn i => i+1))) *)
	   end 


    (* $BBh(Bi$BHVL\$NL?BjJQ?t$,??$+(B *)
       fun isAssignedByTrue resultAr i = 
	   if i <= Array.length resultAr
	      andalso 0 < i
	   then Array.sub (resultAr,i-1) > 0
	   else (print ("isAssingedBy: index out of bouds: " 
			^ (Int.toString (i-1))
			^ "\n"); raise PolyError)

   in 
   fun makeInitialMetaEnv count falist =
(*        L.map (fn (f,n) => (f, makeLinearPoly count n)) falist *)
      L.foldl (fn ((f,n),map) => FM.insert (map, f, makeLinearPoly count n))
(*       L.foldl (fn ((f,n),map) => FM.insert (map, f, makeSimplePoly count n)) *)
	       FM.empty
	       falist

   fun polySolver minisatPath tmpDir rs =
       let val faSet = Trs.funAritySetInRules rs
	   val faList = FunIntSet.listItems faSet
	   val count = ref 0
	   val rho = makeInitialMetaEnv count faList
(* 	   val preRho = makeInitialMetaEnv count faList *)
(* 	   fun makePoly xss = L.foldl (fn ((xs,n),p) => ILM.insert (p,xs,ILM.singleton ([n],1))) *)
(* 				      ILM.empty xss *)
(* 	   val rho = L.foldl (fn ((f,xss),map) => FM.insert (map, f, makePoly xss)) *)
(* 				FM.empty *)
(*  				preRho *)
	   val one = ILM.singleton ([],1)
	   val varCoefficients = 
	       L.map (fn (xs,p) => plus (p, minus one))
			 (mapAppend (fn pp => L.filter (fn (xs,_) => not (null xs))  
						       (ILM.listItemsi pp))
				    (FM.listItems rho))
	   val pc = L.@ (mapAppend (makeGreaterThanConstraints rho) rs, varCoefficients)
	   val _ = debug (fn _ => print (PrintUtil.prList toString pc))
	   val prop = Prop.Conj (L.map polyToProp pc)
	   val (result,ar) = Solver.propSolver minisatPath tmpDir (prop,!count)
	   fun assign i = if (isAssignedByTrue ar i) then 1 else 0
	   val _ = debug (fn () => if result 
		   then
		       L.app (fn (f,_) => 
			     case FM.find (rho,f) of
				 SOME pp => PrintUtil.println ("[" ^ (Fun.toString f)  ^ "]"
							       ^ ":= " 
							       ^ (toString (evalMetaPoly assign pp)))
			       | NONE => raise PolyError)
			 faList
		   else ())
       in
	   result
       end

   fun polySolver2 minisatPath tmpDir (ds,rs) =
       let val faSet = Trs.funAritySetInRules (rs @ ds)
	   val faList = FunIntSet.listItems faSet
	   val count = ref 0
	   val rho = makeInitialMetaEnv count faList
(*	   val preRho = makeInitialMetaEnv count faList *)
(* 	   fun makePoly xss = L.foldl (fn ((xs,n),p) => ILM.insert (p,xs,ILM.singleton ([n],1))) *)
(* 				      ILM.empty xss *)
(* 	   val rho = L.foldl (fn ((f,xss),map) => FM.insert (map, f, makePoly xss)) *)
(* 				FM.empty *)
(*  				preRho *)
	   val one = ILM.singleton ([],1)
	   val pc = L.@ (mapAppend (makeGreaterOrEqualConstraints rho) rs,
			 mapAppend (makeGreaterThanConstraints rho) ds)
	   val _ = debug (fn _ => print (PrintUtil.prList toString pc))
	   val prop = Prop.Conj (L.map polyToProp pc)
	   val (result,ar) = Solver.propSolver minisatPath tmpDir (prop,!count)
	   fun assign i = if (isAssignedByTrue ar i) then 1 else 0
	   val _ = debug (fn () => if result 
		   then
		       L.app (fn (f,_) => 
			     case FM.find (rho,f) of
				 SOME pp => PrintUtil.println ( "[" ^ (Fun.toString f)  ^ "]"
							       ^ ":= " 
							       ^ (toString (evalMetaPoly assign pp)))
			       | NONE => raise PolyError)
			 faList
		   else ())
       in
	   result
       end


   (* for non-confluence check *)
   (* rs_i:  usable rules for t_i *)
   fun polySolver3 minisatPath tmpDir gtRules geRules leRules  =
       let val faSet = Trs.funAritySetInRules (gtRules @ geRules @ leRules)
	   val faList = FunIntSet.listItems faSet
	   val count = ref 0
	   val rho = makeInitialMetaEnv count faList
	   val one = ILM.singleton ([],1)
	   val pc = L.@ (mapAppend (makeGreaterThanConstraints rho) gtRules,
			 L.@ (mapAppend (makeGreaterOrEqualConstraints rho) (L.map (fn (x,y) => (y,x)) leRules),
			      mapAppend (makeGreaterOrEqualConstraints rho) geRules))
	   val _ = debug (fn _ => print (PrintUtil.prList toString pc))
	   val prop = Prop.Conj (L.map polyToProp pc)
	   val (result,ar) = Solver.propSolver minisatPath tmpDir (prop,!count)
	   fun assign i = if (isAssignedByTrue ar i) then 1 else 0
	   val _ = debug (fn () => if result 
		   then
		       L.app (fn (f,_) => 
			     case FM.find (rho,f) of
				 SOME pp => PrintUtil.println ( "[" ^ (Fun.toString f)  ^ "]"
							       ^ ":= " 
							       ^ (toString (evalMetaPoly assign pp)))
			       | NONE => raise PolyError)
			 faList
		   else ())
       in
	   result
       end
   end
(*
   (* for non-confluence check even and odd *)
   fun polySolver4 minisatPath tmpDir (term0,term1) rules  =
       let val faSet = Trs.funAritySetInRules ((term0,term1)::rules)
	   val faList = FunIntSet.listItems faSet
	   val count = ref 0
	   val rho = makeInitialMetaEnv count faList
	   val one = ILM.singleton ([],1)
	   val pc = L.@ (mapAppend (makeOddContraints [(term0,term1)]),
			 mapAppend (makeEvenContraints rho rules))
	   val _ = debug (fn _ => print (PrintUtil.prList toString pc))
	   val prop = Prop.Conj (L.map polyToProp2 pc)
	   val (result,ar) = Solver.propSolver minisatPath tmpDir (prop,!count)
	   fun assign i = if (isAssignedByTrue ar i) then 1 else 0
	   val _ = debug (fn () => if result 
		   then
		       L.app (fn (f,_) => 
			     case FM.find (rho,f) of
				 SOME pp => PrintUtil.println ( "[" ^ (Fun.toString f)  ^ "]"
							       ^ ":= " 
							       ^ (toString (evalMetaPoly assign pp)))
			       | NONE => raise PolyError)
			 faList
		   else ())
       in
	   result
       end
   end

*)



(*    val p = ILM.empty *)
(*    val _ = PrintUtil.println ("p=" ^(toString p)) *)

(*    val p1 = ILM.insert (ILM.insert (ILM.insert (p,[1],2),[2],1),[],1) *)
(*    val _ = PrintUtil.println ("p1=" ^(toString p1)) *)

(*    val _ = PrintUtil.println ("-p2=" ^(toString (minus p1))) *)

(*    val p2 = ILM.insert (ILM.insert (ILM.insert (p,[1],1),[2],3),[],2) *)
(*    val _ = PrintUtil.println ("p2=" ^(toString p2)) *)

(*    val a = ILM.listItemsi (plus (p1,p2)) *)
(*    val _ = PrintUtil.println ("p1+p2=" ^(toString (plus (p1,p2)))) *)

(*    val b = ILM.listItemsi (times (p1,p2)) *)
(*    val _ = PrintUtil.println ("p1*p2=" ^(toString (times (p1,p2)))) *)

(*    val q = ILM.insert (ILM.insert (p,[1,2],3),[],1) *)
(*    val _ = PrintUtil.println ("q=" ^(toString q)) *)

(*    val q1 = ILM.listItemsi (apply q [p1,p2]) *)
(*    val _ = PrintUtil.println ("q[p1,p2]=" ^ (toString (apply q [p1,p2]))) *)

(*    val a1 = ILM.insert (ILM.insert (p,[1],2),[],1) *)
(*    val _ = PrintUtil.println ("a1=" ^(toString a1)) *)
(*    val a2 = ILM.insert (ILM.insert (p,[1],3),[],3) *)
(*    val _ = PrintUtil.println ("a2=" ^(toString a2)) *)

(*    val _ = PrintUtil.println ("meta q=" ^(mToString (polyToMetaPoly q))) *)

(*    val pp1 = mPlus (ILM.insert (p,[1],a1), ILM.insert (p,[2],a2)) *)
(*    val _ = PrintUtil.println ("pp1=a1x1+a2x2=" ^(mToString pp1)) *)

(*    val pp2 = mPlus (ILM.insert (p,[1],a1), ILM.insert (p,[1],a2)) *)
(*    val _ = PrintUtil.println ("pp2=a1x1+a2x1=" ^(mToString pp2)) *)

(*    val _ = PrintUtil.println ("a1*a2=" ^(toString (times (a1,a2)))) *)

(*    val pp3 = mTimes (ILM.insert (p,[1],a1), ILM.insert (p,[1],a2)) *)
(*    val _ = PrintUtil.println ("pp3=a1x1*a2x1=" ^(mToString pp3)) *)

(*    val qq = ILM.insert (ILM.insert (p,[], ILM.insert (p,[2],1)),[1], ILM.insert (p,[1],2)) *)
		       
(*    val _ = PrintUtil.println ("qq=" ^(mToString qq)) *)

(*    val _ = PrintUtil.println ("-qq=" ^(mToString (mMinus qq))) *)
(*    val _ = PrintUtil.println ("qq-qq=" ^(mToString (mPlus (mMinus qq, qq)))) *)

(*    val _ = PrintUtil.println ("qq[pp1,pp2]=" ^ (mToString (mApply qq [pp1,pp2]))) *)

(*    val f = Fun.fromString "f" *)
(*    val g = Fun.fromString "g" *)
(*    val h = Fun.fromString "h" *)
(*    val x = IOFotrs.rdTerm "?x" *)
(*    val s = IOFotrs.rdTerm "f(f(?x))" *)
(*    val t = IOFotrs.rdTerm "g(?x)" *)
(*    val vns = VarSet.listItems (Term.varSetInTerms [s,t]) *)
(*    val pf = ILM.insert (ILM.singleton ([1],1),[],2) *)
(*    val pg = ILM.insert (ILM.singleton ([1],1),[],3) *)
(*    val _ = PrintUtil.println ("pf=" ^(toString pf)) *)
(*    val _ = PrintUtil.println ("pg=" ^(toString pg)) *)

(*    val rho = FM.insert (FM.singleton (f,pf),g,pg) *)
(*    val _ = PrintUtil.println ("s=" ^(Term.toString s)) *)
(*    val _ = PrintUtil.println ("t=" ^(Term.toString t)) *)

(*    val pgx = apply pg [(termToPoly rho vns x)] *)
(*    val _ = PrintUtil.println ("[g(?x)]_rho=" ^(toString pgx)) *)

(*    val ps = termToPoly rho vns s *)
(*    val _ = PrintUtil.println ("[s]_rho=" ^(toString ps)) *)
(*    val pt = termToPoly rho vns t *)
(*    val _ = PrintUtil.println ("[t]_rho=" ^(toString pt)) *)
(*    val _ = if isGreaterThan rho (s,t) *)
(* 	   then print "[s]>[t]\n" *)
(* 	   else print "not [s]>[t]\n" *)

(*    val mpf = ILM.insert (ILM.singleton ([],ILM.singleton ([2],1)), *)
(* 			 [1],ILM.singleton ([1],1)) *)

(*    val mph = ILM.insert ( *)
(* 	     ILM.insert ( *)
(* 	     ILM.insert (ILM.singleton ([],ILM.singleton ([6],1)), *)
(* 			 [1,2],ILM.singleton ([3],1)), *)
(* 			 [1],ILM.singleton ([4],1)), *)
(* 			 [2],ILM.singleton ([5],1)) *)

(*    val l1 = IOFotrs.rdTerm "h(h(?x,?y),?z)" *)
(*    val r1 = IOFotrs.rdTerm "h(?x,h(?y,?z))" *)
(*    val l2 = IOFotrs.rdTerm "h(f(?x),f(?y))" *)
(*    val r2 = IOFotrs.rdTerm "f(h(?x,?y))" *)
(*    val l3 = IOFotrs.rdTerm "h(f(?x),h(f(?y),?z))" *)
(*    val r3 = IOFotrs.rdTerm "h(f(h(?x,?y)),?z)" *)

(*    val mrho = FM.insert (FM.singleton (f,mpf),h,mph) *)

(*    val _ = print ("l1>r1:" ^ (PrintUtil.prList toString (makeGreaterThanConstraints mrho (l1,r1)))) *)
(*    val _ = print ("l2>r2:" ^ (PrintUtil.prList toString (makeGreaterThanConstraints mrho (l2,r2)))) *)
(*    val _ = print ("l3>r3:" ^ (PrintUtil.prList toString (makeGreaterThanConstraints mrho (l3,r3)))) *)

(*    val rs = IOFotrs.rdRules [ "h(h(?x,?y),?z) -> h(?x,h(?y,?z))", *)
(* 			      "h(f(?x),f(?y)) -> f(h(?x,?y))", *)
(* 			      "h(f(?x),h(f(?y),?z)) -> h(f(h(?x,?y)),?z)" ] *)



   end (* of local *)

end (* of struct  *)
