(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/solver.sml
 * description: interface using external solver 
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature SOLVER = 
sig
    val runProfile: bool ref
    val runDebug: bool ref
    val inTmpFileName: string ref
    val outTmpFileName: string ref

    type options = {
         satSolver: string,
         smtSolver: string,
         terminationProver: string,
         relativeTerminationProver: string,
         tmpDir: string
    }

    val cnfSolver: string -> string 
		    -> Prop.prop * int * int
		    -> bool * int array

    val propSolver: string -> string 
		    -> Prop.prop * int 
		    -> bool * int array


    val snSolver: string 
		  -> string 
		  -> (Term.term * Term.term) list
		  -> bool

    val sinSolver: string 
		  -> string 
		  -> (Term.term * Term.term) list
		  -> bool

    val relsnSolver: string 
		  -> string 
		  -> ((Term.term * Term.term) list * (Term.term * Term.term) list)
		  -> bool

    val smtSolver: string 
		   -> string 
		   -> Arith.prop * int 
		   -> bool * ((int * int) list)

    val propSolver2: string 
		   -> string 
		   -> Prop.prop * int 
		   -> bool * ((int * string) list)

   val yicesSolver: string 
		   -> string 
		   -> string
		   -> bool * string StringMap.map

end

structure Solver : SOLVER =
struct
   local
       structure A = Atom
       structure AT = AtomTable
       structure L = List
       structure LP = ListPair
       structure FM = FunMap
       structure FS = FunSet
       structure FIS = FunIntSet
       structure FIT = FunIntTable
       structure FIIT = FunIntIntTable
       structure FPT = FunPairTable
(*       structure CP = Compiler.Profile *)
       fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs

      (* increasing $B$J<+A3?tNs$KBP$7$F!$$=$NHVL\$NMWAG$r<h$j$@$9!%(B*)
      fun select ns xs = 
	  let
	      fun selectSub _ [] _ = []
		| selectSub m (n::ns) (x::xs) = 
		  if m = n
		  then x::(selectSub (m+1) ns xs)
		  else selectSub (m+1) (n::ns) xs
	  in
	      selectSub 0 ns xs
	  end

   in 


   exception SolverError of string
			    
   val runProfile = ref false : bool ref
   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()

   val inTmpFileName = ref ""
   val outTmpFileName = ref ""

    type options = {
         satSolver: string,
         smtSolver: string,
         terminationProver: string,
         relativeTerminationProver: string,
         tmpDir: string
    }

   (* $BL?BjO@M}<0$N=<B-2DG=@-H=Dj4o(B *)
   fun cnfSolver minisatPath tmpDir (cnfProp,numOfClause,maxVarIndex) =
       let fun receiveReport ins str =
	       if TextIO.endOfStream ins then str
	       else receiveReport ins (str ^ (TextIO.inputN (ins,1)))

	   fun main () = 
	       let 
		   val inTmpFile = let val tmpName = 
					   #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
				   in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
				   end
		   val _ = inTmpFileName := inTmpFile

		   val outTmpFile = let val tmpName = 
					    #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
				    in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
				    end
		   val _ = outTmpFileName := outTmpFile

		   val _ = debug (fn _ => print "\nwriting to temporary file...\n")

		   (* tmp$B%U%!%$%k$X=q$-9~$_(B *)
		   val outs = TextIO.openOut inTmpFile
		   val _ = Prop.prCnfInDimacs outs (cnfProp,numOfClause,maxVarIndex)
		   val _ = TextIO.closeOut outs

		   (* minisat $B$N<B9T(B *)
		   val cmd = minisatPath ^ " " ^ inTmpFile ^ " " ^ outTmpFile ^ " 2>&1\n"
 		   val _ = debug (fn _ => print cmd)
 		   val minisatCmd = ("/bin/sh", ["-c",cmd])
		   val proc = Unix.execute minisatCmd
		   val (ins,outs) = Unix.streamsOf proc

		   val _ = TextIO.closeOut outs
		   val report  = receiveReport ins ""
		   val _ = TextIO.closeIn ins
		   val _ = debug (fn _ => print report)
		   val _ = Unix.reap proc  (* $B%W%m%;%9=*N;=hM}(B*)

		   (* $B=PNO7k2L$NFI$_9~$_(B *)
		   val ins = TextIO.openIn outTmpFile
		   val ans = TextIO.inputLine ins

		   val result =  (isSome ans) andalso (valOf ans = "SAT\n")

		   val resultArray =  
		       if result
		       then let
			       val len = maxVarIndex
			       val ar = Array.array (len,maxVarIndex)
			       val ins' = (TextIO.getInstream ins)
			       val scanInt = Int.scan StringCvt.DEC TextIO.StreamIO.input1 
 			       fun scan n ins = if n < len 
						then case scanInt ins of
							 SOME (i,ins') => if i < 0 
									  then (Array.update (ar, n, ~1); 
										scan (n+1) ins')
									  else (Array.update (ar, n, 1); 
										scan (n+1) ins')
						       | NONE => ()
						else ()
			       val _ = scan 0 ins'; 
			   in
			       ar
			   end
		       else
			   Array.array (0,0)

		   (* $B%U%!%$%k$N8e;OKv(B *)			   
		   val _ = TextIO.closeIn ins; 
		   val _ = if !runDebug then () else OS.FileSys.remove inTmpFile   
		   val _ = if !runDebug then () else OS.FileSys.remove outTmpFile  
		   val _ = inTmpFileName := ""
		   val _ = outTmpFileName := ""

		   (* $B=PNO7k2L$NI=<((B *)			   
		   val _ = if !runDebug
			   then
			       if result
			       then (print "\nsatisfiable"; 
				     print "[";
				     Array.appi 
					 (fn (i,v) => print
							  ((if i <> 0 andalso i mod 5 = 0 then "." else "")
							   ^ (if v < 0 then "F" else "T")))
					 resultArray;
				     print "]\n")
			       else print "\nunsatisfiable\n"
			   else ()

	       in
		   (result ,resultArray)
	       end
       in
	  if !runProfile 
	  then TimeUtil.profile (main, "propSolver")
	  else main ()
       end

   (* $BL?BjO@M}<0$N=<B-2DG=@-H=Dj4o(B *)
   fun propSolver minisatPath tmpDir (prop,maxVarIndex) =
       cnfSolver minisatPath tmpDir (Prop.eqCnf (prop,maxVarIndex))

   (* ($B:GFb(B)$BDd;_@-$NH=Dj4o(B *)
   (* ($B:GFb(B)$BDd;_@-$NH=Dj4o(B, relative => ps != [] *)
   fun snSolverSub optionalMessages proverPath tmpDir (rs,ps) =
       let fun receiveReport ins str =
	       if TextIO.endOfStream ins then str
	       else receiveReport ins (str ^ (TextIO.inputN (ins,1)))

	   fun main () = 
	       let 
	(*	 ----- TTT2 needs to be specified by .trs file ------
 	           val inTmpFile = let val tmpName = 
					   #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
				   in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
				   end
	 *)
		   val inTmpFile = let val tmpName = 
					   ((#file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))) ^ ".trs")
				   in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
				   end


		   val _ = debug (fn _ => print "\nwriting to temporary file...\n")

		   (* tmp$B%U%!%$%k$X=q$-9~$_(B *)
		   val outs = TextIO.openOut inTmpFile
		   val _ = inTmpFileName := inTmpFile
		   val _ = Trs.prRelativeRulesInTpdb outs (rs,ps)
		   val _ = TextIO.output (outs, optionalMessages)
		   val _ = TextIO.closeOut outs

		   (* snChecker $B$N<B9T(B *)
		   (* val cmd = proverPath ^ " " ^ inTmpFile ^ " 2>&1\n" *)
		   val cmd = proverPath ^ " " ^ inTmpFile ^ "\n" (* $B7k2L$r%(%i!<$K=P$9>l9g$"$j(B *)
 		   val _ = debug (fn _ => print cmd)
 		   val proverCmd = ("/bin/sh", ["-c",cmd])
		   val proc = Unix.execute proverCmd
		   val (ins,outs) = Unix.streamsOf proc

		   val _ = TextIO.closeOut outs
		   val ans = TextIO.inputLine ins
		   val result =  (isSome ans) andalso (valOf ans = "YES\n")

		   val _ = TextIO.closeIn ins
		   val _ = Unix.reap proc  (* $B%W%m%;%9=*N;=hM}(B*)
		   val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 

		   (* $B=PNO7k2L$NI=<((B *)			   
		   val _ = if !runDebug
			   then
			       if result
			       then print "\nTerminating\n"
			       else print "\nUnknown\n"
			   else ()

	       in
		   result
	       end
       in
	   main ()
       end

   (* $BDd;_@-$NH=Dj4o(B *)
   fun snSolver proverPath tmpDir rs = snSolverSub "" proverPath tmpDir (rs,[])

   (* $B:GFbDd;_@-$NH=Dj4o(B *)
   fun sinSolver proverPath tmpDir rs = snSolverSub "(STRATEGY INNERMOST)\n" proverPath 
						    tmpDir (rs,[])

   (* $B:GFbDd;_@-$NH=Dj4o(B *)
   fun relsnSolver proverPath tmpDir (rs,ps) = snSolverSub "" proverPath tmpDir (rs,ps)

   (* SMT Solver *)
   local 
       open Arith
   in
   fun smtSolver proverPath tmpDir (prop,maxVarIndex) =
       let fun receiveReport ins str =
	       if TextIO.endOfStream ins then str
	       else receiveReport ins (str ^ (TextIO.inputN (ins,1)))

	   val varIndex = L.tabulate (maxVarIndex, fn i => i+1)

	   fun prDefVar i = "(define " ^ (prNum (Var i)) ^ "::int)\n"
	   fun prNonNegVar i = "(assert (<= 0 " ^ (prNum (Var i)) ^ "))\n"

	   fun main () = 
	       let 
		   val inTmpFile = let val tmpName = 
					   #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
				   in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
				   end


		   val _ = debug (fn _ => print "\nwriting to temporary file...\n")

		   (* tmp$B%U%!%$%k$X=q$-9~$_(B *)
		   val outs = TextIO.openOut inTmpFile
		   val _ = inTmpFileName := inTmpFile

		   val _ = L.app (fn i => TextIO.output (outs, prDefVar i)) varIndex
		   val _ = L.app (fn i => TextIO.output (outs, prNonNegVar i)) varIndex
		   val _ = TextIO.output (outs, "(assert " ^  (prProp prop ) ^ ")\n")
		   val _ = TextIO.output (outs, "(set-evidence! true)\n")
		   val _ = TextIO.output (outs, "(check)\n")
		   val _ = TextIO.closeOut outs

		   (* smtChecker $B$N<B9T(B *)
		   val cmd = proverPath ^ " " ^ inTmpFile ^ " 2>&1\n"
 		   val _ = debug (fn _ => print cmd)
 		   val proverCmd = ("/bin/sh", ["-c",cmd])
		   val proc = Unix.execute proverCmd
		   val (ins,outs) = Unix.streamsOf proc

		   val _ = TextIO.closeOut outs
		   val ans = TextIO.inputLine ins
		   val result =  (isSome ans) andalso (valOf ans = "sat\n")

		   fun readOneAssign line =
		       let 
			   val getNum = Int.scan StringCvt.DEC Substring.getc
			   val toks = Substring.tokens (not o Char.isAlphaNum) (Substring.full line)
		       in case toks of 
			      [x,y] => (case (getNum (Substring.triml 1 x), getNum y) of
					   (SOME (i,_), SOME (j,_)) => SOME (i,j)
					 | _ => NONE)
			    | _ => NONE
		       end

		   fun readAssigns ins ans = 
		       if TextIO.endOfStream ins 
		       then rev ans
		       else case TextIO.inputLine ins of
				SOME line => (case readOneAssign line of
						 SOME idval => readAssigns ins (idval::ans)
					       | NONE => readAssigns ins ans)
			      | NONE => readAssigns ins ans

		   val ans = readAssigns ins []

(* 		   val _ = print (ListUtil.toStringCommaLnSquare *)
(* 				    (fn (id,num) => "<" ^ (Int.toString id)  *)
(*						    ^ ", " ^ (Int.toString num) ^ ">") *)
(*				    ans) *)

		   val _ = TextIO.closeIn ins
		   val _ = Unix.reap proc  (* $B%W%m%;%9=*N;=hM}(B*)
		   val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 

		   (* $B=PNO7k2L$NI=<((B *)			   
		   val _ = if !runDebug
			   then
			       if result
			       then print "\nSatisfiable\n"
			       else print "\nUnsatisfiable\n"
			   else ()

	       in
		   (result, ans)
	       end
       in
	   main ()
       end
   end 

   local 
       open Prop
   in
   fun propSolver2 proverPath tmpDir (prop,maxVarIndex) =
       let fun receiveReport ins str =
	       if TextIO.endOfStream ins then str
	       else receiveReport ins (str ^ (TextIO.inputN (ins,1)))

	   val varIndex = L.tabulate (maxVarIndex, fn i => i+1)
	   fun prDefVar i = "(define " ^ (prAtom i) ^ "::bool)\n"

	   fun main () = 
	       let 
		   val inTmpFile = let val tmpName = 
					   #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
				   in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
				   end
		   val _ = debug (fn _ => print "\nwriting to temporary file...\n")

		   (* tmp$B%U%!%$%k$X=q$-9~$_(B *)
		   val outs = TextIO.openOut inTmpFile
		   val _ = inTmpFileName := inTmpFile

		   val _ = L.app (fn i => TextIO.output (outs, prDefVar i)) varIndex
		   val _ = TextIO.output (outs, "(assert " ^  (prProp prop ) ^ ")\n")
		   val _ = TextIO.output (outs, "(set-evidence! true)\n")
		   val _ = TextIO.output (outs, "(check)\n")
		   val _ = TextIO.closeOut outs

		   (* smtChecker $B$N<B9T(B *)
		   val cmd = proverPath ^ " " ^ inTmpFile ^ " 2>&1\n"
 		   val _ = debug (fn _ => print cmd)
 		   val proverCmd = ("/bin/sh", ["-c",cmd])
		   val proc = Unix.execute proverCmd
		   val (ins,outs) = Unix.streamsOf proc

		   val _ = TextIO.closeOut outs
		   val ans = TextIO.inputLine ins
		   val result =  (isSome ans) andalso (valOf ans = "sat\n")

		   fun readOneAssign line =
		       let 
			   val getNum = Int.scan StringCvt.DEC Substring.getc
			   val toks = Substring.tokens (not o Char.isAlphaNum) (Substring.full line)
		       in case toks of 
			      [x,y] => (case getNum (Substring.triml 1 x) of
					   SOME (i,_) => SOME (i,Substring.string y)
					 | _ => NONE)
			    | _ => NONE
		       end

		   fun readAssigns ins ans = 
		       if TextIO.endOfStream ins 
		       then rev ans
		       else case TextIO.inputLine ins of
				SOME line => (case readOneAssign line of
						 SOME idval => readAssigns ins (idval::ans)
					       | NONE => readAssigns ins ans)
			      | NONE => readAssigns ins ans

		   val ans = readAssigns ins []

 		   val _ = print (ListUtil.toStringCommaLnSquare 
				    (fn (id,bv) => "<" ^ (Int.toString id)
						    ^ ", " ^ bv ^ ">")
				    ans)

		   val _ = TextIO.closeIn ins
		   val _ = Unix.reap proc  (* $B%W%m%;%9=*N;=hM}(B*)
		   val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 

		   (* $B=PNO7k2L$NI=<((B *)			   
		   val _ = if !runDebug
			   then
			       if result
			       then print "\nSatisfiable\n"
			       else print "\nUnsatisfiable\n"
			   else ()

	       in
		   (result, ans)
	       end
       in
	   main ()
       end
   end


   fun receiveReport ins str =
       if TextIO.endOfStream ins then str
       else receiveReport ins (str ^ (TextIO.inputN (ins,1)))

   (* SMT Solver *)
   (* $B$b$C$H:Y$+$$;H$$J}$r$9$k(B *)
   fun yicesSolver proverPath tmpDir input =
       let 
	   val inTmpFile = let val tmpName = 
				   #file (OS.Path.splitDirFile (OS.FileSys.tmpName ()))
			   in OS.Path.joinDirFile {dir=tmpDir,file=tmpName}
			   end
			   
	   val _ = debug (fn _ => print "\nwriting to temporary file...\n")
		   
	   (* tmp$B%U%!%$%k$X=q$-9~$_(B *)
	   val outs = TextIO.openOut inTmpFile
	   val _ = inTmpFileName := inTmpFile
	   val _ = TextIO.output (outs, input)
	   val _ = TextIO.closeOut outs

	   (* smtChecker $B$N<B9T(B *)
(*	   val cmd = proverPath ^ " " ^ inTmpFile ^ " 2>&1\n" *)
	   val cmd = proverPath ^ " " ^ inTmpFile ^ " 2>/dev/null\n"
 	   val _ = debug (fn _ => print cmd)
 	   val proverCmd = ("/bin/sh", ["-c",cmd])
	   val proc = Unix.execute proverCmd
	   val (ins,outs) = Unix.streamsOf proc

	   val _ = TextIO.closeOut outs
	   val ans = TextIO.inputLine ins
	   val result =  (isSome ans) andalso (valOf ans = "sat\n")

	   fun readOneAssign line =
	       let 
		   val  toks = Substring.tokens (Char.contains " ()\n") 
						(Substring.full line)
	       in case toks of 
		      [x,y,z] => if (Substring.string x) = "=" 
				 then SOME (Substring.string y, Substring.string z) 
				 else NONE
		    | _ => NONE
	       end

	   fun readAssigns ins ans = 
	       if TextIO.endOfStream ins 
	       then ans
	       else case TextIO.inputLine ins of
			SOME line => (case readOneAssign line of
					  SOME (s1,s2) => 
					  readAssigns ins (StringMap.insert (ans,s1,s2))
					| NONE => readAssigns ins ans)
		      | NONE => readAssigns ins ans
				
	   val ans = readAssigns ins StringMap.empty

	   val _ = TextIO.closeIn ins
	   val _ = Unix.reap proc  (* $B%W%m%;%9=*N;=hM}(B*)
	   val _ = if !runDebug then () else OS.FileSys.remove inTmpFile 

(**
	   val _ = print "----------------\n"
	   val _ = L.app (fn (x,y) => print ("[" ^ x ^ "] |-> ["
					     ^ y ^ "]\n"))
			 (StringMap.listItemsi ans)
	   val _ = print "----------------\n"
**)

	   (* $B=PNO7k2L$NI=<((B *)			   
	   val _ = if !runDebug
		   then
		       if result
		       then print "\nSatisfiable\n"
		       else print "\nUnsatisfiable\n"
		   else ()
			
       in
	   (result, ans)
       end



  end (* of local *)
  end (* of structre *)


