(******************************************************************************
 * Copyright (c) 2017-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/unc.sml
 * description: ingredients for checking unique normal form property w.r.t. convertibility
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature UNC = 
sig
    val runDebug: bool ref
    datatype UncResult = UNC | NotUNC | Unknown

    val useStrongNonOverlapping: bool ref
    val useNonOmegaOverlapping: bool ref

    val useParallelClosedLinearization: bool ref
    val useStronglyClosedLinearization: bool ref

    val useWeightDecreasingJoinable: bool ref

    val useRightReducible: bool ref

    val useUncCompletionByStronglyClosed: bool ref
    val useUncCompletionByDevelopmentClosed: bool ref
    val useGeneralUncCompletion: bool ref
    val maxCPclosingSteps: int ref

(*    val useCheckCriticalPairsInNormalForms: bool ref *)
    val useNonUncByCps: bool ref

    val useRuleReversing: bool ref
    val useConfluenceProver: bool ref

    val useShallowDecProc: bool ref

    val uncSolver: Solver.options
                  -> DpSolver.options
                  -> PoSolver.options
                  -> CrSolver.options
                  -> (Term.term * Term.term) list 
                  -> UncResult

    val checkStronglyNonOverlapping: (Term.term * Term.term) list -> bool

    val uncSolverWithDecomposition: Solver.options
				    -> DpSolver.options
				    -> PoSolver.options
				    -> CrSolver.options
				    -> (Term.term * Term.term) list 
				    -> UncResult

   val runDecProcForShallow: (Term.term * Term.term) list  -> bool
end;

structure Unc : UNC = 
   struct
   local 
       open Term
       open Trs
       open Rewrite
       open Subst
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure PU = PrintUtil
       structure TP = TermPair
       structure TPS = TermPairSet
       structure TS = TermSet
       open PrintUtil
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   exception UncError

   datatype UncResult = UNC | NotUNC | Unknown
   exception SolvedWith of UncResult

   val useStrongNonOverlapping = ref false
   val useNonOmegaOverlapping = ref true

    val useParallelClosedLinearization = ref true
    val useStronglyClosedLinearization = ref true

   val useWeightDecreasingJoinable = ref true

   val useRightReducible = ref true

   val useNonUncByCps = ref true

   val useRuleReversing = ref true

   val useUncCompletionByStronglyClosed = ref true
   val useUncCompletionByDevelopmentClosed  = ref true
   val useGeneralUncCompletion  = ref true

   val useShallowDecProc = ref true
   val useConfluenceProver = ref true

(*   val useCheckCriticalPairsInNormalForms = ref false (* only for experiment *) removed *)


   val nameStrongNonOverlapping = "Strongly Non-Overlapping"
   val nameNonOmegaOverlapping = "Non-Omega-Overlapping"

   val nameParallelClosed = "Parallel Closed Conditional Linearization"
   val nameStronglyClosed = "Strongly Closed Conditional Linearization"

   val nameWeightDecreasingJoinable = "Weight-Decreasing Joinable"

   val nameUncCompletionByStronglyClosed = "UNC Completion (Strongly Closed)"
   val nameUncCompletionByDevelopmentClosed = "UNC Completion (Development Closed)"
   val nameGeneralUncCompletion = "UNC Completion (General)"

   val nameRightReducible = "Right-Reducible"

   val nameShallowDecProc = "Use a decision procedure for shallow TRSs"				

   fun unknown s = "unknown " ^ s


   (***  危険対を使った NotUNC の証明 ***)

   fun checkExistenceOfNomalFormsCP rs =
       let val cps = Cr.criticalPairs rs 
	   val witness = L.filter (fn (u,v) => not (Term.equal(u,v)) 
					       andalso (Rewrite.isNormalForm rs u) 
					       andalso (Rewrite.isNormalForm rs v)) cps
       in if null witness
	  then Unknown
	  else let val _ = print "CP consisting of normal forms:\n"
		   val _ =  print (Trs.prEqs witness)
	       in NotUNC
	       end
       end

   exception DistinctNormalForms of term * term 


   (* reversed rule を使った NotUNC の証明 *)
   fun checkNonUNCbyCP rs = 
       let val _ =  print "Check distinct normal forms in critical pair closure"
	   val _ =  debug (fn () => print "\n")
	   fun checkOne u = 
	       let (* val _ = debug (fn () => println ("check reducts of " ^ (Term.toString u))) *)
		   val uReducts = TS.listItems (Rewrite.manyStepsReductSet rs 3 u)
		   val unf = L.filter (Rewrite.isNormalForm rs) uReducts
	       in if L.length unf > 1
		  then raise DistinctNormalForms (hd unf, hd (tl unf))
		  else if null unf then NONE
		  else SOME (hd unf)
	       end 

	   fun checkTwo (u,v) = 
	       let val _ = debug (fn () => println ("check " ^ (Trs.prEq (u,v))))
	       in case (checkOne u, checkOne v) of
		      (SOME u2, SOME v2) => if Term.equal (u2,v2) 
					    then ()
					    else raise DistinctNormalForms (u2,v2)
		    | _ => ()
	       end
(************
	   fun checkRevCP (u,v) = 
	       let val _ = println ("check " ^ (Trs.prEq (u,v)))
		   val uReducts = TS.listItems (Rewrite.manyStepsReductSet rs 3 u)
		   val vReducts = TS.listItems (Rewrite.manyStepsReductSet rs 3 v)
		   val unf = L.filter (Rewrite.isNormalForm rs) uReducts
		   val vnf = L.filter (Rewrite.isNormalForm rs) vReducts
	       (* val _ = println (LU.toStringCommaCurly Term.toString unf) *)
	       (* val _ = println (LU.toStringCommaCurly Term.toString vnf) *)
	       in if not (null vnf)
		  then case L.find (fn u2 => not (LU.member' Term.equal u2 vnf)) unf of
			   SOME u2 => raise DistinctNormalForms (u2,hd vnf)
			 | NONE => 
			   if not (null unf)
			   then case L.find (fn v2 => not (LU.member' Term.equal v2 unf)) vnf of
				    SOME v2 => raise DistinctNormalForms (hd unf, v2)
				  | NONE => if L.length unf > 1
					    then raise DistinctNormalForms (hd unf, hd (tl unf))
					    else if L.length vnf > 1
					    then raise DistinctNormalForms (hd vnf, hd (tl vnf))
					    else ()
			   else ()
		  else ()
	       end
******)
	   val cps = ListMergeSort.sort (fn ((x1,x2),(y1,y2)) => termSize x1 + termSize x2  > termSize y1 + termSize y2)
					   (L.filter (not o Term.equal) (Cr.criticalPairs2 (rs,rs)))
	   val cps2 = if L.length cps > 100 then L.take (cps,100) else cps (* heulistics *)
	   val _ = debug (fn () => print ("Cps:\n" ^ Trs.prEqs cps2))
	   val _ = L.app checkTwo cps2

	   val revRs = L.map (fn (l,r) => (r,l)) rs
	   val revCps = ListMergeSort.sort (fn ((x1,x2),(y1,y2)) => termSize x1 + termSize x2  > termSize y1 + termSize y2)
					  (L.filter (not o Term.equal) (Cr.criticalPairs2 (revRs,revRs)))

	   val revCps2 = if L.length revCps > 100 then L.take (revCps,100) else revCps (* heulistics *)
	   val _ = debug (fn () => print ("rev Cps:\n" ^ Trs.prEqs revCps2))
	   val _ = L.app checkTwo revCps2
	   val _ =  debug (fn () => print "    ")
       in (println "...failed"; NONE)
       end
       handle DistinctNormalForms (u,v) => 
	      let val _ =  debug (fn () => print "    ...found")
		  val _ =  print "\nconvertible distinct normal forms: "
		  val _ =  println (Trs.prEq (u,v))
	      in SOME NotUNC
	      end


   (***  危険対の close 条件を用いた CR化：抽象版 ***)

   exception OrientationFailed

  (* gen :  条件に閉じていない危険対を生成する関数  *)
   (* fun criticalPairClosingStep gen rs = *)
   (*     let val cand = gen rs *)
   (*     in if null cand *)
   (* 	  then let val _ =  print "CP closed TRS:\n" *)
   (* 		   val _ =  print (Trs.prRules rs) *)
   (* 	       in SOME (UNC,[]) *)
   (* 	       end *)
   (* 	  else let val newRules =  *)
   (* 		       L.map (fn (u,v) => case (Rewrite.isNormalForm rs u, Rewrite.isNormalForm rs v) of *)
   (* 					      (true,true) => raise DistinctNormalForms (u,v) *)
   (* 					    | (true,false) => if LU.subseteq' Var.equal (varListInTerm u, varListInTerm v) *)
   (* 							      then (v,u) *)
   (* 							      else raise OrientationFailed *)
   (* 					    | (false,true) => if LU.subseteq' Var.equal (varListInTerm v, varListInTerm u) *)
   (* 							      then (u,v) *)
   (* 							      else raise OrientationFailed *)
   (* 					    | (false,false) => if termSize u >= termSize v  *)
   (* 							       then if LU.subseteq' Var.equal (varListInTerm v, varListInTerm u) *)
   (* 								    then (u,v) *)
   (* 								    else if LU.subseteq' Var.equal (varListInTerm u, varListInTerm v) *)
   (* 								    then (v,u) *)
   (* 								    else raise OrientationFailed *)
   (* 							       else if LU.subseteq' Var.equal (varListInTerm u, varListInTerm v) *)
   (* 							       then (v,u) *)
   (* 							       else if LU.subseteq' Var.equal (varListInTerm v, varListInTerm u) *)
   (* 							       then (u,v) *)
   (* 							       else raise OrientationFailed) *)
   (* 					cand *)
   (* 	       in SOME (Unknown, newRules) *)
   (* 	       end *)
   (* 	       handle DistinctNormalForms (u,v) =>  *)
   (* 		      let val _ =  print "constructed TRS:\n" *)
   (* 			  val _ =  print (Trs.prRules rs) *)
   (* 			  val _ =  print "convertible distinct normal forms: " *)
   (* 			  val _ =  println (Trs.prEq (u,v)) *)
   (* 		      in SOME (NotUNC,[]) *)
   (* 		      end *)
   (* 		    | OrientationFailed => NONE *)
   (*     end *)


  (* gen :  条件に閉じていない危険対を生成する関数  *)
  (** version 2 **)
  (** each rule in rs supposed to satisfy pred **)
   fun criticalPairClosingStep2 gen pred rs =
       let val cand = gen rs
       in if null cand
	  then let val _ =  print "confluent TRS:\n"
		   val _ =  print (Trs.prRules rs)
	       in SOME (UNC,[])
	       end
	  else let fun checkAns (u,v) = if LU.subseteq' Var.equal (varListInTerm u, varListInTerm v)
					then [(v,u)]
					else let val x = hd (LU.differenceByAll' Var.equal (varListInTerm u, varListInTerm v))
						 val k = maxVarIndexInTerm u
						 val y = Var.increaseIndexBy (k+1) x
						 val sigma = VM.insert (VM.empty,x, Var (y, valOf (sortOfVarInTerm u x)))
						 val u2 = Subst.applySubst sigma u
					     in raise DistinctNormalForms (u,u2)
					     end


		   fun getCand xs = (* xs is supposed to be non-empty *)
		       hd (LU.takeMinimals (fn (x,y) => SOME (Int.compare (termSize x, termSize y))) xs)

		   fun getNewRulesSub (u,uReducts,v,vReducts) =
		       let val uCand = getCand uReducts
			   val vCand = getCand vReducts
		       in if termSize uCand >= termSize vCand
			  then if LU.subseteq' Var.equal (varListInTerm vCand, varListInTerm u)
			       then [(u,vCand)]
			       else if LU.subseteq' Var.equal (varListInTerm uCand, varListInTerm v)
			       then [(v,uCand)]
			       else raise OrientationFailed
			  else if LU.subseteq' Var.equal (varListInTerm uCand, varListInTerm v)
			  then [(v,uCand)]
			  else if LU.subseteq' Var.equal (varListInTerm vCand, varListInTerm u)
			  then [(u,vCand)]
			  else raise OrientationFailed
		       end

		   fun estimateAns (u,v) =
		    (*   let val us = Rewrite.developOneStepReducts rs u
			   val vs = Rewrite.developOneStepReducts rs v
			   val cands = ListMergeSort.sort (fn (x,y) => termSize x > termSize y) (us@vs)
		       in case L.find (fn w => LU.subseteq' Var.equal (varListInTerm w, varListInTerm u)
					       andalso LU.subseteq' Var.equal (varListInTerm w, varListInTerm v))
				      cands of
			      SOME w => (if Term.equal (u,w) then [] else [(u,w)])
					@ (if Term.equal (v,w) then [] else [(v,w)])
			    | NONE => []
		       end *)

		   let val uReducts = Rewrite.developOneStepReducts rs u
		       val vReducts = Rewrite.developOneStepReducts rs v
		   in case (L.find (Rewrite.isNormalForm rs) uReducts,
			    L.find (Rewrite.isNormalForm rs) vReducts)  of
			  (SOME u2, SOME v2) => if not (Term.equal (u2,v2))
						then raise DistinctNormalForms (u2,v2)  (* finish *) 
						else  (* u, v are not normal form *)
						    if Term.termSize u > Term.termSize v
						    then [(v,u2)]
						    else [(u,v2)]
			| (NONE, SOME v2) => if LU.subseteq' Var.equal (varListInTerm v2, varListInTerm u)
					     then [(u,v2)]
					     else getNewRulesSub (u,uReducts,v,vReducts)
			| (SOME u2, NONE) => if LU.subseteq' Var.equal (varListInTerm u2, varListInTerm v)
					     then [(v,u2)]
					     else getNewRulesSub (u,uReducts,v,vReducts)
			| (NONE, NONE) => getNewRulesSub (u,uReducts,v,vReducts)
		   end


		   val newRules0 = 
		       LU.mapAppend (fn (u,v) => case (Rewrite.isNormalForm rs u, Rewrite.isNormalForm rs v) of
						    (true,true) => raise DistinctNormalForms (u,v)
						  | (true,false) => checkAns (u,v)
						  | (false,true) => checkAns (v,u)
						  | (false,false) => estimateAns (u,v))
				    (LU.eliminateDuplication' (fn ((x1,x2),(y1,y2)) => TermPair.equal ((x1,x2),(y1,y2))
										       orelse TermPair.equal ((x1,x2),(y2,y1)))
							      cand)
		   val newRules = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
							   newRules0

	       in if null newRules orelse List.exists (not o pred) newRules
		  then NONE
		  else SOME (Unknown, newRules)
	       end
	       handle DistinctNormalForms (u,v) => 
		      let val _ =  print "constructed TRS:\n"
			  val _ =  print (Trs.prRules rs)
			  val _ =  print "convertible distinct normal forms: "
			  val _ =  println (Trs.prEq (u,v))
		      in SOME (NotUNC,[])
		      end
       		    | OrientationFailed => NONE
       end


   val maxCPclosingSteps = ref 2

  (* gen :  条件に閉じていない危険対を生成する関数  *)
  (* pred:  危険対条件が正しいための条件 *)
   (* fun limitCPclosingSteps gen pred n rs =  *)
   (*     if n > (!maxCPclosingSteps) *)
   (* 	  orelse not (pred rs) *)
   (*     then Unknown *)
   (*     else (case criticalPairClosingStep gen rs of *)
   (* 		NONE => Unknown *)
   (* 	      | SOME (Unknown,newRules) => limitCPclosingSteps gen pred (n+1) (rs@newRules) *)
   (* 	      | SOME (ans,_) => ans) *)
   
  (** version 2 **)
   val largeTermSize = 100

   fun limitCPclosingSteps2 gen pred (repeat,size) rs = 
       if repeat >= (!maxCPclosingSteps) (* non-decreasing two consequtive time 0=>1=>2  *)
       then Unknown
       else case criticalPairClosingStep2 gen pred rs of
		NONE => Unknown
	      | SOME (Unknown,newRules) => 
		let val min = L.foldr (fn ((l,r),n) => Int.min (termSize l,n)) largeTermSize newRules
		    val rs2 = rs@newRules
		in case size of 
		       SOME k => if k <= min
				 then limitCPclosingSteps2 gen pred (repeat+1, SOME min) rs2
				 else limitCPclosingSteps2 gen pred (1, SOME min) rs2
		     | NONE => limitCPclosingSteps2 gen pred (1, SOME min) rs2
		end
	      | SOME (ans,_) => ans

   (*** Strong Closed 条件(簡易版)を用いた CR化 ***)
   fun nonStrongClosedCps rs = L.filter (not o Cr.isStrongClosed rs) (Cr.criticalPairs rs)

(*   fun confStrongClosedCps rs = limitCPclosingSteps nonStrongClosedCps Trs.areLinearRules 0 rs *)

   fun confStrongClosedCps2 rs = 
       if Trs.areLinearRules rs
       then limitCPclosingSteps2 nonStrongClosedCps Trs.isLinearRule (0,NONE) rs
       else Unknown

   (*** Development Closed (Huet-Toyama-Oostrom)条件(簡易版)を用いた CR化 ***)

   fun nonDevelopmentClosedCps rs = 
       let val inCps = Cr.insideCriticalPairs rs 
	   val inAns = L.filter (not o Cr.isOostromClosedInCp rs) inCps
	   val outCps = Cr.outsideCriticalPairs rs 
	   val outAns = L.filter (not o Cr.isOostromClosedOutCp rs) outCps
       in inAns @ outAns
       end

   (* fun confDevelopmentClosedCps rs = limitCPclosingSteps nonDevelompentClosedCps Trs.areLeftLinearRules 0 rs *)

   fun confDevelopmentClosedCps2 rs =
       if Trs.areLeftLinearRules rs
    (*   then limitCPclosingSteps2 nonStrongClosedCps Trs.isLeftLinearRule (0,NONE) rs *) 
       then limitCPclosingSteps2 nonDevelopmentClosedCps Trs.isLeftLinearRule (0,NONE) rs  (* bug fix 2018/04/24 *)
       else Unknown

   exception TermSizeExceedsTheLimit 

   (*** critical pair completion (?) ***)

   fun cpCompStep max constFn num rs cps = 
       let fun getCand xs = (* xs is supposed to be non-empty *)
	       hd (LU.takeMinimals (fn (x,y) => SOME (Int.compare (termSize x, termSize y))) xs)
	   
		  
	   fun checkAns (u,v) = if LU.subseteq' Var.equal (varListInTerm u, varListInTerm v)
				then SOME (v,u)
				else let val x = hd (LU.differenceByAll' Var.equal (varListInTerm u, varListInTerm v))
					 val k = maxVarIndexInTerm u
					 val y = Var.increaseIndexBy (k+1) x
					 val sigma = VM.insert (VM.empty,x, Var (y, valOf (sortOfVarInTerm u x)))
					 val u2 = Subst.applySubst sigma u
				     in raise DistinctNormalForms (u,u2)
				     end

	   fun getNewRulesSub (u,uReducts,v,vReducts) =
	       let val uCand = getCand uReducts
		   val vCand = getCand vReducts
	       in if termSize uCand >= termSize vCand
		  then if LU.subseteq' Var.equal (varListInTerm vCand, varListInTerm u)
		       then SOME (u,vCand)
		       else if LU.subseteq' Var.equal (varListInTerm uCand, varListInTerm v)
		       then SOME (v,uCand)
		       else NONE
		  else if LU.subseteq' Var.equal (varListInTerm uCand, varListInTerm v)
		  then SOME (v,uCand)
		  else if LU.subseteq' Var.equal (varListInTerm vCand, varListInTerm u)
		  then SOME (u,vCand)
		  else NONE
	       end

	   fun getNewRule (u, uReductSet,v,vReductSet) =
	       case (Rewrite.isNormalForm rs u, Rewrite.isNormalForm rs v) of
		   (true,true) => raise DistinctNormalForms (u,v) (* finish *)
		 | (true,false) => checkAns (u,v)
                                   (* if LU.subseteq' Var.equal (varListInTerm u, varListInTerm v)
				   then SOME (v, u) (* getCand (TS.listItems uReductSet)) *)
				   else NONE *)
		 | (false,true) => checkAns (v,u)
                                   (* if LU.subseteq' Var.equal (varListInTerm v, varListInTerm u)
				   then SOME (u, v) (* getCand (TS.listItems vReductSet)) *)
				   else NONE *)
		 | (false,false) => 
		   let val uReducts = TS.listItems uReductSet
		       val vReducts = TS.listItems vReductSet
		   in case (L.find (Rewrite.isNormalForm rs) uReducts,
			    L.find (Rewrite.isNormalForm rs) vReducts)  of
			  (SOME u2, SOME v2) => raise DistinctNormalForms (u2,v2)  (* finish *)
			| (NONE, SOME v2) => if LU.subseteq' Var.equal (varListInTerm v2, varListInTerm u)
					     then SOME (u,v2)
					     else getNewRulesSub (u,uReducts,v,vReducts)
			| (SOME u2, NONE) => if LU.subseteq' Var.equal (varListInTerm u2, varListInTerm v)
					     then SOME (v,u2)
					     else getNewRulesSub (u,uReducts,v,vReducts)
			| (NONE, NONE) => getNewRulesSub (u,uReducts,v,vReducts)
		   end

	   fun checkCp (u,v) = if Term.equal (u,v)
			       then NONE
			       else let (* val _ = print "cp: " *)
					(* val _ =  println (Trs.prEq (u,v)) *)
					val uReductSet = constFn rs u
					val vReductSet = constFn rs v
					val comon = TS.intersection (vReductSet,uReductSet)
				    in if not (TS.isEmpty comon)
				       then NONE (* eliminate that cp *)
				       else getNewRule (u, uReductSet,v, vReductSet)
				    end

	   val newRules0 = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
						 (L.mapPartial checkCp cps)

	   val newRules = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
					      (newRules0, rs)

	   val _ =  debug (fn () => println "added rules:")
	   val _ =  debug (fn () => print (Trs.prRules newRules))

	   val newCps = Cr.criticalPairs2 (newRules,rs)
			@ Cr.criticalPairs2 (rs,newRules)
			@ Cr.criticalPairs newRules

       in if null newRules
	  then let (* val _ =  print "final rules:\n" *)
		   (* val _ =  print (Trs.prRules rs) *)
	       in rs
	       end
	  else if let val (l,_) = hd (LU.takeMinimals (fn ((l1,r1),(l2,r2)) => SOME (Int.compare (termSize l1, termSize l2))) newRules)
		  in termSize l > max
		  end
	  then raise TermSizeExceedsTheLimit 
	  else cpCompStep max constFn (num+1) (rs@newRules) newCps
       end


(********
	   (* val newRules1 = refineRules (newRules0 @ rs) *)
	   val newRules1 = newRules0 @ rs
	   val newRules = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
					      (newRules1, rs)
	   val newCps = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
					    (Cr.criticalPairs newRules1, cps)
	   val _ =  print "added rules:\n"
	   val _ =  print (Trs.prRules newRules)
       in if null newRules
	  then let val _ = print 
		   val _ =  print "final rules:\n"
		   val _ =  print (Trs.prRules newRules1)
	       in newRules1
	       end
	  else if let val (l,_) = hd (LU.takeMinimals (fn ((l1,r1),(l2,r2)) => SOME (Int.compare (termSize l1, termSize l2))) newRules)
		  in termSize l > max
		  end
	  then raise TermSizeExceedsTheLimit 
	  else cpCompStep max constFn (num+1) newRules1 newCps
       end
*********)

   exception EmptyRuleSet
   fun cpComp rs = 
       let val cps = LU.eliminateDuplication' 
			 (fn ((l1,r1),(l2,r2)) => Subst.identicalModuloRenamingRule (l1,r1) (l2,r2)
						  orelse Subst.identicalModuloRenamingRule (r1,l1) (l2,r2))
			 (Cr.criticalPairs rs)
	   (* fun constFn rs = Rewrite.manyStepsReductSet rs 5 *)
	   fun constFn rs = Rewrite.developOneStepReductSet rs
	   val _ = if null rs then raise EmptyRuleSet else ()
	   val max = let val (l,_) = hd (LU.takeMaximals (fn ((l1,r1),(l2,r2)) => SOME (Int.compare (termSize l1, termSize l2))) rs)
		     in (termSize l) * 2
		     end
	   val newRs0 = cpCompStep max constFn 0 rs cps
	   val newRs1 = L.map (fn (l,r) =>
				  case L.find (Rewrite.isNormalForm newRs0) (Rewrite.oneStepReducts newRs0 r) of
				      SOME r' => (l,r')
				    | NONE => (l,r)) newRs0
	   val newRs = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y) newRs1
	   val _ = print ("Obtained TRSs:\n"  ^ (Trs.prRules newRs))
       in if Trs.areLinearRules newRs andalso null (nonStrongClosedCps newRs)
	  then (println "...linear strongly closed TRS"; (UNC,NONE))
	  else if Trs.areLeftLinearRules newRs andalso null (nonDevelopmentClosedCps newRs)
	  then (println "...left-linear development closed TRS"; (UNC,NONE))
	  else (Unknown,SOME newRs)
       end
       handle DistinctNormalForms (u,v) => 
	      let val _ =  print "convertible distinct normal forms: "
		  val _ =  println (Trs.prEq (u,v))
	      in (NotUNC,NONE)
	      end
	    | OrientationFailed => (println "...orientation failed"; (Unknown,NONE))
	    | TermSizeExceedsTheLimit => (println "...stopped (exceeds the limit)"; (Unknown,NONE))
	    | EmptyRuleSet => (UNC,NONE)


   fun addAuxiliaryRulesStep rs = 
       let val lrs = L.filter (fn (l,r) => (Term.termSize l > Term.termSize r)
					  andalso Trs.isNonDuplicatingRule (l,r)) rs
	   val rls = L.mapPartial (fn (l,r) => if (Term.termSize l < Term.termSize r) andalso Trs.isRewriteRule (r,l)
					       then SOME (r,l) else NONE) rs
	   val cand0a = LU.eliminateDuplication' TermPair.equal (Cr.criticalPairs2 (rls,lrs))
	 (*  val cand0b = L.map (fn (x,y) => (y,x)) (Cr.criticalPairs2 (lrs,rls)) *)
	   val cand0 = cand0a (* @ cand0b *)
	   val cand = L.filter (fn (l,r) => not (Term.equal (l,r))
					    orelse Rewrite.isNormalForm rs l) cand0
       in LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
				   (L.filter Trs.isRewriteRule cand)
       end

   fun addAuxiliaryRules orgSize step rs = (* add orgSize/step count 2020/09/21 *)
       let val add = addAuxiliaryRulesStep rs
	   val _ = debug (fn () => print ("Auxiliary rules to add:\n" ^ (Trs.prRules add)))
	   val rs2 = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y) 
					      (rs @ add)
       in if (L.length rs2) = (L.length rs)
	     orelse step > 2
	     orelse L.length rs2 > orgSize * 2 + 5 (* heulistics *)
	  then rs2
	  else addAuxiliaryRules orgSize (step+1) rs2
       end

   fun refineRules2 rs org = 
       let val nums = L.tabulate (L.length rs, fn x => x)
       in case L.find (fn n => let val (l,r) = L.nth (rs,n)
			       in not (Term.equal (l,r))  (* can not remove *)
				  andalso 
				  let val rs' = L.mapPartial (fn i => if i = n then NONE else SOME (L.nth (rs,i))) nums
						@ org
				      val reductSet = Rewrite.manyStepsReductSet rs' 3 l
				      (* val reductSetL = Rewrite.developOneStepReductSet rs' l  *)
				      (* val reductSetR = Rewrite.developOneStepReductSet rs' r *)
				  in (* not (TS.isEmpty (TS.intersection (reductSetL,reductSetR))) *)
				      TS.member (reductSet,r)
				  end
			       end)
		      nums of
	      SOME n => (println ("remove rule: " ^  (Trs.prRule (L.nth (rs,n))));
			 refineRules2 (L.mapPartial (fn i => if i = n then NONE else SOME (L.nth (rs,i))) nums)) org
	    | NONE => rs
       end


   fun checkStronglyNonOverlapping rs =
       let val rs2 = L.map (fn (l,r) => (Term.linearize l,r)) rs
       in Cr.isNonOverlapping rs2
       end

   exception Failure


   fun checkParallelClosedCondition (inccps,outccps) crules =
       L.all (fn (l,r,cond) => 
		 let val ans = Ctrs.constraintParallelOneStepReducts crules (cond, l)
		 (* val _ = println "reducts of: "
		    val _ = print (LU.toStringCommaCurly Trs.prEq cond)
		    val _ = println (" => " ^ (Term.toString l))
		    val _ = println (LU.toStringCommaCurly Term.toString ans) *)
		 in LU.member' Term.equal r ans
		 end)
	     (* LU.member' Term.equal r 
		(Ctrs.constraintParallelOneStepReducts crules (cond, l))) *)
	     inccps
       andalso
       L.all (fn (l,r,cond) => not (LU.disjoint' Term.equal 
						 (Ctrs.constraintParallelOneStepReducts crules (cond, l),
						  Ctrs.constraintParallelOneStepReducts crules (cond, r))))
	     outccps
	     
   fun checkStrongClosedCondition ccps crules =
       L.all (fn (l,r,cond) =>
		 let (* check one one first for efficiency *)
		     val ansLe = l :: Ctrs.constraintZeroOrOneStepReducts crules (cond, l)
		     (* val _ = println "zero or one step reducts of: " *)
		     (* val _ = print (LU.toStringCommaCurly Trs.prEq cond) *)
		     (* val _ = println (" => " ^ (Term.toString l)) *)
		     (* val _ = println (LU.toStringCommaCurly Term.toString ansLe)  *)
		     val ansRe = r :: Ctrs.constraintZeroOrOneStepReducts crules (cond, r)
		     (* val _ = println "zero or one step reducts of: " *)
		     (* val _ = print (LU.toStringCommaCurly Trs.prEq cond) *)
		     (* val _ = println (" => " ^ (Term.toString r)) *)
		     (* val _ = println (LU.toStringCommaCurly Term.toString ansRe)  *)
		 in not (LU.disjoint' Term.equal (ansLe,ansRe))
		    orelse 
		    let val ansLm = Ctrs.constraintParallelTwoStepsReducts crules (cond, l)
			(* val _ = println "many step reducts of: " *)
			(* val _ = print (LU.toStringCommaCurly Trs.prEq cond) *)
			(* val _ = println (" => " ^ (Term.toString l)) *)
			(* val _ = println (LU.toStringCommaCurly Term.toString ansLm)  *)
			val ansRm = Ctrs.constraintParallelTwoStepsReducts crules (cond, r)
			(* val _ = println "many step reducts of: " *)
			(* val _ = print (LU.toStringCommaCurly Trs.prEq cond) *)
			(* val _ = println (" => " ^ (Term.toString r)) *)
			(* val _ = println (LU.toStringCommaCurly Term.toString ansRm)  *)
		    in not (LU.disjoint' Term.equal (ansLm,ansRe))
		       andalso not (LU.disjoint' Term.equal (ansLe,ansRm))
		    end
		 end)
	     ccps
			    
   exception stronglyClosedConditionalLinearizationFound
   exception parallelClosedConditionalLinearizationFound

   fun checkConfluenceOfConditionalLinearization crules =
       let
	   val _ = debug (fn () => println "CTRS to check:")
	   val _ = debug (fn () => print (Ctrs.prRules crules))

	   val inccps = Cr.condInsideCriticalPairs crules
	   val _ = debug (fn () => println "Conditional inside critical pairs:")
	   val _ = debug (fn () => print (Ctrs.prEqs inccps))

	   val outccps = Cr.condOutsideCriticalPairs crules
	   val _ = debug (fn () => println "Conditional outside critical pairs:")
	   val _ = debug (fn () => print (Ctrs.prEqs outccps))

	   val _ = if (!useParallelClosedLinearization)
		   then let (* val _ = debug (fn () => println "Check by parallel closed linearization") *)
			    val result = checkParallelClosedCondition (inccps,outccps) crules
			in if result
			   then raise parallelClosedConditionalLinearizationFound
			   else ()
			end
		   else ()

	   val _ = if (!useStronglyClosedLinearization)
		   then let (* val _ = debug (fn () => println "Check by strongly closed linearization") *)
			in if Ctrs.isRightLinearRules crules
			      andalso checkStrongClosedCondition (inccps @ outccps) crules
			   then raise stronglyClosedConditionalLinearizationFound
			   else ()
			end
		   else ()
       in ()
       end

   fun proofByConditionalLinearization rs =
       let val crulesList = Ctrs.condKdVLinearizationRulesMult rs
	   val _ = debug (fn () => println "Conditional Linearizations:")
	   val _ = debug (fn () =>
			     L.app (fn crules => print (Ctrs.prRules crules))
				   crulesList)
	   val isRightLinear = (Trs.areRightLinearRules rs)
           val _ = L.app checkConfluenceOfConditionalLinearization crulesList
	   val _ = println (unknown nameParallelClosed)
	   val _ = println (unknown nameStronglyClosed)
       in ()
       end
       handle parallelClosedConditionalLinearizationFound => (println nameParallelClosed; raise SolvedWith UNC)
	   |  stronglyClosedConditionalLinearizationFound => (println nameStronglyClosed; raise SolvedWith UNC)

   fun runDecProcForShallow rs =
       let fun transVar ((x,i):Var.ord_key) = ((Atom.toString x,i):NueVar.key)
	   fun transFun (f:Fun.ord_key) = (Atom.toString f:NueFun.key)
	   fun transTerm (Var (x,_)) = NueTerm.Var (transVar x)
	     | transTerm (Fun (f,ts,_)) = NueTerm.Fun (transFun f, L.map transTerm ts)
	val rs2 = L.map (fn (l,r) => (transTerm l, transTerm r)) rs
       in NueEUN.checkUN_print rs2 
	(* NueUN.checkUN_print rs2  ... original (unefficient) precedure *)
       end

   fun uncSolver opt0 opt1 opt2 opt3 rs = 
       let val _ =  print "TRS:\n"
	   val _ =  print (Trs.prRules rs)

	   val trs = Trs.rulesToTrs rs
	   val _ = if  (#VarCond trs) then ()
		   else (print "Variable condition is not satisfied\n";
			 raise Failure)

	   val test = false (* true => evaluating various methods *)

	   val notUseRev = if test 
			   then not (!useRuleReversing)
			   else true
	   val _ = if test
		   then (useGeneralUncCompletion := false;
			 useShallowDecProc := false)
		   else ();
	   
	   val _ = if (!useShallowDecProc)
		      andalso Trs.areShallowRules rs
		   then let val _ =  debug (fn ()=> println "Call a decision procedure for shallow TRSs")
			in if runDecProcForShallow rs
			   then (raise SolvedWith UNC)
			   else (raise SolvedWith NotUNC)
			end
		   else ()
	   
	   val _ = if (!useStrongNonOverlapping) andalso notUseRev
		   then let val _ =  debug (fn ()=> println "Check strong non-overlapping criterion")
			    val _ = if checkStronglyNonOverlapping rs
	   			    then (println nameStrongNonOverlapping; raise SolvedWith UNC)
				    else (println (unknown nameStrongNonOverlapping))
			in () end
		   else ()
			    
	   val _ = if (!useNonOmegaOverlapping) andalso notUseRev
		   then let val _ =  debug (fn ()=> println "Check non-omega-overlapping criterion")
			    val _ = if InfTerm.omegaNonOverlappingTrs rs
	   			    then (println nameNonOmegaOverlapping; raise SolvedWith UNC)
				    else (println (unknown nameNonOmegaOverlapping))
			in () end
		   else ()
			    
	   val _ = if (!useParallelClosedLinearization) andalso notUseRev
		      orelse (!useStronglyClosedLinearization) andalso notUseRev
		   then let val _ =  debug (fn ()=> println "Perform conditional linearization and check critical pair conditions")
			in if Trs.areLeftLinearRules rs
                           then debug (fn ()=> println "...left-linear rules: not trying")
		           else proofByConditionalLinearization rs
			end
		   else ()


	   val _ = if (!useWeightDecreasingJoinable) andalso notUseRev
		   then let val _ =  debug (fn ()=> println "Perform conditional linearization and check weight-decreasing criterion")
			in if Trs.areLeftLinearRules rs
                           then debug (fn ()=> println "...left-linear rules: not trying")
		           else if not (Trs.areNonDuplicatingRules rs)
			   then debug (fn ()=> println "...duplicating rules: failed")
			   else if Cr.isWeightDecreasingJoinableTrs rs
			   then (println nameWeightDecreasingJoinable; raise SolvedWith UNC)
			   else (println (unknown nameWeightDecreasingJoinable))
			end
		   else ()

	   val _ = if (!useRightReducible) andalso notUseRev
		   then let val _ =  debug (fn ()=> println "Check right-reducible criterion")
			    val _ = if L.all (fn (l,r) => not (Rewrite.isNormalForm rs r)) rs
				    then (println nameRightReducible; raise SolvedWith UNC)
				    else (println (unknown nameRightReducible))
			    in () end
		   else ()

	   val _ = if (!useNonUncByCps) andalso notUseRev
	   	   then let  val _ =  debug (fn ()=> println "Check non-UNC by (reversed) critical pairs")
			     val rs0 = addAuxiliaryRules (L.length rs) 0 rs
	   		     val rs0' = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y) rs0
	   		in case checkNonUNCbyCP rs0' of
	   		       SOME ans => raise SolvedWith ans
	   		     | NONE => ()
	   		end
	   	   else ()

	   val _ = if (!useUncCompletionByStronglyClosed) andalso notUseRev andalso test
		   then let val _ = debug (fn ()=> println "UNC completion by strongly closed Cps")
			    val ans = confStrongClosedCps2 rs
			    val _ = if (ans <> Unknown) 
				    then (println nameUncCompletionByStronglyClosed; raise SolvedWith ans)
				    else (println (unknown nameUncCompletionByStronglyClosed))
			in () end
		   else ()

	   val _ = if (!useUncCompletionByDevelopmentClosed) andalso notUseRev andalso test
		   then let val _ = debug (fn () => println "UNC completion by development closed Cps")
			    val ans = confDevelopmentClosedCps2 rs
			    val _ = if (ans <> Unknown) 
				    then (println nameUncCompletionByDevelopmentClosed; raise SolvedWith ans)
				    else (println (unknown nameUncCompletionByDevelopmentClosed))
			in () end
		   else ()


	   fun eqRule (x,y) = Subst.identicalModuloRenamingRule x y

	   val (reversed,rs2) = 
	       if (!useRuleReversing)
	       then let val _ =  debug (fn ()=> println ("Perform rule reversing transformation"))
                                                         (* ^ "if r is reducibe, |l|<|r|, and l -> r is a rewrite rule," *)
							 (* ^ "replace l -> r with l -> l + r -> l")) *)
			val rs1 = LU.mapAppend (fn (l,r) =>
						   if Term.termSize l < Term.termSize r
						      andalso not (Rewrite.isNormalForm rs r)
						      andalso Trs.isRewriteRule (r,l)
						   then [(l,l),(r,l)]
						   else [(l,r)]) rs
			fun step rs = L.find (fn (l,r) => Term.equal (l,r) andalso
							  let val vs = LU.deleteAll' eqRule (l,r) rs
							  in not (Rewrite.isNormalForm vs l)
							  end) rs
			fun elim rs = case step rs of
					  NONE => rs
					| SOME (l,r) => elim (LU.deleteAll' eqRule (l,r) rs)
			val rs2 = elim rs1
		    in if L.exists (fn (l,r) => not (LU.member' eqRule (l,r) rs2)) rs
		       then (print ("New rules by rule reversing:\n" ^ Trs.prRules rs2);
			     (true,rs2))
		       else (debug (fn ()=> println "    ...not changed");
			     (false,rs))
		    end
	       else (false,rs)


	   val _ = if (!useStrongNonOverlapping) andalso (!useRuleReversing) andalso test
	   	   then let val _ =  debug (fn ()=> println "Check strong non-overlapping criterion")
	   		    val _ = if checkStronglyNonOverlapping rs2
	   			    then (println nameStrongNonOverlapping; raise SolvedWith UNC)
	   			    else (println (unknown nameStrongNonOverlapping))
	   		in () end
	   	   else ()
			    
	   val _ = if (!useNonOmegaOverlapping) andalso (!useRuleReversing) andalso test
	   	   then let val _ =  debug (fn ()=> println "Check non-omega-overlapping criterion")
	   		    val _ = if InfTerm.omegaNonOverlappingTrs rs2
	   			    then (println nameNonOmegaOverlapping; raise SolvedWith UNC)
	   			    else (println (unknown nameNonOmegaOverlapping))
	   		in () end
	   	   else ()
			    
	   val _ = if (!useParallelClosedLinearization) andalso (!useRuleReversing) andalso test
	   	      orelse (!useStronglyClosedLinearization) andalso (!useRuleReversing) andalso test
	   	   then let val _ =  debug (fn ()=> println "Perform conditional linearization and check critical pair conditions")
	   		in if Trs.areLeftLinearRules rs2
                           then debug (fn ()=> println "...left-linear rules: not trying")
	   	           else proofByConditionalLinearization rs2
	   		end
	   	   else ()


	   val _ = if (!useWeightDecreasingJoinable) andalso (!useRuleReversing) andalso test
	   	   then let val _ =  debug (fn ()=> println "Perform conditional linearization and check weight-decreasing criterion")
	   		in if Trs.areLeftLinearRules rs2
                           then debug (fn ()=> println "...left-linear rules: not trying")
	   	           else if not (Trs.areNonDuplicatingRules rs2)
	   		   then debug (fn ()=> println "...duplicating rules: failed")
	   		   else if Cr.isWeightDecreasingJoinableTrs rs2
	   		   then (println nameWeightDecreasingJoinable; raise SolvedWith UNC)
	   		   else (println (unknown nameWeightDecreasingJoinable))
	   		end
	   	   else ()

	   val _ = if (!useRightReducible) andalso (!useRuleReversing) andalso test
	   	   then let val _ =  debug (fn ()=> println "Check right-reducible criterion")
	   		    val _ = if L.all (fn (l,r) => not (Rewrite.isNormalForm rs2 r)) rs2
	   			    then (println nameRightReducible; raise SolvedWith UNC)
	   			    else (println (unknown nameRightReducible))
	   		    in () end
	   	   else ()

	   val _ = if (!useNonUncByCps) andalso (!useRuleReversing) andalso test
	   	   then let  val _ =  debug (fn ()=> println "Check non-UNC by (reversed) critical pairs")
	   		     val rs0 = addAuxiliaryRules (L.length rs2) 0 rs2
	   		     val rs0' = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y) rs0
	   		in case checkNonUNCbyCP rs0' of
	   		       SOME ans => raise SolvedWith ans
	   		     | NONE => ()
	   		end
	   	   else ()

	   val runForRev = if test
			   then (!useRuleReversing)
			   else reversed

	   val _ = if (!useUncCompletionByStronglyClosed) andalso runForRev
		   then let val _ = debug (fn ()=> println "UNC completion by strongly closed Cps")
			    val ans = confStrongClosedCps2 rs2
			    val _ = if (ans <> Unknown) 
				    then (println nameUncCompletionByStronglyClosed; raise SolvedWith ans)
				    else (println (unknown nameUncCompletionByStronglyClosed))
			in () end
		   else ()

	   val _ = if (!useUncCompletionByDevelopmentClosed) andalso runForRev
		   then let val _ = debug (fn () => println "UNC completion by development closed Cps")
			    val ans = confDevelopmentClosedCps2 rs2
			    val _ = if (ans <> Unknown) 
				    then (println nameUncCompletionByDevelopmentClosed; raise SolvedWith ans)
				    else (println (unknown nameUncCompletionByDevelopmentClosed))
			in () end
		   else ()


	   val opRs4 = if (!useGeneralUncCompletion) 
			then let val _ = debug (fn () => println "general UNC completion")
				 val (ans,opCompRs) = cpComp rs2
				 val _ = if (ans <> Unknown)
					 then (println nameGeneralUncCompletion; raise SolvedWith ans)
					 else (println (unknown nameGeneralUncCompletion))
				 val _ = case opCompRs of
					     SOME rs4 => if (!useRuleReversing)
							 then case checkNonUNCbyCP rs4 of 
	   							  SOME ans => raise SolvedWith ans
	   							| NONE => ()
							 else ()
					   | NONE => ()
			     in if isSome opCompRs 
				   andalso L.length (valOf opCompRs) > L.length rs2
				then opCompRs
				else NONE
			     end
			else NONE

	   val _ = if (!useConfluenceProver) andalso reversed 
		   then let val _ = println "Try to prove CR of the result of rule reversing transformation..."
			    val ans = CrSolver.crSolver opt0 opt1 opt2 opt3 rs2
			    val _ =  if ans = Cr.CR
				     then (println "...CR proof of the result of rule reversing transformation is successful.";
					   raise SolvedWith UNC)
				     else ()
			in () end
	   	   else ()


	   val _ = if (!useConfluenceProver) andalso isSome opRs4
		   then let val _ = println "Try to prove CR of the result of UNC completion..."
			    val ans = CrSolver.crSolver opt0 opt1 opt2 opt3 (valOf opRs4)
			    val _ =  if ans = Cr.CR
				     then (println "...CR proof of the result of UNC completion is successful.";
					   raise SolvedWith UNC)
				     else ()
			in () end
	   	   else ()

       in Unknown
       end
       handle SolvedWith ans => ans
	    | Failure => Unknown


   fun repeatDecomposition decomp1 decomp2 isCont rs =
       (* isCont = true to repeat
          isCont = false to stop if unsuccessfull *)
       let val components = decomp1 rs
       in if L.length components = 1
	  then if isCont 
	       then repeatDecomposition decomp2 decomp1 false rs
	       else [rs]
	  else LU.mapAppend (repeatDecomposition decomp2 decomp1 true) components
       end
		   
   fun getDecomposition decomp1 decomp2 rs = repeatDecomposition decomp1 decomp2 true rs

   fun uncSolverWithDecomposition opt0 opt1 opt2 opt3 rules = 
       let val _ =  print "input TRS:\n"
	   val _ =  print (Trs.prRules rules)

           val useDirect = #useDirect opt3
	   val useLayer = #useLayer opt3
	   fun persistentDecomp rs =  L.map (fn (_,ys) => ys) (CrDirect.persistentComponents rs)
	   fun layerPreservingDecomp rs = CrLayer.clp_decompose rs

	   val direct = uncSolver opt0 opt1 opt2 opt3

	   val components = case (useDirect, useLayer) of
				(true,true) => (print "Try persistent and layer-preserving decomposition...\n";
						getDecomposition persistentDecomp layerPreservingDecomp rules)
			      | (true,false) => (print "Try persistent decomposition...\n";
						 persistentDecomp rules)
			      | (false,true) => (print "Try layer-preserving decomposition...\n";
						 layerPreservingDecomp rules)
			      | (false,false) => [rules]

	   fun checkNotUNC [] = Unknown
	     | checkNotUNC (rs::rss) = case direct rs of NotUNC => NotUNC | _ => checkNotUNC rss
	   fun checkAll [] = UNC
	     | checkAll (rs::rss) = case direct rs of UNC => checkAll rss | NotUNC => NotUNC| Unknown => checkNotUNC rss

       in if useDirect orelse useLayer
	  then if (L.length components) > 1
	       then let val _ = print "...result of decompositions:\n"
			val _ = L.app (fn rs => print (Trs.prRules rs)) components
		    in checkAll components
		    end
	       else (print "...decomposition failed.\n";
		     direct rules)
	  else direct rules
       end



   end (* of local *)
   
   end; (* of structure UNC *)
