(* file: comp_e.sml *)
(* description: complition of E *)
(* author: Masaomi Yamaguchi *)

signature NUE_COMP_E = 
sig
    exception Inconsistent
    val rename: NueTrs.eq * NueTrs.eq -> NueTrs.eq	
    val rule1: NueTrs.eq -> NueTrs.eqs -> NueTrs.eqs
    val rule2: NueTrs.eq -> NueTrs.eqs -> NueTrs.eqs -> NueTrs.eqs
    val rule3: NueTrs.eq -> NueTrs.eqs -> NueTrs.eqs -> NueTrs.eqs
    val sameEq: NueTrs.eq * NueTrs.eq -> bool
    val comp: NueTrs.eqs -> NueTrs.eqs
    val compTrs: NueTrs.eqs -> NueTrs.trs
end

structure NueCompE: NUE_COMP_E =
struct 

local 
    structure L = List
    structure LU = NueListUtil
    structure T = NueTerm
    structure S = NueSubst
    structure Var = NueVar
    structure Fun = NueFun
in
exception Inconsistent

fun rename ((l1,r1), (l2,r2)) =
    let (* tのインデックスの最大値を計算する *)
	fun maxIndexOfTerm t =
	    foldl (fn (x,m) => Int.max (Var.index x, m)) 0 (T.vars t)
	(* tの各変数のインデックスにそれぞれnを足す *)
	fun addIndex (T.Var x) n = T.Var (Var.name x, (Var.index x) + n)
	  | addIndex (T.Fun (f,ts)) n = T.Fun (f, addIndexList ts n)
	and addIndexList [] n = []
	  | addIndexList (t::ts) n = (addIndex t n) :: addIndexList ts n
	val maxIndex = Int.max (maxIndexOfTerm l1, maxIndexOfTerm r1)
	val l2' = addIndex l2 (maxIndex + 1)
	val r2' = addIndex r2 (maxIndex + 1)
    in 
	(l2',r2')
    end

(* 推論規則infの仮定が可換の場合に，infをs=tとforall l=r in E' に適用する *)
fun inference (s,t) E' NE inf =
    let fun main (s,t) NE =
	    foldl (fn ((l,r),NE') =>
		      inf (s,t) (r,l) (inf (s,t) (l,r) NE')
		  ) NE E'
    in	
	main (t,s) (main (s,t) NE)
    end

(* 推論規則infの仮定が非可換の場合に，infをs=tとforall l=r in E' に適用する．
   2つの仮定s=tとl=r を入れ替えた場合も考慮する *)
fun inferenceMutual (s,t) E' NE inf =
    let fun inf' st lr NE = inf lr st NE
    in inference (s,t) E' (inference (s,t) E' NE inf) inf'
    end
	
(* 引数:等式s=t，E^の部分集合E'(E^に入ることが確定した等式の集合)*)
fun rule1 (s,t) E' =	
    let
	fun inf (T.Var g,d) (l,r) NE = NE
	  | inf (g,d) (T.Var l,r) NE = NE
	  | inf (g,d) (l,r) NE =
	    let
		val (l',r') = rename ((g,d),(l,r)) (* 等式の変数が被らないようにする *)
	    in
		case (S.unify (l',g)) of NONE => NE
				       | SOME sigma => (S.apply sigma d,S.apply sigma r')::NE
	    end
    in
	inference (s,t) E' [] inf
    end

(* 引数:等式s=t，E^の部分集合E'，rule1で新たに生成された等式集合NE(New Equations) *)
fun rule2 (s,t) E' NE =	
    let
	fun inf (T.Var x,d) (T.Var y,r) NE = let val (T.Var y',r') = rename ((T.Var x,d),(T.Var y,r))
					     in
						 (d,S.apply [(y',T.Var x)] r')::NE
					     end
	  | inf (T.Fun (c,[]),d) (T.Var y,r) NE = let val (T.Var y',r') = rename ((T.Fun (c,[]),d),(T.Var y,r))
						  in
						      (d,S.apply [(y',T.Fun (c,[]))] r')::NE
						  end
	  | inf _ _ NE = NE
    in	
	inferenceMutual (s,t) E' NE inf
    end
	
(* 引数:等式s=t，E^の部分集合E'，rule1,2で新たに生成された等式集合NE(New Equations) *)
fun rule3 (s,t) E' NE =	
    let
	fun inf (T.Fun (g,[]),d) (T.Fun (a,[]),T.Fun (b,[])) NE =
	    if g = a then (T.Fun (b,[]),d)::NE
	    else NE
	  | inf (T.Fun (g,ts),d) ((T.Fun (a,[])),(T.Fun (b,[]))) NE =
	    let fun main ts1 [] = []
		  | main ts1 (T.Fun (c,[])::ts2) =
		    if c = a then
			(L.revAppend (ts1,T.Fun (b,[])::ts2)) :: (main ((T.Fun (c,[]))::ts1) ts2)
		    else
			(main ((T.Fun (c,[]))::ts1) ts2)
		  | main ts1 (t::ts2) = main (t::ts1) ts2
	    in
		foldl (fn (ts',NE') => (T.Fun (g,ts'),d)::NE') NE (main [] ts)
	    end
	  | inf _ _ NE = NE 
    in
	inferenceMutual (s,t) E' NE inf
    end

(* 変数の名前変え(x0,x1,…)をし，同じ等式になるかをチェック *)
fun numbering (s,t) = let val vars = LU.union ((T.vars s),(T.vars t))
			  val sigma = L.mapi (fn (n,x) => (x,T.Var ("x",n))) vars
		      in ((S.apply sigma s),(S.apply sigma t))
		      end
	
(* 変数の名前換えにより，同じルールにできるか判定 *)
fun sameEq ((l1,r1),(l2,r2)) =
    let
	fun main (s1,t1) (s2,t2) = ((numbering (s1,t1)) = (numbering (s2,t2)))
    in
	(main (l1,r1) (l2,r2)) orelse (main (r1,l1) (l2,r2))
    end
	
local
    (* E^の部分集合E'，未チェックの等式集合E，Eに新たに追加する等式集合NE *)
    fun checkInconsistent (s,t) =
	let fun checkMain ((T.Var x),s) = if L.exists (fn y => x = y) (T.vars s) then (s,(T.Var x))
					  else raise Inconsistent
	      | checkMain (l,r) = (r,l)
	in
	    checkMain (checkMain (s,t))
	end
    fun main E' E NE =
	let
	    fun member (s,t) Eq = L.exists (fn eq => sameEq ((s,t),eq)) Eq
	    fun canNotAdd (s,t) E = (checkInconsistent (s,t);
				     if s = t then true
				     else (member (s,t) E') orelse (member (s,t) E))
	    fun add (s,t) E = if canNotAdd (s,t) E then E else (s,t)::E 
	in
	    let val E = foldl (fn (e,es) => add e es) E NE
	    in
		case E of [] => E'
			| ((s,t)::es) => let val E' = (s,t)::E'
					     val NE' = rule3 (s,t) E' (rule2 (s,t) E' (rule1 (s,t) E'))
					 in main E' es NE'
					 end
	    end
	end
in
fun comp E = main [] [] E
end

(* 注意：order rewriteをしないと停止しない *)
(* Eはすでにcompleteと仮定 *)
fun compTrs E =
    let fun main (T.Var x) t rs = (t,(T.Var x))::rs
	  | main s (T.Var x) rs = (s,(T.Var x))::rs
	  | main (T.Fun (c,[])) (T.Fun (c1,[])) rs = if c > c1 then
							 ((T.Fun (c,[])),(T.Fun (c1,[])))::rs
						     else (*c = c1は自明の等式なので入っていない*)  
							 ((T.Fun (c1,[])),(T.Fun (c,[])))::rs
	  | main (T.Fun (c,[])) t rs = (t,(T.Fun (c,[])))::rs
	  | main s (T.Fun (c,[])) rs = (s,(T.Fun (c,[])))::rs
	  | main s t rs = if (numbering (s,t)) = (numbering (t,s)) then
			       (s,t)::rs
			   else (s,t)::(t,s)::rs
    in foldl (fn ((s,t),rs) => main s t rs) [] E
    end
end (* of local *)
end (* of struct *)
