(* file: flat_trs.sml *)
(* description: flat term rewriting systems *)
(* author: Masaomi Yamaguchi *)

signature NUE_FLAT_TRS = 
sig
    type rule = (NueTerm.term * NueTerm.term)
    type eq = (NueTerm.term * NueTerm.term)
    type trs = rule list
    type eqs = eq list
    val isFlatTrs: NueTrs.trs -> bool
    val function_symbols_with_arity: trs -> (NueFun.key * int) list
    val max_arity: (NueFun.key * int) list -> int
    val NC_function_symbols_with_arity: (NueFun.key * int) list  -> (NueFun.key * int) list
    val constant: (NueFun.key * int) list  -> NueTerm.term list
    val enum_c: (NueFun.key * int) list -> int	     
end

structure NueFlatTrs: NUE_FLAT_TRS =
struct

local 
    structure T = NueTerm
    structure LU = NueListUtil
    structure AL = NueAssocList
    structure Fun = NueFun
    structure Var = NueVar
    structure Flatting = NueFlatting
in
type rule = (T.term * T.term)
type eq = (T.term * T.term)
type trs = rule list
type eqs = eq list
	      
fun isFlatTrs rs = List.all (fn (l,r) => Flatting.isFlat l andalso Flatting.isFlat r) rs

(* Make a list of all function symbols (including constant). Return pairs of a function symbol and its arity *)
fun function_symbols_with_arity rs =
    let fun sub (T.Var x) = []
	  | sub (T.Fun (f,ts)) = (f, length ts)::
				 foldl (fn (T.Fun (c,[]),list)  => (c,0)::list
				       |  (_,list) => list) [] ts (* Duplications will be omitted at largeUnion in main *)
	fun main [] = []
	  | main ((l,r)::rs) = case AL.largeUnion [sub l, sub r, main rs] of
				   NONE => raise Fail "Arity of a function symbol is defferent"
				 | SOME list => list 
    in
	main rs
    end

(* ルール中に出てくる関数記号の最大のアリティを計算する *)
fun max_arity fs = foldl (fn ((f,n), max_n) => Int.max(n,max_n)) 0 fs

(* 定数以外の関数記号一覧を作成する．arityと関数記号の組のリストを返す *)
fun NC_function_symbols_with_arity fs = List.filter (fn (f,n) => n > 0) fs

(* 定数一覧を作成する *)
fun constant fs = foldl (fn ((f,n),list) => if (n = 0) then (T.Fun (f,[]))::list else list) [] fs

(* ルール中に出てくる定数の個数を数える *)
fun enum_c fs = foldl (fn ((f,0), count) => count + 1
		      |  ((f,_), count) => count) 0 fs		
		      
end (* of local *)

end


