(* file: flatting.sml *)
(* description: flatting rules *)
(* author: Masaomi Yamaguchi *)

signature NUE_FLATTING = 
sig val isFlat: NueTerm.term -> bool
    val isNCGroundFlat: NueTerm.term -> bool
    val stepR: NueTrs.eq -> (NueTrs.trs * NueTrs.trs * int) -> (NueTrs.trs * NueTrs.trs)
    val flattingR: NueTrs.trs -> NueTrs.trs
    val replacel0c: NueTrs.trs ->  NueTerm.term -> NueTerm.term -> NueTrs.trs
    val stepL: NueTrs.eq -> (NueTrs.trs * NueTrs.trs * int) -> (NueTrs.trs * NueTrs.trs)
    val flattingL: NueTrs.trs -> NueTrs.trs
    val flatting: NueTrs.trs -> NueTrs.trs	  
end

structure NueFlatting: NUE_FLATTING =
struct 

local 
    structure L = List
    structure LU = NueListUtil
    structure S = NueSubst
    structure T = NueTerm
    structure R = NueRewrite
in

fun isFlat (T.Var x) = true
  | isFlat (T.Fun (f,ts)) = let fun isXC (T.Var x) = true
				| isXC (T.Fun (c,[])) = true
				| isXC _ = false
			    in List.all isXC ts
			    end

(* non-constant ground flat termかを調べる *)
fun isNCGroundFlat (T.Var x) = false
  | isNCGroundFlat (T.Fun (c,[])) = false
  | isNCGroundFlat (T.Fun (f,ts)) = let fun isC (T.Var x) = false
					  | isC (T.Fun (c,[])) = true
					  | isC _ = false
				    in List.all isC ts
				    end

(* ルールl->rをR-flatにする1ステップ．R_tsはR-flatルールの集合，tsは未チェクのルールの集合 *)
fun stepR (l,r) (r_ts,ts,i) =
    let fun main [] = raise Fail "Not shallow rules!"
	  | main (p::ps) = let val r0 = (T.subterm r p)
			   in
			       if isNCGroundFlat r0 then
				   let val c = T.Fun ("!cr_"^Int.toString i,[])
				   in
				       ((c,r0)::r_ts,(l,T.replaceSubterm r p c)::ts)
				   end
			       else main ps
			   end
    in
	main (T.pos r)
    end

fun flattingR ts =
    let fun main (r_ts,[]) i = r_ts
	  | main (r_ts,(l,r)::ts) i = if isFlat r then main ((l,r)::r_ts,ts) i
				      else main (stepR (l,r) (r_ts,ts,i)) (i+1)
    in
	main ([],ts) 0
    end
	
fun replacel0c ts l0 c =
    let fun pos_l0 t = LU.revFilter (fn p => (T.subterm t p) = l0) (T.pos t)
	fun replace t = foldl (fn (p,t') => T.replaceSubterm t' p c) t (pos_l0 t)
    in
	foldl (fn ((l,r),ts') => ((replace l),r)::ts') [] ts
    end

(* ルールl->rをL-flatにする1ステップ．L_tsはL-flatルールの集合，tsは未チェクのルールの集合 *)
fun stepL (l,r) (l_ts,ts,i) =
    let fun main [] = raise Fail "Not shallow rules!"
	  | main (p::ps) = let val l0 = (T.subterm l p)
			   in
			       if isNCGroundFlat l0 then
				   let val c = T.Fun ("!cl_"^Int.toString i,[])
				       val R = l_ts @ ((l,r)::ts)
				       val ts' = replacel0c ((l,r)::ts) l0 c
				       val l_ts' = replacel0c l_ts l0 c
				   in		
				       if R.isNF R l0 then
					   ((l0,c)::l_ts', ts')
				       else
					   ((c,c)::(l0,c)::l_ts', ts') 
				   end
			       else main ps
			   end
    in
	main (T.pos l)
    end
	
fun flattingL ts =
    let fun main (l_ts,[]) i = l_ts
	  | main (l_ts,(l,r)::ts) i = if isFlat l then main ((l,r)::l_ts,ts) i
				      else main (stepL (l,r) (l_ts,ts,i)) (i+1)
    in
	main ([],ts) 0
    end	

fun flatting ts = (flattingL o flattingR) ts
	
end (* of local *)
   
end
