(* file: list_util.sml *)
(* description: utility functions for list *)
(* author: Takahito Aoto *)

(* 各インデックスは1から始まるものとする *)

signature NUE_LIST_UTIL = 
sig 
    (* functions for set *)
    val member: ''a  -> ''a list -> bool
    val add: ''a  -> ''a list -> ''a list
    val union: ''a list * ''a list -> ''a list
    (* 要素の順番を保存したままunionする *)
    val prevUnion: ''a list * ''a list -> ''a list
    val difference: ''a list * ''a list -> ''a list      
    val eliminate: ''a -> ''a list -> ''a list	  
    (* 集合の集合を1つの集合にする *)
    val largeUnion: ''a list list -> ''a list	  
    (* リストのリストを1つのリストにする *)
    val flatten: 'a list list -> 'a list
    (* リストの重複をなくす *)
    val unique: ''a list -> ''a list
    (* リストの各要素にfを適用する．ただし，fではインデックスiを使用できる *)
    val mapIndex: ('a * int -> 'b) -> 'a list -> 'b list
    (* リスト内のi番目の要素にのみfを適用する *)
    val mapSpecified: ('a -> 'a) -> 'a list -> int -> 'a list option
    (* リストのi番目の要素を取り出す *)
    val get: 'a list -> int -> 'a option
    (* リストのそれぞれの要素にfを適用したえで，flattenする *)
    val mapAppend: ('a -> 'b list) -> 'a list -> 'b list
    (* リストから最小値を取り出す．この際，listの要素xにfを適用したときの大きさを
       lsterによって比較する *)
    val min: ('a -> 'b) -> 'a list -> ('b * 'b -> bool) -> 'a
    (* リストから最小値を取り出す．この際，listの要素xにfを適用したときの大きさを
       "<"によって比較する *)						       
    val mini: ('a -> int) -> 'a list -> 'a
    val sort: ('a * 'a -> bool) -> 'a list -> 'a list
    val revFilter: ('a -> bool) -> 'a list -> 'a list
    (* f(a,b)=trueとなる a,b in xsが存在する時，SOME((a,b))を返す．そうでなければNONEを返す．*)
    (* ただし，a!=b, fのa,bは可換とする *)		 
    val exists_true_pair: ('a * 'a -> bool) -> 'a list -> ('a * 'a) option					 
    val toStringCommaSpaceBrace: ('a -> string) -> 'a list  -> string
    val toStringCommaLnSquare: ('a -> string) -> 'a list  -> string
end
    
structure NueListUtil : NUE_LIST_UTIL =
struct 

fun member x ys = List.exists (fn y => x = y) ys

(* 集合としての演算 *)
fun add x xs = if member x xs then xs else x::xs
fun union (xs,ys) = foldr (fn (x,ys') => add x ys') ys xs
fun prevUnion (xs,ys) = List.rev(foldl (fn (y,list) => add y list) (foldl (fn (x,list) => add x list) [] xs) ys)
fun largeUnion xs = foldr (fn (a,b) => union (a,b)) [] xs
fun difference (xs,ys) = foldr (fn (x,xs') => if (member x ys) then xs' else x::xs') [] xs
fun eliminate x xs = foldr (fn (y,xs') => if x=y then xs' else y::xs') [] xs

fun flatten l = foldr (fn (a,b) => a @ b) [] l
fun unique l = foldr (fn (a,b) => if member a b then b else a :: b) [] l
		     
fun mapIndex f l = let fun g [] i = []
			 | g (x::xs) i = f (x,i) :: g xs (i+1)
		   in g l 1
		   end

fun mapSpecified f l i = if i > length l then NONE
			 else SOME (mapIndex (fn (a,j) => if i = j then f a else a) l)
				   
fun get [] n = NONE
  | get (x::xs) 1 = SOME x
  | get (x::xs) n = get xs (n-1)
			
fun toStringBuilt prElm (start,sep,stop) xs =
    let fun toStringSub [] = ""
          | toStringSub [x] = (prElm x)
          | toStringSub (x::xs)= (prElm x) ^ sep ^ (toStringSub xs)
    in  start ^ (toStringSub xs) ^ stop
    end

fun mapAppend f [] = []
  | mapAppend f (x::xs) = (f x) @ (mapAppend f xs) 

fun min f list lster =
    let fun main f [x] lster = (x,f x)
	  | main f (x::xs) lster =  let val (m,v) = main f xs lster
					val fx = f x
				    in
					if lster (fx,v) then (x,fx)
					else (m,v)
				    end
    in #1 (main f list lster)
    end

fun mini f list = min f list (op <)

(*fun merge xs ys = foldl (fn (x,ys') => x::ys') ys xs*)

fun sort f [] = []
  | sort f (x::xs) =
    let
        fun partition([],a,b) = (a,b)
	  | partition(y::ys,a,b) =
            if f(y,x) then partition(ys,y::a,b)  
            else partition(ys,a,y::b)
			  
        val (a,b) = partition(xs,[],[])
    in
        sort f a @ (x::sort f b)
    end

fun revFilter f xs = foldl (fn (x,xs') => if f x then x::xs' else xs') [] xs
			   
(* f(a,b)=trueとなるa,b in xsが存在する時，SOME((a,b))を返す．そうでなければNONEを返す．*)
(* ただし，a!=b, fのa,bは可換とする *)
fun exists_true_pair f [] = NONE
  | exists_true_pair f (x::xs) =
    let
	fun main [] = NONE
	  | main (y::ys) = if f(x,y) then SOME(x,y) else main ys
    in
	case main xs of
	    SOME (x,y) => SOME (x,y)
	  | NONE =>  exists_true_pair f xs
    end
	
fun toStringCommaSpaceBrace prElm xs = toStringBuilt prElm ("{",", ","}") xs
fun toStringCommaLnSquare prElm xs = toStringBuilt prElm ("   [ ",",\n     "," ]\n") xs
end
