(* file: order_rewrite.sml *)
(* description: ordered rewriting *)
(* author: Masaomi Yamaguchi *)

signature NUE_ORDER_REWRITE = 
sig val posLeftInMost: NueTerm.term -> NueTerm.position list
    val rootStep: NueTrs.trs -> NueTerm.term -> NueTerm.term option
    val step: NueTrs.trs -> NueTerm.term -> NueTerm.position -> NueTerm.term option
    val listep: NueTrs.trs -> NueTerm.term -> NueTerm.term option
    val isNF: NueTrs.trs -> NueTerm.term -> bool
    val listepsToNF: NueTrs.trs -> NueTerm.term -> unit
    val linf: NueTrs.trs -> NueTerm.term -> NueTerm.term
    val isJoinable: NueTrs.trs -> NueTerm.term * NueTerm.term -> bool
end

structure NueOrderRewrite: NUE_ORDER_REWRITE =
struct 

local 
    structure L = List
    structure LU = NueListUtil
    structure S = NueSubst
    structure T = NueTerm
    structure Trs = NueTrs
    structure PathOrder = NuePathOrder
    open PrintUtil
in

(* "!" はasciiコード中で最小の文字(制御文字は除く) *)
val bottom = T.Var ("!",0)

exception notNF

(* 最内最左の順に並んだpositionのリストを返す *)
fun posLeftInMost t =
    let
	(* 組(x,v)のリストのvの最大値を返す *)
	fun max xs = foldl (fn ((x,v), mav) => if v > mav then v else mav) 0 xs 
	(* 組(x,v)のリストPとvの最大値mavを受け取り，vの降順にxを並び替える．*)
	(* ただし，vが等しいものはPの順番通りに並ぶ *)
	fun sort xs =
	    let fun filter xs n = foldr (fn ((x,v),(Eq,Gr)) => if v = n then (x::Eq,Gr) else (Eq,(x,v)::Gr))
					([],[]) xs
		fun main [] n = []
		  | main xs n =
		    let val (Eq,Gr) = filter xs n 
		    in
			Eq @ (main Gr (n-1))
		    end
	    in
		main xs (max xs)
	    end
    in sort (T.posWithDepth t)
    end
	
	

(* rootStep rs term : TRS rs による根位置での書き換え *)
fun rootStep [] term = NONE
  | rootStep ((l,r)::rs) term =
    case S.match l term
     of SOME sigma => let val sigma' = foldl (fn (x,sigma') => (x,bottom) :: sigma')
					     sigma (LU.difference (T.vars r,T.vars l))
			  val lsigma = S.apply sigma' l
			  val rsigma = S.apply sigma' r
		      in
			  if PathOrder.lpoGt (lsigma,rsigma) then 
			      SOME rsigma
			  else
			      rootStep rs term
		      end
      | NONE => rootStep rs term

(* step rs term ps : TRS rs による位置psでの書き換え *)
fun step rs term ps = let val rsigmaOp = rootStep rs (T.subterm term ps)
			  val c = T.makeContext term ps
		      in
			  case rsigmaOp of
			      NONE => NONE
			    | SOME rsigma => SOME (T.fillContext c rsigma)
		      end


			  
(* listep rs term : TRS rsによる最左最外書き換え *)
fun listep rs term = 
    let fun main rs [] = NONE
	  | main rs (p::ps) = case (step rs term p) of
				  NONE => main rs ps
			       |  SOME t => SOME t 
    in
	main rs (posLeftInMost term)
    end
	
fun isNF rs (T.Var x) = true
  | isNF rs (T.Fun (f,ts)) = case rootStep rs (T.Fun (f,ts)) of
				 SOME t' => false
			       | NONE => List.all (isNF rs) ts
						  
fun listepsToNF rs term =
  let fun main term = case (listep rs term) of
			  NONE => ()
			| SOME t => (print ("->R^ "^T.toString t^"\n"); main t)
  in
      (print ("     "^T.toString term^"\n"); main term)
  end

fun linf rs (T.Var x) = T.Var x
  | linf rs (T.Fun (f,ts)) = let val ts' = map (fn t => linf rs t) ts
			     in case rootStep rs (T.Fun (f,ts')) of
				    SOME t' => linf rs t'
				  | NONE => T.Fun (f,ts')
			     end

fun isJoinable rs (t1,t2) = let val t1' = linf rs t1
				val t2' = linf rs t2
			    in
				t1' = t2'
			    end    
end (* of local *)

end
