(* file: subst.sml *)
(* description: substutition *)
(* author: Takahito Aoto *)

signature NUE_SUBST = 
sig 
    type subst
    val apply: subst -> NueTerm.term -> NueTerm.term
    val dom: subst -> NueVar.key list
    val toString: subst -> string
    val fromString: string -> subst
    val match: NueTerm.term -> NueTerm.term -> subst option
    val compose: subst -> subst -> subst
    val unify: NueTerm.term * NueTerm.term -> subst option
end

structure NueSubst: NUE_SUBST =
struct 

local 
    structure L = List
    structure AL = NueAssocList
    structure LU = NueListUtil
    structure T = NueTerm
    structure S = NueSymbol
    structure Var = NueVar
in

type subst = (Var.key * T.term) list

fun apply sigma (T.Var x) = (case AL.find x sigma of NONE => T.Var x
						   | SOME t => t)
  | apply sigma (T.Fun (f,ts)) = T.Fun (f,map (fn t => apply sigma t) ts)

fun dom sigma = foldr (fn ((k,t),d) => case t of (T.Var x) =>
						 if k=x then d else LU.add k d
					       | _ => LU.add k d
		      ) [] sigma

fun toString sigma = LU.toStringCommaSpaceBrace (fn (k,t) => "?"^Var.toString k^" := "^T.toString t) sigma


fun fromString str = 
    let val tps = T.readMultipleKeySepartedTermPairs ("{",",","}") ":=" str
	fun getVar s  = case T.root s of 
			    S.V x => x 
			  | S.F _ => raise Fail ("Syntax error: var expected " ^ (T.toString s))
    in L.map (fn (s,t) => (getVar s, t)) tps
    end

fun match pattern term =
  let fun main [] sigma = SOME sigma
	| main ((T.Var x, t0)::rest) sigma =
	  let fun f x t0 sigma  = case (AL.add (x,t0) sigma) of
				      NONE => NONE
				    | SOME sigma' => main rest sigma'
	  in f x t0 sigma end
	| main ((T.Fun (f,ts), T.Fun (g,us))::rest) sigma = if f = g then
								main (decompose (ts,us) rest) sigma
							    else NONE
	| main ((T.Fun _, T.Var _)::_) sigma = NONE
      and decompose ([],[]) E = E
	| decompose ((t::ts),(u::us)) E = decompose (ts,us) (LU.add (t,u) E)
  in main [(pattern,term)] []
  end
      
fun compose sigma rho = let val rho' =
				foldr (fn (x,rh) => case (AL.add (x,T.Var x) rh) of
							NONE => rh
						      | SOME rh' => rh'
				      ) rho (dom sigma)
			    fun main rho' =
			      map (fn (y,u) => (y,apply sigma u)) rho'
			in
			    main rho'
			end

local fun eqVar (x,T.Var y) = x = y
	| eqVar _ = false
      fun decompose ([],[]) E = E
	| decompose ((t::ts),(u::us)) E = decompose (ts,us) (LU.add (t,u) E)
      fun applyE sigma E = map (fn (t,s) => (apply sigma t, apply sigma s)) E
in
fun unify (term1, term2) =
  let fun main [] sigma = SOME sigma
	| main ((T.Var x, t)::rest) sigma = if eqVar (x,t) then main rest sigma
					    else if LU.member x (T.vars t) then NONE
					    else let val rho = [(x,t)]
						 in main (applyE rho rest) (compose rho sigma)
						 end
	| main ((T.Fun (f,ts), T.Fun (g,us))::rest) sigma = if f = g then main (decompose (ts,us) rest) sigma
							    else NONE
	| main ((t,T.Var x)::rest) sigma = main ((T.Var x, t)::rest) sigma
  in main [(term1,term2)] []
  end
end
			    
end (* of local *)
end
