(* file: trs.sml *)
(* description: first-order term rewriting systems *)
(* author: Takahito Aoto *)

signature NUE_TRS = 
sig
    type rule = (NueTerm.term * NueTerm.term)
    type eq = (NueTerm.term * NueTerm.term)
    type trs = rule list
    type eqs = eq list

    val prRule: rule -> string
    val rdRule: string -> rule

    val prRules: trs -> string
    val rdRules: string -> trs

    val prEq: eq -> string
    val rdEq: string -> eq

    val prEqs: eqs -> string
    val rdEqs: string -> eqs

    val rename: (NueTerm.term * NueTerm.term) * (NueTerm.term * NueTerm.term)
		-> (NueTerm.term * NueTerm.term) * (NueTerm.term * NueTerm.term)

    val filter: NueFun.key -> trs -> trs
end

structure NueTrs: NUE_TRS =
struct

local 
    structure SU = NueStringUtil
    structure Fun = NueFun
    structure Var = NueVar
    structure T = NueTerm
    structure LU = NueListUtil

in
type rule = (T.term * T.term)
type eq = (T.term * T.term)
type trs = rule list
type eqs = eq list

fun prRule (l,r) = T.toString l ^ " -> "  ^ T.toString r
fun rdRule str = T.readKeySeparatedTermPair "->" str 

fun prRules rs = LU.toStringCommaLnSquare prRule rs
fun rdRules str = T.readMultipleKeySepartedTermPairs ("[",",","]") "->" str

fun prEq (l,r) = T.toString l ^ " = "  ^ T.toString r
fun rdEq str = T.readKeySeparatedTermPair "=" str 

fun prEqs es = LU.toStringCommaLnSquare prEq es
fun rdEqs str = T.readMultipleKeySepartedTermPairs ("[",",","]") "=" str

fun rename ((l1,r1), (l2,r2)) =
  let (* tのインデックスの最大値を計算する *)
      fun maxIndexOfTerm t =
	    foldl (fn (x,m) => Int.max (Var.index x, m)) 0 (T.vars t)
      (* tの各変数のインデックスにそれぞれnを足す *)
      fun addIndex (T.Var x) n = T.Var (Var.name x, (Var.index x) + n)
	| addIndex (T.Fun (f,ts)) n = T.Fun (f, addIndexList ts n)
      and addIndexList [] n = []
	| addIndexList (t::ts) n = (addIndex t n) :: addIndexList ts n
      val maxIndex = Int.max (maxIndexOfTerm l1, maxIndexOfTerm r1)
      val l2' = addIndex l2 (maxIndex + 1)
      val r2' = addIndex r2 (maxIndex + 1)
  in 
      ((l1,r1), (l2',r2'))
  end

fun filter f rs = List.filter (fn (T.Fun(g,ts),r) => f = g
				 | (_,r) => false) rs

end (* of local *)

end


