(******************************************************************************
 * Copyright (c) 2020, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/rwchecker/unr.sml
 * description: ingredients for checking unique normal form property w.r.t. reduction
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature UNR = 
sig
    val runDebug: bool ref
    datatype UnrResult = UNR | NotUNR | Unknown

    val unrSolver: Solver.options
                  -> DpSolver.options
                  -> PoSolver.options
                  -> CrSolver.options
                  -> (Atom.atom * int) list option
                  -> (Term.term * Term.term) list 
                  -> UnrResult

end;

structure Unr : UNR = 
   struct
   local 
       structure T = Term
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure PU = PrintUtil
       structure TP = TermPair
       structure TPS = TermPairSet
       structure TS = TermSet
       open PrintUtil
   in

   val runDebug = ref false : bool ref
   fun debug f = if !runDebug then f () else ()
   exception UnrError

   datatype UnrResult = UNR | NotUNR | Unknown
   exception SolvedWith of UnrResult

   val useStrongNonOverlapping = ref true
   val useRightReducible = ref true
   val useShallowDecProcForUNC = ref true
   val useNonUnrByCps = ref true
   val useKB = ref true
   val useTrivial = ref true

   val useTransByStronglyClosed = ref true
   val useTransByParallelClosed = ref true

   val useLinearApprox = ref true

   val useCollapsMapping = ref true
   val useSRRelimination = ref true

   val useConfluenceProver = ref true

   val nameStrongNonOverlapping = "Strongly Non-Overlapping"
   val nameRightReducible = "Right-Reducible"

   fun unknown s = "unknown " ^ s


   (***  危険対を使った NotUNR の証明 ***)
   fun checkExistenceOfNomalFormsCP rs =
       let val cps = Cr.criticalPairs rs 
	   val witness = L.filter (fn (u,v) => not (Term.equal(u,v)) 
					       andalso (Rewrite.isNormalForm rs u) 
					       andalso (Rewrite.isNormalForm rs v)) cps
       in if null witness
	  then Unknown
	  else let val _ = print "CP consisting of normal forms:\n"
		   val _ =  print (Trs.prEqs witness)
	       in NotUNR
	       end
       end

   (* simple counter example check from CP (reducts) *)
   val maxLen = 3 

   fun possibleNormalForms rs maxLen term =
       let val tset = Rewrite.manyStepsReductSet rs maxLen term
       in L.filter (Rewrite.isNormalForm rs) (TS.listItems tset)
       end

   (*** NotUNR の証明 ****)
   fun addAuxiliaryRulesStep rs = 
       let val lrs = L.filter (fn (l,r) => (Term.termSize l >= Term.termSize r)
					  andalso Trs.isNonDuplicatingRule (l,r)) rs
	   val rls = L.mapPartial (fn (l,r) => if (Term.termSize l <= Term.termSize r) andalso Trs.isRewriteRule (r,l)
					       then SOME (r,l) else NONE) rs
	   val cand0a = LU.eliminateDuplication' TermPair.equal (Cr.criticalPairs2 (rls,lrs))
	   val cand0 = cand0a
	   val cand = L.filter (fn (l,r) => not (Term.equal (l,r)))  cand0
       in LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
				   (L.filter Trs.isRewriteRule cand)
       end

   fun addAuxiliaryRules orgSize step rs = 
       let val add = addAuxiliaryRulesStep rs
	   val rs2 = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y) 
					      (rs @ add)
       in if (L.length rs2) = (L.length rs)
	     orelse step > 2
	     orelse L.length rs2 > orgSize * 2 + 5 (* heulistics *)
	  then rs2
	  else (debug (fn () => print ("Auxiliary rules to add:\n"
				       ^ (Trs.prRules (L.drop (rs2, L.length rs)))));
		addAuxiliaryRules orgSize (step+1) rs2)
       end


   exception DistinctNormalForms of Term.term * Term.term 

   fun checkNonUNRbyCP rs = 
       let val _ =  print "Check distinct normal forms in critical pair closure"
	   val _ =  debug (fn () => print "\n")
	   fun checkOne u = 
	       let (* val _ = debug (fn () => println ("check reducts of " ^ (Term.toString u))) *)
		   val uReducts = TS.listItems (Rewrite.manyStepsReductSet rs 4 u)
		   val unf = L.filter (Rewrite.isNormalForm rs) uReducts
	       in if L.length unf > 1
		  then raise DistinctNormalForms (hd unf, hd (tl unf))
		  else if null unf then NONE
		  else SOME (hd unf)
	       end 

	   fun checkTwo (u,v) = 
	       let val _ = debug (fn () => println ("check " ^ (Trs.prEq (u,v))))
	       in case (checkOne u, checkOne v) of
		      (SOME u2, SOME v2) => if Term.equal (u2,v2) 
					    then ()
					    else raise DistinctNormalForms (u2,v2)
		    | _ => ()
	       end

	   val cps = ListMergeSort.sort
			 (fn ((x1,x2),(y1,y2)) => T.termSize x1 + T.termSize x2  > T.termSize y1 + T.termSize y2)
			 (L.filter (not o Term.equal) (Cr.criticalPairs2 (rs,rs)))

	   val cps2 = if L.length cps > 100 then L.take (cps,100) else cps (* heulistics *)
	   val _ = debug (fn () => print ("Cps:\n" ^ Trs.prEqs cps2))

	   val _ = L.app checkTwo cps2

       in (println "...failed"; NONE)
       end
       handle DistinctNormalForms (u,v) => 
	      let val _ =  debug (fn () => print "    ...found")
		  val _ =  print "\ndistinct normal forms of a term: "
		  val _ =  println (Trs.prEq (u,v))
	      in SOME NotUNR
	      end


   fun checkCounterExample rs (reducts,term) =
       let val nfs = LU.eliminateDuplication' Term.equal (LU.mapAppend (possibleNormalForms rs maxLen) reducts)
	   val _ = case L.find (fn x => not (Term.equal (x,term))) nfs of
		      SOME term2 => (print ("CounterExample:\n" ^ Trs.prEq (term2,term) ^ "\n");
				     raise SolvedWith NotUNR)
		    |  NONE => ()
	   val _ = if L.length nfs > 1
		   then (print ("CounterExample:\n" ^ Trs.prEq (hd nfs, hd (tl nfs)) ^ "\n");
			 raise SolvedWith NotUNR)
		   else ()
       in ()
       end


   (* approximating e.g. f(x,x) -> a  by f(x,y) -> a when f(x,y) is not normal *)
   fun nLLapproximation rs (l,r) =
       let val nlvarSetInLHS = Term.nonLinearVarSetInTerm l
	   val varSetInRHS = Term.varSetInTerm r
	   fun select (T.Var (x,ty), T.Var (y,_)) = if VS.member (nlvarSetInLHS,x)
						       andalso not (VS.member (varSetInRHS,x))
						    then T.Var (y,ty)
						    else T.Var (x,ty)
	     | select (T.Fun (f,xs,ty), T.Fun (_,ys,_)) = T.Fun (f,L.map select (LP.zip (xs,ys)),ty)
	     | select (t,_) = t (* should not come here *)
	   val newLHS = select (l,Term.linearize l)
       in if Rewrite.isNormalForm rs newLHS
	  then (l,r) (* approximation not admissible, and return original anyway *)
	  else (* normal form でなくても，l -> l となるだけのときは近似しない *)
	      let val reducts = Rewrite.oneStepReducts (LU.deleteOne' TP.equal (l,r) rs) l
	      in if LU.member' T.equal l reducts
		 then (l,r)
		 else (newLHS,r)  (* use approximaiton *)
	      end
       end

   fun appNonLL rs (l,r) = if Term.isLinearTerm l
			   then (l,r)
			   else nLLapproximation rs (l,r)

   fun appLinearApproximation rs = L.map (appNonLL rs) rs

   exception StopUNRbySC of (Term.term * Term.term) list 
   val LimitIteration = 5

(** not used
   fun removeAuxiliaryRules rs = 
       let fun main (used,[]) = rev used
	     | main (used,(l,r)::rest) =
	       if LU.member' Term.equal r (Rewrite.developTwoStepsReducts used l)
	       then main (used,rest)
	       else main ((l,r)::used,rest)
       in main ([],rs)
       end
**)

   fun proveUNRbyStrongClosed orgSize step (rs,aux) = 
       let
	   val _ =  print "checking TRS:\n"
	   val _ =  print (Trs.prRules rs)
	   val _ =  print "with auxiliary rules:\n"
	   val _ =  print (Trs.prRules aux)

          (* 危険対による合流性のチェックでは， l -> l を除く *)
	   val rs2pre = L.filter (not o Term.equal) (rs @ aux)
	   val _ = print ("Trivial rules are removed for checking CR\n")

	   (* val _ = if not (Trs.areLinearRules rs2)
		   then raise StopUNRbySC rs2 (* proof stopped *)
		   else () *)

	   val rs2 = if Trs.areLinearRules rs2pre
		     then rs2pre
		     else if (!useLinearApprox)
		     then let val _ = print ("Try linear approximation\n")
			      val rs2rev = appLinearApproximation rs2pre
			  in if Trs.areLinearRules rs2rev
			     then rs2rev
			     else raise StopUNRbySC rs2rev (* proof stopped *)
			  end
		     else raise StopUNRbySC rs2pre (* proof stopped *)

	   val cps2 = Cr.criticalPairs rs2
	   val _ = if L.all (Cr.isStrongClosed rs2) cps2
		   then (println ("CP:\n" ^ Trs.prEqs cps2);
			 print "(approximated) TRS is linear, strongly closed; hence confluent\n";
                        raise SolvedWith UNR)
		   else ()
   
	   val _ = if step >= LimitIteration
		      orelse (L.length rs2) > (orgSize + 2) * (orgSize + 2)
		   then raise StopUNRbySC rs2 (* proof stopped *)
		   else ()

          (* 以下では rs に戻る *)
	   val cps = Cr.criticalPairs rs
	   val nonclosedcps = L.filter (not o Cr.isStrongClosed rs) cps

	   fun checkcp isOrgTRS (s,t) =
	       case (Rewrite.isNormalForm rs s, Rewrite.isNormalForm rs t) of
		   (true,true) => if isOrgTRS
				  then (print ("CounterExample:\n" ^ Trs.prEq (s,t) ^ "\n");
					raise SolvedWith NotUNR)
				  else ([],[])
		|  (false,false) => 
		   let val sReducts = Rewrite.developOneStepReducts rs s
		       val tReducts = Rewrite.developOneStepReducts rs t
		       val (ra,sa) = if LU.member' Term.equal t sReducts
				     then ([(s,t)],[])
				     else if Trs.isRewriteRule (s,t)
				     then ([],[(s,t)])
				     else ([],[])
		       val (rb,sb) = if LU.member' Term.equal s tReducts
				     then ([(t,s)],[])
				     else if Trs.isRewriteRule (t,s)
				     then ([],[(t,s)])
				     else ([],[])
		   in (ra@rb,sa@sb)
		   end
		|  (true,false) => 
		   let val tReducts = Rewrite.developOneStepReducts rs t
		       val (rb,sb) = if LU.member' Term.equal s tReducts
				     then ([(t,s)],[])
				     else if Trs.isRewriteRule (t,s)
				     then ([],[(t,s)])
				     else ([],[])
		       val _ = if isOrgTRS then checkCounterExample rs (tReducts,s) else ()
		   in (rb,sb)
		   end
		|  (false,true) => 
		   let val sReducts = Rewrite.developOneStepReducts rs s
		       val (rb,sb) = if LU.member' Term.equal t sReducts
				     then ([(s,t)],[])
				     else if Trs.isRewriteRule (s,t)
				     then ([],[(s,t)])
				     else ([],[])
		       val _ = if isOrgTRS then checkCounterExample rs (sReducts,t) else ()
		   in (rb,sb)
		   end

	   val (raddlist,saddlist) = LP.unzip (L.map (checkcp true) nonclosedcps)

	   val radds = Trs.deleteIdenticalRules (LU.mapAppend (fn x=>x) raddlist)
	   val sadds = Trs.deleteIdenticalRules (LU.mapAppend (fn x=>x) saddlist)

	   val radds2 = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y) (radds,rs)
	   val sadds2 = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y) (sadds,aux)

           (** auxiliary cps **)
	   val cpsS = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
					    (Cr.criticalPairs (rs@aux), cps)
	  (*  val _ = print ("CP(rest):\n" ^ Trs.prEqs cpsS) *)
	   val nonclosedcpsS = L.filter (not o Cr.isStrongClosed rs) cpsS
	   val (raddlistS,saddlistS) = LP.unzip (L.map (checkcp false) nonclosedcpsS)
	   val addlistS = Trs.deleteIdenticalRules (LU.mapAppend (fn x=>x) (raddlistS@saddlistS))
	   val sadds3 = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
					    (addlistS,aux)

       in if not (null radds2) orelse not (null sadds2) orelse not (null sadds3)
	  then proveUNRbyStrongClosed orgSize (step+1) (rs@radds2,aux@sadds2@sadds3)
	  else SOME (rs@aux)
       end
       handle StopUNRbySC rs2 => NONE

   exception StopUNRbyPC of (Term.term * Term.term) list 


   fun proveUNRbyParallelClosed orgSize step (rs,aux) = 
       let 
	   val _ =  print "checking TRS:\n"
	   val _ =  print (Trs.prRules rs)
	   val _ =  print "with auxiliary rules:\n"
	   val _ =  print (Trs.prRules aux)

	   (* val rs2 = rs @ aux  ... revised 2024/1/11 *)
           (* 危険対のチェックでは， l -> l を除く *)
	   val rs2pre = L.filter (not o Term.equal) (rs @ aux)
	   val _ = print ("Trivial rules are removed for checking CR\n")

	   val rs2 = if Trs.areLeftLinearRules rs2pre
		     then rs2pre
		     else if (!useLinearApprox)
		     then let val _ = print ("Try linear approximation\n")
			      val rs2rev = appLinearApproximation rs2pre
			  in if Trs.areLeftLinearRules rs2rev
			     then rs2rev
			     else raise StopUNRbyPC rs2rev (* proof stopped *)
			  end
		     else raise StopUNRbyPC rs2pre (* proof stopped *)
			    
	   val _ = let val inCps = Cr.insideCriticalPairs rs2
		       val outCps = Cr.outsideCriticalPairs rs2
		   in if Cr.isOostromClosedCps rs2 inCps outCps
		      then (println ("CPin(R):\n" ^ Trs.prEqs inCps);
			    println ("CPout(R):\n" ^ Trs.prEqs outCps);
			    print "(approximated) TRS is development closed; hence confluent\n";
			    raise SolvedWith UNR)
		      else ()
		   end
		       
	   val _ = if step >= LimitIteration
		      orelse (L.length rs2) > (orgSize + 2) * (orgSize + 2)
		   then raise StopUNRbyPC rs2 (* proof stopped *)
		   else ()

           (* 以下では rs に戻る *)
	   val inCps = Cr.insideCriticalPairs rs
	   val outCps = Cr.outsideCriticalPairs rs

	   (* val _ = print ("CPin(R):\n" ^ Trs.prEqs inCps) *)
	   (* val _ = print ("CPout(R):\n" ^ Trs.prEqs outCps) *)

	   val nonclosedInCps = L.filter (not o Cr.isOostromClosedInCp rs) inCps
	   val nonclosedOutCps = L.filter (not o Cr.isOostromClosedOutCp rs) outCps

	   val stepNum = 5 (* heulistics parameter *)

	   fun checkOutCp isOrgTRS (s,t) =
	       case (Rewrite.isNormalForm rs s, Rewrite.isNormalForm rs t) of
		   (true,true) => if isOrgTRS
				  then (print ("CounterExample:\n" ^ Trs.prEq (s,t) ^ "\n");
					raise SolvedWith NotUNR)
				  else ([],[])
		|  (false,false) => 
		   let val sReductSet = Rewrite.manyStepsReductSet rs stepNum s
		       val tReductSet = Rewrite.manyStepsReductSet rs stepNum t
		   in case TS.find (fn _=>true) (TS.intersection (sReductSet,tReductSet)) of
		       SOME u => ([(s,u),(t,u)],[])
		    |  NONE => if Trs.isRewriteRule (s,t)
			       then ([],[(s,t)])
			       else if Trs.isRewriteRule (t,s)
			       then ([],[(t,s)])
			       else ([],[])
		   end
		|  (true,false) => 
		   let val tReductSet = Rewrite.manyStepsReductSet rs stepNum t
		       val _ = if isOrgTRS
			       then checkCounterExample rs (TS.listItems tReductSet,s)
			       else ()
		   in if TS.member (tReductSet,s)
		      then ([(t,s)],[])
		      else if Trs.isRewriteRule (t,s)
		      then ([],[(t,s)])
		      else ([],[])
		   end
		|  (false,true) => 
		   let val sReductSet = Rewrite.manyStepsReductSet rs stepNum s
		       val _ = if isOrgTRS
			       then checkCounterExample rs (TS.listItems sReductSet,t)
			       else ()
		   in if TS.member (sReductSet,t)
		      then ([(s,t)],[])
		      else if Trs.isRewriteRule (t,s)
		      then ([],[(t,s)])
		      else ([],[])
		   end


	   fun checkInCp isOrgTRS (s,t) =
	       case (Rewrite.isNormalForm rs s, Rewrite.isNormalForm rs t) of
		   (true,true) => if isOrgTRS
				  then (print ("CounterExample:\n" ^ Trs.prEq (s,t) ^ "\n");
					raise SolvedWith NotUNR)
				  else ([],[])
		|  (false,false) => 
		   let val sReductSet = Rewrite.manyStepsReductSet rs stepNum s
		   in if TS.member (sReductSet,t)
		      then ([(s,t)],[])
		      else if Trs.isRewriteRule (s,t)
		      then ([],[(s,t)])
		      else ([],[])
		   end
		|  (false,true) => 
		   let val sReductSet = Rewrite.manyStepsReductSet rs stepNum s
		       val _ = if isOrgTRS
			       then checkCounterExample rs (TS.listItems sReductSet,t)
			       else ()
		   in if TS.member (sReductSet,t)
		      then ([(s,t)],[])
		      else if Trs.isRewriteRule (s,t)
		      then ([],[(s,t)])
		      else ([],[])
		   end
		|  (true,false) =>
		   let val tReductSet = Rewrite.manyStepsReductSet rs stepNum t
		       val _ = if isOrgTRS
			       then checkCounterExample rs (TS.listItems tReductSet,s)	
			       else ()
		   in if Trs.isRewriteRule (t,s)
		      then ([],[(t,s)])
		      else ([],[])
		   end

	   val nonclosedcps = (L.map (checkInCp true) nonclosedInCps) @ (L.map (checkOutCp true) nonclosedOutCps)
	   val (raddlist,saddlist) = LP.unzip nonclosedcps

	   val radds = Trs.deleteIdenticalRules (LU.mapAppend (fn x=>x) raddlist)
	   val sadds = Trs.deleteIdenticalRules (LU.mapAppend (fn x=>x) saddlist)

	   val radds2 = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y) (radds,rs)
	   val sadds2 = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y) (sadds,aux)
				    
           (** auxiliary cps **)
	   val inCpsS = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y) 
					    (Cr.insideCriticalPairs (rs@aux), inCps)
	   val outCpsS = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y) 
					    (Cr.outsideCriticalPairs (rs@aux), outCps)
	   (* val _ = print ("CPin(rest):\n" ^ Trs.prEqs inCpsS) *)
	   (* val _ = print ("CPout(rest):\n" ^ Trs.prEqs outCpsS) *)
	   val nonclosedInCpsS = L.filter (not o Cr.isOostromClosedInCp rs) inCpsS
	   val nonclosedOutCpsS = L.filter (not o Cr.isOostromClosedOutCp rs) outCpsS
	   val nonclosedcpsS = (L.map (checkInCp false) nonclosedInCpsS) @ (L.map (checkOutCp false) nonclosedOutCpsS)
	   val (raddlistS,saddlistS) = LP.unzip nonclosedcpsS
	   val addlistS = Trs.deleteIdenticalRules (LU.mapAppend (fn x=>x) (raddlistS@saddlistS))
	   val sadds3 = LU.differenceByAll' (fn (x,y) => Subst.identicalModuloRenamingRule x y)
					    (addlistS,aux)

       in if not (null radds2) orelse not (null sadds2) orelse not (null sadds3)
	  then proveUNRbyParallelClosed orgSize (step+1) (rs@radds2,aux@sadds2@sadds3)
	  else SOME (rs@aux)
       end
       handle StopUNRbyPC rs2  => NONE

   fun getSomeApproximation rs = 
       let
	   val cps = L.filter (not o T.equal) (Cr.criticalPairs rs)
	   val stepNum = 5 (* heulistics parameter *)
	   fun getReductSet t = TS.union (Rewrite.developOneStepReductSet rs t,
					  Rewrite.manyStepsReductSet rs stepNum t)

	   fun checkcp (s,t) =
	       case (Rewrite.isNormalForm rs s, Rewrite.isNormalForm rs t) of
		   (true,true) => (print ("CounterExample:\n" ^ Trs.prEq (s,t) ^ "\n");
				   raise SolvedWith NotUNR)
		|  (false,false) => 
		   let val sReductSet = getReductSet s
		       val tReductSet = getReductSet t
		       val comon = TS.listItems (TS.intersection (sReductSet,tReductSet))
		       fun addSize u = (t,T.termSize u)
		       val comonSorted = L.map (fn (t,_) => t)
					       (ListMergeSort.sort (fn ((_,i),(_,j))=> Int.> (i,j)) (L.map addSize comon))
		   in if not (null comon)
		      then [(s,hd comonSorted),(t,hd comonSorted)]
		      else if (T.termSize t < T.termSize s) andalso Trs.isRewriteRule (t,s)
		      then [(t,s)]
		      else if Trs.isRewriteRule (s,t)
		      then [(s,t)]
		      else []
		   end
		|  (true,false) => if Trs.isRewriteRule (t,s) then [(t,s)] else []
		|  (false,true) => if Trs.isRewriteRule (s,t) then [(s,t)] else []

	   val newRules = Trs.deleteIdenticalRules
			      (rs @ (L.filter (not o T.equal) (LU.mapAppend checkcp cps)))

       in newRules
       end

		   
   fun getRulesByCollapseMapping rs =
       case L.find (fn (l,r) => Term.isFun l
				andalso not (Term.equal (l,r))
				andalso null (Term.argsOfTerm l)
				andalso FS.member (T.funSetInTerm r, valOf (Term.funRootOfTerm l))) rs of
	   NONE => NONE
	|  SOME (l0,r0) =>
	   let val c = valOf (Term.funRootOfTerm l0)
	       fun condI (l,r) = if FS.member (T.funSetInTerm l, c)
				 then FS.member (T.funSetInTerm r, c)
				 else true
	       fun condII (l,r) = if FS.member (T.funSetInTerm r, c)
				  then true
				  else Trs.isNonErasingRule (l,r)
	       fun trans (l,r) = if FS.member (T.funSetInTerm l, c)
				 then NONE
				 else if FS.member (T.funSetInTerm r, c)
				 then SOME (l,l)
				 else SOME (l,r)
	   in if L.all (fn (l,r) => condI (l,r) andalso condII (l,r)) rs
	      then (println "Modify rules by collapse mapping";
		    SOME (c, L.mapPartial trans rs))
	      else NONE
	   end
				      

   fun getRulesByRepeatedCollapseMapping rs =
       let fun iterateCollapseMapping rs =
	       case getRulesByCollapseMapping rs of
		   SOME (_, rs2) => (print "(an iteration of Collapse Mapping)\n"; iterateCollapseMapping rs2)
		|  NONE => rs
       in case getRulesByCollapseMapping rs of
	      SOME (_,rs2) => SOME (iterateCollapseMapping rs2)
	   |  NONE => NONE
       end


   fun hasFiniteReducts limit rs term =
       let fun main step (checked,border) =
	       let val reductSet = List.foldl (fn (u,set)=>  TS.union (Rewrite.oneStepReductSet rs u,set)) TS.empty border
		   val checked2 = TS.addList (checked,border)
		   val border2 = TS.listItems (TS.difference (reductSet,checked2))
	       in if null border2
		  then SOME (TS.listItems checked2)
		  else if step > limit
		  then NONE
		  else main (step+1) (checked2,border2)
	       end	       
       in main 1 (TS.empty, [term])
       end

       
   fun getRulesByGeneralizeCollapsMapping rs (* obsolute *) =
       let val limit = 3 (* heulistics *)
	   fun trans (l,r) =
	       (* returns (type III rule, (modified) type II rule, descendants to check) *)
	       if Term.isGroundTerm r
	       then case hasFiniteReducts limit rs r of
			SOME ts => if L.all (not o (Rewrite.isNormalForm rs)) ts
				   then if Term.isGroundTerm l
					then case hasFiniteReducts (limit+1) rs l of
						 SOME ss => if L.all (not o (Rewrite.isNormalForm rs)) ss
							    then (NONE,NONE,ts)
							    else (NONE,SOME (l,l),ts)
					       | NONE => (NONE,SOME (l,l),ts)
					else (NONE,SOME (l,l), ts)
				   else (SOME (l,r),NONE,[])
		      | NONE => (SOME (l,r),NONE,[])
	       else (SOME (l,r), NONE, [])

	   val results = L.map trans rs
	   val descendants = LU.eliminateDuplication' Term.equal (LU.mapAppend (fn (x,y,z) => z) results)
	   val typeIIIrules = L.mapPartial (fn (x,y,z) => x) results
	   val typeIIrules = L.mapPartial (fn (x,y,z) => y) results
	   fun checkOverlap t = null (Cr.insideCriticalPairs2 ([(t,t)],typeIIIrules))
       in if Trs.areNonErasingRules typeIIIrules
	     andalso L.all checkOverlap descendants
	  then SOME (typeIIIrules @ typeIIrules)
	  else NONE
       end

   fun SRRelim (srWithDesc,neRules) =
       if null srWithDesc
       then NONE
       else let 
	   fun checkOverlap t = null (Cr.insideCriticalPairs2 ([(t,t)],neRules))
	   val (srSuccess, srFail) = L.partition (fn ((l,r),ts) => L.all checkOverlap ts) srWithDesc
       in if null srFail
	  then SOME (L.map (fn (lr,ts)=>lr) srSuccess,neRules)
	  else if Trs.areNonErasingRules (L.map (fn (lr,ts)=>lr) srFail)
	  then SRRelim (srSuccess, neRules @ L.map (fn (lr,ts)=>lr) srFail)
	  else NONE
       end

   fun getRulesBySRRelimination rs =
       let val limit = 3 (* heulistics *)
	   fun checkStronglyReducible (l,r) = 
	       if not (Term.isGroundTerm r)
	       then (false,(l,r),[])
	       else case hasFiniteReducts limit rs r of
			SOME ts => (if L.all (not o (Rewrite.isNormalForm rs)) ts
				    then (true,(l,r),ts)
				    else (false,(l,r),[]))
		      | NONE => (false,(l,r),[])

	   val (nonErasingRules,erasingRules) = L.partition Trs.isNonErasingRule rs 
	   val erasingRulesChecked = L.map checkStronglyReducible erasingRules
	   val isStartable = L.all (fn (x,y,z)=>x) erasingRulesChecked

       in if not isStartable
	  then NONE
	  else let
	      val SRrules0 = L.map (fn (x,y,z)=>(y,z)) erasingRulesChecked
	      val nonErasingRulesChecked = L.map checkStronglyReducible nonErasingRules
	      val (SRrules1pre,NErules1pre) = L.partition (fn (x,y,z)=>x) nonErasingRulesChecked
	      val SRrules1 = L.map (fn (x,y,z)=>(y,z)) SRrules1pre
	      val NErules1 = L.map (fn (x,y,z)=>y)NErules1pre
	      val (initSR,initNE) = (SRrules0 @ SRrules1, NErules1)
	  in SRRelim (initSR,initNE)
	  end
       end

   fun unrSolver opt0 opt1 opt2 opt3 sigsOp rules = 
       let val _ =  print "input TRS:\n"
	   val faSetInRules = Trs.funAritySetInRules rules
	   val faSet =  case sigsOp of
			    SOME ls => FunIntSet.addList (faSetInRules, ls)
			  | NONE => faSetInRules
	   val faList = FunIntSet.listItems faSet
	   val faExList = FunIntSet.listItems (FunIntSet.difference (faSet, faSetInRules))

	   fun prFunArity (f,i) = (Fun.toString f) ^ " : " ^ (Int.toString i)
	   val _ =  print (PrintUtil.prList prFunArity faList)
	   val _ =  print (Trs.prRules rules)

	   val _ = if (!useStrongNonOverlapping)
		   then let val _ =  debug (fn ()=> println "Check strong non-overlapping criterion ")
			    val _ = if Unc.checkStronglyNonOverlapping rules
	   			    then (println nameStrongNonOverlapping; raise SolvedWith UNR)
				    else (println (unknown nameStrongNonOverlapping))
			in () end
		   else ()

	   val _ = if (!useRightReducible)
		   then let val _ =  debug (fn ()=> println "Check right-reducibility criterion ")
			    val _ = if L.all (fn (l,r) => not (Rewrite.isNormalForm rules r)) rules
				    then (println nameRightReducible; raise SolvedWith UNR)
				    else (println (unknown nameRightReducible))
			in () end
		   else ()
		
	   val _ = if (!useShallowDecProcForUNC)
		      andalso Trs.areShallowRules rules
		   then let val _ =  println "Call a UNC decision procedure for shallow TRSs..."
			in if Unc.runDecProcForShallow rules
			   then (println "...UNC found by the decision procedure (hence UNR).";
				 raise SolvedWith UNR)
			   else (println "...Not UNC found by the decision procedure.")
			end
		   else ()

	   val _ = if (!useNonUnrByCps)
	   	   then let  val _ =  debug (fn ()=> println "Check non-UNR by critical pairs")
			     val rs0 = addAuxiliaryRules (L.length rules) 0 rules
	   		     val rs0' = LU.eliminateDuplication' (fn (x,y) => Subst.identicalModuloRenamingRule x y) rs0
	   		in case checkNonUNRbyCP rs0' of
	   		       SOME ans => raise SolvedWith NotUNR
	   		     | NONE => ()
	   		end
	   	   else ()

	   val _ = if (!useTrivial)
	   	   then let  val _ =  debug (fn ()=> println "Check UNR by trivial rules")
			in if L.all (fn (l,r)=> Term.equal (l,r)) rules
			   then (println "Consisting of trivial rules (hence UNR)";
				 raise SolvedWith UNR)
			   else ()
			end
		   else ()

           val satSolver = (#satSolver opt0)
           val smtSolver = (#smtSolver opt0)
           val snProver = (#terminationProver opt0)
           val tmpDir = (#tmpDir opt0)

          fun isTerminating isOverlay rs = 
	      if snProver = ""
	      then 
		  if (#useDp opt1)
		  then if isOverlay 
		       then DpSolver.dpSolverInner satSolver tmpDir opt1 opt2 rs
		       else DpSolver.dpSolver satSolver tmpDir opt1 opt2 rs
		  else PoSolver.poSolver satSolver tmpDir opt2 (rs,[])
	      else 
		  if isOverlay 
		  then Solver.sinSolver snProver tmpDir rs
		  else Solver.snSolver snProver tmpDir rs
			    
	  val nameKnuthBendix = "Knuth & Bendix"
	  val _ = if not (!useKB)
		  then ()
		  else let val _ = print "UNR Proof by the Knuth-Bendix criterion\n"
			   val _ = print "checking TRS:\n"
			   val _ = print (Trs.prRules rules)
			   val rules2 = L.filter (not o Term.equal) rules
			   val containTrivRules = (L.length rules2 = L.length rules)
			   val _ = print "containing trivial rules..remove trivial rules and forget nonUNR\n"
			   val inCps = Cr.insideCriticalPairs rules2
			   val outCps = Cr.outsideCriticalPairs rules2
		      in if null inCps  (* overlay *)
			 then (print "Overlay, check Innermost Termination...\n";
			       if isTerminating true rules2
			       then let val _ = print "Innermost Terminating (hence Terminating)"
					val joinableCps = Cr.isInnerJoinableCps rules2 outCps
					val _ = if joinableCps
						then println ", WCR"
						else println ", not WCR"
					val _ = println nameKnuthBendix
				    in if joinableCps then raise SolvedWith UNR
				       else if not containTrivRules
				       then raise SolvedWith NotUNR
				       else println  "unknown CR...failed\n"
				    end
			       else println  "unknown/not Terminating...failed\n")
			 else (print "not Overlay, check Termination...\n";
			       if isTerminating false rules2
			       then let val _ = print "Terminating"
					val joinableCps = Cr.isJoinableCps rules2 (inCps @ outCps)
					val _ = if joinableCps
						then println ", WCR"
						else println ", not WCR"
					val _ = println nameKnuthBendix
				    in if joinableCps then raise SolvedWith UNR
				       else if not containTrivRules
				       then raise SolvedWith NotUNR
				       else println  "unknown CR...failed\n"
				    end
			       else println  "unknown/not Terminating...failed\n")
		      end
			    

	   val approxRulesSC = if not (!useTransByStronglyClosed)
			       then NONE
			       else
				   let val _ = print "UNR proof by Transformation (Strongly Closed)\n"
				   in if Trs.areLinearRules rules
				      then let val rsop = proveUNRbyStrongClosed (L.length rules) 0 (rules,[])
					       val _ = print "...failed\n"
					   in rsop
					   end
				      else if not (!useLinearApprox)
				      then (print "Not tried ... not linear\n"; NONE)
				      else
					  let val _  = print "Not linear, apply linear-approximation if possible\n"
					      val rules2 = appLinearApproximation rules
					  in if Trs.areLinearRules rules2
					     then let val rsop2 = proveUNRbyStrongClosed (L.length rules2) 0 (rules2,[])
						      val _ = print "...failed\n"
						  in rsop2
						  end
					     else (print "Approximation failed...\n"; NONE)
					  end
				   end

	   
	   val approxRulesPC = if not (!useTransByParallelClosed)
			       then NONE
			       else let val _ =  print "UNR proof by Transformation (Parallel Closed)\n"
				    in if Trs.areLeftLinearRules rules
				       then let val rsop = proveUNRbyParallelClosed (L.length rules) 0 (rules,[])
						val _ = print "...failed\n"
					    in rsop
					    end
				       else if not (!useLinearApprox)
				       then (print "Not tried ... not left-linear\n"; NONE)
				       else let val _  = print "Not left-linear, apply linear-approximation if possible\n"
						val rules2 = appLinearApproximation rules
					    in if Trs.areLeftLinearRules rules2
					       then let val rsop2 = proveUNRbyParallelClosed (L.length rules2)
											     0 (rules2,[])
							val _ = print "...failed\n"
						    in rsop2
						    end
					       else (print "Approximation failed...\n"; NONE)
					    end
				    end


	   val _ = if not (!useCollapsMapping) 
		   then ()
		   else let val _ =  print "UNR proof by Collapse Mapping\n"
			in case getRulesByCollapseMapping rules of
			       NONE => println "...failed"
			     | SOME (c,rules2) =>
			       let val _ = useKB := true
				   val _ = useTrivial := true
				   val _ = useTransByStronglyClosed := true
				   val _ = useTransByParallelClosed := true
				   val _ = useLinearApprox := true
			       in case unrSolver opt0 opt1 opt2 opt3 
					      (case sigsOp of
						   SOME ls => SOME (LU.deleteOne' FunInt.equal (c,0) ls)
						 | NONE => NONE)
					      rules2 of
				      UNR =>  raise SolvedWith UNR
				   |  NotUNR =>  raise SolvedWith NotUNR
				   |  Unknown => ()
			       end
			end

(********* old version ***
	   val _ = if not (!useCollapsMapping)
		   then ()
		   else let val _ =  print "UNR proof by Collapse Mapping\n"
			(* in case getRulesByCollapseMapping rules of *)
			in case getRulesByRepeatedCollapseMapping rules of
			       NONE => println "...failed"
			     | SOME rules2pre =>
			       let val _ = print "Collapse Mapping succeeds...\n"
				   val _ =  print (Trs.prRules rules2pre)
   				   val rules2 = if not (!useLinearApprox)
						then (print "Apply linear approximation\n";
						      appLinearApproximation rules2pre)
						else rules2pre

				   val _ = if Trs.areLinearRules rules2
					   then let val rsop = proveUNRbyStrongClosed (L.length rules2) 0 (rules2,[])
						    val _ = print "...failed\n"
						in rsop
						end
					   else (print "Not tried ... not linear\n"; NONE)
				   val _ = if Trs.areLeftLinearRules rules2
					   then let val rsop = proveUNRbyParallelClosed (L.length rules2) 0 (rules2,[])
						    val _ = print "...failed\n"
						in rsop
						end
					   else (print "Not tried ... not left-linear\n"; NONE)
			       in () end
			end
************)
	   

	   val _ = if not (!useSRRelimination)
		   then ()
		   else let val _ =  print "UNR proof by SRR elimination\n"
			in case getRulesBySRRelimination rules of
			       NONE => println "...failed"
			     | SOME (srr,rest) =>
			       let val rules2 =  rest @ (L.map (fn (l,r) => (l,l)) srr)
				   val _ = print "SRR elimination succeeds...\n"
				   val _ = useKB := true
				   val _ = useTrivial := true
				   val _ = useTransByStronglyClosed := true
				   val _ = useTransByParallelClosed := true
				   val _ = useLinearApprox := true
			       in case unrSolver opt0 opt1 opt2 opt3 sigsOp rules2 of
				      UNR =>  raise SolvedWith UNR
				   |  NotUNR =>  raise SolvedWith NotUNR
				   |  Unknown => ()
			       end
			end


(******* old version
	   val _ = if not (!useSRRelimination)
		   then ()
		   else let val _ =  print "UNR proof by SRR elimination\n"
			in case getRulesBySRRelimination rules of
			       NONE => println "...failed"
			     | SOME (srr,rest) =>
			       let val rules2 =  rest @ (L.map (fn (l,r) => (l,l)) srr)
				   val _ = print "SRR elimination succeeds...\n"
				   val _ =  print (Trs.prRules rules2)
				   val _ = if Trs.areLinearRules rules2
					   then let val rsop = proveUNRbyStrongClosed (L.length rules2) 0 (rules2,[])
						    val _ = print "...failed\n"
						in rsop
						end 
					   else (print "Not tried ... not linear\n"; NONE)
				   val _ = if Trs.areLeftLinearRules rules2
					   then let val rsop = proveUNRbyParallelClosed (L.length rules2) 0 (rules2,[])
						    val _ = print "...failed\n"
						in rsop
						end 
					   else (print "Not tried ... not left-linear\n"; NONE)
			       in () end
			end
*********)			    


	   val result = if (!useConfluenceProver)
			then
			    let val approxRules = let val rules2 = case (approxRulesSC,approxRulesPC) of
								       (NONE,NONE) => getSomeApproximation rules 
								     | (SOME rs, NONE) => rs
								     | (_,SOME rs) => rs
						      fun appNonLL rs (l,r) = if Term.isLinearTerm l
									      then (l,r)
									      else nLLapproximation rs (l,r)
						  in L.map (appNonLL rules2) rules2
						  end
			    in if L.length rules = L.length approxRules
				  andalso LP.all (fn (rule1,rule2) => TP.equal (rule1,rule2)) (rules,approxRules)
			       then (println "...failed to get an approximation."; Unknown)
			       else let 
				   (* val _ =  print (Trs.prRules rules) *)
				   val _ = println "Try to prove CR of the approximated TRS..."
				   val ans = CrSolver.crSolver opt0 opt1 opt2 opt3 approxRules
					val _ =  if ans = Cr.CR
						 then (println "...CR proof of the (approximated) TRS is successful.";
						       raise SolvedWith UNR)
						 else ()
					val _ = println "...CR proof of approximated TRS failed"
					val _ = println "Try to prove UNC of the approximated TRS..."
					val _ = Unc.useShallowDecProc := false (* done before *)
					val ans2 = Unc.uncSolverWithDecomposition opt0 opt1 opt2 opt3 approxRules
					val _ = if ans2 = Unc.UNC
   						then (println "...UNC proof of the (approximated) TRS is successful.";
						      raise SolvedWith UNR)
						else ()
					val _ = println "...UNC proof of approximated TRS failed"
				    in Unknown end
	   		    end
			else Unknown
							 



       in result
       end
       handle SolvedWith ans => ans


   end (* of local *)
   
   end; (* of structure UNC *)
