(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/fun.sml
 * description: definition and utility functions for Function Symbols
 * author: AOTO Takahito
 * 
 ******************************************************************************)

structure Fun : sig
   include ORD_KEY2 (* { val ord_key, fun compare, fun equal, fun toString }*)
   val fromString: string -> ord_key
   val toProofTree: ord_key -> unit -> string 
end = struct  
   type ord_key = Atom.atom
   fun compare (f,g) = Atom.compare (f,g)
   fun equal (f,g) = (compare (f,g)) = EQUAL
   fun toString f = Atom.toString f
   fun fromString s = Atom.atom s
   fun toProofTree f () = CertifyUtil.encloseProofLeafBy "name" (Atom.toString f)
end;

structure FunSet = RedBlackSetFn (Fun) : ORD_SET
structure FunMap = RedBlackMapFn (Fun) : ORD_MAP
structure FunPair = OrdKey2PairFn (structure A = Fun structure B = Fun) :ORD_KEY2
structure FunPairSet = RedBlackSetFn (FunPair) : ORD_SET
structure FunPairMap = RedBlackMapFn (FunPair) : ORD_MAP

structure FunInt = OrdKey2PairFn (structure A = Fun structure B = Int2) : ORD_KEY2
structure FunIntSet = RedBlackSetFn (FunInt) : ORD_SET
structure FunIntMap = RedBlackMapFn (FunInt) : ORD_MAP
structure FunIntPair = OrdKey2PairFn (structure A = FunInt structure B = FunInt) :ORD_KEY2

structure FunIntInt = OrdKey2TripleFn (structure A = Fun structure B = Int2 structure C = Int2) : ORD_KEY2
structure FunIntIntSet = RedBlackSetFn (FunIntInt) : ORD_SET
structure FunIntIntMap = RedBlackMapFn (FunIntInt) : ORD_MAP

structure FunPairTable : MONO_HASH_TABLE = 
let structure FP : HASH_KEY = 
       struct 
       type hash_key = FunPair.ord_key
       fun hashVal (f,g) = Word.+(Word.*(Word.fromInt 33, Atom.hash f), 
                                  Word.+(Word.fromInt 720, Atom.hash g))
       fun sameKey ((f1,f2),(g1,g2)) =  (Fun.equal (f1,g1)) andalso (Fun.equal (f2,g2))
       end
in HashTableFn (FP)
end

structure FunIntTable : MONO_HASH_TABLE = 
let structure FI : HASH_KEY = 
       struct 
       type hash_key = Fun.ord_key * int
       fun hashVal (f,i) = Word.+(Word.*(Word.fromInt 33, Atom.hash f), 
                                  Word.+(Word.fromInt 720, Word.fromInt i))
       fun sameKey ((f,i),(g,j)) =  (Fun.equal (f,g)) andalso i = j 
       end
in HashTableFn (FI)
end

structure FunIntPairTable : MONO_HASH_TABLE = 
let structure FIP : HASH_KEY = 
    struct 
    type hash_key = (Fun.ord_key * int) * (Fun.ord_key * int)
    fun hashVal ((f,i),(g,j)) = 
	Word.+(Word.*(Word.fromInt 33, 
		      Word.+(Word.*(Word.fromInt 33, 
				    Word.+(Word.*(Word.fromInt 33, Atom.hash f), 
					   Word.+(Word.fromInt 720, Word.fromInt i))),
			     Word.+(Word.fromInt 720, Atom.hash g))),
	       Word.+(Word.fromInt 720, Word.fromInt j))
    fun sameKey ((fi1,gj1),(fi2,gj2)) =  FunInt.equal (fi1,fi2)
					andalso FunInt.equal (gj1,gj2)
    end
in HashTableFn (FIP)
end

structure FunIntIntTable : MONO_HASH_TABLE = 
let structure FII : HASH_KEY = 
       struct 
       type hash_key = Fun.ord_key * int * int
       fun hashVal (f,i,j) =  Word.+(Word.*(Word.fromInt 33, 
                                            Word.+(Word.*(Word.fromInt 33, Atom.hash f), 
                                                   Word.+(Word.fromInt 720, Word.fromInt i))),
                                     Word.+(Word.fromInt 720, Word.fromInt j))
       fun sameKey ((f,i1,i2),(g,j1,j2)) =  (Fun.equal (f,g)) andalso i1 = j1 andalso i2 = j2
       end
in HashTableFn (FII)
end

