(******************************************************************************
 * Copyright (c) 2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/inf.sml
 * description: definition and utility functions for Regular Infinitary Terms
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature INF_TERM = 
   sig
   datatype iterm = Inf of Var.ord_key * Subst.subst
   val toString: iterm -> string
   val fromString: string * string -> iterm option
   val renameDisjointly: iterm * iterm -> iterm * iterm
   val disjointUnify: iterm * iterm -> Subst.subst option
   val unify: iterm -> iterm -> Subst.subst option
   val unifyFiniteTerms: Term.term -> Term.term -> Subst.subst option
   val omegaNonOverlappingTrs: (Term.term * Term.term) list -> bool
   val matchPath: (Term.term * Term.term) list -> Term.term -> Term.position -> bool
   val test: unit -> unit
   val test2: unit -> unit
   val test3: int -> unit
   end;


structure InfTerm : INF_TERM = 
   struct
   local 
       structure FS = FunSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure VM = VarMap
       structure TS = TermSet
       structure VS = VarSet
       open Term
   in

   val debug = false

   exception InfTermErr of string

   datatype iterm = Inf of Var.ord_key * Subst.subst

   fun toString (Inf (x,sub)) = "<" ^ (Var.toString x) ^ "," ^ (Subst.toString sub) ^ ">"

   fun fromString (vstr,substr) = 
       case (varRootOfTerm (IOFotrs.rdTerm vstr),Subst.fromString substr)  of 
	   (SOME v, SOME sub) => SOME (Inf (v,sub)) | _ => NONE

   fun maxVarIndex (Inf (x,sub)) =
       let val cod = VarMap.listItems sub
           val dom = L.map (fn x => Var (x,Sort.null)) (VarMap.listKeys sub)
       in       	   
	   Int.max (Var.index x, 
		    Term.maxVarIndexInTerm (Fun ((Fun.fromString ""),
					   dom @ cod, 
					   Sort.null)))
       end

   fun chomskyNormalForm (term as (Inf (x,sub))) = 
       let val maxIdx = maxVarIndex term
	   val varIndex = ref maxIdx 
	   fun newVar () = (varIndex := (!varIndex) + 1; Var.fromStringAndInt ("x",!varIndex))
	   fun varTerm x = Var (x,Sort.null)
	   fun replace [] = ([],[])
	     | replace (s::ss) = if isVar s 
				 then let val (xs,ys) = replace ss
				      in (s::xs,ys)
				      end
				 else let val n = newVar ()
					  val (xs,ys) = replace ss
				      in ((varTerm n)::xs,(n,s)::ys)
				      end
	   fun flatEq [] = []
	     | flatEq ((x,t as (Var _))::xts) = (x,t)::(flatEq xts)
	     | flatEq ((x,t as (Fun (f,ts,ty)))::xts) =
	       if L.all Term.isVar ts
	       then (x,t)::(flatEq xts)
	       else let val (ts2,xts2) = replace ts 
		    in (x,Fun (f,ts2,ty))::flatEq (xts2 @ xts)
		    end
	   val sub2 = List.foldl (fn ((x,t),map) => VM.insert (map,x,t)) VM.empty 
				 (flatEq (VM.listItemsi sub))
       in 
	   Inf (x,sub2)
       end

   fun simplify ([],sub) = SOME sub
     | simplify ((s,t)::yet,sub) = 
       if Term.equal (s,t) then simplify (yet,sub)
       else 
	   case (s,t) of 
	       (Var (x,_),Var (y,_)) => 
	       let val s2 = case VM.find (sub,x) of SOME s2 => s2 | NONE => s 
		   val t2 = case VM.find (sub,y) of SOME t2 => t2 | NONE => t
		   val rev = VM.singleton (x,t)
		   val sub2 = case VM.find (sub,x) of SOME s2 => #1 (VM.remove (sub,x))
						    | NONE => sub
	       in simplify 
		      (L.map (fn (u,v) => (Subst.applySubst rev u, Subst.applySubst rev v))
			     ((s2,t2)::yet),
		       Subst.compose rev sub2)
	       end
	     | (Fun (f,ss,_),Fun (g,ts,_)) => 
	       if Fun.equal (f,g) 
	       then simplify ((LP.zip (ss,ts)) @ yet, sub)
	       else NONE
	     | (Fun (f,ss,_),Var (y,_)) => simplify ((t,s)::yet, sub)
	     | (Var (x,_),Fun (g,ts,_)) => 
	       case VM.find (sub,x) of 
		   NONE => simplify (yet, VM.insert (sub,x,t))
		 | SOME (Var _) => raise InfTermErr "simplify"
		 | SOME (Fun (f,ss,_)) => if Fun.equal (f,g)
					  then simplify ((LP.zip (ss,ts)) @ yet, sub)
					  else NONE

   fun renameDisjointly (Inf (x,xsub), Inf (y,ysub)) = 
       let val nullSort = Sort.null
	   val cod = VarMap.listItems xsub
           val dom = L.map (fn x => Var (x,nullSort)) (VarMap.listKeys xsub)
	   val maxIdx = Int.max (Var.index x, 
				 Term.maxVarIndexInTerm (Fun ((Fun.fromString ""),
							dom @ cod, 
							Sort.null)))
	   fun incVar (name,idx) = (name, idx + maxIdx + 1)
	   fun incTerm (Var (z,s)) =  Var (incVar z,s)
	     | incTerm (Fun (f,ts,s)) =  Fun (f,L.map incTerm ts, s)
	   val y2 = incVar y
	   val ysub2 = VM.foldli (fn (k,i,vmap) => 
				     VM.insert (vmap, incVar k, incTerm i))
				 VM.empty 
				 ysub 
       in (Inf (x,xsub), Inf (y2, ysub2))
       end

   fun minMaxLocalVarIndex (Inf (x,xsub)) =
       let val lvs = VM.listKeys xsub
       in
	   Term.rangeVarIndexInTerms (L.map (fn y => Var (y,Sort.null)) (x::lvs))
       end

   fun increaseLocalVarIndexBy n (Inf (y,ysub)) = 
       let    
	   fun incVar x = Var.increaseIndexBy n x
	   val keys = VarMap.listKeys ysub
	   val rename = 
	       L.foldl (fn (k,map)=> VM.insert (map,k, Var (incVar k,Sort.null)))
		       VM.empty keys
	   val ysub2 = 
	       VM.foldli (fn (k,t,vmap) => 
			     VM.insert (vmap, incVar k, Subst.applySubst rename t))
			 VM.empty ysub 
       in Inf (incVar y, ysub2)
       end


   fun renameLocalVarDisjointly (itermx as (Inf (x,xsub)), itermy as (Inf (y,ysub))) = 
       let val (lx,gx) = minMaxLocalVarIndex itermx
	   val (ly,gy) = minMaxLocalVarIndex itermy
	   val (itermx1,itermy1) = 
	       if (gy < lx) orelse (gx < ly)
	       then (itermx,itermy)
	       else
		   if lx <= ly 
		   then (itermx, increaseLocalVarIndexBy (gx + 1 - ly) itermy)
		   else (increaseLocalVarIndexBy (gy + 1 - lx) itermx, itermy)
	   val _ = if debug
		   then print ("rename local vars: " ^ (toString itermx1)
			  ^  " and " ^ (toString itermy1) ^ "\n")
		   else ()
       in (itermx1, itermy1)
       end



   (* from J.Jaffer, "Efficient Unification over Infinite Terms", 
      New Generation Computing, Vol.2, 1984, pp.207-219 *)

   local
  (* checkRoots (NONE,NONE,[]) = SOME (op1,op2,argslist)
        $BJQ?t0J30$N(B root $B$,6&DL$J$i(B SOME (op1,op2,sublist)
           where         op1 ...  $BJQ?t$,=P8=$7$F$$$?$i!$$=$NJQ?t(B
                         op2 ...  root $B$N4X?t5-9f(B(NONE...funterm$B$,$^$@L$=P8=(B)
                         argslist ... funterm $B$N0z?t$N%j%9%H(B *)
   fun checkRoots (op1,op2,argslist) [] = SOME (op1,op2,L.map rev argslist)
     | checkRoots (op1,op2,argslist) (t::ts) =
       case (op1,op2,t) of 
	   (NONE,_,Var _) => checkRoots (SOME t,op2,argslist) ts
	 | (SOME s,_,Var _) => checkRoots (SOME s,op2,argslist) ts
	 | (_,NONE,Fun (f,us,ty)) => checkRoots (op1,SOME f,L.map (fn u=>[u]) us) ts
	 | (_,SOME g,Fun (f,us,ty)) => if Fun.equal (f,g)
				       then checkRoots (op1,SOME g,
							LP.map (fn (v,vs)=>v::vs) (us,argslist))
						       ts
				       else NONE
   in
   fun frontierCommon [] = raise (InfTermErr "frontierCommon")
     | frontierCommon ts = 
       case checkRoots (NONE,NONE,[]) ts of
	   NONE => NONE
	 | SOME (SOME t,_     ,_) => let val x = valOf (Term.varRootOfTerm t)
				     in  SOME (L.mapPartial (fn u => if Term.equal (t,u)
								     then NONE else SOME (x,u)) ts,
					       t)
				     end
	 | SOME (NONE  ,NONE  ,_) => NONE
	 | SOME (NONE  ,SOME f,argslist) => 
	   if null argslist 
	   then SOME ([], Fun (f,[],Sort.null))
	   else let val argsop = L.map frontierCommon argslist
		in if L.all isSome argsop
		   then let val (fargs,cargs) = LP.unzip (L.map valOf argsop)
			in SOME (L.concat fargs,
				 Fun (f,cargs,Sort.null))
			end
		   else NONE
		end


   end


   datatype entry = V of Var.ord_key | TL of TermSet.set

   local

   fun uparrow vmap x = case VM.find (vmap,x) of SOME (V y) => uparrow vmap y | _ => x
   fun lookup vmap x = case VM.find (vmap,x) of SOME (TL tset) => tset | _ => raise (InfTermErr "lookup")
   fun remove x vmap = 
       let val (vmap2,_) = VM.remove (vmap,x) in vmap2 end
       handle NotFound => vmap
   fun removeList [] vmap = vmap
     | removeList (x::xs) vmap = removeList xs (remove x vmap) 
   fun update (x,xent) vmap =
       let val (vmap2,_) = VM.remove (vmap,x) in VM.insert (vmap2,x,xent)end
       handle NotFound => VM.insert (vmap,x,xent)

   fun prVMap vmap = 
       let fun printEnt (V x) = (Var.toString x)
	     | printEnt (TL ts) = LU.toStringCommaSquare Term.toString (TS.listItems ts)
       in LU.toStringCommaCurly (fn (v,ent) => "(" ^ (Var.toString v) ^ "," ^ (printEnt ent) ^ ")")
				(VM.listItemsi vmap)
       end

   fun findEntry vmap = 
       let exception Found of (Var.ord_key * TS.set)
	   fun check (_,V _) = ()
	     | check (x,TL tset) = if (TS.numItems tset) > 1 then raise (Found (x,tset)) else ()
       in (VM.appi check vmap; NONE)
	  handle (Found ans) => SOME ans
       end

   fun getMgu vmap =
       let val sub = VM.mapPartial (fn ent => case ent of V y => SOME (Var (uparrow vmap y,Sort.null))
							| _ => NONE)  vmap
	   fun select tset = if TS.isEmpty tset 
			     then NONE 
			     else SOME (Subst.applySubst sub (hd (TS.listItems tset)))
	   fun getTerm (V y) = (case select (lookup vmap (uparrow vmap y)) of
				   SOME (t as (Fun (f,[],ty))) => SOME t
				 | _ => SOME (Var (uparrow vmap y, Sort.null)))
	     | getTerm (TL tset) = select tset
       in VM.mapPartial getTerm vmap
       end 

   exception NonUnify

   fun decompose x ts =
       case frontierCommon ts of
	   NONE => raise NonUnify
	 | SOME (front, com) => case com of 
				    (Var _) => (TL TS.empty, (x,com)::front)
				  | (Fun _) => (TL (TS.singleton com), front)
		
   fun step vmap = 
       let val _ = if debug
		   then print ((prVMap vmap) ^ "\n")
		   else ()
       in
	   case findEntry vmap of
	       SOME (x,tset) => let val (ent,ts) = decompose x (TS.listItems tset)
				    val vmap2 = update (x,ent) vmap
				in reflect vmap2 ts
				end
	     | NONE => let val mgu = getMgu vmap
			   val _ = if debug
				   then print "success(unify): "
				   else ()
			   val _ = if debug
				   then print ("mgu = " ^ (Subst.toString mgu) ^ "\n")
				   else ()
		       in SOME mgu
		       end
       end
   and reflect vmap [] = step vmap
     | reflect vmap ((x,(Var (y,_)))::rest) = 
       let val x2 = uparrow vmap x
	   val y2 = uparrow vmap y
       in if Var.equal (x2,y2) 
	  then reflect vmap rest
	  else let val xtset = lookup vmap x2
		   val ytset = lookup vmap y2
		   val vmap2 = update (x2, TL (TS.union (xtset,ytset))) (update (y2,V x2) vmap)
	       in reflect vmap2 rest
	       end
       end
     | reflect vmap ((x,t as (Fun _))::rest) = 
       let val x2 = uparrow vmap x
	   val tset = lookup vmap x2
	   val vmap2 = update (x2,TL (TS.add (tset,t))) vmap
       in reflect vmap2 rest
       end



   in

   fun unify itermx itermy =
       let val (Inf (x,xsub), Inf (y,ysub)) = renameLocalVarDisjointly (itermx,itermy)
	   val vmap = VM.unionWith (fn (u,v)=>u) (xsub,ysub)
	   val vmap1 = VM.foldli (fn (x,t,map)=> VM.insert (map,x,TL (TS.singleton t)))
				 VM.empty vmap
           val allVarSet = Term.varSetInTerms (VM.listItems vmap)
	   val globalVarSet = VS.difference (allVarSet, VS.addList (VS.empty, VM.listKeys vmap))
	   val vmap2 = L.foldl (fn (x,map) => VM.insert (map,x,TL (TS.empty)))
			       vmap1 (VS.listItems globalVarSet)
       in reflect vmap2 [(x,Var (y,Sort.null))]
       end
       handle NonUnify => (if debug then print "fail(unify)" else (); NONE)

   fun disjointUnify (itermx,itermy) = 
       let val (itermx1 as (Inf (x,xsub)),  itermy1 as (Inf (y,ysub))) = renameDisjointly (itermx,itermy)
	   val vmap = VM.unionWith (fn (u,v)=>u) (xsub,ysub)
	   val vmap1 = VM.foldli (fn (x,t,map)=> VM.insert (map,x,TL (TS.singleton t)))
				 VM.empty vmap
           val allVarSet = Term.varSetInTerms (VM.listItems vmap)
	   val globalVarSet = VS.difference (allVarSet, VS.addList (VS.empty, VM.listKeys vmap))
	   val vmap2 = L.foldl (fn (x,map) => VM.insert (map,x,TL (TS.empty)))
			       vmap1 (VS.listItems globalVarSet)
	   val _ = if debug
		   then print ("unify " ^ (toString itermx1) ^ " and " ^ (toString itermy1) ^ ".\n")
		   else ()
       in reflect vmap2 [(x,Var (y,Sort.null))]
       end
       handle NonUnify => (if debug then print "fail(unify)" else (); NONE)

   end

   fun unifyFiniteTerms t1 t2 =
       let val count2 = 1 + (Term.maxVarIndexInTerms [t1,t2])
	   val rootVar1 = Var.fromStringAndInt ("x", count2)
	   val rootVar2 = Var.fromStringAndInt ("y", count2)
	   val iterm1 = Inf (rootVar1, VM.singleton (rootVar1,t1))
	   val iterm2 = Inf (rootVar2, VM.singleton (rootVar2,t2))
       in unify iterm1 iterm2
       end

   fun omegaNonOverlappingTrs rs =
       let val lhs = L.map (fn (l,r)=>l) rs
	   val indexes = L.tabulate (L.length rs, fn i =>i)
	   fun checkForEq (l1,l0) = let val [l1',l0'] = Subst.renameTerms [l1,l0]
					val sub0' = Term.nonVarProperSubterms l0'
				    in L.all (fn t0' => not (isSome (unifyFiniteTerms t0' l1'))) sub0'
				    end
	   fun checkForNonEq (l1,l0) = let val [l1',l0'] = Subst.renameTerms [l1,l0]
					val sub0' = Term.nonVarSubterms l0'
				    in L.all (fn t0' => ((* print "check: <";
					      print (Term.toString t0');
					      print ", ";
					      print (Term.toString l1');
					      print ">\n"; *)
					      not (isSome (unifyFiniteTerms t0' l1')))) sub0'
				    end
       in  L.all (fn i => L.all (fn j => if i = j
					 then checkForEq (L.nth (lhs,i),L.nth (lhs,j))
					 else checkForNonEq (L.nth (lhs,i), L.nth (lhs,j)))
				indexes)
		 indexes
       end





  fun removeUnusedSub (xs,vmap) = 
      let fun removeUnusedSub (_,[]) ans = ans
	    | removeUnusedSub (done, y::ys) ans = 
	      case VM.find (vmap,y) of
		  SOME t => if Term.equal (t, Var (y,Sort.null))
			    then removeUnusedSub (done, ys) ans
			    else
				let val zs = Term.varListInTerm t
				val new = LU.differenceByAll' Var.equal (zs, done @ (y::ys))
			    in removeUnusedSub (y::done, ys @ new) (VM.insert (ans,y,t))
			    end
		| NONE => removeUnusedSub (done, ys) ans
      in  removeUnusedSub ([],xs) VM.empty
      end

   local
       val count = ref 1
       fun fakeFun i = Fun.fromString ("_gv" ^ (Int.toString ((!count) + i)))
       fun fakeConsts n = List.tabulate (n, fn i => fakeFun i)
       fun fakeConstSet n = FS.addList (FS.empty, fakeConsts n)
       fun makeTranslationTable (pvset,tfset) =
	   let val size = VS.numItems pvset
	       fun setCounter () = if FS.isEmpty (FS.intersection (tfset,fakeConstSet size))
				   then ()
				   else (count:= (!count) + 1; setCounter ())
	       val sub = LP.foldl (fn (f,v,map) => VM.insert (map,v,Fun (f,[],Sort.null)))
				  VM.empty 
				  (List.tabulate (size, fn i => fakeFun i), VS.listItems pvset)
	   in (setCounter ();  LP.zip (fakeConsts size, VS.listItems pvset))
	   end

   in
   fun match (ipattern as (Inf (x,xsub)), iterm as (Inf (y,ysub))) = 
       let 
	   val _ = if debug
		   then print ("match a term " ^ (toString iterm) ^ "\n      with a pattern "
			  ^ (toString ipattern) ^ ".\n")
		   else ()
	   val patFunSet  = Term.funSetInTerms (VM.listItems xsub)

	   val termLocalVars = VM.listKeys ysub
	   val termVarSet  = Term.varSetInTerms (VM.listItems ysub)
	   val termGlobalVarSet  = VS.difference (termVarSet, VS.addList (VS.empty,termLocalVars))

	   val patLocalVars = VM.listKeys xsub
	   val patVarSet  = Term.varSetInTerms (VM.listItems xsub)
	   val patGlobalVarSet  = VS.difference (patVarSet, VS.addList (VS.empty,patLocalVars))

	   val table = makeTranslationTable (termGlobalVarSet,patFunSet)

	   fun mapV2C x = L.find (fn (_,y) => Var.equal (x,y)) table
	   fun mapC2V c = L.find (fn (d,_) => Fun.equal (c,d)) table
	   fun replaceV2C (t as (Var (x,ty))) = (case mapV2C x of SOME (c,_) => Fun (c,[],ty) | _ => t)
	     | replaceV2C (Fun (f,ts,ty)) = Fun (f, L.map replaceV2C ts,ty)
	   fun replaceC2V (t as (Fun (f,[],ty))) = (case mapC2V f of SOME (_,x) => Var (x,ty) | _ => t)
	     | replaceC2V (Fun (f,ts,ty)) = Fun (f, L.map replaceC2V ts,ty)
	     | replaceC2V (t as (Var _)) = t
	   fun replaceC2Vop (x, (t as (Fun (f,[],ty)))) = 
	       (case mapC2V f of 
		   SOME (_,y) => if Var.equal (x,y) then NONE else SOME (Var (y,ty))
		 | NONE => SOME t)
	     | replaceC2Vop (x, t) = SOME (replaceC2V t)

	   val iterm1 = Inf (y,VM.map replaceV2C ysub)
	   val _ = if debug
		   then print ("unify " ^ (toString iterm1) ^ "\n      and "
			       ^ (toString ipattern) ^ ".\n")
		   else ()

       in case unify iterm1 ipattern of
	      SOME mgu => let val matcher = removeUnusedSub (VS.listItems patGlobalVarSet, 
							     VM.mapPartiali replaceC2Vop mgu)
			      val _ = if debug
				      then print "success(match): "
				      else ()
			      val _ = if debug
				      then print ("matcher = " ^ (Subst.toString matcher) ^ "\n")
				      else ()
			  in SOME matcher
			  end
	    | NONE => (if debug then print "fail(match)\n" else (); NONE)
       end
   end

   fun rootRewrite [] iterm = NONE
     | rootRewrite ((l,r)::rs) iterm = 
       let val count2 = 1 + (Term.maxVarIndexInTerm l)
	   val rootVar = Var.fromStringAndInt ("x", count2+1)
	   val lhs = Inf (rootVar, VM.singleton (rootVar,l))
       in
	   case match (lhs,iterm)
            of SOME sigma => let val reduct = Inf (rootVar,
						   VM.foldli (fn (v,t,map) => VM.insert (map,v,t)) 
							     (VM.singleton (rootVar,r))
							     sigma)
				 val _ = if debug
					 then print ("reduct: " ^ (toString reduct) ^ "\n")
					 else ()
			     in SOME reduct
			     end
             | NONE => rootRewrite rs iterm
       end

   fun applyInfSubst vmap term =
       let val idx = Term.maxVarIndexInTerm term
	   val idx2 = List.foldr Int.max 0 (L.map Var.index (VM.listKeys vmap))
	   val range = VM.listItems vmap
	   val idx3 = List.foldr Int.max 0 (L.map Term.maxVarIndexInTerm range)
	   val max = Int.max (idx, Int.max (idx2,idx3))
	   val v = Var.fromStringAndInt ("x",max+1)
       in Inf (v, removeUnusedSub ([v],VM.insert (vmap,v,term)))
       end

   fun collapsedSubs vmap =
       let val lvars = VM.listKeys vmap
	   val range = VM.listItems vmap
	   val gvarSet = VS.difference (Term.varSetInTerms range, VS.addList (VS.empty,lvars))
          (** ad hoc **)
	   fun isSameSortVars x y  = (String.size (Var.name x) = String.size (Var.name y))
	   val cand1 = 
	       L.mapPartial
		   (fn (v,t) => let val tvSet = Term.varSetInTerm t
				in if (LU.member' Var.equal v lvars)
				      andalso VS.isSubset (tvSet, gvarSet)
				   then SOME (L.foldl (fn (x,map)=>
							  if isSameSortVars x v
							  then VM.insert (map,x,Var (v,Sort.null))
							  else map)
						      VM.empty (VS.listItems tvSet))
				   else NONE
				end)
		   (VM.listItemsi vmap)
       in VM.empty::cand1 (*  $B=EJ#$,$G$-$F$7$^$&!%!%!%(B*)
       end


   fun disproveOmegaSHN rs =
       let val timer = Timer.startCPUTimer ()
	   val start = TimeUtil.checkTime timer 
	   val _ = print "\n-------------------------\n"
	   val _ = print   "start disproveOmegaSHN...\n"
	   val _ = print   "-------------------------\n"
	   val _ = print ("TRS:" ^ (LU.toStringCommaLnSquare IOFotrs.prRule rs))

	   fun check (l,t,sigma) = 
	       let val lsigma = Subst.applySubst sigma l
		   val idx = Term.maxVarIndexInTerm t
		   val lsigmaPrime = Term.increaseVarIndexBy (idx + 1) lsigma
		   val _ = if debug
			   then print ("REN(lsigma) := " ^ (Term.toString lsigmaPrime) ^ "\n")
			   else ()
		   exception Found of iterm 
	       in (** case Subst.unify lsigmaPrime t of **)
		   case unifyFiniteTerms lsigmaPrime t of
		       NONE => NONE
		    | SOME theta => let val _ = if debug
						then print ("theta := " ^ (Subst.toString theta) ^ "\n")
						else ()
				    in case L.find (fn rho =>
						       let val _ = if debug
								   then print ("\ncollapse by " 
									  ^ (Subst.toString rho)
									  ^ "\n")
								   else ()
							   val thetap = Subst.compose rho theta
							   val lsigmathetap = applyInfSubst thetap lsigma
							   val tthetap = applyInfSubst thetap t
						       in
						        case match (lsigmathetap,tthetap) of
							    SOME _ => raise (Found lsigmathetap)
							  | NONE => false
						       end)
						   (collapsedSubs theta) of
					   _  => NONE
				    end
		      handle (Found ans) => SOME ans
	       end
	   val limit = 20
	   fun step i [] = (print "fail"; NONE)
	     | step i ((l,t,sigma)::rest) = 
	       let val _ = print ("\nSTEP " ^(Int.toString i) ^ "\n")
		   val _ = print ("l:= " ^ (Term.toString l) ^ ", "
				  ^ "t:= " ^ (Term.toString t) ^ ", "
				  ^ "sigma:= " ^ (Subst.toString sigma) ^ "\n")
	       in
		   if i >= limit 
		   then (print "fail"; NONE)
		   else case check (l,t,sigma) of 
			    SOME t  => (print "success(disproveOmegaSHN):"; 
					print ("iterm = " ^ (toString t) ^ "\n");
					SOME t)
			  | NONE => let val new = L.map (fn (t2,sigma2) =>
							    (l,t2, Subst.compose sigma2 sigma))
							(Rewrite.oneStepNarrow rs t)
				    in step (i+1) (rest @ new)
				    end
	       end

	   val result = step 1 (L.map (fn (l,r) => (l,r,VM.empty)) rs)
	   val stop = TimeUtil.checkTime timer 
	   val _ = print (" ("  ^ (TimeUtil.reportTime (start,stop)) ^ " msec.)\n")
       in
	   result	   
       end

   fun isRedex rs iterm = isSome (rootRewrite rs iterm)

(*    fun matchPathRoot (Var _) _ _ = true *)
(*      | matchPathRoot (Fun _) (Var _) _ = false *)
(*      | matchPathRoot (s as (Fun (f,ts,_))) (t as (Fun (g,ss,_))) path =  *)
(*        let val _ = if debug then print "matchPathRoot: " else () *)
(* 	   val _ = if debug then print ((Term.toString s) ^ " : " ^ (Term.toString t)) else () *)
(* 	   val _ = if debug then print (" at " ^ (Term.prPosition path) ^ "\n") else () *)
(*        in *)
(*        if Fun.equal (f,g)  *)
(*        then if (null path) *)
(* 	    then (\** L.all Term.isVar ts **\)  *)
(* 		 (\** true **\) *)
(* 		(\** isSome (Subst.match s t) **\) *)
(* 		not (Fun.equal (f,Fun.fromString "cons")) *)
(* 	    else matchPathRoot (L.nth (ts, (hd path)-1)) (L.nth (ss,(hd path)-1)) (tl path) *)
(*        else false *)
(*        end  *)


   fun matchPathRoot2 (s as (Var _)) _ path  = 
       let val _ = if debug then print "stream path match: " else ()
	   val _ = if debug then print ((Term.toString s) ^ " in Var") else ()
	   val _ = if debug then print (" at " ^ (Term.prPosition path) ^ "\n") else ()
       in
	   true
       end

     | matchPathRoot2 (s as (Fun (f,_,_))) (t as (Var _)) path = 
       let val _ = if debug then print "stream path match: " else ()
	   val _ = if debug 
		   then if not (null path)
			then print ((Term.toString s) ^ " : " ^ (Term.toString t)) 
			else print ((Term.toString s) ^ "")
		   else ()
	   val _ = if debug then print (" at " ^ (Term.prPosition path) ^ " ") else ()
	   val _ = if debug 
		   then if null path
			   andalso not (Fun.equal (f,Fun.fromString "cons"))
			then print " <> \"cons\"\n"
			else print "\n"
		   else ()
       in
	   (null path) andalso not (Fun.equal (f,Fun.fromString "cons"))
       end
     | matchPathRoot2 (s as (Fun (f,ts,_))) (t as (Fun (g,ss,_))) path = 
       let val _ = if debug then print "stream path match: " else ()
	   val _ = if debug 
		   then if not (null path)
			then print ((Term.toString s) ^ " : " ^ (Term.toString t))
			else print ((Term.toString s) ^ "")
		   else ()
	   val _ = if debug then print (" at " ^ (Term.prPosition path) ^ " ") else ()
	   val _ = if debug 
		   then if null path
			   andalso not (Fun.equal (f,Fun.fromString "cons"))
			then print " <> \"cons\"\n"
			else print "\n"
		   else ()
       in
	   if (null path)
	   then not (Fun.equal (f,Fun.fromString "cons"))
	   else Fun.equal (f,g) 
		andalso 
		matchPathRoot2 (L.nth (ts, (hd path)-1)) (L.nth (ss,(hd path)-1)) (tl path)
       end 

   fun matchPath rs (Var _) path = false
     | matchPath rs (term as (Fun (f,ts,_))) path = 
       let val _ = if debug then print "\ncheck stream-stable-pos: " else ()
	   val _ = if debug then print (Term.toString term) else ()
	   val _ = if debug then print (" at " ^ (Term.prPosition path) ^ "\n") else ()
       in 
       L.exists (fn (l,_) => 
		    let val _ = if debug then print ("lhs: " ^ (Term.toString l) ^ "\n") else ()
			val result = matchPathRoot2 l term path
			val _ = if debug then
				    if result then (print "...match\n") 
				    else (print "...unmatch\n")
				else ()
		    in result
		    end)
		rs
       orelse
       if (L.length path <= 1)
       then false
       else matchPath rs (L.nth (ts,(hd path)-1)) (tl path)
       end 

   fun disproveGenProd rs term0 =
       let 
	   val timer = Timer.startCPUTimer ()
	   val start = TimeUtil.checkTime timer 
	   val _ = print "\n------------------------\n"
	   val _ = print   "start disproveGenProd...\n"
	   val _ = print   "------------------------\n"
	   val _ = print ("TRS:" ^ (LU.toStringCommaLnSquare IOFotrs.prRule rs))

	   val (_,dSymSet,_,_) =  Trs.fdcSetInRules rs
	   fun dsymIsContainedInPath t [] = false
	     | dsymIsContainedInPath (Var _) _ = false
	     | dsymIsContainedInPath (Fun (f,ts,_)) (p::ps) = 
	       FS.member (dSymSet,f)
	       orelse dsymIsContainedInPath (L.nth (ts,p-1)) ps

	   val cons = Fun.fromString "cons"
	   fun removeHead (t as (Var _)) = NONE
	     | removeHead (t as (Fun (f,ts,ty))) = 
	       if Fun.equal (f,cons) 
	       then case removeHead (L.nth (ts,1)) of 
			SOME u => SOME u
		      | NONE => SOME (L.nth (ts,1))
	       else NONE


	   fun check (hasRootStep,t0,t,sigma) = 
	       let val t0sigma = Subst.applySubst sigma t0
		   val idx = Term.maxVarIndexInTerm t
		   val t0sigmaPrime = Term.increaseVarIndexBy (idx + 1) t0sigma
		   val _ = if debug
			   then print ("REN(t0sigma) := " ^ (Term.toString t0sigmaPrime) ^ "\n")
			   else ()
		   exception Found of iterm 
		   fun checkSubterm tp revpos = 
		       let val _ = if debug
				   then print ("\ncheck subterm at " ^ (Term.prPosition (rev revpos)) ^ " ")
				   else ()
			   val _ = if debug 
				   then print ("of term " ^ (Term.toString t) ^ "\n")
				   else ()
			   val dcheck = ((null revpos) andalso hasRootStep)
					orelse dsymIsContainedInPath t (rev revpos)
			   val _ = if not dcheck andalso debug
				   then print "non-empty path with no defined symbols ... skip check\n"
				   else ()
		       in
			   if not dcheck 
			   then checkProperSubterm tp revpos
			   else

			   (* case Subst.unify t0sigmaPrime tp of 2010/11/19 *)
			   case unifyFiniteTerms t0sigmaPrime tp of
			       NONE => (if debug then print " not unify\n" else ();
					checkProperSubterm tp revpos)

			 | SOME theta => 
			   let val _ = if debug
				       then print (" unify theta := " ^ (Subst.toString theta) ^ "\n")
				       else ()
			   in case L.find 
				       (fn rho =>
					   let val _ = if debug
						       then print ("\ncollapse by " 
							      ^ (Subst.toString rho)
							      ^ "\n")
						       else ()
					       val thetap = Subst.compose rho theta
					       val t0sigmathetap = applyInfSubst thetap t0sigma
					       val tpthetap = applyInfSubst thetap tp
					   in
					       case match (t0sigmathetap,tpthetap) of
						   SOME _ => raise (Found t0sigmathetap)
						    | NONE => false
					      end)
					  (collapsedSubs theta) of
				  _  => checkProperSubterm tp revpos
			   end
		       end
		   and checkProperSubterm tp revpos = 
		       case tp of 
			   (Var _) => NONE
			 | Fun (_,ts,_) => 
			   case L.find
				    (fn i => let val tq = L.nth (ts,i) 	
						 val revq = (i+1)::revpos
						 val q = rev revq
					     in if matchPath rs t q
						then (if debug 
						      then print ("stop further check by non-stability: " ^
								  (Term.prPosition q) ^ "\n")
						      else (); false)
						else 
						    (if debug
						     then print ("stability check passed.\n")
						     else ();
							    isSome (checkSubterm tq revq))
					     end)
				    (L.tabulate (length ts, fn i => i)) of
			       _ => NONE
	       in checkSubterm t []
		  handle (Found ans) => SOME ans
	       end
	       
	   val limit = 20

	   fun step i [] = (print "fail"; NONE)
	     | step i ((hasRootStep,t0,t,sigma)::rest) = 
	       let val _ = print ("\nSTEP " ^(Int.toString i) ^ "\n")
		   val _ = print ("t0:= " ^ (Term.toString t0) ^ ", "
				  ^ "t:= " ^ (Term.toString t) ^ ", "
				  ^ "sigma:= " ^ (Subst.toString sigma) ^ "\n")
	       in
		   if i >= limit 
		   then (print "fail"; NONE)
		   else case removeHead t of
			    SOME s0 => 
			    let val cand = L.map (fn (pos,u,sigma2) => 
						     (hasRootStep orelse (null pos),s0,u,sigma2))
						 (Rewrite.oneStepOutermostNarrowWithPosition rs s0)
				val _ = if debug
					then print "remove head and outermost narrow\n"
					else ()
				val _ = if debug
					then print ("add:\n" 
					       ^ (LU.toStringCommaLnSquare 
						      (fn (_,u0,u,sub) => 
							   ("<" ^ (Term.toString u0) ^ ", " 
							    ^ (Term.toString u) ^ ", " 
							   ^ (Subst.toString sub) ^ ">"))
						      cand)
					       ^ "\n")
					else ()
			    in step (i+1) (rest @ cand)
			    end
			  | NONE => 
			    case check (hasRootStep,t0,t,sigma) of 
				SOME t  => (print "success(disproveGenProd):"; 
					    print ("iterm = " ^ (toString t) ^ "\n");
					    SOME t)
			      | NONE => let val cand = L.map 
							  (fn (pos,t2,sigma2) =>
							      (hasRootStep orelse (null pos),
							       t0,t2, Subst.compose sigma2 sigma))
							  (Rewrite.oneStepOutermostNarrowWithPosition rs t)
					    val _ = if debug 
						    then print "\noutermost narrow\n"
						    else ()
					    val _ = if debug
						    then print ("add:\n" 
							   ^ (LU.toStringCommaLnSquare 
								  (fn (_,u0,u,sub) => 
								      ("<" ^ (Term.toString u0) ^ ", " 
								       ^ (Term.toString u) ^ ", " 
								       ^ (Subst.toString sub) ^ ">"))
								  cand)
							   ^ "\n")
						    else ()
					in step (i+1) (rest @ cand)
					end
	       end

	   val result = step 1 (L.map (fn (pos,t,sigma) => (null pos,term0,t,sigma))
				      (Rewrite.oneStepOutermostNarrowWithPosition rs term0))

	   val stop = TimeUtil.checkTime timer 
	   val _ = print (" ("  ^ (TimeUtil.reportTime (start,stop)) ^ " msec.)\n")
       in
	   result
       end



   fun test3 1 = let val term = IOFotrs.rdTerm "A"
		     val R = IOFotrs.rdRules 
				 [
				  "A -> A",
				  "A -> B"
				 ]
		     val _ = disproveGenProd R term
		 in () end
     | test3 _ = ()


   fun test2 () = let

   val _ = let val term = IOFotrs.rdTerm "ones"
	       val R = IOFotrs.rdRules 
			   [
			    "ones -> cons(1,ones)"
			   ]
	       val _ = print "\nR1\n"
	   in disproveGenProd R term
	   end


   (*** Proving the Correctness of Reactive Systems Using Sized Types, p411 ***)
   val _ = let val term = IOFotrs.rdTerm "ones2"
	       val R = IOFotrs.rdRules 
			   [
			    "ones2 -> cons(1,tl(ones2))",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\nR2\n"
	   in disproveGenProd R term
	   end


   val _ = let val term = IOFotrs.rdTerm "zero(?xs)"
	       val R = IOFotrs.rdRules 
			   [
			    "zero(cons(0,?xs)) -> cons(0,zero(?xs))",
			    "zero(cons(1,?xs)) -> zero(?xs)"
			   ]
	       val _ = print "\nR3\n"
	   in disproveGenProd R term
	   end

   (* Ishihara  **)
   val _ = let val term = IOFotrs.rdTerm "f(?x)"
	       val R = IOFotrs.rdRules 
			   [
			    "f(cons(bullet,cons(bullet,?xs))) -> cons(bullet,f(f(?xs)))"
			   ]
	       val _ = print "\nR4\n"
	   in disproveGenProd R term
	   end

(* productivity of stream definitions, p769 *)

   val _ = let val term = IOFotrs.rdTerm "alt"
	       val R = IOFotrs.rdRules 
			   [
			    "alt -> tl (alt2)",
			    "alt2 -> cons(0,cons(1,alt2))"
			   ]
	       val _ = print "\nR5\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

(* productivity of stream definitions, p769 *)
   val _ = let val term = IOFotrs.rdTerm "zeros"
	       val R = IOFotrs.rdRules 
			   [
			    "zeros -> f(c)",
			    "c -> c",
			    "f(?x) -> cons(0,f(?x))"
			   ]
	       val _ = print "\nR6\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

(*  W. Buchholz: "A term calculus for (co-)recursive definitions on streamlike data-structures" *)
       
       val _ =  let val term = IOFotrs.rdTerm "F2(?x)" 
		    val R = IOFotrs.rdRules 
				[
				 "F2(cons(?x,?xs)) -> cons(add(S(0),?x),F2(F2(tl(tl(?xs)))))",
				 "tl(cons(?a,?xs)) -> ?xs",
				 "add(0,?y) -> ?y",
				 "add(S(x),?y) -> S(add(?x,?y))"
				]
	       val _ = print "\nR7\n"
                in disproveGenProd R term
		end


(* productivity of stream definitions, p771 Ex3.8 *)
   val _ = let val term = IOFotrs.rdTerm "J"
	       val R = IOFotrs.rdRules 
			   [
			    "J -> cons(0,cons(1,even(J)))",
			    "even(cons(?x,?xs)) -> cons(?x,odd(?xs))",
			    "odd(cons(?x,?xs)) -> even(?xs)"
			   ]
	       val _ = print "\nR8\n"
	   in disproveGenProd R term
	   end


   (* Zantema p.165  **)
   val _ = let val term = IOFotrs.rdTerm "f(?xs)"
	       val R = IOFotrs.rdRules 
			   [
			    "f(cons(?x,?xs)) -> cons(?x,tl(f(?xs)))",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\nR9\n"
	   in disproveGenProd R term
	   end


   (* Zantema p.171  **)
   val _ = let val term = IOFotrs.rdTerm "f(?xs)"
	       val R = IOFotrs.rdRules 
			   [
			    "c -> cons(1,c)",
			    "f(cons(?x,?xs)) -> g(?x,?xs)",
			    "g(0,?xs) -> f(?xs)",
			    "g(1,?xs) -> cons(1,f(?xs))"
			   ]
	       val _ = print "\nR10\n"
	   in disproveGenProd R term
	   end


   (* Zantema p.175  **)
   val _ = let val term = IOFotrs.rdTerm "f(c)"
	       val R = IOFotrs.rdRules 
			   [
			    "c -> cons(1,c)",
			    "f(cons(?x,?xs)) -> g(f(?xs))",
			    "g(cons(?x,?xs)) -> c"
			   ]
	       val _ = print "\nR11\n"
	   in disproveGenProd R term
	   end


   (* Zantema p.177  **)
   val _ = let val term = IOFotrs.rdTerm "f(c)"
	       val R = IOFotrs.rdRules 
			   [
			    "c -> cons(1,c)",
			    "f(cons(0,?xs)) -> cons(1,f(?xs))",
			    "f(cons(1,?xs)) -> cons(0,f(f(?xs)))"
			   ]
	       val _ = print "\nR12\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end


(* productivity of stream definitions, p767 Fig.1 *)
   val _ = let val term = IOFotrs.rdTerm "M"
	       val R = IOFotrs.rdRules 
			   [
			    "M -> cons(0,zip(inv(even(M)),tail(M)))",
			    "zip(cons(?x,?xs),?ys) -> cons(?x,zip(?ys,?xs))",
			    "inv(cons(?x,?ys)) -> cons(i(?x),inv(?ys))",
			    "even(cons(?x,?xs)) -> cons(?x,odd(?xs))",
			    "odd(cons(?x,?xs)) -> even(?xs)",
			    "tail(cons(?x,?xs)) -> ?xs",
			    "i(0) -> 1",
			    "i(1) -> 0"
			   ]
	       val _ = print "\nR13\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

(* productivity of stream definitions, p770 *)
   val _ = let val term = IOFotrs.rdTerm "M"
	       val R = IOFotrs.rdRules 
			   [
			    "M -> cons(0,zipstar(inv(even(M)),tail(M)))",
			    "zipstar(cons(?x,?xs),cons(?y,?ys)) -> cons(?x,cons(?y,zipstar(?xs,?ys)))",
			    "inv(cons(?x,?ys)) -> cons(i(?x),inv(?ys))",
			    "even(cons(?x,?xs)) -> cons(?x,odd(?xs))",
			    "odd(cons(?x,?xs)) -> even(?xs)",
			    "tail(cons(?x,?xs)) -> ?xs",
			    "i(0) -> 1",
			    "i(1) -> 0"
			   ]
	       val _ = print "\nR14\n"
	   in disproveGenProd R term
	   end

(* productivity of stream definitions, p770 *)
   val _ = let val term = IOFotrs.rdTerm "M"
	       val R = IOFotrs.rdRules 
			   [
			    "M -> cons(0,zip(inv(M),tail(M)))",
			    "zip(cons(?x,?xs),?ys) -> cons(?x,zip(?ys,?xs))",
			    "inv(cons(?x,?ys)) -> cons(i(?x),inv(?ys))",
			    "tail(cons(?x,?xs)) -> ?xs",
			    "i(0) -> 1",
			    "i(1) -> 0"
			   ]
	       val _ = print "\nR15\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end


(* productivity of stream definitions, p770 *)
   val _ = let val term = IOFotrs.rdTerm "M"
	       val R = IOFotrs.rdRules 
			   [
			    "M -> cons(0,zipstar(inv(M),tail(M)))",
			    "zipstar(cons(?x,?xs),cons(?y,?ys)) -> cons(?x,cons(?y,zipstar(?xs,?ys)))",
			    "inv(cons(?x,?ys)) -> cons(i(?x),inv(?ys))",
			    "tail(cons(?x,?xs)) -> ?xs",
			    "i(0) -> 1",
			    "i(1) -> 0"
			   ]
	       val _ = print "\nR16\n"
	   in disproveGenProd R term
	   end



   (*** B.A.Sijtsma p.636 ***)
   val _ = let val term = IOFotrs.rdTerm "L"
	       val R = IOFotrs.rdRules 
			   [
			    "L -> cons(0,cons(s(0),plusl(L,tl(L))))",
			    "plusl(cons(?x,?xs),cons(?y,?ys)) -> cons(plus(?x,?y),plusl(?xs,?ys))",
			    "plus(0,?y) -> ?y",
			    "plus(s(?x),?y) -> s(plus(?x,?y))",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\nR17\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

   (*** B.A.Sijtsma p.636 ***)
   val _ = let val term = IOFotrs.rdTerm "L"
	       val R = IOFotrs.rdRules 
			   [
			    "L -> cons(0,cons(s(0),minl(tl(tl(tl(L))),tl(L))))",
			    "minl(cons(?x,?xs),cons(?y,?ys)) -> cons(minus(?x,?y),minl(?xs,?ys))",
			    "min(?x,0) -> ?x",
			    "min(0,s(?y)) -> 0",
			    "min(s(?x),s(?y)) -> min(?x,?y)",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\nR18\n"
	   in disproveGenProd R term
	   end

   val _ = let val term = IOFotrs.rdTerm "L" (* p642, Sijstma *)
	       val R = IOFotrs.rdRules 
			   [
			    "L -> cons(s(0),cons(hd(tl(tl(tl(tl(L))))),L))",
			    "hd(cons(?x,?xs)) -> ?x",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\nR19\n"
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

   val _ = let val term = IOFotrs.rdTerm "L" (* p642, Sijstma this fails*)
	       val R = IOFotrs.rdRules 
			   [
			    "L -> cons(s(0),cons(hd(tl(tl(tl(L)))),L))",
			    "hd(cons(?x,?xs)) -> ?x",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\nR20\n"
               val _ = print "\n\nThe next example fails by some reason."
	   in disproveGenProd R term
	   end

(*        val _ =  let val term = IOFotrs.rdTerm "J" (\* p642, Sijstma, modified  --- fail *\) *)
(* 		    val R = IOFotrs.rdRules  *)
(* 				[ *)
(* 				 "L -> cons(1,cons(hd(J),L))", *)
(* 				 "J -> tl(tl(tl(tl(L))))", *)
(* 				 "hd(cons(?x,?xs)) -> ?x", *)
(* 				 "tl(cons(?x,?xs)) -> ?xs" *)
(* 				] *)
(* 		    val _ = print "\n\nThe next example should fail." *)
(*                 in disproveGenProd R term *)
(* 		end *)

(*        val _ =  let val term = IOFotrs.rdTerm "J" (\* p642, Sijstma, modified  --- success *\) *)
(* 		    val R = IOFotrs.rdRules  *)
(* 				[ *)
(* 				 "L -> cons(1,cons(hd(J),L))", *)
(* 				 "J -> tl(tl(tl(L)))", *)
(* 				(\* "J -> hd(tl(tl(tl(tl(tl(L))))))", --- OK *\) *)
(* 				 "hd(cons(?x,?xs)) -> ?x", *)
(* 				 "tl(cons(?x,?xs)) -> ?xs" *)
(* 				] *)
(*                 in disproveGenProd R term *)
(* 		end *)

(*        val _ =  let val term = IOFotrs.rdTerm "L" (\* p642, Sijstma, modified  --- fail *\) *)
(* 		    val R = IOFotrs.rdRules  *)
(* 				[ *)
(* 				 "L -> cons(1,cons(J,L))", *)
(* 				 "J -> hd(tl(tl(tl(tl(L)))))", *)
(* 				 "hd(cons(?x,?xs)) -> ?x", *)
(* 				 "tl(cons(?x,?xs)) -> ?xs" *)
(* 				] *)
(* 		    val _ = print "\n\nThe next example should fail." *)
(*                 in disproveGenProd R term *)
(* 		end *)

(*        val _ =  let val term = IOFotrs.rdTerm "L" (\* p642, Sijstma, modified --- fail *\) *)
(* 		    val R = IOFotrs.rdRules  *)
(* 				[ *)
(* 				 "L -> cons(1,cons(J,L))", *)
(* 				 "J -> hd(tl(tl(tl(L))))", *)
(* 				 "hd(cons(?x,?xs)) -> ?x", *)
(* 				 "tl(cons(?x,?xs)) -> ?xs" *)
(* 				] *)
(*                 in disproveGenProd R term *)
(* 		end *)


   (*** B.A.Sijtsma p.648***)
   val _ = let val term = IOFotrs.rdTerm "dup(?xs)"
	       val R = IOFotrs.rdRules 
			   [
			    "dup(cons(?x,?xs)) -> cons(?x,cons(?x,dup(?xs)))"
			   ]
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end


   (*** Telford 98 p1 ***)
   val _ = let val term = IOFotrs.rdTerm "evens"
	       val R = IOFotrs.rdRules 
			   [
			    "evens -> cons(s(s(0)),add(s(s(0)),evens))",
			    "add(?x,cons(?y,?ys)) -> cons(plus(?x,?y),add(?x,?ys))",
			    "plus(0,?y) -> ?y",
			    "plus(s(?x),?y) -> s(plus(?x,?y))"
			   ]
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

   (*** Telford 98 p33 ***)
   val _ = let val term = IOFotrs.rdTerm "f"
	       val R = IOFotrs.rdRules 
			   [
			    "f -> cons(hd(tl(f)),cons(1,f))",
			    "hd(cons(?x,?xs)) -> ?x",
			    "tl(cons(?x,?xs)) -> ?xs"
			   ]
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

(* productivity of stream definitions, p771 Ex3.9 *)
   val _ = let val term = IOFotrs.rdTerm "T"
	       val R = IOFotrs.rdRules 
			   [
			    "T -> cons(0,cons(1,f(tl(T))))",
			    "tl(cons(?x,?xs)) -> ?xs",
			    "f(cons(0,?xs)) -> cons(0,cons(1,f(?xs)))",
			    "f(cons(1,?xs)) -> cons(1,cons(0,f(?xs)))"
			   ]
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end

   (* Zantema p.164  **)
   val _ = let val term = IOFotrs.rdTerm "Fib"
	       val R = IOFotrs.rdRules 
			   [
			    "Fib -> cons(0,c)",
			    "c -> cons(1,f(c))",
			    "f(cons(0,?xs)) -> cons(1,cons(0,f(?xs)))",
			    "f(cons(1,?xs)) -> cons(0,f(?xs))"
			   ]
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end



(*** 
   val _ = let val term = IOMstrs.rdTerm "f(?xs:STREAM)"
	       val R = IOMstrs.rdRules 
			   [
			    "c -> cons(1,c)",
			    "f(cons(?x:Nat,?xs:STREAM)) -> g(f(?xs:STREAM))",
			    "g(cons(?x:Nat,?xs:STREAM)) -> c"
			   ]
	   in disproveGenProd R term
	   end
***)


   val _ = let val term = IOFotrs.rdTerm "zipstar(?xs,?ys)"
	       val R = IOFotrs.rdRules 
			   [
			    "zipstar(cons(?x,?xs),cons(?y,?ys)) -> cons(?x,cons(?y,zipstar(?xs,?ys)))"
			   ]
	       val _ = print "\n\nThe next example should fail."
	   in disproveGenProd R term
	   end


   in () end


   fun test () = let
       
   val _ = let val R = IOFotrs.rdRules ["ap(ap(L,?x),?y) -> ap(?x,ap(?y,?y))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(J,?x),?y),?z),?w) -> ap(ap(?x,?y),ap(ap(?x,?w),?z))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["a -> a"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(?x) -> f(f(?x))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(f(?x)) -> f(?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["a -> b", "b -> a"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(?x) -> g(?x)", "g(?x)->h(?x)", "h(?x)->f(?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(?x) -> g(?x)", "g(h(?x))->f(?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(?x) -> g(a)", "a -> b", "g(b)->f(a)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(g(?x)) -> g(f(?x))", "g(f(?x))->f(?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(g(?x)) -> g(f(?x))", "g(f(?x))->f(?x)", "a -> g(f(a))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(?x) -> h(g(?x),g(x))", "h(?x,g(?x))->f(?x)"]
	       val _ = print "\n\nThe next example does not succeed, because only finite narrowing is used in the current implementation."
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["f(?x) -> h(g(?x),g(x))", "h(?x,g(?x))->f(?x)", "a -> g(a)"]
	   in disproveOmegaSHN R
	   end
 
  val _ = let val R = IOFotrs.rdRules ["f(?x) -> h(g(?x),g(?x))", "h(?x,g(?x))->f(?x)", "g(a) -> a"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(S,?x),?y),?z) -> ap(ap(?x,?z),ap(?y,?z))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(K,?x),?y) -> ?x"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(I,?x) -> ?x"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(H,?x),?y),?z) -> ap(ap(ap(?x,?y),?z),?y)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(M,?x) -> ap(?x,?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(W,?x),?y) -> ap(ap(?x,?y),?y)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(W1,?x),?y) -> ap(ap(?y,?x),?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(Wa,?x),?y),?z) -> ap(ap(ap(?x,?y),?z),?z)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(Waa,?x),?y),?z),?w) -> ap(ap(ap(ap(?x,?y),?z),?w),?w)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(B,?x),?y),?z) -> ap(?x,ap(?y,?z))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(C,?x),?y),?z) -> ap(ap(?x,?z),?y)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(D,?x),?y),?z),?w) -> ap(ap(?x,?y),ap(?z,?w))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(ap(E,?x),?y),?z),?w),?v) -> ap(ap(?x,?y),ap(ap(?z,?w),?v))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(F,?x),?y),?z) -> ap(ap(?z,?y),?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(G,?x),?y),?z),?w) -> ap(ap(?x,?w),ap(?y,?z))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(Q,?x),?y),?z) -> ap(?y,ap(?x,?z))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(Q1,?x),?y),?z) -> ap(?x,ap(?z,?y))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(Q3,?x),?y),?z) -> ap(?z,ap(?x,?y))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(R,?x),?y),?z) -> ap(ap(?y,?z),?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(T,?x),?y) -> ap(?y,?x)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(V,?x),?y),?z) -> ap(ap(?z,?x),?y)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(V,?x),?y),?z) -> ap(ap(?z,?x),?y)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(Ca,?x),?y),?z),?w) -> ap(ap(ap(?x,?y),?w),?z)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(ap(Caa,?x),?y),?z),?w),?v) -> ap(ap(ap(ap(?x,?y),?z),?v),?w)"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(ap(ap(ap(Fa,?x),?y),?z),?k),?f) -> ap(?x,ap(?y,ap(?k,ap(?f,?z))))"]
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(Y,?x) -> ap(?x,ap(Y,?x))"]
	       val _ = print "\n\nThe next example should fail."
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(U,?x),?y) -> ap(?y,ap(ap(?x,?x),?y))"]
	       val _ = print "\n\nThe next example should fail."
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(U2,?x),?y) -> ap(?y,ap(ap(ap(?x,?x),?x),?y))"]
	       val _ = print "\n\nThe next example should fail."
	   in disproveOmegaSHN R
	   end

   val _ = let val R = IOFotrs.rdRules ["ap(ap(O,?x),?y) -> ap(?y,ap(?x,?y))"]
	       val _ = print "\n\nThe next example should fail."
	   in disproveOmegaSHN R
	   end

   in ()
   end 


   val R = IOFotrs.rdRules ["ap(ap(L,?x),?y) -> ap(?x,ap(?y,?y))"]
   val Lterm = valOf (fromString ("?z","{?z:=ap(ap(L,?x),?y),?x:=ap(L,?x)}"))
   val Lterm2 = valOf (rootRewrite R Lterm)
   val Lterm3 = valOf (rootRewrite R Lterm2)


   fun testUnify () = let

       val fff = valOf (fromString ("?x","{?x:=f(f(?x,?x),?x)}"))
       val _ = print ("fff = " ^ (toString fff) ^ "\n")

       val fff2 = valOf (fromString ("?x","{?x:=f(?x,f(?x,?x))}"))
       val _ = print ("fff2 = " ^ (toString fff2) ^ "\n")

       val _ = disjointUnify (fff,fff2)

       val fomega = valOf (fromString ("?x","{?x:=f(?x)}"))
       val _ = print ("fomega = " ^ (toString fomega) ^ "\n")
	       
       val fomega2 = valOf (fromString ("?x","{?x:=f(?y),?y:=f(?x)}"))
       val _ = print ("fomega2 = " ^ (toString fomega2) ^ "\n")

       val fomega3 = valOf (fromString ("?x","{?x:=f(?y),?y:=f(?y)}"))
       val _ = print ("fomega3 = " ^ (toString fomega3) ^ "\n")

       val _ = disjointUnify (fomega,fomega2)
       val _ = disjointUnify (fomega,fomega3)
       val _ = disjointUnify (fomega2,fomega3)

       val nonlinear = valOf (fromString ("?x","{?x:=f(?y,?y)}"))
       val _ = print ("nonlinear = " ^ (toString nonlinear) ^ "\n")

       val nonlinear2 = valOf (fromString ("?x","{?x:=f(?y,g(?y))}"))
       val _ = print ("nonlinear2 = " ^ (toString nonlinear2) ^ "\n")

       val nonlinear3 = valOf (fromString ("?x","{?x:=f(?y,g(?z)),?z:=g(?z)}"))
       val _ = print ("nonlinear3 = " ^ (toString nonlinear3) ^ "\n")

       val _ = disjointUnify (nonlinear,nonlinear2)
       val _ = disjointUnify (nonlinear,nonlinear3)
       val _ = disjointUnify (nonlinear2,nonlinear3)

       val nonunify = valOf (fromString ("?x","{?x:=f(?y,?y),?y:=h(?y)}"))
       val _ = print ("nonunify = " ^ (toString nonunify) ^ "\n")

       val nonunify2 = valOf (fromString ("?x","{?x:=f(?y,?y),?y:=g(?y)}"))
       val _ = print ("nonunify2 = " ^ (toString nonunify2) ^ "\n")

       val _ = disjointUnify (nonunify,nonunify2)

       val glob1 = valOf (fromString ("?x","{?x:=f(?y,?z,?w)}"))
       val _ = print ("glob1 = " ^ (toString glob1) ^ "\n")

       val glob2 = valOf (fromString ("?x","{?x:=f(?z,?z,?u)}"))
       val _ = print ("glob2 = " ^ (toString glob2) ^ "\n")

       val _ = disjointUnify (glob1,glob2)

   in () end


   end (* of local *)


   end (* of structure InfTerm *)




