(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/io_fotrs.sml
 * description: string I/O for first-order term rewriting systems
 * author: AOTO Takahito
 * 
 ******************************************************************************)

structure IOFotrs : sig

   include IO_TRS

   val rdRuleSet: string -> (TermKey.term * TermKey.term) list
   val rdEqSet: string -> (TermKey.term * TermKey.term) list
   val rdAssignmentSet: string -> (TermKey.term * TermKey.term) list

   val prRuleSet: (TermKey.term * TermKey.term) list -> string
   val prEqSet: (TermKey.term * TermKey.term) list -> string
   val prAssignmentSet: (TermKey.term * TermKey.term) list -> string

end = 
   struct

   structure FotrsKey =
      struct 
      val alphas = [ ]
      and symbols = ["{", "}", "(", ")", ",", "?", "->", "=", ":=", "_"]
      end;
   
   structure FotrsLex = Lexical (FotrsKey);
   structure FotrsParsing = Parsing (FotrsLex);

   local 
       open Term
       structure A = Atom
       structure S = Sort
       structure L = List
       structure LU = ListUtil
   in

   local
       infix 6 $--
       infix 5 --
       infix 3 >>
       infix 0 ||

       fun makeFun (id, ts) = Fun (A.atom id, ts, S.null)
       fun makeConst id = Fun (A.atom id, nil, S.null)
       fun makeConstSubsc (id, sub)  = Fun (A.atom (id ^ "_" ^ sub) , nil, S.null)
       fun makeVar id  = Var ((A.atom id, 0), S.null)
       fun makeVarSubsc (id, sub)  = 
		   case Int.fromString sub of
			   SOME n => Var ((A.atom id, n), S.null)
			 | NONE => raise  (FotrsParsing.SyntaxErr "non-numeric subscript")
       fun makeList (t, ts) = t::ts
       fun makeList1 t = t::nil
       fun makeNil _ = nil
       fun makeNum id = Int.fromString id

       open FotrsParsing

       fun rule toks =
           ( term -- "->" $-- term ) toks
       and equation toks =
           ( term -- "=" $-- term ) toks
       and assignment toks =
           ( term -- ":=" $-- term ) toks
       and term toks =
           ( id --  "(" $-- termlist >> makeFun || atom ) toks
       and termlist toks =
           ( $ ")" >> makeNil || termseq -- $ ")" >> #1 ) toks
       and termseq toks =
           ( term -- "," $-- termseq >> makeList || term >> makeList1 ) toks
       and atom toks  =
           ( "?" $-- id -- "_" $-- id >> makeVarSubsc  ||  "?" $-- id >> makeVar  
		 || id -- "_" $-- id >> makeConstSubsc || id >> makeConst ) toks
       and ruleset toks =
           ( "{" $-- rulelist) toks
       and rulelist toks =
           ( $ "}" >> makeNil || ruleseq -- $ "}" >> #1 ) toks
       and ruleseq toks =
           ( rule -- "," $-- ruleseq >> makeList || rule >> makeList1 ) toks
       and equationset toks =
           ( "{" $-- equationlist) toks
       and equationlist toks =
           ( $ "}" >> makeNil || equationseq -- $ "}" >> #1 ) toks
       and equationseq toks =
           ( equation -- "," $-- equationseq >> makeList || equation >> makeList1 ) toks
       and assignmentset toks =
           ( "{" $-- assignmentlist) toks
       and assignmentlist toks =
           ( $ "}" >> makeNil || assignmentseq -- $ "}" >> #1 ) toks
       and assignmentseq toks =
           ( assignment -- "," $-- assignmentseq >> makeList || assignment >> makeList1 ) toks

   in 
   val rdTerm = reader term
   val rdRule = reader rule
   val rdRuleSet = reader ruleset
   val rdEq = reader equation
   val rdEqSet = reader equationset
   val rdAssignment = reader assignment
   val rdAssignmentSet = reader assignmentset
   val rdTerms = L.map rdTerm
   val rdRules = L.map rdRule
   val rdEqs = L.map rdEq
   end

   fun prTerm (Var (x,ty)) = (Var.toString x)
	 | prTerm (Fun (f,[],ty)) = (Fun.toString f) 
	 | prTerm (Fun (f,ts,ty)) = (Fun.toString f) ^ (ListUtil.toStringCommaRound prTerm ts)
   fun prRule (l,r) = (prTerm l) ^ " -> " ^ (prTerm r)
   fun prEq (l,r) = (prTerm l) ^ " = " ^ (prTerm r)
   fun prAssignment (l,r) = (prTerm l) ^ " := " ^ (prTerm r)

   val prRuleSet = LU.toStringCommaCurly prRule
   val prEqSet = LU.toStringCommaCurly prEq
   val prAssignmentSet = LU.toStringCommaCurly prAssignment

   val prTerms = L.map prTerm
   val prRules = L.map prRule
   val prEqs = L.map prEq


   end (* of local *)

   end; (* of structure IOFotrs *)

