(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/io_mstrs.sml
 * description: string I/O for many-sorted term rewriting systems
 * author: AOTO Takahito
 * 
 ******************************************************************************)

structure IOMstrs : sig

   include IO_TRS

   val rdSort: string -> Sort.sort
   val rdSorts: string list -> Sort.sort list
   val rdDecl: string -> Term.decl
   val rdDecls: string list -> Term.decl list

   val rdTerm': string -> Term.term
   val rdRule': string -> Term.term * Term.term
   val rdEq': string -> Term.term * Term.term
   val rdTerms': string list -> Term.term list
   val rdRules': string list -> (Term.term * Term.term) list
   val rdEqs': string list -> (Term.term * Term.term) list

   val prTerm': Term.term -> string
   val prRule': Term.term * Term.term -> string
   val prEq': Term.term * Term.term -> string
   val prTerms': Term.term list -> string list
   val prRules': (Term.term * Term.term) list -> string list
   val prEqs': (Term.term * Term.term) list -> string list

end  =
  struct

  structure MstrsKey =
      struct 
      val alphas = [ ]
      and symbols = ["(", ")", ",", ":", "->", ">", "=", "*", "_"]
      end;

   structure MstrsLex = Lexical (MstrsKey);
   structure MstrsParsing = Parsing (MstrsLex);

   local 
       open Term
       structure A = Atom
       structure S = Sort
       structure L = List
   in

   local
       infix 6 $--
       infix 5 --
       infix 3 >>
       infix 0 ||
       fun makeFun (id,ts) = Fun (A.atom id, ts, S.null)
       fun makeConst id = Fun (A.atom id, nil, S.null)
       fun makeFun' ((id,ts),ty) = Fun (A.atom id, ts, ty)
       fun makeConst' (id, ty) = Fun (A.atom id, nil, ty)
       fun makeVar (id, ty)  = Var ((A.atom id, 0), ty)
       fun makeVarSubsc ((id, sub), ty)  = 
		   case Int.fromString sub of
			   SOME n => Var ((A.atom id, n), ty)
			 | NONE => raise  (MstrsParsing.SyntaxErr "non-numeric subscript")
       fun makeTermList (t, ts) = t::ts
       fun makeTermList1 t = t::nil
       fun makeNil _ = nil
       fun makeBase id = S.Base (A.atom id)
       fun makeProc (args,ty) = (args,ty)
       fun makeProc1 ty = (nil,ty)
       fun makeArgs (ty,args) = ty :: args
       fun makeArgs1 ty = ty :: nil
       fun makeSort (args,ty) = if (null args) then ty else S.Proc (args,ty)
       fun makeDecl (id,(ss,s)) = {sym=A.atom id, sort=makeSort (ss,s)}

       open MstrsParsing

       fun rule toks =
           ( term -- "->" $-- term ) toks
       and equation toks =
           ( term -- "=" $-- term ) toks
       and term toks =
           ( id --  "(" $-- termlist >> makeFun || atom ) toks
       and termlist toks =
           ( $ ")" >> makeNil || termseq -- $ ")" >> #1 ) toks
       and termseq toks =
           ( term -- "," $-- termseq >> makeTermList || term >> makeTermList1 ) toks
       and atom toks  =
           ( "?" $-- id -- "_" $-- id -- ":" $-- sort >> makeVarSubsc
             || "?" $-- id -- ":" $-- sort >> makeVar 
             || id >> makeConst ) toks
       and rule' toks =
           ( term' -- "->" $-- term' ) toks
       and equation' toks =
           ( term' -- "=" $-- term' ) toks
       and term' toks =
           ( id --  "(" $-- termlist' -- ":" $-- sort >> makeFun' || atom ) toks
       and termlist' toks =
           ( $ ")" >> makeNil || termseq' -- $ ")" >> #1 ) toks
       and termseq' toks =
           ( term' -- "," $-- termseq' >> makeTermList || term' >> makeTermList1 ) toks
       and atom' toks  =
           ( "?" $-- id -- "_" $-- id -- ":" $-- sort >> makeVarSubsc 
                || "?" $-- id -- ":" $-- sort >> makeVar 
                || id -- ":" $-- sort >> makeConst' ) toks
       and sort toks  =
           ( id >> makeBase) toks
       and decl toks  =
           ( id -- ":" $-- proc >> makeDecl) toks
       and proc toks  =
           ( args -- "=" $-- (">" $-- sort ) >> makeProc || sort >> makeProc1 ) toks        
       and args toks  =
           ( sort -- "*" $-- args >> makeArgs || sort >> makeArgs1 ) toks        

   in 
   val rdTerm = reader term
   val rdRule = reader rule
   val rdEq' = reader equation'
   val rdTerm' = reader term'
   val rdRule' = reader rule'
   val rdEq = reader equation
   fun rdSort str = makeSort (reader proc str)
   val rdDecl = reader decl
   end

   val rdTerms = map rdTerm
   val rdRules = map rdRule
   val rdEqs = map rdEq
   val rdTerms' = map rdTerm'
   val rdRules' = map rdRule'
   val rdEqs' = map rdEq'
   val rdDecls = map rdDecl
   val rdSorts = map rdSort

   fun prTerm (Var (x,ty)) = (Var.toString x) ^ ":" ^ (Sort.toString ty)
	 | prTerm (Fun (f,[],ty)) = (Fun.toString f) 
	 | prTerm (Fun (f,ts,ty)) = (Fun.toString f) ^ (ListUtil.toStringCommaRound prTerm ts)
   fun prRule (l,r) = (prTerm l) ^ " -> " ^ (prTerm r)
   fun prEq (l,r) = (prTerm l) ^ " = " ^ (prTerm r)

   val prTerms = L.map prTerm
   val prRules = L.map prRule
   val prEqs = L.map prEq

   fun prTerm' (Var (x,ty)) = (Var.toString x) ^ ":" ^ (Sort.toString ty)
	 | prTerm' (Fun (f,[],ty)) = (Fun.toString f) ^ ":" ^ (Sort.toString ty)
	 | prTerm' (Fun (f,ts,ty)) = (Fun.toString f) 
								 ^ (ListUtil.toStringCommaRound prTerm ts)
								 ^ ":" ^ (Sort.toString ty)
   fun prRule' (l,r) = (prTerm' l) ^ " -> " ^ (prTerm' r)
   fun prEq' (l,r) = (prTerm' l) ^ " = " ^ (prTerm' r)

   val prTerms' = L.map prTerm'
   val prRules' = L.map prRule'
   val prEqs' = L.map prEq'


   end (* of local *)

   end; (* of structure IoMstrs *)


