(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/pos.sml
 * description: definition and utility functions for Term Positions
 * author: AOTO Takahito
 * 
 ******************************************************************************)

structure Pos : sig

   include ORD_KEY2 (* { val ord_key, fun compare, fun equal, fun toString }*)

   type position = ord_key
   val isPrefix: position -> position -> bool
   val isStrictPrefix: position -> position -> bool
   val cut: position -> position -> position option
   val insert: (position * position list) -> position list
   val cons: (position * position list) -> position list
   val insertMinimal: position -> position list -> position list
   val selectMinimals: position list -> position list
   val unionMinimals: position list -> position list -> position list
   val toProofTree: position -> unit -> string

end = struct  

   local 
       structure L = List
   in
   exception TermPositionError;
   type ord_key = int list 
   type position = ord_key

   fun compare ([], []) = EQUAL
     | compare (x::xs, []) = GREATER
     | compare ([], y::ys) = LESS
     | compare (x::xs, y::ys) = case Int.compare (x,y) of
				    GREATER => GREATER
				  | LESS => LESS
				  | EQUAL => compare (xs,ys)

   fun equal (xs,ys) = (compare (xs,ys)) = EQUAL

   fun toString [] = "e" 
     | toString (x::[]) = (Int.toString x)
     | toString (x::xs) = (Int.toString x) ^ "." ^ (toString' xs)
   and toString' [] = "" 
     | toString' xs = toString xs

   (* isPrefix pos1 pos2: pos1 <= pos2 $B$+(B *)
    fun isPrefix [] _ = true
      | isPrefix (x::xs) [] = false
      | isPrefix (x::xs) (y::ys) = (x = y) andalso (isPrefix xs ys)

   (* isStrictPrefix pos1 pos2: pos1 < pos2 $B$+(B *)
    fun isStrictPrefix [] [] = false
      | isStrictPrefix [] (y::ys) = true
      | isStrictPrefix (x::xs) [] = false
      | isStrictPrefix (x::xs) (y::ys) = (x = y) andalso (isStrictPrefix xs ys)

   (* cut pos1 pos2: w such that pos1.w = pos2 *)
    fun cut [] ys = SOME ys
      | cut (x::xs) [] = NONE
      | cut (x::xs) (y::ys) = if (x = y) 
			      then cut xs ys
			      else NONE

    (* compare $B$N>:=g$rJ];}$9$k$h$&$K(B insert$B!$$?$@$7F1$8$b$N$,$"$l$PDI2C$7$J$$(B *)
    fun insert (x,[]) = [x]
      | insert (x,y::ys) = case compare (x,y) of
			       GREATER => y::(insert (x,ys))
			     | LESS => x::y::ys
			     | EQUAL => y::ys

    (* compare $B$N>:=g$rJ];}$9$k$h$&$K(B insert$B!$F1$8$b$N$,$"$C$F$bDI2C(B *)
    fun cons (x,[]) = [x]
      | cons (x,y::ys) = case compare (x,y) of
			       GREATER => y::(cons (x,ys))
			     | LESS => x::y::ys
			     | EQUAL => x::y::ys

   (* minimal positions (w.r.t. prefix ordering) 
      $B$+$D(B compare $B$N>:=g$K$J$k$h$&$K(B insert *)
    fun insertMinimal x ys = 
	if L.exists (fn y=> isStrictPrefix y x) ys
	then ys
	else insert (x, L.filter (fn y=> not (isPrefix x y)) ys)

    fun selectMinimals [] = []
      | selectMinimals (x::xs) = insertMinimal x (selectMinimals xs)

    fun unionMinimals [] ys = ys
      | unionMinimals (x::xs) ys = insertMinimal x (unionMinimals xs ys)

   end

   fun toProofTree pos () = 
       if null pos
       then CertifyUtil.encloseProofLeafBy "positionInTerm" ""
       else CertifyUtil.encloseProofTreesBy "positionInTerm"
	    (List.map (fn p => fn _ => CertifyUtil.encloseProofLeafBy "position" (Int.toString p)) pos)

end;
