(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/rewrite.sml
 * description: various rewriting 
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature REWRITE = 
   sig
   val linf: (Term.term * Term.term) list -> Term.term -> Term.term
   val linfcount: (Term.term * Term.term) list -> int -> Term.term -> Term.term
   val statusLinf: (Term.term * Term.term) list -> Term.term -> (bool * Term.term)
   val linfPartial: (Term.term * Term.term) list -> Term.term -> Term.term option
   val listep: (Term.term * Term.term) list -> Term.term -> Term.term option
   val listepWithIndex: (Term.term * Term.term) list -> Term.term -> (int * Term.term) option
   val listepsToNF: (Term.term * Term.term) list -> Term.term -> Term.term list
   val listepsToNFwithIndex: (Term.term * Term.term) list -> Term.term -> (int option * Term.term) list
   val listepsToNFwithRule: (Term.term * Term.term) list -> Term.term
			    -> ((Term.term * Term.term) option * Term.term) list
   val rootRewrite: (Term.term * Term.term) list -> Term.term -> Term.term option
   val rootRewriteWithSubst: (Term.term * Term.term) list -> Term.term -> (Term.term * Subst.subst) option
   val rootRewriteAll: (Term.term * Term.term) list -> Term.term -> Term.term list
   val rootRewriteWithSubstAll: (Term.term * Term.term) list -> Term.term -> (Term.term * Subst.subst) list
   val rootRewriteAllSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val isNormalForm: (Term.term * Term.term) list -> Term.term -> bool
   val oneStepReducts: (Term.term * Term.term) list -> Term.term -> Term.term list
   val oneStepReductSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val oneStepReductsWithPos: 
       (Term.term * Term.term) list -> Term.term ->  (Term.position * Term.term) list
   val oneStepReductsWithRestrictedPosition:
       (Term.term * Term.term) list -> int list -> Term.term -> Term.term list
   val isMinimalForm: (Term.term * Term.term) list -> Term.term -> bool
   val isHeadStable: (Term.term * Term.term) list -> Term.term -> bool

   val oneStepReductsWithSubst: (Term.term * Term.term) list -> 
			       Term.term -> (Term.term * Subst.subst) list
   val oneStepReductsWithIndex: (Term.term * Term.term) list -> 
			       Term.term -> (int * Term.term) list
   val oneStepReductsWithRule: (Term.term * Term.term) list -> 
			       Term.term -> ((Term.term * Term.term) * Term.term) list
   val oneStepReductsWithIndexAndPos: (Term.term * Term.term) list -> 
			       Term.term -> (Term.position * int * Term.term) list
   val oneStepReductWithIndexSet: (Term.term * Term.term) list -> Term.term -> IntTermSet.set

   val statusOneStepRewrite: (Term.term * Term.term) list -> Term.term -> (bool * Term.term)
   val oneStepRewritePartial: (Term.term * Term.term) list -> Term.term -> Term.term option

   val liOneStepReductWithIndex: (Term.term * Term.term) list -> Term.term -> (int * Term.term) option
   val riOneStepReductWithIndex: (Term.term * Term.term) list -> Term.term -> (int * Term.term) option
   val loOneStepReductWithIndex: (Term.term * Term.term) list -> Term.term -> (int * Term.term) option

   val loOneStepReductWithIndexAndFs:
       (Term.term * Term.term) list -> Term.term -> (Fun.ord_key list * int * Term.term) option

   val parallelOneStepReducts: (Term.term * Term.term) list -> Term.term -> Term.term list
   val nonRootParallelOneStepReducts: (Term.term * Term.term) list -> Term.term -> Term.term list
   val parallelOneStepReductSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val parallelOneStepReductsWithSubst: (Term.term * Term.term) list -> Term.term 
					-> (Term.term * Subst.subst list) list
   val nonRootParallelOneStepReductSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val parallelOneStepReductsOfRestrictedDepthLength:
       (Term.term * Term.term) list -> int -> Term.term -> Term.term list
   val parallelOneStepReductsWithVar: (Term.term * Term.term) list -> Term.term 
				      -> (Term.term * VarSet.set) list
   val parallelOneStepReductsWithIndex: 
       (Term.term * Term.term) list -> Term.term -> (IntSet.set * Term.term) list
   val developOneStepReducts: (Term.term * Term.term) list -> Term.term -> Term.term list
   val developOneStepReductSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val developOneStepReductsWithIndex: 
       (Term.term * Term.term) list -> Term.term -> (IntSet.set * Term.term) list
   val developTwoStepsReducts: (Term.term * Term.term) list -> Term.term -> Term.term list
   val developTwoStepsReductSet: (Term.term * Term.term) list -> Term.term -> TermSet.set

   val minimalReductSet: (Term.term * Term.term) list -> int -> Term.term -> TermSet.set option 
   val manyStepsReductSet: (Term.term * Term.term) list -> int -> Term.term -> TermSet.set
   val manyStepsReductsWithSubst: (Term.term * Term.term) list -> int ->
			       Term.term -> (Term.term * Subst.subst list) list
   val manyStepsReductsWithPosAndIndex: 
       (Term.term * Term.term) list -> int -> Term.term 
       -> (Term.position list * int list * Term.term) list
   val reductSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val normalFormSet: (Term.term * Term.term) list -> Term.term -> TermSet.set
   val isEquivalent: (Term.term * Term.term) list -> (Term.term * Term.term) -> bool
   val isEquivalentWithLimit: int -> (Term.term * Term.term) list 
			      -> (Term.term * Term.term) -> bool
   val containedInReductSet:
       (Term.term * Term.term) list -> Term.term -> Term.term -> bool
   val oneStepModuloRewriteWithNFset: (Term.term * Term.term) list 
				      -> (Term.term * Term.term) list 
				      -> Term.term -> (Term.term option * TermSet.set)
   val moduloLinf: (Term.term * Term.term) list 
		   -> (Term.term * Term.term) list 
		   -> Term.term -> TermSet.set
   val isModuloReachable:(Term.term * Term.term) list 
		   -> (Term.term * Term.term) list 
		   -> Term.term -> TermSet.set -> bool
   val isModuloJoinable:(Term.term * Term.term) list 
		   -> (Term.term * Term.term) list 
		   -> (Term.term * Term.term) -> bool

   val isModuloJoinableWithLimit: int -> (Term.term * Term.term) list 
				  -> (Term.term * Term.term) list 
				  -> (Term.term * Term.term) -> bool

   val statusModuloLinf: (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> (bool * Term.term)
   val moduloLinfPartial: (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> Term.term option
   val moduloLinfWithNFset:(Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> (Term.term option * TermSet.set)

   val rootConstrainedRewrite: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> Term.term -> Term.term option

   val rootConstrainedRewrite2 : (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> Term.term option

   val rootConstrainedRewriteAll: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> Term.term -> Term.term list

   val rootConstrainedRewriteAll2: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> Term.term list

   val statusOneStepConstrainedRewrite: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> (bool * Term.term)

   val oneStepConstrainedRewritePartial: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> (Term.term option)

   val oneStepConstrainedReducts: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> Term.term list

   val constrainedReductSet: (Term.term * Term.term -> bool)
			 -> (Term.term * Term.term) list 
			 -> (Term.term * Term.term) list 
			 -> Term.term -> TermSet.set

   val clinf: (Term.term * Term.term -> bool)
	      -> (Term.term * Term.term) list 
	      -> (Term.term * Term.term) list 
	      -> Term.term -> Term.term

   val statusClinf: (Term.term * Term.term -> bool)
	      -> (Term.term * Term.term) list 
	      -> (Term.term * Term.term) list 
	      -> Term.term -> bool * Term.term

   val clinfPartial:(Term.term * Term.term -> bool)
	      -> (Term.term * Term.term) list 
	      -> (Term.term * Term.term) list 
	      -> Term.term -> Term.term option

   val moduloClinf: (Term.term * Term.term -> bool)
		    -> (Term.term * Term.term -> bool)
		    -> (Term.term * Term.term) list 
		    -> (Term.term * Term.term) list 
		    -> (Term.term * Term.term) list 
		    -> Term.term -> Term.term

   val statusModuloClinf: (Term.term * Term.term -> bool)
		    -> (Term.term * Term.term -> bool)
		    -> (Term.term * Term.term) list 
		    -> (Term.term * Term.term) list 
		    -> (Term.term * Term.term) list 
		    -> Term.term -> bool * Term.term

   val moduloClinfPartial:(Term.term * Term.term -> bool)
		    -> (Term.term * Term.term -> bool)
		    -> (Term.term * Term.term) list 
		    -> (Term.term * Term.term) list 
		    -> (Term.term * Term.term) list 
		    -> Term.term -> Term.term option

   val strengthenRewriteSeq: (Term.term * Term.term) list 
			     -> Term.term list list 
			     -> Term.term list list 

   val strengthenRewriteSeqWithIndex: (Term.term * Term.term) list 
				      -> (int * Term.term) list list 
				      -> (int * Term.term) list list 

   val nStepsRewriteSeqWithIndex: (Term.term * Term.term) list
				  -> int
				  -> (int * Term.term)
				  -> (int * Term.term) list list 

   val strengthenRewriteSeqWithIndexAndFs: (Term.term * Term.term) list 
				      -> (Fun.ord_key list * int * Term.term) list list 
				      -> (Fun.ord_key list * int * Term.term) list list 

   val strengthenRewriteSeqWithIndexAndPosFs: (Term.term * Term.term) list 
				      -> ((Fun.ord_key * int) list * int * Term.term) list list 
				      -> ((Fun.ord_key * int) list * int * Term.term) list list 


   val narrow: Term.position -> (Term.term * Term.term) -> Term.term -> (Term.term * Subst.subst) option
   val repeatNarrow: 
       Term.term -> (Pos.ord_key list * (Term.term * Term.term) list) -> (Term.term * Subst.subst) option
   val rootNarrow: (Term.term * Term.term) list -> Term.term -> (Term.term * Subst.subst) option
   val rootNarrowAll: (Term.term * Term.term) list -> Term.term -> (Term.term * Subst.subst) list
   val oneStepNarrow: (Term.term * Term.term) list -> Term.term -> (Term.term * Subst.subst) list
   val oneStepOutermostNarrow: (Term.term * Term.term) list -> Term.term -> (Term.term * Subst.subst) list
   val oneStepOutermostNarrowWithPosition: 
       (Term.term * Term.term) list -> Term.term -> (Term.position * Term.term * Subst.subst) list

   val forwardOneStepUnfoldingsOfRule: 
       (Term.term * Term.term) list -> Term.term * Term.term -> (Term.term * Term.term) list
   val backwardOneStepUnfoldingsOfRule:
       (Term.term * Term.term) list -> Term.term * Term.term -> (Term.term * Term.term) list
   val forwardOneStepUnfoldingsOfRules: 
       (Term.term * Term.term) list -> (Term.term * Term.term) list -> (Term.term * Term.term) list
   val backwardOneStepUnfoldingsOfRules:
       (Term.term * Term.term) list -> (Term.term * Term.term) list -> (Term.term * Term.term) list
   val oneStepUnfoldingsOfRules: 
       (Term.term * Term.term) list -> (Term.term * Term.term) list
   val multiStepsUnfoldingsOfRules: 
       (Term.term * Term.term) list -> int -> (Term.term * Term.term) list

   val findRewriteSequence: (Term.term * Term.term) list -> int -> Term.term -> Term.term 
			    -> (Term.position * (Term.term * Term.term) * Term.term) list option

   val maxRewriteStepsForRev: int ref
   val isReversibleRules: (Term.term * Term.term) list -> bool

   val prRewriteStepInProofTree: (Term.position * (Term.term * Term.term) * Term.term)
				 -> unit -> string

   val prRewriteSeqInProofTree: (Term.term * (Term.position * (Term.term * Term.term) * Term.term) list)
				-> unit -> string

   val prRewriteStepInProofTree2: ((Term.term * Term.term) * Term.term)
				 -> unit -> string

   val prRewriteSeqInProofTree2: (Term.term * ((Term.term * Term.term) * Term.term) list)
				-> unit -> string

end;

structure Rewrite : REWRITE = 
   struct

   local 
       open Term
       open Subst
       structure CU = CertifyUtil
       structure VS = VarSet
       structure VM = VarMap
       structure FS = FunSet
       structure FM = FunMap
       structure SS = SortSet
       structure FIS = FunIntSet
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
       structure TS = TermSet
       structure ITS = IntTermSet
       structure IS = IntSet
       fun cartesianProduct xs ys = ListXProd.mapX (fn xy => xy) (xs,ys)
       fun allCombinations [] = [[]]
	 | allCombinations (xs::[]) = L.map (fn x => [x]) xs
	 | allCombinations (xs::xss) = 
	   L.map (fn (x,xs) => x::xs) (cartesianProduct xs (allCombinations xss))

   in

   (* rootRewrite R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥå׽񤭴 *)
   fun rootRewrite [] term = NONE
     | rootRewrite ((l,r)::rs) term = 
       case match l term
	of SOME sigma => SOME (applySubst sigma r)
	 | NONE => rootRewrite rs term


   (* rootRewrite R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥå׽񤭴 *)
   (* Ѥդ *)
   fun rootRewriteWithSubst [] term = NONE
     | rootRewriteWithSubst ((l,r)::rs) term = 
       case match l term
	of SOME sigma => SOME (applySubst sigma r, sigma)
	 | NONE => rootRewriteWithSubst rs term

  (* rootRewrite Υǥåդǡ
        R=[l_0->r_0,...,l_n->r_n] ȤƻȤä롼ΥǥåĤ *)

   fun rootRewriteWithIndex R term = 
       let
	   fun rootRewriteWI [] _ term = NONE
	     | rootRewriteWI ((l,r)::rs) n term = 
	       case match l term
		of SOME sigma => SOME (n, applySubst sigma r)
		 | NONE => rootRewriteWI rs (n+1) term
       in rootRewriteWI R 0 term
       end


   (* rootRewriteAll R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥå׽񤭴Υꥹ *)
   fun rootRewriteAll R term = 
       L.mapPartial (fn rule => rootRewrite [rule] term)  R

   fun rootRewriteAllSet R term = 
       TS.addList(TS.empty, rootRewriteAll R term)

   fun rootRewriteWithSubstAll R term = 
       L.mapPartial (fn rule => rootRewriteWithSubst [rule] term)  R

  (* rootRewriteAll Υǥåդǡ
        R=[l_0->r_0,...,l_n->r_n] ȤƻȤä롼Υǥå
     ȤƤĤ *)

   fun rootRewriteAllWithIndex R term = 
       let
	   fun rootRewriteWI [] _ term = []
	     | rootRewriteWI ((l,r)::rs) n term = 
	       case match l term
		of SOME sigma => (n, applySubst sigma r) :: (rootRewriteWI rs (n+1) term)
		 | NONE => rootRewriteWI rs (n+1) term
       in rootRewriteWI R 0 term
       end


   (* normal form  *)
   fun isNormalForm R term = 
       let fun isNF [] = true
	     | isNF (t::ts) = 
	       case t 
		of Var _ => isNF ts
		 | Fun (f,ss,sort) => case rootRewrite R t
				       of SOME _ => false
					| NONE => isNF (ss @ ts)
       in 
	   isNF [term]
       end

  (* left innermost ά 1 step (rule index դ) *)
   fun liOneStepReductWithIndex R term =
       case term 
	of Var _ => NONE
	 | Fun (f,ts,sort) =>case liOneStepReductWithIndexList R ts of
				 NONE => rootRewriteWithIndex R term
			       | SOME (i,ss) => SOME (i, Fun (f,ss,sort))
   and liOneStepReductWithIndexList R [] = NONE
     | liOneStepReductWithIndexList R (t::ts) = 
       case liOneStepReductWithIndex R t of
	   NONE => (case liOneStepReductWithIndexList R ts of
			NONE => NONE
		      | SOME (i,ss) => SOME (i,t::ss))
	 | SOME (i,s) => SOME (i,s::ts)

  (* right innermost ά 1 step (rule index դ) *)
   fun riOneStepReductWithIndex R term =
       case term 
	of Var _ => NONE
	 | Fun (f,ts,sort) =>case liOneStepReductWithIndexList R (rev ts) of
				 NONE => rootRewriteWithIndex R term
			       | SOME (i,ss) => SOME (i, Fun (f,rev ss,sort))

  (* left outermost ά 1 step (rule index դ) *)
   fun loOneStepReductWithIndex R term =
       case term 
	of Var _ => NONE
	 | Fun (f,ts,sort) =>case rootRewriteWithIndex R term of
				 NONE => (case loOneStepReductWithIndexList R ts of
					      NONE => NONE
					    | SOME (i,ss) => SOME (i, Fun (f,ss,sort)))
			       | SOME (i,t) => SOME (i,t)
   and loOneStepReductWithIndexList R [] = NONE
     | loOneStepReductWithIndexList R (t::ts) = 
       case loOneStepReductWithIndex R t of
	   NONE => (case loOneStepReductWithIndexList R ts of
			NONE => NONE
		      | SOME (i,ss) => SOME (i,t::ss))
	 | SOME (i,s) => SOME (i,s::ts)


  (* left outermost ά 1 step (rule index , root ޤǤδؿ դ) *)
   fun loOneStepReductWithIndexAndFs R term =
       case term 
	of Var _ => NONE
	 | Fun (f,ts,sort) =>case rootRewriteWithIndex R term of
				 NONE => (case loOneStepReductWithIndexAndFsList R ts of
					      NONE => NONE
					    | SOME (fs,i,ss) => SOME (f::fs, i, Fun (f,ss,sort)))
			       | SOME (i,t) => SOME ([],i,t)
   and loOneStepReductWithIndexAndFsList R [] = NONE
     | loOneStepReductWithIndexAndFsList R (t::ts) = 
       case loOneStepReductWithIndexAndFs R t of
	   NONE => (case loOneStepReductWithIndexAndFsList R ts of
			NONE => NONE
		      | SOME (fs,i,ss) => SOME (fs,i,t::ss))
	 | SOME (fs,i,s) => SOME (fs,i,s::ts)



   (* linf R term : 񤭴§ R ΤȤǤ term  *)
   fun linf R term =
       case term 
	of Var _ => term
	 | Fun (f,ts,sort) => let val ss = map (linf R) ts
			      in case rootRewrite R (Fun (f,ss,sort))
				  of SOME t => linf R t
				   | NONE => Fun (f,ss,sort)
			      end

   (* linfcount R count term : 񤭴§ R ΤȤǤ term   
                               󥿤 *)
   fun linfcount R 0 term = term
     | linfcount R n term = 
       case term 
	of Var _ => term
	 | Fun (f,ts,sort) => let val ss = map (linfcount R n) ts
			      in case rootRewrite R (Fun (f,ss,sort))
				  of SOME t => linfcount R (n-1) t
				   | NONE => Fun (f,ss,sort)
			      end

   (* 񤭴Ƥ (true, nf)Ƥʤ (false, nf) ֤ *)
   fun statusLinf R term =
       case term 
	of Var _ => (false, term)
	 | Fun (f,ts,sort) => let val statusXss = L.map (statusLinf R) ts
				  val ss = L.map #2 statusXss
			      in case rootRewrite R (Fun (f,ss,sort))
				  of SOME t => (true, linf R t)
				   | NONE => (L.exists (fn x => #1 x) statusXss, Fun (f,ss,sort))
			      end

   (* 񤭴Ƥ SOME nfƤʤ NONE ֤ *)
   fun linfPartial R term =
       let val (status, nf) = statusLinf R term
       in if status then SOME nf else NONE
       end

   fun listep R (Var _) = NONE
     | listep R (term as Fun (f,ts,sort)) = case listepList R ts of
					SOME ts2 => SOME (Fun (f,ts2,sort))
				      | NONE => rootRewrite R term
   and listepList R [] = NONE
     | listepList R (t::ts) = (case listep R t of
				  SOME t2 => SOME (t2::ts)
				| NONE => (case listepList R ts of
					      SOME ts2 => SOME (t::ts2)
					    | NONE => NONE))

   fun listepWithIndex R (Var _) = NONE
     | listepWithIndex R (term as Fun (f,ts,sort)) = case listepListWithIndex R ts of
						 SOME (i,ts2) => SOME (i, Fun (f,ts2,sort))
					       | NONE => rootRewriteWithIndex R term
   and listepListWithIndex R [] = NONE
     | listepListWithIndex R (t::ts) = (case listepWithIndex R t of
				  SOME (i,t2) => SOME (i,t2::ts)
				| NONE => (case listepListWithIndex R ts of
					      SOME (i,ts2) => SOME (i,t::ts2)
					    | NONE => NONE))

   (* linfseq R term : 񤭴§ R ΤȤǤ term 餽ޤǤν񤭴ƥåס
      񤭴衤Ƭterm Ǹ *)
   fun listepsToNF R term = case listep R term of
				SOME t => term:: listepsToNF R t
			      | NONE => [term]


   fun listepsToNFwithIndex R term = case listepWithIndex R term of
				SOME (i,t) => (NONE,term) :: (SOME i,t) :: tl (listepsToNFwithIndex R t)
			      | NONE => [(NONE,term)]

   fun listepsToNFwithRule R term =
       let fun getRule (SOME i) = SOME (L.nth (R,i))
	     | getRule NONE = NONE
       in L.map (fn (i,t) => (getRule i, t)) (listepsToNFwithIndex R term)
       end


   (* 1ƥåפ¹Խ񤭴轸 (0ƥåפޤ) *)
    (* 0ƥåפޤळȤϡѿξOKޤallCombinations [] = [[]] ʤΤ
       ξ(ih )ޤޤ롤ȤϵǼˡǼ *)
   fun parallelOneStepReducts R term =
       case term 
	of Var _ => [term]
	 | Fun (f,ts,sort) => 
	   let val ih = L.map 
			    (fn ss => Fun (f,ss,sort))
			    (allCombinations (map (parallelOneStepReducts R) ts))
	   in
	       L.@ (ih, rootRewriteAll R term)
	   end

  (* root Ǥν񤭴¹Խ񤭴Υꥹ *)
   fun nonRootParallelOneStepReducts R term =
       case term 
	of Var _ => [term]
	 | Fun (f,ts,sort) => 
	   L.map 
	       (fn ss => Fun (f,ss,sort))
	       (allCombinations (map (parallelOneStepReducts R) ts))

   fun nonRootParallelOneStepReductSet R term =
       TS.addList(TS.empty, nonRootParallelOneStepReducts R term)

   fun parallelOneStepReductSet R term =
       TS.addList(TS.empty, parallelOneStepReducts R term)


   fun parallelOneStepReductsWithSubst R term =
       case term 
	of Var _ => [(term,[])]
	 | Fun (f,ts,sort) => 
	   let val ih = L.map 
			    (fn ss => let val (args,subs) = LP.unzip ss
				      in (Fun (f,args,sort),LU.mapAppend (fn x=>x) subs) end)
			    (allCombinations (map (parallelOneStepReductsWithSubst R) ts))
	       val rootReducts = L.map (fn (t,sigma) => (t,[sigma])) (rootRewriteWithSubstAll R term)
	   in
	       L.@ (ih, rootReducts)
	   end


   (* (1ƥåפ¹Խ񤭴衤񤭴ʬreductѿ) ν *)
   fun parallelOneStepReductsWithVar R term =
       case term 
	of Var _ => [(term,VS.empty)]
	 | Fun (f,ts,sort) => 
	   let val ih = L.map 
			    (fn sswv => let val (ss,vSetList) = LP.unzip sswv
					    val vSet = L.foldl VS.union VS.empty vSetList
					in (Fun (f,ss,sort), vSet)
					end)
			    (allCombinations (map (parallelOneStepReductsWithVar R) ts))
	   in
	       L.@ (ih, L.map (fn u => (u, varSetInTerm u)) (rootRewriteAll R term))
	   end

   (* 1ƥåפ¹Խ񤭴ꥹȡ depth length k ʲǽ񤭴 *)
   fun parallelOneStepReductsOfRestrictedDepthLength R k term = 
       if k < 0
       then [term]
       else case term 
	     of Var _ => [term]
	      | Fun (f,ts,sort) => 
		let val ih = L.map 
				 (fn ss => Fun (f,ss,sort))
				 (allCombinations 
				      (map (parallelOneStepReductsOfRestrictedDepthLength R (k-1))
					   ts))
		in
		    L.@ (ih, rootRewriteAll R term)
		end


   (* rootDevelop R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥådevelop 񤭴ν *)
   fun rootDevelopReducts R term = 
       let fun rootDevelopSub [] term = [term]
	     | rootDevelopSub ((l,r)::rs) term =
	       case match l term
		of SOME sigma => 
		   let val subst = 
			   L.map (fn x => (x, 
					   case VM.find (sigma,x) of 
					       SOME t => developOneStepReducts R t
					     | NONE => []))
				 (VS.listItems (varSetInTerm r))
		       val devSubsts = developSubst subst [VM.empty]
		   in
		       (L.map (fn sigma => applySubst sigma r) devSubsts)
		       @ (rootDevelopSub rs term)
		   end
		 | NONE => rootDevelopSub rs term
       in
	   rootDevelopSub R term
       end

   and developSubst [] sigmas = sigmas
     | developSubst ((x,ts)::xs) sigmas = 
       developSubst xs 
		    (ListXProd.mapX (fn (t,sigma) => VM.insert (sigma,x,t))
				    (ts,sigmas))

   and developOneStepReducts R term =
       case term 
	of Var _ => [term]
	 | Fun (f,ts,sort) => 
	   let val ih = L.map 
			    (fn ss => Fun (f,ss,sort))
			    (allCombinations (map (developOneStepReducts R) ts))
	   in
(*	       term :: (L.@ (rootDevelopReducts R term, ih)) ; corrected to eliminate duplication *)
	       L.@ (ih, rootDevelopReducts R term)
	   end

   fun developOneStepReductSet R term =
       TS.addList(TS.empty, developOneStepReducts R term)

   fun developTwoStepsReducts R term =
       let val tset = developOneStepReductSet R term
       in  LU.mapAppend (developOneStepReducts R) (TS.listItems tset)
       end

   fun developTwoStepsReductSet R term =
       let val tset = developOneStepReductSet R term
       in  TS.addList (TS.empty, LU.mapAppend (developOneStepReducts R) (TS.listItems tset))
       end

   local
       fun cartesianProduct f xs ys = ListXProd.mapX (fn xy => f xy) (xs,ys)
       fun allCombinations _ [] = [(IS.empty,[])]
	 | allCombinations _ (xs::[]) = L.map (fn (x,y) => (x,[y])) xs
	 | allCombinations f (xs::xss) = 
	   (cartesianProduct f xs (allCombinations f xss))
   in
   (* ǥåդ1ƥåפ¹Խ񤭴轸 (0ƥåפޤ) *)
   fun parallelOneStepReductsWithIndex R term =
       case term 
	of Var _ => [(IS.empty, term)]
	 | Fun (f,ts,sort) => 
	   let val ih = L.map 
			    (fn (iset,ss) => (iset, Fun (f,ss,sort)))
			    (allCombinations 
				 (fn ((a,x),(b,xs)) => (IS.union (a,b), x::xs))
				 (L.map (parallelOneStepReductsWithIndex R) ts))
	   in
(*	       (IS.empty, term) :: 
	       (L.map (fn (i,t) => (IS.singleton i, t)) (rootRewriteAllWithIndex R term))
	       @ ih  ; corrected to eliminate duplication  *)
	     L.@(ih,
		 L.map (fn (i,t) => (IS.singleton i, t)) (rootRewriteAllWithIndex R term))
	   end

   (* rootDevelop R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥådevelop 񤭴ν *)
   fun rootDevelopReductsWithIndex R term =
       let fun applySubstWI n sigma t =
	       let 
		   val sub = VM.map (fn (_,t) => t) sigma
		   val aset = VM.foldl (fn ((a,_),b) => IS.union (a,b)) IS.empty sigma
	       in (IS.add (aset,n), applySubst sub t)
	       end

	   fun rootDevelopSub [] _ term = [(IS.empty,term)]
	     | rootDevelopSub ((l,r)::rs) n term =
	       case match l term
		of SOME sigma =>
		   let 
		       val subst =
			   L.map (fn x => (x,
					   case VM.find (sigma,x) of
					       SOME t => developOneStepReductsWithIndex2 R t
					     | NONE => []))
				 (VS.listItems (varSetInTerm r))
		       val devSubsts = developSubstWithIndex subst [VM.empty]
		       val ys = L.map (fn sigma => applySubstWI n sigma r) devSubsts
		   in
		       ys @ (rootDevelopSub rs (n+1) term)
		   end
		 | NONE => rootDevelopSub rs (n+1) term
(*	   val _ = print "rootDevelopOneStepReductsWithIndex: " *)
(*	   val _ = print (Trs.prRules R) *)
(*	   val _ = print ((Term.toString term) ^ "\n")  *)
       in
	   rootDevelopSub R 0 term
       end

   and developSubstWithIndex [] sigmas = sigmas
     | developSubstWithIndex ((x,ts)::xs) sigmas =
       developSubstWithIndex xs
			     (ListXProd.mapX (fn (t,sigma) => VM.insert (sigma,x,t))
					     (ts,sigmas))

   and developOneStepReductsWithIndex R term =
       let
	   fun deleteDuplication xs =
	       let fun dupsub [] ans = ans
		     | dupsub ((a,t)::xs) ans = 
		       if L.exists (fn (b,s) => IS.equal (a,b) andalso Term.equal (s,t)) ans
		       then dupsub xs ans 
		       else dupsub xs ((a,t)::ans)
	       in dupsub xs []
	       end
       in
	   deleteDuplication 
	       (developOneStepReductsWithIndex2 R term)
       end

   and developOneStepReductsWithIndex2 R term =
       case term 
	of Var _ => [(IS.empty, term)]
	 | Fun (f,ts,sort) => 
	   let val ih = L.map 
			    (fn (iset,ss) => (iset, Fun (f,ss,sort)))
			    (allCombinations 
				 (fn ((a,x),(b,xs)) => (IS.union (a,b), x::xs))
				 (L.map (developOneStepReductsWithIndex2 R) ts))
(*	       val _ = L.app (fn (_,u) => print ("[" ^ (Term.toString u) ^ "]\n")) ih *)
(*	       val _ = print "...\n" *)
	   in
	       L.@(ih, rootDevelopReductsWithIndex R term)
	   end
   end



   (* statusOneStepRewrite R term : 񤭴§ R ΤȤǤ1ƥå׽񤭴 *)
   (* 񤭴Ƥ (true, reduct)Ƥʤ (false, term) ֤ *)
   fun statusOneStepRewrite R term = 
       case term 
	of Var _ => (false,term)
	 | Fun (f,ts,sort) => 
	   case rootRewrite R term of
	       SOME reduct => (true,reduct)
	     | NONE => let val (status,ss) = statusOneStepRewriteList R ts
		       in (status, Fun (f,ss,sort))
		       end
   and statusOneStepRewriteList R [] = (false, [])
     | statusOneStepRewriteList R (t::ts) =
       let val (status,t') = statusOneStepRewrite R t
       in if status 
	  then (true, t'::ts)
	  else let val (status,ts') = statusOneStepRewriteList R ts
	       in (status,t::ts')
	       end
       end

   (* 񤭴Ƥ SOME nfƤʤ NONE ֤ *)
   fun oneStepRewritePartial R term =
       let val (status, reduct) = statusOneStepRewrite R term
       in if status then SOME reduct else NONE
       end


   (* 1 ƥåפν񤭴ꥹ *)
   fun oneStepReducts R term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn u => 
					    Fun (f,
						 L.@(L.take (ts,i), u::(L.drop (ts,i+1))),
						 sort))
					(oneStepReducts R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(rootRewriteAll R term, ss)
	   end

   fun oneStepReductSet R term =
       TS.addList(TS.empty, oneStepReducts R term)


   (* 1 ƥåפν񤭴轸: pos 겼ΰ֤Ǥν񤭴Ǥʤ *)
   fun oneStepReductsWithRestrictedPosition R [] term = rootRewriteAll R term
     | oneStepReductsWithRestrictedPosition R (p::ps) term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn u => 
					    Fun (f,
						 L.@(L.take (ts,i), u::(L.drop (ts,i+1))),
						 sort))
					(if i+1 = p
					 then oneStepReductsWithRestrictedPosition R ps (L.nth (ts,i))
					 else oneStepReducts R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(rootRewriteAll R term, ss)
	   end


   (* minimal form  *)
   fun isMinimalForm R term = 
       let val rset = oneStepReductSet R term
	   val rsetnum = TS.numItems rset
       in 
	   rsetnum = 0
	   orelse (rsetnum = 1) andalso (TS.member (rset,term))
       end

   (* head stable form  *)
   fun isHeadStable R term = 
       (L.all (fn (l,r) => not (Term.haveSameRoots (l,term))
			  orelse (Term.haveSameRoots (r,term))) R)
       orelse isMinimalForm R term


   (* 1 ƥåפν񤭴轸 *)
   fun oneStepReductsWithSubst R term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn (u,phi) => 
					    (Fun (f,L.@(L.take (ts,i), u::(L.drop (ts,i+1))),sort),phi))
					(oneStepReductsWithSubst R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(rootRewriteWithSubstAll R term, ss)
	   end

   (* 1 ƥåפν񤭴轸 *)
   fun oneStepReductsWithIndex R term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn (k,u) => 
					    (k,Fun (f,
						 L.@(L.take (ts,i), u::(L.drop (ts,i+1))),
						 sort)))
					(oneStepReductsWithIndex R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(rootRewriteAllWithIndex R term, ss)
	   end

   fun oneStepReductsWithRule R term =
       L.map (fn (i,t) => (L.nth (R,i), t)) (oneStepReductsWithIndex R term)

   fun oneStepReductWithIndexSet R term =
       ITS.addList(ITS.empty, oneStepReductsWithIndex R term)

   (* 1 ƥåפν񤭴轸 *)
   fun oneStepReductsWithIndexAndFs R term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn (fs,k,u) => 
					    (f::fs,k,
					     Fun (f, L.@(L.take (ts,i),u::(L.drop (ts,i+1))),sort)))
					(oneStepReductsWithIndexAndFs R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(L.map (fn (i,s) => ([],i,s)) (rootRewriteAllWithIndex R term), ss)
	   end


   (* 1 ƥåפν񤭴轸 *)
   fun oneStepReductsWithIndexAndPosFs R term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn (fps,k,u) => 
					    ((f,i+1)::fps,k,
					     Fun (f, L.@(L.take (ts,i),u::(L.drop (ts,i+1))),sort)))
					(oneStepReductsWithIndexAndPosFs R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(L.map (fn (i,s) => ([],i,s)) (rootRewriteAllWithIndex R term), ss)
	   end

   (* 1 ƥåפν񤭴轸硤դ*)
   fun oneStepReductsWithPos R term =
       L.map (fn (fps,_,t)=> (L.map (fn (_,i)=>i) fps,t))
	     (oneStepReductsWithIndexAndPosFs R term)

   (* 1 ƥåפν񤭴轸硤դ*)
   fun oneStepReductsWithIndexAndPos R term =
       L.map (fn (fps,k,t)=> (L.map (fn (_,i)=>i) fps,k,t))
	     (oneStepReductsWithIndexAndPosFs R term)

   (* minimalReductSet term  reducts  iƥå(0<=i<=k) ĤƤСιླྀ *)
   (* maxLen ΰ̣ȤȰ㤦  ĿiıۤNONE *)
   fun minimalReductSet R maxLen term = 
       let fun oneStepReductSet [] ans = ans
	     | oneStepReductSet (t::ts) ans = 
	       oneStepReductSet ts (TS.addList (ans, oneStepReducts R t))
	   fun reductSetSub done yet = 
	       let val done' = TS.union (done,yet)
	       in if  (TS.numItems done') <= maxLen
		  then (let val yet' = TS.difference (oneStepReductSet (TS.listItems yet) TS.empty,
						      done')
			in if TS.isEmpty yet'
			   then SOME done'
			   else reductSetSub done' yet'
			end)
		  else NONE
	       end
       in reductSetSub TS.empty (TS.singleton term)
       end

   (* manyStepsReductSet term  iƥå(0<=i<=k) ླྀ *)
   fun manyStepsReductSet R maxLen term = 
       let fun oneStepReductSet [] ans = ans
	     | oneStepReductSet (t::ts) ans = 
	       oneStepReductSet ts (TS.addList (ans, oneStepReducts R t))
	   fun reductSetSub k done new = 
	       if k <= 0 orelse (TS.isEmpty new)
	       then TS.union (done,new)
	       else let val done' = TS.union (done,new)
			val new' = TS.difference (oneStepReductSet (TS.listItems new) TS.empty,
						  done')
		    in reductSetSub (k-1) done' new'
		    end			   
       in reductSetSub maxLen TS.empty (TS.singleton term)
       end


   fun manyStepsReductsWithSubst R maxLen term = 
       let 
	   fun reductsSub k done new = 
	       if k <= 0 
	       then new @ done
	       else let val done' = new @ done
			val new' = LU.mapAppend (fn (t,sublist) => 
						    L.map (fn (u,phi) => (u,phi::sublist))
							  (oneStepReductsWithSubst R t))
						new
		    in reductsSub (k-1) done' new'
		    end			   
       in reductsSub maxLen [] [(term,[])]
       end


   fun manyStepsReductsWithPosAndIndex R maxLen term = 
       let 
	   fun union (xs,ys) = LU.union' (fn ((_,_,t1),(_,_,t2))=> Term.equal (t1,t2))
					 (xs,ys)
	   fun difference (xs,ys) = LU.differenceByAll' 
					(fn ((_,_,t1),(_,_,t2))=> Term.equal (t1,t2))
					(xs,ys)
	   fun expand [] ans = ans
	     | expand ((ps,xs,t)::ts) ans =  
	       expand ts (union (L.map (fn (q,y,s)=> (ps@[q],xs@[y],s))
				       (oneStepReductsWithIndexAndPos R t), 
				 ans))
	   fun reductsSub k done new = 
	       if k <= 0 orelse (null new)
	       then union (new,done)
	       else let val done' = union (new,done)
			val new' = difference (expand new [], done')
		    in reductsSub (k-1) done' new'
		    end			   
       in reductsSub maxLen [] [([],[],term)]
       end

(*    fun manyStepsReductsWithPos R maxLen term = *)
(*        if maxLen <= 0  *)
(*        then [] *)
(*        else if maxLen = 1  *)
(*        then L.map (fn (pos,t) => ([pos],t)) (oneStepReductsWithPos R term) *)
(*        else let val pre = manyStepsReductsWithPos R (maxLen - 1) term *)
(* 	    in *)
(* 		pre @ (LU.mapAppend *)
(* 			   (fn (ps,t) => L.map (fn (p,u)=> (Pos.insertMinimal p ps,u)) *)
(* 					       (oneStepReductsWithPos R t)) *)
(* 			   pre) *)
(* 	    end *)

   (* reductSet R term :  R ¿(>=0)ƥå׽񤭴ˤν *)
   fun reductSet R term = 
       let fun reductSetSub R [] set = set
	     | reductSetSub R (t::ts) set = 
	       if TS.member (set, t)
	       then reductSetSub R ts set
	       else
		   let val ss = oneStepReducts R t
		   in
			reductSetSub R (ss @ ts)  (TS.add (set, t))
		   end
       in
	   reductSetSub R [term] TS.empty 
       end

   (* normalFormSet R term :  R ν񤭴ˤ term ν *)
   (* ƥǡºݤ˻ȤˤϡäȤޤˡɬפꡥ *)
   fun normalFormSet R term = 
       let fun nfSetSub termset nfset =
	       case TS.find (fn _ => true) termset of
		   NONE => nfset
		 | SOME t => let val termset2 = TS.delete (termset,t)
			     in if isNormalForm R t
				then nfSetSub termset2 (TS.add (nfset,t))
				else nfSetSub (TS.addList (termset2, oneStepReducts R t)) nfset
			     end
           (* fun nfSetSub [] set = set
	     | nfSetSub (t::ts) set = 
	       if isNormalForm R t
	       then nfSetSub ts (TS.add (set, t)) 
	       else let val ss = oneStepReducts R t
		    in nfSetSub R (LU.union' Term.equal (ss,ts)) set
		    end *)
       in
	   nfSetSub (TS.singleton term) TS.empty 
       end


   (* term1 <-*->E term2 *)
    fun isEquivalent E (term1,term2) =
	let fun check [] n1 set1 [] n2 set2 = false
	      | check (t::ts1) n1 set1 ts2 n2 set2 = 
		if TS.member (set1,t)
		then check ts1 n1 set1 ts2 n2 set2
		else if TS.member (set2,t) 
		then true
		else let val ss = oneStepReducts E t
		     in check0 (ts1@ss) (n1+1) (TS.add (set1,t)) ts2 n2 set2
		     end
	      | check [] n1 set1 (t::ts2) n2 set2 = 
		if TS.member (set2,t)
		then check [] n1 set1 ts2 n2 set2
		else if TS.member (set1,t) 
		then true
		else let val ss = oneStepReducts E t
		     in check [] n1 set1 ts2 (n2+1) (TS.add (set2,t))
		     end
	    and check0 ts1 n1 set1 ts2 n2 set2 =
		if n2 > n1
		then check ts2 n2 set2 ts1 n1 set1
		else check ts1 n1 set1 ts2 n2 set2
	in check [term1] 0 TS.empty [term2] 0 TS.empty
	end

   (* term1 <-*->E term2 *)
    fun isEquivalentWithLimit maxStep E (term1,term2) =
	let fun check [] n1 set1 [] n2 set2 = false
	      | check (t::ts1) n1 set1 ts2 n2 set2 = 
		if TS.member (set1,t)
		then check ts1 n1 set1 ts2 n2 set2
		else if TS.member (set2,t) 
		then true
		else let val ss = oneStepReducts E t
		     in check0 (ts1@ss) (n1+1) (TS.add (set1,t)) ts2 n2 set2
		     end
	      | check [] n1 set1 (t::ts2) n2 set2 = 
		if TS.member (set2,t)
		then check [] n1 set1 ts2 n2 set2
		else if TS.member (set1,t) 
		then true
		else let val ss = oneStepReducts E t
		     in check [] n1 set1 ts2 (n2+1) (TS.add (set2,t))
		     end
	    and check0 ts1 n1 set1 ts2 n2 set2 =
		if n2 > n1
		then if n2 > maxStep
		     then false
		     else check ts2 n2 set2 ts1 n1 set1
		else if n1 > maxStep
		then false
		else check ts1 n1 set1 ts2 n2 set2
	in check [term1] 0 TS.empty [term2] 0 TS.empty
	end

(*
val ACadd = IOFotrs.rdRules 
 [ "add(?x,?y) -> add(?y,?x)",
   "add(add(?x,?y),?z) -> add(?x,add(?y,?z))",
   "add(?x,add(?y,?z)) -> add(add(?x,?y),?z)" ];

fun check (name1,name2) () = 
    let val t1 = IOFotrs.rdTerm name1
	val t2 = IOFotrs.rdTerm name2
	val result = isEquivalent ACadd (t1,t2)
	val _ = print (name1 ^ " <-*->E " ^ name2 ^ " ?\n")
    in  if result 
	then print "Equivalent!\n"
	else print "Not Equivalent!\n"
    end

fun test name1 name2 = TimeUtil.profile (check (name1,name2),"")
val _ = test "add(?x,?y)" "add(?x,?y)"
val _ = test "add(?x,?y)" "add(?y,?x)"
val _ = test "add(add(?x,?y),?z)" "add(?z,add(?y,?x))"
val _ = test "add(add(?x,?y),?z)" "add(add(?x,?y),add(?y,?z))"
*)

   (* containedInReductSet R source reduct: source -*->R  reduct  *)
   fun containedInReductSet R source reduct = 
       let fun checkReduct [] set = false
	     | checkReduct  (t::ts) set = 
	       if Term.equal (t,reduct) 
	       then true
	       else if TS.member (set, t)
	       then checkReduct ts set
	       else let val ss = oneStepReducts R t
		    in checkReduct (ss @ ts)  (TS.add (set, t))
		    end
       in checkReduct [source] TS.empty 
       end

  (* oneStepModuloRewriteWithNFset: 
       (SOME nf, TS.empty): term  R/E 񤭴ȤE-eq  term  R-nf nf
       (NONE, set): term R/E񤭴ǤʤȤ term  E-equivalent class  set *)
   fun oneStepModuloRewriteWithNFset R E term = 
       let fun check [] set = (NONE,set)
	     | check (t::ts) set =
	       if TS.member (set, t)
	       then check ts set
	       else case linfPartial R t of
			SOME nf => (SOME nf,TS.empty)
		      | NONE => check (ts @ (oneStepReducts E t)) (TS.add (set, t))
       in check [term] TS.empty
       end

  (* moduloLinf R E term:  
        term  R/E 񤭴 E-equivalent class  set֤ *)
   fun moduloLinf R E term = 
       case oneStepModuloRewriteWithNFset R E term of
	   (SOME nf, _) => moduloLinf R E nf
	 | (NONE,set) => set

  (* term  R/E 񤭴 termset ãǤ뤫 *)
   fun isModuloReachable R E term termset = 
       if TS.member (termset, term)
       then true
       else case oneStepModuloRewriteWithNFset R E term of
		(SOME nf, _) => isModuloReachable R E nf termset
	      | (NONE,set) => false

  (* term1  term2  R/E 񤭴 joinable  *)
   fun isModuloJoinable R E (term1,term2) =
       let fun check [] [] n1 n2 tset1 tset2 = false
	     | check [] (t2::ts2) n1 n2 tset1 tset2 = 
               (* do not use *) check (t2::ts2) [] n2 n1 tset2 tset1
	     | check (t1::ts1) ts2 n1 n2 tset1 tset2 =
	       ((* print ("check t1= " ^ (Term.toString t1) ^ "\n");
	        print ("    set1= " ^ (ListUtil.toStringCommaCurly Term.toString (TS.listItems tset1)) ^ "\n");
	        print ("    set2= " ^ (ListUtil.toStringCommaCurly Term.toString (TS.listItems tset2)) ^ "\n"); *)
	       if TS.member (tset1,t1)
	       then check ts1 ts2 n1 n2 tset1 tset2
	       else if TS.member (tset2,t1)
	       then ((* print ("< "^ (Term.toString term1)
			    ^ ", " ^ (Term.toString term2)
			    ^ "> ( " ^ (Int.toString (TS.numItems tset1))
			    ^ ", " ^ (Int.toString (TS.numItems tset2))
			    ^ ")\n");*)
		     true)
	       else case linfPartial R t1 of
			SOME nf1 => ((* print ("reduce to " ^ (Term.toString nf1) ^ "\n");*)
				     isModuloJoinableSub [nf1] ts2 0 n2 TS.empty tset2)
		      | NONE => isModuloJoinableSub 
				    (ts1 @ (oneStepReducts E t1)) ts2
				    (n1+1) n2 (TS.add (tset1,t1)) tset2)
	   and isModuloJoinableSub ts1 ts2 n1 n2 tset1 tset2 =
	       if n1 <= n2 andalso (not (null ts1))
	       then check ts1 ts2 n1 n2 tset1 tset2
	       else check ts2 ts1 n2 n1 tset2 tset1
       in isModuloJoinableSub [term1] [term2] 0 0 TS.empty TS.empty
       end


  (* term1  term2  R/E 񤭴 joinable  *)
   fun isModuloJoinableWithLimit maxStep R E (term1,term2) =
       let fun check [] [] n1 n2 tset1 tset2 = false
	     | check [] (t2::ts2) n1 n2 tset1 tset2 = 
               (* do not use *) check (t2::ts2) [] n2 n1 tset2 tset1
	     | check (t1::ts1) ts2 n1 n2 tset1 tset2 =
	       ((* print ("check t1= " ^ (Term.toString t1) ^ "\n");
	        print ("    set1= " ^ (ListUtil.toStringCommaCurly Term.toString (TS.listItems tset1)) ^ "\n");
	        print ("    set2= " ^ (ListUtil.toStringCommaCurly Term.toString (TS.listItems tset2)) ^ "\n"); *)
		if n1 >= maxStep
		then false
		else
	       if TS.member (tset1,t1)
	       then check ts1 ts2 n1 n2 tset1 tset2
	       else if TS.member (tset2,t1)
	       then ((* print ("< "^ (Term.toString term1)
			    ^ ", " ^ (Term.toString term2)
			    ^ "> ( " ^ (Int.toString (TS.numItems tset1))
			    ^ ", " ^ (Int.toString (TS.numItems tset2))
			    ^ ")\n");*)
		     true)
	       else case linfPartial R t1 of
			SOME nf1 => ((* print ("reduce to " ^ (Term.toString nf1) ^ "\n");*)
				     isModuloJoinableSub [nf1] ts2 0 n2 TS.empty tset2)
		      | NONE => isModuloJoinableSub 
				    (ts1 @ (oneStepReducts E t1)) ts2
				    (n1+1) n2 (TS.add (tset1,t1)) tset2)
	   and isModuloJoinableSub ts1 ts2 n1 n2 tset1 tset2 =
	       if n1 <= n2 andalso (not (null ts1))
	       then check ts1 ts2 n1 n2 tset1 tset2
	       else check ts2 ts1 n2 n1 tset2 tset1
       in isModuloJoinableSub [term1] [term2] 0 0 TS.empty TS.empty
       end


  (* moduloLinfWithNFset: 
       (SOME nf, TS.empty): term  R/E 񤭴Ȥterm  nf 
       (NONE, set): term 񤭴ǤʤȤ E-equivalent class  set֤ *)
   fun moduloLinfWithNFset R E term = 
       case oneStepModuloRewriteWithNFset R E term of
	   (SOME nf, _) => let val tset = moduloLinf R E nf
			       val fin = TS.find (fn x=> true) tset
			   in (fin,TS.empty)
			   end
	 | (NONE,set) => (NONE,set)

 (*
   fun moduloLinfWithNFset R E term = 
       let fun reduceOrNot nfset = (* nfset : a set of R-normal terms *)
	       let fun checkOne s = 
		       let val ss = oneStepReducts E s
		       in case List.find (not o (isNormalForm R)) ss
			   of SOME t => (SOME t,[]) 
			    | NONE => (NONE,ss)
		       end
		   exception FOUND of Term.term
	       in (NONE, TS.foldl (fn (u,set) 
				      => case checkOne u of 
					     (SOME t,_) => raise FOUND t
					   | (NONE,uu) => TS.addList (set,uu))
				  TS.empty nfset)
		  handle FOUND t => (SOME t,TS.empty)
	       end

	   fun incrementReduceOrNot oldset newset  = 
	       case reduceOrNot newset of
		   (SOME t,_) => reduceOneStep t
		 | (NONE,nfset) => 
		   let val oldset2 = TS.union (oldset,newset)
		       val newset2  = TS.difference (nfset,oldset2)
		   in if TS.isEmpty newset2
		      then (NONE,oldset2)
		      else incrementReduceOrNot oldset2 newset2
		   end
	   and reduceOneStep t = 
	       case linfPartial R t
		of SOME nf => (SOME nf,TS.empty)
		 | NONE => incrementReduceOrNot TS.empty (TS.singleton t)
       in
	   reduceOneStep term
       end
*)

(*
val S = IOFotrs.rdRules 
   [ "not(T) -> F",
     "not(F) -> T",
     "not(not(?x)) -> ?x",
     "exor(?x,T) -> not(?x)",
     "exor(?x,F) -> ?x",
     "not(exor(?x,?y)) -> exor(not(?x),?y)",
     "not(exor(?x,?y)) -> exor(?x,not(?y))" ]

val P = IOFotrs.rdRules 
   [ "exor(exor(?x,?y),?z) -> exor(?x,exor(?y,?z))",
     "exor(?x,exor(?y,?z)) -> exor(exor(?x,?y),?z)",
     "exor(?y,?x) -> exor(?x,?y)" ]

val t1 = IOFotrs.rdTerm "not(exor(?x,exor(?y,?z)))"
val t2 = IOFotrs.rdTerm "exor(exor(?x,?y),not(?z))"

fun check name () = 
    let val t = IOFotrs.rdTerm name
	val set = moduloLinf S P t
	val _ = print "\n"
    in  print (ListUtil.toStringCommaCurly Term.toString (TS.listItems set))
	(* print (Int.toString (TS.numItems set)) *)
    end

fun test name = TimeUtil.profile (check name,name)
val _ = test "not(exor(?x,exor(?y,?z)))"
val _ = test "exor(exor(?x,?y),not(?z))"

val _ = if isModuloJoinable S P (t1,t2)
	then print "TRUE\n"
	else print "FALSE\n"
*)

(***
val Radd = IOFotrs.rdRules 
 [ "add(0,?y) -> ?y", "add(s(?x),?y) -> s(add(?x,?y))",
   "add(?x,0) -> ?x", "add(?x,s(?y)) -> s(add(?x,?y))" ];
val Rtimes = IOFotrs.rdRules 
 [ "times(0,?y) -> 0", "times(s(?x),?y) -> add(times(?x,?y),?y)",
   "times(?x,0) -> 0", "times(?x,s(?y)) -> add(times(?x,?y),?x)" ];
val Rdist = IOFotrs.rdRules 
 [ "times(add(?x,?y),?z) -> add(times(?x,?z),times(?y,?z))",
   "times(?x,add(?y,?z)) -> add(times(?x,?y),times(?x,?z))" ];
val ACadd = IOFotrs.rdRules 
 [ "add(?x,?y) -> add(?y,?x)",
   "add(add(?x,?y),?z) -> add(?x,add(?y,?z))",
   "add(?x,add(?y,?z)) -> add(add(?x,?y),?z)" ];
val ACtimes  = IOFotrs.rdRules 
  ["times(?x,?y) -> times(?y,?x)",
   "times(times(?x,?y),?z) -> times(?x,times(?y,?z))",
   "times(?x,times(?y,?z)) -> times(times(?x,?y),?z)"];
val S = Radd@Rtimes@Rdist;
val P = ACadd@ACtimes;

fun check name () = 
    let val t = IOFotrs.rdTerm name
	val set = moduloLinf S P t
	val _ = print "\n"
    in  (* print (ListUtil.toStringCommaCurly Term.toString (TS.listItems set)) *)
	print (Int.toString (TS.numItems set))
    end

fun test name = TimeUtil.profile (check name,name)
val _ = test "add(?x,?y)"
val _ = test "add(?x,s(?y))"
val _ = test "add(s(?x),s(?y))"
val _ = test "times(s(s(?x)),s(s(?y)))"
val _ = test "add(times(add(?x,?y),?z),times(add(?x,?y),?w))"
val _ = test "add(times(?x,add(?z,?w)),times(?y,add(?z,?w)))"

val t1 = IOFotrs.rdTerm "add(times(add(?x,?y),?z),times(add(?x,?y),?w))"
val t2 = IOFotrs.rdTerm "add(times(?x,add(?z,?w)),times(?y,add(?z,?w)))"
val t3 = IOFotrs.rdTerm "add(times(?x,add(?z,?w)),times(?y,add(?z,?y)))"

val _ = if isModuloJoinable S P (t1,t2)
	then print "TRUE\n"
	else print "FALSE\n"

val _ = if isModuloJoinable S P (t2,t3)
	then print "TRUE\n"
	else print "FALSE\n"
***)

   (* moduloLinf R E term : R/E 񤭴ˤ term  *)
   (*                       E^{-1} \subseteq E            *)
   (***
   fun moduloLinf R E term =
       let fun findReduct set = 
	       case TS.find (fn _ => true) set
		of SOME s => (case linfPartial R s
			      of SOME nf => SOME nf
			       | NONE => findReduct (TS.delete (set,s)))
		 | NONE => NONE
	   fun moduloLinfSub R E nf =
	       case nf
		of Var _ => nf
		 | Fun _ => let val equivalents = reductSet E nf
			    in case findReduct equivalents
				of SOME nf2 => moduloLinfSub R E nf2
				 | NONE => nf
			    end
       in
	   moduloLinfSub R E (linf R term)
       end

   fun moduloLinf R E term = 
       case moduloLinfWithNFset R E term 
	of (SOME nf,_) => nf
	 | (NONE,_) => term
    ***)

   (* moduloLinf R E term : R/E 񤭴˴ؤ          *)
   (*                       E^{-1} \subseteq E            *)
   (*
   fun isModuloNormalForm R E term  =
       let val set = reductSet E term
       in not (TS.exists (fn t => not (isNormalForm R t)) set)
       end*)
   fun isModuloNormalForm R E term  =
       case oneStepModuloRewriteWithNFset R E term 
	of (SOME nf,_) => false
	 | (NONE,_) => true


   (* 񤭴Ƥ (true, nf)Ƥʤ (false, nf) ֤ *)
   (*
   fun statusModuloLinf R E term =
       let fun findReduct set = 
	       case TS.find (fn _ => true) set
		of SOME s => (case linfPartial R s
			      of SOME nf => SOME nf
			       | NONE => findReduct (TS.delete (set,s)))
		 | NONE => NONE
	   fun statusModuloLinfSub R E term =
	       case term
		of Var _ => (false, term)
		 | Fun _ => let val equivalents = reductSet E term
			    in case findReduct equivalents
				of SOME nf => (true, moduloLinf R E nf)
				 | NONE => (false, term)
			    end
       in
	   case linfPartial R term of
	       SOME nf => (true, moduloLinf R E nf)
	     | NONE => statusModuloLinfSub R E term
       end
   *)
   (* ǡ *)
   fun statusModuloLinf R E term =
       case moduloLinfWithNFset R E term 
	of (SOME nf,_) => (true,nf)
	 | (NONE,_) => (false,term)

   fun moduloLinfPartial R E term =
       let val (status, nf) = statusModuloLinf R E term
       in if status then SOME nf else NONE
       end

   (* ֤ constrained rewrite step ǽ *)
   fun rootConstrainedRewrite isSatisfiable [] term = NONE
     | rootConstrainedRewrite isSatisfiable ((l,r)::rs) term = 
       let val sigma1 = match l term
       in if isSome sigma1
	  then let val reduct = applySubst (valOf sigma1) r
	       in if isSatisfiable (term,reduct)
		  then SOME reduct
		  else rootConstrainedRewrite isSatisfiable rs term
	       end
	  else rootConstrainedRewrite isSatisfiable rs term
       end

   (* ֤ constrained rewrite step ǽ *)
   (* R ˤĤƤϽ­Ȳ *)
   fun rootConstrainedRewrite2 isSatisfiable R K term = 
       let val reduct1 = rootRewrite R term
       in if isSome reduct1
	  then reduct1
 	  else rootConstrainedRewrite isSatisfiable K term
       end

   (* ֤ constrained rewrite step 񤭴ν *)
   fun rootConstrainedRewriteAll isSatisfiable K term = 
       L.mapPartial (fn rule => rootConstrainedRewrite isSatisfiable [rule] term) K

   (* ֤ constrained rewrite step 񤭴ν *)
   (* R ˤĤƤϽ­Ȳ *)
   fun rootConstrainedRewriteAll2 isSatisfiable R K term = 
       L.@ (L.mapPartial (fn rule => rootRewrite [rule] term)  R,
	    L.mapPartial (fn rule => rootConstrainedRewrite isSatisfiable [rule] term) K)

   (* statusOneStepConstrainedRewrite R term : constrained rewrite step *)
   (* 񤭴Ƥ (true, reduct)Ƥʤ (false, term) ֤ *)
   fun statusOneStepConstrainedRewrite isSatisfiable R K term =
       case term
	of Var _ => (false,term)
	 | Fun (f,ts,sort) =>
	   case rootConstrainedRewrite2 isSatisfiable R K term of
	       SOME reduct => (true,reduct)
	     | NONE => let val (status,ss) = statusOneStepConstrainedRewriteList 
						 isSatisfiable R K ts
		       in (status, Fun (f,ss,sort))
		       end
   and statusOneStepConstrainedRewriteList isSatisfiable R K [] = (false, [])
     | statusOneStepConstrainedRewriteList isSatisfiable R K (t::ts) =
       let val (status,t') = statusOneStepConstrainedRewrite isSatisfiable R K t
       in if status
	  then (true, t'::ts)
	  else let val (status,ts') = statusOneStepConstrainedRewriteList 
					  isSatisfiable R K ts
	       in (status,t::ts')
	       end
       end

   (* 񤭴Ƥ SOME nfƤʤ NONE ֤ *)
   fun oneStepConstrainedRewritePartial isSatisfiable R K term =
       let val (status, reduct) = statusOneStepConstrainedRewrite isSatisfiable R K term
       in if status then SOME reduct else NONE
       end

   (* constrained rewrite step 1 ƥå׽񤭴ν *)
   (* R ˤĤƤϽ­Ȳ *)
   fun oneStepConstrainedReducts isSatisfiable R K term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn u => 
					    Fun (f,
						 L.@(L.take (ts,i), u::(L.drop (ts,i+1))),
						 sort))
					(oneStepConstrainedReducts isSatisfiable R K (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(rootConstrainedRewriteAll2 isSatisfiable R K term, ss)
	   end

   (* constrained rewrite step ¿(>=0)ƥå׽񤭴ν *)
   (* R ˤĤƤϽ­Ȳ *)
   fun constrainedReductSet isSatisfiable R K term = 
       let fun reductSetSub [] set = set
	     | reductSetSub (t::ts) set = 
	       if TS.member (set, t)
	       then reductSetSub ts set
	       else
		   let val ss = oneStepConstrainedReducts isSatisfiable R K t
		   in
			reductSetSub (ss @ ts)  (TS.add (set, t))
		   end
       in
	   reductSetSub [term] TS.empty 
       end

   (* clinf isSatisfiable R K term *)
   (* constrained rewrite step of R \cup K ΤȤǤ term  *)
   (* R ˤĤƤϽ­Ȳ *)
   fun clinf isSatisfiable R K term =
       case term 
	of Var _ => term
	 | Fun (f,ts,sort) => let val ss = map (clinf isSatisfiable R K) ts
			      in case rootConstrainedRewrite2 isSatisfiable R K (Fun (f,ss,sort))
				  of SOME t => clinf isSatisfiable R K t
				   | NONE => Fun (f,ss,sort)
			      end

   (* 񤭴Ƥ (true, nf)Ƥʤ (false, nf) ֤ *)
   fun statusClinf isSatisfiable R K term =
       case term 
	of Var _ => (false, term)
	 | Fun (f,ts,sort) => let 
				  val statusXss = L.map (statusClinf isSatisfiable R K) ts
				  val ss = L.map #2 statusXss
			      in case rootConstrainedRewrite2 isSatisfiable R K (Fun (f,ss,sort))
				  of SOME t => (true, clinf isSatisfiable R K t)
				   | NONE => (L.exists (fn x => #1 x) statusXss, Fun (f,ss,sort))
			      end

   (* 񤭴Ƥ SOME nfƤʤ NONE ֤ *)
   fun clinfPartial isSatisfiable R K term =
       let 
	   val (status, nf) = statusClinf isSatisfiable R K term
	   
       in if status then SOME nf else NONE
       end

   (* moduloClinf isOrdered isEquivalent R E K term : *)
   (*         R\cup K^> / E \cup K^\approx 񤭴ˤ term  *)
   (*                       E^{-1} \subseteq E, K^{-1} \subseteq K            *)
   fun moduloClinf isOrdered isEquivalent R E K term =
       let fun findReduct set = 
	       case TS.find (fn _ => true) set
		of SOME s => (case clinfPartial isOrdered R K s
			      of SOME nf => SOME nf
			       | NONE => findReduct (TS.delete (set,s)))
		 | NONE => NONE
	   fun moduloClinfSub nf =
	       case nf
		of Var _ => nf
		 | Fun _ => let val equivalents = constrainedReductSet isEquivalent E K nf
			    in case findReduct equivalents
				of SOME nf2 => moduloClinfSub nf2
				 | NONE => nf
			    end
       in
	   moduloClinfSub (clinf isOrdered R K term)
       end

   (* 񤭴Ƥ (true, nf)Ƥʤ (false, nf) ֤ *)
   fun statusModuloClinf isOrdered isEquivalent R E K term =
       let fun findReduct set = 
	       case TS.find (fn _ => true) set
		of SOME s => (case clinfPartial isOrdered R K s
			      of SOME nf => SOME nf
			       | NONE => findReduct (TS.delete (set,s)))
		 | NONE => NONE
	   fun statusModuloClinfSub term =
	       case term
		of Var _ => (false, term)
		 | Fun _ => let val equivalents = constrainedReductSet isEquivalent E K term
			    in case findReduct equivalents
				of SOME nf => (true, moduloClinf isOrdered isEquivalent R E K nf)
				 | NONE => (false, term)
			    end
       in
	   case clinfPartial isOrdered R K term of
	       SOME nf => (true, moduloClinf isOrdered isEquivalent R E K nf)
	     | NONE => statusModuloClinfSub term
       end


   fun moduloClinfPartial isOrdered isEquivalent R E K term =
       let val (status, nf) = statusModuloClinf isOrdered isEquivalent R E K term
       in if status then SOME nf else NONE
       end


   fun strengthenRewriteSeq R tss = 
       let fun strengthen R [] = []
	     | strengthen R (t::ts) = 
	       let val old = TS.addList (TS.empty, t::ts)
		   val new = TS.difference (oneStepReductSet R t, old)
	       in
		   if TS.isEmpty new
		   then [t::ts]
		   else L.map (fn s => s::t::ts)  (TS.listItems new)
	       end
       in
	   LU.mapAppend (strengthen R) tss
       end



   (* fun strengthenRewriteSeqWithIndex R itss =  *)
   (*     let fun member s ts = isSome (L.find (fn y => Term.equal (s,y)) ts) *)
   (* 	   fun strengthen R [] = [] *)
   (* 	     | strengthen R ((i,t)::its) =  *)
   (* 	       let val ts  = L.map (fn (j,s) => s) its *)
   (* 		   val old = TS.addList (TS.empty, t::ts) *)
   (* 		   val new = TS.difference (oneStepReductSet R t, old) *)
   (* 	       in *)
   (* 		   if TS.isEmpty new *)
   (* 		   then [((i,t) :: its)] *)
   (* 		   else L.mapPartial (fn (j,u) => if TS.member (old, u) *)
   (* 						  then NONE *)
   (* 						  else SOME ((j,u)::(i,t)::its)) *)
   (* 				     (ITS.listItems (oneStepReductWithIndexSet R t)) *)
   (* 	       end *)
   (*     in *)
   (* 	   LU.mapAppend (strengthen R) itss *)
   (*     end *)
   (* revised 2014/1/21 *)
   fun strengthenRewriteSeqWithIndex R itss = 
       let fun strengthen R [] = []
	     | strengthen R ((i,t)::its) = 
	       let val old = t:: (L.map (fn (j,s) => s) its)
		   val ans = L.mapPartial (fn (j,u) => if LU.member' Term.equal u old 
						       then NONE
						       else SOME ((j,u)::(i,t)::its))
					  (ITS.listItems (oneStepReductWithIndexSet R t))
	       in if null ans
		  then [((i,t) :: its)]  (* do we need this? *)
		  else ans
	       end
       in
	   LU.mapAppend (strengthen R) itss
       end

   fun nStepsRewriteSeqWithIndex R N it0 = 
       let fun strengthen [] = []
	     | strengthen ((i,t)::its) = 
	       let val old = t:: (L.map (fn (j,s) => s) its)
	       in L.mapPartial (fn (j,u) => if LU.member' Term.equal u old 
					    then NONE
					    else SOME ((j,u)::(i,t)::its))
			       (ITS.listItems (oneStepReductWithIndexSet R t))
	       end
	   fun main 0 itss = itss
	     | main n itss = itss @ (main (n-1) (LU.mapAppend strengthen itss))
			     
       in main N [[it0]]
       end


   fun strengthenRewriteSeqWithIndexAndFs R itss = 
       let fun member s ts = isSome (L.find (fn y => Term.equal (s,y)) ts)
	   fun strengthen R [] = []
	     | strengthen R ((fs,i,t)::its) = 
	       let val ts  = L.map (fn (_,_,s) => s) its
		   val old = TS.addList (TS.empty, t::ts)
		   val new = TS.difference (oneStepReductSet R t, old)
	       in
		   if TS.isEmpty new
		   then [((fs,i,t) :: its)]
		   else L.mapPartial (fn (gs,j,u) => if TS.member (old, u)
						     then NONE
						     else SOME ((gs,j,u)::(fs,i,t)::its))
				     (oneStepReductsWithIndexAndFs R t)
	       end
       in
	   LU.mapAppend (strengthen R) itss
       end

   fun strengthenRewriteSeqWithIndexAndPosFs R itss = 
       let fun member s ts = isSome (L.find (fn y => Term.equal (s,y)) ts)
	   fun strengthen [] = []
	     | strengthen ((fps,i,t)::its) = 
	       let val ts  = L.map (fn (_,_,s) => s) its
		   val old = TS.addList (TS.empty, t::ts)
		   val new = TS.difference (oneStepReductSet R t, old)
	       in
		   L.mapPartial (fn (gqs,j,u) => if TS.member (old, u)
						 then NONE
						 else SOME ((gqs,j,u)::(fps,i,t)::its))
				(oneStepReductsWithIndexAndPosFs R t)
	       end
       in
	   LU.mapAppend strengthen itss
       end

   (* rootNarrow R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥåץʥ *)
   fun rootNarrow [] term = NONE
     | rootNarrow ((l,r)::rs) term = 
       let val idx = 1 + (Term.maxVarIndexInTerm term)
	   val (l',r') = (increaseVarIndexBy idx l, increaseVarIndexBy idx r)
       in
	   case unify l' term
	    of SOME sigma => SOME (applySubst sigma r', 
				   Subst.restrictDomain (Term.varSetInTerm term) sigma)
	 | NONE => rootNarrow rs term
       end

   (* rootNarrowAll R term : 񤭴§ R ΤȤǤκ֤Ǥ1ƥåץʥ󥰤Υꥹ *)
   fun rootNarrowAll R term = 
       L.mapPartial (fn rule => rootNarrow [rule] term)  R

   (* narrow pos (l,r) term : 񤭴§ (l,r) ΤȤǤΰposǤ1ƥåץʥ *)
   fun narrow [] (l,r)(Var _) = NONE
     | narrow [] (l,r) term = rootNarrow [(l,r)] term
     | narrow (i::ps) (l,r) (Var _) = NONE
     | narrow (i::ps) (l,r) (Fun (f,ts,ty)) = 
       if 0 < i andalso i <= L.length ts
       then case narrow ps (l,r) (L.nth (ts,i-1)) of
		NONE => NONE
	      | SOME (u,sigma) => 
		SOME (Fun (f,
			   LU.replaceNth (L.map (Subst.applySubst sigma) ts,i-1, u),
			   ty),
		      sigma)
       else NONE
	   
   (* initial term  narrowing 򤯤꤫*)
   fun repeatNarrow init (ps,rs) = 
       let fun repeatNarrow0 (t,sub) ([],[]) = SOME (t,sub)
	     | repeatNarrow0 (t,sub) (p::ps,[]) = SOME (t,sub)
	     | repeatNarrow0 (t,sub) ([],lr::rest) = SOME (t,sub)
	     | repeatNarrow0 (t,sub) (p::ps,(l,r)::rest) =
	       case narrow p (l,r) t of
		   NONE => NONE
		 | SOME (t2,sub2) => repeatNarrow0 (t2,compose sub2 sub) (ps,rest)
       in 
	   repeatNarrow0 (init,Subst.empty) (ps,rs)
       end



   (* 1 ƥåפΥʥꥹ *)
   fun oneStepNarrow R term =
       case term
	of Var _ => []
	 | Fun (f,ts,sort) =>
	   let val ss = LU.mapAppend 
			 (fn i => L.map (fn (u,sub) => 
					    (Fun (f,
						  L.@(L.map (Subst.applySubst sub) (L.take (ts,i)), 
						      u::(L.map (Subst.applySubst sub) (L.drop (ts,i+1)))),
						  sort), 
					     sub))
			  		(oneStepNarrow R (L.nth (ts,i))))
			 (L.tabulate (length ts, fn x=>x))
	   in
	       L.@(rootNarrowAll R term, ss)
	   end

   (* 1 ƥåפoutermost ʥꥹ *)
   fun oneStepOutermostNarrow R (Var _) = []
     | oneStepOutermostNarrow R (term as (Fun (f,ts,sort))) =
       let val ans = rootNarrowAll R term
       in if null ans
	  then LU.mapAppend 
		   (fn i => L.map (fn (u,sub) => 
				      (Fun (f,
					    L.@(L.map (Subst.applySubst sub) (L.take (ts,i)), 
						u::(L.map (Subst.applySubst sub) (L.drop (ts,i+1)))),
					    sort), 
				       sub))
			  	  (oneStepOutermostNarrow R (L.nth (ts,i))))
		   (L.tabulate (length ts, fn x=>x))
	  else ans
       end

   (* 1 ƥåפoutermost ʥꥹ, 񤭴դ*)
   fun oneStepOutermostNarrowWithPosition R (Var _) = []
     | oneStepOutermostNarrowWithPosition R (term as (Fun (f,ts,sort))) =
       let val ans = rootNarrowAll R term
       in if null ans
	  then LU.mapAppend 
		   (fn i => L.map (fn (pos,u,sub) => 
				      (i+1::pos,
				       Fun (f,
					    L.@(L.map (Subst.applySubst sub) (L.take (ts,i)), 
						u::(L.map (Subst.applySubst sub) (L.drop (ts,i+1)))),
					    sort), 
				       sub))
			  	  (oneStepOutermostNarrowWithPosition R (L.nth (ts,i))))
		   (L.tabulate (length ts, fn x=>x))
	  else L.map (fn (u,sub) => ([],u,sub)) ans
       end

   fun forwardOneStepUnfoldingsOfRule R (l,r) =
       let val pos = Term.positionsInTerm r
	   val idx = 1 + (Term.maxVarIndexInTerms (L.map (fn (x,y)=>x) R))
	   val (l0,r0) = (increaseVarIndexBy idx l, increaseVarIndexBy idx r)
       in L.mapPartial (fn x=>x)
		       (ListXProd.mapX
			    (fn (p,(l',r')) => case unify (valOf (Term.subterm p r0)) l' of
						   SOME sigma => let val l1 = valOf (Term.replaceSubterm r0 p r')
								 in SOME (Subst.applySubst sigma l0, 
									  Subst.applySubst sigma l1)
								 end
						 | NONE => NONE)
			    (pos,R))
       end

   fun forwardOneStepUnfoldingsOfRules R X =
       LU.mapAppend (forwardOneStepUnfoldingsOfRule R) X

   fun backwardOneStepUnfoldingsOfRule R (l,r) =
       let val pos = Term.positionsInTerm l
	   val idx = 1 + (Term.maxVarIndexInTerms (L.map (fn (x,y)=>x) R))
	   val (l0,r0) = (increaseVarIndexBy idx l, increaseVarIndexBy idx r)
       in L.mapPartial (fn x=>x)
		       (ListXProd.mapX
			    (fn (p,(l',r')) => case unify (valOf (Term.subterm p l0)) r' of
						   SOME sigma => let val l1 = valOf (Term.replaceSubterm l0 p l')
								 in SOME (Subst.applySubst sigma l1, 
									  Subst.applySubst sigma r0)
								 end
						 | NONE => NONE)
			    (pos,R))
       end

   fun backwardOneStepUnfoldingsOfRules R X =
       LU.mapAppend (backwardOneStepUnfoldingsOfRule R) X

   fun oneStepUnfoldingsOfRules R =
       (LU.mapAppend (forwardOneStepUnfoldingsOfRule R) R)
       @ (LU.mapAppend (backwardOneStepUnfoldingsOfRule R) R)

   fun multiStepsUnfoldingsOfRules R n =
       let fun sub n X = if n <= 0 then X
			 else sub (n-1) 
				  ((LU.mapAppend (forwardOneStepUnfoldingsOfRule R) X)
				   @ (LU.mapAppend (backwardOneStepUnfoldingsOfRule R) X))
       in sub n R 
       end

  (* find rewrite sequence from start-term to target-term witin max steps *)
   fun findRewriteSequence R max start target =
       let fun expand done (hist,term) = 
	       L.mapPartial (fn (pos,k,reduct) => 
				if LU.member' Term.equal reduct done
				then NONE else SOME ((pos, L.nth (R,k),reduct)::hist,reduct))
			    (oneStepReductsWithIndexAndPos R term)
	   fun expandList done cands = LU.mapAppend (fn t => expand done t) cands
	   fun findRewSeq num done cands  = 
	       case L.find (fn (h,t)=> Term.equal (t,target)) cands of
		   SOME (h,t) => SOME (rev h)
		 | NONE => if num <= 0 
			   then NONE
			   else let val done2 = done@ (L.map (fn (_,t)=> t) cands)
				    val cands2 = expandList done2 cands
				in findRewSeq (num-1) done2 cands2
				end
       in findRewSeq max [] [([],start)]
       end


   val maxRewriteStepsForRev = ref 10 (* ForCheckingReversibility *)
   fun isReversibleRules P = 
       L.all (fn (l,r) => TS.member 
			      (manyStepsReductSet P (!maxRewriteStepsForRev) r,
			       l))
	     P

   fun prRewriteStepInProofTree (pos,rule,term) () =
       CU.encloseProofTreesBy "rewriteStep" 
			      [(Pos.toProofTree pos),
			       (Trs.prRuleInProofTree rule),
			       (Term.toProofTree term)]

   fun prRewriteSeqInProofTree (term,rseq) () =
       CU.encloseProofTreesBy "rewriteSequence" 
       ((fn _ => CU.encloseProofTreeBy "startTerm" (Term.toProofTree term))
	:: L.map prRewriteStepInProofTree rseq)
       
   fun prRewriteStepInProofTree2 (rule,term) () =
       CU.encloseProofTreesBy "rewriteStep" 
			      [(fn _=> CU.encloseProofLeafBy "positionInTerm" ""),
			       (Trs.prRuleInProofTree rule),
			       (Term.toProofTree term)]

   fun prRewriteSeqInProofTree2 (term,rseq) () =
       CU.encloseProofTreesBy "rewriteSequence" 
       ((fn _ => CU.encloseProofTreeBy "startTerm" (Term.toProofTree term))
	:: L.map prRewriteStepInProofTree2 rseq)

   end (* of local *)

   end; (* of structure Rewrite *)

