(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/sort.sml
 * description: definition and utility functions for Sort Symbols
 * author: AOTO Takahito
 * 
 ******************************************************************************)

structure Sort : sig
   include ORD_KEY2 (* { val ord_key, fun compare, fun equal, fun toString }*)
   datatype sort = Base of Atom.atom | Proc of (sort list) * sort
                                       (* sort list should be non-empty *)
   val toString2: ord_key -> string
   val null: ord_key
   val fromString: string -> sort
   val isASubsort: sort -> sort -> bool
   val supersorts: sort -> sort list
   val isBaseType: sort -> bool
   val args: sort -> sort list
   val return: sort -> sort
   val basicSortSetInSort: sort -> AtomSet.set
   val plus: sort -> int -> sort option
   val isFunctionType: sort -> bool
   val arityOfSort: sort -> int

end = struct  
   datatype sort = Base of Atom.atom | Proc of (sort list) * sort
                                       (* sort list should be non-empty *)

   exception SortError;
   type ord_key = sort
   fun compare (Base s, Base t) = Atom.compare (s,t)
     | compare (Proc _, Base _) = GREATER
     | compare (Base _, Proc _) = LESS
     | compare (Proc (ts,t), Proc (ss,s)) = 
       case compare (s,t) of
	   GREATER => GREATER
	 | LESS => LESS
	 | EQUAL => compareList ts ss
   and compareList [] [] = EQUAL
     | compareList (t::ts) [] = GREATER
     | compareList [] (s::ss) = LESS
     | compareList (t::ts) (s::ss) = 
       case compare (t,s)
	of EQUAL => compareList ts ss
	 | LESS => LESS
	 | GREATER => GREATER

   fun equal (S,T) = (compare (S,T)) = EQUAL

   fun toString (Base s) = Atom.toString s
     | toString (Proc (ss,s)) = (ListUtil.toStringComma toString' ss) ^ " -> " ^ (toString' s)
   and toString' (Base s) = Atom.toString s
     | toString' (Proc (ss,s)) = "(" ^ (ListUtil.toStringComma toString' ss) ^ " -> " ^ (toString' s) ^ ")"


   fun toString2 (Base s) = "-> " ^ (Atom.toString s) 
     | toString2 (Proc (ss,s)) = (ListUtil.toStringSpace toString ss) ^ " -> " ^ (toString s)

   val null = Base (Atom.atom "")
   fun fromString str  = Base (Atom.atom str)

  (* $B7?(B t $B$K0z?t$rM?$($F$$$/$H(B $B7?(B s $B$K$J$k$+(B *)
   fun isASubsort (Base s) (Base t) = Atom.compare (s,t) = EQUAL
     | isASubsort (Proc _) (Base _) = false
     | isASubsort s (t as (Proc (ts,t0))) = (equal (s,t)) orelse (isASubsort s t0)

  (* $B0z?t$rM?$($F$$$/$3$H$K$h$C$FF@$i$l$k7?(B $B$N%j%9%H(B *)
   fun supersorts (ty as (Base s)) = [ty]
     | supersorts (ty as (Proc (ts,t0))) = ty::(supersorts t0)

(*    fun plus ty n =  *)
(*        if n <= 0  *)
(*        then ty *)
(*        else case ty of  *)
(* 		Base _ => (print "Error: Sort.plus\n"; raise SortError) *)
(* 	      | Proc (ts,t0) => plus t0 (n-1)  *)

   fun plus ty n = 
       if n < 0 then NONE
       else if n = 0 then SOME ty
       else case ty of 
		Base _ => NONE
	      | Proc (ts,t0) => plus t0 (n-1) 

   fun isBaseType (Base _) = true
     | isBaseType (Proc _) = false

   fun args (Base _) = []
     | args (Proc (ts,_)) = ts

   fun return (s as (Base ty)) = s
     | return (Proc (_,ty)) = ty

   fun basicSortSetInSort sort =
       let
	   fun basicSortSet (Base a) set = AtomSet.add (set,a)
	     | basicSortSet (Proc (ts,ty)) set =
	       List.foldl
		   (fn (ty',set') => basicSortSet ty' set')
		   set
		   (ty::ts)
       in basicSortSet sort AtomSet.empty
       end


   fun isFunctionType (Base _) = false
     | isFunctionType (Proc _) = true

   fun arityOfSort (Base _) = 0
     | arityOfSort (Proc (ts,t0)) = List.length ts

end;


structure SortSet = RedBlackSetFn (Sort) : ORD_SET
structure SortPair = OrdKey2PairFn (structure A = Sort structure B = Sort) : ORD_KEY2
structure SortPairSet = RedBlackSetFn (SortPair) : ORD_SET

