(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/subst.sml
 * description: substitution, matching, unification
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature SUBST = 
   sig
   type subst = Term.term VarMap.map
   exception SubstError
   val empty: subst
   val applySubst: subst -> Term.term -> Term.term
   val inDomain: Term.var_key -> subst -> bool
   val isLinear: subst -> bool
   val isVarSubst: subst -> bool
   val toList: subst -> (Term.var_key * Term.term) list
   val toEqs: subst -> (Term.term * Term.term) list
   val restrictDomain: VarSet.set -> subst -> subst
   val compose: subst -> subst -> subst
   val merge: subst * subst -> subst option
   val mergeList: subst list -> subst option
   val match: Term.term -> Term.term -> subst option
   val matchSubterm: Term.term -> Term.term -> bool
   val isVariant: Term.term -> Term.term -> bool
   val identicalModuloRenamingRule: (Term.term * Term.term) -> (Term.term * Term.term) -> bool
   val renameTermToStd: Term.term -> Term.term
   val renameTerms: Term.term list -> Term.term list
   val renameTermDisjointFrom: Var.ord_key list -> Term.term -> Term.term
   val renameTermsDisjointFrom: Var.ord_key list -> Term.term list -> Term.term list
   val unify: Term.term -> Term.term -> subst option 
   val disjointUnify: Term.term * Term.term -> bool
   val mergeUnify: subst * subst -> subst option
   val mergeUnifyList: subst list -> subst option
   val linearize: Term.term -> Term.term
   val toString: subst -> string
   val toStringWithVarSort: subst -> string 
   val fromString: string -> subst option

    val complementOfTerm : Term.term list -> Term.term -> Term.term list
    val complementOfSubst : Term.term list -> subst -> subst list 
    val patCovered: Term.term list -> (Term.term list list * Term.term list list) -> bool
    val coveringPatternSequence: Term.term list -> Term.term list list -> bool

    val subtractPatternSequence: Term.term list -> (Term.term list * Term.term list) ->  Term.term list list option
    val patSubtraction: Term.term list -> (Term.term list list * Term.term list list) ->  Term.term list list

    val applySublist: Term.term VarMap.map list -> Term.term -> Term.term list
    val complementOfPattern: Term.term list -> Term.term list -> Term.term list
    val coveringPattern: Term.term list -> Term.term list -> bool
(*
    val deleteAuxFromTerm: Term.term list -> Term.term -> Term.ord_key list
    val equalPatterns: Term.term list -> Term.term list * Term.term list -> bool
    val extendTermAtPos: Term.term list -> Term.term -> int list -> Term.term list
    val isNonEmptyPattern: Term.term list -> Term.term list -> bool
    val selectPatternBySort: Term.term list -> Sort.ord_key -> Term.term list
*)
   end;

structure Subst : SUBST = 
   struct

   local 
       open Term;
       structure VS = VarSet
       structure VM = VarMap
       structure L = List
       structure LP = ListPair
       structure LU = ListUtil
   in

   type subst = term VM.map
   exception SubstError

   val empty = VM.empty

   (* applySubst sigma term :  sigma  term Ŭ  *)

   fun applySubst sigma term = 
       case term 
	of Var (x,_) =>  (case (VM.find (sigma, x))
			   of SOME t => t | NONE => term)
	 | Fun (f,ts,sort) => Fun (f, L.map (applySubst sigma) ts, sort)

   fun inDomain x sigma =  VM.inDomain (sigma, x)
   fun isLinear sigma =  L.all Term.isLinearTerm (VM.listItems sigma)
   fun isVarSubst sigma =  L.all Term.isVar (VM.listItems sigma)
   fun toList sigma =  VM.listItemsi sigma
   fun toEqs sigma =  L.map (fn (x,t) => (Term.Var (x,Term.sortOfTerm t), t)) (toList sigma)
			    
   (* f = compose g h defined by f(x) = g(h(x)) *)
   fun compose sub1 sub2 = 
       let val sub3 = 
	       VM.foldli (fn (v,t,map) => VM.insert (map,v,applySubst sub1 t)) VM.empty sub2
       in
	   VM.foldli (fn (v,t,map) => VM.insert (map,v,t)) sub1 sub3
       end

   (*** Example 
   - val sub1 = valOf (Subst.fromString "{?x:=f(?y),?y:=a}");
   - val sub2 = valOf (Subst.fromString "{?y:=g(?z),?z:=h(?y)}");
   - Subst.toString (Subst.compose sub1 sub2);
     val it = "{?x:=f(?y),?y:=g(?z),?z:=h(a)}" : string
   - Subst.toString (Subst.compose sub2 sub1);
     val it = "{?x:=f(g(?z)),?y:=a,?z:=h(?y)}" : string
    ***)

   fun restrictDomain vSet sigma =  VM.filteri (fn (v,t)=> VS.member (vSet,v)) sigma

   fun merge (sigma1,sigma2) =
       let val common = VM.intersectWith (fn (u,v)=> if Term.equal (u,v)
						     then SOME u
						     else NONE) (sigma1,sigma2)
       in if List.all isSome (VM.listItems common)
	  then SOME (VM.unionWith (fn (u,v) => u) (sigma1,sigma2))
	  else NONE
       end

   fun mergeList [] = SOME VM.empty
     | mergeList [sigma] = SOME sigma
     | mergeList (sigma1::sigma2::rest)  = 
       case merge (sigma1,sigma2) of
	   SOME sigma => mergeList (sigma::rest)
	 | NONE => NONE


   (* match pattern term : term  pattern ˾ȹ礹֤ *)
   local
       exception UnMatched
       fun matchMain [] sigma = SOME sigma
	 | matchMain ((s0,t0)::rest) sigma = 
	   case s0 
	    of Var (x,_) => (case (VM.find (sigma,x))
			      of SOME t1 => (case compare (t0,t1) 
					      of EQUAL => matchMain rest sigma 
					       | _ => raise UnMatched)
			       | NONE => matchMain rest (VM.insert (sigma, x,t0)))
	     | Fun (f,ts,_) => case t0 
				of Var _ => raise UnMatched
				 | Fun (g,ss,_) => case Fun.compare (f,g)
						    of EQUAL => matchMain (L.@(LP.zip (ts,ss), rest)) sigma
						     | _ => raise UnMatched
   in 
   fun match s t = matchMain [(s,t)] VM.empty
       handle UnMatched => NONE
   end

   fun matchSubterm pat (t as (Fun (f,ts,_))) =  isSome (match pat t) orelse
						 L.exists (fn ti => matchSubterm pat ti) ts
     | matchSubterm pat t = isSome (match pat t)




(*    fun identicalModuloRenamingRule (l,r) (l',r') = *)
(*        let val null = Atom.atom "" *)
(* 	   val rule1 = Fun (null,[l,r],Sort.null) *)
(* 	   val rule2 = Fun (null,[l',r'],Sort.null) *)
(*        in *)
(* 	   isSome (match rule1 rule2) *)
(* 	   andalso isSome (match rule2 rule1) *)
(*        end *)


   local
       exception NonRenaming

       fun isMono ren = 
	   let fun checkMono [] ys = true
		 | checkMono (x::xs) ys = 
		   if L.exists (fn y => Var.equal (x,y)) ys 
		   then false
		   else checkMono xs (x::ys)
	   in checkMono (L.map (fn (k,v) => v) (VM.listItemsi ren)) []
	   end

       fun isRenaming (Var _) (Fun _) ren = raise NonRenaming
	 | isRenaming (Fun _) (Var _) ren = raise NonRenaming
	 | isRenaming (Var (x,_)) (Var (y,_)) ren = 
	   (case VM.find (ren, x) of
		SOME z => if Var.equal (y,z) then ren 
			  else raise NonRenaming
	      | NONE => VM.insert (ren,x,y))
	 | isRenaming (Fun (f,ts,_)) (Fun (g,ss,_)) ren = 
	   if Fun.equal (f,g)
	   then LP.foldl 
		    (fn (ti,si,ans) => isRenaming ti si ans)
		    ren
		    (ts,ss)
	   else raise NonRenaming
   in
   fun isVariant s t = 
       let val sigma = isRenaming s t VM.empty
       in isMono sigma
       end
       handle NonRenaming => false
   fun identicalModuloRenamingRule (l1,r1) (l2,r2) = 
       let val sigma = isRenaming r1 r2 (isRenaming l1 l2 VM.empty)
       in isMono sigma
       end
       handle NonRenaming => false
   end


   fun renameTermToStd term =
       let val count = ref 0
	   val ren = ref VM.empty
	   fun newvar () = (count := (!count+1); 
			    Var.fromStringAndInt ("x",!count))
	   fun renaming (Var (x,ty)) = 
	       (case VM.find (!ren,x) of 
		    SOME y => Var (y,ty)
		  | NONE => let val y = newvar ()
				val _ = ren := VM.insert (!ren,x,y)
			    in Var (y,ty)
			    end)
	     | renaming (Fun (f,ts,ty)) =
	       Fun (f, List.map renaming ts,ty)
       in
	   renaming term
       end

   (* renameTerms [t0,t1,...] : t0,t1,...ѿ dijoint ˤʤ褦 term Υǥå
						  ѹΥꥹȤ֤ *)

   local
       fun varIndexes (Var (x,_)) = [#2 x]
	 | varIndexes (Fun (_,ts,_)) = L.concat (L.map varIndexes ts)
       fun renameTermsMain [] _ accum = (L.rev accum)
	 | renameTermsMain (t0::ts) n accum =
	   let val idx = (varIndexes t0)
	   in if null idx then renameTermsMain ts n (t0::accum)
	      else let val (i, j) = (L.foldr Int.min (L.hd idx) (L.tl idx), 
				     L.foldr Int.max (L.hd idx) (L.tl idx))
		   in
		       renameTermsMain ts (n-i+j+1) ((increaseVarIndexBy (n-i) t0)::accum)
		   end
	   end
   in
   fun renameTerms ts = renameTermsMain ts 0 []
   fun renameTermDisjointFrom vs t = 
       if null vs then t
       else let val (n::ns) = L.map (fn (x,y) => y) vs
		val m = L.foldl Int.max n ns
       		val i = L.foldl Int.min 0 (varIndexes t)
	    in if m < i then t
	       else increaseVarIndexBy (m-i+1) t
	    end
   fun renameTermsDisjointFrom vs ts = 
       if null vs then ts
       else let val (n::ns) = L.map (fn (x,y) => y) vs
		val m = L.foldl Int.max n ns
       		val i = L.foldl Int.min 0 (L.concat (L.map varIndexes ts))
	    in if m < i then ts
	       else L.map (increaseVarIndexBy (m-i+1)) ts
	    end
   end


   (* unify term1 term2 : term1  term2 ñ첽ǽʤ mgu ֤ *)
   (* ɬפʤ unify  renaming Τ˺ʤ *)
   local
       exception UnUnified
       fun applySubst1 (x,t) term = 
	   case term 
	    of Var (y,_) =>  (case Var.compare (x,y) of EQUAL => t | _ => term)
	     | Fun (f,ts,sort) => Fun (f, map (applySubst1 (x,t)) ts, sort)
       fun unifyMain [] sigma = SOME sigma
	 | unifyMain ((s0,t0)::rest) sigma = 
	   case (s0,t0)
	    of (Var (x,ty1), Var (y,ty2)) => 
	       (case Var.compare (x,y) 
		 of EQUAL => unifyMain rest sigma 
		  | _ => if Sort.equal (ty1,ty2) (* add sort check 2017/03/04 *)
			 then eliminate x t0 rest sigma
			 else raise UnUnified)
	     | (Fun (f,ts,_), Fun (g,ss,_)) => 
	       (case Fun.compare (f,g)
		 of EQUAL => unifyMain ((LP.zip (ts,ss))@rest) sigma
		  | _ => raise UnUnified)
	     | (Var (x,ty1), Fun(_,_,ty2)) => (* add sort check 2017/03/04 *)
	       if VS.member (varSetInTerm t0, x) orelse not (Sort.equal (ty1,ty2))
	       then raise UnUnified
	       else eliminate x t0 rest sigma
	     | (Fun (_,_,ty1), Var (y,ty2)) =>   (* add sort check 2017/03/04 *)
	       if VS.member (varSetInTerm s0, y) orelse not (Sort.equal (ty1,ty2))
	       then raise UnUnified
	       else eliminate y s0 rest sigma
       and eliminate x t0 rest sigma =
	   let
	       val rest' = (map (fn (s,t) => (applySubst1 (x,t0) s, applySubst1 (x,t0) t)) rest)
	       val sigma' = VM.map (fn s => applySubst1 (x,t0) s) sigma
	   in 
               unifyMain rest' (VM.insert (sigma',x,t0))
	   end
   in 
   fun unify s t = unifyMain [(s,t)] VM.empty
       handle UnUnified => NONE
   end (* of local *)


   (* disjointUnify term1 term2 : term1  term2 ñ첽ǽʤ true ֤ *)
   (* term1  term2 ѿ disjoint ȸ *)
   fun disjointUnify (term1,term2) = 
       let val [s,t] = renameTerms [term1,term2]
       in isSome (unify s t)
       end

(*
   fun disjointUnify term1 term2 = 
       let val count = (maxVarIndexInTerm term1) + 1
	   val term2' = increaseVarIndexBy count term2
	   val varset1 = varSetInTerm term1
	   val varset2 = varSetInTerm term2'
	   fun incIndexesBy n (Var ((x,i),sort)) = if VS.member (varset2,(x,i)) then Var ((x,i+n), sort)
						   else Var ((x,i), sort)
	     | incIndexesBy n (Fun (f,ts,sort)) = Fun (f, L.map (incIndexesBy n) ts, sort)


	   val makeSigma1 = VM.foldli (fn (v,t,map) => if VS.member (varset1,v)
						       then VM.insert (map,v,incIndexesBy (~count) t)
						       else map)
				      VM.empty 
	   val makeSigma2 = VM.foldli (fn (v,t,map) => if VS.member (varset2,v)
						       then VM.insert (map,Var.increaseIndexBy (~count) v,
								       incIndexesBy (~count) t)
						       else map)
				      VM.empty
	   val sigmaOp = unify term1 term2'
       in
	   case sigmaOp of 
	       SOME sigma => SOME (makeSigma1 sigma, makeSigma2 sigma)
	     | NONE => NONE
       end
Bug?: +(x,y)  +((y,z),x)  disjointUnify  
      (x := y+z, z:=y) Ȥʤꡤ   +(+(y,z),y) ȤʤäƤޤ

*)
   (*** Example
       - Subst.disjointUnify (IOFotrs.rdTerm "ap(ap(L,?x),?y)") (IOFotrs.rdTerm "ap(?y0,?y0)");
       - print (let val (sub1,sub2) = valOf it in (Subst.toString sub1) ^ (Subst.toString sub2) end);
         {?y:=ap(L,?x)}{?y0:=ap(L,?x)}val it = () : unit
    ******)


   fun mergeUnify (sigma1,sigma2) =
       let val common = VM.intersectWith (fn (u,v)=> unify u v) (sigma1,sigma2)
       in if List.all isSome (VM.listItems common)
	  then SOME (VM.unionWithi (fn (k,u,v) => let val sub = valOf (valOf (VM.find (common,k)))
						  in applySubst sub u
						  end)
				   (sigma1,sigma2))
	  else NONE
       end

   fun mergeUnifyList [] = SOME VM.empty
     | mergeUnifyList [sigma] = SOME sigma
     | mergeUnifyList (sigma1::sigma2::rest)  = 
       case mergeUnify (sigma1,sigma2) of
	   SOME sigma => mergeUnifyList (sigma::rest)
	 | NONE => NONE



   (* 2ܤиѿ̾ؤơ *)
   fun linearize term = 
       let val count = ref (maxVarIndexInTerm term)
	   fun newvar () = (count := (!count+1); 
			    Var.fromStringAndInt ("x",!count))
	   fun rename vset (Var (x,ty)) = if VS.member (vset,x)
					  then (Var (newvar (),ty), vset)
					  else (Var (x,ty), VS.add (vset,x))
	     | rename vset (Fun (f,ts,ty)) = let val (ts',vset') = renameList vset ts
					     in (Fun (f,ts',ty),vset') 
					     end
	   and renameList vset []  = ([],vset)
	     | renameList vset (t::ts) = 
	       let val (t',vset1) = rename vset t
		   val (ts',vset2) = renameList vset1 ts
	       in (t'::ts', vset2)
	       end
	   val (new,_) = rename VS.empty term
       in new
       end

   fun toString sub = let fun prPair (x,t) = (Var.toString x) ^ ":=" ^ (Term.toString t)
		      in ListUtil.toStringCommaCurly prPair (VM.listItemsi sub)
		      end

   fun toStringWithVarSort sub = 
       let fun prPair (x,t) = (Var.toString x) ^ ":=" ^ (Term.toStringWithVarSort t)
       in ListUtil.toStringCommaCurly prPair (VM.listItemsi sub)
       end


   fun fromString str = let val ts = IOFotrs.rdAssignmentSet str
			in
			    if L.all (fn (x,_) => Term.isVar x) ts
			    then 
				SOME (L.foldl (fn ((v,t),vmap) => VM.insert (vmap,valOf (varRootOfTerm v),t))
					      VM.empty
					      ts)
			    else NONE
			end
				   

   fun sameSubst (sigma, rho) = 
       let val kvlist0 = VM.listItemsi sigma
	   val klist0 = L.map (fn (k,v) => k) kvlist0
	   val kvlist1 = VM.listItemsi rho
	   val klist1 = L.map (fn (k,v) => k) kvlist1
       in  LU.setEqual' Var.equal (klist0,klist1)
	   andalso L.all (fn k => Term.equal (VM.lookup (sigma,k), VM.lookup (rho,k))) klist0
       end

(****************************)
(*** pattern ˴ؤؿ ***)
(****************************)

   type single_pattern = term (* should be linear term *)
   type pattern = term list (* should be linear *)
   (* W: Υѥ    *)
   (* P: ѥ   *)
   (* pat \in P: signle pattern   *)

   fun selectPatternBySort P ty = L.filter (fn pat => Sort.equal (ty, sortOfTerm pat)) P

  (* pat \subseteq P*)
   fun isNonEmptyTerm W pat = 
       L.exists (fn p =>  isSome (match p pat)) W

   fun isNonEmptyPattern W P = 
       L.exists (fn pat => isNonEmptyTerm W pat) P

   fun isEmptyPattern W P = not (isNonEmptyPattern W P)

   (* pos is supposed to be a variable position of pat *)
   fun extendTermAtPos W pat pos = 
       let  fun extendPatSub W pat pos = 
		case pos of [] =>  (* root position *)
			    (case pat of Var (x,ty) => selectPatternBySort W ty
				       | Fun (f,ts,ty) => (print "extendTermAtPos: not a variable position\n"; raise SubstError))
			  | i::ps => (* non-root position *)
			    (case pat of Var (x,ty) => (print "extendTermAtPos: not a variable position\n"; raise SubstError)
				       | Fun (f,ts,ty) => let val pati = extendPatSub  W (L.nth (ts,i)) ps
							  in L.map (fn pi => Fun (f, LU.replaceNth (ts,i,pi), ty)) pati 
							  end)
       in L.map Term.linearize (extendPatSub W pat pos)
       end

   fun intersectionOfSingePatterns (pat1,pat2) =
       let val [pat1',pat2'] = renameTerms [pat1,pat2]
       in case unify pat1' pat2' of
   	      NONE => []
   	    | SOME sigma => [applySubst sigma pat1']
       end

  (* pat1  pat2  subsume 뤫 pat2  pat1 üʾ礫 *)
   fun subsumeSingePattern pat1 pat2 = isSome (match pat1 pat2)

   fun isEmptyTerm W pat = L.all (fn p=> null (intersectionOfSingePatterns (p,pat))) W
   fun isEmptyPattern W P = L.all (fn p => isEmptyTerm W p) P

   fun isNonEmptyTerm W pat = L.exists (fn p=> not (null (intersectionOfSingePatterns (p,pat)))) W
   fun isNonEmptyPattern W P = L.exists (fn p => isNonEmptyTerm W p) P

   fun deleteAuxFromTerm P pat = 
       let val (pat'::P')  = renameTerms (pat::P)
       in  LU.eliminateDuplication' Term.equal 
				    (LU.mapAppend (fn p => intersectionOfSingePatterns (p,pat')) P')
       end

   fun applySublist sublist t = L.map (fn sigma => (applySubst sigma t)) sublist

   fun intersectionOfPatterns (P,Q) = 
       LU.mapAppend (fn p => deleteAuxFromTerm P p) Q

   local
       fun listToSubst ks vs = LP.foldl (fn (k,v,map) => VM.insert (map,k,v)) VM.empty (ks,vs)
   in
   (* աsort 㤦Τ complement ʤ *)
   (*  W Ǥϡ f(..variable only ...) η *)
   (* fun complementOfTerm W t = *)
   (*     let fun compOfTerm [] t ans = L.rev ans *)
   (* 	     | compOfTerm (p::ps) t ans = *)
   (* 	       if not (Term.haveSameSort (p,t)) *)
   (* 	       then compOfTerm ps t ans *)
   (* 	       else case match p t of *)
   (* 			NONE => compOfTerm ps t (p::ans) *)
   (* 		      | SOME sigma => *)
   (* 			let (* val _ = PrintUtil.println ("\nP: " ^ (LU.toStringCommaSquare Term.toString W)) *) *)
   (* 			    (* val _ = PrintUtil.println ("pattern: " ^ (Term.toString p)) *) *)
   (* 			    (* val _ = PrintUtil.println ("term: " ^ (Term.toString t)) *) *)
   (* 			    (* val _ = PrintUtil.println ("sub: " ^ (toString sigma)) *) *)
   (* 			    (* val _ = PrintUtil.println ("compute complement sub of " ^ (toString sigma) ^ "...") *) *)
   (* 			    val ss = complementOfSubst W sigma *)
   (* 			    (* val _ = PrintUtil.println ("...complement sub of " ^ (toString sigma) *) *)
   (* 			    (* 			       ^ " := " ^ (LU.toStringCommaSquare toString ss)) *) *)
   (* 			    val pcomp = L.map Term.linearize (applySublist ss p) *)
   (* 			in compOfTerm ps t (pcomp @ ans) *)
   (* 			end *)
   (*     in compOfTerm W t [] *)
   (*     end *)
   fun complementOfTerm W t =
       let fun compOfTermList [] _ = []
   	     | compOfTermList ps [] = ps
   	     | compOfTermList ps (t::ts) =
   	       let val compt = compOfTerm ps t
   	       in compOfTermList compt ts
   	       end
   	   and compOfTerm [] _ = []
   	     | compOfTerm (p::ps) t =
   	       let val [p',t'] = renameTerms [p,t]
   	       in if isSome (match t' p')
   		  then compOfTerm ps t
   		  else case unify p' t' of
   			   NONE => p::(compOfTerm ps t)
   			 | SOME sigma =>
   			let (* val _ = PrintUtil.println ("\nP: " ^ (LU.toStringCommaSquare Term.toString W)) *)
   			    (* val _ = PrintUtil.println ("pattern: " ^ (Term.toString p)) *)
   			    (* val _ = PrintUtil.println ("term: " ^ (Term.toString t)) *)
   			    (* val _ = PrintUtil.println ("sub: " ^ (toString sigma)) *)
   			    (* val _ = PrintUtil.println ("compute complement sub of " ^ (toString sigma) ^ "...") *)
   			    val ss = complementOfSubst W sigma
   			    (* val _ = PrintUtil.println ("...complement sub of " ^ (toString sigma) *)
   			    (* 			       ^ " := " ^ (LU.toStringCommaSquare toString ss)) *)
   			    val pcomp = L.map Term.linearize (applySublist ss p')
   			    val tcomp = L.map Term.linearize (applySublist ss t')
   			    val pcomp' = LU.eliminateDuplication' (fn (x,y) => isVariant x y) (pcomp @ ps)
   			in compOfTermList pcomp' tcomp
   			end
   	       end
   	   val (W1,W2) = L.partition (fn p => Term.haveSameSort (p,t)) W
       in W2 @ (compOfTerm W1 t)
       end
   and complementOfSubst W sigma = 
       let fun compOfSubst sigma = 
	       let val sigma' = VM.filteri (fn (k,v) => (not (Term.isVar v))) sigma
	       in if VM.isEmpty sigma'
		  then []
		  else let val kvlist = VM.listItemsi sigma'
			   val klist = L.map (fn (k,v) => k) kvlist
			   val vlist = L.map (fn (k,v) => v) kvlist
			   (* val _ = PrintUtil.println ("take complements of terms: " ^ (LU.toStringComma Term.toString vlist)) *)
		           val entlist =  L.map (fn t => (if isNonEmptyPattern W [t] then [t] else [])
							 @  (L.filter (fn p => Term.haveSameSort (p,t)) (complementOfTerm W t)))
						vlist
			   val sublist0 = LU.allCombinations entlist
			   val result = LU.deleteOne' sameSubst sigma' (L.map (listToSubst klist) sublist0)
		       in result 
		       end
	       end
       in compOfSubst sigma
       end
   end (* local *)

   fun complementOfPattern W [] =  W
     | complementOfPattern W (p::ps) = 
       intersectionOfPatterns (complementOfTerm W p, complementOfPattern W ps)

   fun subtractPattern W (P,Q) = 
       intersectionOfPatterns (P, complementOfPattern W Q)

   fun equalPatterns W (P,Q) = 
       isEmptyPattern W (subtractPattern W (P,Q))
       andalso isEmptyPattern W (subtractPattern W (Q,P))

  (* P  Τ򥫥СƤ뤫 *)
   fun coveringPattern W P = 
       isEmptyPattern W (subtractPattern W (W,P))
       (* let fun coverSub [] _ = true *)
       (* 	     | coverSub (w::ws) qs  = *)
       (* 	       let val (w'::qs')  = renameTerms (w::qs) *)
       (* 		   val cand = L.mapPartial (fn q => case unify w' q of  *)
       (* 							NONE => NONE *)
       (* 						      | SOME sigma => SOME (q,sigma)) qs' *)
       (* 	       in if null cand  *)
       (* 		  then false *)
       (* 		  else let val (q,sigma) = hd cand *)
       (* 		       in if isSome (match q w') *)
       (* 			  then coverSub ws qs *)
       (* 			  else let val qs2 = LU.deleteOne' Term.equal q qs' *)
       (* 				   val sublist = complementOfSubst W sigma *)
       (* 				   val wcomp = applySublist sublist w' *)
       (* 				   val qcomp = applySublist sublist q *)
       (* 			       in coverSub (LU.eliminateDuplication' (fn (x,y) => isVariant x y) (wcomp @ ws)) *)
       (* 					   (LU.eliminateDuplication' (fn (x,y) => isVariant x y) (qcomp @ qs')) *)
       (* 			       end *)
       (* 		       end *)
       (* 	       end *)
       (* in coverSub W P *)
       (* end *)


   fun subtractPatternSequence cPat (ps,qs) = 
       if LP.all (fn (p,q) => isSome (match q p)) (ps,qs)
       then SOME []  (* qs is more general than ps *)
       else let val len = L.length ps
		val terms = renameTerms (ps @ qs)
		val ps' = L.take (terms, len)
		val qs' = L.drop (terms, len)
		val unifs = LP.map (fn (p,q) => unify p q) (ps',qs')
		fun complement sigma = if VM.isEmpty sigma
					  orelse L.all Term.isVar (VarMap.listItems sigma)
				       then NONE
				       else SOME (complementOfSubst cPat sigma)
		fun apply sublistlist ps = let val ls = L.map (fn n => 
								  LU.eliminateDuplication' (fn (x,y) => isVariant x y)
											   (applySublist (L.nth (sublistlist,n))  
													 (L.nth (ps,n))))
							      (L.tabulate (L.length ps, fn n => n))
					   in LU.allCombinations ls end
	    in if L.exists (not o isSome) unifs
	       then NONE (* no common part *)
	       else let val sublistlist = L.map (fn sigmaop => case complement (valOf sigmaop) of
								   SOME sublist => ((valOf sigmaop)::sublist)
								 | NONE => [valOf sigmaop]) unifs 
			val inst = LP.map (fn (sigmaop,p) => applySubst (valOf sigmaop) p) (unifs,ps')
			val ans = LU.deleteOne' (fn (ps,qs) => LP.all (fn (p,q) => isVariant p q) (ps,qs))
						inst
						(apply sublistlist ps')
		    in SOME ans
		    end
	    end

val patE = IOFotrs.rdTerm "e"
val patI = IOFotrs.rdTerm "i(?x)"
val patX = IOFotrs.rdTerm "?x"
val patY = IOFotrs.rdTerm "?y"
val P = [patE, patI]
val pslist = subtractPatternSequence P ([patX,patY], [patE,patX])


(*
   val _ = print "coveringPattern\n"
   val PNat0 = [patS, pat0]
   val pslist = valOf (subtractPatternSequence PNat0 ([patX,patX], [pat0,patS]))
(*   val pslist = valOf (subtractPatternSequence PNat0 ([patX], [pat0])) *)
   val _ = PU.println ("Constructor patterns: " ^
    		 (LU.toStringSemicolonSquare  
		     (fn ps => LU.toStringCommaCurly Term.toString ps) pslist))
*)

  (* cPat = [0, s(x)] 
     patlist = [[0,y] [s(x),y]] 
     ps = [0,const]
   *)


  (* cPat = [0, s(x)] 
     pslist=[x,y], patlist=[[0,y] [s(x),y]]     => true
     pslist=[0,y], patlist = [[0,0],[x,s(y)]]   => true
     pslist=[0,y], patlist = [[0,0],[0,s(y)]]   => true
   *)

   (* pslist  patlist ǥСƤ뤫 *)
   fun patCovered cPat (pslist,patlist)=
       let fun subsumeseq (ps,qs) = LP.all (fn (p,q) => isSome (match p q)) (ps,qs)
   	   fun elimdup [] ans = L.rev ans
   	     | elimdup (ps::rest) ans = (L.filter (fn qs => not (subsumeseq (ps,qs))) rest)
   					@ (ps::(L.filter (fn qs => not (subsumeseq (ps,qs))) ans))
   	   fun covers [] qslist = true
   	     | covers pslist [] = false
   	     | covers (ps::pslist) qslist =
	       if L.exists (fn qs => subsumeseq (qs,ps)) qslist
	       then covers pslist qslist
	       else case L.find (fn qs => let val len = L.length ps
					      val terms = renameTerms (ps @ qs)
					      val ps' = L.take (terms, len)
					      val qs' = L.drop (terms, len)
					  in  LP.all (fn (p,q) => isSome (unify p q)) (ps',qs') 
					  end) 
				qslist of
			NONE => false
		      | SOME qs => case subtractPatternSequence cPat (ps,qs) of
				       NONE => (print "Error: no common part found ???\n"; raise SubstError)
				     | SOME comp => covers (elimdup (comp @ pslist) []) qslist
          in covers pslist patlist
       end

(*
   val _ = if patCovered PNat0 ([[patX,patX]], [[pat0,patX], [patS,patX]])
   	   then print "[[x,y]] is covered by [[0,y],[s(x),y]].\n"
   	   else print "[[x,y]] is not covered by [[0,y],[s(x),y]].???\n"

   val _ = if patCovered PNat0 ([[pat0,patX]], [[pat0,pat0], [pat0,patS]])
   	   then print "[[0,y]] is covered by [[0,0],[0,s(y)]].\n"
   	   else print "[[0,y]] is not covered by [[0,0],[0,s(y)]].???\n"

   val _ = if patCovered PNat0 ([[pat0,patX]], [[pat0,pat0], [patS,patX]])
   	   then print "[[0,y]] is covered by [[0,0],[s(x),y]].???\n"
   	   else print "[[0,y]] is not covered by [[0,0],[s(x),y]].\n"
*)

   (* pslist - patlist  *)
   fun patSubtraction cPat (pslist,patlist)=
       let fun subsumeseq (ps,qs) = LP.all (fn (p,q) => isSome (match p q)) (ps,qs)
   	   fun elimdup [] ans = L.rev ans
   	     | elimdup (ps::rest) ans = (L.filter (fn qs => not (subsumeseq (ps,qs))) rest)
   					@ (ps::(L.filter (fn qs => not (subsumeseq (ps,qs))) ans))
   	   fun subtract [] qslist = []
   	     | subtract pslist [] = pslist
   	     | subtract (ps::pslist) qslist =
	       if L.exists (fn qs => subsumeseq (qs,ps)) qslist
	       then subtract pslist qslist
	       else case L.find (fn qs => let val len = L.length ps
					      val terms = renameTerms (ps @ qs)
					      val ps' = L.take (terms, len)
					      val qs' = L.drop (terms, len)
					  in  LP.all (fn (p,q)
							 => isSome (unify p q)) (ps',qs') 
					  end) 
				qslist of
			NONE => ps::(subtract pslist qslist)
		      | SOME qs => case subtractPatternSequence cPat (ps,qs) of
				       NONE => (print "Error: no common part found ???\n"; raise SubstError)
				     | SOME comp => subtract (elimdup (comp @ pslist) []) qslist
	   val ans = subtract pslist patlist
(*	   val ans = L.map Term.linearize ans0
	   val _ = PrintUtil.println ("ans patterns: " ^
    			       (LU.toStringSemicolonSquare  
				    (fn ps => LU.toStringCommaAngle Term.toString ps) ans)) *)
          in ans


       end


  (* cPat = [0, s(x)] 
     patlist = [[0,y] [s(x),y]]
     patlist = [[0,0] [s(x),0],[x,s(y)]]
     patlist = [[0,y] [s(0),y],[s(s(x)),y]]
   *)

   fun coveringPatternSequence cPat [] = null cPat
     | coveringPatternSequence cPat patlist =
       let val hdpat = hd patlist
   	   val count = ref 0
   	   fun newVar () = (count := (!count) + 1; Var.fromStringAndInt ("x",!count))
   	   fun newSortedVar ty = Term.Var (newVar (), ty)
   	   val basepat = L.map (fn t => newSortedVar (Term.sortOfTerm t)) hdpat
       in patCovered cPat ([basepat],patlist)
       end


(*
   val _ = if coveringPatternSequence PNat0 [[pat0,patX], [patS,patX]]
   	   then print "[[0,y],[s(x),y]]covering Nat over 0, s.\n"
   	   else print "[[0,y],[s(x),y]] not covering Nat over 0, s.???\n"

   val _ = if coveringPatternSequence PNat0 [[pat0,pat0], [patS,pat0], [patX,patS]]
   	   then print "[[0,0],[s(x),0],[x,s(y)]] covering Nat over 0, s.\n"
   	   else print "[[0,0],[s(x),0],[x,s(y)]] not covering Nat over 0, s.???\n"

   val _ = if coveringPatternSequence PNat0 [[pat0,patX], [patS0,patX], [patSS, patX]]
   	   then print "[[0,y],[s(0),y],[s(s(x)),y]] covering Nat over 0, s.\n"
   	   else print "[[0,y],[s(0),y],[s(s(x)),y]] not covering Nat over 0, s.???\n"
*)

   end (* of local *)

   end;

