(******************************************************************************
 * Copyright (c) 2013-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/tree_automata.sml
 * description: utility for tree automata
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature TREE_AUTOMATA = 
sig
   type tree_automaton = { Signature: FunIntSet.set,
			   States: TermSet.set, 
			   Final: TermSet.set,
			   Rules: (Term.term * Term.term) list (* without epsilon transition *) }
   val toString: tree_automaton -> string
   val toProofTree: tree_automaton -> unit -> string
   val isStateFun: Fun.ord_key -> bool
   val isState: Term.term -> bool
   val reachableStates: (Term.term * Term.term) list -> Term.term -> Term.term list
   val isWellFormed: tree_automaton -> bool

   val member: Term.term -> tree_automaton -> bool
   val isEmpty: tree_automaton -> bool
   val isComplementEmpty: tree_automaton -> bool
   val isSubseteq: tree_automaton * tree_automaton -> bool
   val isIntersectionEmpty: tree_automaton * tree_automaton -> bool
   val haveEmptyIntersection: tree_automaton -> Term.term * Term.term -> bool

   val anonymousSymbol: string
   val anonymousTerm: Term.term
   val anonymize: Term.term -> Term.term
   val qOfTerm: Term.term -> Term.term 
   val qOfNum: int -> Term.term 
   val anonymousState: Term.term
   val makeRuleForTerm: Term.term -> Term.term * Term.term
   val rulesForAcceptingAllTerms: FunIntSet.set -> (Term.term * Term.term) list
   val taAcceptingAllTerms: FunIntSet.set -> tree_automaton
   val taAcceptingGroundInstancesOfLinearTermSet: FunIntSet.set -> TermSet.set -> tree_automaton
   val taAcceptingGroundInstancesOfLinearTerm: FunIntSet.set -> Term.term -> tree_automaton
   val degrowingCompletion: FunIntSet.set -> (Term.term list) -> (Term.term * Term.term) list -> tree_automaton
   val runDebug: bool ref

end;

structure TreeAutomata: TREE_AUTOMATA =
struct
   local
     structure CU = CertifyUtil
     structure FIS = FunIntSet
     structure L = List
     structure LP = ListPair
     structure LU = ListUtil
     structure PU = PrintUtil
     structure TP = TermPair
     structure TPS = TermPairSet
     structure TS = TermSet
     structure VM = VarMap
     structure VS = VarSet
     exception Error
     fun consList [] [] = []
       | consList [] _  = []
       | consList _ []  = []
       | consList (x::xs) (ys::yss) = (x::ys):: consList xs yss
     open PrintUtil
  in
  val runDebug = ref false
  fun debug f = if (!runDebug) then f () else ()

   type tree_automaton = { Signature: FunIntSet.set,
			   States: TermSet.set, 
			   Final: TermSet.set,
			   Rules: (Term.term * Term.term) list (* without epsilon transition *) }

   fun toString (ta:tree_automaton) =
       let val faSet = #Signature ta
	   val Q = #States ta
	   val Qf = #Final ta
	   val Delta = #Rules ta
       in ("<F,Q,Q_f,Delta> where\n"
	   ^ "   F = " ^ (PU.prSetInOneLine FunInt.toString (FIS.listItems faSet)) ^ "\n"
	   ^ "   Q = " ^ (PU.prSetInOneLine Term.toString (TS.listItems Q)) ^ "\n"
	   ^ "   Qf = " ^ (PU.prSetInOneLine Term.toString (TS.listItems Qf)) ^ "\n"
	   ^ "   Delta = \n" ^ (Trs.prRules Delta))
       end

  fun prState t () = CU.encloseProofLeafBy "state" (Term.toString t)

  fun prTransition (l,r) () = 
      if Term.isVar l then raise Error
      else CU.encloseProofTreesBy "transition"
				  [fn _=> CU.encloseProofTreesBy "lhs"
			((Fun.toProofTree (valOf (Term.funRootOfTerm l)))
			 :: L.map prState (Term.argsOfTerm l)),
			   fn _=> CU.encloseProofTreeBy "rhs" (prState r)]

  fun toProofTree (ta:tree_automaton) () =
      let val faSet = #Signature ta
	  val Q = #States ta
	  val Qf = #Final ta
	  val Delta = #Rules ta
      in CU.encloseProofTreesBy "treeAutomaton" 
	 [fn _ => CU.encloseProofTreesBy "finalStates" (L.map prState (TS.listItems Qf)),
	  fn _ => CU.encloseProofTreesBy "transitions" (L.map prTransition Delta)]
      end

   (* 状態は "q_" から始まる関数名とする 
         * 項に対する状態は "q_{t}" を使う，ただし t は変数を anonymousSymbol に置き換えた項の文字列表現
         * 変数に対する状態は "q_{*}" を使う，ここで * は anonymousSymbol
         * プログラム中の一時的な状態は "q_n" (n = 0,1,2,...) を使う 
    *)
   fun isStateFun f = String.isPrefix "q_" (Fun.toString f)
   fun isState t = Term.isFun t andalso isStateFun (valOf (Term.funRootOfTerm t))

   fun reachableStates Delta t = 
       if Term.isVar t
       then (print "reachableStates in tree_automata.sml\n"; raise Error)
       else if isState t
       then [t]
       else let val f = valOf (Term.funRootOfTerm t)
		val args = Term.argsOfTerm t
		val tss = L.map (reachableStates Delta) args
	    in LU.mapAppend (fn ts => Rewrite.rootRewriteAll Delta (Term.mkFunTerm (f,ts,Sort.null)))
			    (List.foldr (ListXProd.mapX (fn (x,xs)=>x::xs)) [[]] tss)
	    end

   fun isWellFormed (ta:tree_automaton) =
       let val faSet = #Signature ta
	   val Q = #States ta
	   val Qf = #Final ta
	   val Delta = #Rules ta
	   fun isWellFormedRule (l,r) = Term.isFun l
					andalso let val f = valOf (Term.funRootOfTerm l)
						    val args = Term.argsOfTerm l
						in FIS.member (faSet, (f,L.length args))
						   andalso L.all (fn u=> TS.member (Q,u)) args
						   andalso TS.member (Q,r)
						end
       in L.all isState (TS.listItems Q)
	  andalso L.all (fn (f,_) => not (isStateFun f )) (FIS.listItems faSet)
	  andalso TS.isSubset (Qf, Q)
	  andalso L.all isWellFormedRule Delta
       end 

   (* member: bottom up version *)
   fun member_bot term (ta:tree_automaton) =
       let val Qf = #Final ta
	   val Delta = #Rules ta
	   val nfset = Rewrite.normalFormSet Delta term
       in not (TS.isEmpty (TS.intersection (nfset, Qf)))
       end

   (* member: top down version *)
   fun member_top term (ta:tree_automaton) =
       let val Delta = #Rules ta
	   val Qf = #Final ta
	   fun check (t,q) =
	       L.exists (fn (l,r) => 
			    Term.haveSameRoots (t,l)				 
			    andalso Term.equal (r,q)
			    andalso let val subs = LP.zip (Term.argsOfTerm t, Term.argsOfTerm l)
				    in L.all check subs
				    end)
			Delta
       in L.exists (fn q => check (term,q)) (TS.listItems Qf)
       end

   (* member: mixed version *)
   local
       fun acceptAmong Delta (t,qs) =
	   let val ts = Term.argsOfTerm t
	       val Delta2 = L.filter (fn (l,r) =>
					 Term.haveSameRoots (t,l) 
					 andalso LU.member' Term.equal r qs) 
				     Delta
	       val qss =  L.foldl (fn ((l,_),xss) => consList (Term.argsOfTerm l) xss)
				  (L.tabulate (L.length ts,fn _=> []))
				  Delta2
	       val qss2 = L.map (acceptAmong Delta) (LP.zip (ts,qss))
	   in LU.eliminateDuplication' Term.equal
	      (L.mapPartial (fn (l,r) => 
			       if L.all (fn (q',qs') => LU.member' Term.equal q' qs')
					(LP.zip (Term.argsOfTerm l,qss2))
			       then SOME r
			       else NONE)
			    Delta2)
	   end
   in
   fun member term (ta:tree_automaton) =
       let val Delta = #Rules ta
	   val Qf = #Final ta
       in not (L.null (acceptAmong Delta (term,TS.listItems Qf)))
       end
   end

   (* isEmpty: bottom up version *)
   fun isEmpty_bot (ta:tree_automaton) =
       let val Delta = #Rules ta
	   val Qf = #Final ta
	   fun check Qold = 
	       let val qs = L.mapPartial
				  (fn (l,r) => if L.all (fn t => TS.member (Qold,t))
							(Term.argsOfTerm l)
					       then SOME r
					       else NONE) 
				  Delta
	       in if L.all (fn q => TS.member (Qold,q)) qs
		  then Qold
		  else check (TS.addList (Qold,qs))
	       end
       in TS.isEmpty (TS.intersection (check TS.empty, Qf))
       end

   (* isEmpty: top down version *)
   fun isEmpty (ta:tree_automaton) =
       let val Delta = #Rules ta
	   val Qf = #Final ta
	   val Delta2 = L.filter (fn (l,r) => not (LU.member' Term.equal r (Term.argsOfTerm l))) Delta
	   fun empCheck (q,qs) =
	       if LU.member' Term.equal q qs
	       then true
	       else L.all (fn (l,r) => 
				 not (Term.equal (q,r))
				 orelse
				 let val xs = Term.argsOfTerm l
				 in if L.exists (fn q'=> LU.member' Term.equal q' qs) xs
				    then true
				    else if L.all (fn q' => not (empCheck (q', q::qs))) 
						  (LU.eliminateDuplication' Term.equal xs)
				    then false
				    else true
				 end)
			  Delta2
       in L.all (fn q => empCheck (q,[])) (TS.listItems Qf)
       end

   (* isCompentEmpty: bottom up version *)

   local
       exception DisjointSetFound
       val runDebug = ref false
       fun debug f = if (!runDebug) then f () else ()
   in
   fun isComplementEmpty (ta:tree_automaton) =
       let val Delta = #Rules ta
	   val Qf = #Final ta
	   val faSet = #Signature ta
	   val fas = ListMergeSort.sort (fn ((_,i),(_,j))=> Int.> (i,j))
					(FIS.listItems faSet)
	   fun printQ Q = LU.toStringCommaCurly Term.toString (TS.listItems Q)
	   fun printQs Qs = LU.toStringSpaceLnSquare printQ Qs
	   fun printQs2 Qs = LU.toStringCommaRound printQ Qs

           (* n個の状態集合のすべて組み合せ *)
	   fun getAllTuples 0 Qs = [[]]
	     | getAllTuples 1 Qs = L.map (fn x=> [x]) Qs
	     | getAllTuples n Qs = ListXProd.mapX (fn (x,ys) => x::ys) (Qs, getAllTuples (n-1) Qs)

           (* getAllTuples but newQs からの状態集合を1つは含む *)
	   fun getAllTuples' 0 (Qs,newQs,prevQs) = if null newQs 
						then [[]]  (* very first time *)
						else [] (* after second time *)
	     | getAllTuples' 1 (Qs,newQs,prevQs)= L.map (fn x=> [x]) newQs
	     | getAllTuples' n (Qs,newQs,prevQs)= 
	       (ListXProd.mapX (fn (x,ys) => x::ys) (newQs, getAllTuples (n-1) prevQs))
	       @ (ListXProd.mapX (fn (x,ys) => x::ys) (Qs, getAllTuples' (n-1) (Qs,newQs,prevQs)))

	   (*  Qs = newQs \cup prevQs, candQs \subseteq Qs
               Qs: 前回までの状態集合のリスト 
               newQs: 前回追加された状態集合のリスト 
               prevQs = Qs \setminus newQs
               candQs: Qsに今回追加中の状態集合も加えたリスト  *)
	   fun check0 (f,ar) (Qs,newQs,prevQs,candQs) = 
	       let val _ = debug (fn _ => print ("check0: " ^ (Fun.toString f) ^ "\n") )
		   val tuples = getAllTuples' ar (Qs,newQs,prevQs)
		   val _ = debug (fn _ => print "possible tuples: ")
		   val _ = debug (fn _ => print (LU.toStringCommaSquare printQs2 tuples))
		   val _ = debug (fn _ => print "\n")
		   fun getNewQ tmpQs Qs' =
		       let val Qcand =
			       TS.addList (TS.empty,
					   L.mapPartial 
					       (fn (l,r) => if Fun.equal (f,valOf (Term.funRootOfTerm l))
							    then
								let val args = Term.argsOfTerm l
								in if L.all (fn i=> TS.member (List.nth (Qs',i), 
											       List.nth (args,i)))
									    (L.tabulate (ar, fn x=>x))
								   then SOME r
								   else NONE
								end
							    else NONE)
					       Delta)
			   val _ = debug (fn _ => print ("candidate: " ^ (printQ Qcand) ^ "\n"))
		       in if LU.member' TS.equal Qcand tmpQs
			  then NONE
			  else if TS.isEmpty (TS.intersection (Qcand, Qf))
			  then (debug (fn _ => print ("containing no final states!\n")); 
				raise DisjointSetFound)
			  else SOME Qcand
		       end
	       in L.foldl (fn (Qs',tmpQs) => case getNewQ tmpQs Qs' of
						NONE => tmpQs
					      | SOME Q => (debug (fn _ => print ("added: " ^ (printQ Q) ^ "\n")); 
							   Q::tmpQs))
			  candQs 
			  tuples
	       end

           (* Qs = newQs U prevQs, 
	    newQs:前回で追加された状態集合のリスト *)
	   fun check (Qs,newQs,prevQs) = 
	       let val _ = debug (fn _ => print "current states:\n")
		   val _ = debug (fn _ => print (printQs Qs))
		   val Qs' = L.foldl (fn (far,candQs) => check0 far (Qs,newQs,prevQs,candQs))
					Qs
					fas
		   val newQs' = L.take (Qs', (L.length Qs') - (L.length Qs))
	       in if null newQs' (* saturated *)
		  then true
		  else check (Qs',newQs',Qs)
	       end

       in check ([],[],[])
	  handle DisjointSetFound=>false  (* 追加される状態集合で，Qfとの積が空のものがあれば empty *)
       end
   end (* local *)


   (* isSubseteq: bottom up version, see Note 2012 (5)*)
   local
       exception NonSubset
       exception SignatureNotSubseteq
       val runDebug = ref false
       fun debug f = if (!runDebug) then f () else ()
   in
   fun isSubseteq (ta0:tree_automaton, ta1:tree_automaton) =
       let val Delta0 = #Rules ta0
	   val Qf0 = #Final ta0
	   val Delta1 = #Rules ta1
	   val Sig0 = #Signature ta0
	   val Sig1 = #Signature ta1
	   val Qf1 = #Final ta1
	   val sig0 = ListMergeSort.sort (fn ((_,i),(_,j))=> Int.> (i,j))
						(FIS.listItems Sig0)
	   val sig1 = ListMergeSort.sort (fn ((_,i),(_,j))=> Int.> (i,j))
						(FIS.listItems Sig1)
	   val _ = if FIS.isSubset (Sig0,Sig1)
		   then ()
		   else raise SignatureNotSubseteq

	   fun printQ Q = LU.toStringCommaCurly Term.toString (TS.listItems Q)
	   fun printQs Qs = LU.toStringSpaceLnSquare printQ Qs
	   fun printQs2 Qs = LU.toStringCommaRound printQ Qs

           (* n個の状態集合のすべて組み合せ *)
	   fun getAllTuples 0 Qs = [[]]
	     | getAllTuples 1 Qs = L.map (fn x=> [x]) Qs
	     | getAllTuples n Qs = ListXProd.mapX (fn (x,ys) => x::ys) (Qs, getAllTuples (n-1) Qs)

           (* getAllTuples but newQs からの状態集合を1つは含む *)
	   fun getAllTuples' 0 (Qs,newQs,prevQs) = if null newQs 
						then [[]]  (* very first time *)
						else [] (* after second time *)
	     | getAllTuples' 1 (Qs,newQs,prevQs)= L.map (fn x=> [x]) newQs
	     | getAllTuples' n (Qs,newQs,prevQs)= 
	       (ListXProd.mapX (fn (x,ys) => x::ys) (newQs, getAllTuples (n-1) prevQs))
	       @ (ListXProd.mapX (fn (x,ys) => x::ys) (Qs, getAllTuples' (n-1) (Qs,newQs,prevQs)))

	   fun lookup Q [] = NONE
	     | lookup Q ((Q',Q0')::rest) = if TS.equal (Q,Q')
					   then SOME Q0'
					   else lookup Q rest
	   fun update (Q,Q0) [] = (debug (fn _ => print ("added: " ^ (printQ Q) 
					  ^ " |=> " ^ (printQ Q0) ^ "\n"));
				   [(Q,Q0)])
	     | update (Q,Q0) ((Q',Q0')::rest) = if TS.equal (Q,Q')
						then (debug (fn _ => print ("update: " ^ (printQ Q) 
							     ^ " |=> " ^ (printQ (TS.union (Q0,Q0'))) ^ "\n"));
						      (Q,TS.union (Q0,Q0'))::rest)
						else (Q',Q0')::(update (Q,Q0) rest)

	   (*  Qs = newQs \cup prevQs, candQs \subseteq Qs
               Qs: 前回までの状態集合のリスト 
               newQs: 前回追加された状態集合のリスト 
               prevQs = Qs \setminus newQs
               candQs: Qsに今回追加中の状態集合のリスト
               candAssoc: Q1からQ0への対応
	    *)
	   fun check0 (f,ar) (Qs,newQs,prevQs,candQs,candAssoc) = 
	       let val _ = debug (fn _ => print ("check0: " ^ (Fun.toString f) ^ "\n") )
		   val tuples = getAllTuples' ar (Qs,newQs,prevQs)
		   val _ = debug (fn _ => print "possible tuples: ")
		   val _ = debug (fn _ => print (LU.toStringCommaSquare printQs2 tuples))
		   val _ = debug (fn _ => print "\n")
		   fun getQcand Qs' Delta' = 
		       TS.addList (TS.empty,
				   L.mapPartial 
				       (fn (l,r) => if Fun.equal (f,valOf (Term.funRootOfTerm l))
						    then
							let val args = Term.argsOfTerm l
							in if L.all (fn i=> TS.member (List.nth (Qs',i), 
										       List.nth (args,i)))
								    (L.tabulate (ar, fn x=>x))
							   then SOME r
							   else NONE
							end
						    else NONE)
				       Delta')
		   fun getNewQ (tmpQs,tmpAssoc) Qs' =
		       let val Qcand = getQcand Qs' Delta1
			   val _ = debug (fn _ => print ("candidate: " ^ (printQ Qcand) ^ "\n"))
			   val Qs0' = L.map (fn Q'=> valOf (lookup Q' tmpAssoc)) Qs'
			   val Qcand0 = getQcand Qs0' Delta0
			   val _ = debug (fn _ => print ("corresponding Q in A0: " ^ (printQ Qcand0) ^ "\n"))
		       in if (LU.member' TS.equal Qcand Qs
			      orelse LU.member' TS.equal Qcand tmpQs)
			     andalso (debug (fn _ => 
						print ("prev: " ^ (printQ (valOf (lookup Qcand tmpAssoc))) ^ "\n"));
				      true)
			     andalso TS.isSubset (Qcand0, valOf (lookup Qcand tmpAssoc))
			  then (debug (fn _ => print "already added.\n"); NONE)
			  else if TS.isEmpty Qcand0 
			  then NONE
			  else if not (TS.isEmpty (TS.intersection (Qcand0, Qf0)))
				  andalso TS.isEmpty (TS.intersection (Qcand, Qf1))
			  then (debug (fn _ => print ("not subseteq!\n")); 
				raise NonSubset)
			  else SOME (Qcand,Qcand0)
		       end
	       in L.foldl (fn (Qs',(tmpQs,tmpAssoc)) => case getNewQ (tmpQs,tmpAssoc) Qs' of
						NONE => (tmpQs,tmpAssoc)
					      | SOME (Q,Q0) => (debug (fn _ => print ("added/updated: " 
										      ^ (printQ Q) ^ "\n")); 
								(if LU.member' TS.equal Q tmpQs
								 then tmpQs else Q::tmpQs, 
								 update (Q,Q0) tmpAssoc)))
			  (candQs,candAssoc)
			  tuples
	       end

           (* Qs = newQs U prevQs, 
	    newQs:前回で追加された状態集合のリスト
            assoc: Qset1 |-> Qset0 *)
	   fun check (Qs,newQs,prevQs,assoc) = 
	       let val _ = debug (fn _ => print "current states:\n")
		   val _ = debug (fn _ => print (printQs Qs))
		   val (newQs',assoc') = L.foldl (fn (far,(candQs,candAssoc)) => 
						  check0 far (Qs,newQs,prevQs,candQs,candAssoc))
					      ([],assoc)
					      sig1
		   val QsRev = LU.differenceByOne' TS.equal (Qs,newQs')
		   val Qs' = newQs' @ QsRev
	       in if null newQs' (* saturated *)
		  then true
		  else check (Qs',newQs',QsRev,assoc')
	       end

       in check ([],[],[],[])
	  handle NonSubset=>false
	       | SignatureNotSubseteq=>false  
       end
   end (* local *)

   (* isIntersectionEmpty: bottom up version *)
   fun isIntersectionEmpty_bot (ta0:tree_automaton, ta1:tree_automaton) =
       let val Delta0 = #Rules ta0
	   val Qf0 = #Final ta0
	   val Sig0 = #Signature ta0
           val Delta1 = #Rules ta1
	   val Qf1 = #Final ta1
	   val Sig1 = #Signature ta1
	   val faSet = FIS.intersection (Sig0,Sig1)
           fun filterRules Delta = L.filter (fn (l,r)=> FIS.member (faSet,
							       (valOf (Term.funRootOfTerm l),
								L.length (Term.argsOfTerm l))))
				       Delta
	   val Delta0com = filterRules Delta0
	   val Delta1com = filterRules Delta1

	   val fas = ListMergeSort.sort (fn ((_,i),(_,j))=> Int.> (i,j))
					(FIS.listItems faSet)

	   fun select f Q Delta = 
	       L.mapPartial
		   (fn (l,r) => if Fun.equal (valOf (Term.funRootOfTerm l), f)
				then let val ts = Term.argsOfTerm l
				     in if L.all (fn t => TS.member (Q,t)) ts
					then SOME (r::ts)
					else NONE
				     end
				else NONE)
		   Delta
	   fun check0 f (prevQ0,prevQ1,prevQQ) (tmpQ0,tmpQ1,tmpQQ) = 
	       let val qss0 = select f prevQ0 Delta0com
		   val qss1 = select f prevQ1 Delta1com
	       in ListXProd.foldX (fn (p::xs,q::ys,(Q0',Q1',QQ'))=> 
				       if LP.allEq (fn (x,y)=> TPS.member (prevQQ, (x,y))) (xs,ys)
				       then (TS.add (Q0',p),TS.add (Q1',q),TPS.add (QQ',(p,q)))
				       else (Q0',Q1',QQ'))
				  (qss0,qss1)
				  (tmpQ0,tmpQ1,tmpQQ)
	       end

	   fun check (Q0,Q1,QQ) = 
	       let val (Q0',Q1',QQ') = L.foldl (fn ((f,_),(tmpQ0,tmpQ1,tmpQQ)) => 
						   check0 f (Q0,Q1,QQ) (tmpQ0,tmpQ1,tmpQQ))
					       (Q0,Q1,QQ)
					       fas
	       in if TPS.isSubset (QQ',QQ)
		  then QQ'
		  else check (Q0',Q1',QQ')
	       end
       in L.all (fn (q0,q1)=> not (TS.member (Qf0,q0)) orelse not (TS.member (Qf1,q1)))
		(TPS.listItems (check (TS.empty,TS.empty,TPS.empty)))
       end

   (* isIntersectionEmpty: top down version *)
   fun isIntersectionEmpty (ta0:tree_automaton, ta1:tree_automaton) =
       let val Delta0 = #Rules ta0
	   val Qf0 = #Final ta0
	   val Sig0 = #Signature ta0

	   val Delta1 = #Rules ta1
	   val Qf1 = #Final ta1
	   val Sig1 = #Signature ta1

	   val faSet = FIS.intersection (Sig0,Sig1)
           fun filterRules Delta = L.filter (fn (l,r)=> FIS.member (faSet,
							    (valOf (Term.funRootOfTerm l),
							     L.length (Term.argsOfTerm l))))
				    Delta
	   val Delta0com = filterRules Delta0
	   val Delta1com = filterRules Delta1

	   (* val Delta0a = L.filter (fn (l,r) => not (LU.member' Term.equal r (Term.argsOfTerm l))) Delta0com *)
	   (* val Delta1a = L.filter (fn (l,r) => not (LU.member' Term.equal r (Term.argsOfTerm l))) Delta1com *)
	   val Delta0a = Delta0com
	   val Delta1a = Delta1com

	   val qfs = ListXProd.mapX (fn xy => xy) (TS.listItems Qf0, TS.listItems Qf1)
	   val Delta = ListXProd.foldX (fn (rule0,rule1,rs) => let val (l0,_) = rule0
								     val (l1,_) = rule1
								 in if Term.haveSameRoots (l0,l1)
								    then (rule0,rule1)::rs
								    else rs
								 end)
				       (Delta0a,Delta1a)
				       []

	   fun TPequal ((l0,r0),(l1,r1)) = Term.equal (l0,l1) andalso Term.equal (r0,r1)
(*
	   fun empCheck (qq,qqs) =
	       if LU.member' TP.equal qq qqs
	       then true
	       else L.all (fn ((l0,r0),(l1,r1)) => 
				 not (TP.equal (qq,(r0,r1)))
				 orelse
				 let val xs0 = Term.argsOfTerm l0
				     val xs1 = Term.argsOfTerm l1
				 in if LP.exists (fn (q0,q1) => LU.member' TP.equal (q0,q1) qqs) (xs0,xs1)
				    then true
				    else if L.all (fn qq' => not (empCheck (qq', qq::qqs)))
						  (LU.eliminateDuplication' TPequal (LP.zipEq (xs0,xs1)))
                                         (* LP.allEq (fn qq' => not (empCheck (qq', qq::qqs))) (xs0,xs1) *)
 				    then false
				    else true
				 end)
			  Delta
*)

	   fun empCheck ((q0,q1),qqs) =
	       if LU.member' TP.equal (q0,q1) qqs
	       then true
	       else let val Delta0b = L.filter (fn (_,r)=> Term.equal (r,q0)) Delta0a
			val Delta1b = L.filter (fn (_,r)=> Term.equal (r,q1)) Delta1a
			val DeltaU = ListXProd.foldX (fn (rule0,rule1,rs) => let val (l0,_) = rule0
										 val (l1,_) = rule1
									     in if Term.haveSameRoots (l0,l1)
										then (rule0,rule1)::rs
										else rs
									     end)
						     (Delta0b,Delta1b)
						     []
		    in
			L.all (fn ((l0,r0),(l1,r1)) => 
				  let val xs0 = Term.argsOfTerm l0
				      val xs1 = Term.argsOfTerm l1
				  in if LP.exists (fn (q0,q1) => LU.member' TP.equal (q0,q1) qqs) (xs0,xs1)
				     then true
				     else if L.all (fn qq' => not (empCheck (qq', (q0,q1)::qqs)))
						  (LU.eliminateDuplication' TPequal (LP.zipEq (xs0,xs1)))
                                         (* LP.allEq (fn qq' => not (empCheck (qq', qq::qqs))) (xs0,xs1) *)
 				     then false
				     else true
				 end)
			  DeltaU
		    end
       in L.all (fn qq => empCheck (qq,[])) qfs
       end

   (* L(A,q_0) \cap L(A,q_1) \neq emptyset ? *)
   fun haveEmptyIntersection (ta:tree_automaton) (q0,q1)  =
       let val Delta = #Rules ta
	   val Sig = #Signature ta
	   val Delta2 = L.filter (fn (l,r) => not (LU.member' Term.equal r (Term.argsOfTerm l))) Delta
	   val emptyStates = ref TS.empty
	   val nonEmptyStates = ref TS.empty

	   fun isEmpty (q,qs) = 
	       if LU.member' Term.equal q qs
	       then true
	       else L.all (fn (l,r) => 
                              not (Term.equal (q,r))
			      orelse
			      let val xs = Term.argsOfTerm l
			      in if L.exists (fn q'=> LU.member' Term.equal q' qs) xs
				 then true
				 else if L.all (fn q' => not (isEmpty (q', q::qs))) 
					       (LU.eliminateDuplication' Term.equal xs)
				 then false
				 else true
			      end)
			  Delta2

	   fun isEqualEq ((qa,qb),(qa',qb')) = 
	       (Term.equal (qa,qa') andalso Term.equal (qb,qb'))
	       orelse
	       (Term.equal (qa,qb') andalso Term.equal (qb,qa'))

	   fun isIntersectionEmpty (q0,q1) qqs = 
	       ((* print ("check " ^ (LU.toStringCommaCurly TP.toString qqs)
		       ^ "|- " ^ (Term.toString q0) ^ "-" ^ (Term.toString q1) ^ "\n"); *)
	       if Term.equal (q0,q1)
	       then if TS.member (!emptyStates,q0)
		    then true
		    else if TS.member (!nonEmptyStates,q0)
		    then false
		    else let val ans = isEmpty (q0,[])
			 in if ans then (emptyStates := TS.add (!emptyStates,q0);ans)
			    else (nonEmptyStates := TS.add (!nonEmptyStates,q0);ans)
			 end
	       else if LU.member' isEqualEq (q0,q1) qqs
	       then true
	       else let val Delta0 = L.filter (fn (_,r)=> Term.equal (r,q0)) Delta
			val Delta1 = L.filter (fn (_,r)=> Term.equal (r,q1)) Delta
			val DeltaU = ListXProd.foldX 
					 (fn (rule0,rule1,rs) => let val (l0,_) = rule0
								     val (l1,_) = rule1
								 in if Term.haveSameRoots (l0,l1)
								    then (rule0,rule1)::rs
								    else rs
								 end)
					 (Delta0,Delta1)
					 []
			val ans = L.all (fn ((l0,r0),(l1,r1)) => 
					    let val xs0 = Term.argsOfTerm l0
						val xs1 = Term.argsOfTerm l1
						(* val _ = print "<"
						val _ = print (Trs.prRule (l0,r0))
						val _ = print " & "
						val _ = print (Trs.prRule (l1,r1))
						val _ = print ">\n" *)
					    in if LP.exists (fn (qa,qb) => LU.member' isEqualEq (qa,qb) qqs)
							    (xs0,xs1)
					       then true
					       else if L.all (fn qq' => not (isIntersectionEmpty qq' ((q0,q1)::qqs)))
							     (LU.eliminateDuplication' isEqualEq (LP.zipEq (xs0,xs1)))
					             (* LP.allEq (fn qq' => not (isIntersectionEmpty qq' 
												    ((q0,q1)::qqs)))
							     (xs0,xs1) *)
					       then false
					       else true
					    end)
					DeltaU
		    in ans
		    end)
       in isIntersectionEmpty (q0,q1) []
       end

(*********
       val faSet2 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q2 = TS.addList (TS.empty, IOFotrs.rdTerms ["qnat","qzero","qone","qtwo","qeven","qodd"])
       val Qf2 = TS.addList (TS.empty, IOFotrs.rdTerms ["qodd"])   
       val Rules2 = IOFotrs.rdRules ["0 -> qnat",
				     "0 -> qeven",
				     "0 -> qzero",
				     "s(qnat) -> qnat",
				     "s(qeven) -> qodd",
				     "s(qodd) -> qeven",
				     "s(qzero) -> qone",
				     "s(qone) -> qtwo",
				     "s(qtwo) -> qzero"]
       val ta2 = { Signature = faSet2, States = Q2, Final = Qf2, Rules = Rules2 }:tree_automaton
       val _ = TimeUtil.profile (fn _=> print (Bool.toString (
					       haveEmptyIntersection ta2 (IOFotrs.rdTerm "qtwo",
									   IOFotrs.rdTerm "qodd"))),
				 "haveEmptyIntersection")

       val faSet3 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q3 = TS.addList (TS.empty, IOFotrs.rdTerms ["qnat","qeven","qodd"])
       val Qf3 = TS.addList (TS.empty, IOFotrs.rdTerms ["qodd"])   
       val Rules3 = IOFotrs.rdRules ["0 -> qnat",
				     "0 -> qeven",
				     "s(qnat) -> qnat",
				     "s(qeven) -> qodd",
				     "s(qodd) -> qeven",
				     "plus(qnat,qnat) -> qnat",
				     "plus(qeven,qeven) -> qeven",
				     "plus(qeven,qodd) -> qodd",
				     "plus(qodd,qnat) -> qodd", (* incorrect *)
				     "plus(qodd,qodd) -> qeven",
				     "plus(qnat,qnat) -> qnat"]
       val ta3 = { Signature = faSet3, States = Q3, Final = Qf3, Rules = Rules3 }:tree_automaton
       val _ = TimeUtil.profile (fn _=> print (Bool.toString (
					       haveEmptyIntersection ta3 (IOFotrs.rdTerm "qeven",
									   IOFotrs.rdTerm "qodd"))),
				 "haveEmptyIntersection")


       fun qnat n = let fun nat n = if n <= 0 then "0"
				    else ("s" ^ (nat (n-1)))
		    in "q" ^ (nat n)
		    end

       val _ = print "check intersection\n"
       val max = 3
       val faSet4 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q4 = TS.addList (TS.empty, IOFotrs.rdTerms (L.tabulate(max, fn n => qnat n)))
       val Qf4 = TS.addList (TS.empty, IOFotrs.rdTerms [ (qnat (max - 1)) ])
       val Rules4 = IOFotrs.rdRules 
			(["0 -> q0"]
			 @ (L.tabulate (max - 1, fn n => "s(" ^ (qnat n) ^ ") -> " ^ (qnat (n+1))))
			 @ (let val ns = L.tabulate (max,fn x=>x)
			    in List.mapPartial (fn x=>x)
					       (ListXProd.mapX (fn (n,m) => if n+m < max
									    then SOME ("plus(" ^ (qnat n) ^ "," 
										       ^ (qnat m) ^ ") -> " 
										       ^ (qnat (n+m)))
									    else NONE) (ns,ns))
			    end))

       val qf0 = IOFotrs.rdTerm (qnat (max - 2))
       val qf1 = IOFotrs.rdTerm (qnat (max - 1))

       val ta4a = { Signature = faSet4, States = Q4, Final = (TS.singleton qf0), Rules = Rules4 }:tree_automaton
       val ta4b = { Signature = faSet4, States = Q4, Final = (TS.singleton qf1), Rules = Rules4 }:tree_automaton

       val _ = print "start\n"

       val _ = TimeUtil.profile (fn _=> print (Bool.toString (haveEmptyIntersection ta4a (qf0,qf1))), "haveEmptyIntersection")

       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta4a,ta4b))),
				 "isIntersectionEmpty")

***********)

   (* common functions *)
   val anonymousSymbol  = "*"
   val anonymousTerm = Term.mkFunTerm (Fun.fromString anonymousSymbol,[],Sort.null)
   fun anonymize t = L.foldl (fn (pos,u) => valOf (Term.replaceSubterm u pos anonymousTerm))
			     t
			     (Term.varPositionsInTerm t)

   fun qOfTerm t = Term.mkFunTerm (Fun.fromString ("q_{" ^ (Term.toString (anonymize t)) 
						   ^ "}"),[],Sort.null)

   fun qOfNum n = Term.mkFunTerm (Fun.fromString ("q_" ^ (Int.toString n)
						   ^ ""),[],Sort.null)

   val anonymousState = qOfTerm anonymousTerm


   fun makeRuleForTerm t = 
       if Term.isVar t 
       then (print "Error: makeRuleForTerm in tree_automata.sml\n"; raise Error)
       else let val f = valOf (Term.funRootOfTerm t)
		val args = L.map qOfTerm (Term.argsOfTerm t)
	    in (Term.mkFunTerm (f,args,Sort.null),qOfTerm t)
	    end

   fun rulesForAcceptingAllTerms faSet =
       L.map (fn (f,ar)=> (Term.mkFunTerm (f,L.tabulate(ar,fn x=>anonymousState),Sort.null),
			   anonymousState))
	     (FIS.listItems faSet)

   fun taAcceptingAllTerms faSet =
       { Signature = faSet,
	 States = TS.singleton anonymousState,
	 Final = TS.singleton anonymousState,
	 Rules = rulesForAcceptingAllTerms faSet }: tree_automaton

   local
       exception allInstances
   in
   fun taAcceptingGroundInstancesOfLinearTermSet faSet termSet =
      let 
	  val terms0 = TS.listItems termSet
	  val _ = if L.exists Term.isVar terms0
		  then raise allInstances
		  else ()

	  fun deleteInstances [] = []
	    | deleteInstances (t::ts) = 
	      let val ts0 = deleteInstances ts
		  val ts1 = L.filter (fn u => not (isSome (Subst.match t u))) ts0
	      in if L.exists (fn u => isSome (Subst.match u t)) ts1
		 then ts1
		 else t::ts1
	      end
	  val terms = deleteInstances terms0
	  (* val _ = print (LU.toStringCommaCurly Term.toString terms0) *)
	  (* val _ = print (LU.toStringCommaCurly Term.toString terms) *)

	  val isVariableContained = L.exists (fn t => not (null (Term.varListInTerm t))) terms

	  val anonyterms = L.map anonymize terms
	  (* val _ = print (LU.toStringCommaCurly Term.toString anonyterms) *)
	  val properSubtermSet = 
	      TS.difference (L.foldl (fn (t,tset) => TS.union (tset, Term.setOfProperSubterms t))
				 TS.empty
				 anonyterms,
			     TS.singleton anonymousTerm)
	  val properSubterms = TS.listItems properSubtermSet
	  (* val _ = print (LU.toStringCommaCurly Term.toString properSubterms) *)

	  val Delta0 = L.map makeRuleForTerm properSubterms

	  val qfinal = qOfTerm (Term.mkFunTerm (Fun.fromString "**",[],Sort.null))
	  fun makeRule1 t = let val f = valOf (Term.funRootOfTerm t)
				val args = L.map qOfTerm (Term.argsOfTerm t)
			     in (Term.mkFunTerm (f,args,Sort.null), qfinal)
			     end
	  val Delta1 = L.map makeRule1 anonyterms
	  val Q = TS.addList (TS.singleton qfinal, L.map qOfTerm properSubterms)
      in if isVariableContained
	 then { Signature = faSet,
		States = TS.add (Q, anonymousState),
		Final = TS.singleton qfinal,
		Rules = Delta0 @ Delta1 @ (#Rules (taAcceptingAllTerms faSet))  }:tree_automaton
	 else { Signature = faSet,
		States = Q,
		Final = TS.singleton qfinal,
		Rules = Delta0 @ Delta1  }:tree_automaton
      end
       handle allInstances => taAcceptingAllTerms faSet

   end (* local *)

   fun taAcceptingGroundInstancesOfLinearTerm faSet term =
      taAcceptingGroundInstancesOfLinearTermSet faSet (TS.singleton term)


   (* rules: lhs should be non-variable *)
   (* rules should be linear, fresh variable can occur in rhs *)
   (* term: should be ground *)
   local
     (*  val runDebug = ref false *)
     (*  fun debug f = if (!runDebug) then f () else () *)
   in
   fun degrowingCompletion faSet terms rules = 
       let val _ = debug (fn _ => print ("\n*** degrowing completion ***\n"))
	   val _ = debug (fn _ => print ("Terms: " ^ (LU.toStringCommaCurly Term.toString terms) ^ "\n"))
	   val _ = debug (fn _ => print ("Rules:\n" ^ (Trs.prRules rules)))

	   (* val termSet = TS.addList (TS.addList (TS.empty, terms), L.map (fn (_,r) => r) rules) *)
	   (* val properSubtermSet = *)
	   (*     TS.difference (L.foldl (fn (t,tset) => TS.union (tset, Term.setOfProperSubterms *)
	   (* 								  (anonymize t))) *)
	   (* 			      TS.empty *)
	   (* 			      (TS.listItems termSet), *)
	   (* 		      TS.singleton anonymousTerm) *)
	   (* val properSubterms = TS.listItems properSubtermSet *)

	   (* fun newsubsInRule (l,r) = *)
	   (*     let val news = LU.differenceByAll' Term.equal (Term.nonVarProperSubterms r, *)
	   (* 						      Term.nonVarSubterms l)  *)
	   (*     in L.map anonymize news *)
	   (*     end *)
           (*  bug? fix 2014/4/18 *)
	   fun newsubsInRule (l,r) =
	       let val nvargs = L.filter Term.isFun (Term.argsOfTerm r)
		   val nvargs2 = LU.differenceByAll' Term.equal (nvargs, Term.nonVarSubterms l)
		   val news = LU.mapAppend Term.nonVarSubterms nvargs2
	       in L.map anonymize news
	       end

	   val properSubtermSet = let val ts = L.map anonymize terms
				      val ts2 = LU.mapAppend Term.properSubterms ts
				  in L.foldl (fn (rule,tset) => TS.addList (tset, newsubsInRule rule))
					     (TS.addList (TS.empty, ts2))
	   				     rules
				  end

	   val properSubterms = TS.listItems properSubtermSet

           (* bug fix 2014/2/6*)
	   (* val instantiationNotNeeded = (* Trs.areDegrowingRules rules *) *)
	   (*     not (FunSet.member (Term.funSetInTerms properSubterms, Fun.fromString anonymousSymbol)) *)

	   val instantiationNotNeeded = (* Trs.areDegrowingRules rules *)
	       Trs.areRewriteRules rules  (* need V(r) \subseteq V(l) *)
	       andalso
	       not (FunSet.member (Term.funSetInTerms properSubterms, Fun.fromString anonymousSymbol))

	   val _ = debug (fn _=> print ((LU.toStringCommaCurly Term.toString properSubterms)  ^ "\n"))

	   val qfs = L.tabulate (L.length terms, fn n => qOfNum n)
	   val Qf = TS.addList(TS.empty, qfs)

	   val rules0 = if instantiationNotNeeded
			then []
			else rulesForAcceptingAllTerms faSet
	   val rules1 = L.map makeRuleForTerm properSubterms
	   val rules2 = if L.exists Term.isVar terms
			then (debug (fn _=> print "degrowingCompletion in tree_automata.sml\n");raise Error)
			else L.map (fn n => let val t = L.nth (terms, n)
					       val q = L.nth (qfs, n)
					   in
					       (Term.mkFunTerm (valOf (Term.funRootOfTerm t),
							       L.map qOfTerm (Term.argsOfTerm t),
							       Sort.null),
						q)
					   end)
				  (L.tabulate (L.length terms, fn n => n))

          (* (** terms が rules1 を作るのに使った項を含むときは，その qOfTerm を q_i に置きかえるがよいのでは？ **) *)
          (*  => あとで，qOfTerm を使うかもしれないのでダメ! *)
	  (*  val rules1' = let val corresp  = LP.zip (L.map qOfTerm terms,  qfs) *)
	  (* 		     fun repl [] t = t *)
	  (* 		       | repl ((u,q)::xs) t = if Term.equal(t,u) then q  *)
	  (* 					      else repl xs t *)
	  (* 		 in L.map (fn (l,r) =>  *)
	  (* 			       (Term.mkFunTerm (valOf (Term.funRootOfTerm l),  *)
	  (* 						L.map (repl corresp) (Term.argsOfTerm l), *)
	  (* 						Sort.null), *)
	  (* 				repl corresp r)) rules1 *)
	  (* 		 end *)
          (* val Delta0 = Trs.deleteIdenticalRules (rules0 @ rules1' @ rules2) *)

	   val Delta0 = rules0 @ rules1 @ rules2

	   val _ = debug (fn _ => print ("Delta0 = \n" ^ (Trs.prRules Delta0)))

	   val Q = TS.addList (TS.empty,
			       if instantiationNotNeeded 
			       then qfs @ (L.map qOfTerm properSubterms)
			       else anonymousState::(qfs @ (L.map qOfTerm properSubterms)))

	   val fas = ListMergeSort.sort (fn ((_,i),(_,j))=> Int.> (i,j))
						(FIS.listItems faSet)

	   fun oneStep0 (l,r) theta Delta =
	       let val largs = Term.argsOfTerm l
		   val lvarTermSet = Term.varTermSetInTerm l
		   val lsubtermSet = Term.setOfSubterms l
		   val ss0 =  L.map (fn t => Subst.applySubst theta t) largs
		   val sss1 = List.foldr (fn (u,vs) => ListXProd.mapX (fn (x,xs)=>x::xs)
								      (reachableStates Delta u,vs))
					 [[]] 
					 ss0
		   val f = valOf (Term.funRootOfTerm l)
		   val ss1 = L.map (fn us => Term.mkFunTerm (f,us,Sort.null)) sss1
		   (* 左辺-*->f(q_1,...,qk)なる項 *)
		   val rstates = let val rtheta = Subst.applySubst theta r
				 in if Term.isGroundTerm rtheta
				    then reachableStates Delta rtheta
				    else []
				 end
	       in if Term.isVar r
		  then let val q' = if TS.member (lvarTermSet,r)
				    then Subst.applySubst theta r
				    else anonymousState
		       in LU.mapAppend (fn (u',v') => 
					    if not (Term.equal (v',anonymousState))
					       andalso LU.member' Term.equal u' ss1
					       andalso not (LU.member' Term.equal v' rstates) (* added *)
					    then L.mapPartial (fn (l',w') =>
								  if Term.equal (q',w')
								  then SOME (l',v')
								  else NONE)
							      Delta
					    else [])
				       Delta
		       end
		  else
		      let val ts0 =  L.map (fn u=> if TS.member (lvarTermSet,u)
						   then Subst.applySubst theta u
						   else if TS.member (lsubtermSet,u) (* new 2013/4/29 *)
						   then let val cand = reachableStates Delta (Subst.applySubst theta u)
							in if not (null cand)
							   then hd cand 
							   else qOfTerm u
							end
						   else qOfTerm u)
					   (Term.argsOfTerm r)
			  val g = valOf (Term.funRootOfTerm r)
			  val u = Term.mkFunTerm (g,ts0,Sort.null)
		      in L.mapPartial (fn (u',v)=> if not (Term.equal (v,anonymousState))
						      andalso LU.member' Term.equal u' ss1
						      andalso not (LU.member' Term.equal v rstates) (* added *)
						   then SOME (u,v)
						   else NONE)
				  Delta
		  end
	       end

	   fun allsubst vs qs = L.foldl
				    (fn (v,subs)=> LU.mapAppend (fn theta=> L.map (fn q=> VM.insert (theta,v,q)) qs)
								subs)
				    [VM.empty]
				    vs

	   fun oneStep (l,r) Delta =
	       let val subs = allsubst (VS.listItems (Term.varSetInTerm l)) (TS.listItems Q)
		   val new = LU.mapAppend (fn theta => oneStep0 (l,r) theta Delta) subs
	       in new
	       end

	   fun doCompletion DeltaSet =
	       let val Delta = TPS.listItems DeltaSet
		   val new = LU.mapAppend (fn lr=> oneStep lr Delta) rules 
		   val _ = debug (fn _ => print "added rules:\n")
		   val _ = debug (fn _ => print (Trs.prRules (LU.differenceByAll' TP.equal (new,Delta))))
		   val DeltaSet' = TPS.addList (DeltaSet,new)
	       in if TPS.isSubset (DeltaSet', DeltaSet)
		  then TPS.listItems DeltaSet
		  else doCompletion DeltaSet'
	       end
       in { Signature = faSet,
            States = Q,
	    Final = Qf,
	    Rules = doCompletion (TPS.addList (TPS.empty, Delta0))  }:tree_automaton
       end
   end (* of local *)


(***
   val faSet0 = Term.funAritySetInTerm (IOFotrs.rdTerm "f(g(a,?y),h(b))")
   val t0 = IOFotrs.rdTerm "f(g(a,b),h(h(b)))"
(*   val rules0 = IOFotrs.rdRules ["h(?x) -> g(?x,b)", "g(a,?x) -> h(?x,g(?y))"] *)
   val rules0 = IOFotrs.rdRules ["h(?x) -> ?y"]
   val _ = TimeUtil.profile (fn _=> print (toString (degrowingCompletion faSet0 [t0] rules0)),
			     "degrowingCompletion")
***)

(******
   val faSet1 = Term.funAritySetInTerm (IOFotrs.rdTerm "F(G(C),A)")
   val t1a = IOFotrs.rdTerm "A"
(*   val rules1 = IOFotrs.rdRules ["F(?x,?x) -> A","G(?x) -> F(?x,?y)", "C -> G(C)"] 
                  F(q_a,q_{*}) -> r \in Delta makes the result incorrect...
*)
   val rules1 = IOFotrs.rdRules ["F(?x,?x) -> A","G(?x) -> F(?x,?x)", "C -> G(C)"]
   val ta1a = degrowingCompletion faSet1 [t1a] rules1

(*   val t1b = IOFotrs.rdTerm "F(A,G(A))" *)
   val t1b = IOFotrs.rdTerm "G(C)"
   val ta1b = degrowingCompletion faSet1 [t1b] rules1

   val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta1a,ta1b))),
			     "isIntersectionEmpty")


   val faSet0 = FIS.addList(FIS.empty, 
			    [(Fun.fromString "a", 0),
			     (Fun.fromString "b", 0),
			     (Fun.fromString "f", 1)])
   val t0a = IOFotrs.rdTerm "b"
   val rules0 = IOFotrs.rdRules ["a -> b", "a -> f(a)", "b -> f(f(b))"]
   val ta0a = degrowingCompletion faSet0 [t0a] rules0
   val _ = print (toString ta0a)

   val t0b = IOFotrs.rdTerm "f(b)"
   val ta0b = degrowingCompletion faSet0 [t0b] rules0
   val _ = print (toString ta0b)

   val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta0a,ta0b))),
			     "isIntersectionEmpty")
****)
(*
   val faSet0 = FIS.addList(FIS.empty, 
			    [(Fun.fromString "a", 0),
			     (Fun.fromString "b", 0),
			     (Fun.fromString "c", 0),
			     (Fun.fromString "d", 0),
			     (Fun.fromString "h", 1)])
   val t0a = IOFotrs.rdTerm "h(a)"
   val rules0 = IOFotrs.rdRules ["b -> a", "b -> c", "c -> h(b)", "c -> d", "a -> h(a)", "d -> h(d)"]
   val ta0a = degrowingCompletion faSet0 [t0a] rules0
   val _ = print (toString ta0a)

   val t0b = IOFotrs.rdTerm "h(d)"
   val ta0b = degrowingCompletion faSet0 [t0b] rules0
   val _ = print (toString ta0b)

   val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta0a,ta0b))),
			     "isIntersectionEmpty")
*)

   fun test () = let
       val testMember = true
       val testIsEmpty = true
       val testIsComplentEmpty = true
       val testIsSubseteq = true
       val testIsIntersectionEmpty = true

       val faSet0 = Term.funAritySetInTerm (IOFotrs.rdTerm "f(g(a,?y),h(b))")
       val ta0a = taAcceptingGroundInstancesOfLinearTerm faSet0 (IOFotrs.rdTerm "f(g(a,?y),h(b))")
       val _ = if false then print (toString ta0a) else ()

       val t0 = IOFotrs.rdTerm "f(g(a,h(h(a))),h(g(b,h(b))))"

       val _ = if testMember
	       then let 
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (member t0 ta0a)), "member")
		   in () end
	       else ()

       val ta0b = taAcceptingGroundInstancesOfLinearTerm faSet0 (IOFotrs.rdTerm "f(g(a,b),h(b))")
       val _ = if true then print (toString ta0b) else ()

       val ta0b2 = degrowingCompletion faSet0 [(IOFotrs.rdTerm "f(g(a,b),h(b))")]
				       (IOFotrs.rdRules ["h(?x) -> g(?x,?y)"])
       val _ = if true then print (toString ta0b2) else ()

       val _ = if testIsIntersectionEmpty 
	       then let 
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty_bot (ta0a,ta0b))),
						 "isIntersectionEmpty_bot")
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta0a,ta0b))),
						 "isIntersectionEmpty")
		    in () end
	       else ()
		    
       val _ = if testIsSubseteq
	       then let 
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isSubseteq (ta0a,ta0b))), "isSubseteq")
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isSubseteq (ta0b,ta0a))), "isSubseteq")
		   in () end
	       else ()

       val _ = if false 
	       then print (toString (taAcceptingAllTerms
					 (Term.funAritySetInTerm (IOFotrs.rdTerm "f(g(a,?y),h(b))"))))
	       else ()

       val ta0c = taAcceptingGroundInstancesOfLinearTermSet
		      (Term.funAritySetInTerm (IOFotrs.rdTerm "f(g(a,?y),h(b))"))
		      (TS.addList (TS.empty, IOFotrs.rdTerms ["f(g(a,?y),h(?x))", "h(b)", "?x"]))

       val _ = if false then print (toString ta0c) else ()

       val faSet1 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q1 = TS.addList (TS.empty, IOFotrs.rdTerms ["qnat"])
       val Qf1 = TS.addList (TS.empty, IOFotrs.rdTerms ["qnat"])
       val Rules1 = IOFotrs.rdRules ["0 -> qnat",
				     "s(qnat) -> qnat",
				     "plus(qnat,qnat) -> qnat"]
       val ta1 = { Signature = faSet1, States = Q1, Final = Qf1, Rules = Rules1 }:tree_automaton
       val term1 = IOFotrs.rdTerm ("plus(s(s(s(0))),s(s(0)))")

       val _ = if testMember 
	       then TimeUtil.profile (fn _=> member_bot term1 ta1, "member_bot")
	       else true

       val _ = if testIsComplentEmpty 
	       then TimeUtil.profile (fn _=> print (Bool.toString (isComplementEmpty ta1)), "isComplementEmpty")
	       else ()

       val faSet2 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q2 = TS.addList (TS.empty, IOFotrs.rdTerms ["qz","qsz","qssz"])
       val Qf2 = TS.addList (TS.empty, IOFotrs.rdTerms ["qz","qsz","qssz"])
       (* val Qf2 = TS.addList (TS.empty, IOFotrs.rdTerms ["qssz"]) *)
       (*   val Qf2 = TS.addList (TS.empty, IOFotrs.rdTerms ["qsssz"]) *)
       val Rules2 = IOFotrs.rdRules ["0 -> qz",
				     "s(qz) -> qsz",
				     "s(qsz) -> qssz",
				     "plus(qz,qz) -> qz",
				     "plus(qz,qsz) -> qsz",
				     "plus(qz,qssz) -> qssz",
				     "plus(qsz,qz) -> qsz",
				     "plus(qsz,qsz) -> qssz",
				     "plus(qssz,qz) -> qssz" ]
       val ta2 = { Signature = faSet2, States = Q2, Final = Qf2, Rules = Rules2 }:tree_automaton
       val term2 = IOFotrs.rdTerm ("plus(s(s(s(0))),s(s(0)))")

       val _ = if testMember 
	       then TimeUtil.profile (fn _=> member_bot term2 ta2, "member_bot")
	       else true

       val _ = if testIsEmpty
	       then let val _ = TimeUtil.profile (fn _=> print (Bool.toString (isEmpty_bot ta2)), "isEmpty")
			val _ = TimeUtil.profile (fn _=> print (Bool.toString (isEmpty ta2)), "isEmpty")
		    in ()
		    end
	       else ()

       val _ = if testIsComplentEmpty 
	       then TimeUtil.profile (fn _=> print (Bool.toString (isComplementEmpty ta2)), "isComplementEmpty")
	       else ()

       val _ = if testIsSubseteq
	       then
		   let val _ = print ("A0:" ^ (toString ta1))
		       val _ = print ("A1:" ^ (toString ta2))
		       val _ = print "check A0 \\subseteq A1 \n" 
		   in TimeUtil.profile (fn _=> print (Bool.toString (isSubseteq (ta1,ta2))), "isSubseteq")
		   end
	       else ()

       val _ = if testIsSubseteq
	       then let val _ = print ("A0:" ^ (toString ta2))
			val _ = print ("A1:" ^ (toString ta1))
			val _ = print "check A0 \\subseteq A1 \n" 
		    in TimeUtil.profile (fn _=> print (Bool.toString (isSubseteq (ta2,ta1))), "isSubseteq")
		    end
	       else ()

       val _ = if testIsIntersectionEmpty 
	       then let 
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty_bot (ta1,ta2))),
						 "isIntersectionEmpty_bot")
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta1,ta2))),
						 "isIntersectionEmpty")
		   in () end
	       else ()

       val faSet3 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q3 = TS.addList (TS.empty, IOFotrs.rdTerms ["qnat","qeven","qodd","qdummy"])
       (*   val Qf3 = TS.addList (TS.empty, IOFotrs.rdTerms ["qeven"])   *)
       val Qf3 = TS.addList (TS.empty, IOFotrs.rdTerms ["qodd"])   
       (*   val Qf3 = TS.addList (TS.empty, IOFotrs.rdTerms ["qeven","qodd"]) *)
       val Rules3 = IOFotrs.rdRules ["0 -> qnat",
				     "0 -> qeven",
				     "s(qnat) -> qnat",
				     "s(qeven) -> qodd",
				     "s(qodd) -> qeven",
				     "plus(qnat,qnat) -> qnat",
				     "plus(qeven,qeven) -> qeven",
				     "plus(qeven,qodd) -> qodd",
				     "plus(qodd,qeven) -> qodd",
				     "plus(qodd,qodd) -> qeven",
				     "plus(qnat,qnat) -> qnat"]
       val ta3 = { Signature = faSet3, States = Q3, Final = Qf3, Rules = Rules3 }:tree_automaton
       fun nat n = if n <= 0 then "0"
		   else ("s(" ^ (nat (n-1)) ^ ")")
       (*   val term3 = IOFotrs.rdTerm ("plus(" ^ (nat 20) ^ "," ^ (nat 21) ^ ")") *)
       val _ = if testIsComplentEmpty 
	       then TimeUtil.profile (fn _=> print (Bool.toString (isComplementEmpty ta3)), "isComplementEmpty")
	       else ()

       val _ = if testIsSubseteq
	       then
		   let val _ = print ("A0:" ^ (toString ta2))
		       val _ = print ("A1:" ^ (toString ta3))
		       val _ = print "check A0 \\subseteq A1 \n" 
		   in TimeUtil.profile (fn _=> print (Bool.toString (isSubseteq (ta2,ta3))), "isSubseteq")
		   end
	       else ()

       val _ = if testIsSubseteq
	       then let val _ = print ("A0:" ^ (toString ta3))
			val _ = print ("A1:" ^ (toString ta2))
			val _ = print "check A0 \\subseteq A1 \n" 
		    in TimeUtil.profile (fn _=> print (Bool.toString (isSubseteq (ta3,ta2))), "isSubseteq")
		    end
	       else ()

       val _ = if testIsIntersectionEmpty 
	       then let 
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty_bot (ta2,ta3))),
						 "isIntersectionEmpty_bot")
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta2,ta3))),
						 "isIntersectionEmpty")
		   in () end
	       else ()


       fun qnat n = let fun nat n = if n <= 0 then "0"
				    else ("s" ^ (nat (n-1)))
		    in "q" ^ (nat n)
		    end
       val max = 1 (* 300 *) 
       val faSet4 = Term.funAritySetInTerm (IOFotrs.rdTerm "plus(s(0),?y)")
       val Q4 = TS.addList (TS.empty, IOFotrs.rdTerms (L.tabulate(max, fn n => qnat n)))
       val Qf4 = TS.addList (TS.empty, IOFotrs.rdTerms [ (qnat (max - 1)) ])
       val Rules4 = IOFotrs.rdRules 
			(["0 -> q0"]
			 @ (L.tabulate (max - 1, fn n => "s(" ^ (qnat n) ^ ") -> " ^ (qnat (n+1))))
			 @ (let val ns = L.tabulate (max,fn x=>x)
			    in List.mapPartial (fn x=>x)
					       (ListXProd.mapX (fn (n,m) => if n+m < max
									    then SOME ("plus(" ^ (qnat n) ^ "," 
										       ^ (qnat m) ^ ") -> " 
										       ^ (qnat (n+m)))
									    else NONE) (ns,ns))
			    end))

       val ta4 = { Signature = faSet4, States = Q4, Final = Qf4, Rules = Rules4 }:tree_automaton
       (* val _ = print (toString ta4) *)
       val Qf4' = TS.addList (TS.empty, IOFotrs.rdTerms [ (qnat max) ])
       val ta4' = { Signature = faSet4, States = Q4, Final = Qf4', Rules = Rules4 }:tree_automaton

       val _ = if testIsEmpty 
	       then let val _ = TimeUtil.profile (fn _=> print (Bool.toString (isEmpty_bot ta4)), "isEmpty")
			val _ = TimeUtil.profile (fn _=> print (Bool.toString (isEmpty ta4)), "isEmpty")
			val _ = TimeUtil.profile (fn _=> print (Bool.toString (isEmpty_bot ta4')), "isEmpty")
			val _ = TimeUtil.profile (fn _=> print (Bool.toString (isEmpty ta4')), "isEmpty")
		    in ()
		    end
	       else ()

       val _ = if testIsComplentEmpty 
	       then (print "now starting\n";
		     TimeUtil.profile (fn _=> print (Bool.toString (isComplementEmpty ta4)), "isComplementEmpty"))
	       else ()

       val _ = if testIsIntersectionEmpty 
	       then let 
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty_bot (ta3,ta4))),
						 "isIntersectionEmpty_bot")
		       val _ = TimeUtil.profile (fn _=> print (Bool.toString (isIntersectionEmpty (ta3,ta4))),
						 "isIntersectionEmpty")
		   in () end
	       else ()

   in ()
   end

   (* val _ =  test () *)


   end (* of local *)

end;


