(******************************************************************************
 * Copyright (c) 2014-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/term_rewriting/trs.sml
 * description: term rewriting systems
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature TRS = 
   sig

   type trs = { FSymSet:FunSet.set, 
		DSymSet:FunSet.set,
		CSymSet:FunSet.set,
		VarCond:bool,    (* Does satisfy l \notin V, V(r) \subseteq (l) ? *)
		Rules: (Term.term * Term.term) list }

   type rule = Term.term * Term.term
   type rules = rule list
   type equation =  Term.term * Term.term
   type equations = equation list

   val prRule: (Term.term * Term.term) -> string
   val prRules: (Term.term * Term.term) list -> string
   val prRuleWithVarSort: (Term.term * Term.term) -> string
   val prRulesWithVarSort: (Term.term * Term.term) list -> string
   val mkRule: (Term.term * Term.term) -> (Term.term * Term.term)
   val prEqs: (Term.term * Term.term) list -> string
   val prEq: (Term.term * Term.term) -> string
   val prRulesInTpdb: TextIO.outstream -> (Term.term * Term.term) list -> unit
   val prRelativeRulesInTpdb: TextIO.outstream -> 
			      ((Term.term * Term.term) list * (Term.term * Term.term) list) 
			      -> unit
   val prRuleInProofTree: (Term.term * Term.term) -> unit -> string
   val prRulesInProofTree: (Term.term * Term.term) list -> unit -> string
   val prTrsInProofTree: (Term.term * Term.term) list -> unit -> string
   val prEquationsInProofTree: (Term.term * Term.term) list -> unit -> string

   val sortSetInRule: (Term.term * Term.term) -> SortSet.set
   val sortSetInRules: (Term.term * Term.term) list -> SortSet.set

   val varSetInRule: (Term.term * Term.term) -> VarSet.set
   val varSetInRules: (Term.term * Term.term) list -> VarSet.set

   val ruleSize: (Term.term * Term.term) -> int

   val funSetInRule: (Term.term * Term.term) -> FunSet.set
   val funSetInRules: (Term.term * Term.term) list -> FunSet.set

   val fdcSetInRules: (Term.term * Term.term) list -> FunSet.set * FunSet.set * FunSet.set * bool
       (* function symbols , defined symbols, constructor symbols, variable condition *)
   val rulesToTrs: (Term.term * Term.term) list -> trs
   val checkVarCond: (Term.term * Term.term) list -> bool

   val dfunArityOfRule: (Term.term * Term.term) -> (Fun.ord_key * int) option
   val dfunAritySetOfRules: (Term.term * Term.term) list -> FunIntSet.set

   val funAritySetInRule: (Term.term * Term.term) -> FunIntSet.set
   val funAritySetInRules: (Term.term * Term.term) list -> FunIntSet.set

   val funArityMapInRule: (Term.term * Term.term) -> int FunMap.map
   val funArityMapInRules: (Term.term * Term.term) list -> int FunMap.map

   val prFunArityInProofTree: Fun.ord_key * int -> unit -> string
   val prFunArityListInProofTree: (Fun.ord_key * int) list -> unit -> string

(*   val isWellSortedRule: (Term.term * Term.term) -> Term.sort_key option *)
   val sortOfRule: (Term.term * Term.term) -> Term.sort_key option 
   val signatureInRule: (Term.term * Term.term) -> (Term.sort_key list * Term.sort_key) FunMap.map
   val signatureInRules: (Term.term * Term.term) list -> (Term.sort_key list * Term.sort_key) FunMap.map

   val attachSortToRule: Term.decl list -> Term.term * Term.term -> (Term.term * Term.term) option
   val attachSortToRules: Term.decl list -> (Term.term * Term.term) list 
			  -> (Term.term * Term.term) list option
   val attachSortToRuleWithEnv: Term.decl list -> Term.sort_key VarMap.map -> Term.term * Term.term -> (Term.term * Term.term) option
   val attachSortToRulesWithEnv: Term.decl list -> Term.sort_key VarMap.map -> (Term.term * Term.term) list 
				 -> (TermKey.term * Term.term) list option
   val addFunSortToRule: Term.decl list -> Term.term * Term.term -> (Term.term * Term.term) option
   val addFunSortToRules: Term.decl list -> (Term.term * Term.term) list 
			  -> (Term.term * Term.term) list option

   val isRewriteRule : Term.term * Term.term -> bool
   val areRewriteRules : (Term.term * Term.term) list -> bool
   val isLinearRule : Term.term * Term.term -> bool
   val areLinearRules : (Term.term * Term.term) list -> bool
   val isLeftLinearRule : Term.term * Term.term -> bool
   val areLeftLinearRules : (Term.term * Term.term) list -> bool
   val isRightLinearRule : Term.term * Term.term -> bool
   val areRightLinearRules : (Term.term * Term.term) list -> bool
   val isNonDuplicatingRule : Term.term * Term.term -> bool
   val areNonDuplicatingRules : (Term.term * Term.term) list -> bool
   val isNonErasingRule : Term.term * Term.term -> bool
   val areNonErasingRules : (Term.term * Term.term) list -> bool								    

   val renameRules : (Term.term * Term.term) list -> (Term.term * Term.term) list
   val renameRuleDisjointFrom: Var.ord_key list -> Term.term * Term.term -> Term.term * Term.term
   val renameRuleSet : ((Term.term * Term.term) list) * ((Term.term * Term.term) list)
                       -> ((Term.term * Term.term) list) * ((Term.term * Term.term) list)

   val normalizeVarNameRule: (Term.term * Term.term) -> (Term.term * Term.term * Subst.subst)
   val deleteIdenticalRules: (Term.term * Term.term) list 
			     -> (Term.term * Term.term) list

   val isBidirectionalRule: Term.term * Term.term -> bool
   val areBidirectionalRules: (Term.term * Term.term) list -> bool
   val isDegrowingRule: Term.term * Term.term -> bool
   val areDegrowingRules: (Term.term * Term.term) list -> bool

   val isShallowRule: Term.term * Term.term -> bool
   val areShallowRules: (Term.term * Term.term) list -> bool
   val isCollapsingTrs: (Term.term * Term.term) list -> bool

end;

structure Trs : TRS = 
   struct

   type trs = { FSymSet:FunSet.set, 
		DSymSet:FunSet.set,
		CSymSet:FunSet.set,
		VarCond:bool,    (* Does satisfy l \notin V, V(r) \subseteq (l) ? *)
		Rules: (Term.term * Term.term) list }

   type rule = Term.term * Term.term
   type rules = rule list
   type equation =  Term.term * Term.term
   type equations = equation list


   local 
       open Term
       open Subst
       structure CU = CertifyUtil
       structure FIS = FunIntSet
       structure FS = FunSet
       structure FM = FunMap
       structure L = List
       structure LU = ListUtil
       structure LP = ListPair
       structure SS = SortSet
       structure TS = TermSet
       structure VM = VarMap
       structure VS = VarSet
       fun mapAppend f xs = List.foldr (fn (x,ys) => List.@(f x, ys)) [] xs
   in

   fun prRule (l,r) = (Term.toString l) ^ " -> " ^ (Term.toString r)  
(*    fun prRule (l,r) = "\"" ^ (Term.toString l) ^ " -> " ^ (Term.toString r) ^ "\""  *)
   fun prRuleWithVarSort (l,r) = (Term.toStringWithVarSort l) ^ " -> " ^ (Term.toStringWithVarSort r)
   fun prEq (l,r) = (Term.toString l) ^ " = " ^ (Term.toString r) 
   val prRules = PrintUtil.prList prRule
   val prRulesWithVarSort = PrintUtil.prList prRuleWithVarSort
   val prEqs = PrintUtil.prList prEq
   fun mkRule (l,r) = (l,r)

   fun prRuleInProofTree (l,r) () = 
       CU.encloseProofTreesBy "rule"
			[fn _ => (CU.encloseProofTreeBy "lhs" (Term.toProofTree l)),
			 fn _ => (CU.encloseProofTreeBy "rhs" (Term.toProofTree r))]
   fun prRulesInProofTree rs () 
     = CU.encloseProofTreesBy "rules" (L.map (fn f => prRuleInProofTree f) rs)
   fun prTrsInProofTree rs () = CU.encloseProofTreeBy "trs" (prRulesInProofTree rs)

   fun prEquationsInProofTree rs () = CU.encloseProofTreeBy "equations" (prRulesInProofTree rs)


   fun sortSetInRule (l,r) = SS.union (sortSetInTerm l, sortSetInTerm r)
   fun sortSetInRules rs = 
       L.foldr (fn (r,xs) => SS.union (sortSetInRule r, xs)) SS.empty rs

   fun varSetInRule (l,r) = VS.union (varSetInTerm l, varSetInTerm r)
   fun varSetInRules rs = 
       L.foldr (fn (r,xs) => VS.union (varSetInRule r, xs)) VS.empty rs

   fun ruleSize (l,r) = Term.termSize l + Term.termSize r

   fun funSetInRule (l,r) = FS.union (funSetInTerm l, funSetInTerm r)
   fun funSetInRules rs = 
       L.foldr (fn (r,xs) => FS.union (funSetInRule r, xs)) FS.empty rs

(*    fun fdcSetInRules rs =  *)
(*        let fun rootFun (Var _) = FS.empty *)
(*              | rootFun (Fun (f,_,_)) = FS.singleton f *)
(* 	   val dSyms = L.foldr (fn ((l,_),xs) => FS.union (rootFun l, xs)) FS.empty rs *)
(* 	   val fSyms = funSetInRules rs *)
(* 	   val cSyms = FS.difference (fSyms, dSyms) *)
(*        in  *)
(* 	   (fSyms,dSyms,cSyms) *)
(*        end *)

   fun fdcSetInRules rs = 
       let fun rootFun (Var _) = NONE
             | rootFun (Fun (f,_,_)) = SOME f
	   val roots = L.map (fn (l,_) => rootFun l) rs
	   val vars = L.map (fn (l,r) => (varSetInTerm l, varSetInTerm r)) rs
	   val varCond = (L.all isSome roots)
			 andalso (L.all (fn (vl,vr) => VS.isSubset (vr,vl)) vars)

(* 	   val _ = if varCond then () *)
(* 		   else if (L.all isSome roots) *)
(* 		   then (print "The following rules have a extra variable in rhs.\n"; *)
(* 			 L.app (fn (l,r) => if not (VS.isSubset (varSetInTerm r, varSetInTerm l)) *)
(* 					    then print (prRule (l,r) ^ "\n") *)
(* 					    else ()) *)
(* 			       rs) *)
(* 		   else (print "The following rules have a variable lhs:\n"; *)
(* 			 L.app (fn (l,r) => if (isVar l)  *)
(* 					    then print (prRule (l,r) ^ "\n") *)
(* 					    else ()) *)
(* 			       rs) *)

	   val dSyms = L.foldr 
			   (fn (x,xset) => if isSome x 
					   then FS.add (xset, valOf x) 
					   else xset)
			   FS.empty 
			   roots
 	   val fSyms = funSetInRules rs 
	   val cSyms = FS.difference (fSyms, dSyms)
       in 
	   (fSyms,dSyms,cSyms,varCond)
       end

   fun prRelativeRulesInTpdb outs (rs,ps) = 
       let 
	   fun prVar (x,i) = ((Atom.toString x) ^ "_" ^ (Int.toString i))
	   val vars = L.foldl
			  (fn (x,str) => str ^ " " ^ (prVar x))
			  ""
			  (VS.listItems (varSetInRules (rs@ps)))
	   fun prTerm (Var (x,ty)) = (prVar x) 
	     | prTerm (Fun (f,[],ty)) = (Fun.toString f) 
	     | prTerm (Fun (f,ts,ty)) = (Fun.toString f) 
					^ (ListUtil.toStringCommaRound prTerm ts)

	   val rules = L.foldl
			  (fn ((l,r),str) => str ^ "   " 
					     ^ (prTerm l) ^ " -> " ^ (prTerm r) ^ "\n")
			  ""
			  rs

	   val prules = L.foldl
			  (fn ((l,r),str) => str ^ "   " 
					     ^ (prTerm l) ^ " ->= " ^ (prTerm r) ^ "\n")
			  ""
			  ps
       in
	   (TextIO.output (outs, "(VAR " ^ vars  ^ ")\n");
	    TextIO.output (outs, "(RULES\n" ^ rules  ^ prules ^ ")\n"))
       end

   fun prRulesInTpdb outs rs = prRelativeRulesInTpdb outs (rs,[])


   fun rulesToTrs rs =
       let val (fSyms,dSyms,cSyms,varCond) = fdcSetInRules rs
       in { FSymSet = fSyms,
	    DSymSet = dSyms,
	    CSymSet = cSyms,
	    VarCond = varCond,
	    Rules = rs }
       end

   fun checkVarCond rs = 
       (L.all (fn (l,_) => isSome (Term.funRootOfTerm l)) rs)
       (*** andalso (L.all (fn (l,r) => VS.isSubset (varSetInTerm l, varSetInTerm r)) rs)  ... bug fix 2020/2/16 ***)
       andalso (L.all (fn (l,r) => VS.isSubset (varSetInTerm r, varSetInTerm l)) rs) 



   fun dfunArityOfRule (Var _,_) =  NONE
     | dfunArityOfRule (Fun (f,ts,_),_) =  SOME (f,L.length ts)
   fun dfunAritySetOfRules rs =  FIS.addList (FIS.empty,L.mapPartial dfunArityOfRule rs)

   fun funAritySetInRule (l,r) = FIS.union (funAritySetInTerm l, funAritySetInTerm r)
   fun funAritySetInRules rs = 
       L.foldr (fn (r,xs) => FIS.union (funAritySetInRule r, xs)) FIS.empty rs

   fun funArityMapInRule (l,r) = FM.unionWith (fn (x,y) => x) 
					      (funArityMapInTerm l, funArityMapInTerm r)
   fun funArityMapInRules rs = 
       L.foldr (fn (r,xs) => FM.unionWith (fn (x,y) => x) (funArityMapInRule r, xs)) FM.empty rs


   fun prFunArityInProofTree (f,n) () =
       CU.encloseProofTreesBy "symbol"
			[fn _ => (CU.encloseProofLeafBy "name" (Fun.toString f)),
			 fn _ => (CU.encloseProofLeafBy "arity" (Int.toString n))]

   fun prFunArityListInProofTree ps () =
       CU.encloseProofTreesBy "signature" (L.map prFunArityInProofTree ps)
       

   (* well-sorted ʤ SOME sortedRule ֤Ǥʤ NONE ֤ *)
   (* fun attachSortToRule (decls:Term.decl list) (l,r) = *)
	   (*        case (attachSortToTerm decls l, attachSortToTerm decls r) of *)
	   (* 	   (SOME l', SOME r') => SOME (l',r') *)
	   (* 	 | (_,_) => NONE *)
   fun attachSortToRule (decls:Term.decl list) (l,r) =
       case attachSortToTermWithEnv decls (l,VM.empty,NONE) of 
	   SOME (l',env,opty) => (case attachSortToTermWithEnv decls (r, env, opty) of
				      SOME (r',_,_) => SOME (l',r')
				    | NONE => NONE)
	 | _ => NONE

   fun attachSortToRuleWithEnv (decls:Term.decl list) vartype (l,r) =
       case attachSortToTermWithEnv decls (l,vartype,NONE) of 
	   SOME (l',env,opty) => (case attachSortToTermWithEnv decls (r, env, opty) of
				      SOME (r',env2,NONE) => NONE (* sort of term is not definable. both of lhs and rhs are variables? *)
				   |  SOME (r',env2,opty2) => SOME (Term.attachRootSortToTerm (valOf opty2) l', r')
				   | NONE => NONE)
	 | _ => NONE

   (* well-sorted ʤ SOME sortedRule ֤Ǥʤ NONE ֤ *)
(*    fun attachSortToRules (decls:Term.decl list) [] = SOME [] *)
(*      | attachSortToRules (decls:Term.decl list) (e::es) = *)
(*        case (attachSortToRule decls e) of *)
(* 	   SOME e' => (case attachSortToRules decls es of *)
(* 			   SOME es' => SOME (e'::es') *)
(* 			 | NONE => NONE) *)
(* 	 | NONE => NONE *)
   fun attachSortToRules (decls:Term.decl list) [] = SOME []
     | attachSortToRules (decls:Term.decl list) (e::es) =
       case (attachSortToRule decls e) of
	   SOME e' => (case attachSortToRules decls es of
			   SOME es' => SOME (e'::es')
			 | NONE => NONE)
	 | NONE => NONE

   fun attachSortToRulesWithEnv (decls:Term.decl list) vartype [] = SOME []
     | attachSortToRulesWithEnv (decls:Term.decl list) vartype (e::es) =
       case (attachSortToRuleWithEnv decls vartype e) of
	   SOME e' => (case attachSortToRulesWithEnv decls vartype es of
			   SOME es' => SOME (e'::es')
			 | NONE => NONE)
	 | NONE => NONE


  (* ѿηϤȤĤƤȤơå򤷤ơ
     ؿrootιη򤭤ɲä *)
   fun addFunSortToRule (decls:Term.decl list) (l,r) =
       let val l' = attachSortToTerm decls l
	   val r' = attachSortToTerm decls r
       in if isSome l' 
	     andalso isSome r'
	     andalso Sort.equal (sortOfTerm (valOf l'),  sortOfTerm (valOf r'))
	  then SOME (valOf l', valOf r')
	  else NONE
       end

   fun addFunSortToRules (decls:Term.decl list) rs =
       let val rs' = L.map (fn lr => addFunSortToRule decls lr) rs
       in if L.all isSome rs'
	  then SOME (L.mapPartial (fn x=>x)  rs')
	  else NONE
       end

   fun sortOfRule (l,r) =
       let val s1 = sortOfTerm l
	   val s2 = sortOfTerm r
       in
	   if Sort.equal (s1,s2)
	   then SOME s1
	   else NONE
       end


   fun signatureInRule (l,r) = FM.unionWith (fn (x,y) => x) (signatureInTerm l, signatureInTerm r)
   fun signatureInRules rs = 
       L.foldr (fn (r,f) => FM.unionWith (fn (x,y) => x) (signatureInRule r, f)) FM.empty rs


   fun isRewriteRule (l,r) = isSome (funRootOfTerm l) andalso VS.isSubset (varSetInTerm r, varSetInTerm l)
   fun areRewriteRules rs = List.all isRewriteRule rs

   fun isLinearRule (l,r) = (isLinearTerm l) andalso (isLinearTerm r)
   fun areLinearRules rs = List.all isLinearRule rs

   fun isLeftLinearRule (l,r) = isLinearTerm l
   fun areLeftLinearRules rs = List.all isLeftLinearRule rs

   fun isRightLinearRule (l,r) = isLinearTerm r
   fun areRightLinearRules rs = List.all isRightLinearRule rs

   fun isNonDuplicatingRule (l,r) = null (LU.differenceByOne' Var.equal (varListInTerm r, varListInTerm l))
   fun areNonDuplicatingRules rs = List.all isNonDuplicatingRule rs

   fun isNonErasingRule (l,r) = VS.isSubset (varSetInTerm l, varSetInTerm r)
   fun areNonErasingRules rs = List.all isNonErasingRule rs

   (* renameRules [(l1,r1),(l2,r2),...] : §˴ޤޤѿ dijoint ˤʤ褦 
                                          ǥåѹ§ΥꥹȤ֤ *)
   fun renameRules rs =
       let val ts = List.map (fn (l,r) => Fun (Atom.atom "$eq",
					       [l,r],
					       Sort.null))
			     rs
	   val rts = renameTerms ts
	   val new = List.map (fn (Fun (_,[l,r],_)) => (l,r)) rts
       in
	   new
       end

   fun renameRuleDisjointFrom vs (l,r) =
       let val t = Fun (Atom.atom "$eq",
			[l,r],
			Sort.null)
	   val rt = renameTermDisjointFrom vs t
	   val new = (fn (Fun (_,[l,r],_)) => (l,r)) rt
       in
	   new
       end

  (* renameRuleSet rs1 rs2 : rs1˴ޤޤ뵬§˴ޤޤѿ rs2˴ޤޤ뵬§˴ޤޤѿ
                                dijoint ˤʤ褦 
                               ǥåѹ§ΥꥹȤ֤ *)
   fun renameRuleSet (rs1,rs2)  =
       let val (ss1,ss2) = ListPair.unzip rs1   
           val (ts1,ts2) = ListPair.unzip rs2
           val s1 = Fun (Atom.atom "$left",ss1, Sort.null)
           val s2 = Fun (Atom.atom "$right",ss2, Sort.null)
           val s = Fun (Atom.atom "$top",[s1,s2], Sort.null)
           val t1 = Fun (Atom.atom "$left",ts1, Sort.null)
           val t2 = Fun (Atom.atom "$right",ts2, Sort.null)
           val t = Fun (Atom.atom "$top",[t1,t2], Sort.null)
           val [Fun (_,[s1',s2'],_), Fun (_,[t1',t2'],_)] = renameTerms [s,t]
           val Fun (_,ss1',_) = s1'
           val Fun (_,ss2',_) = s2'
           val Fun (_,ts1',_) = t1'
           val Fun (_,ts2',_) = t2'
       in
           (ListPair.zip (ss1',ss2'), ListPair.zip (ts1',ts2'))
       end

   fun deleteIdenticalRules rules =
       let fun delete [] ans = ans
	     | delete (lr1::rs) ans = 
	       if L.exists (fn lr2 => Subst.identicalModuloRenamingRule lr1 lr2) ans
	       then delete rs ans
	       else delete rs (lr1::ans)
       in delete rules []
       end

   (* ѿ̾ɸಽ *)
   fun normalizeVarNameRule (l,r) =
       let val svlist = Term.sortedVars l
	   val size = L.length svlist
	   val newVars = L.tabulate (size, fn i => Var.fromStringAndInt ("x",i))
	   val renaming = LP.map (fn ((x,ty),y) => (x, Term.mkVarTerm (y,ty))) (svlist,newVars)
	   val sigma = L.foldl (fn ((k,v),map) => VM.insert (map,k,v)) VM.empty renaming
       in (Subst.applySubst sigma l, Subst.applySubst sigma r, sigma)
       end
       

   fun isBidirectionalRule (l,r) = VS.equal (Term.varSetInTerm l, Term.varSetInTerm r)

   fun areBidirectionalRules rules = List.all isBidirectionalRule rules

   fun isDegrowingRule (l,r) = 
       Term.isVar r
       orelse
       let val rvset = Term.varSetInTerms (L.filter  (not o Term.isVar)  (Term.argsOfTerm r))
	   val lvset = Term.varSetInTerm l
       in VS.isEmpty (VS.intersection (rvset,lvset))
       end

   fun areDegrowingRules rules = List.all isDegrowingRule rules

   fun isShallowRule (l,r) = Term.isShallowTerm l andalso Term.isShallowTerm r
   fun areShallowRules rules = List.all isShallowRule rules

   fun isCollapsingTrs rs = L.exists (fn (l,r) => isVar r) rs

   end (* of local *)

   end; (* of structure Trs *)

