(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/arith.sml
 * description: utility for sat modulo linear arithmetic theory
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature ARITH = 
sig
   datatype num = Var of int | Const of int
   type mono = num list
   type poly = mono list
   datatype atom = Eq of poly * poly 
		 | Lt of poly * poly 
		 | Gt of poly * poly 
		 | Le of poly * poly 
		 | Ge of poly * poly 
   datatype prop = Atom of atom
		 | Neg of prop
		 | Conj of prop list 
		 | Disj of prop list
		 | Imp of prop * prop 
		 | IfThenElse of prop * prop * prop
   val True: prop
   val False: prop
   val allFalse: prop list -> prop
   val one: prop list -> prop
   val atMostOne: prop list -> prop
   val atLeastOne: prop list -> prop

   val prNum: num -> string
   val prMono: mono -> string
   val prPoly: poly -> string
   val prAtom: atom -> string
   val prProp: prop -> string

end;

structure Arith: ARITH =
struct
   datatype num = Var of int | Const of int
   type mono = num list
   type poly = mono list
   datatype atom = Eq of poly * poly 
		 | Lt of poly * poly 
		 | Gt of poly * poly 
		 | Le of poly * poly 
		 | Ge of poly * poly 
   datatype prop = Atom of atom
		 | Neg of prop
		 | Conj of prop list 
		 | Disj of prop list
		 | Imp of prop * prop 
		 | IfThenElse of prop * prop * prop

   val True = Conj []
   val False = Disj []
   fun allFalse xs = Conj (List.map (fn x=> Neg x) xs)
   fun one [] = False
     | one (x::xs) = IfThenElse (x, allFalse xs, one xs)
   fun atMostOne [] = True
     | atMostOne (x::xs) = IfThenElse (x, allFalse xs, atMostOne xs)
   fun atLeastOne xs = Disj xs

   fun prNum (Var i) = if i > 0 
		       then "x" ^ (Int.toString i)
		       else if i < 0 
		       then "y" ^ (Int.toString (~i))
		       else "z"
     | prNum (Const i) = if i >= 0
			 then (Int.toString i)
			 else ("(- 0 " ^ (Int.toString (~i)) ^ ")")

   fun prMono [] = "0"
     | prMono [n] = prNum n
     | prMono ns = "(*" ^ (List.foldr (fn (n,str) => " " ^ (prNum n) ^ str) ")" ns)

   fun prPoly [] = "0"
     | prPoly [m] = prMono m
     | prPoly ms = "(+" ^ (List.foldr (fn (n,str) => " " ^ (prMono n) ^ str) ")" ms)

   fun prAtom (Eq (p,q)) = "(= " ^ (prPoly p) ^ " " ^ (prPoly q) ^ ")"
     | prAtom (Lt (p,q)) = "(< " ^ (prPoly p) ^ " " ^ (prPoly q) ^ ")"
     | prAtom (Gt (p,q)) = "(> " ^ (prPoly p) ^ " " ^ (prPoly q) ^ ")"
     | prAtom (Le (p,q)) = "(<= " ^ (prPoly p) ^ " " ^ (prPoly q) ^ ")"
     | prAtom (Ge (p,q)) = "(>= " ^ (prPoly p) ^ " " ^ (prPoly q) ^ ")"

   fun prProp (Atom a) = prAtom a
     | prProp (Neg p) = "(not " ^ (prProp p) ^ ")"
     | prProp (Conj []) = "true"
     | prProp (Conj ps) = "(and " ^ (List.foldr (fn (p,str) => " " ^ (prProp p) ^ str) ")" ps)
     | prProp (Disj []) = "false"
     | prProp (Disj ps) = "(or " ^ (List.foldr (fn (p,str) => " " ^ (prProp p) ^ str) ")" ps)
     | prProp (Imp (p,q)) = "(implies " ^ (prProp p) ^ " " ^ (prProp q) ^ ")"
     | prProp (IfThenElse (p,q,r)) = "(if " ^ (prProp p) ^ " " 
				     ^ (prProp q) ^ " " 
				     ^ (prProp r) ^ ")"


end;


