(******************************************************************************
 * Copyright (c) 2014-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/certify_util.sml
 * description: utility for generating proofs for cerfitication
 * author: AOTO Takahito
 * 
 * $Id: certify_util.sml,v 1.8 2013/06/09 11:57:01 aoto Exp $
 ******************************************************************************)

signature CERTIFY_UTIL = 
sig
   val outputFile: string ref
(*   val outStream: TextIO.outstream ref *)
   val output: (unit -> string) -> unit

(*   val prOpening: string *)
(*   val prClosing: string *)
   val toolName: string ref
   val versionNumber: string ref

   val encloseProofLeafBy: string -> string -> string
   val encloseProofTreeBy: string -> (unit -> string) -> string
   val encloseProofTreesBy: string -> ((unit -> string) list) -> string
(*   val encloseProofTreeByLn: string -> string -> string *)

   val beginInput: (unit -> string) -> unit -> string
   val inputTrs: (unit -> string) -> unit -> string
   val beginCommutation: (unit -> string) list -> unit -> string
   val beginInductiveValidity: (unit -> string) list -> unit -> string
   val version: string -> unit -> string
   val beginProof: (unit -> string) -> unit -> string
   val beginCrProof: (unit -> string) -> unit -> string
   val beginCrDisproof: (unit -> string) -> unit -> string
   val beginComProof: (unit -> string) -> unit -> string
   val beginComDisproof: (unit -> string) -> unit -> string
   val beginIndProof: (unit -> string) -> unit -> string
   val beginRewritingInductionProof: (unit -> string) list -> unit -> string
   val beginSwapTRSs: (unit -> string) -> unit -> string
   val beginNonJoinableFork: (unit -> string) list -> unit -> string

   val start: unit -> unit
   val finish: unit -> unit
end

structure CertifyUtil : CERTIFY_UTIL = 
struct
   val outputFile = ref ""
   val indentLevel = ref 0
   val toolName = ref ""
   val versionNumber = ref ""
   val outStream = ref TextIO.stdOut

   fun output mkstr = TextIO.output (!outStream,mkstr ())

   fun prSpace n = if n >= 0
		   then String.concat (List.tabulate (n,fn _=> " "))
		   else ""

   fun encloseProofLeafBy key str = 
       if str = ""
       then (prSpace (!indentLevel)) ^ "<" ^ key ^ "/>\n"
       else let val str1 = (prSpace (!indentLevel)) ^ "<" ^ key ^ ">" ;
		val str3 = "</" ^ key ^ ">\n" ;
	    in str1^str^str3
	    end

   fun encloseProofTreeBy key mkstr = 
       let val str1 = (prSpace (!indentLevel)) ^ "<" ^ key ^ ">\n" ;
	   val _ = indentLevel := (!indentLevel) + 1
	   val str2 = mkstr ()
	   val _ = indentLevel := (!indentLevel) - 1
	   val str3 = (prSpace (!indentLevel)) ^ "</" ^ key ^ ">\n" ;
       in str1^str2^str3
       end

   fun encloseProofTreesBy key mkstrs = 
       let val str1 = (prSpace (!indentLevel)) ^ "<" ^ key ^ ">\n" ;
	   val _ = indentLevel := (!indentLevel) + 1
	   val str2 = String.concat (List.map (fn f => f ()) mkstrs)
	   val _ = indentLevel := (!indentLevel) - 1
	   val str3 = (prSpace (!indentLevel)) ^ "</" ^ key ^ ">\n" ;
       in str1^str2^str3
       end

   val prOpening = "<?xml version=\"1.0\"?>\n"
		  ^ "<?xml-stylesheet type=\"text/xsl\" href=\"cpfHTML.xsl\"?>\n"
		  ^ "<certificationProblem xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" "
		  ^ "xsi:noNamespaceSchemaLocation=\"../xml/cpf.xsd\">\n"

   fun beginInput mkstr () = encloseProofTreeBy "input" mkstr
   fun beginCommutation mkstrs ()  = encloseProofTreesBy "commutationInput" mkstrs
   fun beginInductiveValidity mkstrs ()  = encloseProofTreesBy "inductiveValidityInput" mkstrs
   fun inputTrs mkstr ()  = encloseProofTreeBy "trsInput" mkstr
   fun version str () =  encloseProofLeafBy "cpfVersion" str
   fun beginProof mkstr () = encloseProofTreeBy "proof" mkstr
   fun beginCrProof mkstr () = encloseProofTreeBy "crProof" mkstr
   fun beginCrDisproof mkstr () = encloseProofTreeBy "crDisproof" mkstr
   fun beginComDisproof mkstr () = encloseProofTreeBy "comDisproof" mkstr
   fun beginComProof mkstr () = encloseProofTreeBy "comProof" mkstr
   fun beginIndProof mkstr () = encloseProofTreeBy "indProof" mkstr
   fun beginRewritingInductionProof mkstrs () = encloseProofTreesBy "rewritingInductionProof" mkstrs
   fun beginSwapTRSs mkstr () = encloseProofTreeBy "swapTRSs" mkstr
   fun beginNonJoinableFork mkstrs () = encloseProofTreesBy "nonJoinableFork" mkstrs

   fun prOrigin () = encloseProofTreeBy "origin"
		(fn _ => encloseProofTreeBy "proofOrigin"
                      (fn _ => encloseProofTreesBy "tool"
                           [fn _ => encloseProofLeafBy "name" (!toolName),
                            fn _ => encloseProofLeafBy "version" (!versionNumber)]))

   val prClosing = "</certificationProblem>\n"

   fun start () =  let (* val _ = print (!outputFile)  *)
		       val _ = outStream := TextIO.openOut (!outputFile)
		   in output (fn _ => prOpening)
		   end

   fun finish () =  let val _ = output (fn _ => prOrigin ())
                        val _ = output (fn _ => prClosing)
			val _ = TextIO.closeOut (!outStream)
			 val _ = outStream := TextIO.stdOut
		    in ()
		    end

end
