(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/graph.sml
 * description: utilities for manupilating graphs
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature GRAPH = 
sig
   type graph = int list array
   val scc: graph -> IntSet.set list
   val reflexiveTransitiveClosure: graph -> graph
   val toString: graph -> string
end;


signature STACK = 
sig
   type ''a stack 
   exception Empty
   val empty: ''a stack
   val isEmpty: ''a stack -> bool
   val push: ''a stack * ''a -> ''a stack
   val pop: ''a stack -> ''a stack
   val top: ''a stack -> ''a
   val member: ''a stack * ''a -> bool
end;

structure Stack: STACK =
struct
   type ''a stack = ''a list
   exception Empty
   val empty = []
   fun isEmpty st = st = []
   fun push (st,a) = a::st
   fun pop [] = raise Empty | pop (_::st) = st
   fun top [] = raise Empty | top (a::_) = a
   fun member (st,a) = List.exists (fn y => a = y) st
end;


structure Graph: GRAPH =
struct

  local 
      structure IIS = IntPairSet 
      structure II = IntPair
      structure IM = IntMap
      fun member (i:int, js) = List.exists (fn j => i = j) js
   in


  type graph = int list array

(*
  val graph1 = Array.fromList [[1,3,4],[2,3],[0],[2],[3],[6,7],[4],[3,5,6]]
*)

(*********************
SAMPLE:
0->{1 3 4}
1->{2 3}
2->{0}
3->{2}
4->{3}
5->{6 7}
6->{4}
7->{3 5 6}
**********************)

  fun toString ar = 
      let fun prLine n [] = (Int.toString n) ^ "->{};\n"
	    | prLine n (x::xs) = 
	      (Int.toString n) ^ "->{" ^ (Int.toString x) 
	      ^ (List.foldr (fn (x,str) => " " ^ (Int.toString x) ^ str) 
			    "};\n" 
			    xs)
      in
	  Array.foldri
	      (fn (n,xs,str) => (prLine n xs) ^ str)
	      "" 
	      ar
      end

(*
  val _ = print (toString graph1);
*)

  fun depthFirstSearch g = 
    let val treeEdges = ref IIS.empty
	val backEdges = ref IIS.empty
	datatype status = OLD | NEW
	val len =  Array.length g
	val done =  Array.array (len, NEW)

	fun search v = 
	    let val _ = Array.update (done,v,OLD)
	    in
		List.app 
		(fn w => if (Array.sub (done, w)) = NEW
			 then let val _ = treeEdges := IIS.add (!treeEdges,(v,w))
			      in search w
			      end
			 else backEdges := IIS.add (!backEdges,(v,w)))
		(Array.sub (g, v))
	    end
    in
	List.app (fn v => if (Array.sub (done, v)) = NEW
			  then search v
			  else ())
		 (List.tabulate (len, fn x=>x));
	(!treeEdges,!backEdges)
    end


(*
  val _ =  let val (tree,back) = depthFirstSearch graph1
	       fun prIISet xs = IIS.app (print o II.toString) xs
	   in
	       print "Tree Edges: {"; prIISet tree; print "}\n";
	       print "Back Edges: {"; prIISet back; print "}\n"
	   end
*)

  fun reflexiveTransitiveClosure g =
      let val len =  Array.length g
	  val indexes = List.tabulate (len, fn i => i)
	  fun edge (v,w) = member (w, Array.sub (g, v))
	  fun existsEdge (v,w) = if edge (v,w) then 1 else 0
	  val ar0 = Array2.array (len,len,1)
	  val _ = ListXProd.appX 
		      (fn (i,j) => 
			  if i = j
			  then ()
			  else Array2.update (ar0,i,j, existsEdge (i,j)))
 		      (indexes,indexes)
 	  val C = Array.array (len+1, ar0) 
 	  val _ = List.app 
		      (fn i => 
 			  let val ar = Array2.array (len, len, 0)
 			  in  Array.update (C,i+1,ar)
			  end)
 		      indexes

 	  val _ = List.app 
		      (fn k => 
			  let val Cl = Array.sub (C, k)
			      val Ck = Array.sub (C, k+1)
			  in
			      (ListXProd.appX 
				   (fn (i,j) => 
				       let val a = Array2.sub (Cl,i,j)
					   val b = Array2.sub (Cl,i,k)
					   val c = Array2.sub (Cl,k,j)
				       in
					   Array2.update (Ck,i,j, Int.max (a, Int.min (b,c)))
				       end)
 				   (indexes,indexes))
			  end)
 		      indexes

	  val result = Array.array (len,[])
	  val _ = let val arN = Array.sub (C,len)
		  in
		      ListXProd.appX
			  (fn (i,j) => 
				if Array2.sub (arN,i,j) = 1
				then let val ns = Array.sub (result,i)
				     in Array.update (result,i,j::ns)
				     end
				else ())
			  (indexes,rev indexes)
		  end
      in
	  result
      end

(*
  val _ =  let val graph2 =  reflexiveTransitiveClosure graph1
	   in
	       (print "reflexive transitive closure:\n";
		print (toString graph2))
	   end
*)

  fun scc g = 
    let 
	datatype status = OLD | NEW
	val len =  Array.length g
	val done =  Array.array (len, NEW)
	val count = ref 0
	val dfNumber =  Array.array (len, 0)
	val lowLink =  Array.array (len, 0)
	val revisited =  Array.array (len, false)
	val st =  ref Stack.empty
	val answer =  ref []
	fun getSccFromStack v = let val w = Stack.top (!st)
				    val _ = st := Stack.pop (!st)
				in if w = v
				   then IntSet.singleton v
				   else IntSet.add (getSccFromStack v, w)
				end

	fun search v = 
	    let val _ = Array.update (done,v,OLD)
		val _ = count := !count + 1
		val _ = Array.update (dfNumber,v,!count)
		val _ = Array.update (lowLink,v,!count)
		val _ = st := Stack.push (!st,v)
		val _ = List.app 
			    (fn w => if (Array.sub (done, w)) = NEW
				     then let val _ = search w
					  in 
					      Array.update (lowLink,v,
							    Int.min (Array.sub (lowLink,v),
								     Array.sub (lowLink,w)))
					  end
				     else if (Array.sub (dfNumber,w)) <= (Array.sub (dfNumber,v))
					     andalso (Stack.member (!st,w))
				     then
					 let val _ = Array.update (revisited, w, true)
					 in
					     Array.update (lowLink,v,
							   Int.min (Array.sub (dfNumber,w),
								    Array.sub (lowLink,v)))
					 end
				     else ())
			    (Array.sub (g, v))
	    in
		if Array.sub (lowLink,v) = (Array.sub (dfNumber,v))
		then if Array.sub (revisited,v)
		     then answer := ((getSccFromStack v) :: (!answer))
		     else (getSccFromStack v; ())
		else ()
	    end
    in
	List.app (fn v => if (Array.sub (done, v)) = NEW
			  then search v
			  else ()) (List.tabulate (len, fn x=>x));
	rev (!answer)
    end

(*
  val _ =  let val vss = scc graph1
	   in
	       List.map (print o IntSet.toString) vss
	   end
*)

  end

end;


