(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/lexer.sml
 * description: lexer generator from "ML for Working Programmer", C.Paulson
 * 
 ******************************************************************************)

signature LEXICAL =
  sig
  datatype token = Id of string | Key of string
  val scan : string -> token list
  end;

signature KEYWORD =
  sig
  val alphas : string list
  and symbols : string list
  end;

functor Lexical (Keyword: KEYWORD) : LEXICAL =
  struct
  datatype token = Key of string | Id of string;

  fun member (x:string, l) = List.exists (fn y => x = y) l;

  fun alphaTok a =
      if member (a, Keyword.alphas) then Key (a) else Id (a);

  fun symbolic (sy, ss) =
      case Substring.getc ss of
	  NONE => (Key sy, ss)
	| SOME (c, ss1) =>
	  if member (sy, Keyword.symbols) orelse not (Char.isPunct c)
	  then (Key sy, ss)
	  else symbolic (sy ^ String.str c, ss1);

  fun scanning (toks, ss) =
      case Substring.getc ss of
	  NONE => rev toks
	| SOME (c, ss1) => 
	  if Char.isAlphaNum c
	  then let val (id, ss2) = Substring.splitl Char.isAlphaNum ss
		   val tok = alphaTok (Substring.string id)
	       in scanning (tok::toks, ss2)
	       end
	  else if Char.isPunct c
	  then let val (tok, ss2) = symbolic (String.str c, ss1)
	       in scanning (tok::toks, ss2)
	       end
	  else
	      scanning (toks, Substring.dropl (not o Char.isGraph) ss);

  fun scan a = scanning ([], Substring.full a);
  end;



