(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/list_util.sml
 * description: utility for list functions
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature LIST_UTIL =  
sig
    val mapAppend: ('a -> 'b list) -> 'a list -> 'b list
    val replaceNth: ('a list * int * 'a) -> 'a list
    val exceptNth: ('a list * int) -> 'a list
    val exceptLast: 'a list -> 'a list

    val member: ''a -> ''a list -> bool
    val member': ('a * 'a -> bool) -> 'a -> 'a list -> bool

    val setEqual: (''a list * ''a list) -> bool
    val setEqual': ('a * 'a -> bool) -> ('a list * 'a list) -> bool

    val subseteq: (''a list * ''a list) -> bool
    val subseteq': ('a * 'a -> bool) -> ('a list * 'a list) -> bool

    val union: (''a list * ''a list) -> ''a list
    val union': ('a * 'a -> bool) -> ('a list * 'a list) -> 'a list

    val intersection: (''a list * ''a list) -> ''a list
    val intersection': ('a * 'a -> bool) -> ('a list * 'a list) -> 'a list

    val disjoint: (''a list * ''a list) -> bool
    val disjoint': ('a * 'a -> bool) -> ('a list * 'a list) -> bool

    val indexOf: ''a -> ''a list -> int option
    val indexOf': ('a * 'b -> bool) -> 'a -> 'b list -> int option

    val deleteOne: ''a -> ''a list -> ''a list 
    val deleteExactlyOne: ''a -> ''a list -> ''a list option
    val multisetEqual: (''a list * ''a list) -> bool
    val multisetUnion: (''a list * ''a list) -> ''a list

    val deleteAll: ''a -> ''a list -> ''a list 
    val differenceByOne: ''a list * ''a list -> ''a list 
    val differenceByAll: ''a list * ''a list -> ''a list 

    val deleteOne': ('a * 'a -> bool) -> 'a -> 'a list -> 'a list
    val deleteExactlyOne': ('a * 'a -> bool) -> 'a -> 'a list -> 'a list option
    val multisetEqual': ('a * 'a -> bool) -> ('a list * 'a list) -> bool
    val multisetUnion': ('a * 'a -> bool) -> ('a list * 'a list) -> 'a list

    val deleteAll': ('a * 'a -> bool) -> 'a -> 'a list -> 'a list
    val differenceByOne': ('a * 'a -> bool) -> 'a list * 'a list -> 'a list
    val differenceByAll': ('a * 'a -> bool) -> 'a list * 'a list -> 'a list

    val eliminateDuplication: ''a list -> ''a list
    val eliminateDuplication': ('a * 'a -> bool) -> 'a list -> 'a list

    val takeMinimals: ('a * 'a -> order option) -> 'a list -> 'a list
    val takeMaximals: ('a * 'a -> order option) -> 'a list -> 'a list
   (*  order option = quasi_order = SOME LESS | SOME EQUAL | SOME GREATER | NONE *)

    val assemble: 'a list list -> 'a list list
    val classify: ('a * 'a -> bool) -> 'a list -> 'a list list

    val toString: ('a -> string) -> 'a list -> string
    val toStringComma: ('a -> string) -> 'a list -> string
    val toStringSpace: ('a -> string) -> 'a list -> string
    val toStringAst: ('a -> string) -> 'a list -> string
    val toStringPlus: ('a -> string) -> 'a list -> string
    val toStringPlusRound: ('a -> string) -> 'a list -> string
    val toStringSlashSquare: ('a -> string) -> 'a list -> string

    val toStringCommaSquare: ('a -> string) -> 'a list -> string
    val toStringCommaAngle: ('a -> string) -> 'a list -> string
    val toStringCommaCurly: ('a -> string) -> 'a list -> string
    val toStringCommaRound: ('a -> string) -> 'a list -> string
    val toStringCommaLnSquare: ('a -> string) -> 'a list -> string
    val toStringCommaLnCurly: ('a -> string) -> 'a list -> string

    val toStringSpaceSquare: ('a -> string) -> 'a list -> string
    val toStringSpaceAngle: ('a -> string) -> 'a list -> string
    val toStringSpaceCurly: ('a -> string) -> 'a list -> string
    val toStringSpaceRound: ('a -> string) -> 'a list -> string
    val toStringSpaceLnSquare: ('a -> string) -> 'a list -> string
    val toStringSpaceLnCurly: ('a -> string) -> 'a list -> string

    val toStringSemicolonSquare: ('a -> string) -> 'a list -> string
    val toStringSemicolonAngle: ('a -> string) -> 'a list -> string
    val toStringSemicolonCurly: ('a -> string) -> 'a list -> string
    val toStringSemicolonRound: ('a -> string) -> 'a list -> string
    val toStringSemicolonLnSquare: ('a -> string) -> 'a list -> string
    val toStringSemicolonLnCurly: ('a -> string) -> 'a list -> string

    val powerlist: 'a list -> 'a list list
    val powerlistWithComplement: 'a list -> ('a list * 'a list) list
    val cartesianProduct: 'a list * 'b list -> ('a * 'b) list
    val allCombinations: 'a list list -> 'a list list

end

structure ListUtil: LIST_UTIL =
struct
  local 
  structure L = List

  in 

  fun mapAppend f xs = L.foldr (fn (x,ys) => L.@(f x, ys)) [] xs

    fun replaceNth (xs,i,y) = (L.take (xs,i)) @ [y] @ (L.drop (xs,i+1))

    fun exceptNth ([],n) = []
      | exceptNth (x::xs,n) = if n = 0 then xs else (x::(exceptNth (xs,n-1)))

    fun exceptLast xs = 
	let fun exceptLast0 ([],ys) = rev ys
	      | exceptLast0 ([x],ys) = rev ys
	      | exceptLast0 (x::xs,ys) = exceptLast0 (xs,x::ys)
	in exceptLast0 (xs,[])
	end

   (* ys $B$NCf$K(B x $B$,$"$k$+!)(B *)
    fun member x ys = L.exists (fn y => x = y) ys

   (* ys $B$NCf$K(B x $B$,$"$k$+!)(B *)
    fun member' pred x ys = L.exists (fn y => pred (x,y)) ys

    (* $B=89g$H$7$FEy$7$$$+!)(B *)
    fun setEqual (xs,ys) = L.all (fn x => member x ys) xs
			 andalso L.all (fn y => member y xs) ys

    fun setEqual' pred (xs,ys) = L.all (fn x => member' pred x ys) xs
			       andalso L.all (fn y => member' pred y xs) ys

    (* $B=89g$H$7$FItJ,=89g$+!)(B xs is a subset of ys ?*)
    fun subseteq (xs,ys) = L.all (fn x => member x ys) xs
    fun subseteq' pred (xs,ys) = L.all (fn x => member' pred x ys) xs

   (* xs $B$H(B ys $B$r!$=EJ#$r=|$$$F9gJ;(B *)
    fun union (xs,ys) = 
	let fun unionSub [] ys ans = L.revAppend (ans,ys)
	      | unionSub (x::xs) ys ans = 
		if member x ys
		then unionSub xs ys ans
		else unionSub xs ys (x::ans)
	in unionSub xs ys []
	end

   (* xs $B$H(B ys $B$r!$=EJ#$r=|$$$F9gJ;(B *)
    fun union' pred (xs,ys) = 
	let fun unionSub [] ys ans = L.revAppend (ans,ys)
	      | unionSub (x::xs) ys ans = 
		if member' pred x ys
		then unionSub xs ys ans
		else unionSub xs ys (x::ans)
	in unionSub xs ys []
	end

    fun intersection (xs,ys) = L.filter (fn x => member x ys) xs
    fun intersection' pred (xs,ys) = L.filter (fn x => member' pred x ys) xs

    fun disjoint (xs,ys) = not (L.exists (fn x => member x ys) xs)
    fun disjoint' pred (xs,ys) = not (L.exists (fn x => member' pred x ys) xs)

   (* x $B$,(B ys $B$NCf$G2?HVL\$NMWAG$H$7$F=P8=$9$k$+$r(B option $B7?$GJV$9(B *)
   (* $B@hF,$J$i(B 0$BHVL\!"=P8=$7$J$$$H$-$O(B NONE *)
    fun indexOf x ys = 
	let fun indexOfSub [] _= NONE
	      | indexOfSub (z::zs) num = 
		if x = z then SOME num else indexOfSub zs (1+num)
	in indexOfSub ys 0 
	end

    fun indexOf' pred x ys = 
	let fun indexOfSub [] _= NONE
	      | indexOfSub (z::zs) num = 
		if pred (x,z) then SOME num else indexOfSub zs (1+num)
	in indexOfSub ys 0 
	end

   (* ys $B$NCf$+$i(B x $B$r(B1$B$D:o=|(B($B$"$l$P(B)$B$9$k(B *)
    fun deleteOne x ys = 
	let fun delete [] ans = rev ans
	      | delete (y'::ys') ans = 
		if x = y' then L.revAppend (ans, ys')
		else delete ys' (y'::ans)
	in delete ys [] 
	end

   (* ys $B$NCf$+$i(B x $B$rCzEY(B1$B$D:o=|$7!$$G$-$J$l$P(B NONE *)
    fun deleteExactlyOne x ys = 
	let fun delete [] ans = NONE
	      | delete (y'::ys') ans = 
		if x = y' then SOME (L.revAppend (ans, ys'))
		else delete ys' (y'::ans)
	in delete ys [] 
	end

   (* xs $B$,(B ys $BB?=E=89g$H$7$FEy$7$$(B *)
    fun multisetEqual (xs,[]) = List.null xs
      | multisetEqual (xs,y::ys) = case deleteExactlyOne y xs of 
				       SOME xs' => multisetEqual (xs',ys)
				     | NONE => false


   (* xs $B$H(B ys $B$r!$B?=E=89g$H$7$F9gJ;(B *)
    fun multisetUnion (xs,ys) = 
	let fun unionSub [] ys ans = L.revAppend (ans,ys)
	      | unionSub (x::xs) ys ans = unionSub xs (deleteOne x ys) (x::ans)
	in unionSub xs ys []
	end

   (* ys $B$NCf$+$i(B x $B$r$9$Y$F:o=|$9$k(B *)
    fun deleteAll x ys = 
	let fun delete [] ans = rev ans
	      | delete (y'::ys') ans = 
		if x = y' then delete ys' ans
		else delete ys' (y'::ans)
	in delete ys [] 
	end

   (* xs $B$NCf$+$i(B ys $B$HF1$8MWAG$rF1$88D?t$@$1:o=|$9$k(B *)
    fun differenceByOne (xs,[]) = xs
      | differenceByOne (xs,y::ys) = differenceByOne (deleteOne y xs, ys)

   (* xs $B$NCf$+$i!$(Bys $B$K$b$"$kMWAG$r!$A4It:o=|$9$k(B *)
    fun differenceByAll (xs,[]) = xs
      | differenceByAll (xs,y::ys) = differenceByAll (deleteAll y xs, ys)

   (* ys $B$NCf$+$i(B x $B$r(B1$B$D:o=|(B($B$"$l$P(B)$B$9$k(B *)
    fun deleteOne' pred x ys = 
	let fun delete [] ans = rev ans
	      | delete (y'::ys') ans = 
		if pred (x,y') then L.revAppend (ans, ys')
		else delete ys' (y'::ans)
	in delete ys [] 
	end

   (* ys $B$NCf$+$i(B x $B$rCzEY(B1$B$D:o=|$7!$$G$-$J$l$P(B NONE *)
    fun deleteExactlyOne' pred x ys = 
	let fun delete [] ans = NONE
	      | delete (y'::ys') ans = 
		if pred (x,y') then SOME (L.revAppend (ans, ys'))
		else delete ys' (y'::ans)
	in delete ys [] 
	end

   (* xs $B$,(B ys $BB?=E=89g$H$7$FEy$7$$(B *)
    fun multisetEqual' pred (xs,[]) = List.null xs
      | multisetEqual' pred (xs,y::ys) = case deleteExactlyOne' pred y xs of 
					     SOME xs' => multisetEqual' pred (xs',ys)
					   | NONE => false


   (* xs $B$H(B ys $B$r!$B?=E=89g$H$7$F9gJ;(B *)
    fun multisetUnion' pred (xs,ys) = 
	let fun unionSub [] ys ans = L.revAppend (ans,ys)
	      | unionSub (x::xs) ys ans = unionSub xs (deleteOne' pred x ys) (x::ans)
	in unionSub xs ys []
	end

   (* ys $B$NCf$+$i(B x $B$r$9$Y$F:o=|$9$k(B *)
    fun deleteAll' pred x ys = 
	let fun delete [] ans = rev ans
	      | delete (y'::ys') ans = 
		if pred (x,y') then delete ys' ans
		else delete ys' (y'::ans)
	in delete ys [] 
	end

   (* xs $B$NCf$+$i(B ys $B$HF1$8MWAG$rF1$88D?t$@$1:o=|$9$k(B *)
    fun differenceByOne' _ (xs,[]) = xs
      | differenceByOne' pred (xs,y::ys) = differenceByOne' pred (deleteOne' pred y xs, ys)

   (* xs $B$NCf$+$i!$(Bys $B$K$b$"$kMWAG$r!$A4It:o=|$9$k(B *)
    fun differenceByAll' _  (xs,[]) = xs
      | differenceByAll' pred (xs,y::ys) = differenceByAll' pred (deleteAll' pred y xs, ys)

    fun eliminateDuplication xs = 
	let fun elimDup [] ans = rev ans
	      | elimDup (x::xs) ans = 
		if L.exists (fn y => x = y) ans
		then elimDup xs ans
		else elimDup xs (x::ans)
	in elimDup xs []
	end

    fun eliminateDuplication' pred xs = 
	let fun elimDup [] ans = rev ans
	      | elimDup (x::xs) ans = 
		if L.exists (fn y => pred (x,y)) ans
		then elimDup xs ans
		else elimDup xs (x::ans)
	in elimDup xs []
	end


    local exception FoundSmaller
	  fun inv pred (x,y) = case pred (x,y) of 
				   SOME LESS => SOME GREATER
				 | SOME GREATER => SOME LESS
				 | ans => ans
    in fun takeMinimals pred [] = []
	 | takeMinimals pred (x::ys) = case (takeMinimalsSub pred x ys) of
				       SOME ys' => x::(takeMinimals pred ys')
				     | NONE => takeMinimals pred ys
       and takeMinimalsSub pred x ys = 
	   let val ys' = L.mapPartial (fn y => case pred (x,y) of 
						   SOME GREATER => raise FoundSmaller
						 | SOME LESS => NONE
						 | SOME EQUAL => NONE
						 | NONE => SOME y)  ys
	   in SOME ys' 
	   end handle FoundSmaller => NONE
       fun takeMaximals pred t  = takeMinimals (inv pred) t
    end

    (* assemble [[1,2],[3,4,5]] = [[1,3],[1,4],[1,5],[2,3],[2,4],[2,5]] *)
    fun assemble [] =  []
     |  assemble [xs] =  L.map (fn x=>[x]) xs
     |  assemble (xs::yss) = ListXProd.mapX (fn (x,ys)=> x::ys) (xs, assemble yss)

(* classify (fn (x,y) => x = y) [1,2,5,1,2,3,1,4,5]  = [[1,1,1],[2,2],[5,5],[3],[4]] *)
    fun classify isEq [] = []
      | classify isEq (x::ys) =
	let val (xEquiv,xNonEquiv) = L.partition (fn y => isEq (x,y)) ys
	in (x::xEquiv) :: (classify isEq xNonEquiv)
	end
							   


    fun toStringBuilt prElm (start,sep,stop) xs =
	let fun toStringSub [] = ""
	      | toStringSub [x] = (prElm x)
	      | toStringSub (x::xs)= (prElm x) ^ sep ^ (toStringSub xs)
	in  start ^ (toStringSub xs) ^ stop
	end

    fun toString prElm xs = toStringBuilt prElm ("",  "", "") xs

    fun toStringComma prElm xs = toStringBuilt prElm ("",  ",", "") xs
    fun toStringSpace prElm xs = toStringBuilt prElm ("",  " ", "") xs
    fun toStringAst prElm xs = toStringBuilt prElm   ("",  "*", "") xs
    fun toStringPlus prElm xs = toStringBuilt prElm   ("",  "+", "") xs
    fun toStringPlusRound prElm xs = toStringBuilt prElm   ("(",  "+", ")") xs
    fun toStringSlashSquare prElm xs = toStringBuilt prElm   ("[",  "/", "]") xs

    fun toStringSpaceSquare prElm xs = toStringBuilt prElm ("[",  " ",  "]") xs
    fun toStringSpaceAngle prElm xs = toStringBuilt prElm  ("<",  " ",  ">") xs
    fun toStringSpaceCurly prElm xs = toStringBuilt prElm  ("{",  " ",  "}") xs
    fun toStringSpaceRound prElm xs = toStringBuilt prElm  ("(",  " ",  ")") xs
    fun toStringSpaceLnSquare prElm xs = toStringBuilt prElm ("   [ ",  "\n     ",  " ]\n") xs
    fun toStringSpaceLnCurly prElm xs = toStringBuilt prElm ("   { ",  "\n     ",  " }\n") xs


    fun toStringCommaSquare prElm xs = toStringBuilt prElm ("[",  ",",  "]") xs
    fun toStringCommaAngle prElm xs = toStringBuilt prElm  ("<",  ",",  ">") xs
    fun toStringCommaCurly prElm xs = toStringBuilt prElm  ("{",  ",",  "}") xs
    fun toStringCommaRound prElm xs = toStringBuilt prElm  ("(",  ",",  ")") xs
    fun toStringCommaLnSquare prElm xs = toStringBuilt prElm ("   [ ",  ",\n     ",  " ]\n") xs
    fun toStringCommaLnCurly prElm xs = toStringBuilt prElm ("   { ",  ",\n     ",  " }\n") xs

    fun toStringSemicolonSquare prElm xs = toStringBuilt prElm ("[",  ";",  "]") xs
    fun toStringSemicolonAngle prElm xs = toStringBuilt prElm  ("<",  ";",  ">") xs
    fun toStringSemicolonCurly prElm xs = toStringBuilt prElm  ("{",  ";",  "}") xs
    fun toStringSemicolonRound prElm xs = toStringBuilt prElm  ("(",  ";",  ")") xs
    fun toStringSemicolonLnSquare prElm xs = toStringBuilt prElm ("   [ ",  ";\n     ",  " ]\n") xs
    fun toStringSemicolonLnCurly prElm xs = toStringBuilt prElm ("   { ",  ";\n     ",  " }\n") xs


    (* powerlist: [0,1,2] 
     =>  [[],[0],[1],[2],[0,1],[0,2],[1,2],[0,1,2]]  *)
    fun powerlist [] = [[]]
      | powerlist (x::xs) = 
	let val yss = powerlist xs
	in yss @ (L.map (fn ys => x::ys) yss)
	end

    (* powerlistWithComplement: 
     [0,1,2]  => 
     [([1,2,3],[]),
      ([1,2],[3]),([1,3],[2]),([1],[2,3]),
      ([2,3],[1]),([2],[1,3]),([3],[1,2]),([],[1,2,3])] *)
    fun powerlistWithComplement [] = [([],[])]
      | powerlistWithComplement (x::xs) = 
	let val yss = powerlistWithComplement xs
	in (L.map (fn (ys,zs) => (x::ys,zs)) yss)
	   @ (L.map (fn (ys,zs) => (ys,x::zs)) yss)
	end

    (* cartesianProduct [1,2] [a,b,c] = [(1,a),(1,b),(1,c),(2,a),(2,b),(2,c)] *)
    fun cartesianProduct (xs,ys) = ListXProd.mapX (fn xy => xy) (xs,ys)

    (* allCombinations [[1,2],[a,b],[A,B]] = 
     =  [ [1,a,A], [1,a,B], [1,b,A], [1,b,B],
          [2,a,A], [2,a,B], [2,b,A], [2,b,B] ]
     *)
    fun allCombinations [] = [[]]
      | allCombinations (xs::[]) = L.map (fn x => [x]) xs
      | allCombinations (xs::xss) = 
	L.map (fn (x,xs) => x::xs) (cartesianProduct (xs, allCombinations xss))

  end
end
