(******************************************************************************
 * Copyright (c) 2012-2013, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/ord_key2.sml
 * description: some general sets and mappings
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature ORD_KEY2 = 
sig 
   include ORD_KEY (* { val ord_key, fun compare }*)
   val equal: ord_key * ord_key -> bool
   val toString: ord_key -> string
end

functor OrdKey2PairFn (structure A : ORD_KEY2
                       structure B : ORD_KEY2) : ORD_KEY2 = 
struct  
   type ord_key = A.ord_key * B.ord_key
   fun compare ((a1,b1),(a2,b2)) =
       case A.compare (a1,a2)
	of EQUAL => B.compare (b1,b2)
	 | GREATER => GREATER
	 | LESS => LESS
   fun equal (aa, bb) = (compare (aa,bb)) = EQUAL
   fun toString (a,b) = "(" ^ (A.toString a) ^ "," ^ (B.toString b) ^ ")"
end;

signature ORD_SET2 =
sig 
   include ORD_SET 
   val toString: set -> string
end

functor OrdKey2SetFn (Key : ORD_KEY2) : ORD_SET2 = 
struct  
   structure Key = Key : ORD_KEY2
   structure Set = RedBlackSetFn (Key) : ORD_SET
   type item = Key.ord_key
   type set = Set.set
   val empty = Set.empty
   val singleton = Set.singleton
   val add = Set.add 
   val add' = Set.add' 
   val addList = Set.addList 
   val delete = Set.delete 
   val member = Set.member 
   val isEmpty = Set.isEmpty 
   val equal = Set.equal 
   val compare = Set.compare 
   val isSubset = Set.isSubset 
   val numItems = Set.numItems 
   val listItems = Set.listItems 
   val union = Set.union 
   val intersection = Set.intersection 
   val difference = Set.difference 
(* comment out following 4 line to use SML/NJ versions < 110.75 *)
   val subtract = Set.subtract
   val subtract' = Set.subtract'
   val subtractList = Set.subtractList
   val all = Set.all
(*****************)
(* comment out the following 3 lines to use SML/NJ versions < 110.80 *)
(*   val minItem = Set.minItem *)
(*   val maxItem = Set.maxItem *)
(*   val toList = Set.toList *)
(*****************)
   val fromList = Set.fromList
   val partition = Set.partition
   val map = Set.map 
   val app = Set.app 
   val foldl = Set.foldl 
   val foldr = Set.foldr 
   val filter = Set.filter 
   val exists = Set.exists 
   val find = Set.find 
   fun toString xs = 
       if Set.isEmpty xs
       then "{}"
       else "{" ^ (PrintUtil.prSeq Key.toString (listItems xs)) ^ "}"
end;

structure Int2 : ORD_KEY2  =
struct
   type ord_key = int
   val compare = Int.compare
   val equal = op=
   val toString = Int.toString
end

(* structure IntSet = RedBlackSetFn (Int2) : ORD_SET *)
structure IntSet = OrdKey2SetFn (Int2) : ORD_SET2

structure IntMap = RedBlackMapFn (Int2) : ORD_MAP
structure IntPair = OrdKey2PairFn (structure A = Int2 structure B = Int2) : ORD_KEY2
structure IntPairSet = RedBlackSetFn (IntPair) : ORD_SET
structure IntPairMap = RedBlackMapFn (IntPair) : ORD_MAP

functor OrdKey2TripleFn (structure A : ORD_KEY2
                         structure B : ORD_KEY2
                         structure C : ORD_KEY2) : ORD_KEY2 = 
struct  
   type ord_key = A.ord_key * B.ord_key * C.ord_key
   fun compare ((a1,b1,c1),(a2,b2,c2)) =
       case A.compare (a1,a2)
	of EQUAL => (case B.compare (b1,b2)
		      of EQUAL => C.compare (c1,c2)
		       | GREATER => GREATER
		       | LESS => LESS)
	 | GREATER => GREATER
	 | LESS => LESS
   fun equal (aaa, bbb) = (compare (aaa,bbb)) = EQUAL
   fun toString (a,b,c) = "(" ^ (A.toString a) ^ "," ^ (B.toString b) ^ "," ^ (C.toString c) ^ ")"
end;

structure IntTriple = OrdKey2TripleFn (structure A = Int2 
                                       structure B = Int2
                                       structure C = Int2) : ORD_KEY2

structure IntTripleMap = RedBlackMapFn (IntTriple) : ORD_MAP

structure IntList : ORD_KEY2 = 
struct  
   type ord_key = int list
   fun compare ([],[]) = EQUAL
     | compare ([],y::ys) = LESS
     | compare (x::xs,[]) = GREATER
     | compare (x::xs,y::ys) = 
       case Int.compare (x,y) 
	of GREATER => GREATER
	 | LESS => LESS
	 | EQUAL => compare (xs,ys)
   fun equal (x,y) = (compare (x,y)) = EQUAL
   fun toString xs = "[ " ^ (PrintUtil.prSeq Int.toString xs) ^ "]"
end;

structure IntListSet2 = RedBlackSetFn (IntList) : ORD_SET 
 (* IntListSet structure already exists in SML/NJ library *)
structure IntListMap2 = RedBlackMapFn (IntList) : ORD_MAP 
 (* IntListSet structure already exists in SML/NJ library *)
structure IntListPair = OrdKey2PairFn (structure A = IntList structure B = IntList) : ORD_KEY2
structure IntListPairMap = RedBlackMapFn (IntListPair) : ORD_MAP

(**
structure AtomKey : ORD_KEY2 =
   struct
   type ord_key = Atom.atom
   val compare = Atom.compare
   fun equal (a,b) = compare (a,b) = EQUAL
   val toString = Atom.toString 
   end 

structure AtomSet = RedBlackSetFn (AtomKey) : ORD_SET
**)

structure String2 : ORD_KEY2  =
struct
   type ord_key = string
   val compare = String.compare
   val equal = op=
   val toString = String.toString
end

structure StringSet = OrdKey2SetFn (String2) : ORD_SET2
structure StringMap = RedBlackMapFn (String2) : ORD_MAP
structure StringPair = OrdKey2PairFn (structure A = String2 
                                      structure B = String2) : ORD_KEY2
structure StringPairSet = RedBlackSetFn (StringPair) : ORD_SET
structure StringPairMap = RedBlackMapFn (StringPair) : ORD_MAP
