(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * Copyright (c) 2016-2023, Aoto Laboratory, Niigata University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/print_util.sml
 * description: utility for printing 
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature PRINT_UTIL = 
sig
  (* to be deleted *)
   val prSeq : ('a -> string) -> 'a list -> string
   val prSeqWith: ('a -> string) -> string -> 'a list -> string
   val prSpace : ('a -> string) -> 'a list -> string
   val prProd : ('a -> string) -> 'a list -> string
   val prListInOneLine : ('a -> string) -> 'a list -> string
   val prListInOneLineWith : ('a -> string) -> string -> 'a list -> string
   val prSetInOneLine : ('a -> string) -> 'a list -> string
   val prSetInOneLineWith : ('a -> string) -> string -> 'a list -> string

   val prList : ('a -> string) -> 'a list -> string
   val prListWith : ('a -> string) -> string -> 'a list -> string
   val prPair : ('a -> string) -> 'a * 'a -> string

   val printStdOut : string -> unit
   val printStdErr : string -> unit
   val printlnStdOut : string -> unit
   val printlnStdErr : string -> unit
   val outs: TextIO.outstream ref
   val print : string -> unit
   val println : string -> unit
   val error : string -> unit
end

structure PrintUtil : PRINT_UTIL = 
struct

   (* to be deleted => see ListUtil *)
   fun prSeq f [] = ""  
     | prSeq f [a] = f a 
     | prSeq f (x::xs) = (f x) ^ "," ^ (prSeq f xs)

   (* to be deleted => see ListUtil *)
   fun prSeqWith f str [] = ""
     | prSeqWith f str [a] = f a
     | prSeqWith f str (x::xs) = (f x) ^ str ^ (prSeqWith f str xs)

  (* to be deleted => see ListUtil *)
   fun prSpace f [] = ""
     | prSpace f [a] = f a
     | prSpace f (x::xs) = (f x) ^ " " ^ (prSpace f xs)

  (* to be deleted => see ListUtil *)
   fun prProd f [] = ""
     | prProd f [a] = f a
     | prProd f (x::xs) = (f x) ^ "*" ^ (prProd f xs)

   (* to be deleted => see ListUtil *)
   fun prListInOneLine f xs = "[" ^ (prSeq f xs) ^ "]"
   fun prListInOneLineWith f str xs = "[" ^ (prSeqWith f str xs) ^ "]"
   fun prSetInOneLine f xs = "{" ^ (prSeq f xs) ^ "}"
   fun prSetInOneLineWith f str xs = "{" ^ (prSeqWith f str xs) ^ "}"

   fun prList f [] = "   [ ]\n"
     | prList f (x::xs)  = "   [ " ^ (f x) ^ (prListSub f xs)
   and prListSub f [] = " ]\n"
     | prListSub f (x::xs) = ",\n     " ^ (f x) ^ (prListSub f xs)

   fun prListWith f str [] = "   [ ]\n"
     | prListWith f str (x::xs)  = "   [ " ^ (f x) ^ (prListWithSub f str xs)
   and prListWithSub f str [] = " ]\n"
     | prListWithSub f str (x::xs) = str ^ "\n     " ^ (f x) ^ (prListWithSub f str xs)

   fun prPair f (a,b) = "<" ^ (f a) ^ "," ^ (f b) ^ ">"

   fun printStdOut str = TextIO.output (TextIO.stdOut,str)
   fun printStdErr str = TextIO.output (TextIO.stdErr,str)

   fun printlnStdErr str = printStdErr (str ^ "\n")
   fun printlnStdOut str = printStdOut (str ^ "\n")

   val outs = ref TextIO.stdOut

   fun print str = TextIO.output (!outs,str)
   fun println str = print (str ^ "\n")

   val err = ref TextIO.stdErr
   fun error str = TextIO.output (!err,str)
end
