(******************************************************************************
 * Copyright (c) 2012-2015, Toyama&Aoto Laboratory, Tohoku University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  1. Redistributions of source code must retain the above copyright notice, 
 *     this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright 
 *     notice, this list of conditions and the following disclaimer in the 
 *     documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR 
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS 
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
 * POSSIBILITY OF SUCH DAMAGE.
 ******************************************************************************)
(******************************************************************************
 * file: rwtools/util/prop.sml
 * description: utility for propositional logic
 * author: AOTO Takahito
 * 
 ******************************************************************************)

signature PROP = 
sig
   datatype prop = Atom of int | Neg of prop
		  | Conj of prop list | Disj of prop list
		  | Imp of prop * prop | Iff of prop * prop 
		  | IfThenElse of prop * prop * prop

   val True: prop
   val False: prop
   val allFalse: prop list -> prop
   val one: prop list -> prop
   val atMostOne: prop list -> prop
   val atLeastOne: prop list -> prop
   val equalNumber: (prop list * prop list) -> prop
   val moreThanOrEqualNumber: (prop list * prop list) -> prop
   val moreThanNumber: (prop list * prop list) -> prop
   val eval: prop -> bool option

   val runProfile: bool ref
   val printProp: prop -> string

   val eqCnf: (prop * int) -> (prop * int * int)
   val eqCnf': int ref -> prop  -> prop
   val appendCnf: prop list -> prop list

   val simplifyProp: prop -> prop
   val prCnfInDimacs: TextIO.outstream -> (prop * int * int) -> unit
(*    val solveByMinisat: string * string list -> prop * cnfInfo -> bool *)

   val prAtom: int -> string
   val prProp: prop -> string
end;

structure Prop: PROP =
struct
   datatype prop = Atom of int | Neg of prop
		  | Conj of prop list | Disj of prop list
		  | Imp of prop * prop | Iff of prop * prop 
		  | IfThenElse of prop * prop * prop

   val True = Conj []
   val False = Disj []
   fun allFalse xs = Conj (List.map (fn x=> Neg x) xs)
   fun one [] = False
     | one (x::xs) = IfThenElse (x, allFalse xs, one xs)
   fun atMostOne [] = True
     | atMostOne (x::xs) = IfThenElse (x, allFalse xs, atMostOne xs)
   fun atLeastOne xs = Disj xs
   fun equalNumber ([],[]) = True
     | equalNumber ([],ys) = allFalse ys
     | equalNumber (xs,[]) = allFalse xs
     | equalNumber (x::xs,y::ys) = 
       IfThenElse (Iff (x,y), 
		   equalNumber (xs,ys),
		   Disj [Conj [x, equalNumber (x::xs,ys)],
			 Conj [y, equalNumber (xs,y::ys)]])
   fun moreThanOrEqualNumber (xs,[]) = True
     | moreThanOrEqualNumber ([],ys) = allFalse ys
     | moreThanOrEqualNumber (x::xs,y::ys) = 
       IfThenElse (Iff (x,y), 
		   moreThanOrEqualNumber (xs,ys),
		   Disj [Conj [x, moreThanOrEqualNumber (x::xs,ys)],
			 Conj [y, moreThanNumber (xs,ys)]])
   and moreThanNumber ([],ys) = False
     | moreThanNumber (xs,[]) = atLeastOne xs
     | moreThanNumber (x::xs,y::ys) = 
       IfThenElse (Iff (x,y), 
		   moreThanNumber (xs,ys),
		   Disj [Conj [x, moreThanOrEqualNumber (xs,ys)],
			 Conj [y, moreThanNumber (xs,y::ys)]])



   fun eval (Atom _)  = NONE
     | eval (Neg p)  = (case (eval p) of
			   SOME b => SOME (not b)
			 | NONE => NONE)
     | eval (Conj ps)  = evalConj ps
     | eval (Disj ps)  = evalDisj ps
     | eval (Imp (p,q))  = (case (eval p) of
			       SOME b => if b then eval q
					 else SOME true
			     | NONE => NONE)
     | eval (Iff (p,q))  = (case (eval p, eval q) of
				(SOME bp, SOME bq) => if bp then (SOME bq)
						      else SOME (not bq)
			      | _ => NONE)
     | eval (IfThenElse (p,q,r)) = (case (eval p) of
					NONE => NONE					
				      | SOME bp => if bp then (eval q)
						   else (eval r))
   and evalConj []  = SOME true
     | evalConj (p::ps) = case (eval p, evalConj ps) of
			     (SOME b1, SOME b2) => SOME (b1 andalso b2)
			   | _ => NONE
   and evalDisj []  = SOME false
     | evalDisj (p::ps) = case (eval p, evalDisj ps) of
			     (SOME b1, SOME b2) => SOME (b1 orelse b2)
			   | _ => NONE



   val runProfile = ref false : bool ref

   fun printProp (Atom i) = "(Atom " ^ (Int.toString i) ^ ")"
     | printProp (Neg p) = "(Neg " ^ (printProp p) ^ ")"
     | printProp (Conj ps) = "(Conj [" ^ (printPropList ps) ^ "])"
     | printProp (Disj ps) = "(Disj [" ^ (printPropList ps) ^ "])"
     | printProp (Imp (p,q)) = "(Imp (" ^ (printProp p) ^ "," ^ (printProp q) ^ "))"
     | printProp (Iff (p,q)) = "(Iff (" ^ (printProp p) ^ "," ^ (printProp q) ^ "))"
     | printProp (IfThenElse (p,q,r)) = "(IfThenElse (" ^ (printProp p) ^ "," 
					^ (printProp q) ^ "," 
					^ (printProp r) ^ "))"
									    
   and printPropList [] = ""
     | printPropList (p::[]) = (printProp p)
     | printPropList (p::q::ps) = (printProp p) ^ "," ^ (printPropList (q::ps))


  (*  makePropInfo: $BD9$5(B(= $BItJ,O@M}<0$N?t(B)$B!$%"%H%`HV9f$N:GBg@dBPCM(B *)
   fun makePropInfo prop =
       let
	   fun lenXmaxAtomNum (Atom i) = (1, Int.abs i)
	     | lenXmaxAtomNum (Neg p) = let val (i,x) = lenXmaxAtomNum p
					in (i+1, x)
					end
	     | lenXmaxAtomNum (Conj ps) = List.foldr lenXmaxAtomNum' (1,0) ps
	     | lenXmaxAtomNum (Disj ps) = List.foldr lenXmaxAtomNum' (1,0) ps
	     | lenXmaxAtomNum (Imp (p1,p2)) = let val (i,x) = lenXmaxAtomNum p1
						  val (j,y) = lenXmaxAtomNum p2
					      in (i+j+1, Int.max (x,y))
					      end
	     | lenXmaxAtomNum (Iff (p1,p2)) = let val (i,x) = lenXmaxAtomNum p1
						  val (j,y) = lenXmaxAtomNum p2
					      in (i+j+1, Int.max (x,y))
					      end
	     | lenXmaxAtomNum (IfThenElse (p1,p2,p3)) = let 
		   val (i,x) = lenXmaxAtomNum p1
		   val (j,y) = lenXmaxAtomNum p2
		   val (k,z) = lenXmaxAtomNum p3
					      in (i+j+k+1, Int.max(Int.max (x,y), z))
					      end
	   and lenXmaxAtomNum' (p, (i,x)) =
	       let val (j,y) = lenXmaxAtomNum p 
	       in (i+j+1, Int.max (x,y))
	       end
	   val (len,max) = lenXmaxAtomNum prop
       in
	   {lenOfProp = len, maxAtomIndex=max}
       end

   fun eprint s = TextIO.output (TextIO.stdErr, s)
   exception PropError of string


   structure RBM : ORD_MAP = 
   let structure Key : ORD_KEY = 
       struct 
       type ord_key = prop
       fun compare (Atom i, Atom j) = Int.compare (i,j)
	 | compare (Atom _, _) = GREATER
	 | compare (_, Atom _) = LESS
	 | compare (Neg p, Neg q) = compare (p,q)
	 | compare (Neg _, _) = GREATER
	 | compare (_, Neg _) = LESS
	 | compare (IfThenElse (p1,p2,p3), IfThenElse (q1,q2,q3)) = 
	   (case compare (p1,q1)
		of EQUAL => (case compare (p2,q2) 
				 of EQUAL => compare (p3,q3) 
				  | GREATER => GREATER 
				  | LESS => LESS)
		 | GREATER => GREATER 
		 | LESS => LESS)
  	 | compare (IfThenElse _, _) = GREATER 
  	 | compare (_, IfThenElse _) = LESS 
	 | compare (Imp (p1,p2), Imp (q1,q2)) = 
	   (case compare (p1,q1)
		of EQUAL => compare (p2,q2)
		 | GREATER => GREATER 
		 | LESS => LESS)
	 | compare (Imp _, _) = GREATER
	 | compare (_, Imp _) = LESS
	 | compare (Iff (p1,p2), Iff (q1,q2)) = 
	   (case compare (p1,q1)
		of EQUAL => compare (p2,q2)
		 | GREATER => GREATER 
		 | LESS => LESS)
  	 | compare (Iff _, _) = GREATER 
  	 | compare (_, Iff _) = LESS 
	 | compare (Conj ps, Conj qs) =  compareLex (ps, qs)
	 | compare (Conj _, _) = GREATER
	 | compare (_,Conj _) = LESS
	 | compare (Disj ps, Disj qs) =  compareLex (ps, qs)
(* 	 | compare (Disj _, _) = GREATER *)
(* 	 | compare (_, Disj _) = LESS *)
       and compareLex ([], []) = EQUAL
	 | compareLex (p::ps, []) = GREATER
	 | compareLex ([], q::qs) = LESS
	 | compareLex (p::ps, q::qs) = 
	   case compare (p,q)
		of EQUAL => compareLex (ps,qs)
		 | GREATER => GREATER 
		 | LESS => LESS
       end
   in RedBlackMapFn (Key)
   end


   local
       fun retrieve _ _ (Atom i) = i
	 | retrieve table count (Neg q) = ~(retrieve table count q)
	 | retrieve table count p =
	   case (RBM.find (!table, p))
	    of SOME v => v
	     | NONE => (count := !count + 1; table := RBM.insert (!table, p,!count);
(* 			 			print ("["^ (Int.toString (!count)) ^ "]"); *)
			!count)
       val top = Conj []
       val bot = Disj []
       fun renBoth table count (p as Neg q) = 
	   let val B = retrieve table count p
	       val C = retrieve table count q
	   in Conj [Disj [(Atom (~B)), (Atom (~C))], Disj [(Atom B), (Atom C)] ]
	   end
	 | renBoth table count (p as Conj ps) = 
	   let val B = retrieve table count p
	   in Conj ((Disj ((Atom B)::(List.map (fn pi => (Atom (~(retrieve table count pi)))) ps)))
		    ::(List.map (fn pi => Disj [(Atom (~B)), (Atom (retrieve table count pi))]) ps))
	   end
	 | renBoth table count (p as Disj ps) = 
	   let val B = retrieve table count p
	   in Conj ((Disj ((Atom (~B))::(List.map (fn pi => (Atom (retrieve table count pi))) ps)))
		    ::(List.map (fn pi => Disj [(Atom B), (Atom (~(retrieve table count pi)))]) ps))
	   end
	 | renBoth table count (p as Imp (p1,p2)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	   in Conj [Disj [(Atom (~B)), (Atom(~B1)), (Atom B2)], 
		    Disj [(Atom B), (Atom B1)], 
		    Disj [(Atom B), (Atom (~B2))]]
	   end
	 | renBoth table count (p as Iff (p1,p2)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	   in Conj [Disj [(Atom (~B)), (Atom B1), (Atom (~B2))], 
		    Disj [(Atom (~B)), (Atom (~B1)), (Atom B2)],
		    Disj [(Atom B), (Atom B1), (Atom B2)], 
		    Disj [(Atom B), (Atom (~B1)), (Atom (~B2))]]
	   end
	 | renBoth table count (p as IfThenElse (p1,p2,p3)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	       val B3 = retrieve table count p3
	   in Conj [Disj [(Atom (~B)), (Atom (~B1)), (Atom B2)], 
		    Disj [(Atom (~B)), (Atom B1), (Atom B3)],
		    Disj [(Atom B), (Atom (~B1)), (Atom (~B2))], 
		    Disj [(Atom B), (Atom B1), (Atom (~B3))], 
		    Disj [(Atom B), (Atom (~B2)), (Atom (~B3))]]
	   end
	 | renBoth _ _ _ = raise PropError "renBoth\n"

       fun renPos table count (p as Conj ps) = 
	   let val B = retrieve table count p
	   in Conj (List.map (fn pi => Disj [(Atom (~B)), (Atom (retrieve table count pi))]) ps)
	   end
	 | renPos table count (p as Disj ps) = 
	   let val B = retrieve table count p
	   in Disj ((Atom (~B))::(List.map (fn pi => (Atom (retrieve table count pi))) ps))
	   end
	 | renPos table count (p as Imp (p1,p2)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	   in Disj [(Atom (~B)), (Atom (~B1)), (Atom B2)] 
	   end
	 | renPos table count (p as Iff (p1,p2)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	   in Conj [Disj [(Atom (~B)), (Atom B1), (Atom (~B2))], 
		    Disj [(Atom (~B)), (Atom (~B1)), (Atom B2)]]
	   end
	 | renPos table count (p as IfThenElse (p1,p2,p3)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	       val B3 = retrieve table count p3
	   in Conj [Disj [(Atom (~B)), (Atom (~B1)), (Atom B2)], 
		    Disj [(Atom (~B)), (Atom B1), (Atom B3)]]
	   end
	 | renPos _ _ _ = raise PropError "renPos\n"

       fun renNeg table count (p as Conj ps) = 
	   let val B = retrieve table count p
	   in Disj ((Atom B)::(List.map (fn pi => (Atom (~(retrieve table count pi)))) ps))
	   end
	 | renNeg table count (p as Disj ps) = 
	   let val B = retrieve table count p
	   in Conj (List.map (fn pi => Disj [(Atom B), (Atom (~(retrieve table count pi)))]) ps)
	   end
	 | renNeg table count (p as Imp (p1,p2)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	   in Conj [Disj [(Atom B), (Atom B1)],
		    Disj [(Atom B), (Atom (~B2))]]
	   end
	 | renNeg table count (p as Iff (p1,p2)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	   in Conj [Disj [(Atom B), (Atom B1), (Atom B2)], 
		    Disj [(Atom B), (Atom (~B1)), (Atom (~B2))]]
	   end
	 | renNeg table count (p as IfThenElse (p1,p2,p3)) = 
	   let val B = retrieve table count p
	       val B1 = retrieve table count p1
	       val B2 = retrieve table count p2
	       val B3 = retrieve table count p3
	   in Conj [Disj [(Atom B), (Atom (~B1)), (Atom (~B2))], 
		    Disj [(Atom B), (Atom B1), (Atom (~B3))], 
		    Disj [(Atom B), (Atom (~B2)), (Atom (~B3))]]
	   end
	 | renNeg _ _ _ = raise PropError "renNeg\n"

       fun ctBoth _ _ (Atom _) = []
	 | ctBoth table count (p as Neg q) = 
	   let val r = renBoth table count p
	       val cs = ctBoth table count q
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctBoth\n"
	   end
	 | ctBoth table count (p as Conj ps) = 
	   let val r = renBoth table count p
	       val cs = List.foldr (fn (q,qs) => List.@(ctBoth table count q, qs))  [] ps
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctBoth\n"
	   end
	 | ctBoth table count (p as Disj ps) = 
	   let val r = renBoth table count p
	       val cs = List.foldr (fn (q,qs) => List.@(ctBoth table count q, qs))  [] ps
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctBoth\n"
	   end
	 | ctBoth table count (p as Imp (p1,p2)) = 
	   let val r = renBoth table count p
 	       val cs = List.@ (ctBoth table count p1, ctBoth table count p2)
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctBoth\n"
	   end
	 | ctBoth table count (p as Iff (p1,p2)) = 
	   let val r = renBoth table count p
 	       val cs = List.@ (ctBoth table count p1, ctBoth table count p2)
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctBoth\n"
	   end
	 | ctBoth table count (p as IfThenElse (p1,p2,p3)) = 
	   let val r = renBoth table count p
 	       val cs = List.@ (ctBoth table count p1, 
				List.@ (ctBoth table count p2,
					ctBoth table count p3))
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctBoth\n"
	   end

       fun ctPos _ _ (Atom _) = []
	 | ctPos table count (p as Neg q) = ctNeg table count q
	 | ctPos table count (p as Conj ps) = 
	   let val r = renPos table count p
	       val cs = List.foldr (fn (q,qs) => List.@(ctPos table count q, qs))  [] ps
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctPos\n"
	   end
	 | ctPos table count (p as Disj ps) = 
	   let val r = renPos table count p
	       val cs = List.foldr (fn (q,qs) => List.@(ctPos table count q, qs))  [] ps
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctPos\n"
	   end
	 | ctPos table count (p as Imp (p1,p2)) = 
	   let val r = renPos table count p
 	       val cs = List.@ (ctNeg table count p1, ctPos table count p2)
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctPos\n"
	   end
	 | ctPos table count (p as Iff (p1,p2)) = 
	   let val r = renPos table count p
 	       val cs = List.@ (ctBoth table count p1, ctBoth table count p2)
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctPos\n"
	   end
	 | ctPos table count (p as IfThenElse (p1,p2,p3)) = 
	   let val r = renPos table count p
 	       val cs = List.@ (ctBoth table count p1, 
				List.@ (ctPos table count p2,
					ctPos table count p3))
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctPos\n"
	   end
       and ctNeg _ _ (Atom _) = []
	 | ctNeg table count (p as Neg q) = ctPos table count q
	 | ctNeg table count (p as Conj ps) = 
	   let val r = renNeg table count p
	       val cs = List.foldr (fn (q,qs) => List.@(ctNeg table count q, qs))  [] ps
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctNeg\n"
	   end
	 | ctNeg table count (p as Disj ps) = 
	   let val r = renNeg table count p
	       val cs = List.foldr (fn (q,qs) => List.@(ctNeg table count q, qs))  [] ps
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctNeg\n"
	   end
	 | ctNeg table count (p as Imp (p1,p2)) = 
	   let val r = renNeg table count p
 	       val cs = List.@ (ctPos table count p1, ctNeg table count p2)
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctNeg\n"
	   end
	 | ctNeg table count (p as Iff (p1,p2)) = 
	   let val r = renNeg table count p
 	       val cs = List.@ (ctBoth table count p1, ctBoth table count p2)
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctNeg\n"
	   end
	 | ctNeg table count (p as IfThenElse (p1,p2,p3)) = 
	   let val r = renNeg table count p
 	       val cs = List.@ (ctBoth table count p1, 
				List.@ (ctNeg table count p2,
					ctNeg table count p3))
	   in case r 
	       of Conj xs => List.@(xs,cs)
		| Disj _ => r::cs
		| _ => raise PropError "ctNeg\n"
	   end

   in
   fun eqCnf (prop,max) = 
       let val count = ref max      (* $B:GBg(Batom$BHV9f$N%+%&%s%?(B *)
	   val table = ref RBM.empty
	   val ds = (Disj [Atom (retrieve table count prop)]) :: ctPos table count prop
	   val dlen = List.length ds
	   val cnf = Conj ds
       in
	   (cnf, dlen,!count)
       end
   fun eqCnf' symCount prop  = 
       let val count = symCount     (* $B:GBg(Batom$BHV9f$N%+%&%s%?(B *)
	   val table = ref RBM.empty
	   val ds = (Disj [Atom (retrieve table count prop)]) :: ctPos table count prop
	   val dlen = List.length ds
	   val cnf = Conj ds
       in
	   cnf
       end
   end

   fun appendCnf ps = 
       let
	   fun appcnf [] ans = rev ans
	     | appcnf (p::ps) ans = 
	       case p of 
	           Conj xs => appcnf ps (xs @ ans)
		 | _ => appcnf ps (p::ans)
       in
	   appcnf ps []
       end


   fun prCnfInDimacs outs (cnf,numOfClause,maxVarIndex) = 
       let 
	   fun prAtom atom = 
	       case atom of
		   (Atom i) => if i > 0 
			       then TextIO.output (outs, (Int.toString (Int.abs i)) ^ " ")
			       else TextIO.output (outs, "-" ^ (Int.toString (Int.abs i)) ^ " ")
		 | _ => (print "prCnfInDimacs: atom expected.\n";
			 raise PropError "prCnfInDimacs: atom expected.\n")

	   fun prDisj clause = 
	       case clause of 
		   (Atom i) => (if i > 0 
				then TextIO.output (outs, (Int.toString (Int.abs i)) ^ " ")
				else TextIO.output (outs, "-" ^ (Int.toString (Int.abs i)) ^ " ");
				TextIO.output (outs, "0\n"))
		 | (Disj ys) => (List.app prAtom ys; TextIO.output (outs, "0\n"))
		 | _ => (print "prCnfInDimacs: Disj expected.\n";
			 raise PropError "prCnfInDimacs: Disj expected.\n")

	   fun prCnf cnf = 
	       case cnf of 
		   (Atom i) => (if i > 0 
				then TextIO.output (outs, (Int.toString (Int.abs i)) ^ " ")
				else TextIO.output (outs, "-" ^ (Int.toString (Int.abs i)) ^ " ");
				TextIO.output (outs, "0\n"))
		 | (Disj ys) => (List.app prAtom ys; TextIO.output (outs, "0\n"))
		 | (Conj xs) => List.app prDisj xs
		 | _ => (print "prCnfInDimacs: Conj expected.\n";
			 raise PropError "prCnfInDimacs: Conj expected.\n")

	   fun main () = 
	       let 
		   val _ = TextIO.output (outs, "p cnf " 
						^ (Int.toString maxVarIndex) ^ " "
						^ (Int.toString numOfClause) ^ "\n")
		   val _ = prCnf cnf
		   val _ = TextIO.flushOut outs
	       in ()
	       end
       in
	   if !runProfile
	   then TimeUtil.profile (main, "prCnfInDimacs")
	   else main ()
       end


(*    local *)
(*        fun printAtom outs (Atom i) =  *)
(* 	   if i > 0  *)
(* 	   then TextIO.output (outs, (Int.toString (Int.abs i)) ^ " ") *)
(* 	   else TextIO.output (outs, "-" ^ (Int.toString (Int.abs i)) ^ " ") *)
(* 	 | printAtom _ _ = raise PropError "printAtom: atom expected.\n" *)
(*        fun printDisj outs (Disj ys) = (List.map (printAtom outs) ys; TextIO.output (outs, "0\n")) *)
(* 	 | printDisj _ _ = raise PropError "printDisj: disjunction expected.\n" *)
(*        fun printCnf outs (Conj xs) (info:cnfInfo) =  *)
(* 	   let val mx = #maxAtomIndex info *)
(* 	       val len = #numOfClause info *)
(* 	   in *)
(* 	       (TextIO.output (outs, "p cnf " ^ (Int.toString mx)); *)
(* 		TextIO.output (outs, " " ^ (Int.toString len) ^ "\n"); *)
(* 		List.map (printDisj outs) xs) *)
(* 	   end *)
(* 	 | printCnf _ _ _ = raise PropError "printCnf: cnf expected.\n" *)
(*    in *)
(*    fun solveByMinisat minisat_command (cnf, info:cnfInfo) =  *)
(*        let (\* val proc = Unix.execute(minisat, []) *\) *)
(* 	   val proc = Unix.execute minisat_command *)
(* 	   val (ins,outs) = Unix.streamsOf proc *)
(* 	   val _ = (printCnf outs cnf info; TextIO.flushOut outs) *)
(* 	   val _ = TextIO.closeOut outs *)
(* 	   fun receiveReport str = *)
(*  	       if str <> "" then str *)
(* 	       else receiveReport (str ^ (TextIO.inputN(ins,1))) *)
(* 	   val report = receiveReport "" *)
(* 	   val flag = substring(report,0,1) *)
(* 	   val _ = Unix.reap proc *)
(*        in *)
(* 	   (\* 	   print report; *\) *)
(* 	   if flag = "S" then true *)
(* 	   else if flag = "U" then false *)
(* 	   else raise PropError ("solveByMinisat: ???[" ^ report ^ "]\n") *)
(*        end *)
(*        handle PropError s => (eprint s; false) *)
(*    end *)


   local
       fun member x ys = isSome (List.find (fn y => x = y) ys)
   in
   fun simplifyProp (p as Atom _) = p
     | simplifyProp (Neg q) = Neg (simplifyProp q)
     | simplifyProp (Conj ps) =
       let val ps' = List.map simplifyProp ps
       in if (List.length ps' = 1) then (hd ps')
	  else if (member (Disj []) ps') then (Disj [])
	  else Conj (List.filter (fn p => p <> (Conj [])) ps')
       end
     | simplifyProp (Disj ps) =
       let val ps' = List.map simplifyProp ps
       in if (List.length ps' = 1) then (hd ps')
	  else if (member (Conj []) ps') then (Conj [])
	  else Disj (List.filter (fn p => p <> (Disj [])) ps')
       end
      | simplifyProp (Imp (q1,q2)) = Imp (simplifyProp q1, simplifyProp q2)
      | simplifyProp (Iff (q1,q2)) = Iff (simplifyProp q1, simplifyProp q2)
      | simplifyProp (IfThenElse (q1,q2,q3)) = IfThenElse (simplifyProp q1, 
							   simplifyProp q2,
							   simplifyProp q3)
   end


   (* print proposition in yices format *)
   fun prAtom i = if i >= 0 then "x" ^ (Int.toString i)
		 else "y" ^ (Int.toString (~i))
   fun prProp (Atom i) = prAtom i
     | prProp (Neg p) = "(not " ^ (prProp p) ^ ")"
     | prProp (Conj []) = "true"
     | prProp (Conj ps) = "(and " ^ (List.foldr (fn (p,str) => " " ^ (prProp p) ^ str) ")" ps)
     | prProp (Disj []) = "false"
     | prProp (Disj ps) = "(or " ^ (List.foldr (fn (p,str) => " " ^ (prProp p) ^ str) ")" ps)
     | prProp (Imp (p,q)) = "(implies " ^ (prProp p) ^ " " ^ (prProp q) ^ ")"
     | prProp (Iff (p,q)) = "(and " ^ "(implies " ^ (prProp p) ^ " " ^ (prProp q) ^ ")"
			    ^ "(implies " ^ (prProp q) ^ " " ^ (prProp p) ^ "))"
     | prProp (IfThenElse (p,q,r)) = "(if " ^ (prProp p) ^ " " 
				     ^ (prProp q) ^ " " 
				     ^ (prProp r) ^ ")"


end;


